/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: super.c,v 1.15 2005/01/03 21:10:41 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


STATIC void
wrapfs_read_inode(inode_t *inode)
{
    print_entry_location();

    itopd(inode) = kmalloc(sizeof(struct wrapfs_inode_info), GFP_KERNEL);
    ASSERT(itopd(inode));
    itohi(inode) = NULL;
    inode->i_version = ++global_event;	/* increment inode version */
    inode->i_op = &wrapfs_main_iops;

    print_exit_location();
}


#ifdef NOT_NEEDED
/*
 * Not needed.  This function will be called by sync_one(), which is
 * called by sync_list(), which is called by sync_inodes(kdev_t dev),
 * in order to sync dirty inodes of a vfs to disk.  But the higher
 * vfs code will call sync_inodes() on the lower mounted device anyway
 * so we should not have call write_inode() on the hidden_inode.
 */
STATIC void
wrapfs_write_inode(inode_t *inode)
{
    return;
}
#endif /*  NOT_NEEDED */


STATIC void
wrapfs_put_inode(inode_t *inode)
{
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();
    fist_dprint(8, __FUNCTION__ " i_count = %d, i_nlink = %d\n", inode->i_count, inode->i_nlink);
    /*
     * This is really funky stuff:
     * Basically, if i_count == 1, iput will then decrement it and this inode will be destroyed.
     * It is currently holding a reference to the hidden inode.
     * Therefore, it needs to release that reference by calling iput on the hidden inode.
     * iput() _will_ do it for us (by calling our clear_inode), but _only_ if i_nlink == 0.
     * The problem is, NFS keeps i_nlink == 1 for silly_rename'd files.
     * So we must for our i_nlink to 0 here to trick iput() into calling our clear_inode.
     */
    if (inode->i_count == 1)
	inode->i_nlink = 0;
    print_exit_location();
    return;
}


/*
 */
STATIC void
wrapfs_delete_inode(inode_t *inode)
{
    print_entry_location();

    fist_checkinode(inode, "wrapfs_delete_inode IN");
    inode->i_size = 0;		/* every f/s seems to do that */
    fist_checkinode(inode, "wrapfs_delete_inode OUT");

    print_exit_location();
}


/*
 * Called by iput() when the inode reference count reached zero
 * and the inode is not hashed anywhere.  Used to clear anything
 * that needs to be, before the inode is completely destroyed and put
 * on the inode free list.
 */
STATIC void
wrapfs_clear_inode(inode_t *inode)
{
    print_entry_location();

    fist_checkinode(inode, "wrapfs_clear_inode IN");
    fist_print_inode("clear_inode inode", inode);
    fist_print_inode("clear_inode hidden_inode", itohi(inode));
    /*
     * Decrement a reference to a hidden_inode, which was incremented
     * by our read_inode when it was created initially.
     */
    iput(itohi(inode));
    kfree_s(itopd(inode), sizeof(struct wrapfs_inode_info));
    itopd(inode) = NULL;

    print_exit_location();
}


STATIC int
wrapfs_notify_change(dentry_t *dentry, struct iattr *ia)
{
    int err = 0;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    inode_t *inode = dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();
    fist_checkinode(inode, "wrapfs_notify_change");

    err = notify_change(hidden_dentry, ia);

    /*
     * The lower file system has changed the attributes,
     * so we copy the hidden_inode's attributes (and a few more) to
     * our inode.
     */
    fist_copy_attr_all(inode, hidden_inode);

out:
    fist_checkinode(inode, "post wrapfs_notify_change");
    print_exit_status(err);
    return err;
}


STATIC void
wrapfs_put_super(super_block_t *sb)
{
    print_entry_location();

    if (stopd(sb)) {
	kfree_s(stopd(sb), sizeof(struct wrapfs_sb_info));
	stopd(sb) = NULL;
    }
    fist_dprint(6, "wrapfs: released super\n");
#ifdef MODULE
    fist_mod_dec_use_count();
#endif /* MODULE */

    print_exit_location();
}


#ifdef NOT_NEEDED
/*
 * This is called in do_umount before put_super.
 * The superblock lock is not held yet.
 * We probably do not need to define this or call write_super
 * on the hidden_sb, because sync_supers() will get to hidden_sb
 * sooner or later.  But it is also called from file_fsync()...
 */
STATIC void
wrapfs_write_super(super_block_t *sb)
{
    return;
}
#endif /* NOT_NEEDED */


STATIC int
wrapfs_statfs(super_block_t *sb, struct statfs *buf, int bufsiz)
{
    int err = 0;
    super_block_t *hidden_sb = stohs(sb);

    print_entry_location();

    if (hidden_sb->s_op && hidden_sb->s_op->statfs)
	err = hidden_sb->s_op->statfs(hidden_sb, buf, bufsiz);

    print_exit_status(err);
    return err;
}


/*
 * XXX: not implemented.  This is not allowed yet.
 * Should we call this on the hidden_sb?  Probably not.
 */
STATIC int
wrapfs_remount_fs(super_block_t *sb, int *flags, char *data)
{
    return -ENOSYS;
}


/*
 * Called in do_umount() if the MNT_FORCE flag was used and this
 * function is defined.  See comment in linux/fs/super.c:do_umount().
 * Used only in nfs, to kill any pending RPC tasks, so that subsequent
 * code can actually succeed and won't leave tasks that need handling.
 *
 * PS. I wonder if this is somehow useful to undo damage that was
 * left in the kernel after a user level file server (such as amd)
 * dies.
 */
STATIC void
wrapfs_umount_begin(super_block_t *sb)
{
    super_block_t *hidden_sb = stohs(sb);

    print_entry_location();

    if (hidden_sb->s_op->umount_begin)
	hidden_sb->s_op->umount_begin(hidden_sb);

    print_exit_location();
}


struct super_operations wrapfs_sops =
{
    read_inode:		wrapfs_read_inode,
    delete_inode:	wrapfs_delete_inode,
    clear_inode:	wrapfs_clear_inode,
    notify_change:	wrapfs_notify_change,
    put_super:		wrapfs_put_super,
    statfs:		wrapfs_statfs,
    remount_fs:		wrapfs_remount_fs,
    umount_begin:	wrapfs_umount_begin,
    put_inode:		wrapfs_put_inode,
#ifdef NOT_NEEDED
    write_inode:	wrapfs_write_inode,
    write_super:	wrapfs_write_super,
#endif /* not NOT_NEEDED */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
