/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: dentry.c,v 1.16 2005/01/03 21:10:42 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


/*
 * THIS IS A BOOLEAN FUNCTION: returns 1 if valid, 0 otherwise.
 */
STATIC int
wrapfs_d_revalidate(dentry_t *dentry, int flags)
{
    int err = 1; // default is valid (1); invalid is 0.
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);

    print_entry_location();

    if (!hidden_dentry->d_op || !hidden_dentry->d_op->d_revalidate)
	goto out;

    err = hidden_dentry->d_op->d_revalidate(hidden_dentry, flags);

 out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_d_hash(dentry_t *dentry, qstr_t *name)
{
    int err = 0;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);

    print_entry_location();

    if (!hidden_dentry->d_op || !hidden_dentry->d_op->d_hash)
	goto out;

    err = hidden_dentry->d_op->d_hash(hidden_dentry, name);

 out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_d_compare(dentry_t *dentry, qstr_t *a, qstr_t *b)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);

    print_entry_location();

    if (hidden_dentry->d_op && hidden_dentry->d_op->d_compare) {
	// XXX: WRONG: should encode_filename on a&b strings
	err = hidden_dentry->d_op->d_compare(hidden_dentry, a, b);
    } else {
	err = ((a->len != b->len) || memcmp(a->name, b->name, b->len));
    }

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_d_delete(dentry_t *dentry)
{
    dentry_t *hidden_dentry;
    int err = 0;

    print_entry_location();

    hidden_dentry = dtohd(dentry);
    /* added b/c of changes to dput(): it calls d_drop on us */
    if (hidden_dentry &&
	hidden_dentry->d_op &&
	hidden_dentry->d_op->d_delete) {
	err = hidden_dentry->d_op->d_delete(hidden_dentry);
    }

    print_exit_status(err);
    return err;
}


STATIC void
wrapfs_d_release(dentry_t *dentry)
{
    dentry_t *hidden_dentry;

    print_entry_location();
    fist_print_dentry("wrapfs_d_release IN dentry", dentry);

    // call dtohd directly because hidden_dentry could be NULL
    if (!dtopd(dentry)) {
	fist_dprint(6, "wrapfs_d_release: dentry without private data\n");
	goto out;
    }
    hidden_dentry = dtohd(dentry);
    fist_print_dentry("wrapfs_d_release IN hidden_dentry", hidden_dentry);
    if (hidden_dentry->d_inode)
	fist_dprint(6, "wrapfs_d_release: hidden_inode->i_count %d, i_num %lu.\n",
		    hidden_dentry->d_inode->i_count,
		    hidden_dentry->d_inode->i_ino);

#ifdef FIST_FILTER_SCA
    if (dtopd(dentry)->idx_dentry)
	dput(dtopd(dentry)->idx_dentry);
#endif /* FIST_FILTER_SCA */
    /* free private data (wrapfs_dentry_info) here */
    kfree_s(dtopd(dentry), sizeof(struct wrapfs_dentry_info));
    dtopd(dentry) = NULL;	/* just to be safe */
    /* decrement hidden dentry's counter and free its inode */
    dput(hidden_dentry);
 out:
    print_exit_location();
}


#ifdef FIST_DEBUG
/*
 * we don't need wrapfs_iput, because dentry_iput will call iput() if
 * d_iput is not defined. We left this implemented for ease of
 * tracing/debugging.
 */
STATIC void
wrapfs_d_iput(dentry_t *dentry, inode_t *inode)
{
    print_entry_location();
    iput(inode);
    print_exit_location();
}
#endif /* FIST_DEBUG */


struct dentry_operations wrapfs_dops = {
    d_revalidate:	wrapfs_d_revalidate,
    d_hash:		wrapfs_d_hash,
    d_compare:		wrapfs_d_compare,
    d_release:		wrapfs_d_release,
    d_delete:		wrapfs_d_delete,
#ifdef FIST_DEBUG
    d_iput:		wrapfs_d_iput,
#endif /* FIST_DEBUG */
};


/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
