/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: file.c,v 1.34 2005/01/03 21:10:42 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


/*******************
 * File Operations *
 *******************/


STATIC loff_t
wrapfs_llseek(file_t *file, loff_t offset, int origin)
{
    loff_t err;
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    fist_dprint(6, "wrapfs_llseek: file=%x, offset=0x%x, origin=%d\n",
		(int) file, (int) offset, origin);

    /* always set hidden position to this one */
    hidden_file->f_pos = file->f_pos;
    if (file->f_reada) { /* update readahead information if needed */
	hidden_file->f_reada = file->f_reada;
	hidden_file->f_ramax = file->f_ramax;
	hidden_file->f_raend = file->f_raend;
	hidden_file->f_ralen = file->f_ralen;
	hidden_file->f_rawin = file->f_rawin;
    }
    ASSERT(hidden_file->f_reada == file->f_reada);
    ASSERT(hidden_file->f_ramax == file->f_ramax);
    ASSERT(hidden_file->f_raend == file->f_raend);
    ASSERT(hidden_file->f_ralen == file->f_ralen);
    ASSERT(hidden_file->f_rawin == file->f_rawin);

    if (hidden_file->f_op && hidden_file->f_op->llseek)
	err = hidden_file->f_op->llseek(hidden_file, offset, origin);
    else
	err = default_llseek(hidden_file, offset, origin);

    if (err < 0)
	goto out;

    if (err != file->f_pos) {
	file->f_pos = err;
	// ION maybe this?
	// 	file->f_pos = hidden_file->f_pos;
	file->f_reada = 0;
	file->f_version = ++event;
    }

out:
    print_exit_status(err);
    return err;
}


#ifndef FIST_FILTER_DATA
#error do not use wrapfs_read
STATIC ssize_t
wrapfs_read(file_t *file, char *buf, size_t count, loff_t *ppos)
{
    int err = -EINVAL;
    file_t *hidden_file = ftohf(file);
    loff_t pos = *ppos;

    print_entry_location();

    if (!hidden_file->f_op || !hidden_file->f_op->read)
	goto out;

    err = hidden_file->f_op->read(hidden_file, buf, count, &pos);

    if (err > 0) {
	fist_copy_attr_atime(file->f_dentry->d_inode,
			     hidden_file->f_dentry->d_inode);
    }

    // MAJOR HACK
    /*
     * because pread() does not have any way to tell us that it is
     * our caller, then we don't know for sure if we have to update
     * the file positions.  This hack relies on read() having passed us
     * the "real" pointer of its struct file's f_pos field.
     */
    if (ppos == &file->f_pos)
	hidden_file->f_pos = *ppos = pos;
    if (hidden_file->f_reada) { /* update readahead information if needed */
	file->f_reada = hidden_file->f_reada;
	file->f_ramax = hidden_file->f_ramax;
	file->f_raend = hidden_file->f_raend;
	file->f_ralen = hidden_file->f_ralen;
	file->f_rawin = hidden_file->f_rawin;
    }

out:
    print_exit_status(err);
    return err;
}
#endif /* not FIST_FILTER_DATA */


#ifndef FIST_FILTER_DATA
#error do not use wrapfs_write
/* this wrapfs_write() does not modifying data pages! */
STATIC ssize_t
wrapfs_write(file_t *file, const char *buf, size_t count, loff_t *ppos)
{
    int err = -EINVAL;
    file_t *hidden_file = ftohf(file);
    inode_t *inode = file->f_dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);
    loff_t pos = *ppos;

    print_entry_location();

    if (!hidden_file->f_op || !hidden_file->f_op->read)
	goto out;

    /* adjust for append -- seek to the end of the file */
    if (file->f_flags & O_APPEND)
	pos = inode->i_size;

    err = hidden_file->f_op->write(hidden_file, buf, count, &pos);

    if (err > 0)
	fist_copy_attr_atime(inode, hidden_inode);

    /*
     * XXX: MAJOR HACK
     *
     * because pread() does not have any way to tell us that it is
     * our caller, then we don't know for sure if we have to update
     * the file positions.  This hack relies on read() having passed us
     * the "real" pointer of its struct file's f_pos field.
     */
    if (ppos == &file->f_pos)
	hidden_file->f_pos = *ppos = pos;

    /* update this inode's size */
    if (pos > inode->i_size)
	inode->i_size = pos;

out:
    print_exit_status(err);
    return err;
}
#endif /* not FIST_FILTER_DATA */


#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
struct wrapfs_getdents_callback {
    void *dirent;
    dentry_t *dentry;
    filldir_t filldir;
};

/* copied from generic filldir in fs/readir.c */
STATIC int
wrapfs_filldir(void *dirent, const char *name, int namlen, off_t offset, ino_t ino)
{
    struct wrapfs_getdents_callback *buf = (struct wrapfs_getdents_callback *) dirent;
    int err;
#ifdef FIST_FILTER_NAME
    char *decoded_name;
    int decoded_length;
#endif /* FIST_FILTER_NAME */

#ifdef FIST_FILTER_SCA
    /* skip .idx files */
    if (namlen > INDEX_EXTENSION_LEN-1 &&
	strncmp(&name[namlen-INDEX_EXTENSION_LEN+1],
		INDEX_EXTENSION,
		INDEX_EXTENSION_LEN-1) == 0) {
	//don't do this, it's not zero-terminated!!!
	//fist_dprint(7, "filldir: skip \"%s\"\n", name);
	return 0;
    }
#endif /* FIST_FILTER_SCA */

#ifdef FIST_FILTER_NAME
    decoded_length = wrapfs_decode_filename(name, namlen, &decoded_name, SKIP_DOTS,
					    buf->dentry->d_inode, buf->dentry->d_sb);
    if (decoded_length < 0)
	return 0;			/* no error, just skip the entry */

    FIST_OP_READDIR_CALL;

    err = buf->filldir(buf->dirent, decoded_name, decoded_length, offset, ino);
    kfree_s(decoded_name, decoded_length);
#else /* ! FIST_FILTER_NAME */
    err = buf->filldir(buf->dirent,name,namlen,offset,ino);
#endif /* FIST_FILTER_NAME */

    return err;
}
#endif /* FIST_FILTER_NAME || FIST_FILTER_SCA */


STATIC int
wrapfs_readdir(file_t *file, void *dirent, filldir_t filldir)
{
    int err = -ENOTDIR;
    file_t *hidden_file = ftohf(file);
    inode_t *inode = file->f_dentry->d_inode;
#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
    struct wrapfs_getdents_callback buf;
#endif /* FIST_FILTER_NAME || FIST_FILTER_SCA */

    print_entry_location();
    fist_checkinode(inode, "wrapfs_readdir");

#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
    /* prepare for callback */
    buf.dirent = dirent;
    buf.dentry = file->f_dentry;
    buf.filldir = filldir;
    err = vfs_readdir(hidden_file, wrapfs_filldir, (void *) &buf);
#else /* not FIST_FILTER_NAME */
    err = vfs_readdir(hidden_file, filldir, dirent);
#endif /* not FIST_FILTER_NAME */

    file->f_pos = hidden_file->f_pos;
    if (err > 0)
	fist_copy_attr_atime(inode, hidden_file->f_dentry->d_inode);

    fist_checkinode(inode, "post wrapfs_readdir");
    print_exit_status(err);
    return err;
}


STATIC unsigned int
wrapfs_poll(file_t *file, poll_table *wait)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    if (!hidden_file->f_op || !hidden_file->f_op->poll)
	goto out;

    err = hidden_file->f_op->poll(hidden_file, wait);

out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_ioctl(inode_t *inode, file_t *file, unsigned int cmd, unsigned long arg)
{
    int err = 0;
    file_t *hidden_file;
    vfs_t *this_vfs;
    vnode_t *this_vnode;
    int val;
# ifdef FIST_COUNT_WRITES
    extern unsigned long count_writes, count_writes_middle;
# endif /* FIST_COUNT_WRITES */

    print_entry_location();

    this_vfs = inode->i_sb;
    this_vnode = inode;

    /* check if asked for local commands */
    switch (cmd) {
#ifdef FIST_DEBUG
    case FIST_IOCTL_GET_DEBUG_VALUE:
	val = fist_get_debug_value();
	printk("IOCTL GET: send arg %d\n", val);
	err = put_user(val, (int *) arg);
# ifdef FIST_COUNT_WRITES
	printk("COUNT_WRITES:%lu:%lu\n", count_writes, count_writes_middle);
# endif /* FIST_COUNT_WRITES */
	break;

    case FIST_IOCTL_SET_DEBUG_VALUE:
	err = get_user(val, (int *) arg);
	if (err)
	    break;
	fist_dprint(6, "IOCTL SET: got arg %d\n", val);
	if (val < 0 || val > 20) {
	    err = -EINVAL;
	    break;
	}
	fist_set_debug_value(val);
	break;
#endif /* FIST_DEBUG */
#ifdef FIST_FILTER_SCA
    case FIST_IOCTL_GET_FAST_TAIL_VALUE:
	val = (stopd(inode->i_sb)->sca_flags != 0);
	printk("IOCTL GET: send arg %d\n", val);
	err = put_user(val, (int *) arg);
	break;

#ifdef FIST_FAST_TAILS
    case FIST_IOCTL_SET_FAST_TAIL_VALUE:
	err = get_user(val, (int *) arg);
	if (err)
	    break;
	fist_dprint(6, "IOCTL SET: got arg %d\n", val);
	if (val == 1) {
	    stopd(inode->i_sb)->sca_flags |= SCA_FLAG_FASTTAIL;
	} else if (val == 0) {
	    stopd(inode->i_sb)->sca_flags &= ~SCA_FLAG_FASTTAIL;
	} else {
	    err = -EINVAL;
	}
	break;
#endif /* FIST_FAST_TAILS */
#endif /* FIST_FILTER_SCA */

    /* add non-debugging fist ioctl's here */
FIST_IOCTL_ECLS

    default:
	hidden_file = ftohf(file);
	/* pass operation to hidden filesystem, and return status */
	if (hidden_file->f_op && hidden_file->f_op->ioctl)
	    err = hidden_file->f_op->ioctl(itohi(inode), hidden_file, cmd, arg);
    } /* end of outer switch statement */

    print_exit_status(err);
    return err;
}


#ifdef not_needed
STATIC int
wrapfs_mmap(file_t *file, vm_area_t *vma)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);
    inode_t *inode = file->f_dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);
    vm_area_t *hidden_vma = (vm_area_t *) 0xdeadc0de;

    print_entry_location();

    fist_dprint(6, "MMAP1: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, (int) inode->i_count, (int) hidden_inode->i_count);

    if (!hidden_file->f_op || !hidden_file->f_op->mmap) {
	err = -ENODEV;
	goto out;
    }

    /*
     * Most of this code comes straight from generic_file_mmap
     * in mm/filemap.c.
     */
    if ((vma->vm_flags & VM_SHARED) && (vma->vm_flags & VM_MAYWRITE)) {
	vma->vm_ops = &wrapfs_shared_vmops;
    } else {
	vma->vm_ops = &wrapfs_private_vmops;
    }
    if (!inode->i_sb || !S_ISREG(inode->i_mode)) {
	err = -EACCES;
	goto out;
    }
    if (!hidden_inode->i_op || !hidden_inode->i_mapping->a_ops->readpage) {
	err = -ENOEXEC;
	goto out;
    }
    UPDATE_ATIME(inode);

    /*
     * Now we do the hidden stuff, but only for shared maps.
     */
    if ((vma->vm_flags & VM_SHARED) && (vma->vm_flags & VM_MAYWRITE)) {
	hidden_vma = kmalloc(sizeof(vm_area_t), GFP_KERNEL);
	if (!hidden_vma) {
	    printk("MMAP: Out of memory\n");
	    err = -ENOMEM;
	    goto out;
	}
	// ION, is this right?
	memcpy(hidden_vma, vma, sizeof(vm_area_t));
	vmatohvma(vma) = hidden_vma;
	err = hidden_file->f_op->mmap(hidden_file, hidden_vma);
	hidden_vma->vm_file = hidden_file;
	atomic_inc(&hidden_file->f_count);
    }

    /*
     * XXX: do we need to insert_vm_struct and merge_segments as is
     * done in do_mmap()?
     */
 out:
    fist_dprint(6, "MMAP2: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, (int) inode->i_count, (int) hidden_inode->i_count);
#if 0
    if (!err) {
	inode->i_mmap = vma;
	hidden_inode->i_mmap = hidden_vma;
    }
#endif
    print_exit_status(err);
    return err;
}
#endif /* not_needed */


STATIC int
wrapfs_open(inode_t *inode, file_t *file)
{
    int err = 0;
    int hidden_mode, hidden_flags;
    file_t *hidden_file;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(file->f_dentry);

    print_entry_location();
    fist_print_dentry(__FUNCTION__ " IN hidden_dentry", hidden_dentry);

    dget(hidden_dentry);

    hidden_flags = file->f_flags;
#ifdef FIST_FILTER_DATA
    hidden_mode = file->f_mode | FMODE_READ;
    if (file->f_flags & O_APPEND) {
	fist_dprint(5, "file is opened in append-only mode\n");
	hidden_flags &= ~O_APPEND; /* turn off O_APPEND flag */
    }
#else /* not FIST_FILTER_DATA */
    hidden_mode = file->f_mode;
#endif /* not FIST_FILTER_DATA */

    /*
     * dentry_open will decrement mnt refcnt if err.
     * otherwise fput() will do an mntput() for us upon file close.
     */
    mntget(stopd(inode->i_sb)->hidden_mnt);
    hidden_file = dentry_open(hidden_dentry, stopd(inode->i_sb)->hidden_mnt, hidden_mode, hidden_flags);
    if (IS_ERR(hidden_file)) {
	err = PTR_ERR(hidden_file);
	goto out;
    }

    ftohf(file) = hidden_file;	/* link two files */

 out:
    fist_print_dentry(__FUNCTION__ " OUT hidden_dentry", hidden_dentry);
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_flush(file_t *file)
{
    int err = 0;		/* assume ok (see open.c:close_fp) */
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    if (!hidden_file->f_op || !hidden_file->f_op->flush)
	goto out;

    err = hidden_file->f_op->flush(hidden_file);

out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_release(inode_t *inode, file_t *file)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);
    dentry_t *hidden_dentry;

    print_entry_location();
    ASSERT(hidden_file != NULL);

    fist_print_dentry("wrapfs_release IN hidden_dentry", hidden_file->f_dentry);
    fist_checkinode(inode, "wrapfs_release");
    fist_dprint(6, "wrapfs_release IN, file->f_count=%d\n", file->f_count);
    /*
     * will decrement file refcount, and if 0, destroy the file,
     * which will call the lower file system's file release function.
     */
    hidden_dentry = hidden_file->f_dentry;
    fput(hidden_file);

    fist_dprint(6, "wrapfs_release done\n");
    fist_checkinode(inode, "post wrapfs_release");

    fist_print_dentry("wrapfs_release OUT hidden_dentry", hidden_dentry);
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_fsync(file_t *file, dentry_t *dentry)
{
    int err = -EINVAL;
    file_t *hidden_file = ftohf(file);
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);

    print_entry_location();

    if (hidden_file->f_op && hidden_file->f_op->fsync) {
	down(&hidden_dentry->d_inode->i_sem);
	err = hidden_file->f_op->fsync(hidden_file, hidden_dentry);
	up(&hidden_dentry->d_inode->i_sem);
    }

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_fasync(int fd, file_t *file, int flag)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    if (hidden_file->f_op && hidden_file->f_op->fasync)
	err = hidden_file->f_op->fasync(fd, hidden_file, flag);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_lock(file_t *file, int cmd, struct file_lock *fl)
{
    int err = 0;
    file_t *hidden_file = ftohf(file);

    print_entry_location();

    if (hidden_file->f_op->lock) {
	err = hidden_file->f_op->lock(hidden_file, F_GETLK, fl);
    } else {
	posix_test_lock(hidden_file, fl);
    }

    print_exit_status(err);
    return err;
}


struct file_operations wrapfs_fops =
{
    llseek:	wrapfs_llseek,
#ifdef FIST_FILTER_DATA
    read:	generic_file_read,
    write:	generic_file_write,
#else /* not FIST_FILTER_DATA */
    read:	wrapfs_read,
    write:	wrapfs_write,
#endif /* not FIST_FILTER_DATA */
    readdir:	wrapfs_readdir,
    poll:	wrapfs_poll,
    ioctl:	wrapfs_ioctl,
#ifdef NOT_NEEDED
    mmap:	wrapfs_mmap,
#else /* NOT_NEEDED */
    mmap:	generic_file_mmap,
#endif /* NOT_NEEDED */
    open:	wrapfs_open,
    flush:	wrapfs_flush,
    release:	wrapfs_release,
    fsync:	wrapfs_fsync,
    fasync:	wrapfs_fasync,
    lock:	wrapfs_lock,
    /* not needed: readv */
    /* not needed: writev */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
