/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: inode.c,v 1.57 2005/01/03 21:10:42 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


STATIC int
wrapfs_create(inode_t *dir, dentry_t *dentry, int mode)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    ASSERT(hidden_dentry != NULL);
    fist_checkinode(dir, "wrapfs_create");

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = PTR_ERR(hidden_dir_dentry);
    if (IS_ERR(hidden_dir_dentry))
	goto out;

    err = vfs_create(hidden_dir_dentry->d_inode, hidden_dentry, mode);
    if (err)
	goto out_lock;

    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);

#ifdef FIST_FILTER_SCA
    err = wrapfs_idx_create(hidden_dir_dentry->d_inode, dentry, mode);
    if (err) {
	/*
	 * PROBLEM: we created an empty data file at this point.
	 * should we unlink it?  maybe.  Another solution might be
	 * to change SCA semantics so that a zero-length file need not
	 * have an index file.
	 */
	printk("wrapfs_create: couldn't create idx file %s: %d\n",
	       dtopd(dentry)->idx_dentry->d_name.name, err);
	goto out_lock;
    }
    wrapfs_idx_open(dentry, FMODE_WRITE, O_WRONLY);
#endif /* FIST_FILTER_SCA */

 out_lock:
    unlock_dir(hidden_dir_dentry);
 out:
    fist_checkinode(dir, "post wrapfs_create");
    print_exit_status(err);
    return err;
}


STATIC dentry_t *
wrapfs_lookup(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dir_dentry = wrapfs_hidden_dentry(dentry->d_parent);
    dentry_t *hidden_dentry;
    const char *name = dentry->d_name.name;
    vnode_t *this_vnode;
    dentry_t *this_dir;
#ifdef FIST_FILTER_NAME
    unsigned int namelen = dentry->d_name.len;
    char *encoded_name;
    unsigned int encoded_namelen;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    fist_checkinode(dir, "wrapfs_lookup");

    this_vnode = dir;
    this_dir = hidden_dir_dentry;

    fist_print_dentry("LOOKUP: dentry IN", dentry);
    fist_print_dentry("LOOKUP: dentry->d_parent IN", dentry->d_parent);
    fist_print_dentry("LOOKUP: hidden_dir_dentry IN", hidden_dir_dentry);
    fist_print_inode("LOOKUP: dir IN", dir);

    if (hidden_dir_dentry->d_inode)
	fist_print_inode("LOOKUP: hidden_dir_dentry->d_inode",
			 hidden_dir_dentry->d_inode);

    /* must initialize dentry operations */
    dentry->d_op = &wrapfs_dops;

    FIST_OP_LOOKUP_PRECALL;

    /* increase refcount of base dentry (lookup_one will decrement) */
    // THIS IS RIGHT! (don't "fix" it)
    // NO THIS IS WRONG IN 2.3.99-pre6. lookup_one will NOT decrement
    // dget(hidden_dir_dentry);

#ifndef FIST_FILTER_NAME
    /* will allocate a new hidden dentry if needed */
    hidden_dentry = lookup_one(name, hidden_dir_dentry);
#else /* FIST_FILTER_NAME */
    encoded_namelen = wrapfs_encode_filename(name,
					     namelen,
					     &encoded_name,
					     SKIP_DOTS, dir, dir->i_sb);
    /* will allocate a new hidden dentry if needed */
    hidden_dentry = lookup_one(encoded_name, hidden_dir_dentry);
    kfree_s(encoded_name, encoded_namelen);
#endif /* FIST_FILTER_NAME */

    if (IS_ERR(hidden_dentry)) {
	printk("ERR from hidden_dentry!!!\n");
	err = PTR_ERR(hidden_dentry);
	goto out;
    }

    FIST_OP_LOOKUP_POSTCALL;

    /* update parent directory's atime */
    fist_copy_attr_atime(dir, hidden_dir_dentry->d_inode);
    /* link the upper and lower dentries */
    dtopd(dentry) = (struct wrapfs_dentry_info *) kmalloc(sizeof(struct wrapfs_dentry_info), GFP_KERNEL);
    if (!dtopd(dentry)) {
	err = -ENOMEM;
	goto out_dput;
    }
    dtohd(dentry) = hidden_dentry;

#ifdef FIST_FILTER_SCA
    /*
     * Lookup index file.  Note that we create index files only
     * for real files, but at this point we may not know the type
     * of the to-be created inode.  We release idx_dentry's in d_release.
     */
    if (hidden_dentry->d_inode == NULL ||
	S_ISREG(hidden_dentry->d_inode->i_mode)) {
	dtopd(dentry)->idx_dentry = wrapfs_idx_lookup(hidden_dentry);
	if (IS_ERR(dtopd(dentry)->idx_dentry)) {
	    err = PTR_ERR(dtopd(dentry)->idx_dentry);
	    goto out_free;
	}
	/* index file exists and data file does not, or viceversa */
	if ((dtopd(dentry)->idx_dentry->d_inode == NULL) ^
	    (hidden_dentry->d_inode == NULL)) {
	    err = -ENOENT;
	    goto out_free;
	}
    }
    else
	/* no need for index file */
	dtopd(dentry)->idx_dentry = NULL;
#endif /* FIST_FILTER_SCA */

    /* lookup is special: it needs to handle negative dentries */
    if (!hidden_dentry->d_inode) {
	d_add(dentry, NULL);
	fist_print_dentry("lookup hidden", hidden_dentry);
	goto out;
    }

    fist_dprint(6, "lookup \"%s\" -> inode %d\n", name, hidden_dentry->d_inode->i_ino);
    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 1);
    if (err)
	goto out_free;

    fist_checkinode(dentry->d_inode, "wrapfs_lookup OUT: dentry->d_inode:");
    fist_checkinode(dir, "wrapfs_lookup OUT: dir");

    fist_print_dentry(__FUNCTION__ " OUT hidden_dentry", hidden_dentry);
    fist_print_inode(__FUNCTION__ " OUT hidden_inode", hidden_dentry->d_inode);

#ifdef FIST_FILTER_SCA
    if (S_ISREG(hidden_dentry->d_inode->i_mode)) {
	err = wrapfs_idx_open(dentry, FMODE_READ, O_RDONLY);
	if (err)
	    goto out_free;
	err = wrapfs_idx_read(dentry->d_inode);
	wrapfs_idx_release(dentry->d_inode);
	if (err)
	    goto out_free;
	wrapfs_idx_open(dentry, FMODE_WRITE, O_WRONLY);
    }
#endif /* FIST_FILTER_SCA */

    /* All is well */
    goto out;

 out_free:
    d_drop(dentry);		/* so that our bad dentry will get destroyed */
#ifdef FIST_FILTER_SCA
    if (dtopd(dentry)->idx_dentry)
	dput(dtopd(dentry)->idx_dentry);
#endif /* FIST_FILTER_SCA */
    kfree_s(dtopd(dentry), sizeof(struct wrapfs_dentry_info));
    dtopd(dentry) = NULL;	/* be safe */

 out_dput:
    dput(hidden_dentry);

 out:
    fist_print_dentry("LOOKUP: dentry OUT", dentry);
    print_exit_status(err);
    return ERR_PTR(err);
}


STATIC int
wrapfs_link(dentry_t *old_dentry, inode_t *dir, dentry_t *new_dentry)
{
    int err;
    dentry_t *hidden_old_dentry = wrapfs_hidden_dentry(old_dentry);
    dentry_t *hidden_new_dentry = wrapfs_hidden_dentry(new_dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();

    fist_checkinode(dir, "wrapfs_link-dir");
    fist_checkinode(old_dentry->d_inode, "wrapfs_link-oldinode");

    dget(hidden_old_dentry);
    dget(hidden_new_dentry);
    hidden_dir_dentry = lock_parent(hidden_new_dentry);
    err = -ENOENT;

#ifdef FIST_FILTER_SCA
    /* Also link the index file */
    if (old_dentry->d_inode && S_ISREG(old_dentry->d_inode->i_mode)) {
	/* These must be in the same directory as the main file */
	dentry_t *idx_old_dentry = dtopd(old_dentry)->idx_dentry;
	dentry_t *idx_new_dentry = dtopd(new_dentry)->idx_dentry;

	if (!check_parent(hidden_dir_dentry, idx_new_dentry))
	    goto out_lock;
	err = vfs_link(idx_old_dentry,
		       hidden_dir_dentry->d_inode,
		       idx_new_dentry);
	if (err)
	    goto out_lock;

	fist_copy_attr_timesizes(dir, idx_new_dentry->d_inode);
    }
#endif

    if (!check_parent(hidden_dir_dentry, hidden_new_dentry))
	goto out_lock;

    err = vfs_link(hidden_old_dentry,
		   hidden_dir_dentry->d_inode,
		   hidden_new_dentry);
    if (err)
	goto out_lock;

    err = wrapfs_interpose(hidden_new_dentry, new_dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;

    fist_copy_attr_timesizes(dir, hidden_new_dentry->d_inode);
    /* propagate number of hard-links */
    old_dentry->d_inode->i_nlink = itohi(old_dentry->d_inode)->i_nlink;

 out_lock:
    unlock_dir(hidden_dir_dentry);
    dput(hidden_new_dentry);
    dput(hidden_old_dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_unlink(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    inode_t *hidden_dir = itohi(dir);
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    ASSERT(hidden_dentry != NULL);
    fist_checkinode(dir, "wrapfs_unlink-dir");

    dget(dentry);
    hidden_dir_dentry = lock_parent(hidden_dentry);

#ifdef FIST_FILTER_SCA
    /* first unlink the index file */
    if (dentry->d_inode) {
	if (S_ISREG(dentry->d_inode->i_mode)) {
	    dentry_t *idx_dentry = dtopd(dentry)->idx_dentry;

	    if (!check_parent(hidden_dir_dentry, idx_dentry))
		goto out_index;

	    dget(idx_dentry);
	    err = vfs_unlink(hidden_dir, idx_dentry);
	    dput(idx_dentry);
	    if (err)
		printk("SCA unlink \"%s\" failed with error %d\n", dentry->d_name.name, err);
	    fist_print_dentry("unlink idx_dentry", idx_dentry);
	    if (idx_dentry->d_inode)
		fist_print_inode("unlink idx_inode", idx_dentry->d_inode);
	}
    } else {
	printk("No index file for %s!\n", dentry->d_name.name);
    }
 out_index:
#endif /* FIST_FILTER_SCA */

    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out_lock;

    /* avoid destroying the hidden inode if the file is in use */
    dget(hidden_dentry);
    err = vfs_unlink(hidden_dir, hidden_dentry);
    dput(hidden_dentry);
#if 0
    if (err && err != -ENOENT)
	goto out_lock;
#endif
    if (!err)			/* dput does that */
	d_delete(hidden_dentry);

 out_lock:
    fist_copy_attr_times(dir, hidden_dir);
    /* propagate number of hard-links */
    dentry->d_inode->i_nlink = itohi(dentry->d_inode)->i_nlink;

    unlock_dir(hidden_dir_dentry);

    /*
     * call d_drop so the system "forgets" about us
     */
    if (!err || err == -ENOENT)
	d_drop(dentry);

    dput(dentry);

    fist_checkinode(dir, "post wrapfs_unlink-dir");
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_symlink(inode_t *dir, dentry_t *dentry, const char *symname)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;
#ifdef FIST_FILTER_NAME
    char *encoded_symname;
    unsigned int encoded_symlen;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    fist_checkinode(dir, "wrapfs_symlink-dir");

    dget(hidden_dentry);
    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = -ENOENT;
    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out_lock;

#ifndef FIST_FILTER_NAME
    err = vfs_symlink(hidden_dir_dentry->d_inode,
		      hidden_dentry,
		      symname);
#else /* FIST_FILTER_NAME */
    encoded_symlen = wrapfs_encode_filename(symname,
					    strlen(symname),
					    &encoded_symname,
					    DO_DOTS, dir, dir->i_sb);
    err = vfs_symlink(hidden_dir_dentry->d_inode,
		      hidden_dentry,
		      encoded_symname);
    kfree_s(encoded_symname, encoded_symlen);
#endif /* FIST_FILTER_NAME */
    if (err)
	goto out_lock;
    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    fist_checkinode(dir, "post wrapfs_symlink-dir");

 out_lock:
    unlock_dir(hidden_dir_dentry);
    dput(hidden_dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_mkdir(inode_t *dir, dentry_t *dentry, int mode)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    fist_checkinode(dir, "wrapfs_mkdir-dir");

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = -ENOENT;
    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out;

    err = vfs_mkdir(hidden_dir_dentry->d_inode,
		    hidden_dentry,
		    mode);
    if (err)
	goto out;

    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    /* update number of links on parent directory */
    dir->i_nlink = hidden_dir_dentry->d_inode->i_nlink;

    fist_checkinode(dir, "post wrapfs_mkdir-dir");

 out:
    unlock_dir(hidden_dir_dentry);
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_rmdir(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    fist_checkinode(dir, "wrapfs_rmdir-dir");

    dget(dentry);
    hidden_dir_dentry = lock_parent(hidden_dentry);

    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out_lock;

    /* avoid destroying the hidden inode if the file is in use */
    dget(hidden_dentry);
    err = vfs_rmdir(hidden_dir_dentry->d_inode, hidden_dentry);
    dput(hidden_dentry);
#if 0
    if (err && err != -ENOENT)
	goto out_lock;
#endif
    if (!err)			/* dput does that */
	d_delete(hidden_dentry);

 out_lock:
    fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
    /* copy the nlink count for our dentry and our parent's dentry */
    dir->i_nlink =  hidden_dir_dentry->d_inode->i_nlink;
    /* copy nlink (has to be 0) to ours so iput destroys it */
    dentry->d_inode->i_nlink = 0;

    unlock_dir(hidden_dir_dentry);

    /*
     * call d_drop so the system "forgets" about us
     */
    if (!err || err == -ENOENT)
	d_drop(dentry);

    dput(dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_mknod(inode_t *dir, dentry_t *dentry, int mode, int dev)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    fist_checkinode(dir, "wrapfs_mknod-dir");

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = -ENOENT;
    if (!check_parent(hidden_dir_dentry, hidden_dentry))
	goto out;

    err = vfs_mknod(hidden_dir_dentry->d_inode,
		    hidden_dentry,
		    mode,
		    dev);
    if (err)
	goto out;

    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out;
    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);

 out:
    unlock_dir(hidden_dir_dentry);
    fist_checkinode(dir, "post wrapfs_mknod-dir");
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_rename(inode_t *old_dir, dentry_t *old_dentry,
	      inode_t *new_dir, dentry_t *new_dentry)
{
    int err;
    dentry_t *hidden_old_dentry = wrapfs_hidden_dentry(old_dentry);
    dentry_t *hidden_new_dentry = wrapfs_hidden_dentry(new_dentry);
    dentry_t *hidden_old_dir_dentry;
    dentry_t *hidden_new_dir_dentry;

    print_entry_location();
    fist_checkinode(old_dir, "wrapfs_rename-old_dir");
    fist_checkinode(new_dir, "wrapfs_rename-new_dir");

    dget(hidden_old_dentry);
    dget(hidden_new_dentry);
    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_new_dentry);
    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

    err = -ENOENT;

#ifdef FIST_FILTER_SCA
    /* Also rename the index file */
    if (old_dentry->d_inode && S_ISREG(old_dentry->d_inode->i_mode)) {
	/* These must be in the same directory as the main file */
	dentry_t *idx_old_dentry = dtopd(old_dentry)->idx_dentry;
	dentry_t *idx_new_dentry = dtopd(new_dentry)->idx_dentry;

	if (check_parent(hidden_old_dir_dentry, idx_old_dentry) &&
	    check_parent(hidden_new_dir_dentry, idx_new_dentry))
	    err = vfs_rename(hidden_old_dir_dentry->d_inode, idx_old_dentry,
			     hidden_new_dir_dentry->d_inode, idx_new_dentry);
	if (err)
	    goto out_lock;
    }
#endif

    if (check_parent(hidden_old_dir_dentry, hidden_old_dentry) &&
	check_parent(hidden_new_dir_dentry, hidden_new_dentry))
	err = vfs_rename(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
			 hidden_new_dir_dentry->d_inode, hidden_new_dentry);
    if (err)
	goto out_lock;

    fist_copy_attr_all(new_dir, hidden_new_dir_dentry->d_inode);
    if (new_dir != old_dir)
	fist_copy_attr_all(old_dir, hidden_old_dir_dentry->d_inode);

 out_lock:
    // double_unlock will dput the new/old parent dentries whose refcnts
    // were incremented via get_parent above.
    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
    dput(hidden_new_dentry);
    dput(hidden_old_dentry);

    fist_checkinode(new_dir, "post wrapfs_rename-new_dir");
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_readlink(dentry_t *dentry, char *buf, int bufsiz)
{
    int err;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
#ifdef FIST_FILTER_NAME
    char *decoded_name, *hidden_buf;
    mm_segment_t old_fs;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    fist_print_dentry("wrapfs_readlink IN", dentry);

    if (!hidden_dentry->d_inode->i_op ||
	!hidden_dentry->d_inode->i_op->readlink) {
	err = -EINVAL;
	goto out;
    }

#ifndef FIST_FILTER_NAME
    err = hidden_dentry->d_inode->i_op->readlink(hidden_dentry,
						 buf,
						 bufsiz);
    if (err > 0)
	fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
#else /* FIST_FILTER_NAME */
    hidden_buf = kmalloc(bufsiz, GFP_KERNEL);
    if (hidden_buf == NULL) {
	printk("Out of memory.\n");
	err = -ENOMEM;
	goto out;
    }

    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = hidden_dentry->d_inode->i_op->readlink(hidden_dentry,
						 hidden_buf,
						 bufsiz);
    set_fs(old_fs);
    if (err >= 0) {
	// don't do this, it's not zero-terminated!!!
	//	fist_dprint(7, "READLINK: link \"%s\", length %d\n", hidden_buf, err);
	err = wrapfs_decode_filename(hidden_buf, err,
				     &decoded_name, DO_DOTS, dentry->d_inode, dentry->d_sb);
	if (err > 0) {
	    if (copy_to_user(buf, decoded_name, err))
		err = -EFAULT;
	    kfree_s(decoded_name, err);
	}
	fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
    }

    kfree_s(hidden_buf, bufsiz);
#endif /* FIST_FILTER_NAME */

 out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_follow_link(dentry_t *dentry, struct nameidata *nd)
{
    char *buf;
    int len = PAGE_SIZE, err;
    mm_segment_t old_fs;

    print_entry_location();
    //    fist_print_dentry("wrapfs_follow_link dentry IN", dentry);

    buf = kmalloc(len, GFP_KERNEL);
    if (!buf) {
	err = -ENOMEM;
	goto out;
    }

    /* read the symlink, and then we will follow it */
    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = dentry->d_inode->i_op->readlink(dentry, buf, len);
    set_fs(old_fs);
    if (err < 0)
	goto out_free;

    buf[err] = 0;	// terminate the buffer -- XXX still needed?

    // XXX: FIXME w/ wrapfs_encode_filename()
    /*
     * vfs_follow_link will increase the nd's mnt refcnt
     * we assume that some other VFS code decrements it.
     */
    err = vfs_follow_link(nd, buf);

 out_free:
    kfree_s(buf, len);
 out:
#if 0
    if (err < 0) {
	dput(nd->dentry);
	printk("EZK follow_link() mnt_cnt %d\n", atomic_read(&nd->mnt->mnt_count));
	mntput(nd->mnt);
    }
#endif

    print_exit_status(err);
    return err;
}


#ifdef FIST_FILTER_SCA
STATIC int
wrapfs_truncate_and_enlarge(dentry_t *dentry)
{
    dentry_t *hidden_dentry = dtohd(dentry);
    inode_t *hidden_inode = hidden_dentry->d_inode;
    inode_t *inode = dentry->d_inode;
    int err = 0;
    page_t *page = NULL;
    int index, new_index;
    unsigned offset, new_offset, real_size;
    struct encoded_page *ep = NULL, *next_ep, **cur_ep;
    struct scafs_header *hdr = &itopd(inode)->hdr;
    void *opaque;
    file_t *hidden_file;
    struct iattr attr;
    off_t abs_offset;
    int need_to_call = 1;
    file_t fake_file; /* needed b/c notify change doesn't get a struct file */

    print_entry_location();

    if (dentry->d_inode->i_size == hdr->real_size)
	goto out;
    down(&hidden_inode->i_sem);

    // calculate page index and page offset (offset)
    new_index = inode->i_size >> PAGE_CACHE_SHIFT;
    new_offset = inode->i_size & ~PAGE_CACHE_MASK;
    if (inode->i_size > hdr->real_size) {
	/* enlarge */
	index = hdr->real_size >> PAGE_CACHE_SHIFT;
	offset = hdr->real_size & ~PAGE_CACHE_MASK;
	/* abs_offset is the absolute offset within the hidden_file
	 * where the encoded representation of the current page begins.
	 * Whee... */
	abs_offset = HIDDEN_PAGE_STARTING_OFFSET(index, inode);
    } else {
	/* shrink */
	index = new_index;
	offset = new_offset;
	abs_offset = HIDDEN_PAGE_STARTING_OFFSET(new_index, inode);
    }

    real_size = 0;

    if (offset == 0 || IS_HOLE(index, inode))
	goto page_boundary;

    if (index == new_index)
	offset = new_offset;
    else
	offset = PAGE_CACHE_SIZE;

    /*
     * get our page at offset inode->i_size
     * this is either the page within which we're truncating,
     *		if we are shrinking,
     * or the former last page, if we are enlarging
     */
    /*
     * XXX: set up fake file.  A hack b/c address_ops need a struct file
     * but notify_change only gets a dentry.
     */
    memset(&fake_file, 0, sizeof(file_t));
    // XXX: may need to initialize other fields!
    fake_file.f_dentry = dentry;
    page = wrapfs_get1page(&fake_file, index);
    if (IS_ERR(page)) {
	err = PTR_ERR(page);
	goto out_unlock;
    }

    /* re-encode our page from 0 to "offset" within page */
    kmap(page);
    cur_ep = &ep;
    while (need_to_call) {
	*cur_ep = kmalloc(sizeof(struct encoded_page), GFP_KERNEL);
	(*cur_ep)->data = (char *) __get_free_page(GFP_KERNEL);
#ifdef FIST_FAST_TAILS
	if (CAN_BE_FASTTAIL_PAGE(offset, inode)) {
	    memcpy((*cur_ep)->data, (char *) page_address(page), offset);
	    need_to_call = 0;
	    err = offset;
	    hdr->flags |= SCA_FLAG_FASTTAIL;
	} else
#endif /* FIST_FAST_TAILS */
	{
	    err = wrapfs_encode_buffers((*cur_ep)->data,
					(char *) page_address(page),
					&need_to_call,
					offset,
					inode,
					inode->i_sb,
					&opaque);
	    if (page->index + 1 >= hdr->num_chunks)
		hdr->flags &= ~SCA_FLAG_FASTTAIL;
	}

	cur_ep = &(*cur_ep)->next;
	*cur_ep = NULL;
	if (err < 0)
	    goto out_free;
	real_size += err;
    }
    wrapfs_idx_set_entry(hdr, index, abs_offset + real_size);
    // write new bytes
    dget(hidden_dentry);
    /*
     * dentry_open will decrement mnt refcnt if err.
     * otherwise fput() will do an mntput() for us upon file close.
     */
    mntget(stopd(inode->i_sb)->hidden_mnt);
    hidden_file = dentry_open(hidden_dentry, stopd(inode->i_sb)->hidden_mnt, FMODE_WRITE, O_WRONLY);
    if (IS_ERR(hidden_file)) {
	err = PTR_ERR(hidden_file);
	goto out_free;
    }
    err = wrapfs_copy_encoded_data(hidden_file,
				   abs_offset,
				   ep,
				   real_size);
    fput(hidden_file);
    if (err < 0)
	goto out_free;

 page_boundary:
    index = new_index;
    offset = new_offset;
    abs_offset = HIDDEN_PAGE_STARTING_OFFSET(new_index, inode);

    if (index >= hdr->num_chunks)
	real_size = 0;		/* hole, even though imcomplete */

    // call notify change on lower data file w/ new size
    attr.ia_size = abs_offset + real_size;
    attr.ia_valid = ATTR_SIZE;
    err = notify_change(hidden_dentry, &attr);
    if (err < 0)
	goto out_free;

    // update IT for the new encoded bytes
    hdr->real_size = inode->i_size;
    if (offset == 0) {
	if (index > 0)
	    err = wrapfs_idx_set_entry(hdr, index - 1, attr.ia_size);
	hdr->num_chunks = index;
    } else {
	err = wrapfs_idx_set_entry(hdr, index, attr.ia_size);
	hdr->num_chunks = index + 1;
    }
    if (err < 0)
	goto out_free;
    mark_inode_dirty(inode);	/* so it'll write the index table */

    /* all is ok */
 out_free:
    /* free encoded_pages */
    while (ep) {
	next_ep = ep->next;
	free_page((unsigned long) ep->data);
	kfree_s(ep, sizeof(encoded_pages_t));
	ep = next_ep;
    }
    kunmap(page);
    if (page)
	page_cache_release(page);
 out_unlock:
    up(&hidden_inode->i_sem);
 out:
    print_exit_location();
    return err;
}
#endif /* not FIST_FILTER_SCA */


STATIC int
wrapfs_permission(inode_t *inode, int mask)
{
    inode_t *hidden_inode = itohi(inode);
    int mode = inode->i_mode;
    int err;

    print_entry_location();
    if ((mask & S_IWOTH) && IS_RDONLY(inode) &&
	(S_ISREG(mode) || S_ISDIR(mode) || S_ISLNK(mode))) {
	err = -EROFS; /* Nobody gets write access to a read-only fs */
	goto out;
    }
    err = permission(hidden_inode, mask);
 out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_inode_revalidate(dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    inode_t *hidden_inode = hidden_dentry->d_inode;

    print_entry_location();
    //    fist_print_dentry("inode_revalidate IN", dentry);

    if (hidden_inode->i_op && hidden_inode->i_op->revalidate) {
	err = hidden_inode->i_op->revalidate(hidden_dentry);
	if (!err)
	    fist_copy_attr_all(dentry->d_inode, hidden_inode);
    }

    //    fist_print_dentry("inode_revalidate OUT", dentry);
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_setattr(dentry_t *dentry, struct iattr *ia)
{
    int err = 0;
    dentry_t *hidden_dentry = wrapfs_hidden_dentry(dentry);
    inode_t *inode = dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();
    fist_checkinode(inode, "wrapfs_setattr");

#ifdef FIST_FILTER_SCA
    if (ia->ia_valid & ATTR_SIZE) {
	dentry->d_inode->i_size = ia->ia_size;
	err = wrapfs_truncate_and_enlarge(dentry);
	if (err < 0)
	    goto out;
	ia->ia_valid &= ~ATTR_SIZE; /* b/c we handled it */
    }
#endif /* FIST_FILTER_SCA */
    err = notify_change(hidden_dentry, ia);

#ifdef FIST_FILTER_SCA
 out:
#endif /* FIST_FILTER_SCA */
    /*
     * The lower file system might has changed the attributes, even if
     * notify_change above resulted in an error(!)  so we copy the
     * hidden_inode's attributes (and a few more) to our inode.
     */
    fist_copy_attr_all(inode, hidden_inode);

    fist_checkinode(inode, "post wrapfs_setattr");
    print_exit_status(err);
    return err;
}


#if NOT_USED_YET
STATIC int
wrapfs_getattr(dentry_t *dentry, struct iattr *ia)
{
    return -ENOSYS;
}
#endif /* NOT_USED_YET */


struct inode_operations wrapfs_iops_symlink =
{
    create:	wrapfs_create,
    lookup:	wrapfs_lookup,
    link:	wrapfs_link,
    unlink:	wrapfs_unlink,
    symlink:	wrapfs_symlink,
    mkdir:	wrapfs_mkdir,
    rmdir:	wrapfs_rmdir,
    mknod:	wrapfs_mknod,
    rename:	wrapfs_rename,
    readlink:	wrapfs_readlink,
    follow_link: wrapfs_follow_link,
    // off because we have setattr
    //    truncate:	wrapfs_truncate,
    permission:	wrapfs_permission,
    revalidate:	wrapfs_inode_revalidate,
    setattr:	wrapfs_setattr,
#if 0
    // XXX: off, b/c the VFS doesn't call getattr yet
    getattr:	wrapfs_getattr,
#endif
};

struct inode_operations wrapfs_iops =
{
    create:	wrapfs_create,
    lookup:	wrapfs_lookup,
    link:	wrapfs_link,
    unlink:	wrapfs_unlink,
    symlink:	wrapfs_symlink,
    mkdir:	wrapfs_mkdir,
    rmdir:	wrapfs_rmdir,
    mknod:	wrapfs_mknod,
    rename:	wrapfs_rename,
    //readlink:	wrapfs_readlink,
    //follow_link: wrapfs_follow_link,
    // off because we have setattr
    //    truncate:	wrapfs_truncate,
    permission:	wrapfs_permission,
    revalidate:	wrapfs_inode_revalidate,
    setattr:	wrapfs_setattr,
#if 0
    // XXX: off, b/c the VFS doesn't call getattr yet
    getattr:	wrapfs_getattr,
#endif
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
