/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: main.c,v 1.23 2005/01/03 21:10:42 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <linux/module.h>	/* include only once in a logical module */
#include <linux/init.h>
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


int
wrapfs_interpose(dentry_t *hidden_dentry, dentry_t *dentry, super_block_t *sb, int flag)
{
    inode_t *hidden_inode = hidden_dentry->d_inode;
    int err = 0;
    inode_t *inode;

    print_entry_location();

    ASSERT(hidden_inode != NULL);
    ASSERT(dentry->d_inode == NULL);

    /* check that the lower file system didn't cross a mount point */
    if (hidden_inode->i_dev != stohs(sb)->s_dev) {
	err = -EXDEV;
	goto out;
    }

    /*
     * We allocate our new inode below, by calling iget.
     * iget will call our read_inode which will initialize some
     * of the new inode's fields
     */
    /*
     * XXX: fix assumption that hidden and our inode numbers are same -- maybe
     */
    inode = iget(sb, hidden_inode->i_ino);
    if (!inode) {
	err = -EACCES;		/* should be impossible??? */
	goto out;
    }

    /*
     * interpose the inode if not already interposed
     *   this is possible if the inode is being reused
     */
    if (itohi(inode) == NULL)
	itohi(inode) = igrab(hidden_inode);

    /* XXX Hack hack hack */
    if (S_ISLNK(hidden_inode->i_mode))
	inode->i_op = &wrapfs_iops_symlink;

    /* only (our) lookup wants to do a d_add */
    if (flag)
	d_add(dentry, inode);
    else
	d_instantiate(dentry, inode);

    ASSERT(dtopd(dentry) != NULL);

    /* all well, copy inode attributes */
    fist_copy_attr_all(inode, hidden_inode);

 out:
    print_exit_status(err);
    return err;
}


#ifdef FIST_DEBUG
/* find hidden dentry given this wrapfs dentry */
dentry_t *
__wrapfs_hidden_dentry(char *file, char *func, int line, dentry_t *dentry)
{
    dentry_t *hidden_dentry;

    ASSERT(dentry != NULL);
    ASSERT(dentry->d_op != NULL);
    ASSERT(dentry->d_op == &wrapfs_dops);
    ASSERT(dentry->d_sb->s_op == &wrapfs_sops);
    if (dentry->d_inode)
	ASSERT(dentry->d_inode->i_op == &wrapfs_iops ||
	       dentry->d_inode->i_op == &wrapfs_iops_symlink);

    hidden_dentry = dtohd(dentry);
    ASSERT(hidden_dentry != NULL);
    return hidden_dentry;
}
#endif /* FIST_DEBUG */


STATIC dentry_t *
wrapfs_parse_options(super_block_t *sb, char *options)
{
    dentry_t *hidden_root = ERR_PTR(-EINVAL);
    struct nameidata nd;
    char *name;
    int err = 0;

    print_entry_location();

    while (*options) {
	if (!strncmp("dir=", options, 4)) {
	    fist_dprint(4, "wrapfs: using directory: %s\n", options + 4);
	    /* note: the name passed need not be encoded */
	    name = options + 4;
	    if (path_init(name, LOOKUP_FOLLOW, &nd))
		err = path_walk(name, &nd);
	    if (!err) {
		hidden_root = nd.dentry;
		stopd(sb)->hidden_mnt = nd.mnt;
	    } else
		hidden_root = ERR_PTR(err);
	    fist_dprint(6, "parse_options: new s_root, inode: 0x%x, 0x%x\n",
			(int) hidden_root, (int) hidden_root->d_inode);
	} else {
	    printk(KERN_WARNING "wrapfs: unrecognized options '%s'\n", options);
	}
	while (*options && *options != ',')
	    options++;
    }

    print_exit_location();
    return hidden_root;
}


super_block_t *
wrapfs_read_super(super_block_t *sb, void *raw_data, int silent)
{
    super_block_t *ret_sb = NULL;
    dentry_t *hidden_root;

    print_entry_location();

    if (!raw_data) {
	printk(KERN_WARNING "wrapfs_read_super: missing data argument\n");
	goto out;
    }
    /*
     * Allocate superblock private data
     */
    stopd(sb) = kmalloc(sizeof(struct wrapfs_sb_info), GFP_KERNEL);
    stohs(sb) = NULL;
#ifdef FIST_FILTER_SCA
    stopd(sb)->sca_flags = 0;
#endif /* FIST_FILTER_SCA */

    hidden_root = wrapfs_parse_options(sb, raw_data);
    if (IS_ERR(hidden_root)) {
	printk(KERN_WARNING "wrapfs_read_super: lookup_dentry failed (err = %ld)\n", PTR_ERR(hidden_root));
	goto out_free;
    }
    if (!hidden_root->d_inode) {
	printk(KERN_WARNING "wrapfs_read_super: no directory to interpose on\n");
	goto out_free;
    }
    stohs(sb) = hidden_root->d_sb;

    sb->s_op = &wrapfs_sops;
    /*
     * we can't use d_alloc_root if we want to use
     * our own interpose function unchanged,
     * so we simply replicate *most* of the code in d_alloc_root here
     */
    sb->s_root = d_alloc(NULL, &(const struct qstr) { "/", 1, 0 });
    if (IS_ERR(sb->s_root)) {
	printk(KERN_WARNING "wrapfs_read_super: d_alloc failed\n");
	goto out_dput;
    }

    sb->s_root->d_op = &wrapfs_dops;
    sb->s_root->d_sb = sb;
    sb->s_root->d_parent = sb->s_root;
    /* link the upper and lower dentries */
    dtopd(sb->s_root) = (struct wrapfs_dentry_info *) kmalloc(sizeof(struct wrapfs_dentry_info), GFP_KERNEL);
    if (!dtopd(sb->s_root)) {
	goto out_dput2;
    }
    dtohd(sb->s_root) = hidden_root;
#ifdef FIST_FILTER_SCA
    dtopd(sb->s_root)->idx_dentry = NULL;
#endif /* FIST_FILTER_SCA */

    if (wrapfs_interpose(hidden_root, sb->s_root, sb, 0))
	goto out_dput2;

    ret_sb = sb;
    fist_print_dentry(__FUNCTION__ " OUT hidden_dentry", hidden_root);
    fist_print_inode(__FUNCTION__ " OUT hidden_inode", hidden_root->d_inode);
    // next line causes null ptr deref at mount(2) time
    // fist_print_dentry(__FUNCTION__ "OUT sb->s_root", sb->s_root);
    goto out;

 out_dput2:
    dput(sb->s_root);
 out_dput:
    dput(hidden_root);
 out_free:
    kfree_s(stopd(sb), sizeof(struct wrapfs_sb_info));
    stopd(sb) = NULL;
 out:
    print_exit_location();
    return ret_sb;
}


/*----*/
// this structure *must* be static!!! (or the memory for 'name' gets
// corrupted in 2.3...)
static DECLARE_FSTYPE(wrapfs_fs_type, "wrapfs", wrapfs_read_super, 0);

static int __init init_wrapfs_fs(void)
{
    printk("Registering wrapfs version $Id: main.c,v 1.23 2005/01/03 21:10:42 ezk Exp $\n");
    return register_filesystem(&wrapfs_fs_type);
}
static void __exit exit_wrapfs_fs(void)
{
    printk("Unregistering wrapfs version $Id: main.c,v 1.23 2005/01/03 21:10:42 ezk Exp $\n");
    unregister_filesystem(&wrapfs_fs_type);
}

EXPORT_NO_SYMBOLS;
MODULE_AUTHOR("Erez Zadok <ezk@cs.sunysb.edu>");
module_init(init_wrapfs_fs)
module_exit(exit_wrapfs_fs)

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
