/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: vm_area.c,v 1.11 2005/01/03 21:10:42 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


#ifdef not_needed
STATIC void
wrapfs_vm_open(vm_area_t *vma)
{
    vm_area_t *hidden_vma = vmatohvma(vma), *hidden_vma2;
    file_t *file = vma->vm_file;
    file_t *hidden_file = ftohf(file);

    print_entry_location();
    fist_dprint(6, "VM_OPEN: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) atomic_read(&file->f_count),
		(int) atomic_read(&hidden_file->f_count));

    if (hidden_vma->vm_ops->open)
	hidden_vma->vm_ops->open(hidden_vma);
    atomic_inc(&hidden_file->f_count);

    /* we need to duplicate the private data */
    hidden_vma2 = kmalloc(sizeof(vm_area_t), GFP_KERNEL);
    if (!hidden_vma2) {
	printk("VM_OPEN: Out of memory\n");
	goto out;
    }
    memcpy(hidden_vma2, hidden_vma, sizeof(vm_area_t));
    vmatohvma(vma) = hidden_vma2;

 out:
    fist_dprint(6, "VM_OPEN2: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) atomic_read(&file->f_count),
		(int) atomic_read(&hidden_file->f_count));
    print_exit_location();
}


STATIC void
wrapfs_vm_close(vm_area_t *vma)
{
    vm_area_t *hidden_vma = vmatohvma(vma);
    file_t *file = vma->vm_file;
    file_t *hidden_file = ftohf(file);

    print_entry_location();
    fist_dprint(6, "VM_CLOSE1: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) atomic_read(&file->f_count),
		(int) atomic_read(&hidden_file->f_count));

    ASSERT(hidden_vma != NULL);

    if (hidden_vma->vm_ops->close)
	hidden_vma->vm_ops->close(hidden_vma);
    fput(hidden_file);
    kfree_s(hidden_vma, sizeof(vm_area_t));
    vmatohvma(vma) = NULL;

    fist_dprint(6, "VM_CLOSE2: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) atomic_read(&file->f_count),
		(int) atomic_read(&hidden_file->f_count));
    print_exit_location();
}


STATIC void
wrapfs_vm_shared_unmap(vm_area_t *vma, unsigned long start, size_t len)
{
    vm_area_t *hidden_vma = vmatohvma(vma);
#ifdef FIST_DEBUG
    file_t *file = vma->vm_file;
    file_t *hidden_file = ftohf(file);
#endif /* FIST_DEBUG */

    print_entry_location();
    fist_dprint(6, "VM_S_UNMAP1: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) atomic_read(&file->f_count),
		(int) atomic_read(&hidden_file->f_count));

    /*
     * call sync (filemap_sync) because the default filemap_unmap
     * calls it too.
     */
    filemap_sync(vma, start, len, MS_ASYNC);

    if (hidden_vma->vm_ops->unmap)
	hidden_vma->vm_ops->unmap(hidden_vma, start, len);

    fist_dprint(6, "VM_S_UNMAP2: file 0x%x, hidden_file 0x%x, file->f_count %d, hidden_file->f_count %d\n",
		(int) file, (int) hidden_file,
		(int) atomic_read(&file->f_count),
		(int) atomic_read(&hidden_file->f_count));
    print_exit_location();
}


/*
 * Shared mappings need to be able to do the right thing at
 * close/unmap/sync. They will also use the private file as
 * backing-store for swapping..
 */
struct vm_operations_struct wrapfs_shared_vmops = {
    open:	wrapfs_vm_open,
    close:	wrapfs_vm_close,
    unmap:	wrapfs_vm_shared_unmap,
    sync:	filemap_sync,
    nopage:	filemap_nopage,
    swapout:	filemap_swapout,
};


/*
 * Private mappings just need to be able to load in the map.
 *
 * (This is actually used for shared mappings as well, if we
 * know they can't ever get write permissions..)
 */
struct vm_operations_struct wrapfs_private_vmops = {
    nopage:	filemap_nopage,
};
#endif not_needed

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
