/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: file.c,v 1.54 2005/03/12 04:57:48 satnur Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


/*******************
 * File Operations *
 *******************/


STATIC loff_t
wrapfs_llseek(file_t *file, loff_t offset, int origin)
{
    loff_t err;
    file_t *hidden_file = NULL;

    print_entry_location();
    if (ftopd(file) != NULL) {
	hidden_file = ftohf(file);
    }

    fist_dprint(6, "wrapfs_llseek: file=%p, offset=0%llx, origin=%d\n",
		file, offset, origin);

#ifndef FIST_FILTER_SCA
    ASSERT(hidden_file != NULL);

    /* always set hidden position to this one */
    hidden_file->f_pos = file->f_pos;
    if (file->f_reada) { /* update readahead information if needed */
	hidden_file->f_reada = file->f_reada;
	hidden_file->f_ramax = file->f_ramax;
	hidden_file->f_raend = file->f_raend;
	hidden_file->f_ralen = file->f_ralen;
	hidden_file->f_rawin = file->f_rawin;
    }
    ASSERT(hidden_file->f_reada == file->f_reada);
    ASSERT(hidden_file->f_ramax == file->f_ramax);
    ASSERT(hidden_file->f_raend == file->f_raend);
    ASSERT(hidden_file->f_ralen == file->f_ralen);
    ASSERT(hidden_file->f_rawin == file->f_rawin);

    if (hidden_file->f_op && hidden_file->f_op->llseek)
	err = hidden_file->f_op->llseek(hidden_file, offset, origin);
    else
	err = generic_file_llseek(hidden_file, offset, origin);

    if (err < 0)
	goto out;

    if (err != file->f_pos) {
	file->f_pos = err;
	// ION maybe this?
	// 	file->f_pos = hidden_file->f_pos;
	file->f_reada = 0;
	file->f_version = ++event;
    }
#else /* FIST_FILTER_SCA */
    fist_print_file("wrapfs_llseek IN Virtual file", file);
    fist_print_inode("wrapfs_llseek IN Virtual inode", file->f_dentry->d_inode);
	/*
	 * There is no need to deal with any read ahead information, since
	 * the SCA is not matched with the real read aheads.  Adjust to the
	 * new position
	 */
	switch (origin) {
		case SEEK_END:
			offset += file->f_dentry->d_inode->i_size;
			break;
		case SEEK_CUR:
			offset += file->f_pos;
	}

	err = -EINVAL;

        /*
         * Check file boundaries:
         * offset less than zero is not allowed, however, offset beyond
         * file size is allowed. And it doesnot change the file size
         */
	if (offset >= 0) {
		if (offset != file->f_pos) {
			file->f_pos = offset;
			file->f_reada = 0;
			file->f_version = ++event;
		}
		err = offset;
	}
    fist_print_inode("wrapfs_llseek OUT Virtual inode", file->f_dentry->d_inode);
    fist_print_file("wrapfs_llseek OUT Virtual file", file);
#endif /* FIST_FILTER_SCA */
out:
    print_exit_status((int) err);
    return err;
}


#ifdef FIST_FILTER_DATA
/*
 * generic_file_read updates the atime of upper layer inode.
 * But, it doesn't give us a chance to update the atime of
 * the lower layer inode. This function is a wrapper to
 * generic_file_read. it Updates the atime of the lower level
 * inode if generic_file_read returns without any errors
 * This is to be used only for file reads and only if
 * FIST_FILTER_DATA is defined. The function to be
 * used for directory reads is wrapfs_read
 */
STATIC ssize_t
wrapfs_read_update_atime(file_t *file, char* buf, size_t count, loff_t *ppos)
{
    int err = 0;

    print_entry_location();

    err = generic_file_read(file, buf, count, ppos);

    if (err >= 0)
	UPDATE_ATIME(itohi(file->f_dentry->d_inode));

    print_exit_status(err);
    return err;
}
#endif /* FIST_FILTER_DATA */

STATIC ssize_t
wrapfs_read(file_t *file, char *buf, size_t count, loff_t *ppos)
{
    int err = -EINVAL;
    file_t *hidden_file = NULL;
    loff_t pos = *ppos;

    print_entry_location();
    if (ftopd(file) != NULL)
	hidden_file = ftohf(file);
    ASSERT(hidden_file != NULL);

    if (!hidden_file->f_op || !hidden_file->f_op->read)
	goto out;

    err = hidden_file->f_op->read(hidden_file, buf, count, &pos);

    if (err >= 0) {
	/* atime should also be updated for reads of size zero or more */
	fist_copy_attr_atime(file->f_dentry->d_inode,
			     hidden_file->f_dentry->d_inode);
    }

    // MAJOR HACK
    /*
     * because pread() does not have any way to tell us that it is
     * our caller, then we don't know for sure if we have to update
     * the file positions.  This hack relies on read() having passed us
     * the "real" pointer of its struct file's f_pos field.
     */
    if (ppos == &file->f_pos)
	hidden_file->f_pos = *ppos = pos;
    if (hidden_file->f_reada) { /* update readahead information if needed */
	file->f_reada = hidden_file->f_reada;
	file->f_ramax = hidden_file->f_ramax;
	file->f_raend = hidden_file->f_raend;
	file->f_ralen = hidden_file->f_ralen;
	file->f_rawin = hidden_file->f_rawin;
    }

out:
    print_exit_status(err);
    return err;
}


/* this wrapfs_write() does not modify data pages! */
STATIC ssize_t
wrapfs_write(file_t *file, const char *buf, size_t count, loff_t *ppos)
{
    int err = -EINVAL;
    file_t *hidden_file = NULL;
    inode_t *inode;
    inode_t *hidden_inode;
    loff_t pos = *ppos;

    print_entry_location();
    if (ftopd(file) != NULL)
	hidden_file = ftohf(file);
    ASSERT(hidden_file != NULL);

    inode = file->f_dentry->d_inode;
    hidden_inode = itohi(inode);

    if (!hidden_file->f_op || !hidden_file->f_op->write)
	goto out;

    /* adjust for append -- seek to the end of the file */
    if ((file->f_flags & O_APPEND) && (count != 0))
	pos = inode->i_size;

    if (count != 0)
    	err = hidden_file->f_op->write(hidden_file, buf, count, &pos);
    else
    	err = 0;

    /*
     * copy ctime and mtime from lower layer attributes
     * atime is unchanged for both layers
     */
    if (err >= 0)
	fist_copy_attr_times(inode, hidden_inode);

    /*
     * XXX: MAJOR HACK
     *
     * because pwrite() does not have any way to tell us that it is
     * our caller, then we don't know for sure if we have to update
     * the file positions.  This hack relies on write() having passed us
     * the "real" pointer of its struct file's f_pos field.
     */
    if (ppos == &file->f_pos)
	hidden_file->f_pos = *ppos = pos;

    /* update this inode's size */
    if (pos > inode->i_size)
	inode->i_size = pos;

out:
    print_exit_status(err);
    return err;
}


#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
struct wrapfs_getdents_callback {
    void *dirent;
    dentry_t *dentry;
    filldir_t filldir;
    int err;
    int filldir_called;
    int entries_written;
};

/* copied from generic filldir in fs/readir.c */
STATIC int
# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
wrapfs_filldir(void *dirent, const char *name, int namlen, loff_t offset, ino_t ino, unsigned int d_type)
# else /* kernel older than 2.4.9 */
wrapfs_filldir(void *dirent, const char *name, int namlen, off_t offset, ino_t ino, unsigned int d_type)
# endif /* kernel older than 2.4.9 */
{
    struct wrapfs_getdents_callback *buf = (struct wrapfs_getdents_callback *) dirent;
    int err;
# ifdef FIST_FILTER_NAME
    char *decoded_name;
    int decoded_length;
# endif /* FIST_FILTER_NAME */

    print_entry_location();

    buf->filldir_called++;

# ifdef FIST_FILTER_SCA
    /* skip .idx files */
    if (namlen > INDEX_EXTENSION_LEN-1 &&
	strncmp(&name[namlen-INDEX_EXTENSION_LEN+1],
		INDEX_EXTENSION,
		INDEX_EXTENSION_LEN-1) == 0) {
	//don't do this, it's not zero-terminated!!!
	//fist_dprint(7, "filldir: skip \"%s\"\n", name);
	return 0;
    }
# endif /* FIST_FILTER_SCA */

# ifdef FIST_FILTER_NAME
    decoded_length = wrapfs_decode_filename(name, namlen, &decoded_name, SKIP_DOTS,
					    buf->dentry->d_inode, buf->dentry->d_sb);
    if (decoded_length < 0) {
	return 0;			/* no error, just skip the entry */
    }

    FIST_OP_READDIR_CALL;

    err = buf->filldir(buf->dirent, decoded_name, decoded_length, offset, ino, d_type);
    KFREE(decoded_name);
# else /* not FIST_FILTER_NAME */
    err = buf->filldir(buf->dirent, name, namlen, offset, ino, d_type);
# endif /* not FIST_FILTER_NAME */

    if (err >= 0) {
    	buf->entries_written++;
    }

out:
    print_exit_status(err);
    return err;
}
#endif /* FIST_FILTER_NAME || FIST_FILTER_SCA */


STATIC int
wrapfs_readdir(file_t *file, void *dirent, filldir_t filldir)
{
    int err = -ENOTDIR;
    file_t *hidden_file = NULL;
    inode_t *inode;
#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
    struct wrapfs_getdents_callback buf;
#endif /* FIST_FILTER_NAME || FIST_FILTER_SCA */

    print_entry_location();
    if (ftopd(file) != NULL)
	hidden_file = ftohf(file);
    ASSERT(hidden_file != NULL);

    inode = file->f_dentry->d_inode;

    fist_checkinode(inode, "wrapfs_readdir");
    hidden_file->f_pos = file->f_pos;

#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
    /* prepare for callback */
    buf.dirent = dirent;
    buf.dentry = file->f_dentry;
    buf.filldir = filldir;
retry:
    buf.filldir_called = 0;
    buf.entries_written = 0;
    buf.err = 0;
    err = vfs_readdir(hidden_file, wrapfs_filldir, (void *) &buf);
    if (buf.err) {
	err = buf.err;
    }
    if (buf.filldir_called && !buf.entries_written) {
	goto retry;
    }
#else /* not FIST_FILTER_NAME || FIST_FILTER_SCA */
    err = vfs_readdir(hidden_file, filldir, dirent);
#endif /* not FIST_FILTER_NAME || FIST_FILTER_SCA */

    file->f_pos = hidden_file->f_pos;
    if (err >= 0)
	fist_copy_attr_atime(inode, hidden_file->f_dentry->d_inode);

    fist_checkinode(inode, "post wrapfs_readdir");
    print_exit_status(err);
    return err;
}


STATIC unsigned int
wrapfs_poll(file_t *file, poll_table *wait)
{
    unsigned int mask = DEFAULT_POLLMASK;
    file_t *hidden_file = NULL;

    print_entry_location();
    if (ftopd(file) != NULL)
	hidden_file = ftohf(file);
    ASSERT(hidden_file != NULL);

    if (!hidden_file->f_op || !hidden_file->f_op->poll)
	goto out;

    mask = hidden_file->f_op->poll(hidden_file, wait);

out:
    print_exit_status(mask);
    return mask;
}


STATIC int
wrapfs_ioctl(inode_t *inode, file_t *file, unsigned int cmd, unsigned long arg)
{
    int err = 0;		/* don't fail by default */
    file_t *hidden_file = NULL;
    vfs_t *this_vfs;
    vnode_t *this_vnode;
    int val;
#ifdef FIST_COUNT_WRITES
    extern unsigned long count_writes, count_writes_middle;
#endif /* FIST_COUNT_WRITES */

    print_entry_location();

    this_vfs = inode->i_sb;
    this_vnode = inode;

    /* check if asked for local commands */
    switch (cmd) {
#ifdef FIST_DEBUG
    case FIST_IOCTL_GET_DEBUG_VALUE:
	val = fist_get_debug_value();
	printk("IOCTL GET: send arg %d\n", val);
	err = put_user(val, (int *) arg);
# ifdef FIST_COUNT_WRITES
	printk("COUNT_WRITES:%lu:%lu\n", count_writes, count_writes_middle);
# endif /* FIST_COUNT_WRITES */
	break;

    case FIST_IOCTL_SET_DEBUG_VALUE:
	err = get_user(val, (int *) arg);
	if (err)
	    break;
	fist_dprint(6, "IOCTL SET: got arg %d\n", val);
	if (val < 0 || val > 20) {
	    err = -EINVAL;
	    break;
	}
	fist_set_debug_value(val);
	break;
#endif /* FIST_DEBUG */
#ifdef FIST_FILTER_SCA
    case FIST_IOCTL_GET_FAST_TAIL_VALUE:
	val = (stopd(inode->i_sb)->sca_flags != 0);
	printk("IOCTL GET: send arg %d\n", val);
	err = put_user(val, (int *) arg);
	break;

# ifdef FIST_FAST_TAILS
    case FIST_IOCTL_SET_FAST_TAIL_VALUE:
	err = get_user(val, (int *) arg);
	if (err)
	    break;
	fist_dprint(6, "IOCTL SET: got arg %d\n", val);
	if (val == 1) {
	    stopd(inode->i_sb)->sca_flags |= SCA_FLAG_FASTTAIL;
	} else if (val == 0) {
	    stopd(inode->i_sb)->sca_flags &= ~SCA_FLAG_FASTTAIL;
	} else {
	    err = -EINVAL;
	}
	break;
# endif /* FIST_FAST_TAILS */
#endif /* FIST_FILTER_SCA */

    /* add non-debugging fist ioctl's here */
FIST_IOCTL_ECLS

    default:
        if (ftopd(file) != NULL) {
	    hidden_file = ftohf(file);
	}
	/* pass operation to hidden filesystem, and return status */
	if (hidden_file && hidden_file->f_op && hidden_file->f_op->ioctl)
	    err = hidden_file->f_op->ioctl(itohi(inode), hidden_file, cmd, arg);
	else
	    err = -ENOTTY;	/* this is an unknown ioctl */
    } /* end of outer switch statement */

    print_exit_status(err);
    return err;
}


#ifndef FIST_FILTER_DATA
/* FIST-LITE special version of mmap */
STATIC int
wrapfs_mmap(file_t *file, vm_area_t *vma)
{
    int err = 0;
    file_t *hidden_file = NULL;
    inode_t *inode;
    inode_t *hidden_inode;

    print_entry_location();
    if (ftopd(file) != NULL)
	hidden_file = ftohf(file);

    ASSERT(hidden_file != NULL);

    if (!hidden_file->f_op || !hidden_file->f_op->mmap) {
	err = -ENODEV;
	goto out;
    }	

    ASSERT(hidden_file->f_op != NULL);
    ASSERT(hidden_file->f_op->mmap != NULL);

    vma->vm_file = hidden_file;
    err = hidden_file->f_op->mmap(hidden_file, vma);
    get_file(hidden_file); /* make sure it doesn't get freed on us */
    fput(file);			/* no need to keep extra ref on ours */

 out:
    print_exit_status(err);
    return err;
}
#endif /* not FIST_FILTER_DATA */


STATIC int
wrapfs_open(inode_t *inode, file_t *file)
{
    int err = 0;
    int hidden_mode, hidden_flags;
    file_t *hidden_file = NULL;
    dentry_t *hidden_dentry = NULL;

    print_entry_location();
    ftopd(file) = KMALLOC(sizeof(struct wrapfs_file_info), GFP_KERNEL);
    if (!ftopd(file)) {
	err = -ENOMEM;
	goto out;
    }

    hidden_dentry = wrapfs_hidden_dentry(file->f_dentry); /* CPW: Moved after print_entry */
    fist_print_dentry("wrapfs_open IN hidden_dentry", hidden_dentry);

    dget(hidden_dentry);

    hidden_flags = file->f_flags;
#ifdef FIST_FILTER_DATA
    /* If a file is opened for write only, make read-write. */
    if ((hidden_flags & O_ACCMODE) == O_WRONLY)
	hidden_flags = (hidden_flags & ~O_ACCMODE) | O_RDWR;
    if (file->f_flags & O_APPEND) {
	fist_dprint(5, "file is opened in append-only mode\n");
	hidden_flags &= ~O_APPEND; /* turn off O_APPEND flag */
    }
#endif /* not FIST_FILTER_DATA */

    /*
     * dentry_open will decrement mnt refcnt if err.
     * otherwise fput() will do an mntput() for us upon file close.
     */
    mntget(stopd(inode->i_sb)->hidden_mnt);
    hidden_file = dentry_open(hidden_dentry, stopd(inode->i_sb)->hidden_mnt, hidden_flags);
    if (IS_ERR(hidden_file)) {
	err = PTR_ERR(hidden_file);
	goto out;
    }

    ftohf(file) = hidden_file;	/* link two files */

 out:
    if (err < 0 && ftopd(file)) {
	KFREE(ftopd(file));
    }
    fist_print_dentry("wrapfs_open OUT hidden_dentry", hidden_dentry);
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_flush(file_t *file)
{
    int err = 0;		/* assume ok (see open.c:close_fp) */
    file_t *hidden_file = NULL;

    print_entry_location();

    if (ftopd(file) != NULL)
	hidden_file = ftohf(file); /* CPW: Moved after print_entry_location */
    ASSERT(hidden_file != NULL);

    if (!hidden_file->f_op || !hidden_file->f_op->flush)
	goto out;

    err = hidden_file->f_op->flush(hidden_file);

out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_release(inode_t *inode, file_t *file)
{
    int err = 0;
    file_t *hidden_file = NULL;
    dentry_t *hidden_dentry;
    inode_t *hidden_inode = NULL;

    print_entry_location();

    if (ftopd(file) != NULL)
	hidden_file = ftohf(file);
    KFREE(ftopd(file));

    ASSERT(hidden_file != NULL);

    ASSERT(inode);
    hidden_inode = itohi(inode);
    ASSERT(hidden_inode != NULL);

    fist_print_dentry("wrapfs_release IN hidden_dentry", hidden_file->f_dentry);
    fist_checkinode(inode, "wrapfs_release");
    fist_dprint(6, "wrapfs_release IN, file->f_count=%d\n", atomic_read(&file->f_count));
    /*
     * will decrement file refcount, and if 0, destroy the file,
     * which will call the lower file system's file release function.
     */
    hidden_dentry = hidden_file->f_dentry;
    fput(hidden_file);

#ifndef FIST_FILTER_SCA
   /*
     * Ext2 does something strange: if you compile Ext2 with
     * EXT2_PREALLOCATE, then it will prealocate N more blocks of data to
     * the file in anticipation of the file growing in the near future.  If,
     * however, the file doesn't grow, then ext2 will deallocate these
     * pre-allocated blocks on ext2_release.  However, our stacking template
     * copies the number of blocks in a file in prepare_write and
     * commit_write, at which point the lower file reports more blocks than
     * it would have later on once the file is closed.  This was confirmed
     * by using a C program that runs fsync() on an ext2 file that's open in
     * the middle of the session.  The solution here is to copy the number
     * of blocks of the lower file to the upper file.  (It's possible to
     * copy all attributes, but might be overkill -- I'd rather copy only
     * what we need.)  WARNING: this code is definitely wrong for SCA!
     *   -Erez.
     */
    inode->i_blocks = hidden_inode->i_blocks;
#endif /* not FIST_FILTER_SCA */

    fist_dprint(6, "wrapfs_release done\n");
    fist_checkinode(inode, "post wrapfs_release");

    fist_print_dentry("wrapfs_release OUT hidden_dentry", hidden_dentry);
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_fsync(file_t *file, dentry_t *dentry, int datasync)
{
    int err = -EINVAL;
    file_t *hidden_file = NULL;
    dentry_t *hidden_dentry;

    print_entry_location();

    /* when exporting upper file system through NFS with sync option,
       nfsd_sync_dir() sets struct file as NULL. Use inode's i_fop->fsync instead of file's.
       see fs/nfsd/vfs.c */
    if (file == NULL) {
        hidden_dentry = wrapfs_hidden_dentry(dentry);

        if (hidden_dentry->d_inode->i_fop && hidden_dentry->d_inode->i_fop->fsync) {
            down(&hidden_dentry->d_inode->i_sem);
            err = hidden_dentry->d_inode->i_fop->fsync(hidden_file, hidden_dentry, datasync);
            up(&hidden_dentry->d_inode->i_sem);
        }

    } else {
        if (ftopd(file) != NULL) {
            hidden_file = ftohf(file);
            ASSERT(hidden_file != NULL);

            hidden_dentry = wrapfs_hidden_dentry(dentry);

            if (hidden_file->f_op && hidden_file->f_op->fsync) {
                down(&hidden_dentry->d_inode->i_sem);
                err = hidden_file->f_op->fsync(hidden_file, hidden_dentry, datasync);
                up(&hidden_dentry->d_inode->i_sem);
            }
        }
    }

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_fasync(int fd, file_t *file, int flag)
{
    int err = 0;
    file_t *hidden_file = NULL;

    print_entry_location();
    if (ftopd(file) != NULL)
	hidden_file = ftohf(file);
    ASSERT(hidden_file != NULL);

    if (hidden_file->f_op && hidden_file->f_op->fasync)
	err = hidden_file->f_op->fasync(fd, hidden_file, flag);

    print_exit_status(err);
    return err;
}

STATIC int
wrapfs_getlk(file_t *file, struct file_lock *fl)
{
    int error=0;
    struct file_lock *tempfl=NULL;

    if (file->f_op && file->f_op->lock) {
        error = file->f_op->lock(file, F_GETLK, fl);
        if (error < 0)
            goto out;
        else if (error == LOCK_USE_CLNT)
            /* Bypass for NFS with no locking - 2.0.36 compat */
            tempfl = posix_test_lock(file, fl);
    } else {
        tempfl = posix_test_lock(file, fl);
    }

    if (!tempfl)
        fl->fl_type = F_UNLCK;
    else
        memcpy(fl, tempfl, sizeof(struct file_lock));

out:
    return error;
}

STATIC int
wrapfs_setlk(file_t *file, int cmd, struct file_lock *fl)
{
    int error;
    inode_t *inode;

    error = -EINVAL;
    inode = file->f_dentry->d_inode;

    /* Don't allow mandatory locks on files that may be memory mapped
     * and shared.
     */
    if (IS_MANDLOCK(inode) &&
       (inode->i_mode & (S_ISGID | S_IXGRP)) == S_ISGID) {

        struct address_space *mapping = inode->i_mapping;

        if (mapping->i_mmap_shared != NULL) {
            error = -EAGAIN;
            goto out;
        }
    }

    error = -EBADF;
    switch (fl->fl_type) {
        case F_RDLCK:
            if (!(file->f_mode & FMODE_READ))
                goto out;
            break;
        case F_WRLCK:
            if (!(file->f_mode & FMODE_WRITE))
                goto out;
            break;
        case F_UNLCK:
            break;
        case F_SHLCK:
        case F_EXLCK:
#ifdef __sparc__
/* warn a bit for now, but don't overdo it */
{
        static int count = 0;
        if (!count) {
            count=1;
            printk(KERN_WARNING
                   "fcntl_setlk() called by process %d (%s) with broken flock() emulation\n",
                   current->pid, current->comm);
        }
}
        if (!(file->f_mode & 3))
            goto out;
            break;
#endif
        default:
            error = -EINVAL;
            goto out;
        }

        if (file->f_op && file->f_op->lock != NULL) {
            error = file->f_op->lock(file, cmd, fl);
            if (error < 0)
                goto out;
        }
        error = posix_lock_file(file, fl, cmd == F_SETLKW);

out:
        return error;
}

STATIC int
wrapfs_lock(file_t *file, int cmd, struct file_lock *fl)
{
    int err = 0;
    file_t *hidden_file = NULL;
    struct file_lock *tmpfl = NULL;

    print_entry_location();
    if (ftopd(file) != NULL)
        hidden_file = ftohf(file);
    ASSERT(hidden_file != NULL);

    err = -EINVAL;
    if (!fl)
        goto out;

    fl->fl_file = hidden_file;
    switch(cmd) {
        case F_GETLK:
            err = wrapfs_getlk(hidden_file, fl);
            break;

        case F_SETLK:
        case F_SETLKW:
            err = wrapfs_setlk(hidden_file, cmd, fl);
            break;

        default:
            err = -EINVAL;
    }
    fl->fl_file = file;

out:
    print_exit_status(err);
    return err;
}

struct file_operations wrapfs_dir_fops =
{
    llseek:	wrapfs_llseek,
    read:	wrapfs_read,
    write:	wrapfs_write,
    readdir:	wrapfs_readdir,
    poll:	wrapfs_poll,
    ioctl:	wrapfs_ioctl,
#ifdef FIST_FILTER_DATA
    mmap:	generic_file_mmap,
#else /* not FIST_FILTER_DATA */
    mmap:	wrapfs_mmap,
#endif /* not FIST_FILTER_DATA */
    open:	wrapfs_open,
    flush:	wrapfs_flush,
    release:	wrapfs_release,
    fsync:	wrapfs_fsync,
    fasync:	wrapfs_fasync,
    lock:	wrapfs_lock,
    /* not needed: readv */
    /* not needed: writev */
    /* not implemented: sendpage */
    /* not implemented: get_unmapped_area */
};

struct file_operations wrapfs_main_fops =
{
    llseek:	wrapfs_llseek,
#ifdef FIST_FILTER_DATA
    read:	wrapfs_read_update_atime,
    write:	generic_file_write,
#else /* not FIST_FILTER_DATA */
    read:	wrapfs_read,
    write:	wrapfs_write,
#endif /* not FIST_FILTER_DATA */
    readdir:	wrapfs_readdir,
    poll:	wrapfs_poll,
    ioctl:	wrapfs_ioctl,
#ifdef FIST_FILTER_DATA
    mmap:	generic_file_mmap,
#else /* not FIST_FILTER_DATA */
    mmap:	wrapfs_mmap,
#endif /* not FIST_FILTER_DATA */
    open:	wrapfs_open,
    flush:	wrapfs_flush,
    release:	wrapfs_release,
    fsync:	wrapfs_fsync,
    fasync:	wrapfs_fasync,
    lock:	wrapfs_lock,
    /* not needed: readv */
    /* not needed: writev */
    /* not implemented: sendpage */
    /* not implemented: get_unmapped_area */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
