/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: inode.c,v 1.47 2005/02/04 14:02:07 cwright Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


STATIC int
wrapfs_create(inode_t *dir, dentry_t *dentry, int mode)
{
    int err;
    dentry_t *hidden_dentry;
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    hidden_dentry = wrapfs_hidden_dentry(dentry);
    ASSERT(hidden_dentry != NULL);
    fist_checkinode(dir, "wrapfs_create");

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = PTR_ERR(hidden_dir_dentry);
    if (IS_ERR(hidden_dir_dentry))
	goto out;

    err = vfs_create(hidden_dir_dentry->d_inode, hidden_dentry, mode);
    /* XXX this could potentially return a negative hidden_dentry! */
    if (err)
	goto out_lock;

    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);

#ifdef FIST_FILTER_SCA
    err = wrapfs_idx_create(hidden_dir_dentry->d_inode, dentry, mode);
    if (err) {
	/*
	 * PROBLEM: we created an empty data file at this point.
	 * should we unlink it?  maybe.  Another solution might be
	 * to change SCA semantics so that a zero-length file need not
	 * have an index file.
	 */
	printk("wrapfs_create: couldn't create idx file %s: %d\n",
	       dtopd(dentry)->idx_dentry->d_name.name, err);
	goto out_lock;
    }
    wrapfs_idx_open(dentry, O_WRONLY);
#endif /* FIST_FILTER_SCA */

 out_lock:
    unlock_dir(hidden_dir_dentry);
 out:
    fist_checkinode(dir, "post wrapfs_create");
    print_exit_status(err);
    return err;
}


STATIC dentry_t *
wrapfs_lookup(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dir_dentry;
    dentry_t *hidden_dentry = NULL;
    const char *name;
    vnode_t *this_vnode;
    dentry_t *this_dir;
    unsigned int namelen;
#ifdef FIST_FILTER_NAME
    char *encoded_name;
    int encoded_namelen;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    hidden_dir_dentry = wrapfs_hidden_dentry(dentry->d_parent); /* CPW: Moved below print_entry_location */
    name = dentry->d_name.name;
    namelen = dentry->d_name.len;
    fist_checkinode(dir, "wrapfs_lookup");

    this_vnode = dir;
    this_dir = hidden_dir_dentry;

    fist_print_dentry("wrapfs_lookup IN", dentry);
    fist_print_dentry("wrapfs_lookup: dentry->d_parent IN", dentry->d_parent);
    fist_print_dentry("wrapfs_lookup: hidden_dir_dentry IN", hidden_dir_dentry);
    fist_print_inode("wrapfs_lookup: dir IN", dir);

    if (hidden_dir_dentry->d_inode)
	fist_print_inode("wrapfs_lookup: hidden_dir_dentry->d_inode",
			 hidden_dir_dentry->d_inode);

    /* must initialize dentry operations */
    dentry->d_op = &wrapfs_dops;

    FIST_OP_LOOKUP_PRECALL;

    /* increase refcount of base dentry (lookup_one[_len] will decrement) */
    // THIS IS RIGHT! (don't "fix" it)
    // NO THIS IS WRONG IN 2.3.99-pre6. lookup_one[_len] will NOT decrement
    // dget(hidden_dir_dentry);

#ifndef FIST_FILTER_NAME
    /* will allocate a new hidden dentry if needed */
    hidden_dentry = lookup_one_len(name, hidden_dir_dentry, namelen);
#else /* FIST_FILTER_NAME */
    encoded_namelen = wrapfs_encode_filename(name,
					     namelen,
					     &encoded_name,
					     SKIP_DOTS, dir, dir->i_sb);

    /*if there was an error in encoding filenames*/
    if (encoded_namelen < 0) {
	err = encoded_namelen;
	goto out_free;
    }
    /* will allocate a new hidden dentry if needed */
    hidden_dentry = lookup_one_len(encoded_name, hidden_dir_dentry, encoded_namelen - 1);
    KFREE(encoded_name);
#endif /* FIST_FILTER_NAME */

    if (IS_ERR(hidden_dentry)) {
	/*
	 * this produces an unusual dentry: one that has neither an
	 * inode, nor a private structure attached to it. All cleanup
	 * methods (d_delete, d_release, etc) must be prepared to deal
	 * with such dentries. Ion 09/29/2001
	 */
	printk("ERR from hidden_dentry!!!\n");
	err = PTR_ERR(hidden_dentry);
	goto out;
    }

    FIST_OP_LOOKUP_POSTCALL;

    /* update parent directory's atime */
    fist_copy_attr_atime(dir, hidden_dir_dentry->d_inode);
    /* link the upper and lower dentries */
    dtopd(dentry) = (struct wrapfs_dentry_info *) KMALLOC(sizeof(struct wrapfs_dentry_info), GFP_KERNEL);
    if (!dtopd(dentry)) {
	err = -ENOMEM;
	goto out_dput;
    }
    dtohd(dentry) = hidden_dentry;

#ifdef FIST_FILTER_SCA
    /*
     * Lookup index file.  Note that we create index files only
     * for real files, but at this point we may not know the type
     * of the to-be created inode.  We release idx_dentry's in d_release.
     */
    if (hidden_dentry->d_inode == NULL ||
	S_ISREG(hidden_dentry->d_inode->i_mode)) {
	dtopd(dentry)->idx_dentry = wrapfs_idx_lookup(hidden_dentry);
	if (IS_ERR(dtopd(dentry)->idx_dentry)) {
	    err = PTR_ERR(dtopd(dentry)->idx_dentry);
	    goto out_free;
	}
	/* index file exists and data file does not, or viceversa */
	if ((dtopd(dentry)->idx_dentry->d_inode == NULL) ^
	    (hidden_dentry->d_inode == NULL)) {
	    err = -ENOENT;
	    goto out_free;
	}
    }
    else
	/* no need for index file */
	dtopd(dentry)->idx_dentry = NULL;
#endif /* FIST_FILTER_SCA */

    /* lookup is special: it needs to handle negative dentries */
    if (!hidden_dentry->d_inode) {
	d_add(dentry, NULL);
	fist_print_dentry("wrapfs_lookup OUT hidden_dentry", hidden_dentry);
	goto out;
    }

    fist_dprint(6, "lookup \"%s\" -> inode %lu\n", name, hidden_dentry->d_inode->i_ino);
    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 1);
    if (err)
	goto out_free;

    fist_checkinode(dentry->d_inode, "wrapfs_lookup OUT: dentry->d_inode:");
    fist_checkinode(dir, "wrapfs_lookup OUT: dir");

    fist_print_dentry("wrapfs_lookup OUT hidden_dentry", hidden_dentry);
    fist_print_inode("wrapfs_lookup OUT hidden_inode", hidden_dentry->d_inode);

#ifdef FIST_FILTER_SCA
    if (S_ISREG(hidden_dentry->d_inode->i_mode)) {
	err = wrapfs_idx_open(dentry, O_RDONLY);
	if (err)
	    goto out_free;
	err = wrapfs_idx_read(dentry->d_inode);
	wrapfs_idx_release(dentry->d_inode);
	if (err)
	    goto out_free;
	wrapfs_idx_open(dentry, O_WRONLY);
    }
#endif /* FIST_FILTER_SCA */

    /* All is well */
    goto out;

 out_free:
    d_drop(dentry);		/* so that our bad dentry will get destroyed */
#ifdef FIST_FILTER_SCA
    if (dtopd(dentry)->idx_dentry)
	dput(dtopd(dentry)->idx_dentry);
#endif /* FIST_FILTER_SCA */
    KFREE(dtopd(dentry));
    dtopd(dentry) = NULL;	/* be safe */

 out_dput:
    if (hidden_dentry)
	dput(hidden_dentry);

 out:
    fist_print_dentry("wrapfs_lookup OUT", dentry);
    print_exit_status(err);
    return ERR_PTR(err);
}


STATIC int
wrapfs_link(dentry_t *old_dentry, inode_t *dir, dentry_t *new_dentry)
{
    int err;
    dentry_t *hidden_old_dentry;
    dentry_t *hidden_new_dentry;
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    hidden_old_dentry = wrapfs_hidden_dentry(old_dentry); /* CPW: Moved below print_entry_location */
    hidden_new_dentry = wrapfs_hidden_dentry(new_dentry);

    fist_checkinode(dir, "wrapfs_link-dir");
    fist_checkinode(old_dentry->d_inode, "wrapfs_link-oldinode");

    dget(hidden_old_dentry);
    dget(hidden_new_dentry);
    hidden_dir_dentry = lock_parent(hidden_new_dentry);

#ifdef FIST_FILTER_SCA
    /* Also link the index file */
    if (old_dentry->d_inode && S_ISREG(old_dentry->d_inode->i_mode)) {
	/* These must be in the same directory as the main file */
	dentry_t *idx_old_dentry = dtopd(old_dentry)->idx_dentry;
	dentry_t *idx_new_dentry = dtopd(new_dentry)->idx_dentry;

	err = vfs_link(idx_old_dentry,
		       hidden_dir_dentry->d_inode,
		       idx_new_dentry);
	/* XXX this could potentially return a negative dentry! */
	if (err)
	    goto out_lock;

	fist_copy_attr_timesizes(dir, idx_new_dentry->d_inode);
    }
#endif /* FIST_FILTER_SCA */

    err = vfs_link(hidden_old_dentry,
		   hidden_dir_dentry->d_inode,
		   hidden_new_dentry);
    if (err || !hidden_new_dentry->d_inode)
	goto out_lock;

    err = wrapfs_interpose(hidden_new_dentry, new_dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    /* propagate number of hard-links */
    old_dentry->d_inode->i_nlink = itohi(old_dentry->d_inode)->i_nlink;

 out_lock:
    unlock_dir(hidden_dir_dentry);
    dput(hidden_new_dentry);
    dput(hidden_old_dentry);
    if (!new_dentry->d_inode)
	d_drop(new_dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_unlink(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    inode_t *hidden_dir;
    dentry_t *hidden_dentry;
    dentry_t *hidden_dir_dentry;

    print_entry_location();

    hidden_dir = itohi(dir); /* CPW: Moved below print_entry_location */
    hidden_dentry = wrapfs_hidden_dentry(dentry);

    ASSERT(hidden_dentry != NULL);
    fist_checkinode(dir, "wrapfs_unlink-dir");

    dget(dentry);
    hidden_dir_dentry = lock_parent(hidden_dentry);

#ifdef FIST_FILTER_SCA
    /* first unlink the index file */
    if (dentry->d_inode) {
	if (S_ISREG(dentry->d_inode->i_mode)) {
	    dentry_t *idx_dentry = dtopd(dentry)->idx_dentry;

	    dget(idx_dentry);
	    err = vfs_unlink(hidden_dir, idx_dentry);
	    dput(idx_dentry);
	    if (err)
		printk("SCA unlink \"%s\" failed with error %d\n", dentry->d_name.name, err);
	    fist_print_dentry("wrapfs_unlink: idx_dentry", idx_dentry);
	    if (idx_dentry->d_inode)
		fist_print_inode("wrapfs_unlink: idx_inode", idx_dentry->d_inode);
	}
    } else {
	printk("No index file for %s!\n", dentry->d_name.name);
    }
 out_index:
#endif /* FIST_FILTER_SCA */

    /* avoid destroying the hidden inode if the file is in use */
    dget(hidden_dentry);
    err = vfs_unlink(hidden_dir, hidden_dentry);
    dput(hidden_dentry);

    if (!err)			/* vfs_unlink does that */
	d_delete(hidden_dentry);

 out_lock:
    fist_copy_attr_times(dir, hidden_dir);
    /* propagate number of hard-links */
    dentry->d_inode->i_nlink = itohi(dentry->d_inode)->i_nlink;
    fist_copy_attr_ctime(dentry->d_inode, dir);

    unlock_dir(hidden_dir_dentry);

    /*
     * call d_drop so the system "forgets" about us
     */
    if (!err) {
	//	fist_print_dentry("wrapfs_unlink: 0", dentry);
	d_drop(dentry);
	//	fist_print_dentry("wrapfs_unlink: 1", dentry);
    }

    dput(dentry);

    fist_checkinode(dir, "post wrapfs_unlink-dir");
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_symlink(inode_t *dir, dentry_t *dentry, const char *symname)
{
    int err;
    dentry_t *hidden_dentry;
    dentry_t *hidden_dir_dentry;
#ifdef FIST_FILTER_NAME
    char *encoded_symname;
    int encoded_symlen;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    hidden_dentry = wrapfs_hidden_dentry(dentry); /* CPW: Moved below print_entry_location */
    fist_checkinode(dir, "wrapfs_symlink-dir");

    dget(hidden_dentry);
    hidden_dir_dentry = lock_parent(hidden_dentry);

#ifndef FIST_FILTER_NAME
    err = vfs_symlink(hidden_dir_dentry->d_inode,
		      hidden_dentry,
		      symname);
#else /* FIST_FILTER_NAME */
    encoded_symlen = wrapfs_encode_filename(symname,
					    strlen(symname),
					    &encoded_symname,
					    DO_DOTS, dir, dir->i_sb);
    /* if length is negative, return back */
    if (encoded_symlen < 0) {
	err = encoded_symlen;
	goto out_lock;
    }

    err = vfs_symlink(hidden_dir_dentry->d_inode,
		      hidden_dentry,
		      encoded_symname);
    KFREE(encoded_symname);
#endif /* FIST_FILTER_NAME */
    if (err || !hidden_dentry->d_inode)
	goto out_lock;
    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out_lock;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    fist_checkinode(dir, "post wrapfs_symlink-dir");

 out_lock:
    unlock_dir(hidden_dir_dentry);
    dput(hidden_dentry);
    if (!dentry->d_inode)
	d_drop(dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_mkdir(inode_t *dir, dentry_t *dentry, int mode)
{
    int err;
    dentry_t *hidden_dentry;
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    hidden_dentry = wrapfs_hidden_dentry(dentry); /* CPW: Moved below print_entry_location */
    fist_checkinode(dir, "wrapfs_mkdir-dir");

    hidden_dir_dentry = lock_parent(hidden_dentry);

    err = vfs_mkdir(hidden_dir_dentry->d_inode,
		    hidden_dentry,
		    mode);
    if (err || !hidden_dentry->d_inode)
	goto out;

    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out;

    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    /* update number of links on parent directory */
    dir->i_nlink = hidden_dir_dentry->d_inode->i_nlink;

    fist_checkinode(dir, "post wrapfs_mkdir-dir");

 out:
    unlock_dir(hidden_dir_dentry);
    if (!dentry->d_inode)
	d_drop(dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_rmdir(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dentry;
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    hidden_dentry = wrapfs_hidden_dentry(dentry); /* CPW: Moved below print_entry_location */
    fist_checkinode(dir, "wrapfs_rmdir-dir");

    dget(dentry);
    hidden_dir_dentry = lock_parent(hidden_dentry);

    /* avoid destroying the hidden inode if the file is in use */
    dget(hidden_dentry);
    err = vfs_rmdir(hidden_dir_dentry->d_inode, hidden_dentry);
    dput(hidden_dentry);

    if (!err)			/* vfs_rmdir does that */
	d_delete(hidden_dentry);

 out_lock:
    fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
    /* copy the nlink count for our dentry and our parent's dentry */
    dir->i_nlink =  hidden_dir_dentry->d_inode->i_nlink;

    unlock_dir(hidden_dir_dentry);

    /*
     * call d_drop so the system "forgets" about us
     */
    if (!err)
	d_drop(dentry);

    dput(dentry);

    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_mknod(inode_t *dir, dentry_t *dentry, int mode, int dev)
{
    int err;
    dentry_t *hidden_dentry;
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    hidden_dentry = wrapfs_hidden_dentry(dentry); /* CPW: Moved below print_entry_location */
    fist_checkinode(dir, "wrapfs_mknod-dir");

    hidden_dir_dentry = lock_parent(hidden_dentry);

    err = vfs_mknod(hidden_dir_dentry->d_inode,
		    hidden_dentry,
		    mode,
		    dev);
    if (err || !hidden_dentry->d_inode)
	goto out;

    err = wrapfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (err)
	goto out;
    fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);

 out:
    unlock_dir(hidden_dir_dentry);
    if (!dentry->d_inode)
	d_drop(dentry);

    fist_checkinode(dir, "post wrapfs_mknod-dir");
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_rename(inode_t *old_dir, dentry_t *old_dentry,
	      inode_t *new_dir, dentry_t *new_dentry)
{
    int err;
    dentry_t *hidden_old_dentry;
    dentry_t *hidden_new_dentry;
    dentry_t *hidden_old_dir_dentry;
    dentry_t *hidden_new_dir_dentry;

    print_entry_location();

    hidden_old_dentry = wrapfs_hidden_dentry(old_dentry);/* CPW: Moved below print_entry_location */
    hidden_new_dentry = wrapfs_hidden_dentry(new_dentry);

    fist_checkinode(old_dir, "wrapfs_rename-old_dir");
    fist_checkinode(new_dir, "wrapfs_rename-new_dir");

    dget(hidden_old_dentry);
    dget(hidden_new_dentry);
    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_new_dentry);
    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

#ifdef FIST_FILTER_SCA
    /* Also rename the index file */
    if (old_dentry->d_inode && S_ISREG(old_dentry->d_inode->i_mode)) {
	/* These must be in the same directory as the main file */
	dentry_t *idx_old_dentry = dtopd(old_dentry)->idx_dentry;
	dentry_t *idx_new_dentry = dtopd(new_dentry)->idx_dentry;

	err = vfs_rename(hidden_old_dir_dentry->d_inode, idx_old_dentry,
			 hidden_new_dir_dentry->d_inode, idx_new_dentry);
	if (err)
	    goto out_lock;
    }
#endif /* FIST_FILTER_SCA */

    err = vfs_rename(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
		     hidden_new_dir_dentry->d_inode, hidden_new_dentry);
    if (err)
	goto out_lock;

    fist_copy_attr_all(new_dir, hidden_new_dir_dentry->d_inode);
    if (new_dir != old_dir)
	fist_copy_attr_all(old_dir, hidden_old_dir_dentry->d_inode);

 out_lock:
    // double_unlock will dput the new/old parent dentries whose refcnts
    // were incremented via get_parent above.
    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
    dput(hidden_new_dentry);
    dput(hidden_old_dentry);

    fist_checkinode(new_dir, "post wrapfs_rename-new_dir");
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_readlink(dentry_t *dentry, char *buf, int bufsiz)
{
    int err;
    dentry_t *hidden_dentry;
#ifdef FIST_FILTER_NAME
    char *decoded_name, *hidden_buf;
    mm_segment_t old_fs;
#endif /* FIST_FILTER_NAME */

    print_entry_location();
    hidden_dentry = wrapfs_hidden_dentry(dentry);/* CPW: Moved below print_entry_location */
    fist_print_dentry("wrapfs_readlink IN", dentry);

    if (!hidden_dentry->d_inode->i_op ||
	!hidden_dentry->d_inode->i_op->readlink) {
	err = -EINVAL;
	goto out;
    }

#ifndef FIST_FILTER_NAME
    err = hidden_dentry->d_inode->i_op->readlink(hidden_dentry,
						 buf,
						 bufsiz);
    if (err > 0)
	fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
#else /* FIST_FILTER_NAME */
    hidden_buf = KMALLOC(bufsiz, GFP_KERNEL);
    if (hidden_buf == NULL) {
	printk("Out of memory.\n");
	err = -ENOMEM;
	goto out;
    }

    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = hidden_dentry->d_inode->i_op->readlink(hidden_dentry,
						 hidden_buf,
						 bufsiz);
    set_fs(old_fs);
    if (err >= 0) {
	// don't do this, it's not zero-terminated!!!
	//	fist_dprint(7, "READLINK: link \"%s\", length %d\n", hidden_buf, err);
	err = wrapfs_decode_filename(hidden_buf, err,
				     &decoded_name, DO_DOTS, dentry->d_inode, dentry->d_sb);
	if (err > 0) {
	    if (copy_to_user(buf, decoded_name, err))
		err = -EFAULT;
	    KFREE(decoded_name);
	}
	fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
    }

    KFREE(hidden_buf);
#endif /* FIST_FILTER_NAME */

 out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_follow_link(dentry_t *dentry, struct nameidata *nd)
{
    char *buf;
    int len = PAGE_SIZE, err;
    mm_segment_t old_fs;

    print_entry_location();
    //    fist_print_dentry("wrapfs_follow_link dentry IN", dentry);

    buf = KMALLOC(len, GFP_KERNEL);
    if (!buf) {
	err = -ENOMEM;
	goto out;
    }

    /* read the symlink, and then we will follow it */
    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = dentry->d_inode->i_op->readlink(dentry, buf, len);
    set_fs(old_fs);
    if (err < 0)
	goto out_free;

    buf[err] = 0;	// terminate the buffer -- XXX still needed?

    // XXX: FIXME w/ wrapfs_encode_filename()
    /*
     * vfs_follow_link will increase the nd's mnt refcnt
     * we assume that some other VFS code decrements it.
     */
    err = vfs_follow_link(nd, buf);

 out_free:
    KFREE(buf);
 out:
#if 0
    if (err < 0) {
	dput(nd->dentry);
	printk("EZK follow_link() mnt_cnt %d\n", atomic_read(&nd->mnt->mnt_count));
	mntput(nd->mnt);
    }
#endif

    print_exit_status(err);
    return err;
}


#ifdef FIST_FILTER_SCA
STATIC int
wrapfs_truncate_and_enlarge(dentry_t *dentry, loff_t new_size)
{
    dentry_t *hidden_dentry;
    inode_t *hidden_inode;
    inode_t *inode;
    int err = 0;
    page_t *page = NULL;
    int index, new_index;
    unsigned offset, new_offset, real_size;
    struct encoded_page *ep = NULL, *next_ep, **cur_ep;
    struct scafs_header *hdr;
    void *opaque = 0;
    file_t *hidden_file;
    struct iattr attr;
    off_t abs_offset;
    int need_to_call = 1;
    file_t fake_file; /* needed b/c notify change doesn't get a struct file */

    print_entry_location();

    hidden_dentry = dtohd(dentry);/* CPW: Moved below print_entry_location */
    hidden_inode = hidden_dentry->d_inode;
    inode = dentry->d_inode;
    hdr = &itopd(inode)->hdr;

    down(&hidden_inode->i_sem);

    // calculate page index and page offset (offset)
    new_index = new_size >> PAGE_CACHE_SHIFT;
    new_offset = new_size & ~PAGE_CACHE_MASK;
    if (new_size > hdr->real_size) {
	/* enlarge */
	index = hdr->real_size >> PAGE_CACHE_SHIFT;
	offset = hdr->real_size & ~PAGE_CACHE_MASK;
	/* abs_offset is the absolute offset within the hidden_file
	 * where the encoded representation of the current page begins.
	 * Whee... */
	abs_offset = HIDDEN_PAGE_STARTING_OFFSET(index, inode);
    } else {
	/* shrink */
	index = new_index;
	offset = new_offset;
	abs_offset = HIDDEN_PAGE_STARTING_OFFSET(new_index, inode);
	//truncate_inode_pages(inode->i_mapping, new_size);
    }

    real_size = 0;

    if (offset == 0 || IS_HOLE(index, inode))
	goto page_boundary;

    if (index == new_index)
	offset = new_offset;
    else
	offset = PAGE_CACHE_SIZE;

    /*
     * get our page at offset inode->i_size
     * this is either the page within which we're truncating,
     *		if we are shrinking,
     * or the former last page, if we are enlarging
     */
    /*
     * XXX: set up fake file.  A hack b/c address_ops need a struct file
     * but notify_change only gets a dentry.
     */
    memset(&fake_file, 0, sizeof(file_t));
    // XXX: may need to initialize other fields!
    fake_file.f_dentry = dentry;
    page = wrapfs_get1page(&fake_file, index);
    if (IS_ERR(page)) {
	err = PTR_ERR(page);
	goto out_unlock;
    }

    /* re-encode our page from 0 to "offset" within page */
    kmap(page);
    cur_ep = &ep;
    while (need_to_call) {
	*cur_ep = KMALLOC(sizeof(struct encoded_page), GFP_KERNEL);
	(*cur_ep)->data = (char *) __get_free_page(GFP_KERNEL);
# ifdef FIST_FAST_TAILS
	if (CAN_BE_FASTTAIL_PAGE(offset, inode)) {
	    memcpy((*cur_ep)->data, (char *) page_address(page), offset);
	    need_to_call = 0;
	    err = offset;
	    hdr->flags |= SCA_FLAG_FASTTAIL;
	} else
# endif /* FIST_FAST_TAILS */
	{
	    err = wrapfs_encode_buffers((*cur_ep)->data,
					(char *) page_address(page),
					&need_to_call,
					offset,
					inode,
					inode->i_sb,
					&opaque);
	    if (page->index + 1 >= hdr->num_chunks)
		hdr->flags &= ~SCA_FLAG_FASTTAIL;
	}

	cur_ep = &(*cur_ep)->next;
	*cur_ep = NULL;
	if (err < 0)
	    goto out_free;
	real_size += err;
    }
    wrapfs_idx_set_entry(hdr, index, abs_offset + real_size);
    // write new bytes
    dget(hidden_dentry);
    /*
     * dentry_open will decrement mnt refcnt if err.
     * otherwise fput() will do an mntput() for us upon file close.
     */
    mntget(stopd(inode->i_sb)->hidden_mnt);
    hidden_file = dentry_open(hidden_dentry, stopd(inode->i_sb)->hidden_mnt, O_WRONLY);
    if (IS_ERR(hidden_file)) {
	err = PTR_ERR(hidden_file);
	goto out_free;
    }
    err = wrapfs_copy_encoded_data(hidden_file,
				   abs_offset,
				   ep,
				   real_size);
    fput(hidden_file);
    if (err < 0)
	goto out_free;

 page_boundary:
    index = new_index;
    offset = new_offset;
    abs_offset = HIDDEN_PAGE_STARTING_OFFSET(new_index, inode);

    if (index >= hdr->num_chunks)
	real_size = 0;		/* hole, even though imcomplete */

    // call notify change on lower data file w/ new size
    attr.ia_size = abs_offset + real_size;
    attr.ia_valid = ATTR_SIZE;
    err = notify_change(hidden_dentry, &attr);
    if (err < 0)
	goto out_free;

    // update IT for the new encoded bytes
    hdr->real_size = new_size;
    inode->i_size = new_size;
    if (offset == 0) {
	if (index > 0)
	    err = wrapfs_idx_set_entry(hdr, index - 1, attr.ia_size);
	hdr->num_chunks = index;
    } else {
	err = wrapfs_idx_set_entry(hdr, index, attr.ia_size);
	hdr->num_chunks = index + 1;
    }
    if (err < 0)
	goto out_free;
    mark_inode_dirty(inode);	/* so it'll write the index table */

    /* all is ok */
 out_free:
    /* free encoded_pages */
    while (ep) {
	next_ep = ep->next;
	free_page((unsigned long) ep->data);
	KFREE(ep);
	ep = next_ep;
    }
    kunmap(page);
    if (page)
	page_cache_release(page);
 out_unlock:
    up(&hidden_inode->i_sem);
 out:
    print_exit_location();
    return err;
}

#else /* not FIST_FILTER_SCA */
# ifdef FIST_FILTER_DATA
/*
 * Function to handle truncations extending the size of the file for non-SCA
 * file systems.
 */
STATIC int
wrapfs_truncate(dentry_t *dentry, loff_t new_length)
{
    inode_t *inode = dentry->d_inode;
    dentry_t *hidden_dentry;
    file_t fake_file, *hidden_file = NULL;
    page_t *tmp_page;
    int index;
    unsigned to;
    int err = 0;

    print_entry_location();

    memset(&fake_file, 0, sizeof(fake_file));

    /*
     * if this is a truncate past the end of the file, call wrapfs_fill_zeros
     */
    if (new_length > inode->i_size) {
	fake_file.f_dentry = dentry;
	/* allocate space for private data */
	ftopd(&fake_file) = KMALLOC(sizeof(struct wrapfs_file_info), GFP_KERNEL);
	if (!ftopd(&fake_file)) {
	    err = -ENOMEM;
	    goto out;
	}
	/*
	 * dentry_open will decrement mnt refcnt if err.
	 * otherwise fput() will do an mntput() for us upon file close.
	 */
	mntget(stopd(inode->i_sb)->hidden_mnt);
	hidden_dentry = wrapfs_hidden_dentry(dentry);
	dget(hidden_dentry);
	hidden_file = dentry_open(hidden_dentry, stopd(inode->i_sb)->hidden_mnt, O_RDWR);
	if (IS_ERR(hidden_file)) {
	    err = PTR_ERR(hidden_file);
	    goto out_free;
	}
	ftohf(&fake_file) = hidden_file;	/* link two files */

	index = new_length >> PAGE_CACHE_SHIFT;

	tmp_page = wrapfs_get1page(&fake_file, index);
	if (IS_ERR(tmp_page)) {
	    err = PTR_ERR(tmp_page);
	    goto out_fput;
	}

	to = new_length & ~PAGE_CACHE_MASK;

	/*
	 * 'to' is passed to wrapfs_fill_zeros instead of a 'from'
	 *  because unlike the normal case where we fill in zeros
	 *  till 'from', we need to fill it in with zeros till the
	 *  end of the file
	 */
	err = wrapfs_fill_zeros(&fake_file, tmp_page, to);
    }

 out_fput:
    if (hidden_file)
	filp_close(hidden_file, NULL);
 out_free:
    if (ftopd(&fake_file))
	KFREE(ftopd(&fake_file));
 out:
    print_exit_status(err);
    return err;
}
# endif /* FIST_FILTER_DATA */
#endif /* not FIST_FILTER_SCA */

/* We call only the lower-level permission so that file systems with ACLs
 * or more complex permission models work correctly.  This results in
 * O(n^2) calls to permission with n stacked file systems, because each
 * vfs_foo function first calls permission for the top layer, which in
 * turn makes n calls, then calls vfs_foo on the next layer, which will
 * make n calls, etc.
 *
 * An alternative would be to punt permission checking entirely to the
 * lower-level file system, but then we would need to check that we aren't
 * performing any operations without appropriate permissions checking.
 */
STATIC int
wrapfs_permission(inode_t *inode, int mask)
{
    inode_t *hidden_inode;
    int err;

    print_entry_location();
    hidden_inode = itohi(inode);
    err = permission(hidden_inode, mask);
 out:
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_inode_revalidate(dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dentry;
    inode_t *hidden_inode;

    print_entry_location();
    hidden_dentry = wrapfs_hidden_dentry(dentry);
    hidden_inode = hidden_dentry->d_inode;
    //    fist_print_dentry("wrapfs_inode_revalidate IN", dentry);

    if (hidden_inode->i_op && hidden_inode->i_op->revalidate) {
	err = hidden_inode->i_op->revalidate(hidden_dentry);
	/*
	 * XXX: CPW: I believe that this should happen unconditionally,
	 * whether or not there is a revalidation method, and also
	 * whether or not it fails.  This leads to better consistency
	 * between the upper and lower-level file systems.
	 * EZK: however, if we do that, then we should validate all other
	 * attributes as well.
 	 */
	if (!err) {
	    fist_copy_attr_all(dentry->d_inode, hidden_inode);
	}
    }

    //    fist_print_dentry("wrapfs_inode_revalidate OUT", dentry);
    print_exit_status(err);
    return err;
}


STATIC int
wrapfs_setattr(dentry_t *dentry, struct iattr *ia)
{
    int err = 0;
    dentry_t *hidden_dentry;
    inode_t *inode;
    inode_t *hidden_inode;

    print_entry_location();
    hidden_dentry = wrapfs_hidden_dentry(dentry);
    inode = dentry->d_inode;
    hidden_inode = itohi(inode);
    fist_checkinode(inode, "wrapfs_setattr");

#ifdef FIST_FILTER_SCA
    if (ia->ia_valid & ATTR_SIZE) {
	vmtruncate(dentry->d_inode, ia->ia_size);
	err = wrapfs_truncate_and_enlarge(dentry, ia->ia_size);
	if (err < 0)
	    goto out;
	ia->ia_valid &= ~ATTR_SIZE; /* b/c we handled it */
    }
#else /* not FIST_FILTER_SCA */
# ifdef FIST_FILTER_DATA
    if (ia->ia_valid & ATTR_SIZE) {
	err = wrapfs_truncate(dentry, ia->ia_size);
	if (err < 0)
	    goto out;
    }
# endif /* FIST_FILTER_DATA */
#endif /* not FIST_FILTER_SCA */
    err = notify_change(hidden_dentry, ia);

#if defined(FIST_FILTER_DATA) || defined(FIST_FILTER_SCA)
 out:
#endif /* FIST_FILTER_DATA || FIST_FILTER_SCA */
    /*
     * The lower file system might has changed the attributes, even if
     * notify_change above resulted in an error(!)  so we copy the
     * hidden_inode's attributes (and a few more) to our inode.
     */
    fist_copy_attr_all(inode, hidden_inode);

    fist_checkinode(inode, "post wrapfs_setattr");
    print_exit_status(err);
    return err;
}


#ifdef NOT_USED_YET
STATIC int
wrapfs_getattr(dentry_t *dentry, struct iattr *ia)
{
    return -ENOSYS;
}
#endif /* NOT_USED_YET */


#ifndef FIST_FILTER_SCA
# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20)
/* This is lifted from fs/xattr.c */
static void *
xattr_alloc(size_t size, size_t limit)
{
    void *ptr;

    if (size > limit)
	return ERR_PTR(-E2BIG);

    if (!size)			/* size request, no buffer is needed */
	return NULL;
    else if (size <= PAGE_SIZE)
	ptr = KMALLOC((unsigned long) size, GFP_KERNEL);
    else
	ptr = vmalloc((unsigned long) size);
    if (!ptr)
	return ERR_PTR(-ENOMEM);
    return ptr;
}

static void
xattr_free(void *ptr, size_t size)
{
    if (!size)			/* size request, no buffer was needed */
	return;
    else if (size <= PAGE_SIZE)
	KFREE(ptr);
    else
	vfree(ptr);
}

/*
 * BKL held by caller.
 * dentry->d_inode->i_sem down
 */
STATIC int
wrapfs_getxattr(struct dentry *dentry, const char *name, void *value, size_t size)
{
    struct dentry *hidden_dentry = NULL;
    int err = -ENOTSUPP;
#  ifdef FIST_FILTER_NAME
    char *encoded_suffix = NULL;
    int namelen = 0;
    int sufflen = 0;
    int prelen = 0;
#  endif /* FIST_FILTER_NAME */
    /* Define these anyway so we don't need as much ifdef'ed code. */
    char *encoded_name = NULL;
    char *encoded_value = NULL;

    print_entry_location();

    hidden_dentry = dtohd(dentry);

    ASSERT(hidden_dentry);
    ASSERT(hidden_dentry->d_inode);
    ASSERT(hidden_dentry->d_inode->i_op);

    fist_dprint(18, "getxattr: name=\"%s\", value %d bytes\n", name, size);

    if (hidden_dentry->d_inode->i_op->getxattr) {
#  ifdef FIST_FILTER_NAME
	const char *suffix;

	namelen = strlen(name);

	suffix = strchr(name, '.');
	if (suffix) {
	    suffix ++;
	    prelen = (suffix - name) - 1;
	    sufflen = namelen - (prelen + 1);
	} else {
	    prelen = 0;
	    sufflen = namelen;
	    suffix = name;
	}

	sufflen = wrapfs_encode_filename(suffix, sufflen, &encoded_suffix, DO_DOTS, dentry->d_inode, dentry->d_inode->i_sb) - 1;
	encoded_name = xattr_alloc(prelen + sufflen + 2, XATTR_NAME_MAX);
	if (IS_ERR(encoded_name)) {
	    err = PTR_ERR(encoded_name);
	    encoded_name = NULL;
	    goto out;
	}
	if (prelen) {
	    memcpy(encoded_name, name, prelen);
	    encoded_name[prelen] = '.';
	    memcpy(encoded_name + prelen + 1, encoded_suffix, sufflen);
	    encoded_name[prelen + 1 + sufflen] = '\0';
	} else {
	    memcpy(encoded_name + prelen, encoded_suffix, sufflen);
	    encoded_name[sufflen] = '\0';
	}

	fist_dprint(18, "ENCODED XATTR NAME: %s\n", encoded_name);
#  else /* not FIST_FILTER_NAME */
	encoded_name = (char *)name;
#  endif /* not FIST_FILTER_NAME */

#  ifdef FIST_FILTER_DATA
	encoded_value = xattr_alloc(size, XATTR_SIZE_MAX);
	if (IS_ERR(encoded_value)) {
	    err = PTR_ERR(encoded_value);
	    encoded_name = NULL;
	    goto out;
	}
#  else /* not FIST_FILTER_DATA */
	encoded_value = (char *)value;
#  endif /* not FIST_FILTER_DATA */


	down(&hidden_dentry->d_inode->i_sem);
	/* lock_kernel() already done by caller. */
	err = hidden_dentry->d_inode->i_op->getxattr(hidden_dentry, encoded_name, encoded_value, size);
	/* unlock_kernel() will be done by caller. */
	up(&hidden_dentry->d_inode->i_sem);

#  ifdef FIST_FILTER_DATA
	/* Decode the value. */
	if ((err > 0) && size) {
	    int ret;

	    ret = wrapfs_decode_block(encoded_value, value, err, dentry->d_inode, dentry->d_inode->i_sb, 0);
	    if (ret < 0) {
		err = ret;
		goto out;
	    }
	}
#  endif /* FIST_FILTER_DATA */
    }

 out:
#  ifdef FIST_FILTER_NAME
    if (encoded_name) {
	xattr_free(encoded_name, namelen + 1);
    }
#  endif /* not FIST_FILTER_NAME */
#  ifdef FIST_FILTER_DATA
    if (encoded_value) {
	xattr_free(encoded_value, size);
    }
#  endif /* not FIST_FILTER_DATA */
    print_exit_status(err);
    return err;
}

/*
 * BKL held by caller.
 * dentry->d_inode->i_sem down
 */
STATIC int
/*
 * This is the prototype used by 2.4.20--22, but the extended attributes
 * patch to 2.4.21 from Andreas Gruenbacher uses a const void * for the value.
 * If you want to actually use the EA's you will need to apply that patch,
 * then fix this definition.
 */
#  ifndef FIST_SETXATTR_CONSTVOID
wrapfs_setxattr(struct dentry *dentry, const char *name, void *value, size_t size, int flags)
#  else /* not FIST_SETXATTR_CONSTVOID */
wrapfs_setxattr(struct dentry *dentry, const char *name, const void *value, size_t size, int flags)
#  endif /* FIST_SETXATTR_CONSTVOID */
{
    struct dentry *hidden_dentry = NULL;
    int err = -ENOTSUPP;
#  ifdef FIST_FILTER_NAME
    char *encoded_suffix = NULL;
    int namelen = 0;
    int sufflen = 0;
    int prelen = 0;
#  endif /* FIST_FILTER_NAME */
    /* Define these anyway, so we don't have as much ifdef'ed code. */
    char *encoded_value = NULL;
    char *encoded_name = NULL;

    print_entry_location();

    hidden_dentry = dtohd(dentry);

    ASSERT(hidden_dentry);
    ASSERT(hidden_dentry->d_inode);
    ASSERT(hidden_dentry->d_inode->i_op);

    fist_dprint(18, "setxattr: name=\"%s\", value %d bytes, flags=%x\n", name, size, flags);

    if (hidden_dentry->d_inode->i_op->setxattr) {
#  ifdef FIST_FILTER_NAME
	const char *suffix;

	namelen = strlen(name);

	suffix = strchr(name, '.');
	if (suffix) {
	    suffix ++;
	    prelen = (suffix - name) - 1;
	    sufflen = namelen - (prelen + 1);
	} else {
	    prelen = 0;
	    sufflen = namelen;
	    suffix = name;
	}

	sufflen = wrapfs_encode_filename(suffix, sufflen, &encoded_suffix, DO_DOTS, dentry->d_inode, dentry->d_inode->i_sb) - 1;
	encoded_name = xattr_alloc(prelen + sufflen + 2, XATTR_NAME_MAX);
	if (IS_ERR(encoded_name)) {
	    err = PTR_ERR(encoded_name);
	    encoded_name = NULL;
	    goto out;
	}
	if (prelen) {
	    memcpy(encoded_name, name, prelen);
	    encoded_name[prelen] = '.';
	    memcpy(encoded_name + prelen + 1, encoded_suffix, sufflen);
	    encoded_name[prelen + 1 + sufflen] = '\0';
	} else {
	    memcpy(encoded_name + prelen, encoded_suffix, sufflen);
	    encoded_name[sufflen] = '\0';
	}

	fist_dprint(18, "ENCODED XATTR NAME: %s\n", encoded_name);
#  else /* not FIST_FILTER_NAME */
	encoded_name = (char *)name;
#  endif /* not FIST_FILTER_NAME */

#  ifdef FIST_FILTER_DATA
	/* Encode the value. */
	encoded_value = xattr_alloc(size, XATTR_SIZE_MAX);
	if (IS_ERR(encoded_value)) {
	    err = PTR_ERR(encoded_value);
	    encoded_value = NULL;
	    goto out;
	}

	err = wrapfs_encode_block(value, encoded_value, size, dentry->d_inode, dentry->d_inode->i_sb, 0);
	if (err < 0) {
	    goto out;
	}
#  else /* not FIST_FILTER_DATA */
	encoded_value = (char *)value;
#  endif /* not FIST_FILTER_DATA */

	down(&hidden_dentry->d_inode->i_sem);
	/* lock_kernel() already done by caller. */
	err = hidden_dentry->d_inode->i_op->setxattr(hidden_dentry, encoded_name, encoded_value, size, flags);
	/* unlock_kernel() will be done by caller. */
	up(&hidden_dentry->d_inode->i_sem);
    }

 out:
#  ifdef FIST_FILTER_NAME
    if (encoded_name) {
	xattr_free(encoded_name, namelen + 1);
    }
    if (encoded_suffix) {
	KFREE(encoded_suffix);
    }
#  endif /* FIST_FILTER_NAME */
#  ifdef FIST_FILTER_DATA
    if (encoded_value) {
	xattr_free(encoded_value, size);
    }
#  endif /* FIST_FILTER_DATA */
    print_exit_status(err);
    return err;
}

/*
 * BKL held by caller.
 * dentry->d_inode->i_sem down
 */
STATIC int
wrapfs_removexattr(struct dentry *dentry, const char *name)
{
    struct dentry *hidden_dentry = NULL;
    int err = -ENOTSUPP;
    char *encoded_name;
#  ifdef FIST_FILTER_NAME
    char *encoded_suffix = NULL;
    int namelen = 0;
    int sufflen = 0;
    int prelen = 0;
#  endif /* FIST_FILTER_NAME */
    print_entry_location();

    hidden_dentry = dtohd(dentry);

    ASSERT(hidden_dentry);
    ASSERT(hidden_dentry->d_inode);
    ASSERT(hidden_dentry->d_inode->i_op);

    fist_dprint(18, "removexattr: name=\"%s\"\n", name);

    if (hidden_dentry->d_inode->i_op->removexattr) {
#  ifdef FIST_FILTER_NAME
	const char *suffix;

	namelen = strlen(name);

	suffix = strchr(name, '.');
	if (suffix) {
	    suffix ++;
	    prelen = (suffix - name) - 1;
	    sufflen = namelen - (prelen + 1);
	} else {
	    prelen = 0;
	    sufflen = namelen;
	    suffix = name;
	}

	sufflen = wrapfs_encode_filename(suffix, sufflen, &encoded_suffix, DO_DOTS, dentry->d_inode, dentry->d_inode->i_sb) - 1;
	encoded_name = xattr_alloc(prelen + sufflen + 2, XATTR_NAME_MAX);
	if (IS_ERR(encoded_name)) {
	    err = PTR_ERR(encoded_name);
	    encoded_name = NULL;
	    goto out;
	}
	if (prelen) {
	    memcpy(encoded_name, name, prelen);
	    encoded_name[prelen] = '.';
	    memcpy(encoded_name + prelen + 1, encoded_suffix, sufflen);
	    encoded_name[prelen + 1 + sufflen] = '\0';
	} else {
	    memcpy(encoded_name + prelen, encoded_suffix, sufflen);
	    encoded_name[sufflen] = '\0';
	}

	fist_dprint(18, "ENCODED XATTR NAME: %s\n", encoded_name);
#  else /* not FIST_FILTER_NAME */
	encoded_name = (char *)name;
#  endif /* not FIST_FILTER_NAME */

	down(&hidden_dentry->d_inode->i_sem);
	/* lock_kernel() already done by caller. */
	err = hidden_dentry->d_inode->i_op->removexattr(hidden_dentry, encoded_name);
	/* unlock_kernel() will be done by caller. */
	up(&hidden_dentry->d_inode->i_sem);
    }

 out:
    print_exit_status(err);
    return err;
}

/*
 * BKL held by caller.
 * dentry->d_inode->i_sem down
 */
STATIC int
wrapfs_listxattr(struct dentry *dentry, char *list, size_t size)
{
    struct dentry *hidden_dentry = NULL;
    int err = -ENOTSUPP;
    char *encoded_list = NULL;

    print_entry_location();

    hidden_dentry = dtohd(dentry);

    ASSERT(hidden_dentry);
    ASSERT(hidden_dentry->d_inode);
    ASSERT(hidden_dentry->d_inode->i_op);

    if (hidden_dentry->d_inode->i_op->listxattr) {
#  ifdef FIST_FILTER_NAME
	encoded_list = xattr_alloc(size, XATTR_LIST_MAX);
	if (IS_ERR(encoded_list)) {
	    err = PTR_ERR(encoded_list);
	    encoded_list = NULL;
	    goto out;
	}
#  else /* not FIST_FILTER_NAME */
	encoded_list = list;
#  endif /* not FIST_FILTER_NAME */
	down(&hidden_dentry->d_inode->i_sem);
	/* lock_kernel() already done by caller. */
	err = hidden_dentry->d_inode->i_op->listxattr(hidden_dentry, encoded_list, size);
	/* unlock_kernel() will be done by caller. */
	up(&hidden_dentry->d_inode->i_sem);
#  ifdef FIST_FILTER_NAME
	/* The problem is we don't know how big the xattrs
	 * will be without decoding them ourselves.  So,
	 * we just have to copy it attribute by attribute
	 * to figure out.
	 *
	 * XXX: If the buffer was set to zero then we
	 * actually end up lying and passing through the
	 * error.  For some transformations this works, but
	 * you may need to change it (e.g., using an SCA
	 * in the name ala cryptfs).
	 */
	if (encoded_list) {
	    int i;
	    int newsize = 0;
	    char *cur = encoded_list;
	    char *encoded_suffix;

	    while (cur && (cur < (encoded_list + err))) {
		const char *suffix;
		int namelen, sufflen, prelen;

		ASSERT(cur);
		printk("cur: %s\n", cur);

		namelen = strlen(cur);

		suffix = strchr(cur, '.');
		if (suffix) {
		    suffix ++;
		    prelen = (suffix - cur) - 1;
		    sufflen = namelen - (prelen + 1);
		} else {
		    prelen = 0;
		    sufflen = namelen;
		    suffix = cur;
		}

		sufflen = wrapfs_decode_filename(suffix, sufflen, &encoded_suffix, DO_DOTS, dentry->d_inode, dentry->d_inode->i_sb);

		if (prelen) {
		    if (newsize + sufflen + prelen + 2 < size) {
			/* COPY IT IN */
			memcpy(list + newsize, cur, prelen);
			list[newsize + prelen] = '.';
			memcpy(list + newsize + prelen + 1, encoded_suffix, sufflen);
			list[newsize + prelen + 1 + sufflen] = '\0';
		    }
		    newsize += sufflen + prelen + 2;
		} else {
		    if (newsize + sufflen + 1 < size) {
			memcpy(list + newsize, encoded_suffix, sufflen);
			list[newsize + sufflen] = '\0';
		    }
		    newsize += sufflen + 1;
		}

		cur += (namelen + 1);
	    }

	    err = newsize;
	}
#  endif /* FIST_FILTER_NAME */
    }

 out:
#  ifdef FIST_FILTER_NAME
    if (encoded_list) {
	xattr_free(encoded_list, size);
    }
#  endif /* FIST_FILTER_NAME */
    print_exit_status(err);
    return err;
}
# endif /* LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20) */

#endif /* not FIST_FILTER_SCA */

struct inode_operations wrapfs_symlink_iops =
{
    readlink:	wrapfs_readlink,
    follow_link: wrapfs_follow_link,
    permission:	wrapfs_permission,
    revalidate:	wrapfs_inode_revalidate,
    setattr:	wrapfs_setattr,
#if 0
    // XXX: off, b/c the VFS doesn't call getattr yet
    getattr:	wrapfs_getattr,
#endif

#ifndef FIST_FILTER_SCA
# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20)
    setxattr:	wrapfs_setxattr,
    getxattr:	wrapfs_getxattr,
    listxattr:	wrapfs_listxattr,
    removexattr: wrapfs_removexattr
# endif /* LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20) */
#endif /* not FIST_FILTER_SCA */
};

struct inode_operations wrapfs_dir_iops =
{
    create:	wrapfs_create,
    lookup:	wrapfs_lookup,
    link:	wrapfs_link,
    unlink:	wrapfs_unlink,
    symlink:	wrapfs_symlink,
    mkdir:	wrapfs_mkdir,
    rmdir:	wrapfs_rmdir,
    mknod:	wrapfs_mknod,
    rename:	wrapfs_rename,
    /* no readlink/follow_link for non-symlinks */
    // off because we have setattr
    //    truncate:	wrapfs_truncate,
    permission:	wrapfs_permission,
    revalidate:	wrapfs_inode_revalidate,
    setattr:	wrapfs_setattr,
#if 0
    // XXX: off, b/c the VFS doesn't call getattr yet
    getattr:	wrapfs_getattr,
#endif
#ifndef FIST_FILTER_SCA
# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20)
    setxattr:	wrapfs_setxattr,
    getxattr:	wrapfs_getxattr,
    listxattr:	wrapfs_listxattr,
    removexattr: wrapfs_removexattr
# endif /* LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20) */
#endif /* not FIST_FILTER_SCA */
};

struct inode_operations wrapfs_main_iops =
{
    permission:	wrapfs_permission,
    revalidate:	wrapfs_inode_revalidate,
    setattr:	wrapfs_setattr,
#if 0
    // XXX: off, b/c the VFS doesn't call getattr yet
    getattr:	wrapfs_getattr,
#endif
#ifndef FIST_FILTER_SCA
# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20)
    setxattr:	wrapfs_setxattr,
    getxattr:	wrapfs_getxattr,
    listxattr:	wrapfs_listxattr,
    removexattr: wrapfs_removexattr
# endif /* LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20) */
#endif /* not FIST_FILTER_SCA */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
