/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: dentry.c,v 1.11 2005/01/03 21:10:42 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
# ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"

/* HL
   d_revalidate: check for lower_dentry!=NULL


*/

/*
 * THIS IS A BOOLEAN FUNCTION: returns 1 if valid, 0 otherwise.
 */
STATIC int
wrapfs_d_revalidate(dentry_t *dentry, struct nameidata *nd)
{
	int err = 1; // default is valid (1); invalid is 0.
	dentry_t *lower_dentry;

	print_entry_location();

	lower_dentry  = wrapfs_lower_dentry(dentry); /* CPW: Moved to after print_entry_location */

	if (!lower_dentry || !lower_dentry->d_op || !lower_dentry->d_op->d_revalidate)
		goto out;

	err = lower_dentry->d_op->d_revalidate(lower_dentry, nd);

out:
	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_d_hash(dentry_t *dentry, qstr_t *name)
{
	int err = 0;
	dentry_t *lower_dentry;

	print_entry_location();
	lower_dentry = wrapfs_lower_dentry(dentry);	/* CPW: Moved to after print_entry_location */

	if (!lower_dentry || !lower_dentry->d_op || !lower_dentry->d_op->d_hash)
		goto out;

	err = lower_dentry->d_op->d_hash(lower_dentry, name);

out:
	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_d_compare(dentry_t *dentry, qstr_t *a, qstr_t *b)
{
	int err;
	dentry_t *lower_dentry;

	print_entry_location();

	lower_dentry = wrapfs_lower_dentry(dentry);	/* CPW: Moved to after print_entry_location */

	if (lower_dentry && lower_dentry->d_op && lower_dentry->d_op->d_compare) {
		// XXX: WRONG: should encode_filename on a&b strings
		err = lower_dentry->d_op->d_compare(lower_dentry, a, b);
	} else {
		err = ((a->len != b->len) || memcmp(a->name, b->name, b->len));
	}

	print_exit_status(err);
	return err;
}


int
wrapfs_d_delete(dentry_t *dentry)
{

	dentry_t *lower_dentry = 0;
	int err = 0;

	print_entry_location();
/*
  EZKDBG;

  if (!DENTRY_TO_PRIVATE(dentry)) goto out;

  EZKDBG;

  lower_dentry = DENTRY_TO_LOWER(dentry);

  EZKDBG;

  if (!lower_dentry) goto out;
  fist_dprint(8, "lower_dentry: %p\n", lower_dentry);

  EZKDBG;

  if (!lower_dentry->d_op) goto out;

  EZKDBG;

  if (!lower_dentry->d_op->d_delete) goto out;

  EZKDBG;

  err = lower_dentry->d_op->d_delete(lower_dentry);

  EZKDBG;

  // skip everything else!
  goto out;
*/

        /* dentry can be NULL */
        if (!dentry)
                goto out;

	/* this could be a negative dentry, so check first */
	if (!DENTRY_TO_PRIVATE(dentry)) {
		fist_dprint(6, "dentry without private data: %*s", dentry->d_name.len, dentry->d_name.name);
		//fist_dprint(6, "dentry without private data");
		goto out;
	}

	if (!(lower_dentry = DENTRY_TO_LOWER(dentry))) {
		fist_dprint(6, "dentry without lower_dentry: %*s", dentry->d_name.len, dentry->d_name.name);
		//fist_dprint(6, "dentry without lower_dentry");
		goto out;
	}

	fist_print_dentry("D_DELETE IN", dentry);
        fist_dprint(8, "before lower d_delete: %p, %p, %p\n", lower_dentry, lower_dentry ? lower_dentry->d_op : 0, lower_dentry->d_op ? lower_dentry->d_op->d_delete : 0);
	/* added b/c of changes to dput(): it calls d_drop on us */
	if (lower_dentry && lower_dentry->d_op &&
            lower_dentry->d_op->d_delete) {
                fist_dprint(8, "lower d_delete\n");
                /*
                   XXX: HL: can we just call d_delete? B/c in dput happens
                   much more depending on its outcome...
                */
		err = lower_dentry->d_op->d_delete(lower_dentry);
	}

out:
	//fist_dprint(8, "exit d_delete\n");
	print_exit_status(err);
	return err;
}


void
wrapfs_d_release(dentry_t *dentry)
{
	dentry_t *lower_dentry;

	print_entry_location();
	fist_print_dentry("wrapfs_d_release IN dentry", dentry);

	/* this could be a null dentry, so check first */
	if (!dentry) {
		fist_dprint(6, "null dentry");
		goto out;
	}
	/* this could be a negative dentry, so check first */
	if (!DENTRY_TO_PRIVATE(dentry)) {
		fist_dprint(6, "dentry without private data: %*s", dentry->d_name.len, dentry->d_name.name);
		goto out;
	}
	lower_dentry = DENTRY_TO_LOWER(dentry);
	fist_print_dentry("wrapfs_d_release IN lower_dentry", lower_dentry);
	if (lower_dentry && lower_dentry->d_inode)
		fist_dprint(6, "wrapfs_d_release: lower_inode->i_count %d, i_num %lu.\n",
                            atomic_read(&lower_dentry->d_inode->i_count),
                            lower_dentry->d_inode->i_ino);

#ifdef FIST_FILTER_SCA
	if (DENTRY_TO_PRIVATE(dentry)->idx_dentry)
		dput(DENTRY_TO_PRIVATE(dentry)->idx_dentry);
#endif /* FIST_FILTER_SCA */
	/* free private data (wrapfs_dentry_info) here */
	KFREE(DENTRY_TO_PRIVATE(dentry));
	//DENTRY_TO_PRIVATE(dentry) = NULL;	/* just to be safe */
	/* decrement lower dentry's counter and free its inode */
	if (lower_dentry) dput(lower_dentry);
out:
	print_exit_location();
}


#ifdef FIST_DEBUG
/*
 * we don't really need wrapfs_d_iput, because dentry_iput will call iput() if
 * wrapfs_d_iput is not defined. We left this implemented for ease of
 * tracing/debugging.
 */
void
wrapfs_d_iput(dentry_t *dentry, inode_t *inode)
{
	print_entry_location();
	iput(inode);
	print_exit_location();
}
#endif /* FIST_DEBUG */


struct dentry_operations wrapfs_dops = {
	d_revalidate:   wrapfs_d_revalidate,
	d_hash:         wrapfs_d_hash,
	d_compare:      wrapfs_d_compare,
	d_release:      wrapfs_d_release,
	d_delete:       wrapfs_d_delete,
	//d_delete:       NULL,
#ifdef FIST_DEBUG
	d_iput:     wrapfs_d_iput,
#endif /* FIST_DEBUG */
};


/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
