/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: file.c,v 1.31 2005/04/10 01:10:39 satnur Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
# ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"
#include <linux/mount.h>
#include <linux/security.h>

/*******************
 * File Operations *
 *******************/


STATIC loff_t
wrapfs_llseek(file_t *file, loff_t offset, int origin)
{
	loff_t err;
	file_t *lower_file = NULL;

	print_entry_location();
	if (FILE_TO_PRIVATE(file) != NULL) {	 // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file);
	}

	fist_dprint(6, "wrapfs_llseek: file=%p, offset=0x%llx, origin=%d\n",
                    file, offset, origin);

#ifndef FIST_FILTER_SCA
	ASSERT(lower_file != NULL);

	/* always set lower position to this one */
	lower_file->f_pos = file->f_pos;


	memcpy(&(lower_file->f_ra), &(file->f_ra), sizeof(struct file_ra_state));

	if (lower_file->f_op && lower_file->f_op->llseek)
		err = lower_file->f_op->llseek(lower_file, offset, origin);
	else
		err = generic_file_llseek(lower_file, offset, origin);

	if (err < 0)
		goto out;

	if (err != file->f_pos) {
		file->f_pos = err;
		// ION maybe this?
		// 	file->f_pos = lower_file->f_pos;
		file->f_version ++ ;//
	}
#else /* FIST_FILTER_SCA */
	fist_print_file("wrapfs_llseek IN Virtual file", file);
	fist_print_inode("wrapfs_llseek IN Virtual inode", file->f_dentry->d_inode);
	/*
	 * There is no need to deal with any read ahead information, since
	 * the SCA is not matched with the real read aheads.  Adjust to the
	 * new position
	 */
	switch (origin) {
	case SEEK_END:
		offset += i_size_read(file->f_dentry->d_inode);
		break;
	case SEEK_CUR:
		offset += file->f_pos;
	}

	err = -EINVAL;

	/*
	 * Check file boundaries:
	 * offset less than zero is not allowed, however, offset beyond
	 * file size is allowed. And it doesnot change the file size
	 */
	if (offset >= 0) {
		if (offset != file->f_pos) {
			file->f_pos = offset;
                        //  file->f_reada = 0;
                        // file->f_version = ++event;
		}
		err = offset;
	}
	fist_print_inode("wrapfs_llseek OUT Virtual inode", file->f_dentry->d_inode);
	fist_print_file("wrapfs_llseek OUT Virtual file", file);
#endif /* FIST_FILTER_SCA */
out:
	print_exit_status((int) err);
	return err;
}


#ifdef FIST_FILTER_DATA
/*
 * generic_file_read updates the atime of upper layer inode.
 * But, it doesn't give us a chance to update the atime of
 * the lower layer inode. This function is a wrapper to
 * generic_file_read. it Updates the atime of the lower level
 * inode if generic_file_read returns without any errors
 * This is to be used only for file reads and only if
 * FIST_FILTER_DATA is defined. The function to be
 * used for directory reads is wrapfs_read
 */
STATIC ssize_t
wrapfs_read_update_atime(file_t *file, char* buf, size_t count, loff_t *ppos)
{
	int err = 0;

	print_entry_location();

	err = generic_file_read(file, buf, count, ppos);

	if (err >= 0)
		update_atime(INODE_TO_LOWER(file->f_dentry->d_inode));

	print_exit_status(err);
	return err;
}
#endif /* FIST_FILTER_DATA */

STATIC ssize_t
wrapfs_read(file_t *file, char *buf, size_t count, loff_t *ppos)
{
	int err = -EINVAL;
	file_t *lower_file = NULL;
	loff_t pos = *ppos;

	print_entry_location();
	if (FILE_TO_PRIVATE(file) != NULL)   // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef..Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file);
	ASSERT(lower_file != NULL);

	if (!lower_file->f_op || !lower_file->f_op->read)
		goto out;

	err = lower_file->f_op->read(lower_file, buf, count, &pos);

	if (err >= 0) {
		/* atime should also be updated for reads of size zero or more */
		fist_copy_attr_atime(file->f_dentry->d_inode,
                                     lower_file->f_dentry->d_inode);
	}

	// MAJOR HACK
	/*
	 * because pread() does not have any way to tell us that it is
	 * our caller, then we don't know for sure if we have to update
	 * the file positions.  This hack relies on read() having passed us
	 * the "real" pointer of its struct file's f_pos field.
	 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	if (ppos == &file->f_pos)
#endif
		lower_file->f_pos = *ppos = pos;

//    if (lower_file->f_reada) { /* update readahead information if needed */  // XXX: ZH: any way to check if it changed? flag doesn't exist
	memcpy(&(file->f_ra), &(lower_file->f_ra), sizeof(struct file_ra_state));	// XXX: ZH: this a new structure.
//    }

out:
	print_exit_status(err);
	return err;
}


/* this wrapfs_write() does not modify data pages! */
STATIC ssize_t
wrapfs_write(file_t *file, const char *buf, size_t count, loff_t *ppos)
{
	int err = -EINVAL;
	file_t *lower_file = NULL;
	inode_t *inode;
	inode_t *lower_inode;
	loff_t pos = *ppos;

	print_entry_location();
	if (FILE_TO_PRIVATE(file) != NULL)  // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file);
	ASSERT(lower_file != NULL);

	inode = file->f_dentry->d_inode;
	lower_inode = INODE_TO_LOWER(inode);

	if (!lower_file->f_op || !lower_file->f_op->write)
		goto out;

	/* adjust for append -- seek to the end of the file */
	if ((file->f_flags & O_APPEND) && (count != 0))
		pos = i_size_read(inode);

	if (count != 0)
		err = lower_file->f_op->write(lower_file, buf, count, &pos);
	else
		err = 0;

	/*
	 * copy ctime and mtime from lower layer attributes
	 * atime is unchanged for both layers
	 */
	if (err >= 0)
		fist_copy_attr_times(inode, lower_inode);

	/*
	 * XXX: MAJOR HACK
	 *
	 * because pwrite() does not have any way to tell us that it is
	 * our caller, then we don't know for sure if we have to update
	 * the file positions.  This hack relies on write() having passed us
	 * the "real" pointer of its struct file's f_pos field.
	 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	if (ppos == &file->f_pos)
#endif
		lower_file->f_pos = *ppos = pos;

	/* update this inode's size */
	if (pos > i_size_read(inode))
		i_size_write(inode, pos);

out:
	print_exit_status(err);
	return err;
}


#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
struct wrapfs_getdents_callback {
	void *dirent;
	dentry_t *dentry;
	filldir_t filldir;
	int err;
	int filldir_called;
	int entries_written;
};

/* copied from generic filldir in fs/readir.c */
STATIC int
wrapfs_filldir(void *dirent, const char *name, int namlen, loff_t offset, ino_t ino, unsigned int d_type)
{
	struct wrapfs_getdents_callback *buf = (struct wrapfs_getdents_callback *) dirent;
	int err;
#ifdef FIST_FILTER_NAME
	char *decoded_name;
	int decoded_length;
#endif /* FIST_FILTER_NAME */

	print_entry_location();

	buf->filldir_called++;

#ifdef FIST_FILTER_SCA
	/* skip .idx files */
	if (namlen > INDEX_EXTENSION_LEN-1 &&
             strncmp(&name[namlen-INDEX_EXTENSION_LEN+1],
                     INDEX_EXTENSION,
                     INDEX_EXTENSION_LEN-1) == 0) {
		//don't do this, it's not zero-terminated!!!
		//fist_dprint(7, "filldir: skip \"%s\"\n", name);
		return 0;
	}
#endif /* FIST_FILTER_SCA */

#ifdef FIST_FILTER_NAME
	decoded_length = wrapfs_decode_filename(name, namlen, &decoded_name, SKIP_DOTS,
                                                buf->dentry->d_inode, buf->dentry->d_sb);
	if (decoded_length < 0) {
		return 0;			/* no error, just skip the entry */
	}

	FIST_OP_READDIR_CALL;

	err = buf->filldir(buf->dirent, decoded_name, decoded_length, offset, ino, d_type);
	KFREE(decoded_name);
#else /* not FIST_FILTER_NAME */
	err = buf->filldir(buf->dirent, name, namlen, offset, ino, d_type);
#endif /* not FIST_FILTER_NAME */
        if (err >= 0) {
                buf->entries_written++;
	}

out:
	print_exit_status(err);
	return err;
}
#endif /* FIST_FILTER_NAME || FIST_FILTER_SCA */


STATIC int
wrapfs_readdir(file_t *file, void *dirent, filldir_t filldir)
{
	int err = -ENOTDIR;
	file_t *lower_file = NULL;
	inode_t *inode;
#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
	struct wrapfs_getdents_callback buf;
#endif /* FIST_FILTER_NAME || FIST_FILTER_SCA */

	print_entry_location();
	if (FILE_TO_PRIVATE(file) != NULL)  // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file);
	ASSERT(lower_file != NULL);

	inode = file->f_dentry->d_inode;

	fist_checkinode(inode, "wrapfs_readdir"); // XXX: ZH: Halcrow commented this out, he doesnt like debugging info
	lower_file->f_pos = file->f_pos;

#if defined(FIST_FILTER_NAME) || defined(FIST_FILTER_SCA)
	/* prepare for callback */
	buf.dirent = dirent;
	buf.dentry = file->f_dentry;
	buf.filldir = filldir;
retry:
	buf.filldir_called = 0;
	buf.entries_written = 0;
	buf.err = 0;
	err = vfs_readdir(lower_file, wrapfs_filldir, (void *) &buf);
	if (buf.err) {
		err = buf.err;
	}
	if (buf.filldir_called && !buf.entries_written) {
		goto retry;
	}
#else /* not FIST_FILTER_NAME || FIST_FILTER_SCA */
	err = vfs_readdir(lower_file, filldir, dirent);
#endif /* not FIST_FILTER_NAME || FIST_FILTER_SCA */

	file->f_pos = lower_file->f_pos;
	if (err >= 0)
		fist_copy_attr_atime(inode, lower_file->f_dentry->d_inode);

	fist_checkinode(inode, "post wrapfs_readdir");
	print_exit_status(err);
	return err;
}


STATIC unsigned int
wrapfs_poll(file_t *file, poll_table *wait)
{
	unsigned int mask = DEFAULT_POLLMASK;
	file_t *lower_file = NULL;

	print_entry_location();
	if (FILE_TO_PRIVATE(file) != NULL)  // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file);
	ASSERT(lower_file != NULL);

	if (!lower_file->f_op || !lower_file->f_op->poll)
		goto out;

	mask = lower_file->f_op->poll(lower_file, wait);

out:
	print_exit_status(mask);
	return mask;
}


STATIC int
wrapfs_ioctl(inode_t *inode, file_t *file, unsigned int cmd, unsigned long arg)
{
	int err = 0;		/* don't fail by default */
	file_t *lower_file = NULL;
	vfs_t *this_vfs;
	vnode_t *this_vnode;
	int val;
#ifdef FIST_COUNT_WRITES
	extern unsigned long count_writes, count_writes_middle;
#endif /* FIST_COUNT_WRITES */

	print_entry_location();

	this_vfs = inode->i_sb;
	this_vnode = inode;

	/* check if asked for local commands */
	switch (cmd) {
#ifdef FIST_DEBUG
	case FIST_IOCTL_GET_DEBUG_VALUE:
		val = fist_get_debug_value();
		printk("IOCTL GET: send arg %d\n", val);
		err = put_user(val, (int *) arg);
#ifdef FIST_COUNT_WRITES
		printk("COUNT_WRITES:%lu:%lu\n", count_writes, count_writes_middle);
#endif /* FIST_COUNT_WRITES */
		break;

	case FIST_IOCTL_SET_DEBUG_VALUE:
		err = get_user(val, (int *) arg);
		if (err)
			break;
		fist_dprint(6, "IOCTL SET: got arg %d\n", val);
		if (val < 0 || val > 20) {
			err = -EINVAL;
			break;
		}
		fist_set_debug_value(val);
		break;
#endif /* FIST_DEBUG */
#ifdef FIST_FILTER_SCA
	case FIST_IOCTL_GET_FAST_TAIL_VALUE:
		val = (SUPERBLOCK_TO_PRIVATE(inode->i_sb)->sca_flags != 0);
		printk("IOCTL GET: send arg %d\n", val);
		err = put_user(val, (int *) arg);
		break;

#ifdef FIST_FAST_TAILS
	case FIST_IOCTL_SET_FAST_TAIL_VALUE:
		err = get_user(val, (int *) arg);
		if (err)
			break;
		fist_dprint(6, "IOCTL SET: got arg %d\n", val);
		if (val == 1) {
			SUPERBLOCK_TO_PRIVATE(inode->i_sb)->sca_flags |= SCA_FLAG_FASTTAIL;
		} else if (val == 0) {
			SUPERBLOCK_TO_PRIVATE(inode->i_sb)->sca_flags &= ~SCA_FLAG_FASTTAIL;
		} else {
			err = -EINVAL;
		}
		break;
#endif /* FIST_FAST_TAILS */
#endif /* FIST_FILTER_SCA */

		/* add non-debugging fist ioctl's here */
		FIST_IOCTL_ECLS

                        default:
		if (FILE_TO_PRIVATE(file) != NULL) {	 // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
			lower_file = FILE_TO_LOWER(file);
		}
		/* pass operation to lower filesystem, and return status */
		if (lower_file && lower_file->f_op && lower_file->f_op->ioctl)
			err = lower_file->f_op->ioctl(INODE_TO_LOWER(inode), lower_file, cmd, arg);
		else
			err	= -ENOTTY;	/* this is an unknown ioctl */
	} /* end of outer switch statement */

	print_exit_status(err);
	return err;
}


#ifndef FIST_FILTER_DATA
/* FIST-LITE special version of mmap */
STATIC int
wrapfs_mmap(file_t *file, vm_area_t *vma)
{
	int err = 0;
	file_t *lower_file = NULL;
	inode_t *inode;
	inode_t *lower_inode;

	print_entry_location();
	if (FILE_TO_PRIVATE(file) != NULL)   // XXX: ZH: this is wrong?: WHERE IS ASSERT IN THIS CASE??? Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file);

	ASSERT(lower_file != NULL);
	if (!lower_file->f_op || !lower_file->f_op->mmap) {
		err = -ENODEV;
		goto out;
	}	
	ASSERT(lower_file->f_op != NULL);
	ASSERT(lower_file->f_op->mmap != NULL);

	vma->vm_file = lower_file;
	err = lower_file->f_op->mmap(lower_file, vma);
	get_file(lower_file); /* make sure it doesn't get freed on us */
	fput(file);			/* no need to keep extra ref on ours */

out:
	print_exit_status(err);
	return err;
}
#endif /* not FIST_FILTER_DATA */


STATIC int
wrapfs_open(inode_t *inode, file_t *file)
{
	int err = 0;
	int lower_flags;  // XXX: ZH: killed "lower_mode" because it is not used...
	file_t *lower_file = NULL;
	dentry_t *lower_dentry = NULL;

	print_entry_location();

	FILE_TO_PRIVATE_SM(file) = KMALLOC(sizeof(struct wrapfs_file_info), GFP_KERNEL);
	if (!FILE_TO_PRIVATE(file)) {
		err = -ENOMEM;
		goto out;
	}

	lower_dentry = wrapfs_lower_dentry(file->f_dentry);	/* CPW: Moved after print_entry */	// XXX: ZH: why not use this?
//	lower_dentry = DENTRY_TO_LOWER(file->f_dentry); // XXX: ZH: Halcrow prefers this avoiding the error checking


	fist_print_dentry("wrapfs_open IN lower_dentry", lower_dentry);

	dget(lower_dentry);

	lower_flags = file->f_flags;
#ifdef FIST_FILTER_DATA
        /* If a file is opened for write only, make read-write. */
	if ((lower_flags & O_ACCMODE) == O_WRONLY)
		lower_flags = (lower_flags & ~O_ACCMODE) | O_RDWR;
	if (file->f_flags & O_APPEND) {
		fist_dprint(5, "file is opened in append-only mode\n");
		lower_flags &= ~O_APPEND; /* turn off O_APPEND flag */
	}
#endif /* not FIST_FILTER_DATA */

	/*
	 * dentry_open will decrement mnt refcnt if err.
	 * otherwise fput() will do an mntput() for us upon file close.
	 */
	mntget(SUPERBLOCK_TO_PRIVATE(inode->i_sb)->lower_mnt);
	lower_file = dentry_open(lower_dentry, SUPERBLOCK_TO_PRIVATE(inode->i_sb)->lower_mnt, lower_flags);
	if (IS_ERR(lower_file)) {
		err = PTR_ERR(lower_file);
		goto out;
	}

	FILE_TO_LOWER(file) = lower_file;	/* link two files */

out:
	if (err < 0 && FILE_TO_PRIVATE(file)) {
		KFREE(FILE_TO_PRIVATE(file));
	}
	fist_print_dentry("wrapfs_open OUT lower_dentry", lower_dentry);
	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_flush(file_t *file)
{
	int err = 0;		/* assume ok (see open.c:close_fp) */
	file_t *lower_file = NULL;

	print_entry_location();

	if (FILE_TO_PRIVATE(file) != NULL)  // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file); /* CPW: Moved after print_entry_location */
	ASSERT(lower_file != NULL);

	if (!lower_file->f_op || !lower_file->f_op->flush)
		goto out;

	err = lower_file->f_op->flush(lower_file);

out:
	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_release(inode_t *inode, file_t *file)
{
	int err = 0;
	file_t *lower_file = NULL;
	dentry_t *lower_dentry;
	inode_t *lower_inode = NULL;

	print_entry_location();

	if (FILE_TO_PRIVATE(file) != NULL)  // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file);
	KFREE(FILE_TO_PRIVATE(file));

	ASSERT(lower_file != NULL);

	ASSERT(inode);
	lower_inode = INODE_TO_LOWER(inode);
	ASSERT(lower_inode != NULL);

	fist_print_dentry("wrapfs_release IN lower_dentry", lower_file->f_dentry);
	fist_checkinode(inode, "wrapfs_release");
	fist_dprint(6, "wrapfs_release IN, file->f_count=%d\n", atomic_read(&file->f_count));
	/*
	 * will decrement file refcount, and if 0, destroy the file,
	 * which will call the lower file system's file release function.
	 */
	lower_dentry = lower_file->f_dentry;
	fput(lower_file);

#ifndef FIST_FILTER_SCA
	/*
         * Ext2 does something strange: if you compile Ext2 with
         * EXT2_PREALLOCATE, then it will prealocate N more blocks of data to
         * the file in anticipation of the file growing in the near future.  If,
         * however, the file doesn't grow, then ext2 will deallocate these
         * pre-allocated blocks on ext2_release.  However, our stacking template
         * copies the number of blocks in a file in prepare_write and
         * commit_write, at which point the lower file reports more blocks than
         * it would have later on once the file is closed.  This was confirmed
         * by using a C program that runs fsync() on an ext2 file that's open in
         * the middle of the session.  The solution here is to copy the number
         * of blocks of the lower file to the upper file.  (It's possible to
         * copy all attributes, but might be overkill -- I'd rather copy only
         * what we need.)  WARNING: this code is definitely wrong for SCA!
         *   -Erez.
         */
	inode->i_blocks = lower_inode->i_blocks;
#endif /* not FIST_FILTER_SCA */

	fist_dprint(6, "wrapfs_release done\n");
	fist_checkinode(inode, "post wrapfs_release");

	fist_print_dentry("wrapfs_release OUT lower_dentry", lower_dentry);
	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_fsync(file_t *file, dentry_t *dentry, int datasync)
{
        int err = -EINVAL;
        file_t *lower_file = NULL;
        dentry_t *lower_dentry;

	print_entry_location();

        /* when exporting upper file system through NFS with sync option,
           nfsd_sync_dir() sets struct file as NULL. Use inode's i_fop->fsync instead of file's.
           see fs/nfsd/vfs.c */
        if (file == NULL) {
                lower_dentry = wrapfs_lower_dentry(dentry);

                if (lower_dentry->d_inode->i_fop && lower_dentry->d_inode->i_fop->fsync) {
                        down(&lower_dentry->d_inode->i_sem);
                        err = lower_dentry->d_inode->i_fop->fsync(lower_file, lower_dentry, datasync);
                        up(&lower_dentry->d_inode->i_sem);
                }

        } else {
                if (FILE_TO_PRIVATE(file) != NULL) {
                        lower_file = FILE_TO_LOWER(file);
                        ASSERT(lower_file != NULL);

                        lower_dentry = wrapfs_lower_dentry(dentry);

                        if (lower_file->f_op && lower_file->f_op->fsync) {
                                down(&lower_dentry->d_inode->i_sem);
                                err = lower_file->f_op->fsync(lower_file, lower_dentry, datasync);
                                up(&lower_dentry->d_inode->i_sem);
                        }
                }
        }

	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_fasync(int fd, file_t *file, int flag)
{
	int err = 0;
	file_t *lower_file = NULL;

	print_entry_location();
	if (FILE_TO_PRIVATE(file) != NULL)   // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
		lower_file = FILE_TO_LOWER(file);
	ASSERT(lower_file != NULL);

	if (lower_file->f_op && lower_file->f_op->fasync)
		err = lower_file->f_op->fasync(fd, lower_file, flag);

	print_exit_status(err);
	return err;
}

static inline void __locks_delete_block(struct file_lock *waiter)
{
        list_del_init(&waiter->fl_block);
        list_del_init(&waiter->fl_link);
        waiter->fl_next = NULL;
}

static void locks_delete_block(struct file_lock *waiter)
{
        lock_kernel();
        __locks_delete_block(waiter);
        unlock_kernel();
}

STATIC inline int
wrapfs_posix_lock(file_t *file, struct file_lock *fl, int cmd)
{
    int error;
    for(;;) {
        error = posix_lock_file(file, fl);
        if ((error != -EAGAIN) || (cmd == F_SETLK))
            break;

        error = wait_event_interruptible(fl->fl_wait, !fl->fl_next);
        if(!error)
            continue;

        locks_delete_block(fl);
        break;
    }

    return error;
}

STATIC int
wrapfs_setlk(file_t *file, int cmd, struct file_lock *fl)
{
    int error;
    inode_t *inode, *lower_inode;
    file_t *lower_file = NULL;

    print_entry_location();

    error = -EINVAL;
    if (FILE_TO_PRIVATE(file) != NULL)   // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
            lower_file = FILE_TO_LOWER(file);
    ASSERT(lower_file != NULL);

    inode = file->f_dentry->d_inode;
    lower_inode = lower_file->f_dentry->d_inode;

    /* Don't allow mandatory locks on files that may be memory mapped
     * and shared.
     */
    if (IS_MANDLOCK(lower_inode) &&
       (lower_inode->i_mode & (S_ISGID | S_IXGRP)) == S_ISGID &&
       mapping_writably_mapped(lower_file->f_mapping)) {
                error = -EAGAIN;
                goto out;
    }

    if (cmd == F_SETLKW) {
        fl->fl_flags |= FL_SLEEP;
    }

    error = -EBADF;
    switch (fl->fl_type) {
        case F_RDLCK:
            if (!(lower_file->f_mode & FMODE_READ))
                goto out;
            break;
        case F_WRLCK:
            if (!(lower_file->f_mode & FMODE_WRITE))
                goto out;
            break;
        case F_UNLCK:
            break;
        default:
            error = -EINVAL;
            goto out;
    }

    fl->fl_file = lower_file;
    error = security_file_lock(lower_file, fl->fl_type);
    if (error)
        goto out;

    if (lower_file->f_op && lower_file->f_op->lock != NULL) {
        error = lower_file->f_op->lock(lower_file, cmd, fl);
        if (error)
            goto out;
        goto upper_lock;
    }

    error = wrapfs_posix_lock(lower_file, fl, cmd);
    if (error)
        goto out;


upper_lock:
    fl->fl_file = file;
    error = wrapfs_posix_lock(file, fl, cmd);
    if (error) {
        fl->fl_type = F_UNLCK;
        fl->fl_file = lower_file;
        wrapfs_posix_lock(lower_file, fl, cmd);
    }

out:
    print_exit_status(error);
    return error;
}

STATIC int
wrapfs_getlk(file_t *file, struct file_lock *fl)
{
    int error=0;
    struct file_lock *tempfl=NULL;

    print_entry_location();

    if (file->f_op && file->f_op->lock) {
        error = file->f_op->lock(file, F_GETLK, fl);
        if (error < 0)
            goto out;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,11)
        else if (error == LOCK_USE_CLNT)
            /* Bypass for NFS with no locking - 2.0.36 compat */
            tempfl = posix_test_lock(file, fl);
#endif
    } else {
        tempfl = posix_test_lock(file, fl);
    }

    if (!tempfl)
        fl->fl_type = F_UNLCK;
    else
        memcpy(fl, tempfl, sizeof(struct file_lock));

out:
    print_exit_status(error);
    return error;
}

STATIC int
wrapfs_lock(file_t *file, int cmd, struct file_lock *fl)
{
        int err = 0;
        file_t *lower_file = NULL;
        struct file_lock *tmpfl = NULL;

        print_entry_location();

        if (FILE_TO_PRIVATE(file) != NULL)   // XXX: ZH: this is wrong?: in this case ASSERT is not in ifndef... Put ASSERT before the if statement, kill if statement
                lower_file = FILE_TO_LOWER(file);
        ASSERT(lower_file != NULL);

        err = -EINVAL;
        if (!fl)
                goto out;

        fl->fl_file = lower_file;
        switch(cmd) {
                case F_GETLK:
		case F_GETLK64:
                        err = wrapfs_getlk(lower_file, fl);
                        break;

                case F_SETLK:
                case F_SETLKW:
                case F_SETLK64:
                case F_SETLKW64:
                        fl->fl_file = file;
                        err = wrapfs_setlk(file, cmd, fl);
                        break;

                default:
                        err = -EINVAL;
        }
        fl->fl_file = file;

out:
        print_exit_status(err);
        return err;
}

struct file_operations wrapfs_dir_fops =
{
	llseek:   wrapfs_llseek,
	read:     wrapfs_read,
	write:    wrapfs_write,
	readdir:  wrapfs_readdir,
	poll:     wrapfs_poll,
	ioctl:    wrapfs_ioctl,
#ifdef FIST_FILTER_DATA
	mmap:     generic_file_mmap,
#else /* not FIST_FILTER_DATA */
	mmap:     wrapfs_mmap,
#endif /* not FIST_FILTER_DATA */
	open:     wrapfs_open,
	flush:    wrapfs_flush,
	release:  wrapfs_release,
	fsync:    wrapfs_fsync,
	fasync:   wrapfs_fasync,
	lock:     wrapfs_lock,
	/* not needed: readv */
	/* not needed: writev */
	/* not implemented: sendpage */
	/* not implemented: get_unmapped_area */
};

struct file_operations wrapfs_main_fops =
{
	llseek:   wrapfs_llseek,
#ifdef FIST_FILTER_DATA
	read:     wrapfs_read_update_atime,
	write:    generic_file_write,
#else /* not FIST_FILTER_DATA */
	read:     wrapfs_read,
	write:    wrapfs_write,
#endif /* not FIST_FILTER_DATA */
	readdir:  wrapfs_readdir,
	poll:     wrapfs_poll,
	ioctl:    wrapfs_ioctl,
#ifdef FIST_FILTER_DATA
	mmap:     generic_file_mmap,
#else /* not FIST_FILTER_DATA */
	mmap:     wrapfs_mmap,
#endif /* not FIST_FILTER_DATA */
	open:     wrapfs_open,
	flush:    wrapfs_flush,
	release:  wrapfs_release,
	fsync:    wrapfs_fsync,
	fasync:   wrapfs_fasync,
	lock:     wrapfs_lock,
	/* not needed: readv */
	/* not needed: writev */
	/* not implemented: sendpage */
	/* not implemented: get_unmapped_area */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
