/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: main.c,v 1.17 2005/01/03 21:10:42 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


/* sb we pass is wrapfs's super_block */
int
wrapfs_interpose(dentry_t *lower_dentry, dentry_t *dentry, super_block_t *sb, int flag)
{
        inode_t *lower_inode;
        int err = 0;
	inode_t *inode;

	print_entry_location();

	lower_inode = lower_dentry->d_inode; /* CPW: moved after print_entry_location */

	ASSERT(lower_inode != NULL);
	ASSERT(dentry->d_inode == NULL);

	/*
	 * We allocate our new inode below, by calling iget.
	 * iget will call our read_inode which will initialize some
	 * of the new inode's fields
	 */
#ifdef FIST_DYNAMIC_INODE_NUMBERS
	/* XXX: fist file systems reserve first 10 inodes */
	inode = iget(sb, wrapfs_iunique(sb, 10));
#else /* not FIST_DYNAMIC_INODE_NUMBERS */
	/* check that the lower file system didn't cross a mount point */
	if (lower_inode->i_sb != SUPERBLOCK_TO_LOWER(sb)) {
		err = -EXDEV;
		goto out;
	}
	inode = iget(sb, lower_inode->i_ino);
#endif /* not FIST_DYNAMIC_INODE_NUMBERS */

	if (!inode) {
		err = -EACCES;		/* should be impossible??? */
		goto out;
	}

	/*
	 * interpose the inode if not already interposed
	 * this is possible if the inode is being reused
	 * XXX: what happens if we get_empty_inode() but there's another already?
	 * for now, ASSERT() that this can't happen; fix later.
	 */
	if (INODE_TO_LOWER(inode) == NULL)
		INODE_TO_LOWER(inode) = igrab(lower_inode);

	/* Use different set of inode ops for symlinks & directories*/
	if (S_ISLNK(lower_inode->i_mode))
		inode->i_op = &wrapfs_symlink_iops;
	else if (S_ISDIR(lower_inode->i_mode))
		inode->i_op = &wrapfs_dir_iops;
	/* Use different set of file ops for directories */
	if (S_ISDIR(lower_inode->i_mode))
		inode->i_fop = &wrapfs_dir_fops;

	/* properly initialize special inodes */
	if (S_ISBLK(lower_inode->i_mode) || S_ISCHR(lower_inode->i_mode) ||
            S_ISFIFO(lower_inode->i_mode) || S_ISSOCK(lower_inode->i_mode)) {
		init_special_inode(inode, lower_inode->i_mode, lower_inode->i_rdev);
	}

#ifndef FIST_FILTER_DATA
	/* Fix our inode's address operations to that of the lower inode */
	if (inode->i_mapping->a_ops != lower_inode->i_mapping->a_ops) {
		fist_dprint(7, "fixing inode 0x%x a_ops (0x%x -> 0x%x)\n",
                            (int) inode, (int) inode->i_mapping->a_ops,
                            (int) lower_inode->i_mapping->a_ops);
		inode->i_mapping->a_ops = lower_inode->i_mapping->a_ops;
	}
#endif /* not FIST_FILTER_DATA */

	/* only (our) lookup wants to do a d_add */
	if (flag)
		d_add(dentry, inode);
	else
		d_instantiate(dentry, inode);

	ASSERT(DENTRY_TO_PRIVATE(dentry) != NULL);

	/* all well, copy inode attributes */
	fist_copy_attr_all(inode, lower_inode);

#ifdef FIST_FILTER_SCA
	/*
	 * Make upper level inode's private data to point the lower_dentry.
	 * This is required in writepage method.
	 */
	dget(lower_dentry);
	INODE_TO_PRIVATE(inode)->lower_dentry = lower_dentry;
#endif  /* FIST_FILTER_SCA */
out:
	print_exit_status(err);
	return err;
}


#ifdef FIST_DEBUG
/* find lower dentry given this wrapfs dentry */
dentry_t *
__wrapfs_lower_dentry(const char *file, const char *func, int line, dentry_t *dentry)
{
	dentry_t *lower_dentry;

	ASSERT2(dentry != NULL);
	ASSERT2(dentry->d_op != NULL);
#ifdef FIST_MNTSTYLE_ATTACH
        ASSERT2(dentry->d_op == &wrapfs_dops
                // XXX: we may need a special version of this fxn for attach mode
                || dentry->d_op == &wrapfs_attach_dops
                );
#else
        ASSERT2(dentry->d_op == &wrapfs_dops);
#endif /* FIST_MNTSTYLE_ATTACH */

	ASSERT2(dentry->d_sb->s_op == &wrapfs_sops);
        if (dentry->d_inode) {
#ifdef FIST_MNTSTYLE_ATTACH
                ASSERT2(dentry->d_inode->i_op == &wrapfs_main_iops ||
                        dentry->d_inode->i_op == &wrapfs_dir_iops ||
                        // XXX: we may need a special version of this fxn for attach mode
                        dentry->d_inode->i_op == &wrapfs_dir_attach_iops ||
                        dentry->d_inode->i_op == &wrapfs_symlink_iops);
#else
                ASSERT2(dentry->d_inode->i_op == &wrapfs_main_iops ||
                        dentry->d_inode->i_op == &wrapfs_dir_iops ||
                        dentry->d_inode->i_op == &wrapfs_symlink_iops);
#endif /* FIST_MNTSTYLE_ATTACH */
        }
	lower_dentry = DENTRY_TO_LOWER(dentry);
	ASSERT2(lower_dentry != NULL);
	return lower_dentry;
}
#endif /* FIST_DEBUG */


/*
 * Parse mount options: dir=XXX and debug=N
 *
 * Returns the dentry object of the lower-level (lower) directory;
 * We want to mount our stackable file system on top of that lower directory.
 *
 * Sets default debugging level to N, if any.
 */
dentry_t *
wrapfs_parse_options(super_block_t *sb, char *options)
{
	dentry_t *lower_root = ERR_PTR(-EINVAL);
	struct nameidata nd;
	char *name, *tmp, *end;
	int err = 0;

	print_entry_location();

	/* We don't want to go off the end of our arguments later on. */
	for (end = options; *end; end++);

	while (options < end) {
		tmp = options;
		while (*tmp && *tmp != ',')
			tmp++;
		*tmp = '\0';
#ifndef FIST_MNTSTYLE_ATTACH
		if (!strncmp("dir=", options, 4)) {
			/* note: the name passed need not be encoded */
			name = options + 4;
			fist_dprint(4, "wrapfs: using directory: %s\n", name);
			err = path_lookup(name, LOOKUP_FOLLOW, &nd);
			if (err) {
				printk("wrapfs: error accessing lower directory '%s'\n", name);
				lower_root = ERR_PTR(err);
				goto out;
			}
			lower_root = nd.dentry;
			SUPERBLOCK_TO_PRIVATE(sb)->lower_mnt = nd.mnt;
			fist_dprint(6, "parse_options: new s_root, inode: 0x%lx, 0x%lx\n",
                                    (long) lower_root, (long) lower_root->d_inode);
		}
#else
		if (!strncmp("maxbytes=", options, 9)) {
			if (!strncmp("max_non_lfs", options + 9, 11)) {
				sb->s_maxbytes = MAX_NON_LFS;
			} else if (!strncmp("max_lfs_filesize", options + 9, 16)) {
				sb->s_maxbytes = MAX_LFS_FILESIZE;
			} else {
				sb->s_maxbytes = simple_strtoul(options + 9, NULL, 0);
			}
			if (sb->s_maxbytes > MAX_LFS_FILESIZE) {
				sb->s_maxbytes = MAX_LFS_FILESIZE;
			}
			printk("wrapfs: maximum file size set to %llu\n", sb->s_maxbytes);
		}
#endif
		else if (!strncmp("debug=", options, 6)) {
			int debug = simple_strtoul(options + 6, NULL, 0);
			fist_set_debug_value(debug);
		} else {
			printk(KERN_WARNING "wrapfs: unrecognized option '%s'\n", options);
		}
		options = tmp + 1;
	}

out:
	print_exit_location();
	return lower_root;
}

#ifdef FIST_MALLOC_DEBUG
/* for malloc debugging */
static atomic_t wrapfs_malloc_counter;

void *
wrapfs_KMALLOC(size_t len, int flag, int line, const char *file)
{
	void *ptr = (void *) KMALLOC(len, flag);
	if (ptr) {
		atomic_inc(&wrapfs_malloc_counter);
		printk("KM:%d:%p:%d:%s\n", atomic_read(&wrapfs_malloc_counter),ptr, line, file);
	}
	return ptr;
}

void
wrapfs_KFREE(void *ptr, int line, const char *file)
{
	atomic_inc(&wrapfs_malloc_counter);
	printk("KF:%d:%p:%d:%s\n", atomic_read(&wrapfs_malloc_counter), ptr, line, file);
	KFREE(ptr);
}
#endif /* FIST_MALLOC_DEBUG */




#ifndef FIST_MNTSTYLE_ATTACH
/* for attach mode, we use a different ->read_super() in attach.c */
static int
wrapfs_read_super(super_block_t *sb, void *raw_data, int silent)
{
	dentry_t *lower_root;
	int err = 0;

	print_entry_location();

#ifdef FIST_MALLOC_DEBUG
	atomic_set(&wrapfs_malloc_counter, 0); /* for malloc debugging */
#endif /* FIST_MALLOC_DEBUG */

	if (!raw_data) {
		printk(KERN_WARNING "wrapfs_read_super: missing data argument\n");
		err = -EINVAL;
		goto out_no_raw;
	}
	/*
	 * Allocate superblock private data
	 */
	SUPERBLOCK_TO_PRIVATE_SM(sb) = KMALLOC(sizeof(struct wrapfs_sb_info), GFP_KERNEL);
	if (!SUPERBLOCK_TO_PRIVATE(sb)) {
		printk(KERN_WARNING "%s: out of memory\n", __FUNCTION__);
		err = -ENOMEM;
		goto out;
	}
	memset(SUPERBLOCK_TO_PRIVATE(sb), 0, sizeof(struct wrapfs_sb_info));

#ifdef FIST_FILTER_SCA
	SUPERBLOCK_TO_PRIVATE(sb)->sca_flags = 0;
#endif /* FIST_FILTER_SCA */

	lower_root = wrapfs_parse_options(sb, raw_data);
	if (IS_ERR(lower_root)) {
		printk(KERN_WARNING "wrapfs_read_super: lookup_dentry failed (err = %ld)\n", PTR_ERR(lower_root));
		err = PTR_ERR(lower_root);
		goto out_free;
	}
	if (!lower_root->d_inode) {
		printk(KERN_WARNING "wrapfs_read_super: no directory to interpose on\n");
		goto out_dput;
	}
	SUPERBLOCK_TO_LOWER(sb) = lower_root->d_sb;

	sb->s_maxbytes = lower_root->d_sb->s_maxbytes;

	sb->s_op = &wrapfs_sops;
	/*
	 * we can't use d_alloc_root if we want to use
	 * our own interpose function unchanged,
	 * so we simply replicate *most* of the code in d_alloc_root here
	 */

	sb->s_root = d_alloc(NULL, &(const struct qstr){hash: 0, name: "/", len: 1});
        if (IS_ERR(sb->s_root)) {
		printk(KERN_WARNING "wrapfs_read_super: d_alloc failed\n");
		err = -ENOMEM;
		goto out_dput;
	}

	sb->s_root->d_op = &wrapfs_dops;
	sb->s_root->d_sb = sb;
	sb->s_root->d_parent = sb->s_root;
	/* link the upper and lower dentries */
	DENTRY_TO_PRIVATE_SM(sb->s_root) = (struct wrapfs_dentry_info *) KMALLOC(sizeof(struct wrapfs_dentry_info), GFP_KERNEL);

	if (!DENTRY_TO_PRIVATE(sb->s_root)) {
		err = -ENOMEM;
		goto out_dput2;
	}
	DENTRY_TO_LOWER(sb->s_root) = lower_root;

#ifdef FIST_FILTER_SCA
	DENTRY_TO_PRIVATE(sb->s_root)->idx_dentry = NULL;
#endif /* FIST_FILTER_SCA */

	if ((err = wrapfs_interpose(lower_root, sb->s_root, sb, 0)) != 0)
		goto out_dput2;

	fist_print_dentry("wrapfs_read_super OUT lower_dentry", lower_root);
	fist_print_inode("wrapfs_read_super OUT lower_inode", lower_root->d_inode);
	// next line causes null ptr deref at mount(2) time
	// fist_print_dentry("%s OUT sb->s_root", __FUNCTION__, sb->s_root);
	goto out;

out_dput2:
	dput(sb->s_root);
out_dput:
	dput(lower_root);
out_free:
	// XXX: HL: is that mntput necessary? There is some relation to the path_walk in parse_options -
	//  there the path_release is "missing"
	if (SUPERBLOCK_TO_PRIVATE(sb)) {
		mntput(SUPERBLOCK_TO_PRIVATE(sb)->lower_mnt);
	}
	KFREE(SUPERBLOCK_TO_PRIVATE(sb));
	SUPERBLOCK_TO_PRIVATE_SM(sb) = NULL;
out:
	fist_print_sb("OUT sb", sb);
	if (SUPERBLOCK_TO_PRIVATE(sb))
		fist_print_sb("OUT lower_sb", SUPERBLOCK_TO_LOWER(sb));
out_no_raw:
	print_exit_location();
	return err;
}
#endif /* not FIST_MNTSTYLE_ATTACH */


#ifdef FIST_DYNAMIC_INODE_NUMBERS
/* wrapper to iunique to warn if we have a wrap-around */
ino_t
wrapfs_iunique(struct super_block *sb, ino_t maxreserved)
{
	static ino_t counter = 0;
	ino_t ret;

	ret = iunique(sb, maxreserved);
	if (ret < counter) {
		printk(KERN_WARNING "%s iunique may have wrapped around!\n", __FUNCTION__);
	}
	counter = ret;

	return ret;
}
#endif /* FIST_DYNAMIC_INODE_NUMBERS */


static struct super_block *wrapfs_get_sb(struct file_system_type *fs_type,
                                         int flags, const char *dev_name,
                                         void *raw_data) {
	return get_sb_nodev( fs_type, flags, raw_data, wrapfs_read_super );
}

void wrapfs_kill_block_super(struct super_block *sb)
{
	generic_shutdown_super(sb);
/*
 *	XXX: BUG: Halcrow: Things get unstable sometime after this point:
 *
 *	lib/rwsem-spinlock.c:127: spin_is_locked on uninitialized
 *	spinlock a1c953d8.
 *
 *	fs/fs-writeback.c:402: spin_lock(fs/super.c:a0381828) already
 *	locked by fs/fs-writeback.c/402
 *
 *	Apparently, someone's not releasing a lock on sb_lock...
*/
}

static struct file_system_type wrapfs_fs_type = {
	.owner          = THIS_MODULE,
	.name           = "wrapfs",
	.get_sb         = wrapfs_get_sb,
	.kill_sb        = wrapfs_kill_block_super,
	.fs_flags       = 0,
};

static int __init init_wrapfs_fs(void)
{
	printk("Registering wrapfs version $Id: main.c,v 1.17 2005/01/03 21:10:42 ezk Exp $\n");
	return register_filesystem(&wrapfs_fs_type);
}
static void __exit exit_wrapfs_fs(void)
{
	printk("Unregistering wrapfs version $Id: main.c,v 1.17 2005/01/03 21:10:42 ezk Exp $\n");
	unregister_filesystem(&wrapfs_fs_type);
}

MODULE_AUTHOR("Erez Zadok <ezk@cs.sunysb.edu>");
MODULE_DESCRIPTION("FiST-generated wrapfs filesystem");

/*
 * Note: you must define *some* non-empty license string in your .fist file,
 * using the "license" declaration.  It's up to the author to decide what
 * they want: it can be anything.  But if on Linux you pick something other
 * than the approved licenses as listed in <linux/module.h>, your module may
 * not link into the kernel if it's using GPL-only symbols.  For more
 * information, see the COPYING file which should be in the same directory
 * you found this file in.
 */
/* This definition must only appear after we include <linux/module.h> */
MODULE_LICENSE(FIST_LICENSE);	/* defined via "license" decl in .fist file */
#ifndef MODULE_LICENSE
# error must define module license via fistgen license declaration
#endif /* not MODULE_LICENSE */

MODULE_PARM(fist_debug_var, "i");
MODULE_PARM_DESC(fist_debug_var, "Debug level");

module_init(init_wrapfs_fs)
module_exit(exit_wrapfs_fs)

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
