/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: mmap.c,v 1.18 2005/02/04 15:40:10 cwright Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include "fist.h"
#include "wrapfs.h"


#ifdef FIST_COUNT_WRITES
/* for counting writes in the middle vs. regular writes */
unsigned long count_writes = 0, count_writes_middle = 0;
#endif /* FIST_COUNT_WRITES */

/* forward declaration of commit write and prepare write */
STATIC int wrapfs_commit_write(file_t *file, page_t *page, unsigned from, unsigned to);
STATIC int wrapfs_prepare_write(file_t *file, page_t *page, unsigned from, unsigned to);


/*
 * Function for handling creation of holes when lseek-ing past the
 * end of the file and then writing some data.
 */
int
wrapfs_fill_zeros(file_t* file, page_t *page, unsigned from)
{
	int err = 0;
	dentry_t *dentry = file->f_dentry;
	inode_t *inode = dentry->d_inode;
	page_t *tmp_page;
	int index;

	print_entry_location();

	for (index = i_size_read(inode) >> PAGE_CACHE_SHIFT; index < page->index; index++) {
		tmp_page = wrapfs_get1page(file, index);
		if (IS_ERR(tmp_page)) {
			err = PTR_ERR(tmp_page);
			goto out;
		}
                kmap(tmp_page);

		/*
		 * zero out rest of the contents of the page between the appropriate
		 * offsets.
		 */
		memset((char*)page_address(tmp_page) + (i_size_read(inode) & ~PAGE_CACHE_MASK), 0, PAGE_CACHE_SIZE - (i_size_read(inode) & ~PAGE_CACHE_MASK));

		if (! (err = wrapfs_prepare_write(file, tmp_page, 0, PAGE_CACHE_SIZE)))
			err = wrapfs_commit_write(file, tmp_page, 0, PAGE_CACHE_SIZE);

                kunmap(tmp_page);
		page_cache_release(tmp_page);
		if (err < 0)
			goto out;
		if (need_resched())
                        schedule();
	}

	/* zero out appropriate parts of last page */

#ifdef FIST_ENCODING_TYPE_BLOCK
	/*
	 * if the encoding type is block, then adjust the 'from' (where the
	 * zeroing will start) offset appropriately
	 */
	from = from & (~(FIST_ENCODING_BLOCKSIZE - 1));
#endif /* FIST_ENCODING_TYPE_BLOCK */

	if ((from - (i_size_read(inode) & ~PAGE_CACHE_MASK)) > 0) {

		memset((char*)page_address(page) + (i_size_read(inode) & ~PAGE_CACHE_MASK), 0, from - (i_size_read(inode) & ~PAGE_CACHE_MASK));
		if (! (err = wrapfs_prepare_write(file, page, 0, PAGE_CACHE_SIZE)))
			err = wrapfs_commit_write(file, page, 0, PAGE_CACHE_SIZE);

		if (err < 0)
			goto out;
		if (need_resched())
                        schedule();
	}

out:
	print_exit_status(err);
	return err;
}


#ifdef FIST_FILTER_SCA
/*
 * Basic Idea:
 * Using *page, we find out inode.  Traverse through
 * entire list of files for current process to find out file
 * with given inode.
 * If found, continue prepare_write and commit_write with it.
 * If not found, prepare a dummy_file and make complete
 * F-D-I link by the lower_dentry stored in inode's private
 * data.
 */
STATIC int
wrapfs_writepage(page_t *page, struct writeback_control *wbc)
{
	int err = 0;
	int i, j;
	inode_t *inode = NULL;
	struct files_struct *files;
	file_t *file = NULL, *filet = NULL;
	unsigned long end_index;
	unsigned offset;

	/*
	 * dummy variables required when we don't find
	 * file_t * in current->files
	 */
	struct wrapfs_dentry_info dummy_dentry_info;
	struct wrapfs_file_info dummy_file_info;

	file_t dummy_file;
	file_t dummy_lower_file;
	dentry_t dummy_dentry;
	dentry_t *lower_dentry;

	print_entry_location();

	/* get the inode from page mapping */
	inode = page->mapping->host;

	/* lower dentry is also stored in inode's private data */
	lower_dentry = INODE_TO_PRIVATE(inode)->lower_dentry;

	/* get the files structure for current process */
	files = current->files;

	/*
	 * traverse the list of current files, and compare to find the
	 * given inode to get file_t for the given page
	 */
	atomic_inc_and_test(&files->count);
	spin_lock(&files->file_lock);

	j = 0;
	file = NULL;
	for (; ;) {
		unsigned long set;

		i = j * __NFDBITS;

		/* break off if we get file struct */
		if (file) {
			break;
		}

		/* break off because we have passed over the max limits */
		if (i >= files->max_fdset || i >= files->max_fds) {
			break;
		}

		set = files->open_fds->fds_bits[j++];
		while (set) {
			if (set & 1) {
				filet = files->fd[i];

				if (filet) {
					if (filet->f_dentry->d_inode->i_ino == inode->i_ino) {
						file = filet;
						break;
					}
				}
			}
			i++;
			set >>= 1;
		} /* while (set) */
	}

	spin_unlock(&files->file_lock);
	atomic_dec_and_test(&files->count);

	/*
	 * We have not got the file structure.  maybe, the process
	 * has exited.  So, we initialize a dummy file_t * and
	 * fill it with appropriate lower_dentry
	 * obtained from the inode at upper level.
	 * We have to introduce dummy_file, dummy_lower_file and
	 * dummy_dentry for faking the entire link.
	 * This works around for ext2, ext3 since they don't
	 * use the file_t * in prepare_write and commit_write.
	 *
	 * We create a dummy link like,
	 *
	 *         dummy_F -----> dummy_D ----->  I
	 *            |             |             |
	 *            |             |             |
	 *            |             |             |
	 *        dummy_hF ----->   hD    -----> hI
	 *
	 */
	if (!file) {
		/* Didn't find the file,  make a dummy link as in figure */

		/*
		 * initialize all fields to zero, since only required
		 * fields are initialized.
		 */
		memset(&dummy_file, 0, sizeof(file_t));
		memset(&dummy_dentry, 0, sizeof(dentry_t));
		memset(&dummy_lower_file, 0, sizeof(file_t));
		memset(&dummy_file_info, 0, sizeof(struct wrapfs_file_info));
		memset(&dummy_dentry_info, 0, sizeof(struct wrapfs_dentry_info));

		/* prepare dummy lower file first */
		dummy_lower_file.f_dentry = lower_dentry;

		/* dummy file */
		FILE_TO_PRIVATE(&dummy_file) = &dummy_file_info;
		FILE_TO_LOWER(&dummy_file) = &dummy_lower_file;
		dummy_file.f_dentry = &dummy_dentry;

		/* dummy dentry */
		DENTRY_TO_PRIVATE(&dummy_dentry) = &dummy_dentry_info;
		DENTRY_TO_LOWER(&dummy_dentry) = lower_dentry;
		dummy_dentry.d_inode = inode;

		/* make our file pointer point to new lower dentry */
		file = &dummy_file;
	}

	/* do as it is done in normal writepage methods */
	end_index = i_size_read(inode) >> PAGE_CACHE_SHIFT;
	offset = i_size_read(inode) & (PAGE_CACHE_SIZE-1);
	if (!offset) {
		offset = PAGE_SIZE;
	}

	if (page->index >= end_index+1 || !offset) {
		unlock_page(page);
		err = -EIO;
		goto out;
	}

	err = wrapfs_prepare_write(file, page, 0, offset);
	if (!err) {
		err = wrapfs_commit_write(file, page, 0, offset);
		unlock_page(page);
	}

out:
	print_exit_status(err);
	return err;
}

#else /* not FIST_FILTER_SCA */

STATIC int
wrapfs_writepage(page_t *page, struct writeback_control *wbc)
{
	int err = -EIO;
	inode_t *inode;
	inode_t *lower_inode;
	page_t *lower_page;
	char *kaddr, *lower_kaddr;

	print_entry_location();

	inode = page->mapping->host;
	lower_inode = INODE_TO_LOWER(inode);

	/*
	 * writepage is called when shared mmap'ed files need to write
	 * their pages, while prepare/commit_write are called from the
	 * non-paged write() interface.  (However, in 2.3 the two interfaces
	 * share the same cache, while in 2.2 they didn't.)
	 *
	 * So we pretty much have to duplicate much of what commit_write does.
	 */

	/* find lower page (returns a locked page) */
	lower_page = grab_cache_page(lower_inode->i_mapping, page->index);
	if (!lower_page)
		goto out;

	/* get page address, and encode it */
	kaddr = (char *) kmap(page);
	lower_kaddr = (char*) kmap(lower_page);
	wrapfs_encode_block(kaddr, lower_kaddr, PAGE_CACHE_SIZE, inode, inode->i_sb, page->index);
	/* if encode_block could fail, then return error */
	kunmap(page);
	kunmap(lower_page);

	/* call lower writepage (expects locked page) */
	err = lower_inode->i_mapping->a_ops->writepage(lower_page, wbc);

	/*
	 * update mtime and ctime of lower level file system
	 * wrapfs' mtime and ctime are updated by generic_file_write
	 */
	lower_inode->i_mtime = lower_inode->i_ctime = CURRENT_TIME;

	page_cache_release(lower_page);	/* b/c grab_cache_page increased refcnt */

	if (err)
		ClearPageUptodate(page);
	else
		SetPageUptodate(page);
out:
	unlock_page(page);
	print_exit_status(err);
	return err;
}

#endif /* not FIST_FILTER_SCA */

/*
 * get one page from cache or lower f/s, return error otherwise.
 * returns unlocked, up-to-date page (if ok), with increased refcnt.
 */
page_t *
wrapfs_get1page(file_t *file, int index)
{
	page_t *page;
	dentry_t *dentry;
	inode_t *inode;
	struct address_space *mapping;
	int err;

	print_entry_location();

	dentry = file->f_dentry; /* CPW: Moved below print_entry_location */
	inode = dentry->d_inode;
	mapping = inode->i_mapping;

	fist_dprint(8, "%s: read page index %d pid %d\n", __FUNCTION__, index, current->pid);
	if (index < 0) {
		printk("%s BUG: index=%d\n", __FUNCTION__, index);
		page = ERR_PTR(-EIO);
		goto out;
	}
	page = read_cache_page(mapping,
                               index,
                               (filler_t *) mapping->a_ops->readpage,
                               (void *) file);
	if (IS_ERR(page))
		goto out;
	wait_on_page_locked(page);
	if (!PageUptodate(page)) {
		lock_page(page);
		err = mapping->a_ops->readpage(file, page);
		if (err) {
			page = ERR_PTR(err);
			goto out;
		}
		wait_on_page_locked(page);
		if (!PageUptodate(page)) {
			page = ERR_PTR(-EIO);
			goto out;
		}
	}

out:
	print_exit_pointer(page);
	return page;
}


/*
 * get one page from cache or lower f/s, return error otherwise.
 * similar to get1page, but doesn't guarantee that it will return
 * an unlocked page.
 */
page_t *
wrapfs_get1page_cached(file_t *file, int index)
{
	page_t *page;
	dentry_t *dentry;
	inode_t *inode;
	struct address_space *mapping;
	int err;

	print_entry_location();

	dentry = file->f_dentry; /* CPW: Moved below print_entry_location */
	inode = dentry->d_inode;
	mapping = inode->i_mapping;

	fist_dprint(8, "%s: read page index %d pid %d\n", __FUNCTION__, index, current->pid);
	if (index < 0) {
		printk("%s BUG: index=%d\n", __FUNCTION__, index);
		page = ERR_PTR(-EIO);
		goto out;
	}
	page = read_cache_page(mapping,
                               index,
                               (filler_t *) mapping->a_ops->readpage,
                               (void *) file);
	if (IS_ERR(page))
		goto out;

out:
	print_exit_pointer(page);
	return page;
}


/*
 * readpage is called from generic_page_read and the fault handler.
 * If your file system uses generic_page_read for the read op, it
 * must implement readpage.
 *
 * Readpage expects a locked page, and must unlock it.
 */
STATIC int
wrapfs_do_readpage(file_t *file, page_t *page)
{
	int err = -EIO;
	dentry_t *dentry;
	file_t *lower_file = NULL;
	dentry_t *lower_dentry;
	inode_t *inode;
	inode_t *lower_inode;
	char *page_data;
#ifdef FIST_FILTER_SCA
	int i;
	int num_lower_pages;
	int lower_page_index;
	page_t **lower_pages;
	char **lower_pages_data;
	void *opaque;
#else /* not FIST_FILTER_SCA */
	page_t *lower_page;
	char *lower_page_data;
	int real_size;
#endif /* not FIST_FILTER_SCA */

	print_entry_location();

	dentry = file->f_dentry; /* CPW: Moved below print_entry_location */
	if (FILE_TO_PRIVATE(file) == NULL) {
		err = -ENOENT;
		goto out_err;
	}
	lower_file = FILE_TO_LOWER(file);
	ASSERT(lower_file != NULL);	/* XXX: is this assertion right here? */
	lower_dentry = DENTRY_TO_LOWER(dentry);
	inode = dentry->d_inode;
	lower_inode = INODE_TO_LOWER(inode);

	fist_dprint(7, "%s: requesting page %lu from file %s\n", __FUNCTION__, page->index, dentry->d_name.name);
#ifdef FIST_FILTER_SCA
	num_lower_pages = wrapfs_count_lower_pages(page->index,
                                                   inode,
                                                   &lower_page_index,
                                                   &opaque);
	if (num_lower_pages <= 0) {
		err = num_lower_pages;
		if (err == 0) {
			page_data = (char *) kmap(page);
			memset(page_data, 0, PAGE_CACHE_SIZE);
			kunmap(page);
		}
		goto out;
	}
#endif /* FIST_FILTER_SCA */

	MALLOC_PAGE_POINTERS(lower_pages, num_lower_pages);
	MALLOC_PAGEDATA_POINTERS(lower_pages_data, num_lower_pages);
	FOR_EACH_PAGE
                CURRENT_LOWER_PAGE = NULL;

	/* find lower page (returns a locked page) */
	FOR_EACH_PAGE {
		fist_dprint(8, "%s: Current page index = %lu\n", __FUNCTION__, CURRENT_LOWER_PAGEINDEX);
		CURRENT_LOWER_PAGE = read_cache_page(lower_inode->i_mapping,
                                                     CURRENT_LOWER_PAGEINDEX,
                                                     (filler_t *) lower_inode->i_mapping->a_ops->readpage,
                                                     (void *) lower_file);
		if (IS_ERR(CURRENT_LOWER_PAGE)) {
			err = PTR_ERR(CURRENT_LOWER_PAGE);
			CURRENT_LOWER_PAGE = NULL;
			goto out_release;
		}
	}

	/*
	 * wait for the page data to show up
	 * (signaled by readpage as unlocking the page)
	 */
	FOR_EACH_PAGE {
		wait_on_page_locked(CURRENT_LOWER_PAGE);
		if (!PageUptodate(CURRENT_LOWER_PAGE)) {
			/*
			 * call readpage() again if we returned from wait_on_page with a
			 * page that's not up-to-date; that can happen when a partial
			 * page has a few buffers which are ok, but not the whole
			 * page.
			 */
			lock_page(CURRENT_LOWER_PAGE);
			err = lower_inode->i_mapping->a_ops->readpage(lower_file,
                                                                      CURRENT_LOWER_PAGE);
			if (err) {
				CURRENT_LOWER_PAGE = NULL;
				goto out_release;
			}
			wait_on_page_locked(CURRENT_LOWER_PAGE);
			if (!PageUptodate(CURRENT_LOWER_PAGE)) {
				err = -EIO;
				goto out_release;
			}
		}
	}

	/* map pages, get their addresses */
	page_data = (char *) kmap(page);
	FOR_EACH_PAGE
                CURRENT_LOWER_PAGEDATA = (char *) kmap(CURRENT_LOWER_PAGE);

#ifdef FIST_FILTER_SCA
#ifdef FIST_FAST_TAILS
	/*
	 * If this is a fast tail (page we were asked for is
	 * the last page of the file), then we don't need to
	 * decode this buffer, just stick it into page_data
	 */
	if (IS_FASTTAIL_PAGE(page->index, inode)) {
		int offset, len, len0;
		offset = LOWER_PAGE_STARTING_OFFSET(page->index, inode) & ~PAGE_CACHE_MASK;
		len = LOWER_PAGE_LENGTH(page->index, inode);
		if (offset + len <= PAGE_CACHE_SIZE) {
			/* copy one page */
			ASSERT(num_lower_pages == 1);
			memcpy(page_data, lower_pages_data[0] + offset, len);
		} else {
			/* copy two pages */
			ASSERT(num_lower_pages == 2);
			len0 = PAGE_CACHE_SIZE - offset;
			memcpy(page_data, lower_pages_data[0] + offset, len0);
			memcpy(page_data + len0, lower_pages_data[1], len - len0);
		}
		memset(page_data + len, 0, PAGE_CACHE_SIZE - len);
		err = 0;
	} else
#endif /* FIST_FAST_TAILS */
		err	= wrapfs_decode_buffers(num_lower_pages, lower_pages_data,
                                                page_data, inode, inode->i_sb, opaque);

	if (err > 0)
		err = 0;
#else /* not FIST_FILTER_SCA */
	/* if decode_block could fail, then return error */
	err = 0;
	real_size = i_size_read(lower_inode) - ((loff_t)page->index << PAGE_CACHE_SHIFT);
	if (real_size <= 0)
		memset(page_data, 0, PAGE_CACHE_SIZE);
	else if (real_size < PAGE_CACHE_SIZE) {
		wrapfs_decode_block(lower_page_data, page_data, real_size, inode, inode->i_sb, page->index);
		memset(page_data + real_size, 0, PAGE_CACHE_SIZE - real_size);
	} else
		wrapfs_decode_block(lower_page_data, page_data, PAGE_CACHE_SIZE, inode, inode->i_sb, page->index);
#endif /* not FIST_FILTER_SCA */

	FOR_EACH_PAGE
                kunmap(CURRENT_LOWER_PAGE);
	kunmap(page);

out_release:
	FOR_EACH_PAGE
                if (CURRENT_LOWER_PAGE)
                        page_cache_release(CURRENT_LOWER_PAGE);	/* undo read_cache_page */

	FREE_PAGE_POINTERS(lower_pages, num_lower_pages);
	FREE_PAGEDATA_POINTERS(lower_pages_data, num_lower_pages);

out:
	if (err == 0)
		SetPageUptodate(page);
	else
		ClearPageUptodate(page);

out_err:
	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_readpage(file_t *file, page_t *page)
{
	int err;
	print_entry_location();

	err = wrapfs_do_readpage(file, page);

	/*
	 * we have to unlock our page, b/c we _might_ have gotten a locked page.
	 * but we no longer have to wakeup on our page here, b/c UnlockPage does
	 * it
	 */
	unlock_page(page);

	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_prepare_write(file_t *file, page_t *page, unsigned from, unsigned to)
{
	int err = 0;

	print_entry_location();

	/*
	 * we call kmap(page) only here, and do the kunmap
	 * and the actual downcalls, including unlockpage and uncache
	 * in commit_write.
	 */
	kmap(page);

	/* fast path for whole page writes */
	if (from == 0 && to == PAGE_CACHE_SIZE)
		goto out;
	/* read the page to "revalidate" our data */
	/* call the helper function which doesn't unlock the page */
	if (!PageUptodate(page))
		err = wrapfs_do_readpage(file, page);

out:
	print_exit_status(err);
	return err;
}


#ifdef FIST_FILTER_SCA
/*
 * shift data inwards; return 0 if ok or -errno
 */
int
wrapfs_shift_inward(file_t *file,	/* lower file we're shifting */
                    off_t dst_offset,	/* where to shift to (abs offset) */
                    off_t src_offset,	/* where to shift from (abs offset) */
                    off_t total_bytes_to_shift
        )
{
	int dst_page_index, src_page_index;
	unsigned int dst_page_offset, src_page_offset;
	page_t *dst_page, *src_page;
	off_t bytes_to_copy;
	int dst_page_bytes_left, src_page_bytes_left;
	char *dst_kaddr, *src_kaddr;
	int err = 0;
	struct address_space_operations *a_ops;

	print_entry_location();

	if (total_bytes_to_shift < 0) {
		printk("wrapfs_shift_inward asked to shift %ld bytes\n", total_bytes_to_shift);
		goto out;
	}

	/* don't print any error message when total_bytes_to_shift is zero */
	if (total_bytes_to_shift == 0) {
		goto out;
	}

	/* compute page indices */
	dst_page_index = dst_offset >> PAGE_CACHE_SHIFT;
	src_page_index = src_offset >> PAGE_CACHE_SHIFT;
	/* compute offsets within pages */
	dst_page_offset = dst_offset & ~PAGE_CACHE_MASK;
	src_page_offset = src_offset & ~PAGE_CACHE_MASK;
	/* compute number of bytes left in pages */
	dst_page_bytes_left = PAGE_CACHE_SIZE - dst_page_offset;
	src_page_bytes_left = PAGE_CACHE_SIZE - src_page_offset;
	/* get pages */
	dst_page = wrapfs_get1page(file, dst_page_index);
	if (IS_ERR(dst_page)) {
		err = PTR_ERR(dst_page);
		goto out;
	}
	src_page = wrapfs_get1page(file, src_page_index);
	if (IS_ERR(src_page)) {
		page_cache_release(dst_page);
		err = PTR_ERR(src_page);
		goto out;
	}
	src_kaddr = (char *) kmap(src_page);

	a_ops = file->f_dentry->d_inode->i_mapping->a_ops;
	while (1) {
		src_page_bytes_left = MIN(src_page_bytes_left, total_bytes_to_shift);
		bytes_to_copy = MIN(src_page_bytes_left, dst_page_bytes_left);
		/* lock destination page */
		lock_page(dst_page);
		err = a_ops->prepare_write(file,
                                           dst_page,
                                           dst_page_offset,
                                           dst_page_offset+bytes_to_copy);
		if (err < 0) {	  /* XXX: what about partial writes */
			unlock_page(dst_page);
			goto out_release;
		}
		dst_kaddr = (char *) page_address(dst_page);
		/* actually copy some bytes */
		memmove(dst_kaddr + dst_page_offset,
                        src_kaddr + src_page_offset,
                        bytes_to_copy);
		err = a_ops->commit_write(file,
                                          dst_page,
                                          dst_page_offset,
                                          dst_page_offset+bytes_to_copy);
		unlock_page(dst_page);
		if (err < 0)
			goto out_release;

		total_bytes_to_shift -= bytes_to_copy;
		if (total_bytes_to_shift == 0)
			break;
		/* adjust destinations as needed */
		dst_page_bytes_left -= bytes_to_copy;
		if (dst_page_bytes_left == 0) { /* get new page */
			page_cache_release(dst_page);
			dst_page_index++;
			dst_page_offset = 0;
			dst_page_bytes_left = PAGE_CACHE_SIZE;
			dst_page = wrapfs_get1page(file, dst_page_index);
			if (IS_ERR(dst_page)) {
				page_cache_release(src_page);
				err = PTR_ERR(dst_page);
				goto out;
			}
		} else {
			dst_page_offset += bytes_to_copy;
		}
		/* adjust sources as needed */
		src_page_bytes_left -= bytes_to_copy;
		if (src_page_bytes_left == 0) { /* get new page */
			kunmap(src_page);
			page_cache_release(src_page);
			src_page_index++;
			src_page_offset = 0;
			src_page_bytes_left = PAGE_CACHE_SIZE;
			src_page = wrapfs_get1page(file, src_page_index);
			if (IS_ERR(src_page)) {
				page_cache_release(dst_page);
				err = PTR_ERR(src_page);
				goto out;
			}
			src_kaddr = (char *) kmap(src_page);
		} else {
			src_page_offset += bytes_to_copy;
		}
		if (need_resched())
                        schedule();
	} /* end of "while (1)" */

out_release:
	page_cache_release(dst_page);
	page_cache_release(src_page);

out:
	print_exit_status(err);
	return err;
}


/*
 * shift data inwards; return 0 if ok or -errno
 */
int
wrapfs_shift_outward(file_t *file,	/* lower file we're shifting */
                     off_t dst_offset,	/* where to shift to (abs offset) */
                     off_t src_offset,	/* where to shift from (abs offset) */
                     off_t total_bytes_to_shift
        )
{
	int dst_page_index, src_page_index;
	unsigned int dst_page_offset, src_page_offset;
	page_t *dst_page, *src_page;
	off_t bytes_to_copy;
	int dst_page_bytes_left, src_page_bytes_left;
	char *dst_kaddr, *src_kaddr;
	int err = 0;
	struct address_space_operations *a_ops;

	print_entry_location();

	fist_dprint(8, "%s args: dst_off=%d, src_off=%d, bytes=%d\n",
                    __FUNCTION__, dst_offset, src_offset, total_bytes_to_shift);

	if (total_bytes_to_shift < 0) {
		printk("wrapfs_shift_outward asked to shift %ld bytes\n", total_bytes_to_shift);
		goto out;
	}

	if (total_bytes_to_shift == 0) {
		goto out;
	}

	/* compute page indices */
	dst_page_index = (dst_offset + total_bytes_to_shift) >> PAGE_CACHE_SHIFT;
	src_page_index = (src_offset + total_bytes_to_shift) >> PAGE_CACHE_SHIFT;
	/* compute offsets within pages */
	dst_page_offset = (dst_offset + total_bytes_to_shift) & ~PAGE_CACHE_MASK;
	src_page_offset = (src_offset + total_bytes_to_shift) & ~PAGE_CACHE_MASK;
	/* compute number of bytes left in pages */
	dst_page_bytes_left = dst_page_offset;
	src_page_bytes_left = src_page_offset;
	fist_dprint(8, "%s dst: index=%d, offset=%d, bytesleft=%d\n",
                    __FUNCTION__, dst_page_index,
                    dst_page_offset, dst_page_bytes_left);
	fist_dprint(8, "%s src: index=%d, offset=%d, bytesleft=%d\n",
                    __FUNCTION__, src_page_index,
                    src_page_offset, src_page_bytes_left);
	/* get pages */
	dst_page = wrapfs_get1page(file, dst_page_index);
	if (IS_ERR(dst_page)) {
		err = PTR_ERR(dst_page);
		goto out;
	}
	src_page = wrapfs_get1page(file, src_page_index);
	if (IS_ERR(src_page)) {
		page_cache_release(dst_page);
		err = PTR_ERR(src_page);
		goto out;
	}
	src_kaddr = (char *) kmap(src_page);

	a_ops = file->f_dentry->d_inode->i_mapping->a_ops;
	while (1) {
		src_page_bytes_left = MIN(src_page_bytes_left, total_bytes_to_shift);
		bytes_to_copy = MIN(src_page_bytes_left, dst_page_bytes_left);
		dst_page_offset -= bytes_to_copy;
		src_page_offset -= bytes_to_copy;
		/* lock destination page */
		lock_page(dst_page);
		err = a_ops->prepare_write(file,
                                           dst_page,
                                           dst_page_offset,
                                           dst_page_offset+bytes_to_copy);
		if (err < 0) {	  /* XXX: what about partial writes */
			unlock_page(dst_page);
			goto out_release;
		}
		dst_kaddr = (char *) page_address(dst_page);
		/* actually copy some bytes */
		memmove(dst_kaddr + dst_page_offset,
                        src_kaddr + src_page_offset,
                        bytes_to_copy);
		err = a_ops->commit_write(file,
                                          dst_page,
                                          dst_page_offset,
                                          dst_page_offset+bytes_to_copy);
		unlock_page(dst_page);
		if (err < 0)
			goto out_release;

		total_bytes_to_shift -= bytes_to_copy;
		if (total_bytes_to_shift == 0)
			break;
		/* adjust destinations as needed */
		dst_page_bytes_left -= bytes_to_copy;
		if (dst_page_bytes_left == 0) { /* get new page */
			page_cache_release(dst_page);
			dst_page_index--;
			dst_page_offset = PAGE_CACHE_SIZE;
			dst_page_bytes_left = PAGE_CACHE_SIZE;
			fist_dprint(8, "%s dst: index=%d, offset=%d, bytesleft=%d\n",
                                    __FUNCTION__, dst_page_index,
                                    dst_page_offset, dst_page_bytes_left);
			dst_page = wrapfs_get1page(file, dst_page_index);
			if (IS_ERR(dst_page)) {
				page_cache_release(src_page);
				err = PTR_ERR(dst_page);
				goto out;
			}
		}
		/* adjust sources as needed */
		src_page_bytes_left -= bytes_to_copy;
		if (src_page_bytes_left == 0) { /* get new page */
			page_cache_release(src_page);
			src_page_index--;
			src_page_offset = PAGE_CACHE_SIZE;
			src_page_bytes_left = PAGE_CACHE_SIZE;
			fist_dprint(8, "%s src: index=%d, offset=%d, bytesleft=%d\n",
                                    __FUNCTION__, src_page_index,
                                    src_page_offset, src_page_bytes_left);
			src_page = wrapfs_get1page(file, src_page_index);
			if (IS_ERR(src_page)) {
				page_cache_release(dst_page);
				err = PTR_ERR(src_page);
				goto out;
			}
			src_kaddr = (char *) kmap(src_page);
		}
		if (need_resched())
                        schedule();
	} /* end of "while (1)" */

out_release:
	page_cache_release(dst_page);
	page_cache_release(src_page);

out:
	print_exit_status(err);
	return err;
}


/*
 * copy data bytes; return 0 if ok or -errno
 */
int
wrapfs_copy_encoded_data(file_t *file,	/* lower file */
                         off_t dst_offset, /* where to copy to (abs offset) */
                         encoded_pages_t *encoded_pages,
                         off_t total_bytes_to_copy
        )
{
	int dst_page_index;
	unsigned int dst_page_offset, src_page_offset;
	page_t *dst_page;
	off_t bytes_to_copy;
	int dst_page_bytes_left, src_page_bytes_left;
	char *dst_kaddr, *src_kaddr;
	int err = 0;
	struct address_space_operations *a_ops;

	print_entry_location();
	// XXX: check errors
	//    printk("EZK0: total_bytes_to_copy 0x%x\n", (int) total_bytes_to_copy);

	/* compute page indices */
	dst_page_index = dst_offset >> PAGE_CACHE_SHIFT;
	/* compute offsets within pages */
	dst_page_offset = dst_offset & ~PAGE_CACHE_MASK;
	src_page_offset = 0;
	/* compute number of bytes left in pages */
	dst_page_bytes_left = PAGE_CACHE_SIZE - dst_page_offset;
	src_page_bytes_left = PAGE_CACHE_SIZE; /* adjusted below */
	/* get pages */
	dst_page = wrapfs_get1page(file, dst_page_index);
	if (IS_ERR(dst_page)) {
		err = PTR_ERR(dst_page);
		goto out;
	}
	/* map pages into kernel addresses */
	src_kaddr = encoded_pages->data;

	a_ops = file->f_dentry->d_inode->i_mapping->a_ops;
	while (1) {
		src_page_bytes_left = MIN(src_page_bytes_left, total_bytes_to_copy);
		bytes_to_copy = MIN(src_page_bytes_left, dst_page_bytes_left);
		/* lock destination page */
		lock_page(dst_page);
		err = a_ops->prepare_write(file,
                                           dst_page,
                                           dst_page_offset,
                                           dst_page_offset + bytes_to_copy);
		if (err < 0) {	  /* XXX: what about partial writes */
			unlock_page(dst_page);
			goto out_release;
		}
		dst_kaddr = (char *) page_address(dst_page);
		/* actually copy some bytes */
		memmove(dst_kaddr + dst_page_offset,
                        src_kaddr + src_page_offset,
                        bytes_to_copy);
		err = a_ops->commit_write(file,
                                          dst_page,
                                          dst_page_offset,
                                          dst_page_offset + bytes_to_copy);
		unlock_page(dst_page);
		if (err < 0)
			goto out_release;

		total_bytes_to_copy -= bytes_to_copy;
		//	printk("EZK1: total_bytes_to_copy 0x%x\n", (int) total_bytes_to_copy);
		if (total_bytes_to_copy == 0)
			break;
		/* adjust destinations as needed */
		dst_page_bytes_left -= bytes_to_copy;
		if (dst_page_bytes_left == 0) { /* get new page */
			page_cache_release(dst_page);
			dst_page_index++;
			dst_page_offset = 0;
			dst_page_bytes_left = PAGE_CACHE_SIZE;
			dst_page = wrapfs_get1page(file, dst_page_index);
			if (IS_ERR(dst_page)) {
				err = PTR_ERR(dst_page);
				goto out;
			}
		} else {
			dst_page_offset += bytes_to_copy;
		}
		/* adjust sources as needed */
		src_page_bytes_left -= bytes_to_copy;
		if (src_page_bytes_left == 0) { /* get new page */
			src_page_offset = 0;
			src_page_bytes_left = PAGE_CACHE_SIZE;
			encoded_pages = encoded_pages->next;
			src_kaddr = encoded_pages->data;
		} else {
			src_page_offset += bytes_to_copy;
		}
		if (need_resched())
			schedule();
	} /* end of "while (1)" */

out_release:
	page_cache_release(dst_page);

out:
	print_exit_status(err);
	return err;
}


STATIC int
wrapfs_commit_write(file_t *file, page_t *page, unsigned from, unsigned to)
{
	int err = 0;
	inode_t *inode;
	inode_t *lower_inode;
	file_t *lower_file = NULL;
	loff_t pos = 0;
	struct encoded_page *ep = NULL, *next_ep, **cur_ep;
	unsigned real_to, lower_size;
       	loff_t page_offset;
	struct iattr attr;
	void *opaque = NULL;
	off_t old_lower_size, lower_offset, old_lower_offset, delta;
	dentry_t *lower_dentry;
	struct scafs_header *hdr;
	int need_to_call = 1;

	print_entry_location();

	inode = page->mapping->host;/* CPW: Moved below print_entry_location */
	lower_inode = INODE_TO_LOWER(inode);

	ASSERT(file != NULL);
	if (FILE_TO_PRIVATE(file) != NULL)
		lower_file = FILE_TO_LOWER(file);
	ASSERT(lower_file != NULL);	/* XXX: is this assertion right here? */
	lower_dentry = lower_file->f_dentry;

	down(&lower_inode->i_sem);

	/*
	 * here we have a kmapped page, with data from the user copied
	 * into it.  we need to encode_block it, and then call the lower
	 * commit_write.  We also need to simulate same behavior of
	 * generic_file_write, and call prepare_write on the lower f/s first.
	 */
#ifdef FIST_COUNT_WRITES
	count_writes++;
#endif /* FIST_COUNT_WRITES */

	hdr = &INODE_TO_PRIVATE(inode)->hdr;

	page_offset = (loff_t)page->index << PAGE_CACHE_SHIFT;
	if (i_size_read(inode) > page_offset + to)
		if (i_size_read(inode) > page_offset + PAGE_CACHE_SIZE)
			real_to = PAGE_CACHE_SIZE;
		else
			real_to	= i_size_read(inode) - page_offset;
	else
		real_to	= to;

	lower_size = 0;

	cur_ep = &ep;
	while (need_to_call) {
		*cur_ep = KMALLOC(sizeof(struct encoded_page), GFP_KERNEL);
		if (!*cur_ep) {
			err = -ENOMEM;
			goto out_free;
		}
		(*cur_ep)->data = (char *) __get_free_page(GFP_KERNEL);
#ifdef FIST_FAST_TAILS
		if (CAN_BE_FASTTAIL_PAGE(real_to, inode) &&
                    page->index + 1 >= hdr->num_chunks) {
			memcpy((*cur_ep)->data, (char *) page_address(page), real_to);
			need_to_call = 0;
			err = real_to;
			hdr->flags |= SCA_FLAG_FASTTAIL;
		} else
#endif /* FIST_FAST_TAILS */
		{
			err = wrapfs_encode_buffers((*cur_ep)->data,
                                                    (char *) page_address(page),
                                                    &need_to_call,
                                                    real_to,
                                                    inode,
                                                    inode->i_sb,
                                                    &opaque);
			if (page->index + 1 >= hdr->num_chunks)
				hdr->flags &= ~SCA_FLAG_FASTTAIL;
		}
		cur_ep = &(*cur_ep)->next;
		*cur_ep = NULL;
		if (err < 0)
			goto out_free;
		lower_size += err;

                if (need_resched())
                        schedule();

		old_lower_size = LOWER_PAGE_LENGTH(page->index, inode);
		lower_offset = LOWER_PAGE_STARTING_OFFSET(page->index, inode);

		delta = old_lower_size - lower_size;
		if (delta != 0) {
			/*
			 * update lower size by delta
			 *
			 * Note! we take a snapshot of the i_size *before* we go
			 * into shift_* as those functions can change it; however,
			 * we only actually *write* it to disk after all is done
			 */
			attr.ia_size = i_size_read(lower_inode) - delta;

			if (i_size_read(inode) > page_offset + PAGE_CACHE_SIZE) {
				/* this is write-in-the-middle */
				if (delta > 0) {
#ifdef FIST_COUNT_WRITES
					count_writes_middle++;
					printk("WITM:%lu:%lu:%lu:%lu:\n",
                                               (unsigned long) page_offset,
                                               (unsigned long) i_size_read(inode), count_writes, count_writes_middle);
#endif /* FIST_COUNT_WRITES */
					/* shift inwards */
					err = wrapfs_shift_inward(lower_file,
                                                                  lower_offset + lower_size,
                                                                  lower_offset + old_lower_size,
                                                                  i_size_read(lower_inode) - (lower_offset + old_lower_size));
					if (err < 0)
						goto out_free;
				} else {
					/* shift outwards from the end */
					err = wrapfs_shift_outward(lower_file,
                                                                   lower_offset + lower_size,
                                                                   lower_offset + old_lower_size,
                                                                   i_size_read(lower_inode) - (lower_offset + old_lower_size));
					if (err < 0)
						goto out_free;
				}
			} else {
				/* this is append and/or extend */
				if (page->index >= hdr->num_chunks &&
                                    hdr->num_chunks > 0 &&
                                    (hdr->real_size & ~PAGE_CACHE_MASK) != 0) {
					/* we need to encode the former incomplete tail */
					page_t *old_last_page;
					char *old_last_page_data;
					int temp_size;
					struct encoded_page *temp_ep = NULL;

					old_last_page = wrapfs_get1page_cached(file, hdr->num_chunks - 1);
					/*
					 * Call 'get1page_cached' instead of 'get1page', to
					 * avoid deadlock. The deadlock is tickled while
					 * running fsx-linux on gzipfs.
					 *
					 * The deadlock is between fsx's commit_write and
					 * kupdated which calls writepage.
					 *
					 * When the deadlock occurs, fsx would have downed the
					 * inode->i_sem and gotten to this point when it is
					 * switched out and kupdated takes over. kupdated
					 * calls writepage with a page locked. (page 0 in this
					 * case). writepage then calls commit_write on page
					 * 0. But this cannot proceed as it can't obtain the
					 * semaphore.
					 *
					 * fsx is switched back in and it tries to call
					 * get1page on 'hdr->num_chunks - 1' (evaluates to 0
					 * in this case), but the page has been locked by
					 * kupdated and kupdated is waiting on the semaphore
					 * downed by fsx. And we have a deadlock.
					 *
					 * solution was to use the page returned by
					 * read_cache_page without ensuring that it is
					 * unlocked.
					 */

					if (IS_ERR(old_last_page)) {
						err = PTR_ERR(old_last_page);
						goto out_free;
					}
					old_last_page_data = (char *) kmap(old_last_page);

					temp_size = 0;
					need_to_call = 1;
					cur_ep = &temp_ep;
					opaque = NULL;
					while (need_to_call) {
						*cur_ep = KMALLOC(sizeof(struct encoded_page), GFP_KERNEL);
						if (!*cur_ep) {
							err = -ENOMEM;
							goto out_free;
						}
						(*cur_ep)->data = (char *) __get_free_page(GFP_KERNEL);
						err = wrapfs_encode_buffers((*cur_ep)->data,
                                                                            old_last_page_data,
                                                                            &need_to_call,
                                                                            PAGE_CACHE_SIZE,
                                                                            inode,
                                                                            inode->i_sb,
                                                                            &opaque);
						cur_ep = &(*cur_ep)->next;
						*cur_ep = NULL;
						if (err < 0)
							goto encode_error;
						temp_size += err;
						if (need_resched())
                                                        schedule();
					}

					old_lower_offset = LOWER_PAGE_STARTING_OFFSET(old_last_page->index, inode);
					err = wrapfs_copy_encoded_data(lower_file,
                                                                       old_lower_offset,
                                                                       temp_ep,
                                                                       temp_size);
					if (err < 0)
						goto encode_error;
					/* adjust lower_offset */
					lower_offset = old_lower_offset + temp_size;
					wrapfs_idx_set_entry(hdr, hdr->num_chunks - 1, lower_offset);

					/* Calculate new lower file size */
					attr.ia_size = lower_offset + lower_size;

                                encode_error:
					/* free encoded_pages */
					while (temp_ep) {
						next_ep = temp_ep->next;
						free_page((unsigned long) temp_ep->data);
						KFREE(temp_ep);
						temp_ep = next_ep;
					}

					kunmap(old_last_page);
					page_cache_release(old_last_page);
					if (err < 0)
						goto out_free;
				}
			}

			/* update index table structure */
			err = wrapfs_idx_set_entry(hdr,
                                                   page->index,
                                                   lower_offset + lower_size);
			if (err < 0)
				goto out_free;

			attr.ia_valid = ATTR_SIZE;
			err = notify_change(lower_dentry, &attr);
			if (err < 0)
				goto out_free;

			/* update header information by delta */
			wrapfs_idx_update(hdr, page->index + 1, -delta);
		}
		/*
		 * copy chunk of modified bytes, now that we've
		 * potentially shifted and made space
		 */
		err = wrapfs_copy_encoded_data(lower_file,
                                               lower_offset,
                                               ep,
                                               lower_size);
		if (err < 0)
			goto out;
		/* we return number of bytes written */
		err = to - from;
		/* we may have to update the header size */
		pos = ((loff_t)page->index << PAGE_CACHE_SHIFT) + to;
		if (pos > hdr->real_size)
			hdr->real_size = pos;
		/* update our file size */
		copy_inode_size(inode, lower_inode);

		/*
		 * update mtime and ctime of lower level file system
		 * wrapfs' mtime and ctime are updated by generic_file_write
		 */
		lower_inode->i_mtime = lower_inode->i_ctime = CURRENT_TIME;

        out_free:
		/* free encoded_pages */
		while (ep) {
			next_ep = ep->next;
			free_page((unsigned long) ep->data);
			KFREE(ep);
			ep = next_ep;
		}

        out:
		/* final cleanups */
		kunmap(page);		/* kmap was done in prepare_write */
		if (err < 0)
			ClearPageUptodate(page);
		else
			SetPageUptodate(page);
		up(&lower_inode->i_sem);
		print_exit_status(err);
		return err;			/* assume all is ok */
	}
        return err;
}

#else /* not FIST_FILTER_SCA */


STATIC int
wrapfs_commit_write(file_t *file, page_t *page, unsigned from, unsigned to)
{
        int err = -ENOMEM;
        inode_t *inode;
        inode_t *lower_inode;
        page_t *lower_page;
        file_t *lower_file = NULL;
        loff_t pos;
        unsigned bytes = to - from;
        unsigned lower_from, lower_to, lower_bytes;

        print_entry_location();

        inode = page->mapping->host; /* CPW: Moved below print_entry_location */
        lower_inode = INODE_TO_LOWER(inode);

        ASSERT(file != NULL);
        /*
         * here we have a kmapped page, with data from the user copied
         * into it.  we need to encode_block it, and then call the lower
         * commit_write.  We also need to simulate same behavior of
         * generic_file_write, and call prepare_write on the lower f/s first.
         */
#ifdef FIST_COUNT_WRITES
        count_writes++;
#endif /* FIST_COUNT_WRITES */

        /* this is append and/or extend -- we can't have holes so fill them in */
        if (page->index > (i_size_read(lower_inode) >> PAGE_CACHE_SHIFT)) {
                page_t *tmp_page;
                int index;
                for (index = i_size_read(lower_inode) >> PAGE_CACHE_SHIFT; index < page->index; index++) {
                        tmp_page = wrapfs_get1page(file, index);
                        if (IS_ERR(tmp_page)) {
                                err = PTR_ERR(tmp_page);
                                goto out;
                        }
                        kmap(tmp_page);

                        /* zero out the contents of the page at the appropriate offsets */
                        memset((char*)page_address(tmp_page) + (i_size_read(inode) & ~PAGE_CACHE_MASK), 0, PAGE_CACHE_SIZE - (i_size_read(inode) & ~PAGE_CACHE_MASK));
                        if (!(err = wrapfs_prepare_write(file, tmp_page, 0, PAGE_CACHE_SIZE)))
                                err = wrapfs_commit_write(file, tmp_page, 0, PAGE_CACHE_SIZE);
                        kunmap(tmp_page);

                        page_cache_release(tmp_page);
                        if (err < 0)
                                goto out;
                        if (need_resched())
                                schedule();

                }
        }

        if (FILE_TO_PRIVATE(file) != NULL)
                lower_file = FILE_TO_LOWER(file);
        ASSERT(lower_file != NULL);	/* XXX: is this assertion right here? */

        down(&lower_inode->i_sem);
        /* find lower page (returns a locked page) */
        lower_page = grab_cache_page(lower_inode->i_mapping, page->index);
        if (!lower_page)
                goto out_up;
        kmap(lower_page);

#if FIST_ENCODING_BLOCKSIZE > 1
#error encoding_blocksize greater than 1 is not yet supported
#endif /* FIST_ENCODING_BLOCKSIZE > 1 */

#ifdef FIST_ENCODING_TYPE_NONE
        lower_from = from;
        lower_to = to;

        if (((loff_t)page->index << PAGE_CACHE_SHIFT) + to > i_size_read(lower_inode)) {

                /*
                 * If this call to commit_write had introduced holes and the code
                 * for handling holes was invoked, then the beginning of this page
                 * must be zeroed out as well:
                 *	zero out bytes from 'size_of_file%pagesize' to 'from'.
                 */
                if (from > (i_size_read(inode) & ~PAGE_CACHE_MASK))
                        memset((char*)page_address(page) + (i_size_read(inode) & ~PAGE_CACHE_MASK), 0, from - (i_size_read(inode) & ~PAGE_CACHE_MASK));
        }

#endif  /* FIST_ENCODING_TYPE_NONE */
#ifdef FIST_ENCODING_TYPE_STREAM
        lower_from = 0;
        if (((loff_t)page->index << PAGE_CACHE_SHIFT) + to > i_size_read(lower_inode)) {

                /*
                 * If this call to commit_write had introduced holes and the code
                 * for handling holes was invoked, then the beginning of this page
                 * must be zeroed out as well:
                 *	zero out bytes from 'size_of_file%pagesize' to 'from'.
                 */
                if ((from - (i_size_read(inode) & ~PAGE_CACHE_MASK)) > 0)
                        memset((char*)page_address(page) + (i_size_read(inode) & ~PAGE_CACHE_MASK), 0, from - (i_size_read(inode) & ~PAGE_CACHE_MASK));
                lower_to = to;
        } else if ((((loff_t)page->index + 1) << PAGE_CACHE_SHIFT) <= i_size_read(lower_inode))
                lower_to = PAGE_CACHE_SIZE;
        else
                lower_to = i_size_read(lower_inode) & ~PAGE_CACHE_MASK;
#endif /* FIST_ENCODING_TYPE_STREAM */
#ifdef FIST_ENCODING_TYPE_BLOCK
        lower_from = from & (~(FIST_ENCODING_BLOCKSIZE - 1));
        lower_to = ((to + FIST_ENCODING_BLOCKSIZE - 1) & (~(FIST_ENCODING_BLOCKSIZE - 1)));
        if (((loff_t)page->index << PAGE_CACHE_SHIFT) + to > i_size_read(lower_inode)) {

                /*
                 * if this call to commit_write had introduced holes and the code
                 * for handling holes was invoked, then the beginning of this page
                 * must be zeroed out
                 * zero out bytes from 'size_of_file%pagesize' to 'from'.
                 */
                if ((lower_from - (i_size_read(inode) & ~PAGE_CACHE_MASK)) > 0)
                        memset((char*)page_address(page) + (i_size_read(inode) & ~PAGE_CACHE_MASK), 0, lower_from - (i_size_read(inode) & ~PAGE_CACHE_MASK));

        }
#endif /* FIST_ENCODING_TYPE_NONE */
        lower_bytes = lower_to - lower_from;

        /* call lower prepare_write */
        err = -EINVAL;
        if (lower_inode->i_mapping &&
            lower_inode->i_mapping->a_ops &&
            lower_inode->i_mapping->a_ops->prepare_write)
                err = lower_inode->i_mapping->a_ops->prepare_write(lower_file,
                                                                   lower_page,
                                                                   lower_from,
                                                                   lower_to);
        if (err)
                /* don't leave locked pages behind, esp. on an ENOSPC */
                goto out_unlock;

        fist_dprint(8, "%s: encoding %d bytes\n", __FUNCTION__, lower_bytes);
        wrapfs_encode_block((char *) page_address(page) + lower_from, (char*) page_address(lower_page) + lower_from, lower_bytes, inode, inode->i_sb, page->index);
        /* if encode_block could fail, then goto unlock and return error */

        /* call lower commit_write */
        err = lower_inode->i_mapping->a_ops->commit_write(lower_file,
                                                          lower_page,
                                                          lower_from,
                                                          lower_to);

        if (err < 0)
                goto out_unlock;

        err = bytes;	/* convert error to no. of bytes */

        inode->i_blocks = lower_inode->i_blocks;
        /* we may have to update i_size */
        pos = ((loff_t)page->index << PAGE_CACHE_SHIFT) + to;
        if (pos > i_size_read(inode))
                i_size_write(inode, pos);

        /*
         * update mtime and ctime of lower level file system
         * wrapfs' mtime and ctime are updated by generic_file_write
         */
        lower_inode->i_mtime = lower_inode->i_ctime = CURRENT_TIME;

        mark_inode_dirty_sync(inode);

out_unlock:
        kunmap(lower_page);
        unlock_page(lower_page);
        page_cache_release(lower_page);
        kunmap(page);		/* kmap was done in prepare_write */
out_up:
        up(&lower_inode->i_sem);
out:
        /* we must set our page as up-to-date */
        if (err < 0)
                ClearPageUptodate(page);
        else
                SetPageUptodate(page);
        print_exit_status(err);
        return err;			/* assume all is ok */
}

#endif /* end not FIST_FILTER_SCA */


STATIC sector_t
wrapfs_bmap(struct address_space *mapping, sector_t block)
{
        int err = 0;
        inode_t *inode;
        inode_t *lower_inode;

        print_entry_location();

        inode = (inode_t *) mapping->host;
        lower_inode = INODE_TO_LOWER(inode);

        if (lower_inode->i_mapping->a_ops->bmap)
                err = lower_inode->i_mapping->a_ops->bmap(lower_inode->i_mapping, block);
        print_exit_location();
        return err;
}


/*
 * This function is copied verbatim from mm/filemap.c.
 * XXX: It should be simply moved to some header file instead -- bug Al about it!
 */
static inline int sync_page(struct page *page)
{
        struct address_space *mapping = page->mapping;

        if (mapping && mapping->a_ops && mapping->a_ops->sync_page)
                return mapping->a_ops->sync_page(page);
        return 0;
}


/*
 * XXX: we may not need this function if not FIST_FILTER_DATA.
 * FIXME: for FIST_FILTER_SCA, get all lower pages and sync them each.
 */
STATIC int
wrapfs_sync_page(page_t *page)
{
        int err = 0;
        inode_t *inode;
        inode_t *lower_inode;
        page_t *lower_page;

        print_entry_location();

        inode = page->mapping->host; /* CPW: Moved below print_entry_location */
        lower_inode = INODE_TO_LOWER(inode);

        /* find lower page (returns a locked page) */
        lower_page = grab_cache_page(lower_inode->i_mapping, page->index);
        if (!lower_page) {
                err = -ENOMEM;
                goto out;
        }
        err = sync_page(lower_page);

        unlock_page(lower_page);	/* b/c grab_cache_page locked it */
        page_cache_release(lower_page);	/* b/c grab_cache_page increased refcnt */

out:
        print_exit_status(err);
        return err;
}


#ifdef FIST_FILTER_DATA
struct address_space_operations wrapfs_aops =
{
        writepage:      wrapfs_writepage,
        readpage:       wrapfs_readpage,
        prepare_write:  wrapfs_prepare_write,
        commit_write:   wrapfs_commit_write,
        bmap:           wrapfs_bmap,
        sync_page:      wrapfs_sync_page,
};
#endif /* FIST_FILTER_DATA */

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
