/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 *  $Id: wrapfs.h,v 1.18 2005/02/04 15:40:10 cwright Exp $
 */

#ifndef __WRAPFS_H_
#define __WRAPFS_H_

#ifdef __KERNEL__

/*
 * STRUCTURES:
 */


/* Turn this on to include code that counts writes and writes in the middle */
/* #undef FIST_COUNT_WRITES */

/* fist file systems superblock magic */
#define WRAPFS_SUPER_MAGIC 0xf15f

#ifdef FIST_FILTER_SCA
#define FIST_FAST_TAILS
/*
 * ON DISK format of index file:
 * 4 bytes: num_chunks and flags piggybacked
 * 4 bytes: real_size
 * Nx4 bytes: num_chunks of 4 bytes for the offsets
 */
struct scafs_header {
	unsigned long num_chunks;	/* number of "offset" entries */
	off_t real_size;		/* real decoded file size */
	off_t **offsets;		/* actual offset entries */
	/* the following fields do not get written to disk! */
	unsigned long num_alloc;	/* no. of allocated offsets
                                           0 means hdr is uninitialized/empty */
	unsigned long num_pageslots;/* no. of allocated page slots */
	unsigned int flags;		/* various flags */
};
/* flags in memory */
#define SCA_FLAG_MODIFIED	0x00000001
/* flags on disk */
#define SCA_FLAG_FASTTAIL	0x80000000 /* has fast tail */
#define SCA_FLAG_HOLES	0x40000000 /* supporting holes */
/* we have 12 upper bits to play with */
#define SCA_FLAG_MASK		0xfff00000

typedef struct encoded_page {
	char *data;
	struct encoded_page *next;
} encoded_pages_t;

#define INDEX_EXTENSION	".idx"
#define SCA_EXTENSION		".dat"
#define INDEX_EXTENSION_LEN	(sizeof(INDEX_EXTENSION))
#define SCA_EXTENSION_LEN	(sizeof(SCA_EXTENSION))
#endif /* FIST_FILTER_SCA */

/* wrapfs inode data in memory */
struct wrapfs_inode_info {
	inode_t *wii_inode;
#ifdef FIST_FILTER_SCA
	dentry_t *lower_dentry;
	struct scafs_header hdr;
	file_t *idx_file;
#endif /* FIST_FILTER_SCA */
};

/* wrapfs dentry data in memory */
struct wrapfs_dentry_info {
	dentry_t *wdi_dentry;
#ifdef FIST_FILTER_SCA
	dentry_t *idx_dentry;
#endif /* FIST_FILTER_SCA */
};

#ifdef FIST_MNTSTYLE_ATTACH
typedef struct attached_entry {
	struct list_head list;
	struct dentry *e_dentry;
} attached_entry_t;
int del_attached_ent(super_block_t *sb, const char *name, unsigned short namelen);
#endif /* FIST_MNTSTYLE_ATTACH */

/* wrapfs super-block data in memory */
struct wrapfs_sb_info {
	super_block_t *wsi_sb;
	struct vfsmount *lower_mnt;
#ifdef FIST_FILTER_SCA
	unsigned long sca_flags;	/* global SCA flags, see above */
#endif /* FIST_FILTER_SCA */
#ifdef FIST_MNTSTYLE_ATTACH
	struct list_head attached; /* list of attached entries */
	rwlock_t attachlock;	/* lock for attach list */
#endif /*FIST_MNTSTYLE_ATTACH */
	FIST_VFS_FIELDS;
};

/* file private data. */
struct wrapfs_file_info {
	struct file *wfi_file;
};

/*
 * MACROS:
 */

// File TO Private Data
#define FILE_TO_PRIVATE(file) ((struct wrapfs_file_info *)((file)->private_data))
#define FILE_TO_PRIVATE_SM(file) ((file)->private_data)
// File TO lower File
#define FILE_TO_LOWER(file) ((FILE_TO_PRIVATE(file))->wfi_file)
// Inode TO Private Data
#define INODE_TO_PRIVATE(ino) ((struct wrapfs_inode_info *)(ino)->u.generic_ip)
#define INODE_TO_PRIVATE_SM(ino) ((ino)->u.generic_ip)
// Inode TO lower Inode
#define INODE_TO_LOWER(ino) (INODE_TO_PRIVATE(ino)->wii_inode)
#define vnode2lower INODE_TO_LOWER
// Superblock TO Private Data
#define SUPERBLOCK_TO_PRIVATE(super) ((struct wrapfs_sb_info *)(super)->s_fs_info)
#define SUPERBLOCK_TO_PRIVATE_SM(super) ((super)->s_fs_info)
#define vfs2priv SUPERBLOCK_TO_PRIVATE
// Superblock TO lower Superblock
#define SUPERBLOCK_TO_LOWER(super) (SUPERBLOCK_TO_PRIVATE(super)->wsi_sb)
// Dentry TO lower Dentry (and private data --- for the future)
#define DENTRY_TO_PRIVATE_SM(dentry) ((dentry)->d_fsdata)
//(dentry)->d_fsdata
#define DENTRY_TO_PRIVATE(dentry) ((struct wrapfs_dentry_info *)(dentry)->d_fsdata)
#define DENTRY_TO_LOWER(dent) (DENTRY_TO_PRIVATE(dent)->wdi_dentry)
#define dir2lower DENTRY_TO_LOWER
/* VMArea TO lower VMArea */
#define VMA_TO_LOWER(vma) ((vm_area_t *)((vma)->vm_private_data))

#define sbt(sb) ((sb)->s_type->name)

/* macros to simplify non-SCA code */
#ifdef FIST_FILTER_SCA
#define FIST_SCA_CHUNKSIZE PAGE_CACHE_SIZE
#define MALLOC_PAGE_POINTERS(lower_pages, num_lower_pages) \
  lower_pages = (page_t **)KMALLOC(sizeof(page_t *) * num_lower_pages, GFP_KERNEL)
#define MALLOC_PAGEDATA_POINTERS(lower_pages_data, num_lower_pages) \
  lower_pages_data = (char **)KMALLOC(sizeof(char *) * num_lower_pages, GFP_KERNEL)
#define FREE_PAGE_POINTERS(lower_pages, num) KFREE(lower_pages)
#define FREE_PAGEDATA_POINTERS(lower_pages_data, num) KFREE(lower_pages_data)
#define FOR_EACH_PAGE \
    for (i = 0; i < num_lower_pages; i++)
#define CURRENT_LOWER_PAGE lower_pages[i]
#define CURRENT_LOWER_PAGEDATA lower_pages_data[i]
#define CURRENT_LOWER_PAGEINDEX (lower_page_index + i)
#define IS_FASTTAIL_PAGE(pageno, inode) \
  ((pageno == INODE_TO_PRIVATE(inode)->hdr.num_chunks - 1) && \
   (INODE_TO_PRIVATE(inode)->hdr.flags & SCA_FLAG_FASTTAIL))
#define CAN_BE_FASTTAIL_PAGE(size, inode) \
  ((SUPERBLOCK_TO_PRIVATE(inode->i_sb)->sca_flags & SCA_FLAG_FASTTAIL) && \
   (size < PAGE_CACHE_SIZE))
#define HAS_FASTTAIL_PAGE(inode) \
  (INODE_TO_PRIVATE(inode)->hdr.flags & SCA_FLAG_FASTTAIL == SCA_FLAG_FASTTAIL)
#if (BITS_PER_LONG == 32)
#define INDEX_ENTRIES_PER_PAGE	(PAGE_CACHE_SIZE >> 2)
#define OFFSET_SHIFT (PAGE_CACHE_SHIFT - 2)
#else  /* BITS_PER_LONG == 64 */
#define INDEX_ENTRIES_PER_PAGE	(PAGE_CACHE_SIZE >> 3)
#define OFFSET_MASK (PAGE_CACHE_SHIFT - 3)
#endif /* BITS_PER_LONG == 64 */
#define OFFSET_MASK (INDEX_ENTRIES_PER_PAGE - 1)
#define SCA_OFFSET(hdr, pageno) \
  ((hdr)->offsets[(pageno) >> OFFSET_SHIFT][(pageno) & OFFSET_MASK])

/* Current byte offset within a file of a given encoded page */
static inline off_t
LOWER_PAGE_STARTING_OFFSET(unsigned int pageno, inode_t *inode)
{
	struct scafs_header *hdr;
	off_t ret;

	hdr = &INODE_TO_PRIVATE(inode)->hdr;

	if (hdr->num_chunks == 0 || pageno == 0)
		ret = 0;
	else {
		if (pageno >= hdr->num_chunks)
			ret = SCA_OFFSET(hdr, hdr->num_chunks - 1);
		else
			ret	= SCA_OFFSET(hdr, pageno - 1);
	}
	return ret;
}
/* Current length of an encoded page */
static inline int
LOWER_PAGE_LENGTH(unsigned int pageno, inode_t *inode)
{
	struct scafs_header *hdr;
	off_t ret;

	hdr = &INODE_TO_PRIVATE(inode)->hdr;

	if (pageno >= hdr->num_chunks)
		ret = 0;
	else {
		if (pageno == 0)
			ret = SCA_OFFSET(hdr, 0);
		else
			ret	= SCA_OFFSET(hdr, pageno) - SCA_OFFSET(hdr, pageno - 1);
	}
	return ret;
}
static inline int
IS_HOLE(unsigned int pageno, inode_t *inode)
{
	struct scafs_header *hdr;
	int ret;

	hdr = &INODE_TO_PRIVATE(inode)->hdr;

	if (pageno == 0)
		ret = (hdr->num_chunks == 0 || hdr->offsets[0] == 0);
	else
		ret	= (hdr->offsets[pageno] == hdr->offsets[pageno - 1]);
	return ret;
}
#else  /* not FIST_FILTER_SCA */
#define MALLOC_PAGE_POINTERS(lower_pages, num_lower_pages)
#define MALLOC_PAGEDATA_POINTERS(lower_pages_data, num_lower_pages)
#define FREE_PAGE_POINTERS(lower_pages, num)
#define FREE_PAGEDATA_POINTERS(lower_pages_data, num)
#define FOR_EACH_PAGE
#define CURRENT_LOWER_PAGE lower_page
#define CURRENT_LOWER_PAGEDATA lower_page_data
#define CURRENT_LOWER_PAGEINDEX page->index
#endif /* not FIST_FILTER_SCA */


#ifdef FIST_FILTER_NAME
/*
 * Flags for wrapfs_{en,de}code_filename
 * DO_DOTS means the special entries . and .. should be encoded (for symlink)
 * SKIP_DOTS means they should be preserved intact
 */
#define DO_DOTS   0
#define SKIP_DOTS 1
#endif /* FIST_FILTER_NAME */

/*
 * EXTERNALS:
 */
extern struct file_operations wrapfs_main_fops;
extern struct file_operations wrapfs_dir_fops;
extern struct inode_operations wrapfs_main_iops;
extern struct inode_operations wrapfs_dir_iops;
#ifdef FIST_MNTSTYLE_ATTACH
extern struct inode_operations wrapfs_dir_attach_iops;
extern struct dentry_operations wrapfs_attach_dops;
extern struct file_operations wrapfs_dir_attach_fops;
extern super_block_t * wrapfs_read_super(super_block_t *sb, void *raw_data, int silent);
extern int wrapfs_statfs(super_block_t *sb, struct statfs *buf);
extern dentry_t *wrapfs_parse_options(super_block_t *sb, char *options);
extern void wrapfs_d_release(dentry_t *dentry);
extern int wrapfs_d_delete(dentry_t *dentry);
#ifdef FIST_DEBUG
extern void wrapfs_d_iput(dentry_t *dentry, inode_t *inode);
#endif /* FIST_DEBUG */
#endif /* FIST_MNTSTYLE_ATTACH */
extern struct inode_operations wrapfs_symlink_iops;
extern struct super_operations wrapfs_sops;
extern struct dentry_operations wrapfs_dops;
extern struct vm_operations_struct wrapfs_shared_vmops;
extern struct vm_operations_struct wrapfs_private_vmops;
extern struct address_space_operations wrapfs_aops;

extern int wrapfs_interpose(dentry_t *lower_dentry, dentry_t *this_dentry, super_block_t *sb, int flag);
#if defined(FIST_FILTER_DATA) || defined(FIST_FILTER_SCA)
extern page_t *wrapfs_get1page(file_t *file, int index);
extern int wrapfs_fill_zeros(file_t *file, page_t *page, unsigned from);
#endif /* FIST_FILTER_DATA || FIST_FILTER_SCA */
#ifdef FIST_FILTER_DATA
extern int wrapfs_decode_block(const char *from, char *to, int len, const vnode_t *vp, const vfs_t *vfsp, u_long pagenum);
extern int wrapfs_encode_block(const char *from, char *to, int len, const vnode_t *vp, const vfs_t *vfsp, u_long pagenum);
#endif /* FIST_FILTER_DATA */
#ifdef FIST_FILTER_NAME
extern int wrapfs_decode_filename(const char *name, int length, char **decoded_name, int skip_dots, const vnode_t *vp, const vfs_t *vfsp);
extern int wrapfs_encode_filename(const char *name, int length, char **encoded_name, int skip_dots, const vnode_t *vp, const vfs_t *vfsp);
#endif /* FIST_FILTER_NAME */
#ifdef FIST_DYNAMIC_INODE_NUMBERS
extern ino_t wrapfs_iunique(struct super_block *sb, ino_t maxreserved);
#endif /* FIST_DYNAMIC_INODE_NUMBERS */

#ifdef FIST_DEBUG
#define wrapfs_lower_dentry(d) __wrapfs_lower_dentry(__FILE__,__FUNCTION__,__LINE__,(d))
extern dentry_t *__wrapfs_lower_dentry(const char *file, const char *func, int line, dentry_t *this_dentry);
#else /* not FIST_DEBUG */
#define wrapfs_lower_dentry(dentry) DENTRY_TO_LOWER(dentry)
#endif /* not FIST_DEBUG */

#ifdef FIST_USE_AUX_SRC
extern int wrapfs_read_file(const char *filename, void *buf, int len);
extern int wrapfs_write_file(const char *filename, void *buf, int len);
extern dentry_t *fist_lookup(dentry_t *dir, const char *name, vnode_t **out, uid_t uid, gid_t gid);
#endif /* FIST_USE_AUX_SRC */

#ifdef FIST_FILTER_SCA
extern dentry_t *wrapfs_idx_lookup(dentry_t *lower_dentry);
extern int wrapfs_idx_create(inode_t *lower_dir, dentry_t *dentry, int mode, struct nameidata *nd);
extern int wrapfs_idx_open(dentry_t *dentry, unsigned int flags);
extern int wrapfs_idx_read(inode_t *inode);
extern int wrapfs_idx_write(inode_t *inode);
extern void wrapfs_idx_update(struct scafs_header *hdr, unsigned long entry, int delta);
#define wrapfs_idx_release(inode) \
do { \
    fput(INODE_TO_PRIVATE(inode)->idx_file); \
    INODE_TO_PRIVATE(inode)->idx_file = NULL; \
} while (0)
extern int wrapfs_idx_set_entry(struct scafs_header *hdr, int chunk_idx, off_t encoded_chunk_size);
extern int wrapfs_count_lower_pages(unsigned long page_index, inode_t *inode,
                                    int *lower_page_index, void **opaque);
extern int wrapfs_decode_buffers(int num_lower_pages, char **lower_pages_data, char *page_data, inode_t *inode, vfs_t *vfs, void *opaque);
extern int wrapfs_encode_buffers(char *lower_pages_data, char *page_data, int *need_to_call, unsigned to, inode_t *inode, vfs_t *vfs, void **opaque);
extern void * wrapfs_krealloc(void *oldptr, size_t oldsize, size_t newsize);
extern int wrapfs_copy_encoded_data(file_t *file, off_t dst_offset, encoded_pages_t *encoded_pages, off_t total_bytes_to_shift);

#ifdef FIST_DEBUG
extern void fist_print_scafs_header(const char *str, const struct scafs_header *hdr);
#else /* not FIST_DEBUG */
#define fist_print_scafs_header(a, b)
#endif /* not FIST_DEBUG */

#endif /* FIST_FILTER_SCA */


#ifdef FIST_FILTER_SCA
/* XXX must be updated for fast tails when we do writing */
#define copy_inode_size(dst, src) \
    if (INODE_TO_PRIVATE(dst)->hdr.num_alloc > 0) { \
        i_size_write(dst, INODE_TO_PRIVATE(dst)->hdr.real_size); \
	/* XXX by casting to unsigned we screw up LARGEFILE support */ \
	fist_dprint(8, "file size is %d\n", i_size_read(dst)); \
        dst->i_blocks = ((unsigned)(i_size_read(dst)) + 1023) / 512; \
    } else { \
        i_size_write(dst, i_size_read((struct inode *) src)); \
        dst->i_blocks = src->i_blocks; \
    }
#else  /* not FIST_FILTER_SCA */
#define copy_inode_size(dst, src) \
    i_size_write(dst, i_size_read((struct inode *) src)); \
    dst->i_blocks = src->i_blocks;
#endif /* not FIST_FILTER_SCA */

static inline void
fist_copy_attr_atime(inode_t *dest, const inode_t *src)
{
	ASSERT(dest != NULL);
	ASSERT(src != NULL);
	dest->i_atime = src->i_atime;
}
static inline void
fist_copy_attr_ctime(inode_t *dest, const inode_t *src)
{
    ASSERT(dest != NULL);
    ASSERT(src != NULL);
    dest->i_ctime = src->i_ctime;
}
static inline void
fist_copy_attr_times(inode_t *dest, const inode_t *src)
{
	ASSERT(dest != NULL);
	ASSERT(src != NULL);
	dest->i_atime = src->i_atime;
	dest->i_mtime = src->i_mtime;
	dest->i_ctime = src->i_ctime;
}
static inline void
fist_copy_attr_timesizes(inode_t *dest, const inode_t *src)
{
	ASSERT(dest != NULL);
	ASSERT(src != NULL);
	dest->i_atime = src->i_atime;
	dest->i_mtime = src->i_mtime;
	dest->i_ctime = src->i_ctime;
	copy_inode_size(dest, src);
}
static inline void
fist_copy_attr_all(inode_t *dest, const inode_t *src)
{
	print_entry_location();
	ASSERT(dest != NULL);
	ASSERT(src != NULL);
	dest->i_mode = src->i_mode;
	dest->i_nlink = src->i_nlink;
	dest->i_uid = src->i_uid;
	dest->i_gid = src->i_gid;
	dest->i_rdev = src->i_rdev;
	dest->i_atime = src->i_atime;
	dest->i_mtime = src->i_mtime;
	dest->i_ctime = src->i_ctime;
	dest->i_blksize = src->i_blksize;
	dest->i_blkbits = src->i_blkbits;
	copy_inode_size(dest, src);

	dest->i_flags = src->i_flags;
	print_exit_location();
}

/* auto-generated extern definitions */
FIST_AUTO_GENERATED_EXTERNS

#endif /* __KERNEL__ */

/*
 * Definitions for user and kernel code
 */

/* ioctls */
FIST_IOCTL_DEFS

#endif	/* not __WRAPFS_H_ */

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
