/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 * Copyright (c) 1992 by Sun Microsystems, Inc.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#ifdef FISTGEN
# include "fist_wrapfs.h"
#endif /* FISTGEN */
#include <fist.h>
#include <wrapfs.h>

#if 0
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/proc.h>
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <sys/uio.h>
#include <sys/kmem.h>
#include <sys/cred.h>
#include <sys/statvfs.h>
#include <sys/mount.h>
#include <sys/tiuser.h>
#include <sys/cmn_err.h>
#include <sys/debug.h>
#include <sys/mkdev.h>
#include <sys/systm.h>
#include <sys/sysmacros.h>
#include <sys/pathname.h>
#include <sys/fs_subr.h>

#include <fist.h>
#include <wrapfs.h>
#endif

int wrapfs_init(vfssw_t *, int);

/*
 * This is the loadable module wrapfs.
 */
#include <sys/modctl.h>

extern vfsops_t wrapfs_vfsops;
int wrapfs_major;
int wrapfs_minor;
kmutex_t wrapfs_minor_lock;

static vfssw_t vfw =
{
  "wrapfs",
  wrapfs_init,
  &wrapfs_vfsops,
  0
};

/*
 * Module linkage information for the kernel.
 */
extern struct mod_ops mod_fsops;

static struct modlfs modlfs =
{
  &mod_fsops,
  "filesystem for wrapfs",
  &vfw
};

static struct modlinkage modlinkage =
{
  MODREV_1,
  {
    (void *) &modlfs,
    NULL,
  }
};

/*
 * This is the module initialization routine.
 */

/*
 * A counter for how many filesystems are mounted using this counter.
 * If greater than 0, it means the module is unloadable.
 */
static int module_keepcnt = 0;

/*
 * There are not enough stubs for rpcmod so we must force load it
 */
char _depends_on[] = "strmod/rpcmod";

int
_init(void)
{
  print_location();
  return (mod_install(&modlinkage));
}

int
_fini(void)
{
  if (module_keepcnt != 0) {
    fist_dprint(4, "wrapfs_vfsops: _fini(), module_keepcnt=%d.\n",
		module_keepcnt);
    return (EBUSY);
  }
  fist_dprint(4, "wrapfs_vfsops: _fini(): removing module.\n");

  return (mod_remove(&modlinkage));
}

int
_info(struct modinfo *modinfop)
{
  int error;

  fist_dprint(4, "wrapfs_vfsops: _info(): returning info on module.\n");
  error = mod_info(&modlinkage, modinfop);
  fist_dprint(4, "wrapfs_vfsops: module wrapfs successfully loaded at address 0x%x, size 0x%x\n",
	      modinfop->mi_base, modinfop->mi_size);
  return error;
}

static kmutex_t fist_wrapfs_node_count_lock;
extern struct vnodeops wrapfs_vnodeops;
static int wrapfs_fstype;

/*
 * fist_wrapfs vfs operations.
 */
static int wrapfs_mount(vfs_t *, vnode_t *, struct mounta *, cred_t *);
static int wrapfs_unmount(vfs_t *, cred_t *);
static int wrapfs_root(vfs_t *, vnode_t **);
static int wrapfs_statvfs(vfs_t *, struct statvfs64 *);
static int wrapfs_sync(vfs_t *, short, cred_t *);
static int wrapfs_vget(vfs_t *, vnode_t **, fid_t *);
static int wrapfs_mountroot(vfs_t *, enum whymountroot);
static int wrapfs_swapvp(vfs_t *, vnode_t **, char *);

struct vfsops wrapfs_vfsops =
{
  wrapfs_mount,			/* mount */
  wrapfs_unmount,		/* unmount */
  wrapfs_root,			/* root */
  wrapfs_statvfs,		/* statvfs */
  wrapfs_sync,			/* sync */
  wrapfs_vget,			/* vget */
  wrapfs_mountroot,		/* mountroot */
  wrapfs_swapvp			/* swapvp */
};

#define	MAJOR_MIN 128
#define	vfsNVFS   (&vfssw[nfstype])


int
wrapfs_init(
	    vfssw_t * vswp,
	    int fstype
)
{

  fist_dprint(5, "wrapfs_init: vspw %x, fstype %d\n",
	      (int) vswp, fstype);
  fist_set_debug_value(4);

  if (vswp) {
    if (vswp->vsw_name)
      fist_dprint(5, "wrapfs_init: vswp->vsw_name \"%s\"\n",
		  vswp->vsw_name);
    if (vswp->vsw_init)
      fist_dprint(5, "wrapfs_init: vswp->vsw_init %x\n",
		  (int) vswp->vsw_init);
    fist_dprint(5, "wrapfs_init: wrapfs_init %x\n",
		(int) wrapfs_init);
    if (vswp->vsw_vfsops) {
      fist_dprint(5, "wrapfs_init: vswp->vsw_vfsops %x\n",
		  (int) vswp->vsw_vfsops);
      fist_dprint(5, "wrapfs_init: wrapfs_vfsops %x\n",
		  (int) &wrapfs_vfsops);
      if (vswp->vsw_vfsops->vfs_mount)
	fist_dprint(5, "wrapfs_init: vswp->vsw_vfsops->vfs_mount %x\n",
		    (int) vswp->vsw_vfsops->vfs_mount);
      fist_dprint(5, "wrapfs_init: wrapfs_mount %x\n",
		  (int) wrapfs_mount);
    }
  }
  wrapfs_fstype = fstype;
  ASSERT(wrapfs_fstype != 0);
  /*
   * Associate VFS ops vector with this fstype
   */
  vswp->vsw_vfsops = &wrapfs_vfsops;

  mutex_init(&wrapfs_minor_lock, "fist_wrapfs minor lock",
	     MUTEX_DEFAULT, NULL);
  mutex_init(&fist_wrapfs_node_count_lock, "fist_wrapfs_node count lock",
	     MUTEX_DEFAULT, NULL);

  /*
   * Assign unique major number for all fist_wrapfs mounts
   */
  if ((wrapfs_major = getudev()) == -1) {
    cmn_err(CE_WARN,
	    "fist_wrapfs: wrapfs_init: can't get unique device number");
    wrapfs_major = 0;
  }
  wrapfs_minor = 0;

  print_location();
  return (0);
}


static int
wrapfs_mount(
	     vfs_t * vfsp,	/* pre-made vfs structure to mount */
	     vnode_t * vp,	/* existing vnode to mount on */
	     struct mounta *uap, /* user-area mount(2) arguments */
	     cred_t * cr	/* user credentials */
)
{
  int error = 0;
  char namebuf[TYPICALMAXPATHLEN + 4];	/* +4 because of bug 1170077 */
  int len = 0;
  /* struct wrapfs_args args; */
  /* char datalen = uap->datalen; */
  struct fist_wrapfs_info *fwip;
  /* fist_wrapfs_node_t *fwnp; */
  dev_t wrapfs_dev;
  struct vnode *rootvp;
  vnode_t *interposed_vp;	/* interposed vnode */

  if (vfsp)
    fist_print_vfs("wrapfs_mount", vfsp);
  if (vp) {
    fist_dprint(4,
		"%s: wrapfs_vnodeops %x\n",
		"wrapfs_mount",
		(int) &wrapfs_vnodeops);
    fist_print_vnode("wrapfs_mount", vp);
  }
  if (uap)
    fist_print_uap("wrapfs_mount", uap);

  /*
   * Now we can increment the count of module instances.
   * meaning that from now, the mounting cannot fail.
   */
  ++module_keepcnt;

  /*
   * Make sure we're root
   */
  if (!suser(cr)) {
    error = EPERM;
    goto out;
  }
  /* Make sure we mount on a directory */
  if (vp->v_type != VDIR) {
    error = ENOTDIR;
    goto out;
  }

  /*
   * check if vnode is already a root of a filesystem (i.e., there
   * is already a mount on this vnode).
   */
  mutex_enter(&vp->v_lock);
  if ((uap->flags & MS_REMOUNT) == 0 &&
      (uap->flags & MS_OVERLAY) == 0 &&
      (vp->v_count != 1 || (vp->v_flag & VROOT))) {
    mutex_exit(&vp->v_lock);
    error = EBUSY;
    goto out;
  }
  mutex_exit(&vp->v_lock);

  /*
   * Get arguments: XXX: NOT NEEDED YET.
   */

  /*
   * Get vnode for interposed directory.
   */

  /* make sure special dir is a valid absolute pathname string */
  if (!uap || !uap->spec) {
    error = EINVAL;
    goto out;
  }
  /* check if special dir starts with a '/' */
  error = copyinstr(uap->spec, namebuf, TYPICALMAXPATHLEN, &len);
  if (error)
    goto out;
  if (namebuf[0] != '/') {
    error = EINVAL;
    goto out;
  }
  error = lookupname(uap->spec, UIO_USERSPACE, FOLLOW,
		     NULLVPP, &interposed_vp);
  if (error)
    goto out;
  /* Make sure the thing we just looked up is a directory */
  if (interposed_vp->v_type != VDIR) {
    VN_RELE(interposed_vp);
    error = ENOTDIR;
    goto out;
  }
/**************************************************************************
 * FIST_WRAPFSINFO:
 * The private information stored by the vfs for fist_wrapfs.
 */
  /* this implicitly allocates one vnode to be used for root vnode */
  /* XXX: enter this vnode in dnlc? */
  fwip = (struct fist_wrapfs_info *)
    kmem_alloc(sizeof(struct fist_wrapfs_info), KM_SLEEP);
  /* store the vfs of the stacked filesystem (pushed onto "stack") */
  fwip->fwi_mountvfs = vp->v_vfsp;
  /* initialize number of interposed vnodes */
  fwip->fwi_num_vnodes = 0;
  bzero((caddr_t) fwip->fwi_buckets, sizeof(fist_bucket_t *) * FIST_HT_SIZE);
  mutex_init(&fwip->fwi_ht_lock, "wrapfs HT lock", MUTEX_DEFAULT, NULL);

  /* fwip->fwi_rootvnode: is setup in the "root vnode" section below */

/**************************************************************************
 * VFS FOR THE FIST_WRAPFS FILESYSTEM:
 */
  vfsp->vfs_bsize = vp->v_vfsp->vfs_bsize;	/* was 1024, now inherited */
  vfsp->vfs_fstype = wrapfs_fstype;
  /* Assign a unique device id to the mount */
  mutex_enter(&wrapfs_minor_lock);
  do {
    wrapfs_minor = (wrapfs_minor + 1) & MAXMIN;
    wrapfs_dev = makedevice(wrapfs_major, wrapfs_minor);
  } while (vfs_devsearch(wrapfs_dev));
  mutex_exit(&wrapfs_minor_lock);
  /* set the rest of the fields */
  vfsp->vfs_dev = wrapfs_dev;
  vfsp->vfs_fsid.val[0] = wrapfs_dev;
  vfsp->vfs_fsid.val[1] = wrapfs_fstype;
  vfsp->vfs_bcount = 0;
  /* store private fist_wrapfs info in the pre-made vfs */
  vfsp->vfs_data = (caddr_t) fwip;
  /* fill in the vnode we are mounted on, in the vfs */
  vfsp->vfs_vnodecovered = vp;

/**************************************************************************
 * FIST_WRAPFSNODE:
 * The private information stored by interposing vnodes.
 * The interposing vnode here is the new root vnode of fist_wrapfs.  It
 * interposes upon the uap->spec vnode we are mounting on (the directory,
 * or partition interposed upon).
 */

/**************************************************************************
 * ROOT VNODE OF FIST_WRAPFS:
 */
  rootvp = wrapfs_interpose(interposed_vp, vfsp);
  /* this is a root vnode of this filesystem */
  rootvp->v_flag |= VROOT;
  fwip->fwi_rootvnodep = rootvp;

/**************************************************************************
 * VNODE MOUNTED UPON:
 */
  /* this vnode to mount on is a mount point for fist_wrapfs */
  vp->v_vfsmountedhere = vfsp;

  /* print values after we change them */
  if (vfsp)
    fist_print_vfs("wrapfs_mount2", vfsp);
  if (vp)
    fist_print_vnode("wrapfs_mount2", vp);
  if (rootvp)
    fist_print_vnode("wrapfs_mount2rvp", rootvp);
  fist_print_vnode("wrapfs_mount2rvn", fwip->fwi_rootvnodep);

  print_location();
  return (error);

out:
  /*
   * Cleanup our mess
   */
  module_keepcnt--;
  print_location();
  return (error);
}


/*
 * Undo fist_wrapfs mount
 */
static int
wrapfs_unmount(
	       vfs_t * vfsp,
	       cred_t * cr
)
{
  struct fist_wrapfs_info *fwip;
  vnode_t *vnp;

  fist_dprint(4, "wrapfs_unmount vfsp %x\n", vfsp);

  /* check if permitted */
  if (!suser(cr)) {
    return (EPERM);
  }

  /* free memory used by private data of vfs */
  if (!vfsp->vfs_data) {
    goto out;
  }
  fwip = vfstofwi(vfsp);
  if (fwip->fwi_num_vnodes > 1) {	/* always at least 1 b/c of rootvp */
    return (EBUSY);
  }
  fist_print_vnode("wrapfs_unmount rvp", fwip->fwi_rootvnodep);
  if (fwip->fwi_rootvnodep->v_count != 1) {
    fist_dprint(4,
		"wrapfs_unmount root vnode is busy %d\n",
		fwip->fwi_rootvnodep->v_count);
    return (EBUSY);
  }

  vnp = vfsp->vfs_vnodecovered;
  fist_dprint(4, "WRAPFS_UMOUNT: vnodecovered 0x%x has count %d\n", vnp, vnp->v_count);
  /* XXX: need to VN_RELE(vnp); */
  vnp->v_vfsmountedhere = NULL;

  VN_RELE(fwip->fwi_rootvnodep); /* will free last entry in vp HT */

  /* check if HT is really empty */
  ASSERT(fist_ht_empty_assert(vfsp) == 1);
  mutex_destroy(&fwip->fwi_ht_lock);

  /* free private data */
  kmem_free(fwip, sizeof(struct fist_wrapfs_info));

out:
  module_keepcnt--;
  return (0);
}


/*
 * find root of fist_wrapfs
 */
static int
wrapfs_root(
	    vfs_t * vfsp,
	    vnode_t ** vpp
)
{
  struct fist_wrapfs_info *fwip = NULL;

  fist_dprint(4, "wrapfs_root vfsp %x vnode %x\n",
	      vfsp, *vpp);

  fwip = vfstofwi(vfsp);
  ASSERT(fwip != NULL);
  *vpp = fwip->fwi_rootvnodep;
  fist_dprint(4, "wrapfs_root rootvnode %x\n", *vpp);

  /*
   * Should I apply the same operation to the hidden filesystem?
   * probably not, since this root vnode can lead me to the root vnode it
   * was interposed upon in vnodep->v_data->...
   * ANSWER: no!
   */
  /* error = VFS_ROOT(vfsp, vpp); */
  fist_dprint(4, "WRAPFS_ROOT1 root count %d, hidden count %d\n",
	      (*vpp)->v_count, vntofwn(*vpp)->fwn_vnodep->v_count);
#if 1
  /* XXX: is this right?  Should I HOLD a vnode everytime root is called? */
  /* probably ok, since we are holding the root vnode for fist_wrapfs... */
  VN_HOLD(*vpp);
#else
  if (*vpp) {
    fist_print_vnode("wrapfs_root", *vpp);
  }
  /* mutex_enter(&(*vpp)->v_lock); */
  /* (*vpp)->v_count++; */
  /* mutex_exit(&(*vpp)->v_lock); */
#endif

  fist_dprint(4, "WRAPFS_ROOT2 root count %d, hidden count %d\n",
	      (*vpp)->v_count, vntofwn(*vpp)->fwn_vnodep->v_count);

  print_location();
  return (0);
}


/*
 * Get file system statistics.
 */
static int
wrapfs_statvfs(
	       register vfs_t * vfsp,
	       struct statvfs64 * sbp
)
{
  int error = EPERM;
  struct fist_wrapfs_info *fwip = NULL;
  vfs_t *hidden_vfsp;

  fist_dprint(4, "wrapfs_statvfs %x\n", vfsp);

  if (vfsp) {
    /* find hidden vfs */
    fwip = vfstofwi(vfsp);
    ASSERT(fwip != NULL);
    hidden_vfsp = fwip->fwi_mountvfs;

    /* pass operation to hidden filesystem, and return status */
    error = VFS_STATVFS(hidden_vfsp, sbp);
  } else {			/* error: vfsp is invalid */
    error = EINVAL;
  }

  print_location();
  return (error);
}


/*
 * Flush any pending I/O to file system vfsp.
 * The wrapfs_update() routine will only flush *all* fist_wrapfs files.
 * If vfsp is non-NULL, only sync this fist_wrapfs (in preparation
 * for a umount).
 */
static int
wrapfs_sync(vfs_t *vfsp, short flag, struct cred *cr)
{
  /* commented the print of the sync since i got annoyed a  message when we
   * only return 0 */
#if 0
  fist_dprint(4, "wrapfs_sync %x\n", vfsp);
#endif
  /* do not sync() lower layer f/s.  It'll be sync'ed on its own */
  return 0;

#if 0
  /* if vfsp is NULL, then this is a generic "sync()", treat it as ok */
  if (vfsp) {
    /* find hidden vfs */
    fwip = vfstofwi(vfsp);
    ASSERT(fwip != NULL);
    hidden_vfsp = fwip->fwi_mountvfs;

    /* pass operation to hidden filesystem, and return status */
    error = VFS_SYNC(hidden_vfsp, flag, cr);
  }
  print_location();
  return (error);
#endif
}


static int
wrapfs_vget(
	    vfs_t *vfsp,
	    vnode_t **vpp,
	    struct fid *fidp
)
{
  int error = EPERM;
  struct fist_wrapfs_info *fwip = NULL;
  vfs_t *hidden_vfsp;

  fist_dprint(4, "wrapfs_vget %x\n", vfsp);

  if (vfsp) {
    /* find hidden vfs */
    fwip = vfstofwi(vfsp);
    ASSERT(fwip != NULL);
    hidden_vfsp = fwip->fwi_mountvfs;

    /* pass operation to hidden filesystem, and return status */
    error = VFS_VGET(hidden_vfsp, vpp, fidp);
  } else {			/* error: vfsp is invalid */
    error = EINVAL;
  }

  print_location();
  return (error);
}


/*
 * Mount root file system.
 * "why" is ROOT_INIT on initial call, ROOT_REMOUNT if called to
 * remount the root file system, and ROOT_UNMOUNT if called to
 * unmount the root (e.g., as part of a system shutdown).
 *
 * XXX - this may be partially machine-dependent; it, along with the
 * VFS_SWAPVP operation, goes along with auto-configuration.  A mechanism
 * should be provided by which machine-INdependent code in the kernel can
 * say "get me the right root file system" and "get me the right initial
 * swap area", and have that done in what may well be a machine-dependent
 * fashion.  Unfortunately, it is also file-system-type dependent (NFS gets
 * it via bootparams calls, FIST_WRAPFS gets it from various and sundry
 * machine-dependent mechanisms, as SPECFS does for swap).
 */
static int
wrapfs_mountroot(
		 vfs_t *vfsp,
		 enum whymountroot why
)
{
  /*
   * XXX: stackable filesystems cannot be used to as root filesystems, for
   * now.
   * -Erez Zadok <ezk@cs.sunysb.edu>
   */
  fist_dprint(4, "wrapfs_mountroot %x\n", vfsp);

  print_location();
  return (EINVAL);
}


static int
wrapfs_swapvp(
	      vfs_t *vfsp,
	      vnode_t **vpp,
	      char *nm
)
{
  int error = EPERM;
  struct fist_wrapfs_info *fwip = NULL;
  vfs_t *hidden_vfsp;

  fist_dprint(4, "wrapfs_swapvp %x\n", vfsp);

  if (vfsp) {
    /* find hidden vfs */
    fwip = vfstofwi(vfsp);
    ASSERT(fwip != NULL);
    hidden_vfsp = fwip->fwi_mountvfs;

    /* pass operation to hidden filesystem, and return status */
    error = VFS_SWAPVP(hidden_vfsp, vpp, nm);
  } else {			/* error: vfsp is invalid */
    error = EINVAL;
  }

  print_location();
  return (error);
}
