/*
 * Copyright (c) 1997-2005 Erez Zadok <ezk@cs.stonybrook.edu>
 * Copyright (c) 2001-2005 Stony Brook University
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package, or get one from ftp://ftp.filesystems.org/pub/fistgen/COPYING.
 *
 * This Copyright notice must be kept intact and distributed with all
 * fistgen sources INCLUDING sources generated by fistgen.
 */
/*
 * validate.c: Validate fist syntax.
 * Fistgen sources.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */


/****************************************************************************/

/*
 * List of valid fist rule definition components: callset, optype, and part.
 */
char *valid_rule_callsets[] = {
  "op",
  "ops",
  "readops",
  "writeops",
  NULL				/* must be last entry */
};

char *valid_rule_optypes[] = {
  "all",
  "data",
  "name",
  /* the rest of the ops mirror the NFS v.2 protocol */
  "create",
  "getattr",
  "lstat",
  "stat",
  "link",
  "lookup",
  "mkdir",
  "read",
  "readdir",
  "readlink",
  "rename",
  "rmdir",
  "setattr",
  "statfs",
  "symlink",
  "unlink",
  "write",
  /* now some more ops that correspond to certain system calls */
  "ioctl",
  NULL				/* must be last entry */
};

char *valid_rule_parts[] = {
  "precall",
  "call",
  "postcall",
  NULL				/* must be last entry */
};


/****************************************************************************/

/*
 * Validate accessmode, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_accessmode(const char *str)
{
  if (STREQ(str, "readonly")) {
    fist_globals.fg_accessmode = FF_ACCESSMODE_READONLY;
    return TRUE;
  }
  if (STREQ(str, "writeonly")) {
    fist_globals.fg_accessmode = FF_ACCESSMODE_WRITEONLY;
    return TRUE;
  }
  if (STREQ(str, "readwrite")) {
    fist_globals.fg_accessmode = FF_ACCESSMODE_READWRITE;
    return TRUE;
  }
  return FALSE;
}


/*
 * Validate debug flag, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_debug(const char *str)
{
  if (STREQ(str, "on")) {
    fist_globals.fg_debug = 1;
    return TRUE;
  }
  if (STREQ(str, "off")) {
    fist_globals.fg_debug = 0;
    return TRUE;
  }
  return FALSE;
}


/*
 * Validate dynamic_inode_numbers flag, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_dynamic_inode_numbers(const char *str)
{
  if (STREQ(str, "on")) {
    fist_globals.fg_dynamic_inode_numbers = 1;
    return TRUE;
  }
  if (STREQ(str, "off")) {
    fist_globals.fg_dynamic_inode_numbers = 0;
    return TRUE;
  }
  return FALSE;
}


/*
 * Validate filter names, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_filter(const char *str)
{
  if (STREQ(str, "data")) {
    fist_globals.fg_filter |= FF_FILTER_DATA;
    return TRUE;
  }
  if (STREQ(str, "name")) {
    fist_globals.fg_filter |= FF_FILTER_NAME;
    return TRUE;
  }
  if (STREQ(str, "sca")) {
    fist_globals.fg_filter |= FF_FILTER_SCA; /* size changing algorithm */
    return TRUE;
  }
  return FALSE;
}


/*
 * Validate mntstyle flag, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_mntstyle(const char *str)
{
  if (STREQ(str, "regular")) {
    fist_globals.fg_mntstyle = FF_MNTSTYLE_REGULAR;
    return TRUE;
  }
  if (STREQ(str, "overlay")) {
    fist_globals.fg_mntstyle = FF_MNTSTYLE_OVERLAY;
    return TRUE;
  }
  if (STREQ(str, "attach")) {
    fist_globals.fg_mntstyle = FF_MNTSTYLE_ATTACH;
    return TRUE;
  }
  return FALSE;
}


/*
 * Validate fanout, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_fanout(int fo)
{
  if (fo < 1 || fo > MAX_FAN_OUT)
    return FALSE;

  fist_globals.fg_fanout = fo;
  return TRUE;
}


/*
 * Validate encoding-type, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_encoding_type(const char *str)
{
  if (STREQ(str, "none")) {
    fist_globals.fg_encoding_type = FF_ENCODING_TYPE_NONE;
    return TRUE;
  }
  if (STREQ(str, "stream")) {
    fist_globals.fg_encoding_type = FF_ENCODING_TYPE_STREAM;
    return TRUE;
  }
  if (STREQ(str, "block")) {
    fist_globals.fg_encoding_type = FF_ENCODING_TYPE_BLOCK;
    return TRUE;
  }
  return FALSE;
}


/*
 * Validate encoding-blocksize, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_encoding_blocksize(int ebs)
{
  if (ebs < 1 || ebs > MAX_ENCODING_BLOCKSIZE)
    return FALSE;

  fist_globals.fg_encoding_blocksize = ebs;
  return TRUE;
}


/*
 * Validate errorcodes, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_errorcode(const char *str, char *errmsg)
{
  int cur = fist_globals.fg_num_errorcodes;
  int i;

  /* check if exceeded maximum number of error codes */
  if (cur >= MAX_ERRORCODES) {
    sprintf(errmsg, "too many error codes (max=%d)", MAX_ERRORCODES);
    return FALSE;
  }

  /* check if error code is already defined */
  if (cur > 0)
    for (i=0; i<cur; i++) {
      if (STREQ(str, fist_globals.fg_errorcodes[i])) {
	strcpy(errmsg, "duplicate error code");
	return FALSE;
      }
    }

  /* all is well */
  fist_globals.fg_errorcodes[cur] = strdup(str);
  fist_globals.fg_num_errorcodes++; /* increment */
  //  fprintf(out_fp_h, "#define %s (LAST_OS_ERRNO+%d)\n", str, cur+1);

  return TRUE;
}


/*
 * Validate module sources, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_msources(const char *str, char *errmsg)
{
  int cur = fist_globals.fg_num_msources;
  int i;

  /* check if exceeded maximum number of error codes */
  if (cur >= MAX_SOURCES) {
    sprintf(errmsg, "too many additional module sources (max=%d)", MAX_SOURCES);
    return FALSE;
  }

  /* check if error code is already defined */
  if (cur > 0)
    for (i=0; i<cur; i++) {
      if (STREQ(str, fist_globals.fg_msources[i])) {
	sprintf(errmsg, "duplicate module source file: %s", str);
	return FALSE;
      }
    }

  /* check if really a .c file */
  if (!STREQ(&str[strlen(str)-2], ".c")) {
    sprintf(errmsg, "not a C source file: %s", str);
    return FALSE;
  }

  /* all is well */
  fist_globals.fg_msources[cur] = strdup(str);
  fist_globals.fg_num_msources++; /* increment */

  return TRUE;
}


/*
 * Validate module headers, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_mheaders(const char *str, char *errmsg)
{
  int cur = fist_globals.fg_num_mheaders;
  int i;

  /* check if exceeded maximum number of error codes */
  if (cur >= MAX_SOURCES) {
    sprintf(errmsg, "too many additional module headers (max=%d)", MAX_SOURCES);
    return FALSE;
  }

  /* check if error code is already defined */
  if (cur > 0)
    for (i=0; i<cur; i++) {
      if (STREQ(str, fist_globals.fg_mheaders[i])) {
	sprintf(errmsg, "duplicate module source file: %s", str);
	return FALSE;
      }
    }

  /* check if really a .c file */
  if (!STREQ(&str[strlen(str)-2], ".h")) {
    sprintf(errmsg, "not a C header file: %s", str);
    return FALSE;
  }

  /* all is well */
  fist_globals.fg_mheaders[cur] = strdup(str);
  fist_globals.fg_num_mheaders++; /* increment */

  return TRUE;
}


/*
 * Validate user-level sources, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_usources(const char *str, char *errmsg)
{
  int cur = fist_globals.fg_num_usources;
  int i, len;
  char *cp;

  /* check if exceeded maximum number of error codes */
  if (cur >= MAX_SOURCES) {
    sprintf(errmsg, "too many additional user-level sources (max=%d)", MAX_SOURCES);
    return FALSE;
  }

  /* check if error code is already defined */
  if (cur > 0)
    for (i=0; i<cur; i++) {
      if (STREQ(str, fist_globals.fg_usources[i])) {
	sprintf(errmsg, "duplicate user-level source file: %s", str);
	return FALSE;
      }
    }

  /* valid sources end with .h or .c */
  len = strlen(str);
  if (len < 3) {
    sprintf(errmsg, "invalid length foe C source/header file name: %s", str);
    return FALSE;
  }
  cp = (char *) &str[len-2];
  if (!STREQ(cp, ".c") && !STREQ(cp, ".h")) {
    sprintf(errmsg, "invalid C source/header file name: %s", str);
    return FALSE;
  }

  /* all is well */
  fist_globals.fg_usources[cur] = strdup(str);
  fist_globals.fg_num_usources++; /* increment */

  return TRUE;
}


/*
 * Validate added makefile, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_addmk(const char *str, char *errmsg)
{
  struct stat statbuf;
  char buf[MAX_BUF_LEN];

  /* check if already defined one additional makefile */
  if (fist_globals.fg_addmk) {
    sprintf(errmsg, "must specify only one additional makefile");
    return FALSE;
  }

  sprintf(buf, "%s/%s", in_dir, str);
  if (lstat(buf, &statbuf) < 0) {
    sprintf(errmsg, "\"%s\": %s", str, strerror(errno));
    return FALSE;
  }

  /* all is well */
  fist_globals.fg_addmk = strdup(buf);
  return TRUE;
}


/*
 * Validate licese string, and perform necessary actions.
 * Return TRUE/FALSE.
 *
 * Note: any string is really valid.  It's up to the author to decide what
 * they want.  But if on Linux you pick something other than the approved
 * licenses as listed in <linux/module.h>, your module may not link into the
 * kernel if it's using GPL-only symbols.
 */
int
fist_validate_decl_license(const char *str, char *errmsg)
{
  struct stat statbuf;
  int len = 0;

  /* check if already defined another license */
  if (fist_globals.fg_license) {
    sprintf(errmsg, "must specify only one license string");
    return FALSE;
  }

  /* check if null, just to be sure */
  if (!str) {
    sprintf(errmsg, "must specify a non-null license string");
    return FALSE;
  }

  /* check if it's a string */
  len = strlen(str);
  if (str[0] != '"' || (len > 1 && str[len-1] != '"')) {
    sprintf(errmsg, "license string must be \"double-quoted\"");
    return FALSE;
  }
  if (STREQ(str, "\"\"")) {
    sprintf(errmsg, "license must be a non empty string");
    return FALSE;
  }

  /* all is well */
  fist_globals.fg_license = strdup(str);
  return TRUE;
}


/*
 * Validate mntflags, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_mntflag(const char *str, char *errmsg)
{
  int cur = fist_globals.fg_num_mntflags;
  int i;

  /* check if exceeded maximum number of mount flags */
  if (cur >= MAX_MNTFLAGS) {
    sprintf(errmsg, "too many mount flags (max=%d)", MAX_MNTFLAGS);
    return FALSE;
  }

  /* check if mount flag is already defined */
  if (cur > 0)
    for (i=0; i<cur; i++) {
      if (STREQ(str, fist_globals.fg_mntflags[i])) {
	strcpy(errmsg, "duplicate mount flag");
	return FALSE;
      }
    }

  /* all is well */
  fist_globals.fg_mntflags[cur] = strdup(str);
  fist_globals.fg_num_mntflags++; /* increment */
  //  fprintf(out_fp_h, "#define %s %d\n", str, cur+1);

  return TRUE;
}


/*
 * Validate file formats, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_fileformat(const char *str, bdt_t *bdt, char *errmsg)
{
  int cur = fist_globals.fg_num_fileformats;
  int i;

  /* check if exceeded maximum number of file formats */
  if (cur >= MAX_FILE_FORMATS) {
    sprintf(errmsg, "too many file formats (max=%d)", MAX_FILE_FORMATS);
    return FALSE;
  }

  /* check if file format is already defined */
  if (cur > 0)
    for (i=0; i<cur; i++) {
      if (STREQ(str, fist_globals.fg_fileformat_name[i])) {
	strcpy(errmsg, "duplicate file format");
	return FALSE;
      }
    }

  /* all is well */
  fist_globals.fg_fileformat_name[cur] = strdup(str);
  fist_globals.fg_fileformat[cur] = bdt;
  fist_globals.fg_num_fileformats++; /* increment */

  return TRUE;
}


/*
 * check if an ioctl name exists
 * return TRUE if name exists, FALSE otherwise
 */
static int
fist_exist_ioctl_name(const char *str)
{
  int i;
  int cur = fist_globals.fg_num_ioctls;

  if (cur == 0)			/* no ioctls defined */
    return FALSE;

  for (i=0; i<cur; i++)
    if (STREQ(str, fist_globals.fg_ioctl_name[i]))
      return TRUE;

  return FALSE;
}


/*
 * Validate ioctls, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_decl_ioctl(const char *str, const char *type, bdt_t *bdt, char *errmsg)
{
  int cur = fist_globals.fg_num_ioctls;
  u_int iot;

  /* check if ioctl type is valid */
  if (STREQ(type, "both"))
    iot = FF_IOCTL_BOTH;
  else if (STREQ(type, "fromuser"))
    iot = FF_IOCTL_FROMUSER;
  else if (STREQ(type, "touser"))
    iot = FF_IOCTL_TOUSER;
  else if (STREQ(type, "none"))
    iot = FF_IOCTL_NONE;
  else {
    strcpy(errmsg, "unknown ioctl type");
    return FALSE;
  }

  /* check if exceeded maximum number of ioctls */
  if (cur >= MAX_IOCTLS) {
    sprintf(errmsg, "too many ioctls (max=%d)", MAX_IOCTLS);
    return FALSE;
  }

  /* check if ioctl is already defined */
  if (fist_exist_ioctl_name(str)) {
    strcpy(errmsg, "duplicate ioctl");
    return FALSE;
  }

  /* all is well */
  fist_globals.fg_ioctl_name[cur] = strdup(str);
  fist_globals.fg_ioctl[cur] = bdt;
  fist_globals.fg_ioctl_type[cur] = iot;
  fist_globals.fg_num_ioctls++; /* increment */

  return TRUE;
}


/*
 * Validate the 'callset' part of a fist rule definition.
 * Return TRUE/FALSE.
 */
static int
fist_validate_rule_callset(const char *str)
{
  char **tmp = valid_rule_callsets;

  if (!str || !str[0])
    return FALSE;

  while (*tmp) {
    if (STREQ(*tmp, str))
      return TRUE;
    tmp++;
  }
  return FALSE;
}


/*
 * Validate the 'optype' part of a fist rule definition.
 * Return TRUE/FALSE.
 */
static int
fist_validate_rule_optype(const char *str)
{
  char **tmp = valid_rule_optypes;

  if (!str || !str[0])
    return FALSE;

  while (*tmp) {
    if (STREQ(*tmp, str))
      return TRUE;
    tmp++;
  }
  return FALSE;
}


/*
 * Validate the 'part' part of a fist rule definition.
 * Return TRUE/FALSE.
 */
int
fist_validate_rule_part(const char *str)
{
  char **tmp = valid_rule_parts;

  if (!str || !str[0])
    return FALSE;

  while (*tmp) {
    if (STREQ(*tmp, str))
      return TRUE;
    tmp++;
  }
  return FALSE;
}


/*
 * Validate fist rule definition, and perform necessary actions.
 * Return TRUE/FALSE.
 */
int
fist_validate_rule_def(const char *str, char *errmsg, ecl_t **eclpp)
{
  int cur = fist_rules.fr_num_rules;
  int i;
  char buf[MAX_BUF_LEN], *callset, *optype, *part;

  /* check if exceeded maximum number of ioctls */
  if (cur >= MAX_RULES) {
    sprintf(errmsg, "too many rules (max=%d)", MAX_RULES);
    return FALSE;
  }

  /* simple check */
  if (!str || !str[0]) {
    sprintf(errmsg, "null/empty rule definition");
    return FALSE;
  }

  /* find components of rule definition, and validate them */
  strcpy(buf, str);
  callset = optype = part = NULL;
  callset = &buf[1];		/* skip '%' */
  optype = strchr(callset, ':');
  if (optype)
    *optype++ = '\0';
  if (optype) {
    part = strchr(optype, ':');
    if (part)
      *part++ = '\0';
  }
  if (!fist_validate_rule_callset(callset)) {
    sprintf(errmsg, "invalid callset \"%s\" in rule definition", callset);
    return FALSE;
  }
  if (!fist_validate_rule_optype(optype)) {
    sprintf(errmsg, "invalid optype in rule definition");
    return FALSE;
  }

  if (STREQ(optype, "ioctl")) {
    /* also check for newly defined ioctl names */
    if (!fist_validate_rule_part(part) && !fist_exist_ioctl_name(part)) {
      sprintf(errmsg, "invalid part/name in ioctl rule definition");
      return FALSE;
    }
  } else {
    if (!fist_validate_rule_part(part)) {
      sprintf(errmsg, "invalid part in rule definition");
      return FALSE;
    }
  }

  /* check if rule is already defined */
  if (cur > 0)
    for (i=0; i<cur; i++) {
      if (STREQ(callset, fist_rules.fr_callset[i]) &&
	  STREQ(optype, fist_rules.fr_optype[i]) &&
	  STREQ(part, fist_rules.fr_part[i])) {
	strcpy(errmsg, "duplicate rule definition");
	return FALSE;
      }
    }

  /* all is well */
  fist_rules.fr_callset[cur] = strdup(callset);
  fist_rules.fr_optype[cur] = strdup(optype);
  fist_rules.fr_part[cur] = strdup(part);

  /* create uppercase name of tag */
  sprintf(buf, "%s_%s_%s", callset, optype, part);
  uppercase_string(buf);
  fist_rules.fr_tag[cur] = strdup(buf);

  fist_rules.fr_code[cur] = (ecl_t *) calloc(1, sizeof(ecl_t));
  fist_rules.fr_code[cur]->buf = (*eclpp)->buf;
  (*eclpp)->buf = NULL;
  fist_rules.fr_code[cur]->len = (*eclpp)->len;
  fist_rules.fr_code[cur]->max = (*eclpp)->max;
  (*eclpp)->len = (*eclpp)->max = 0;

  fist_rules.fr_num_rules++;

  return TRUE;
}


/*
 * Validate fist rule tag syntax such as [FIST_]OP_LOOKUP_POSTCALL.
 * Return TRUE/FALSE.
 */
int
fist_validate_rule_tag(const char *str)
{
  char buf[MAX_BUF_LEN], *callset, *optype, *part;

  /* simple check */
  if (!str || !str[0])
    return FALSE;

  /* find components of rule definition, and validate them */
  strcpy(buf, str);
  lowercase_string(buf);
  callset = optype = part = NULL;
  callset = buf;
  optype = strchr(callset, '_');
  if (optype)
    *optype++ = '\0';
  if (optype) {
    part = strchr(optype, '_');
    if (part)
      *part++ = '\0';
  }
  if (!fist_validate_rule_callset(callset)) {
    return FALSE;
  }
  if (!fist_validate_rule_optype(optype)) {
    return FALSE;
  }
  if (!fist_validate_rule_part(part)) {
    return FALSE;
  }

  return TRUE;
}
