program meas1

  use DQMC_SHEET
  use DQMC_HUBBARD
  use DQMC_CFG
  use DQMC_GTAU
  
  implicit none

  ! variables
  type(config)  :: cfg
  type(Hubbard) :: Hub                     ! Hubbard model
  type(GTAU)    :: tau
  integer       :: nx, ny, nz, n, L, nxy, nzl
  integer       :: nBin, nIter, ntausk
  integer       :: iBin, cnt, avg, err
  integer       :: i, j, k
  real(wp), pointer :: Gup(:,:), Gdn(:,:), Gavg(:,:), sgn(:)
  real          :: tt1, tt2
  real(wp), pointer  :: Aup(:,:), Adn(:,:)

  ! --------------- !
  ! Configuration   !
  ! --------------- !

  call CPU_TIME(tt1)
  ! read parameters from config file
  call DQMC_Read_Config(cfg, STDIN)
   
  ! Initialize the geometry
  call CFG_Get(cfg, "nx", nx)
  call CFG_Get(cfg, "ny", ny)
  call CFG_Get(cfg, "nz", nz)
  nxy = nx*ny
  n  = nxy*nz
  call DQMC_Init_Sheet(nx, ny, nz, Hub%S)  


  ! Initialize the rest data
  call DQMC_Hub_Config(Hub, cfg)
  Hub%meas2 = .false.

  call DQMC_Gtau_Init(Hub%n, Hub%L, TAU_T0, &
       Hub%SB%nOrth, Hub%G_up%nWrap, tau, Hub%B, Hub%WS)

  L  = Hub%L
  call CFG_Get(cfg, "nbin", nbin)
  iBin   = 1
  cnt    = 0
  avg    = nBin + 1
  err    = nBin + 2

  ! Initialize meas data
  nzl = nz*L+nz
#ifdef _BIGG
  allocate(Adn(n*L,n*L))
  allocate(Aup(n*L,n*L))
#endif

  allocate(Gup (nzl, nBin+2))
  allocate(Gdn (nzl, nBin+2))
  allocate(Gavg(nzl, nBin+2))
  allocate(sgn(nBin+2))

  Gup  = ZERO
  Gdn  = ZERO
  Gavg = ZERO
  sgn  = ZERO


#ifdef _DEBUG1
  fdebug = 30
  call date_and_time(date,time)
  write(fname,*) date//time(1:6)//".sheet.log"
  open (unit=fdebug, iostat=stat, file=fname, status='new', action='write')
  write(fdebug,*) "Debug file for sheet code at ", time(1:6), " on ", date
#endif

  ! ---------- !
  ! Execution  !
  ! ---------- !
  ! Execution MC loop
  ! Warmup sweep

  do i = 1, Hub%nWarm
     ! The second parameter means no measurement should be made.
     call DQMC_Hub_Sweep(Hub, NO_MEAS0)
  end do
  
  ! We divide all the measurement into nBin,
  ! each having nPass/nBin pass.
  call CFG_Get(cfg, "tausk", ntausk)
  nIter  = Hub%nPass/nBin/ntausk
  do i = 1, nBin
     do j = 1, nIter
        do k = 1, ntausk
           call DQMC_Hub_Sweep(Hub, Hub%nMeas)
        end do

#ifdef _BIGG
        call MeasureWithBigG()
#else
        call Measure()
#endif

     end do
     ! Accumulate results for each bin
     call Average() 
     call DQMC_Phy0_Avg(Hub%P0)
  end do

  ! -------------- !
  ! Postprocessing !
  ! -------------- !
  call GetError()
  call DQMC_Phy0_GetErr(Hub%P0)
  call DQMC_Hub_OutputParam(Hub, STDOUT)
  write(STDOUT, FMT_DBLINE) 
  call DQMC_Phy0_Print(Hub%P0, Hub%S, STDOUT)
  write(STDOUT, FMT_DBLINE)
  call PrintOut(STDOUT, Gup, "sheet, tau, Gup (sheet, tau) +- error")
  call PrintOut(STDOUT, Gdn, "sheet, tau, Gdn (sheet, tau) +- error")
  call PrintOut(STDOUT, Gavg, "sheet, tau, Gavg(sheet, tau) +- error")
  call cpu_time(tt2)
  write(STDOUT, *) "Execution time=", tt2-tt1

  ! Clean up the used storage
  call DQMC_Hub_Free(Hub)
  call DQMC_Config_Free(cfg)
  call DQMC_Gtau_Free(tau)

#ifdef _BIGG
  deallocate(Aup, Adn)
#endif

  deallocate(Gup, Gdn, Gavg, sgn)


#ifdef _DEBUG1
  close(fdebug)
#endif

contains

  ! -------------------- !
  ! supporting functios  !
  ! ---------------------!

  ! ----------------------------------------------------------------- !

  subroutine MeasureWithBigG()
    ! Purpose
    ! =======
    !    compute G(i, tau) for time dependent G_up and down
    !
    ! ... Local variables ...
    integer :: li, lj, dl, j, k, gi, idx
    real(wp) ::factor, s1(4), s2(4)
    real(wp), pointer :: up(:,:), dn(:,:)

    ! ... Executable ...
    ! compute the first row of G tau

    factor = ONE/nxy/L
    Gup(:,avg) = ZERO
    Gdn(:,avg) = ZERO
    Gavg(:,avg) = ZERO
    
    call DQMC_Gtau_Big(tau, Aup, Adn, Hub%G_up, Hub%G_dn)

    s1 = ZERO
    s2 = ZERO

    up => Aup(1:n,n+1:2*n)
    dn => Adn(1:n,n+1:2*n)
    idx = 1
    do j = 1, nz
       ! each site
       do k = 1, nxy
          s1(j)  = s1(j)  + up(idx,idx)
          s2(j)  = s2(j)  + dn(idx,idx)
          idx = idx + 1
       end do
    end do
    
    ! write(*,"(8f12.8)") s1+s2


    ! For each time slice
    do li = 0, L-1
       do lj = 0, L-1
          up => Aup(n*lj+1:n*(lj+1),n*li+1:n*(li+1)) 
          dn => Adn(n*lj+1:n*(lj+1),n*li+1:n*(li+1))
          dl = mod(L+li-lj,L)+1
          !          print *, n*lj+1, n*li+1, dl
          ! in each submatrix
          ! for each layer
          idx = 1
          do j = 1, nz
             gi = dl + (j-1)*L
             ! each site
             do k = 1, nxy
                Gup(gi, avg)  = Gup(gi, avg)  + up(idx,idx)
                Gdn(gi, avg)  = Gdn(gi, avg)  + dn(idx,idx)
                idx = idx + 1
             end do
          end do
       end do
    end do

    Gup(:, iBin)  = Gup(:, iBin)  + Gup(:,avg)*factor
    Gdn(:, iBin)  = Gdn(:, iBin)  + Gdn(:,avg)*factor
    factor = factor/2
    Gavg(:, iBin) = Gavg(:, iBin) +  (Gup(:,avg)+Gdn(:,avg))*factor
    cnt = cnt + 1
    
  end subroutine MeasureWithBigG

  ! ----------------------------------------------------------------- !

  subroutine Measure()
    ! Purpose
    ! =======
    !    compute G(i, tau) for time dependent G_up and down
    !
    ! ... Local variables ...
    integer :: i, j, k, gi, idx, L1
    real(wp) ::factor

    ! ... Executable ...
    ! compute the first row of G tau

    L1 = L + 1
    sgn(avg) = Hub%G_up%sgn*Hub%G_dn%sgn
    factor = sgn(avg)/nxy
    Gup(:,avg) = ZERO
    Gdn(:,avg) = ZERO
    Gavg(:,avg) = ZERO

    ! For each time slice
    do i = 1, L
       call DQMC_MakeGtau(tau, Hub%G_up, Hub%G_dn, 1, i-1)
       idx = 0
       ! For each sheet j
       do j = 1, nz
          gi = i+(j-1)*L1
          ! For each site k on sheet j 
          do k = idx+1, idx+nxy
             Gup(gi, avg)  = Gup(gi, avg)  + tau%upt0(k, k)
             Gdn(gi, avg)  = Gdn(gi, avg)  + tau%dnt0(k, k)
          end do
          idx = idx + nxy
       end do
    end do

    ! special case for L+1
    do i = L1, nzl, L1
       Gup(i,avg) = Gup(i,avg)  + nxy - Gup(i-L1+1,avg)
       Gdn(i,avg) = Gdn(i,avg)  + nxy - Gdn(i-L1+1,avg)
    end do

#ifdef _DEBUG1
    idx = 1
    write(fdebug, FMT_DBLINE)
    do i = 1, nz
       write(fdebug, "('Layer',i3,'       G_up           G_dn ')") i
       do j = 0, L
          write(fdebug, "(i8,2f15.10)") j, Gup(idx,avg), Gdn(idx,avg)
          idx = idx + 1
       end do
    end do
#endif

    ! average
    Gup(:, iBin)  = Gup(:, iBin)  + Gup(:,avg)*factor
    Gdn(:, iBin)  = Gdn(:, iBin)  + Gdn(:,avg)*factor
    factor = factor*HALF
    Gavg(:, iBin) = Gavg(:, iBin) +  (Gup(:,avg)+Gdn(:,avg))*factor

    sgn(iBin)     = sgn(iBin) + sgn(avg)
    cnt = cnt + 1
    
  end subroutine Measure

  ! ----------------------------------------------------------------- !

  subroutine Average()
    ! Purpose
    ! =======
    !    partially averge Green's function
    !
    ! ... Local variables ...
    real(wp) :: factor

    ! ... Executable ...
    
    ! average all the values
    factor = ONE/cnt
    Gup (:, iBin) = Gup (:, iBin)*factor
    Gdn (:, iBin) = Gdn (:, iBin)*factor
    Gavg(:, iBin) = Gavg(:, iBin)*factor
    sgn (ibin)    = sgn(iBin)*factor

    ! reset counter and adavence to next bin
    cnt = 0
    iBin = iBin + 1

  end subroutine Average

  ! ----------------------------------------------------------------- !

  subroutine GetError()
    ! Purpose
    ! =======
    !    Compute std err and average
    !
    ! ... Local variables ...
    integer  :: i
    real(wp) :: sum_sgn, z(nBin), y(nBin)

    ! ... Executable ...
    
    call DQMC_JackKnife(nBin, sgn(avg), sgn(err), sgn(1:nBin), &
         y, z, sum_sgn)

    do i = 1, nzl
       ! average all the values
       call DQMC_SignJackKnife(nBin, Gup(i, avg), Gup(i, err), &
            Gup(i, 1:nBin), y, z, sum_sgn)
       call DQMC_SignJackKnife(nBin, Gdn(i, avg), Gdn(i, err), &
            Gdn(i, 1:nBin), y, z, sum_sgn)
       call DQMC_SignJackKnife(nBin, Gavg(i, avg), Gavg(i, err), &
            Gavg(i, 1:nBin), y, z, sum_sgn)
    end do

  end subroutine GetError

  
  ! ----------------------------------------------------------------- !
  
  subroutine PrintOut(OPT, G, title)
    ! Purpose
    ! =======
    !    Print out values
    !
    ! Argument
    ! ========
    integer, intent(in)  :: OPT
    real(wp), intent(in) :: G(:,:)
    character(*), intent(in) :: title

    ! ... Local variables ...
    integer :: i, j, idx
    character(*), parameter :: OPTFMT = "(i5,i5,f20.16,'   +- ', f20.16)"

    write(OPT,*) title
    idx = 1
    do j = 1, nz
       do i = 0, L
          write(OPT,OPTFMT) j, i, G(idx,avg), G(idx,err)
          idx = idx + 1
       end do
    end do
    write(STDOUT, FMT_DBLINE)

  end subroutine PrintOut
  
  ! ----------------------------------------------------------------- !

end program meas1
