package DataExtraction;
import java.util.*;

public class Element {
	//tag: title for this element, taken from the XML tag
	private String tag;
	//value: the possible value of the XML element
	private String value;
	//elements: the children of this element, taken from the XML structure
	private List<Element> elements;
	//attributes: the attributes of this element, taken from the XML
	private List<Attribute> attributes;
	//parent: the XML parent of this element, null if this element is the root
	private Element parent;

	public Element() {
		tag = "";
		value = "";
		parent = null;
		elements = new ArrayList<Element>();
		attributes = new ArrayList<Attribute>();
	}

	public Element(String t, Element p) {
		tag = t;
		value = "";
		parent = p;
		elements = new ArrayList<Element>();
		attributes = new ArrayList<Attribute>();
	}

	public String toString() {
		return this.toString(0);
	}

	/**
	 * Returns the XML representation of this element
	 * @param tabs The number of tabs to include to maintain the tree look of an XML file
	 * @return String of XML representation of this element
	 */
	public String toString(int tabs) {
		StringBuilder result = new StringBuilder();
		String NEW_LINE = System.getProperty("line.separator");

		for (int i = 0; i < tabs; i++)
			result.append('\t');
		result.append("<" + this.tag);
		for (Attribute a : this.attributes)
			result.append(" " + a.toString());
		result.append(">");
		if (this.elements.size() == 0)
			result.append(this.value + "</" + this.tag + ">" + NEW_LINE);
		else {
			result.append(NEW_LINE);
			for (Element e : this.elements)
				result.append(e.toString(tabs + 1));
			for (int i = 0; i < tabs; i++)
				result.append('\t');
			result.append("</" + this.tag + ">" + NEW_LINE);
		}

		return result.toString();
	}

	public Element getParent() {
		return this.parent;
	}

	public void setParent(Element p) {
		this.parent = p;
	}

	public void setValue(String v) {
		this.value = v;
	}

	public String getValue() {
		return this.value;
	}

	public void addChild(Element e) {
		this.elements.add(e);
	}

	public void addAttribute(String name, String value) {
		this.attributes.add(new Attribute(name, value));
	}

	public String getTag() {
		return this.tag;
	}

	public void setTag(String t) {
		this.tag = t;
	}

	public List<Element> getChildren() {
		return this.elements;
	}

	public List<Attribute> getAttributes() {
		return attributes;
	}

	public Element getFirstChildByName(String tag) {
		for (Element child : this.elements)
			if (child.tag.toUpperCase().equals(tag.toUpperCase()))
				return child;
		return null;
	}

	public String getAttributeByName(String tag) {
		for (Attribute att : this.attributes)
			if (att.getName().toUpperCase().equals(tag.toUpperCase()))
				return att.getValue();
		return null;
	}

	public Element getChildByAttributes(List<Attribute> a_list) {
		for (Element child : this.elements) {
			boolean good = true;
			for (Attribute att : a_list) {
				String a_val = child.getAttributeByName(att.getName());
				if (a_val == null || !a_val.equals(att.getValue())) {
					good = false;
					break;
				}
			}
			if (good)
				return child;
		}
		return null;
	}

	public void removeChild(int i) {
		this.elements.remove(i);
	}

	public void addChildToIndex(Element e, int i) {
		this.elements.add(i, e);
	}

	public Element getFirstDescendentByName(String tag) {
		Element ret = getFirstChildByName(tag);
		int i = 0;
		while (ret == null && i < this.elements.size()) {
			ret = this.elements.get(i++).getFirstDescendentByName(tag);
		}
		return ret;
	}
	
	/**
	 * Follows the XML path given and returns the value of the last element in the path
	 * 		Possibly will add functionality for finding specific attributes
	 * @param path
	 * @return
	 */
	public String followPathToValue(List<String> path) {
		String ret = null;
		Element root = this;

		for (String s : path) {
			if (s.startsWith(">")) {
				s = s.substring(1);
				String[] substrings = s.split(",");
				List<Attribute> atts = new ArrayList<Attribute>();
				for (String subs : substrings) {
					String[] vals = subs.split("=");
					atts.add(new Attribute(vals[0], vals[1]));
				}
				ret = root.getParent().getChildByAttributes(atts).getValue();
			} else {
				root = root.getFirstChildByName(s);
			}
		}

		if (root.getChildren().size() == 0 && ret == null)
			ret = root.getValue();

		return ret;
	}
	
	public Element followPathToElement(List<String> path)
	{
		Element active = this;
		for(String s : path)
		{
			if(s.startsWith(">")){
				s = s.substring(1);
				String[] substrings = s.split(",");
				List<Attribute> atts = new ArrayList<Attribute>();
				for (String subs : substrings) {
					String[] vals = subs.split("=");
					atts.add(new Attribute(vals[0], vals[1]));
				}
				active = active.getParent().getChildByAttributes(atts);
			}
			else
				active = active.getFirstChildByName(s);
			if(active == null)
				return null;
		}
		
		return active;
	}
	
	/**
	 * Creates a path to this element by recursively finding its parent, then adding the path back from there
	 * @return
	 */
	public List<String> createPathTo()
	{
		List<String> path = new ArrayList<String>();
		
		Element e = this;
		
		if(e.getTag().toLowerCase().equals("point"))
		{
			String attTag = ">Dx=" + e.getAttributeByName("Dx") + ",Dy=" + e.getAttributeByName("Dy");
			path.add(attTag);
		}
		
		while(e != null)
		{
			path.add(e.getTag());
			e = e.getParent();
		}
		Collections.reverse(path);
		path.remove(0); //remove root, following path will already be there
		
		return path;
	}
}
