package GUI;

import java.awt.Component;
import java.util.*;

import javax.swing.JOptionPane;

import DataExtraction.Element;
import Functions.*;

public class Equation {

	private List<Function> functions;
	private Map<String, List<String>> tagPathMap;
	private Element element;
	private List<String> fullWords;
	private Map<String, Equation> aliasMap;
	private Component component;
	
	public Equation(String l, Map<String, Equation> am)
	{
		this.tagPathMap = new HashMap<String, List<String>>();
		this.aliasMap = am;
		this.fullWords = createEquationList(l);
		this.functions = getFunctions();
	}
	
	public Equation(List<String> path, String name, Map<String, Equation> am)
	{
		this.tagPathMap = new HashMap<String, List<String>>();
		this.tagPathMap.put(name, path);
		this.aliasMap = am;
		this.fullWords = createEquationList(name);
		this.functions = getFunctions();
	}
	
	public Double solve(Element ele, Component comp)
	{
		try{
			this.element = ele;
			this.component = comp;
			List<String> newWords = new ArrayList<String>();
			for(String s : this.fullWords)
				newWords.add(s);
			return getValue(execute(newWords, ele));
		}
		catch(Exception e){
			e.printStackTrace();
			return null;
		}
	}
	
	public String execute(List<String> words, Element ele)
	{
		if(words.size() == 0)
			return null;
		if(words.get(0).equals("-"))
			words.add(0, "0");

		Boolean hasParens;
		do{
			hasParens = false;
			int parens = 0;
			int start = 0;
			for(int i = 0; i < words.size(); i++)
			{
				String word = words.get(i);
				if(word.equals("("))
				{
					hasParens = true;
					if(parens == 0)
						start = i;
					parens++;
				}
				if(word.equals(")"))
				{
					hasParens = true;
					parens--;
					if(parens == 0)
					{
						List<String> toRemove = words.subList(start, i + 1);
						List<String> toExecute = new ArrayList<String>();
						for(int j = 1; j < toRemove.size() - 1; j++)
							toExecute.add(toRemove.get(j));
						toRemove.clear();
						
						//Separate parameters where applicable. If no commas, words are either a value or one parameter
						List<String> param = new ArrayList<String>();
						for(int j = 0; j < toExecute.size(); j++)
						{
							String s = toExecute.get(j);
							switch(s){
							case ",":
								if(parens == 0)
								{
									words.add(start, execute(param, ele));
									start++;
									param = new ArrayList<String>();
								}
								else
									param.add(s);
								break;
							case "(":
								parens++;
								param.add(s);
								break;
							case ")":
								parens--;
							default:
								param.add(s);
							}
						}
						words.add(start, execute(param, ele).toString());
					}
				}
			}
		} while(hasParens);
		
		for(Function func : this.functions)
		{
			for(int i = 0; i < words.size(); i++)
			{
				if(func.getTitle().equals(words.get(i)))
				{
					try{
						List<String> params = words.subList(i + 1, i + 1 + func.getParameterCount());
						Double value = func.solve(params, this, this.component);
						params.clear();
						words.remove(i);
						words.add(i, value.toString());
					}
					catch(IndexOutOfBoundsException e){
						JOptionPane.showMessageDialog(this.component, 
								func.getTitle() + " requires " + func.getParameterCount() + " parameters.",
							    "Error",
							    JOptionPane.ERROR_MESSAGE);
					}
				}
			}
		}
		
		//Perform multiplication and division
		for(int i = 0; i < words.size(); i++)
		{
			String word = words.get(i);
			if(word.equals("*"))
			{
				Double value = getValue(words.get(i - 1)) * getValue(words.get(i + 1));
				words.subList(i-1, i+2).clear();
				words.add(i-1, value.toString());
				i--;
			}
			else if(word.equals("/"))
			{
				Double value = getValue(words.get(i - 1)) / getValue(words.get(i + 1));
				words.subList(i-1, i+2).clear();
				words.add(i-1, value.toString());
				i--;
			}
		}
		
		//Perform addition and subtraction
		for(int i = 0; i < words.size(); i++)
		{
			String word = words.get(i);
			if(word.equals("+"))
			{
				Double value = getValue(words.get(i - 1)) + getValue(words.get(i + 1));
				words.subList(i-1, i+2).clear();
				words.add(i-1, value.toString());
				i--;
			}
			else if(word.equals("-"))
			{
				Double value = getValue(words.get(i - 1)) - getValue(words.get(i + 1));
				words.subList(i-1, i+2).clear();
				words.add(i-1, value.toString());
				i--;
			}
		}
		
		return words.get(0);
	}
	
	public Double getValue(String word)
	{
		if(this.tagPathMap.containsKey(word))
		{
			String value = this.element.followPathToValue(this.tagPathMap.get(word));
			if(value != null)
				return Double.parseDouble(this.element.followPathToValue(this.tagPathMap.get(word)));
			else
			{
				JOptionPane.showMessageDialog(this.component, 
						word + " is a correlation and has no single value associated with it",
					    "Error",
					    JOptionPane.ERROR_MESSAGE);
				return null;
			}
		}
		
		Equation eq = this.aliasMap.get(word);
		if(eq != null)
			return eq.solve(this.element, this.component);

		try
		{
			return Double.parseDouble(word);
		}
		catch(NumberFormatException e)
		{
			JOptionPane.showMessageDialog(this.component, 
					word + " is not a valid variable, number, or function",
				    "Error",
				    JOptionPane.ERROR_MESSAGE);
			return null;
		}
	}
	
	private List<String> createEquationList(String line)
	{
		List<String> words = new ArrayList<String>();
		
		if(this.tagPathMap.containsKey(line) || this.aliasMap.containsKey(line))
		{
			words.add(line);
			return words;
		}
		
		int start = 0;
		for(int i = 0; i < line.length(); i++)
		{
			Character c = line.charAt(i);
			switch(c){
			case '(':
			case ')':
			case ',':
			case '*':
			case '/':
			case '+':
				if(i > 0)
				{
					String toAdd = line.substring(start, i).trim();
					if(!isOperator(line.charAt(i - 1)) && toAdd.length() > 0)
						words.add(toAdd);
				}
				words.add(c.toString());
				start = i + 1;
				break;
			case '-':
				if(i == 0)
				{
					words.add(c.toString());
					start = i + 1;
					break;
				}
				if(line.charAt(i - 1) != 'E')
				{
					String toAdd = line.substring(start, i).trim();
					if(!isOperator(line.charAt(i - 1)) && toAdd.length() > 0)
						words.add(toAdd);
					words.add(c.toString());
					start = i + 1;
				}
				break;
			default:
				if(i == line.length() - 1)
					words.add(line.substring(start).trim());
			}
		}
		
		return words;
	}
	
	private boolean isOperator(Character c)
	{
		if(c.equals('(') || c.equals(')') || c.equals('*') || c.equals('/') || c.equals('+') || c.equals('-'))
			return true;
		return false;
	}
	
	private List<Function> getFunctions()
	{
		List<Function> funcs = new ArrayList<Function>();

		funcs.add(new SquareRoot());
		funcs.add(new AbsoluteValue());
		funcs.add(new Power());
		funcs.add(new AverageCorrelation());
		
		return funcs;
	}
	
	public Element getElement()
	{
		return this.element;
	}
	
	public String getError(Element root)
	{
		if(isSingleVariable())
		{
			Element ele = root.followPathToElement(this.tagPathMap.get(this.fullWords.get(0)));
			if(ele != null)
				return ele.getAttributeByName("Error");
		}
		
		return null;
	}
	
	public boolean isSingleVariable()
	{
		if(this.tagPathMap.containsKey(this.fullWords.get(0)))
			return true;
		return false;
	}
	
	public boolean hasError()
	{
		if(isSingleVariable())
		{
			Element ele = this.element.followPathToElement(this.tagPathMap.get(this.fullWords.get(0)));
			if(ele != null)
				return ele.getAttributeByName("Error") != null;
		}
		return false;
	}
}
