package GUI;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;

import javax.swing.*;

import Comparators.*;

public class Filters {
	//filters: list of filters to be applied to files
	private final List<ResultFilter> filters;
	//fullPanel: panel containing the filters and the list of files after filtering
	private JPanel fullPanel;
	//filterPanel: panel containing the scrollpane of filters
	private JPanel filterPanel;
	//filterjsp: scroll pane containing the filters and their associated components
	private JScrollPane filterjsp;
	//filejsp: scroll pane containing the files after filtering
	private JScrollPane filejsp;
	//parent: the parent of this class, the main window 
	private GUIFrame parent;
	//comparators: list of all possible comparators. used in listing choices
	private List<ResultFilter> comparators;	
	//filtermap: mapping of ID to the panel associated with a filter
	private Map<UUID, JPanel> filtermap;
	
	public Filters(GUIFrame p)
	{
		this.parent = p;
		this.filters = new ArrayList<ResultFilter>();
		filtermap = new HashMap<UUID, JPanel>();
		initializeComparators();
		initializeFullPanel();
	}
	
	/**
	 * Adds all comparators to the list of comparators
	 * When a comparator is added to the directory "Comparators", an instantiation should be added here
	 * 			Can this be automated?
	 */
	private void initializeComparators()
	{
		comparators = new ArrayList<ResultFilter>();
		comparators.add(new DoubleEQFilter());
		comparators.add(new DoubleGEFilter());
		comparators.add(new DoubleGTFilter());
		comparators.add(new DoubleLEFilter());
		comparators.add(new DoubleLTFilter());
		comparators.add(new DoubleNEFilter());
		//comparators.add(new SameStringFilter());
	}
	
	/**
	 * Initializes the full panel, containing all filters and the filtered file list
	 */
	private void initializeFullPanel()
	{
		this.fullPanel = new JPanel();
		this.fullPanel.setLayout(new BoxLayout(this.fullPanel, BoxLayout.Y_AXIS));
		
		this.filterPanel = new JPanel();
		this.filterPanel.setLayout(new BoxLayout(this.filterPanel, BoxLayout.Y_AXIS));
		this.filterPanel.add(getNewFilter());
		JPanel filterPanelWithGlue = new JPanel();
		filterPanelWithGlue.add(this.filterPanel);
		filterPanelWithGlue.add(Box.createVerticalGlue());
		
		this.filterjsp = new JScrollPane(filterPanelWithGlue);
		this.filterjsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		//make jsp a little bit bigger so the vertical scrollbar doesnt push horizontally, requiring a horizontal scrollbar
		Dimension jspd = this.filterjsp.getPreferredSize();
		this.filterjsp.setPreferredSize(new Dimension(jspd.width + 40, jspd.height * 4));
		this.filejsp = getFilteredFileListScrollPane();
		
		JButton addFilter = new JButton("Add Filter");
		addFilter.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e) {
				filterPanel.add(getNewFilter());
				updateFullPanel();
			}
		});
		
		this.fullPanel.add(addFilter);
		this.fullPanel.add(Box.createVerticalStrut(10));
		this.fullPanel.add(this.filterjsp);
		this.fullPanel.add(Box.createVerticalStrut(10));
		this.fullPanel.add(this.filejsp);
		this.fullPanel.add(Box.createVerticalGlue());
	}
	
	/**
	 * Updates the full panel, removes and replaces the filtered file list
	 */
	public void updateFullPanel()
	{
		this.fullPanel.remove(this.filejsp);
		this.filejsp = getFilteredFileListScrollPane();
		this.fullPanel.add(this.filejsp);
		SwingUtilities.updateComponentTreeUI(this.fullPanel);
		this.parent.updateFileList();
		this.parent.updateResultPanel();
	}
	
	/**
	 * 
	 * @return
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private JPanel getNewFilter()
	{
		/*
		 * Container class to be used with actionListeners
		 */
		class FilterWrapper{
			ResultFilter thisFilter;
			FilterWrapper(ResultFilter f){
				this.thisFilter = f;
			}
			public ResultFilter getThisFilter() {
				return thisFilter;
			}
			public void setThisFilter(ResultFilter thisFilter) {
				this.thisFilter = thisFilter;
			}
		}
		
		final FilterWrapper filter = new FilterWrapper(new DoubleEQFilter());
		final JPanel singleFilter = new JPanel();
		singleFilter.setLayout(new BoxLayout(singleFilter, BoxLayout.X_AXIS));
		final UUID id = UUID.randomUUID();
		filter.thisFilter.setId(id);

		final JComboBox operators = new JComboBox(this.comparators.toArray());
		operators.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				ResultFilter selected = (ResultFilter) operators.getSelectedItem();
				ResultFilter newFilter = selected.createNewFilter();
				newFilter.setTarget(filter.getThisFilter().getTarget());
				newFilter.setEquation(filter.getThisFilter().getEquation());
				newFilter.setId(id);
				replaceFilter(id, newFilter);
				filter.setThisFilter(newFilter);
				updateFullPanel();
			}
		});

		final JTextField targetField = new JTextField("Enter operand", 10);
		targetField.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e) {
				filter.thisFilter.setTarget(targetField.getText());
				updateFullPanel();
			}
		});
		targetField.setMaximumSize(new Dimension(100, 20));

		JButton removeFilterButton = new JButton("-");
		removeFilterButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e) {
				removeFilterById(id, true);
				parent.setFilteredFileList(parent.getFullFileList());
				updateFullPanel();
			}
		});
		
		operators.setMaximumSize(new Dimension(150, 20));
		operators.setPreferredSize(new Dimension(120, 20));
		
		singleFilter.add(this.parent.getVariableMenu(filter.thisFilter));
		singleFilter.add(Box.createRigidArea(new Dimension(10, 0)));
		singleFilter.add(operators);
		singleFilter.add(Box.createRigidArea(new Dimension(10, 0)));
		singleFilter.add(targetField);
		singleFilter.add(Box.createRigidArea(new Dimension(10, 0)));
		singleFilter.add(removeFilterButton);

		this.filtermap.put(id, singleFilter);
		
		this.filters.add(filter.thisFilter);
		
		return singleFilter;
	}
	
	/**
	 * Creates a brand new scroll pane containing the list of filtered file names
	 * @return
	 */
	private JScrollPane getFilteredFileListScrollPane() {
		FileRootList allFiles = this.parent.getFullFileList();
		List<FileRoot> filteredFiles = allFiles.getFilteredRoots(this.filters);
		List<String> fileNames = new ArrayList<String>();
		for(FileRoot fr : filteredFiles)
			fileNames.add(fr.file);
		this.parent.setFilteredFileList(new FileRootList(filteredFiles));
		
		final DefaultListModel<String> model = new DefaultListModel<String>();
		for (FileRoot fr : filteredFiles)
			model.addElement(fr.file.substring(Math.max(fr.file.lastIndexOf("/"), fr.file.lastIndexOf("\\")) + 1));
		final JList<String> fileList = new JList<String>(model);

		JScrollPane jsp = new JScrollPane(fileList);
		jsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		Dimension fileListDim = new Dimension(200, 200);
		jsp.setSize(fileListDim);
		jsp.setMinimumSize(fileListDim);
		jsp.setMaximumSize(fileListDim);
		jsp.setPreferredSize(fileListDim);

		return jsp;
	}
	
	public JPanel getPanel()
	{
		return this.fullPanel;
	}
	
	/**
	 * Removes a filter from the list of filters and possibly from the GUI based on its ID.
	 * The GUI components will be removed when the filter itself is removed, but not when the filter is replaced
	 * @param id unique ID of the filter
	 * @param removeGUI Whether or not the filter should be removed from the GUI
	 * @return index in the list of filters where this filter was found, -1 if it was not found
	 */
	private int removeFilterById(UUID id, boolean removeGUI)
	{
		for(int i = 0; i < this.filters.size(); i++)
		{
			ResultFilter filter = this.filters.get(i);
			if(filter.getId().equals(id))
			{
				this.filters.remove(i);
				if(removeGUI) this.filterPanel.remove(this.filtermap.get(id));
				SwingUtilities.updateComponentTreeUI(this.filterPanel);				
				return i;
			}
		}
		return -1;
	}
	
	/**
	 * Removes the filter without removing its GUI elements then replaces it with a new filter
	 * The ID of the new filter will be the same as that of the old filter
	 * @param id unique ID of the filter to be replaced
	 * @param newFilter new ResultFilter to replace the old one
	 */
	private void replaceFilter(UUID id, ResultFilter newFilter)
	{
		int ind = removeFilterById(id, false);
		if(ind >= 0)
		{
			this.filters.add(ind, newFilter);
		}
	}
}
