package GUI;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.File;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.filechooser.FileFilter;

import DataExtraction.ASCIIParser;
import DataExtraction.Element;
import GUI.FileFormatKeeper.FileFormat;
import GUI.VariableMenuGenerator.VariableMenuListener;
import SpecialOutputs.CorrPrintout;
import SpecialOutputs.CorrReflected;

import java.lang.Math;

import java.util.*;

public class GUIFrame extends JFrame {
	private static final long serialVersionUID = -1609424635299612530L;

	//fullFrl: FileRootList containing the unaltered list of FileRoots taken in the constructor
	private FileRootList fullFrl;
	//minDist: minimum distance used in correlations among all roots in fullFrl, used when correlation points are examined individually 
	private Double minDist = Double.MAX_VALUE;
	//fileListIndices: usually empty, contains the indices of selected file names in the file list panel. used in removing files from the file list
	private final List<Integer> fileListIndices;
	//variables: contains all GUI components related to the Variable selection panel and the results panel
	private Variables variables;
	//filters: contains all GUI components related to the Filter selection panel and filtered file list panel
	private Filters filters;
	//filteredFrl: FileRootList containing the list of files which pass for all applied filters
	private FileRootList filteredFrl;
	//variableGenerator: creates menus for user to select a variable for output or filtering
	private VariableMenuGenerator variableGenerator;
	//includeFileNames: boolean tells whether or not to include file names in results panel
	private Boolean includeFileNames;
	//thisDialog: pointer to this JDialog
	private final GUIFrame thisDialog;
	//fileFormatKeeper: maintains the list of file formats which can be saved to
	private FileFormatKeeper fileFormatKeeper;
	//showFilteredList: tells whether to show the filtered list (shows full list if false)
	private Boolean showFilteredList;
	
	private Boolean includeErrors;
	
	private Boolean showAverages;
	
	private final JList<String> fileList;
	
	private ResultPanel resultPanel;
	
	private List<Runnable> specialOutputs;
	
	private String variableList;
	
	private Element sampleEle;
	
	/**
	 * Constructor for GUIFrame
	 * @param frl_i List of FileRoots which is used as for the initial full list of FileRoots
	 */
	public GUIFrame(List<FileRoot> frl_i, String varlist) {
		this.resultPanel = new ResultPanel(this);
		this.thisDialog = this;
		if(frl_i.size() == 0)
		{
			getInitialFileRoots(frl_i);
		}
		this.fullFrl = new FileRootList(frl_i);
		this.filteredFrl = new FileRootList(frl_i);
		this.includeFileNames = false;
		this.showFilteredList = false;
		this.includeErrors = false;
		this.showAverages = false;
		this.fileListIndices = new ArrayList<Integer>();
		this.fileList = new JList<String>();
		this.fileFormatKeeper = new FileFormatKeeper();
		this.variableList = varlist;
	}
	
	/**
	 * Adds components
	 */
	public void executeGUI()
	{
		if(this.fullFrl.getList().size() == 0)
		{
			this.dispose();
			return; //true cancel from user
		}
		this.minDist = computeMinDist();
		this.variableGenerator = new VariableMenuGenerator(this.minDist, this, this.variableList);
		this.sampleEle = this.variableGenerator.getSample();
		this.variables = new Variables(this);
		this.resultPanel.updateResultsPanel();
		this.filters = new Filters(this);
		setSpecialOutputs();
		final DefaultListModel<String> fileModel = new DefaultListModel<String>();
		for (FileRoot fr : fullFrl.getList())
			fileModel.addElement(fr.getShortenedFile());

		this.setJMenuBar(createMainMenu(fileModel));
		this.setLayout(new BoxLayout(getContentPane(), BoxLayout.X_AXIS));

		//Generate the list GUI element which will hold the file names
		this.fileList.setModel(fileModel);
		this.fileList.addKeyListener(new KeyListener(){	//allow user to remove selected files from the list by hitting DELETE
			public void keyPressed(KeyEvent arg0) {}
			public void keyReleased(KeyEvent arg0) {}
			public void keyTyped(KeyEvent arg0) {
				if(arg0.getKeyChar() == '\u007f')	//if the DELETE key is pressed, then remove all selected files (associated FileRoots)
				{	
					for (int i = fileListIndices.size() - 1; i >= 0; i--) {
						int j = fileListIndices.get(i);
						if(showFilteredList)
						{
							fullFrl.removeFileRoot(filteredFrl.getFile(j));
							filteredFrl.removeFileRoot(j);							
						}
						else
						{
							filteredFrl.removeFileRoot(fullFrl.getFile(j));
							fullFrl.removeFileRoot(j);
						}
						fileModel.remove(j);
					}
					fileListIndices.clear();
					filters.updateFullPanel();
					Double newMin = computeMinDist();
					if(minDist != newMin)
					{
						minDist = newMin;
						variableGenerator.setMenusToMinDist(newMin);
					}
				}
			}
		});
		
		/*
		 * Add all components to the GUIFrame (JDialog)
		 * Struts of length 15 are used to add space between components
		 */
		this.add(this.getFileListPanel());

		this.add(Box.createHorizontalStrut(15));
		this.add(new JSeparator(SwingConstants.VERTICAL));
		
		this.add(Box.createHorizontalStrut(15));
		this.add(this.variables.getVariablePanel());

		this.add(Box.createHorizontalStrut(15));
		this.add(new JSeparator(SwingConstants.VERTICAL));
		
		this.add(Box.createHorizontalStrut(15));
		this.add(this.resultPanel.getResultsPanel());
		
		this.add(Box.createHorizontalGlue());

		setTitle("QUEST ADEPT");
		setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
		pack();
		final Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		setLocation((screenSize.width - this.getWidth()) / 2, (screenSize.height - this.getHeight()) / 2);
		setVisible(true);
	}

	/**
	 * Creates a panel which contains the file list in a scroll pane along with related 
	 * 		components including the filter manager
	 * @return	a JPanel containing all components associated with the file list
	 */
	private JPanel getFileListPanel() {
		final JPanel panel = new JPanel();
		panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
		
		//Setup the scroll pane which holds the list
		final JScrollPane jsp = new JScrollPane(this.fileList);
		jsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		jsp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		Dimension fileListDim = new Dimension(220, 300);
		jsp.setMaximumSize(fileListDim);
		jsp.setPreferredSize(fileListDim);
		
		//Panel for the radio buttons to show the full or filtered list in the scroll pane
		JPanel radioPanel = new JPanel();
		
		//Full file list is shown by default
		JRadioButton showFullFileList = new JRadioButton("Show full list");
		showFullFileList.setSelected(!this.showFilteredList);
		showFullFileList.addActionListener(new ActionListener(){	//if clicked, set fileList to full file list
			public void actionPerformed(ActionEvent arg0) {
				showFilteredList = false;
				updateFileList();
			}
		});
		
		JRadioButton showFilteredFileList = new JRadioButton("Show filtered list");
		showFilteredFileList.setSelected(this.showFilteredList);
		showFilteredFileList.addActionListener(new ActionListener(){//if clicked, set fileList to filtered file list
			public void actionPerformed(ActionEvent arg0) {		
				showFilteredList = true;
				updateFileList();
			}
		});
		
		//ButtonGroup groups the radio buttons and ensures that only one of them is set at any time
		ButtonGroup group = new ButtonGroup();
		group.add(showFullFileList);
		group.add(showFilteredFileList);
		radioPanel.add(showFullFileList);
		radioPanel.add(showFilteredFileList);
		
		fileList.addListSelectionListener(new ListSelectionListener() {	//when items in list are selected, set fileListIndices to match those items 
			public void valueChanged(ListSelectionEvent e) {
				if (!e.getValueIsAdjusting()) {
					int[] indices = fileList.getSelectedIndices();
					fileListIndices.clear();
					for (int i : indices)
						fileListIndices.add(i);
				}
			}
		});
		
		//Button to open the Filter Manager which affects the filtered file list
		JButton manageFiltersButton = new JButton("Manage filters");
		manageFiltersButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent arg0) {
				JDialog filterFrame = new JDialog(thisDialog, "Filter Manager");
				filterFrame.add(filters.getPanel());
				filterFrame.pack();
				filterFrame.setVisible(true);
			}
		});
		
		panel.add(radioPanel);
		panel.add(jsp);
		panel.add(Box.createVerticalStrut(15));
		panel.add(manageFiltersButton);
		panel.add(Box.createVerticalGlue());

		return panel;
	}
	
	/**
	 * Creates a menu bar containing options such as adding files, saving to a file, etc.
	 * @param model contains the list of files to be shown in the JList
	 * @return a JMenuBar containing all menus which will be at the top of the main window
	 */
	private JMenuBar createMainMenu(final DefaultListModel<String> model)
	{
		JMenuBar menubar = new JMenuBar();
		
		/*
		 * Menu items regarding files are stored here:
		 *		Add file - 	add a file to the full file list. if the filtered list is being shown, 
		 *					the new file will be shown in the list if it passes through all filters
		 *		Save as -	saves the contents of the result panel to a file of the designated type (an extension must be picked)
		 *		Exit -		Exits the application
		 */
		JMenu fileMenu = new JMenu("File");
		
		JMenuItem addFileItem = new JMenuItem("Add file");
		addFileItem.addActionListener(new ActionListener() {	//open file chooser to add a file
			public void actionPerformed(ActionEvent event) {
				final JFileChooser fc = new JFileChooser();
				fc.setMultiSelectionEnabled(true);				//allow multiple files to be added at once
				int returnVal = fc.showOpenDialog(GUIFrame.this);

				if (returnVal == JFileChooser.APPROVE_OPTION) {
					File[] files = fc.getSelectedFiles();
					for(File file : files)
					{
						String fileName = file.toString();
						ASCIIParser parser = new ASCIIParser();
						Element root = parser.parserFile(fileName);
						if (root != null) {
							fullFrl.addFileRoot(new FileRoot(fileName, root));
							model.addElement(fileName.substring(Math.max(
									fileName.lastIndexOf("/"),
									fileName.lastIndexOf("\\")) + 1));
						}
					}
					sampleEle = variableGenerator.updateAndGetSample();
				}
				filters.updateFullPanel();
			}
		});
		JMenuItem saveFileItem = new JMenuItem("Save as...");
		saveFileItem.addActionListener(new ActionListener(){	//open file chooser to save results to file
			public void actionPerformed(ActionEvent event) {
				final JFileChooser fc = new JFileChooser();
				fc.setAcceptAllFileFilterUsed(false);			//only allow user to output in a format we have created 
				fileFormatKeeper.setFileChooserFilters(fc);		//apply formats
				int option = fc.showSaveDialog(thisDialog);
				if(option == JFileChooser.APPROVE_OPTION)
				{
					File file = fc.getSelectedFile();
					if(file.isDirectory())
					{
						JOptionPane.showMessageDialog(thisDialog, 
							"Must enter a file name, not a directory",
						    "Error",
						    JOptionPane.ERROR_MESSAGE);
					}
					else
					{
						FileFilter filter = fc.getFileFilter();
						String filterext = filter.getDescription();
						filterext = filterext.replace("*", "");
						FileFormat ff = fileFormatKeeper.getFormat(filterext);
						if(!file.getName().endsWith(filterext))
							file = new File(file.getAbsolutePath() + filterext);
						if (file.exists ()) {
				             int response = JOptionPane.showConfirmDialog (null,
				               "Overwrite existing file?","Confirm Overwrite",
				                JOptionPane.YES_NO_OPTION,
				                JOptionPane.QUESTION_MESSAGE);
				             if(response == JOptionPane.YES_OPTION)
								ff.saveToFile(file, resultPanel.getResultData());
						}
						else
							ff.saveToFile(file, resultPanel.getResultData());
					}
				}
			}
		});
		JMenuItem quitItem = new JMenuItem("Exit");
		quitItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				System.exit(0);
			}
		});
		
		fileMenu.add(addFileItem);
		fileMenu.add(saveFileItem);
		fileMenu.add(quitItem);
		
		/*
		 * Menu items regarding special options are stored here:
		 * 		Include file names in results -	When enabled, adds a column to the results panel containing
		 * 										the files name for each row
		 * 		Show _ variables -	When enabled, all "Variable" menus will show the variable list indicated.
		 * 							The number of these options depends on how many variable lists are input
		 * 							by the variablelists.xml file upon execution. 
		 */
		JMenu optionMenu = new JMenu("Options");
		JCheckBoxMenuItem includeErrorsItem = new JCheckBoxMenuItem("Include value errors in results");
		includeErrorsItem.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent arg0) {
				includeErrors = !includeErrors;
				resultPanel.updateResultsPanel();
			}
		});
		JCheckBoxMenuItem includeFileNamesItem = new JCheckBoxMenuItem("Include file names in results");
		includeFileNamesItem.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				includeFileNames = !includeFileNames;
				resultPanel.updateResultsPanel();
			}
		});
		JCheckBoxMenuItem averageValuesItem = new JCheckBoxMenuItem("Include inter-file average values and errors in last rows");
		averageValuesItem.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e){
				showAverages = !showAverages;
				resultPanel.updateResultsPanel();
			}
		});

		optionMenu.add(includeErrorsItem);
		optionMenu.add(includeFileNamesItem);
		optionMenu.add(averageValuesItem);
		
		Set<String> settableMenus = this.variableGenerator.getSettableMenuNames();
		Map<String,Boolean> isMenuVisible = this.variableGenerator.getIsMenuVisible();
		for(final String name : settableMenus)
		{
			JCheckBoxMenuItem showMenu = new JCheckBoxMenuItem("Show " + name + " variables");
			showMenu.addActionListener(new ActionListener(){
				public void actionPerformed(ActionEvent e){
					variableGenerator.toggleVisible(name);
				}
			});
			if(isMenuVisible.get(name)) showMenu.setSelected(true);
			else showMenu.setSelected(false);
			optionMenu.add(showMenu);
		}

		JMenu outputMenu = new JMenu("Output");
		for(final Runnable output : this.specialOutputs)
		{
			JMenuItem item = new JMenuItem(output.toString());
			item.addActionListener(new ActionListener(){
				public void actionPerformed(ActionEvent arg0) {
					output.run();
				}
			});
			outputMenu.add(item);
		}
		
		JMenu helpMenu = new JMenu("Help");
		JMenuItem item = new JMenuItem("Version");
		item.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent arg0) {
				JOptionPane.showMessageDialog(thisDialog, 
						"ADEPT version 1.0",
					    "Version",
					    JOptionPane.INFORMATION_MESSAGE);
			}
		});
		helpMenu.add(item);
		
		menubar.add(fileMenu);
		menubar.add(optionMenu);
		menubar.add(outputMenu);
		menubar.add(helpMenu);
		
		return menubar;
	}
	
	/**
	 * Returns the full file list
	 * @return
	 */
	public FileRootList getFullFileList()
	{
		return this.fullFrl;
	}
	
	/**
	 * Returns the filtered file list
	 * @return
	 */
	public FileRootList getFilteredFileList()
	{
		return this.filteredFrl;
	}
	
	/**
	 * Sets the filtered file list
	 * @param f FileRootList containing the list of fully filtered files 
	 */
	public void setFilteredFileList(FileRootList f)
	{
		this.filteredFrl = f;
	}
	
	/**
	 * Creates a menu bar for the VariableMenuListener
	 * @param ml a listener which is notified when a menu item is selected
	 * @return a JMenuBar containing all variable lists
	 */
	public JMenuBar getVariableMenu(VariableMenuListener ml)
	{
		return this.variableGenerator.generateMenuBar(ml);
	}
	
	/**
	 * A routine to bridge the Variables and VariableMenuGenerator classes
	 * @param menu
	 */
	public void removeVariableMenu(JMenu menu)
	{
		this.variableGenerator.removeMenu(menu);
	}
	
	/**
	 * Returns whether or not to include file names in the results panel
	 * @return
	 */
	public Boolean getIncludeFileNames()
	{
		return this.includeFileNames;
	}
	
	public Boolean getIncludeErrors()
	{
		return this.includeErrors;
	}
	
	public Boolean getShowAverages()
	{
		return this.showAverages;
	}
	
	public Element getSample()
	{
		return this.sampleEle;
	}
	
	private Double computeMinDist()
	{
		Double d = Double.MAX_VALUE;
		for (FileRoot fr : this.fullFrl.getList()) {
			ValueTypeSplitter data = new ValueTypeSplitter(fr.root);
			if (d > data.getFarthestCorrPoint())
				d = data.getFarthestCorrPoint();
		}
		return d;
	}
	
	/**
	 * Updates the file list in the case that either:
	 * 		The displayed list has been changed
	 * 		The user wishes to switch showing the filtered list or not
	 */
	public void updateFileList()
	{
		DefaultListModel<String> model = (DefaultListModel<String>) fileList.getModel();
		model.clear();
		if(this.showFilteredList)
		{
			for (FileRoot fr : filteredFrl.getList())
				model.addElement(fr.getShortenedFile());
		}
		else
		{
			for (FileRoot fr : fullFrl.getList())
				model.addElement(fr.getShortenedFile());
		}
		this.fileList.setModel(model);
		this.fileList.revalidate();
	}
	
	public List<VariableElement> getVariables()
	{
		if(this.variables == null)
			return null;
		return this.variables.getVariables();
	}
	
	public void updateResultPanel()
	{
		this.resultPanel.updateResultsPanel();
	}

	public void setSpecialOutputs()
	{
		this.specialOutputs = new ArrayList<Runnable>();
		this.specialOutputs.add(new CorrPrintout(this));
		this.specialOutputs.add(new CorrReflected(this));
	}
	
	public List<Runnable> getSpecialOutputs()
	{
		return this.specialOutputs;
	}
	
	public void setResultTable(JTable table, Boolean sort)
	{
		this.resultPanel.setTable(table, sort);
	}
	
	private void getInitialFileRoots(List<FileRoot> frl)
	{	
		final JFileChooser fc = new JFileChooser();
		fc.setMultiSelectionEnabled(true);				//allow multiple files to be added at once
		
		int returnVal = fc.showOpenDialog(GUIFrame.this);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			File[] files = fc.getSelectedFiles();
			if(files.length == 0)
			{
				JOptionPane.showMessageDialog(fc, 
						"Must choose a valid input file",
					    "Error",
					    JOptionPane.ERROR_MESSAGE);
				getInitialFileRoots(frl);
			}
			else
			{
				ASCIIParser parser = new ASCIIParser();
				for(File file : files)
				{
					String fileName = file.toString();
					Element root = parser.parserFile(fileName);
					if (root != null) {
						frl.add(new FileRoot(fileName, root));
					}
				}
				if(frl.size() == 0)
				{
					JOptionPane.showMessageDialog(fc, 
						"No valid input files selected",
					    "Error",
					    JOptionPane.ERROR_MESSAGE);
					getInitialFileRoots(frl);
				}
			}
		}
		if(returnVal == JFileChooser.CANCEL_OPTION)
		{
			JOptionPane.showMessageDialog(fc, 
					"No files chosen, exitting",
				    "Error",
				    JOptionPane.ERROR_MESSAGE);
		}
	}
}
