package GUI;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

public class ResultPanel {

	class sortOrder{
		private Boolean sortincreasing;

		public sortOrder(){
			this.sortincreasing = true;
		}
		
		public Boolean getOrder(){
			return this.sortincreasing;
		}
		public void toggle(){
			this.sortincreasing = ! this.sortincreasing;
		}
	}
	
	private JPanel panel;
	private JScrollPane jsp;
	private JTable table;
	private GUIFrame parent;
	private final sortOrder order;
	
	public ResultPanel(GUIFrame p)
	{

		this.panel = new JPanel();
		this.panel.setLayout(new BoxLayout(this.panel, BoxLayout.Y_AXIS));
		JButton updateButton = new JButton("Update");
		updateButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent event) {
				updateResultsPanel();
			}
		});
		
		this.panel.add(updateButton);
		this.parent = p;
		this.order = new sortOrder();
	}
	
	public void updateResultsPanel() {
		if(this.jsp != null) this.panel.remove(this.jsp);
		List<List<String>> dataGrid = new ArrayList<List<String>>();
		List<String> columnNamesList = new ArrayList<String>();
		FileRootList frl = this.parent.getFilteredFileList();
		
		if(this.parent.getIncludeFileNames())
		{
			List<String> fileNames = new ArrayList<String>();
			columnNamesList.add("File names");
			for(FileRoot fr : frl.getList())
				fileNames.add(fr.getShortenedFile());
			if(this.parent.getShowAverages())
			{
				fileNames.add("Average");
				fileNames.add("Error");
			}
			dataGrid.add(fileNames);
		}
		
		for(VariableElement ve : this.parent.getVariables())
		{
			if(ve.getEquation() != null)
			{
				List<String> varData = new ArrayList<String>();
				columnNamesList.add(ve.getTitle());
				for(FileRoot fr : frl.getList())
				{
					Double d = ve.getEquation().solve(fr.root, null);
					if(d == null)
						varData.add("No value");
					else
						varData.add(d.toString());
				}
				if(this.parent.getShowAverages())
				{
					String error = getError(varData);
					varData.add(getAverage(varData));
					varData.add(error);
				}
				dataGrid.add(varData);
				if(this.parent.getIncludeErrors() && ve.getEquation().hasError())
				{
					List<String> errorData = new ArrayList<String>();
					columnNamesList.add(ve.getTitle() + " (Error)");
					for(FileRoot fr : frl.getList())
					{
						errorData.add(ve.getEquation().getError(fr.root));
					}
					if(this.parent.getShowAverages())
					{
						String error = getError(errorData);
						errorData.add(getAverage(errorData));
						errorData.add(error);
					}
					dataGrid.add(errorData);
				}
			}
		}
		
		//Create a data array grid to be used in the TableModel
		String[] columnNames = new String[columnNamesList.size()];
		for(int i = 0; i < columnNamesList.size(); i++)
			columnNames[i] = columnNamesList.get(i);

		int rows = 0;
		if(dataGrid.size() > 0)
			rows  = dataGrid.get(0).size();
		String[][] data = new String[rows][dataGrid.size()];
		for (int i = 0; i < rows; i++) {
			for (int j = 0; j < dataGrid.size(); j++) {
				data[i][j] = dataGrid.get(j).get(i);
			}
		}

		table = new JTable(data, columnNames){
			private static final long serialVersionUID = 1L;

			public boolean isCellEditable(int row, int col){
				return false;
			}
		};
		table.setRowSelectionAllowed(true);
		table.setColumnSelectionAllowed(true);
		table.getTableHeader().setReorderingAllowed(false);
		JTableHeader header = table.getTableHeader();
		
		//Allow table to sort rows by the values in a column when that column's header is clicked
		header.addMouseListener(new MouseAdapter()
		{
		    public void mouseClicked(MouseEvent e)
		    {
		    	JTableHeader h = (JTableHeader)e.getSource() ;
		    	int nColumn = h.columnAtPoint(e.getPoint());
		    	if(nColumn != -1)
		    		sortColumn(nColumn, h.getTable().getModel());
		    }
		});
		
		this.jsp = new JScrollPane(table);
		this.jsp.setMaximumSize(new Dimension(1000, 1000));

		this.panel.add(this.jsp);
		
		SwingUtilities.updateComponentTreeUI(this.panel);
	}
	
	public String[][] getResultData()
	{
		TableModel dataModel = this.table.getModel();
		TableColumnModel allColumnModel = this.table.getColumnModel();
		
		//Create a grid that contains the contents of the result panel plus a row for the column names
		String[][] dataGrid = new String[dataModel.getColumnCount()][dataModel.getRowCount() + 1];
    	for(int i = 0; i < dataModel.getColumnCount(); i++)
    	{
    		TableColumn columnModel = allColumnModel.getColumn(i);
    		dataGrid[i][0] = (String) columnModel.getHeaderValue();
    		for(int j = 0; j < dataModel.getRowCount(); j++)
    			dataGrid[i][j+1] = (String) dataModel.getValueAt(j, i);
    	}
    	
    	return dataGrid;
	}
	
	public JPanel getResultsPanel()
	{
		return this.panel;
	}
	
	public void resizeResultsp(Dimension delta)
	{
		Dimension current = this.jsp.getSize();
		this.jsp.setPreferredSize(new Dimension(current.width + delta.width, current.height + delta.height));
		SwingUtilities.updateComponentTreeUI(this.panel);
	}
	
	private String getAverage(List<String> data)
	{
		Double sum = 0.0;
		try{
			for(String s : data)
				sum += Double.parseDouble(s);
			sum /= data.size();
			return sum.toString();
		}
		catch(NumberFormatException e)
		{
			return "No average";
		}
	}
	
	private String getError(List<String> data)
	{
		List<Double> list = new ArrayList<Double>();
		try
		{
			for(String s : data)
				list.add(Double.parseDouble(s));
		}
		catch(NumberFormatException e)
		{
			return "No error";
		}
		Double ret = 0.0;
		Double sum = 0.0;
		for(Double d : list)
		{
			ret += d * d;
			sum += d;
		}
		ret /= data.size();
		ret -= Math.pow(sum / data.size(), 2);
		ret = Math.sqrt(ret);
		ret /= Math.sqrt(data.size() - 1);

		if(ret.isNaN() || ret.isInfinite())
			return ret.toString();
		String sret = ret.toString();
		
		int end = sret.indexOf("E");
		if(end < 0)
			end = sret.length();
		String after2 = sret.substring(sret.indexOf(".") + 3, end);
		sret = sret.replace(after2, "");
		return sret;
	}

	void sortColumn(int nColumn, TableModel model)
    {
    	Double[] values = new Double[model.getRowCount()];
    	List<Integer> indices = new ArrayList<Integer>();
    	for(int i = 0; i < model.getRowCount(); i++)
    	{
    		try{
    			values[i] = Double.parseDouble((String) model.getValueAt(i, nColumn));
    		}
    		catch(NumberFormatException e)
    		{
    			values[i] = Double.MIN_VALUE;
    		}
    		indices.add(i);
    	}
    	
    	//basic insertion sort
    	for(int i = 1; i < model.getRowCount(); i++)
    	{
    		Double d = values[i];
    		int index = i;
    		if(order.getOrder())	//sort in increasing order
    		{
	    		while(index > 0 && values[index - 1] > d)
	    		{
	    			values[index] = values[index - 1];
	    			Collections.swap(indices, index, index - 1);
	    			index--;
	    		}
	    		values[index] = d;
    		}
    		else					//sort in decreasing order
    		{
    			while(index > 0 && values[index - 1] < d)
	    		{
	    			values[index] = values[index - 1];
	    			Collections.swap(indices, index, index - 1);
	    			index--;
	    		}
	    		values[index] = d;
    		}
    	}

    	String[][] oldvals = new String[model.getColumnCount()][model.getRowCount()];
    	for(int i = 0; i < model.getColumnCount(); i++)
    		for(int j = 0; j < model.getRowCount(); j++)
    			oldvals[i][j] = (String) model.getValueAt(j, i);
    	for(int i = 0; i < model.getColumnCount(); i++)
    		for(int j = 0; j < model.getRowCount(); j++)
    			model.setValueAt(oldvals[i][indices.get(j)], j, i);
    	order.toggle();
    }
	
	public void setTable(JTable table, Boolean sort)
	{
		if(this.jsp != null) this.panel.remove(this.jsp);

		this.table = table;
		this.jsp = new JScrollPane(table);
		this.jsp.setMaximumSize(new Dimension(1000, 1000));

		this.panel.add(this.jsp);
		
		if(sort)
		{
			JTableHeader header = table.getTableHeader();
			
			//Allow table to sort rows by the values in a column when that column's header is clicked
			header.addMouseListener(new MouseAdapter()
			{
			    public void mouseClicked(MouseEvent e)
			    {
			    	JTableHeader h = (JTableHeader)e.getSource() ;
			    	int nColumn = h.columnAtPoint(e.getPoint());
			    	if(nColumn != -1)
			    		sortColumn(nColumn, h.getTable().getModel());
			    }
			});
		}
		
		SwingUtilities.updateComponentTreeUI(this.panel);
	}
}
