package GUI;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.SwingUtilities;

import DataExtraction.Attribute;
import DataExtraction.Element;
import DataExtraction.XMLParser;

public class VariableMenuGenerator {
	//sample: used to create menus for Parameters, Correlations, and NonCorrelations
	private ValueTypeSplitter sample;
	//varListEle: each child is a variable list
	private Element varlistEle;
	//minDist: the minimum distance from the origin in correlations across all file roots
	private Double minDist;
	//menuDim: the dimensions for a menu
	private Dimension menuDim;
	//settableMenus: a mapping of the name of a menu to a list of instance of this menu. used to set visibility
	private Map<String, List<JMenu>> settableMenus;
	//isMenuVisible: a mapping of the name of a menu to a boolean telling if it should be visible or not
	private Map<String, Boolean> isMenuVisible;
	
	private Map<String, Equation> aliasMap;
	
	private Map<Double, List<JMenuItem>> correlationPointMap;
	
	private Element sampleEle;
	
	private final GUIFrame parent;
	
	/*
	 * Used by any classes which will need a variable menu
	 * Allows the correct menu to set its own path and title
	 */
	public interface VariableMenuListener{
		String getTitle();
		void setTitle(String title);
		Equation getEquation();
		void setEquation(Equation o);
		JMenu getExtended();
		void setExtended(JMenu extended);
		JMenu getMainMenu();
	}
	
	public VariableMenuGenerator(Double m, GUIFrame p, String variableList)
	{
		this.minDist = m;
		this.parent = p;
		this.menuDim = new Dimension(230, 20);
		this.settableMenus = new HashMap<String, List<JMenu>>();
		this.isMenuVisible = new HashMap<String, Boolean>();
		this.aliasMap = new HashMap<String, Equation>();
		this.correlationPointMap = new HashMap<Double, List<JMenuItem>>();
		createFullSample();
		this.sample = new ValueTypeSplitter(this.sampleEle);
		//Always add the Extended variable list, which contains all variables
		if(variableList != null)
			this.isMenuVisible.put("Extended", false);
		else	
			this.isMenuVisible.put("Extended", true);
		this.settableMenus.put("Extended", new ArrayList<JMenu>());
		
		//Parse the variable lists
		XMLParser parser = new XMLParser();
		if(variableList != null)
		{
			this.varlistEle = parser.parserXMLFile(variableList);
			for(Element varlist : this.varlistEle.getChildren())
				if(varlist.getAttributeByName("settable").equals("yes"))
					this.settableMenus.put(varlist.getAttributeByName("name"), new ArrayList<JMenu>());
		}
		else
			this.varlistEle = null;
	}
	
	/**
	 * Generates a menu bar which can be added to a container
	 * @param ml a menu listener; ensures that the correct class sends to/is affected by this menu
	 * @return a menu bar to be added to a container
	 */
	public JMenuBar generateMenuBar(final VariableMenuListener ml)
	{
		JMenuBar menubar = new JMenuBar();
		final JMenu mainMenu = new JMenu("Variables");
		
		if(this.varlistEle != null)
		{
			for(Element varlist : this.varlistEle.getChildren())
			{
				String menuName = varlist.getAttributeByName("name");
				JMenu menu = makeVarList(varlist, mainMenu, ml);
				if(this.isMenuVisible.containsKey(menuName))
				{
					menu.setVisible(this.isMenuVisible.get(menuName));
				}
				else
				{
					if(varlist.getAttributeByName("defaultOn").equals("no")) 
					{
						menu.setVisible(false);
						this.isMenuVisible.put(menuName, false);
					}	
					else
					{
						menu.setVisible(true);
						this.isMenuVisible.put(menuName, true);
					}
				}
				if(this.settableMenus.containsKey(menuName))
					this.settableMenus.get(menuName).add(menu);
							
				mainMenu.add(menu);
			}
		}

		JMenu extendedMenu = new JMenu("Extended");
		fillInExtendedMenu(extendedMenu, mainMenu, ml);
		this.settableMenus.get("Extended").add(extendedMenu);		
		
		mainMenu.setMaximumSize(menuDim);
		mainMenu.setMinimumSize(menuDim);
		mainMenu.setPreferredSize(menuDim);
		
		mainMenu.add(extendedMenu);
		mainMenu.add(getCustomMenu(mainMenu, ml));
		
		menubar.add(mainMenu);
		
		return menubar;
	}
	
	/**
	 * Creates a menu which contains all children of the input root. 
	 * If the root is a correlation, attributes are used instead of the tag when choosing a title.
	 * Called recursively until no children remain
	 * @param root Children are added as menus
	 * @param mainMenu Used to set the title when a menu/menu item is selected
	 * @param ml The listener which is impacted when a menu/menu item is selected
	 * @return A menu containing all children of root, in the same XML tree format 
	 */
	private JMenu getMenuItemOfChildren(Element root, final JMenu mainMenu, final VariableMenuListener ml) {
		JMenu menu = new JMenu(root.getTag());

		for (final Element child : root.getChildren()) {
			if (child.getChildren().size() > 0) {
				JMenu submenu = getMenuItemOfChildren(child, mainMenu, ml);
				menu.add(submenu);
			} else {
				if (!child.getTag().toLowerCase().equals("point")) {
					JMenuItem item = new JMenuItem(child.getTag());
					item.addActionListener(new ActionListener() {
						public void actionPerformed(ActionEvent event) {
							mainMenu.setText(child.getTag());
							ml.setTitle(child.getTag());
							ml.setEquation(aliasMap.get(child.getTag()));
							parent.updateResultPanel();
						}
					});
					menu.add(item);
				} else {

					final Double Dx = Double.parseDouble(child
							.getAttributeByName("Dx"));
					final Double Dy = Double.parseDouble(child
							.getAttributeByName("Dy"));
					JMenuItem item = new JMenuItem(new String(Dx + ", " + Dy));
					item.addActionListener(new ActionListener() {
						public void actionPerformed(ActionEvent event) {
							String title = child.getParent().getTag() + " " + Dx + ", " + Dy;
							mainMenu.setText(title);
							ml.setTitle(title);
							String alias = child.getParent().getTag() + "[" + 
									child.getAttributeByName("Dx") + "," + 
									child.getAttributeByName("Dy") + "]";
							ml.setEquation(aliasMap.get(alias));
							parent.updateResultPanel();
						}
					});
					if(!this.correlationPointMap.containsKey(Math.max(Dx,  Dy)))
					{
						this.correlationPointMap.put(Math.max(Dx,  Dy), new ArrayList<JMenuItem>());
					}
					this.correlationPointMap.get(Math.max(Dx, Dy)).add(item);
					if(Dx > this.minDist || Dy > this.minDist)
						item.setVisible(false);
					menu.add(item);
				}
			}
		}

		return menu;
	}
	
	public JMenu getCustomMenu(final JMenu mainMenu, final VariableMenuListener ml)
	{
		final JMenu customMenu = new JMenu("Custom");
		
		JMenuItem newItem = new JMenuItem("New...");
		newItem.addActionListener(new ActionListener(){		
			public void actionPerformed(ActionEvent arg0) {
				CustomVariable cv = new CustomVariable(aliasMap, customMenu, mainMenu, ml, parent);
				cv.showWindow();
			}
		});
		customMenu.add(newItem);
		
		return customMenu;
	}
	
	public void removeMenu(JMenu menu)
	{
		if(this.settableMenus.containsKey(menu.getText()))
			this.settableMenus.get(menu.getText()).remove(menu);
	}
	
	/**
	 * Toggles the visibility of all menus of the respective name
	 * @param key Name of the menu to be toggled
	 */
	public void toggleVisible(String key)
	{
		List<JMenu> menus = this.settableMenus.get(key);
		Boolean isVis = this.isMenuVisible.get(key);
		for(JMenu menu : menus)
		{
			menu.setVisible(!isVis);
			SwingUtilities.updateComponentTreeUI(menu);			
		}
		this.isMenuVisible.remove(key);
		this.isMenuVisible.put(key, !isVis);
	}
	
	private JMenu makeVarList(Element varlist, final JMenu mainMenu, final VariableMenuListener ml)
	{
		final JMenu menu = new JMenu(varlist.getAttributeByName("name"));
		
		for(Element child : varlist.getChildren())
		{
			if(child.getTag().equals("list"))
			{
				menu.add(makeVarList(child, mainMenu, ml));
			}
			else if(child.getTag().equals("variable"))
			{
				final String itemName = child.getAttributeByName("name");
				JMenuItem item = new JMenuItem(itemName);
				item.setToolTipText(child.getAttributeByName("description"));
				String s = child.getAttributeByName("equation");
				if(s == null) 
				{
					s = child.getAttributeByName("variable");
					if(this.aliasMap.containsKey(s))
						this.aliasMap.put(itemName, this.aliasMap.get(s));
				}
				final Equation eq = new Equation(s, this.aliasMap);

				if(!this.aliasMap.containsKey(itemName))
					this.aliasMap.put(itemName, eq);
				
				item.addActionListener(new ActionListener(){
					public void actionPerformed(ActionEvent event){
						mainMenu.setText(itemName);
						ml.setTitle(itemName);
						ml.setEquation(eq);
						parent.updateResultPanel();
					}
				});
				menu.add(item);
			}
		}
		
		return menu;
	}
	
	public Set<String> getSettableMenuNames()
	{
		return this.settableMenus.keySet();
	}
	
	public Map<String, Boolean> getIsMenuVisible()
	{
		return this.isMenuVisible;
	}
	
	public Element getSample()
	{
		return this.sampleEle;
	}
	
	public Element updateAndGetSample()
	{
		createFullSample();
		return this.sampleEle;
	}
	
	public void setMenusToMinDist(Double min)
	{
		this.minDist = min;
		for(Double d : this.correlationPointMap.keySet())
		{
			if(d <= min)
				for(JMenuItem jmi : this.correlationPointMap.get(d))
					jmi.setVisible(true);
			else
				for(JMenuItem jmi : this.correlationPointMap.get(d))
					jmi.setVisible(false);
		}
	}
	
	public void createFullSample()
	{
		this.sampleEle = new Element("results", null);
		for(int i = 0; i < this.parent.getFullFileList().getList().size(); i++)
		{
			Element root = this.parent.getFullFileList().getRoot(i);
			updateAliasMapAndSample(root, this.sampleEle);
		}
		this.sample = new ValueTypeSplitter(this.sampleEle);
		
		if(this.parent.getVariables() != null)
		{
			for(VariableElement ve : this.parent.getVariables())
			{
				ve.getExtended().removeAll();
				fillInExtendedMenu(ve.getExtended(), ve.getMainMenu(), ve);
				SwingUtilities.updateComponentTreeUI(ve.getMainMenu());
			}
		}
	}
	
	public void updateAliasMapAndSample(Element e, Element sample)
	{
		for(Element child : e.getChildren())
		{
			if(child.getChildren().size() > 0)
			{
				Element sampleChild = sample.getFirstChildByName(child.getTag());
				if(sampleChild == null)
					sampleChild = copySample(child, sample);
				
				updateAliasMapAndSample(child, sampleChild);
			}
			else
			{
				if(child.getTag().toLowerCase().equals("point"))
				{
					List<Attribute> atts = new ArrayList<Attribute>();
					atts.add(new Attribute("Dx", child.getAttributeByName("Dx")));
					atts.add(new Attribute("Dy", child.getAttributeByName("Dx")));
					Element sampleChild = sample.getChildByAttributes(atts);
					if(sampleChild == null)
						sampleChild = copySample(child, sample);
					
					String alias = e.getTag() + "[" + child.getAttributeByName("Dx") + "," + child.getAttributeByName("Dy") + "]";
					if(!this.aliasMap.containsKey(alias))
						this.aliasMap.put(alias, new Equation(child.createPathTo(), alias, aliasMap));
					if(!this.aliasMap.containsKey(e.getTag()))
						this.aliasMap.put(e.getTag(), new Equation(e.createPathTo(), e.getTag(), aliasMap));
				}
				else
				{
					Element sampleChild = sample.getFirstChildByName(child.getTag());
					if(sampleChild == null)
						sampleChild = copySample(child, sample);
					
					if(!this.aliasMap.containsKey(child.getTag()))
						this.aliasMap.put(child.getTag(), new Equation(child.createPathTo(), child.getTag(), aliasMap));
				}
			}
		}
	}
	
	private Element copySample(Element child, Element sample)
	{
		Element sampleChild = new Element(child.getTag(), sample);
		if(child.getTag().toLowerCase().equals("point"))
		{
			sampleChild.addAttribute("Dx", child.getAttributeByName("Dx"));
			sampleChild.addAttribute("Dy", child.getAttributeByName("Dy"));
		}
		if(child.getAttributeByName("Error") != null)
			sampleChild.addAttribute("Error", "0.0000");
		if(child.getValue() != null)
			sampleChild.setValue("0.0000");
		sample.addChild(sampleChild);
		
		return sampleChild;
	}
	
	private JMenu fillInExtendedMenu(JMenu extendedMenu, JMenu mainMenu, VariableMenuListener ml)
	{
		extendedMenu.add(getMenuItemOfChildren(this.sample.getParameters(), mainMenu, ml));
		extendedMenu.add(getMenuItemOfChildren(this.sample.getCorrelationParent(), mainMenu, ml));
		extendedMenu.add(getMenuItemOfChildren(this.sample.getNonCorrelationParent(), mainMenu, ml));
		extendedMenu.setVisible(this.isMenuVisible.get("Extended"));
		return extendedMenu;
	}
}
