package GUI;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;

import javax.swing.*;
import javax.swing.border.Border;

public class Variables {
	
	//variables: a list of variables which are represented in the results panel
	private List<VariableElement> variables;
	//variablePanel: the GUI panel associated with the list of variables, and the add variable button
	private JPanel variablePanel;
	//varListPanel: the GUI panel associated with the list of GUI components for variables
	private final JPanel varListPanel;
	//addVariableButton: the GUI button which adds a variable to the list when clicked
	private final JButton addVariableButton;
	//parent: the parent of this class, the main window 
	private final GUIFrame parent;
	//thisVar: a reference to this class, used in actionListeners
	private final Variables thisVar;

	Variables(GUIFrame p) {
		this.variables = new ArrayList<VariableElement>();
		this.parent = p;
		this.thisVar = this;
		this.variablePanel = new JPanel();
		this.variablePanel.setLayout(new BoxLayout(this.variablePanel, BoxLayout.Y_AXIS));
		this.variablePanel.setMaximumSize(new Dimension(1000,1000));
		this.addVariableButton = new JButton("Add variable");
		this.varListPanel = new JPanel();
		this.varListPanel.setMaximumSize(new Dimension(800,800));

		createVariableListPane();
	}
	
	public JPanel getVariablePanel() {
		return variablePanel;
	}

	private void updateVariableList() {
		setCorrectArrows();
		this.varListPanel.setLayout(new GridLayout(this.variables.size(), 1));
		SwingUtilities.updateComponentTreeUI(this.parent);
	}

	/**
	 * Initializes the varListPanel and variablePanel
	 */
	public void createVariableListPane() {
		if (this.parent.getFullFileList().getList().size() == 0)
			return;

		varListPanel.setLayout(new GridLayout(1, 1));
		
		final JPanel borderPanel = new JPanel();
		borderPanel.setLayout(new BorderLayout());

		UUID id = UUID.randomUUID();
		VariableElement ve = new VariableElement(id, this.parent, this);
		this.variables.add(ve);
		setCorrectArrows(); 
		
		this.varListPanel.add(ve.getPanel());
		
		borderPanel.add(this.varListPanel, BorderLayout.NORTH);
		
		JScrollPane jsp = new JScrollPane(borderPanel);
		jsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		Border border = BorderFactory.createEmptyBorder(0, 0, 0, 0);
		jsp.setBorder(border);
		jsp.setPreferredSize(new Dimension(360, 400));
		this.variablePanel.add(jsp);
		this.variablePanel.add(Box.createVerticalGlue());

		JPanel addPanel = new JPanel();
		addPanel.setLayout(new BoxLayout(addPanel, BoxLayout.X_AXIS));
		addVariableButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				UUID id = UUID.randomUUID();
				VariableElement ve = new VariableElement(id, parent, thisVar);
				variables.add(ve);
				varListPanel.add(ve.getPanel());
				updateVariableList();
			}
		});
		addPanel.add(addVariableButton);
		addPanel.add(Box.createHorizontalGlue());

		this.variablePanel.add(addPanel);
	}

	
	/**
	 * Removes a variable from the list of variables
	 * @param id unique ID for this variable
	 * @param menus	menus to remove from the variable menu generator
	 */
	public void removeVariable(UUID id, List<JMenu> menus)
	{
		for(VariableElement ve : this.variables)
		{
			if(ve.getId().equals(id))
			{
				this.varListPanel.remove(ve.getPanel());
				this.variables.remove(ve);
				this.addVariableButton.setEnabled(true);
				for(JMenu menu : menus)
				{
					this.parent.removeVariableMenu(menu);
				}
				break;
			}
		}
		updateVariableList();
	}
	
	/**
	 * Moves a variable up in the list of variables, both in code and graphically
	 * @param id unique ID for the variable
	 */
	public void moveVariableUp(UUID id)
	{
		for(int i = 0; i < this.variables.size(); i++)
		{
			VariableElement ve = this.variables.get(i);
			if(ve.getId().equals(id))
			{
				if(i > 0)
				{
					Component[] vars = this.varListPanel.getComponents();
					Component thisvar = vars[i];
					this.varListPanel.remove(i);
					this.varListPanel.add(thisvar, i - 1);
					Collections.swap(this.variables, i, i - 1);
					updateVariableList();
				}
				break;
			}
		}
	}

	/**
	 * Moves a variable down in the list of variables, both in code and graphically
	 * @param id unique ID for the variable
	 */
	public void moveVariableDown(UUID id)
	{
		for(int i = 0; i < this.variables.size(); i++)
		{
			VariableElement ve = this.variables.get(i);
			if(ve.getId().equals(id))
			{
				if(i < this.variables.size() - 1)
				{
					Component[] vars = this.varListPanel.getComponents();
					Component thisvar = vars[i];
					this.varListPanel.remove(i);
					this.varListPanel.add(thisvar, i + 1);
					Collections.swap(this.variables, i, i + 1);
					updateVariableList();
				}
				break;
			}
		}
	}
	
	/**
	 * Sets arrows for variables so all can move up and down.
	 * Top cannot move up, bottom cannot move down
	 */
	private void setCorrectArrows()
	{
		for(VariableElement ve : this.variables)
		{
			ve.setDownArrow(true);
			ve.setUpArrow(true);
		}
		if(this.variables.size() > 0)
		{
			this.variables.get(0).setUpArrow(false);
			this.variables.get(this.variables.size() - 1).setDownArrow(false);
		}
	}
	
	public List<VariableElement> getVariables()
	{
		return this.variables;
	}
}
