package SpecialOutputs;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import DataExtraction.Element;
import GUI.FileRoot;
import GUI.FileRootList;
import GUI.GUIFrame;
import GUI.ValueTypeSplitter;

public class CorrPrintout implements Runnable{
	private GUIFrame parent;
	private Element element;
	private int fileIndex;
	private JDialog dialog;
	
	public CorrPrintout(GUIFrame parent){
		this.parent	= parent;
	}
	
	public void run()
	{
		this.dialog = new JDialog(this.parent, "Correlation Printout");
		final JPanel panel = new JPanel();
		panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
		final JButton nextButton = new JButton("Next");
		final JLabel label = new JLabel("Select the file to extract data from:");
		Box labelBox = Box.createHorizontalBox();
		labelBox.add(label);
		labelBox.add(Box.createHorizontalGlue());
		final FileRootList files = parent.getFilteredFileList();

		final DefaultListModel<String> fileModel = new DefaultListModel<String>();
		for (FileRoot fr : files.getList())
			fileModel.addElement(fr.getShortenedFile());
		final JList<String> fileList = new JList<String>(fileModel);
		fileList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		final ListSelectionListener lsl = new ListSelectionListener() {	 
			public void valueChanged(ListSelectionEvent e) {
				if (!e.getValueIsAdjusting()) {
					fileIndex = fileList.getSelectedIndex();
					if(fileIndex >= 0)
					{
						nextButton.setEnabled(true);
						element = files.getRoot(fileIndex);
					}
					else
					{
						nextButton.setEnabled(false);
					}
				}
			}
		};
		fileList.addListSelectionListener(lsl);
		
		JScrollPane jsp = new JScrollPane(fileList);
		jsp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		jsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		
		nextButton.setEnabled(false);
		nextButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e) {
				selectCorrelation(panel);
			}
		});
		
		panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
		panel.add(labelBox);
		panel.add(jsp);
		panel.add(nextButton);
		
		dialog.add(panel);
		dialog.setMinimumSize(new Dimension(350, 250));
		dialog.setLocationRelativeTo(parent);
		dialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
		dialog.setVisible(true);
	}
	
	public String toString()
	{
		return "Correlation Printout";
	}
	
	private void selectCorrelation(JPanel panel)
	{
		panel.removeAll();
		final JButton button = new JButton("Finish");
		button.setEnabled(false);
		final DefaultListModel<String> corrModel = new DefaultListModel<String>();
		final ValueTypeSplitter splitter = new ValueTypeSplitter(this.element);
		for(Element corr : splitter.getCorrelationParent().getChildren())
			corrModel.addElement(corr.getTag());
		final JList<String> corrList = new JList<String>(corrModel);
		corrList.addListSelectionListener(new ListSelectionListener() {	 
			public void valueChanged(ListSelectionEvent e) {
				if (!e.getValueIsAdjusting()) {
					fileIndex = corrList.getSelectedIndex();
					if(fileIndex >= 0)
					{
						button.setEnabled(true);
						element = splitter.getCorrelationParent().getChildren().get(fileIndex);
					}
					else
					{
						button.setEnabled(false);
					}
				}
			}
		});

		JScrollPane jsp = new JScrollPane(corrList);
		jsp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		jsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
		
		JLabel sortLabel = new JLabel("Output:");
		Box sortBox = Box.createHorizontalBox();
		sortBox.add(sortLabel);
		sortBox.add(Box.createHorizontalGlue());
		final JRadioButton indexed = new JRadioButton("Perimeter by index");
		indexed.setSelected(true);
		JRadioButton distance = new JRadioButton("All by Euclidean distance");
		distance.setSelected(false);
		ButtonGroup group = new ButtonGroup();
		group.add(indexed);
		group.add(distance);
		
		Box radios = Box.createHorizontalBox();
		radios.add(indexed);
		radios.add(distance);
		
		final JCheckBox showErrorsBox = new JCheckBox("Show error values");
		
		JLabel label = new JLabel("Select the correlation to extract data from:");
		Box labelBox = Box.createHorizontalBox();
		labelBox.add(label);
		labelBox.add(Box.createHorizontalGlue());
		button.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent e) {
				Boolean showErrors = showErrorsBox.isSelected();
				if(indexed.isSelected())
					setPerimeterResults(showErrors);
				else
					setDistanceResults(showErrors);
			}
		});
		
		panel.add(labelBox);
		panel.add(jsp);
		panel.add(sortBox);
		panel.add(radios);
		panel.add(showErrorsBox);
		panel.add(button);
		
		SwingUtilities.updateComponentTreeUI(panel);
	}
	
	private void setDistanceResults(Boolean showErrors)
	{
		JTable table = new JTable();
		String[] columnNames;
		if(showErrors)
			columnNames = new String[3];
		else
			columnNames = new String[2];
		columnNames[0] = "Distance"; columnNames[1] = this.element.getTag();
		if(showErrors)
			columnNames[2] = "Error";
		
		List<Element> points = this.element.getChildren();
		String[][]data = new String[points.size()][columnNames.length];
		for(int i = 0; i < data.length; i++)
		{
			Element point = points.get(i);
			Double dx = Double.parseDouble(point.getAttributeByName("Dx"));
			Double dy = Double.parseDouble(point.getAttributeByName("Dy"));
			data[i][0] = Double.toString(Math.sqrt(Math.pow(dx, 2) + Math.pow(dy, 2)));
			data[i][1] = points.get(i).getValue();
			if(showErrors)
				data[i][2] = points.get(i).getAttributeByName("Error");
		}
		
		table = new JTable(data, columnNames){
			private static final long serialVersionUID = 1L;

			public boolean isCellEditable(int row, int col){
				return false;
			}
		};

		table.setRowSelectionAllowed(true);
		table.setColumnSelectionAllowed(true);
		table.getTableHeader().setReorderingAllowed(false);
		
		this.dialog.dispose();
		this.parent.setResultTable(table, true);
	}
	
	private void setPerimeterResults(Boolean showErrors)
	{
		JTable table = new JTable();
		String[] columnNames;
		if(showErrors)
			columnNames = new String[3];
		else
			columnNames = new String[2];
		columnNames[0] = "Index"; columnNames[1] = this.element.getTag();
		if(showErrors)
			columnNames[2] = "Error";

		Double farthest = getFarthestCorrPoint();
		List<Element> points = getPoints(farthest);
		
		String[][] data = new String[points.size()][columnNames.length];
		for(Integer i = 0; i < data.length; i++)
		{
			data[i][0] = i.toString();
			data[i][1] = points.get(i).getValue();
			if(showErrors)
				data[i][2] = points.get(i).getAttributeByName("Error");
		}
		
		table = new JTable(data, columnNames){
			private static final long serialVersionUID = 1L;

			public boolean isCellEditable(int row, int col){
				return false;
			}
		};

		table.setRowSelectionAllowed(true);
		table.setColumnSelectionAllowed(true);
		table.getTableHeader().setReorderingAllowed(false);
		
		this.dialog.dispose();
		this.parent.setResultTable(table, false);
	}
	
	private List<Element> getPoints(Double farthest)
	{
		List<Element> points = new ArrayList<Element>();
		//first add points along x axis
		for(Element point : element.getChildren())
		{
			Double dy = Double.parseDouble(point.getAttributeByName("Dy"));
			if(dy.equals(0.0))
				points.add(point);
		}
		//then add points along right edge
		for(Element point : element.getChildren())
		{
			Double dx = Double.parseDouble(point.getAttributeByName("Dx"));
			if(dx.equals(farthest) && !points.contains(point))
				points.add(point);
		}
		//finally add points along long edge
		for(Element point : element.getChildren())
		{
			Double dx = Double.parseDouble(point.getAttributeByName("Dx"));
			Double dy = Double.parseDouble(point.getAttributeByName("Dy"));
			if(dy.equals(dx) && !points.contains(point))
				points.add(point);
		}
		
		return points;
	}
	
	private Double getFarthestCorrPoint() {
		Double max = 0.0;
		for (Element point : this.element.getChildren()) {
			Double thisMax = Math.max(
				Double.parseDouble(point.getAttributeByName("Dx")),
				Double.parseDouble(point.getAttributeByName("Dy")));
			if (thisMax > max)
				max = thisMax;
		}
		return max;
	}
}
