module cubic_module

use DQMC_HAMILT
use DQMC_SYMM
use DQMC_LATT
use DQMC_RECLATT
use DQMC_BONDS
use DQMC_Cfg
use DQMC_STRUCT

implicit none

integer, parameter :: numneig = 19
integer, parameter :: nwave   = 19
integer, parameter :: nirrep  =  9

type cubic_t

   ! This class contains information that is pertinant for the cubic
   ! system that are needed to keep track of parameters used in
   ! calculating the position dependent pairing

   !Total number of sites and dimensions
    integer :: n, nx, ny, nz

   !Output file name
    character(len=30) :: ofile

   !b_out(i,j) returns the i-th outward bonds (i=1,2,3,4) of site j
    integer, pointer :: b_out(:,:)

   !b_in(i,j) returns the i-th inward bonds (i=1,2,3,4) of site j
    integer, pointer :: b_in(:,:)

   !neig(i,j) returns the i-th neighbor of site j
    integer, pointer :: neig(:,:)

   !Bond-matrix: given two sites, it returns the bond amongst them
   !and zero if no bond is present.
    integer, pointer :: bm(:,:) 

   !Given a bond and a symmetry operation it returns the bond after
   !the symmetry operation was applied.
    integer, pointer :: map_symm_b(:,:) 

   !Given a bond, it returns the bond which is equivalent by translation
   !but that stems from the primitive cell.
    integer, pointer :: translback_b(:) 

   !Given a bond and a translation, it returns the bond after translation.
    integer, pointer :: translate_b(:,:) 

   !Reciprocal lattice (useful only for Green's function when one uses
   !boundary condition with a twist)
    type(recip_lattice_t) :: Klatt

   !Reciprocal lattice shifted to include the gamma point
    type(recip_lattice_t) :: Glatt

   !Spin of pairs
    integer, pointer :: wspin(:)

   !Irreducible representation to which the wave belongs
    integer, pointer :: irrep(:)

   !Label of the irreducible representation
    character(len=5), pointer :: irreplab(:)

   !trace of each irrep
    real(wp), pointer :: wtrace(:,:)

end type cubic_t

contains

!------------------------------------------------------------!

subroutine cubic_read_input(cfg, cubic)

    ! Gather input parameters that define the cubic

    type(config), intent(inout) :: cfg
    type(cubic_t), target, intent(inout) :: cubic

    integer :: n
    character(len=30) :: ifile

    integer, parameter :: IPT = 40, JPT = 41
    character(len=30), parameter :: cfgfmt = '(A7, 1x, i1, 1x, L, 1x, A8)'

    call getarg(1,ifile)
    open(unit=IPT,file=ifile,status='old',action='read')

    ! Create temporary file with fields in CFG 
    open(file="config.def", unit=JPT)

    ! First : write the default parameters
    do n = 1, N_Param 
       write(JPT,cfgfmt) PARAM_NAME(n), PARAM_TYPE(n), PARAM_ARRAY(n), PARAM_DVAL(n)
    enddo

    ! Second : add cubic parameters
    write(JPT, cfgfmt) "ofile  ", TYPE_STRING,  .false., "ofile   "

    ! Done with writing the file
    close(JPT)

    ! Read in CFG
    call DQMC_Read_Config(cfg, IPT)

    ! Remove temp definition file
    call system("rm config.def")

    !Set output file name
    call CFG_Get(cfg, 'ofile', cubic%ofile)

    !Set dimension along the the three directions
    call CFG_Get(cfg, 'nx', n)
    cubic%nx = n

    call CFG_Get(cfg, 'ny', n)
    cubic%ny = n

    call CFG_Get(cfg, 'nz', n)
    cubic%nz = n

    !Set total number of sites
    cubic%n = cubic%nx * cubic%ny * cubic%nz
    

end subroutine cubic_read_input

!------------------------------------------------------------!

subroutine cubic_struct_init(cubic, S, cfg)

   !  This routine fills out cubic and S.
   !  It internally generates the info that could be otherwise
   ! read using functions in the  general geomtery interface (GGI). 
   ! This is convenient here as the GGI was designed for systems 
   ! having translational invariance, which is not the
   ! case when one has a cubic.
   !  GGI routines are then called to group distances, bonds and
   ! matrix elements in equivalent classes.
   !  Elements of S are then explicitely initialized using
   ! the GGI classes.
  
   type(cubic_t), target, intent(inout) :: cubic
   type(struct), intent(out) :: S
   type(config), intent(inout) :: cfg

   type(lattice_t) :: lattice
   type(recip_lattice_t), pointer :: Rlattice
   type(recip_lattice_t), pointer :: Glattice
   type(Hamiltonian_t) :: hamilt
   type(Symm_Operations) :: symmOp
   type(bonds_t) :: bonds

   integer :: i, j, k, n
   integer, allocatable :: tmp(:,:) 
   real*8, pointer :: clab(:,:), tvalue(:)

   character(len=5) :: string
   
  !Assign reciprocal lattice pointers
  RLattice => cubic%Klatt 
  GLattice => cubic%Glatt 
  
  !Fill basic info about lattice
   call cubic_init_latt(lattice, cubic)
   call construct_lattice(lattice)

  ! Fill reciprocal lattice
  call init_recip_latt(Lattice, Rlattice,.true.,cfg)
  call init_recip_latt(Lattice, Glattice,.false.,cfg)

  ! construct full lattice
  call construct_recip_lattice(Rlattice)
  call construct_recip_lattice(Glattice)

  !Fill hamiltonian using parameters in cubic
   call cubic_init_hamilt(hamilt, lattice, cfg)
 
  !Define the symmetry of cubic
   call cubic_init_symm(symmOp)

  !Defined the site-to-site mapping of all symmetries
   call map_symm_lattice(symmOp, lattice)
   call map_symm_recip_lattice(symmOp, Rlattice, .true.)
   call map_symm_recip_lattice(symmOp, Glattice, .false.)

  !Construct equivalence classes
   call construct_lattice_classes(symmOp, lattice)
   call construct_recip_lattice_classes(symmOp, Rlattice, .true.)
   call construct_recip_lattice_classes(symmOp, Glattice, .false.)
 
   call count_hop_class(lattice,hamilt)
   call count_local_classes(lattice,hamilt)
  !Assign phase for Green's function
   call assign_gf_phase(lattice,Rlattice%ktwist)

  !Fill Fourier Transform matrix
   call DQMC_Fill_FourierC(Rlattice, lattice)
   call DQMC_Fill_FourierC(Glattice, lattice)

  !Initialize bonds and pairing
   call cubic_init_bonds(lattice, SymmOp, Bonds, cubic)

  !Construct equivalent classes for bonds
   call cubic_analyze_bonds(cubic, Bonds)

  !========= Transfer info on S ==========!

   n  = cubic%n

   S%nsite = n
   write(string,'(i4)') cubic%nx
   S%Name = "Cubic lattice, Nx = "//adjustl(string)
   write(string,'(i4)') cubic%ny
   S%Name = trim(S%Name)//", Ny = "//adjustl(string)
   write(string,'(i4)') cubic%nz
   S%Name = trim(S%Name)//", Nz = "//adjustl(string)
   write(string,'(i4)') n
   S%Name = trim(S%Name)//", Total sites = "//adjustl(string)

   allocate(tmp(n,n))

   !Fill S%T : hopping classes in a compact storage
   call group_hopping(hamilt, n, S%n_t, tmp, tvalue)
   call DQMC_CCS_Compress(n, -1, tmp, S%T)
   deallocate(tmp)
   S%checklist(STRUCT_ADJ) = .true.

   S%nClass = lattice%nclass
   !Save distance classes in S
   allocate(S%D(n,n))
   allocate(S%F(S%nClass))
   allocate(S%clabel(S%nClass))
   S%D(1:n,1:n) = lattice%myclass(0:n-1,0:n-1)
   S%F(:) = Lattice%class_size(:)
   clab => Lattice%class_label
   do i = 1, S%nClass
      write(S%clabel(i),'(3i3)') (int(clab(i,j)), j = 1, 3)
   end do

   !Store local mapping
   S%nGroup = hamilt%nlocclass
   allocate(S%map(n))
   S%map(1:n) = Hamilt%mylocclass(0:n-1)

   !Store Green's function phase
   allocate(S%gf_phase(n,n))
   do i = 1, n
      do j = 1, n
         S%gf_phase(i,j) = lattice%gf_phase(i-1,j-1)
      enddo
   enddo
   S%checklist(STRUCT_CLASS) = .true.    

   !Store bonds in S
   S%n_b = Bonds%ntotbond
   allocate(tmp(S%n_b, S%n_b))

   tmp = 0
   do i = 0, Bonds%ntotbond-1
      k = Bonds%bond_origin(i) + 1
      j = Bonds%bond_target(i) + 1
      tmp(k,j) = i + 1 
   end do
   call DQMC_CCS_Compress(n, -1, tmp, S%B)
   deallocate(tmp)

   ! Store bond symmetry in S
   allocate(S%class_b(S%n_b ,S%n_b))
   allocate(S%size_b(Bonds%nclass_b))
   S%nClass_b = Bonds%nclass_b
   S%size_b = Bonds%class_size_b
   do j = 1, S%n_b
      do i = 1, S%n_b
         S%class_b(i,j) = Bonds%myclass_b(i-1,j-1)
      end do
   end do
   S%checklist(STRUCT_BOND) = .true.

   !Define wave properties
   call cubic_init_wave(S, cubic)

   !Construct checkerboard matrices
   call cubic_init_ckb(S, lattice)

   S%checklist(STRUCT_INIT) = .true.

   !There is some conflict between lattice and pairs when
   !deallocating. The first that deallocate is fine. The
   !second complain about deallocating somthing not allocated.
   call free_lattice(lattice)
   call free_hamilt(Hamilt)
   call free_symm(SymmOp)
   call free_bonds(Bonds)

   deallocate(tvalue)

end subroutine cubic_struct_init

!------------------------------------------------------------!

subroutine cubic_init_latt(lattice, cubic)

   type(lattice_t), intent(out) :: lattice
   type(cubic_t), intent(out)   :: cubic

   integer :: i, j
   
   lattice%ndim = 3

   ! Basis cell vectors
   lattice%ac(:,:) = ZERO
   do i = 1, 3
      lattice%ac(i,i) = ONE
   end do

   ! Super cell in unit of basis ones
   lattice%sc(:,:)  = ZERO
   lattice%sc(1,1)  = cubic%nx
   lattice%sc(2,2)  = cubic%ny
   lattice%sc(3,3)  = cubic%nz
   lattice%scc(:,:) = dble(lattice%sc(:,:))
   lattice%ncell    = cubic%n

   lattice%natom = 1
   lattice%nsites = cubic%n
   
   allocate(lattice%olabel(0:0))
   allocate(lattice%xat(3,0:0))
   write(lattice%olabel(0),*) "s0"
   lattice%xat(:,0) = ZERO

   write(*,'(74("="))') 
   write(*,*) 'Basic real space geometry info'
   write(*,*)
   write(*,*) 'Crystal atomic basis'
   write(*,'(i3,3f14.7)') (j, lattice%xat(1:rdim,j),j=0,lattice%natom-1)
   write(*,*)
   write(*,*) 'Basis cell vectors'
   write(*,'(3f14.7)') ((lattice%ac(i,j),i=1,rdim),j=1,rdim)
   write(*,*)
   write(*,'(/,A)') ' Supercell vectors (fractionary unit)'
   write(*,'(3i5)') ((lattice%sc(i,j),i=1,rdim),j=1,rdim)
   write(*,*)
   write(*,*) 'Super-Lattice vectors (cartesian)'
   write(*,'(3f14.7)') ((lattice%scc(i,j),i=1,rdim),j=1,rdim)
   write(*,*)
   write(*,'(74("="))') 

   lattice%initialized = .true.


end subroutine cubic_init_latt

!------------------------------------------------------------!


subroutine cubic_init_hamilt(hamilt, lattice, cfg)

   type(Hamiltonian_t), intent(out) :: hamilt
   type(config), intent(inout)      :: cfg
   type(lattice_t), intent(inout)   :: lattice

   integer :: i, j, k, n, nx, ny, nz, nxy, idx
   real(wp) :: mu, U, t
   real(wp), pointer :: tcfg(:)
   real(wp) :: dx(3,3)

   ! ... Executable ...

   dx = 0.d0
   do i = 1, 3
      dx(i,i) = 1.d0
   enddo

   ! Read in U, t, mu
   n = 1
   call CFG_Get(cfg, "U", n, tcfg) 
   U = tcfg(1)
   call CFG_Get(cfg, "t", n, tcfg)
   t = tcfg(1)
   call CFG_Get(cfg, "mu", n, tcfg)
   mu = tcfg(1)

   ! Determine lattice dimensions
   call CFG_Get(cfg, "nx", nx)
   call CFG_Get(cfg, "ny", ny)
   call CFG_Get(cfg, "nz", nz)
   nxy = nx * ny
   n = nxy * nz

   ! Allocate arrays
   allocate(hamilt%hop(0:n-1,0:n-1))
   allocate(hamilt%Uv(0:n-1,0:n-1))
   allocate(hamilt%Jv(0:n-1,0:n-1))
   allocate(hamilt%phase(0:n-1))

   hamilt%hop = ZERO
   hamilt%Uv  = ZERO
   hamilt%Jv  = ZERO
   hamilt%mu  = mu

   write(*,*)'Filling Hamiltonian'
   do i = 0, n - 1
      hamilt%Uv(i,i) = U
      do k = 1, 3
         do idx = -1, 1, 2
            j = hoptowho(i, idx*dx(:,k), 0, lattice)
            hamilt%hop(j, i) = t
         enddo
      enddo
   enddo

   hamilt%constructed = .true.

   call find_neighbors(hamilt)
   hamilt%neig_found = .true.

end subroutine cubic_init_hamilt

!------------------------------------------------------------!

 subroutine cubic_init_symm(SymmOp)
    type(Symm_Operations), intent(out) :: SymmOp

    integer, parameter :: nsymm = 30
    real(wp), parameter :: twopi = 2.d0*acos(-1.d0)
    integer :: i
    real(wp) :: xaxis(3,nsymm), xnorm

    SymmOp%ntotsymm = nsymm

    ! Allocate
    allocate(SymmOp%symmangle(nsymm))
    allocate(SymmOp%symmpoint(3,nsymm))
    allocate(SymmOp%symmaxis(3,nsymm))
    allocate(SymmOp%symmlabel(nsymm))

    ! Set labels
    SymmOp%symmlabel(1:13)  = "C"
    SymmOp%symmlabel(14:22) = "D"
    SymmOp%symmlabel(23:29) = "S"
    SymmOp%symmlabel(30)    = "I"

    ! Set point of symmetry
    SymmOp%symmpoint = ZERO

    ! Set symmetry axes
    ! Rotations
    xaxis(:, 1) = (/  ONE,  ONE,  ONE /)
    xaxis(:, 2) = (/ -ONE,  ONE,  ONE /)
    xaxis(:, 3) = (/  ONE, -ONE,  ONE /)
    xaxis(:, 4) = (/ -ONE, -ONE,  ONE /)
    xaxis(:, 5) = (/ ZERO, ZERO,  ONE /)
    xaxis(:, 6) = (/ ZERO,  ONE, ZERO /)
    xaxis(:, 7) = (/  ONE, ZERO, ZERO /)
    xaxis(:, 8) = (/  ONE,  ONE, ZERO /)
    xaxis(:, 9) = (/  ONE, -ONE, ZERO /)
    xaxis(:,10) = (/  ONE, ZERO,  ONE /)
    xaxis(:,11) = (/  ONE, ZERO, -ONE /)
    xaxis(:,12) = (/ ZERO,  ONE,  ONE /)
    xaxis(:,13) = (/ ZERO,  ONE, -ONE /)

    ! Reflections
    xaxis(:,14) = (/  ONE, ZERO, ZERO /)
    xaxis(:,15) = (/ ZERO,  ONE, ZERO /)
    xaxis(:,16) = (/ ZERO, ZERO,  ONE /)
    xaxis(:,17) = (/  ONE,  ONE, ZERO /)
    xaxis(:,18) = (/  ONE, -ONE, ZERO /)
    xaxis(:,19) = (/ ZERO,  ONE,  ONE /)
    xaxis(:,20) = (/ ZERO,  ONE, -ONE /)
    xaxis(:,21) = (/  ONE, ZERO,  ONE /)
    xaxis(:,22) = (/  ONE, ZERO, -ONE /)

    ! Rotoreflection
    xaxis(:,23) = (/  ONE,  ONE,  ONE /)
    xaxis(:,24) = (/ -ONE,  ONE,  ONE /)
    xaxis(:,25) = (/  ONE, -ONE,  ONE /)
    xaxis(:,26) = (/ -ONE, -ONE,  ONE /)
    xaxis(:,27) = (/ ZERO, ZERO,  ONE /)
    xaxis(:,28) = (/ ZERO,  ONE, ZERO /)
    xaxis(:,29) = (/  ONE, ZERO, ZERO /)

    !Inversion
    xaxis(:,30) = (/ ZERO, ZERO, ZERO /)

    ! Normalize symmaxis
    do i = 1, nsymm
       xnorm = max(sqrt(sum(xaxis(:,i)**2)),1.d-8)
       SymmOp%symmaxis(:,i) = xaxis(:,i) / xnorm
    end do

    SymmOp%symmangle        =  ZERO
    SymmOp%symmangle(1:4)   =  twopi / 3
    SymmOp%symmangle(5:7)   =  twopi / 4
    SymmOp%symmangle(8:13)  =  twopi / 2
    SymmOp%symmangle(23:26) =  twopi / 6
    SymmOp%symmangle(27:29) =  twopi / 4

    SymmOp%addTimeRev       = .false.
    SymmOp%initialized      = .true. 

 end subroutine cubic_init_symm

!------------------------------------------------------------!

subroutine cubic_init_bonds(lattice, SymmOp, Bonds, cubic)

   type(Symm_Operations), intent(in) :: SymmOp
   type(lattice_t), intent(in)       :: lattice

   type(bonds_t), intent(out)        :: Bonds
   type(cubic_t), intent(inout)      :: cubic

   integer  :: ib, jb
   integer  :: n, nb, nsymm
   integer  :: i, j, k, ii, jj
   integer, pointer  :: borg(:), btgt(:), map_symm(:,:)
   real(wp), dimension(3,0:numneig-1) :: dx

   ! Set up dx
   dx(:, 0) = (/ ZERO, ZERO, ZERO /)
   dx(:, 1) = (/  ONE, ZERO, ZERO /) 
   dx(:, 2) = (/ -ONE, ZERO, ZERO /) 
   dx(:, 3) = (/ ZERO,  ONE, ZERO /) 
   dx(:, 4) = (/ ZERO, -ONE, ZERO /) 
   dx(:, 5) = (/  ONE,  ONE, ZERO /) 
   dx(:, 6) = (/ -ONE, -ONE, ZERO /) 
   dx(:, 7) = (/ -ONE,  ONE, ZERO /)
   dx(:, 8) = (/  ONE, -ONE, ZERO /)
   dx(:, 9) = (/ ZERO, ZERO,  ONE /)
   dx(:,10) = (/ ZERO, ZERO, -ONE /)
   dx(:,11) = (/  ONE, ZERO,  ONE /)
   dx(:,12) = (/ -ONE, ZERO, -ONE /)
   dx(:,13) = (/ -ONE, ZERO,  ONE /)
   dx(:,14) = (/  ONE, ZERO, -ONE /)
   dx(:,15) = (/ ZERO,  ONE,  ONE /)
   dx(:,16) = (/ ZERO, -ONE, -ONE /)
   dx(:,17) = (/ ZERO, -ONE,  ONE /)
   dx(:,18) = (/ ZERO,  ONE, -ONE /)
          

   !Create aliases
   n     = cubic%n
   nb    = numneig * n
   Bonds%ntotbond = nb
   nsymm = SymmOp%nsymm
   map_symm => SymmOp%map_symm

   allocate(Bonds%bond_origin(0:nb-1))
   allocate(Bonds%bond_target(0:nb-1))
   allocate(Bonds%bond_label(0:nb-1))

   allocate(cubic%bm(0:n-1, 0:n-1))
   allocate(cubic%map_symm_b(0:nb-1, nsymm))
   allocate(cubic%translback_b(0:nb-1))
   allocate(cubic%translate_b(0:nb-1, 0:n-1))
   allocate(cubic%b_out(0:numneig-1, 0:n-1))
   allocate(cubic%b_in(0:numneig-1, 0:n-1))

   borg => Bonds%bond_origin
   btgt => Bonds%bond_target

   !Define bond-origin (bo), bond-target (bt) and
   !bond matrix (bm). 
   cubic%bm = -1
   do i = 0, n-1
      do j = 0, numneig-1
         ib = i*numneig + j
         borg(ib) = i 
         k = hoptowho(i,dx(:,j),0,lattice)
         btgt(ib) = k
         cubic%bm(k,i) = ib
         Bonds%bond_label(ib) = j
      end do
   end do

   do ib = 0, nb-1
      i = Bonds%bond_label(ib) 
      j = borg(ib)
      cubic%b_out(i,j) = ib + 1
      !Determine label of conjugate bond
      if (mod(i,2) == 0) then
         i = max(i-1, 0)
      else
         i = i + 1
      endif
      k = btgt(ib)
      cubic%b_in(i,k) = ib + 1
   enddo

   do ib = 0, nb-1
      i = borg(ib)
      j = btgt(ib)

      ! Fill map_symm_b : given ib and a symmetry operation k
      ! it returns the bond where ib is mapped by k
      do k = 1, nsymm
         ii = symmOp%map_symm(i,k)
         jj = symmOp%map_symm(j,k)
         jb = cubic%bm(jj, ii)
         if (jb < 0) stop'Bonds are not mapped properly: map_symm'
         cubic%map_symm_b(ib, k) = jb
      enddo

      ! Fill translback_b : given ib it returns the translation
      ! that maps ib to the equivalent bond stemming from the
      ! primitive cell
      k  = SymmOp%translback(i)
      cubic%translback_b(ib) = k

      ! Fill translate_b: given a bond and a translation k
      ! it returns the bond where ib is mapped by k
      do k = 0, n-1
         ii = SymmOp%translate(i, k)
         jj = SymmOp%translate(j, k)
         jb = cubic%bm(jj, ii)
         if (jb < 0)stop'Bonds are not mapped properly: translate'
         cubic%translate_b(ib, k) = jb
      enddo
      
   enddo

end subroutine cubic_init_bonds

!------------------------------------------------------------!

subroutine cubic_analyze_bonds(cubic, Bonds)

   type(bonds_t), intent(inout) :: Bonds
   type(cubic_t), intent(in)    :: cubic

   integer               :: i, it, ip, is, j, isymm , iclass , istart , csize, csizenew, itransl, &
                            jclass, idj, id, mclass, ip_transl, is_transl, ip2, is2, jstart,      &
                            nclass, nbond, nsymm
   integer, pointer      :: pbond(:,:), sbond(:,:), csizev(:)
   integer, pointer      :: myclass(:,:)
   
   !initialize local variables
   nbond   = Bonds%ntotbond
   
   !allocate internal arrays
   nclass = (numneig*(numneig+1))/2 + numneig*(nbond-numneig)
   allocate( pbond(2,nclass), sbond(2,nclass), csizev(nclass) )
                                                   
   !At the beginning each distance is a separate class and only
   !pairs with at least one atom in the primitive cell are considered.
   !The pair (pbond(ix,iclass) , sbond(ix,iclas)) is the ix-th
   !element of class "iclass"
   allocate(myclass(0:nbond-1,0:nbond-1))

   nclass = 0
   do ip =0,numneig-1
      nclass = nclass + 1
      !first loop over bonds originating inside the primitive cell
      csizev(nclass)   = 1 
      !Insert 1st member of the class
      pbond(1, nclass) = ip 
      sbond(1, nclass) = ip
      myclass(ip, ip)  = nclass
      do is = ip+1, numneig-1
         nclass = nclass + 1
         csizev(nclass)   = 2
         !Insert 1st member of the class
         pbond(1, nclass) = ip 
         sbond(1, nclass) = is
         myclass(ip, is)  = nclass 
         !Insert 2nd mamber of the class
         pbond(2, nclass) = is 
         sbond(2, nclass) = ip
         myclass(is, ip)  = nclass
      enddo
      do is = numneig, nbond-1
         nclass = nclass+1
         pbond(1,nclass) = ip 
         sbond(1,nclass) = is
         csizev(nclass)  = 1
         myclass(ip,is)  = nclass
      enddo
   enddo

   nsymm = size(cubic%map_symm_b,2)
   
   !Loop over symmetry operations. The "+1" symm op is pair permutation.
   do isymm = 1, nsymm+1

      !Loop over classes (for the first symm op, classes are made
      !of all individual bond pairs in which the first bond originates
      !in the primitive cell and the second anywhere inside the supercell)
      do iclass = 1, nclass
         istart = 1
         do 
            csize    = csizev(iclass)
            csizenew = csize
            do id=istart,csize

               !Map the atoms in the class under the symm operation
               if( isymm == nsymm+1 )then
                  ip = sbond(id, iclass)
                  is = pbond(id, iclass)
               else
                  ip = cubic%map_symm_b(pbond(id,iclass), isymm)
                  is = cubic%map_symm_b(sbond(id,iclass), isymm)
               endif

               !Find the transformed pair translated such that the bond origin of the
               !first pair is inside the primitive cell
               itransl = cubic%translback_b(ip)
               ip2     = cubic%translate_b(ip,itransl)
               is2     = cubic%translate_b(is,itransl)
               !Find the class to which the pair belongs to
               jclass  = myclass(ip2,is2)

               !if classes are different they need to be merged
               if( jclass /= iclass )then
                  !all pairs of class jclass are transfered in class "iclass"
                  jstart   = csizenew
                  csizenew = csizenew + csizev(jclass)
                  call resize_class()
                  do idj = 1, csizev(jclass)
                     ip = pbond(idj, jclass)
                     is = sbond(idj, jclass)
                     myclass(ip, is) = iclass
                     pbond(jstart+idj, iclass) = ip
                     sbond(jstart+idj, iclass) = is
                  enddo
                  !Size of jclass is nullified
                  csizev(jclass) = 0
               endif

            enddo

            !if class size did not change we have found all the classes equivalent 
            !to "iclass"
            if(csizenew == csize)exit

            !Update loop bounds to find new equivalence due to newly added elements
            istart = csizev(iclass) + 1
            csizev(iclass) = csizenew

         enddo
      enddo
   enddo
   
   !Assign a class to all the remaining pair of atoms using translational symmetry
   !redifine nclass as the number of final classes
   mclass = 0
   do i = 1, nclass
      if(csizev(i)>0) mclass = mclass + 1
      do j=1,csizev(i)
         ip = pbond(j,i)
         is = sbond(j,i)
         myclass(ip,is) = mclass
         do it = 1, cubic%n - 1
            ip_transl = cubic%translate_b(ip,it)
            is_transl = cubic%translate_b(is,it)
            myclass(ip_transl,is_transl) = mclass
         enddo
      enddo
   enddo
   deallocate(pbond, sbond, csizev)
   nclass = mclass
   
   !Compute the size of each class
   allocate(Bonds%class_size_b(nclass))
   csizev => Bonds%class_size_b
   csizev(:) = 0
   do i = 0, nbond-1
      do j = 0, nbond-1
         iclass = myclass(i,j)
         csizev(iclass) = csizev(iclass)+1
      enddo
   enddo
   
   !Save parameter in lattice
   Bonds%nclass_b = nclass
   Bonds%myclass_b => myclass
   
   contains
   
   subroutine resize_class()
   implicit none
   integer :: curr_csize
   integer, allocatable :: tmpbond(:,:)
   curr_csize = size(pbond, 1)
   if( csizenew > curr_csize ) then
 
      !Initially allocate temp array
      allocate(tmpbond(csizenew,nclass))

      !Update size of pbond without loosing its content
      tmpbond(1:curr_csize,:) = pbond(1:curr_csize,:)
      deallocate(pbond) 
      allocate(pbond(csizenew,nclass))
      pbond = tmpbond

      !Update size of sbond without loosing its content
      tmpbond(1:curr_csize,:) = sbond(1:curr_csize,:)
      deallocate(sbond) 
      allocate(sbond(csizenew,nclass))
      sbond = tmpbond

      !deallocate temp array
      deallocate(tmpbond)

   endif
   end subroutine resize_class
   
end subroutine cubic_analyze_bonds

!------------------------------------------------------------!

  subroutine cubic_init_wave(S, cubic)

    type(struct), intent(inout)  :: S
    type(cubic_t), intent(inout) :: cubic

    integer :: i

    real(wp) :: norm

    S%nWave = nwave

    allocate(S%wlabel(nWave))
    allocate(S%W(numneig,nWave))
    allocate(cubic%wspin(nWave))
    allocate(cubic%irrep(nWave))
    allocate(cubic%irreplab(nirrep))

    cubic%irreplab(1) = "A1g"
    cubic%irreplab(2) = "A1g*"
    cubic%irreplab(3) = "Eg"
    cubic%irreplab(4) = "T1u"
    cubic%irreplab(5) = "A1g**"
    cubic%irreplab(6) = "Eg*"
    cubic%irreplab(7) = "T2g"
    cubic%irreplab(8) = "T1u*"
    cubic%irreplab(9) = "T2u"

    ! Initialize bond_wgt
    S%W = ZERO
    
    !1:1 s-wave
    S%W( 1, 1) =  ONE
    cubic%wspin(1) = 0
    cubic%irrep(1) = 1

    !2:2 extended s-wave s^*
    S%W( 2, 2) =  HALF
    S%W( 3, 2) =  HALF
    S%W( 4, 2) =  HALF
    S%W( 5, 2) =  HALF
    S%W(10, 2) =  HALF
    S%W(11, 2) =  HALF
    cubic%wspin(2) = 0
    cubic%irrep(2) = 2

    !3:3 d_{x^2-y^2}
    S%W( 2, 3) =  HALF
    S%W( 4, 3) = -HALF
    S%W( 5, 3) = -HALF
    S%W( 3, 3) =  HALF
    cubic%wspin(3) = 0
    cubic%irrep(3) = 3

    !4:3 d_{z^2}
    S%W(10, 4) =   ONE
    S%W(11, 4) =   ONE
    S%W( 2, 4) = -HALF
    S%W( 3, 4) = -HALF
    S%W( 4, 4) = -HALF
    S%W( 5, 4) = -HALF
    cubic%wspin(4) = 0
    cubic%irrep(4) = 3

    !5:4 p_x
    S%W( 2, 5) =  HALF
    S%W( 3, 5) = -HALF
    cubic%wspin(5) = 1
    cubic%irrep(5) = 4

    !6:4 p_y
    S%W( 4, 6) =  HALF
    S%W( 5, 6) = -HALF
    cubic%wspin(6) = 1
    cubic%irrep(6) = 4

    !7:4 p_z
    S%W(10, 7) =  HALF
    S%W(11, 7) = -HALF
    cubic%wspin(7) = 1
    cubic%irrep(7) = 4

    !8:5 s^{**}
    S%W( 6:9, 8) =  HALF
    S%W(12:19,8) =  HALF
    cubic%wspin(8) = 0
    cubic%irrep(8) = 5

    !9:6 d_{x2-y2}*
    S%W( 6: 9, 9)   =  ONE
    S%W(12:15, 9)   = -ONE
    cubic%wspin(9) = 0
    cubic%irrep(9) = 6

    !10:6 d_z2*
    S%W( 6: 9,10)   = -HALF
    S%W(12:15,10)   = -HALF
    S%W(16:19,10)   =  ONE
    cubic%wspin(10) = 0
    cubic%irrep(10) = 6

    !11:7 d_{xy}
    S%W( 6,11) =  HALF
    S%W( 8,11) = -HALF
    S%W( 9,11) = -HALF
    S%W( 7,11) =  HALF
    cubic%wspin(11) = 0
    cubic%irrep(11) = 7

    !12:7 d_{zx}
    S%W(12,12) =  HALF
    S%W(14,12) = -HALF
    S%W(15,12) = -HALF
    S%W(13,12) =  HALF
    cubic%wspin(12) = 0
    cubic%irrep(12) = 7

    !13:7 d_{yz}
    S%W(16,13) =  HALF
    S%W(18,13) = -HALF
    S%W(19,13) = -HALF
    S%W(17,13) =  HALF
    cubic%wspin(13) = 0
    cubic%irrep(13) = 7

    !14:8 p_x^*
    S%W( 6,14) =  ONE
    S%W( 7,14) = -ONE
    S%W( 8,14) = -ONE
    S%W( 9,14) =  ONE
    S%W(12,14) =  ONE
    S%W(13,14) = -ONE
    S%W(14,14) = -ONE
    S%W(15,14) =  ONE
    cubic%wspin(14) = 1
    cubic%irrep(14) = 8

    !15:8 p_y^*
    S%W( 6,15) =  ONE
    S%W( 7,15) = -ONE
    S%W( 8,15) =  ONE
    S%W( 9,15) = -ONE
    S%W(16,15) =  ONE
    S%W(17,15) = -ONE
    S%W(18,15) = -ONE
    S%W(19,15) =  ONE
    cubic%wspin(15) = 1
    cubic%irrep(15) = 8

    !16:8 p_z^*
    S%W(12,16) =  ONE
    S%W(13,16) = -ONE
    S%W(14,16) =  ONE
    S%W(15,16) = -ONE
    S%W(16,16) =  ONE
    S%W(17,16) = -ONE
    S%W(18,16) =  ONE
    S%W(19,16) = -ONE
    cubic%wspin(16) = 1
    cubic%irrep(16) = 8

    !17:9 f_{xy^2-xz^2}
    S%W( 6,17) =  ONE
    S%W( 7,17) = -ONE
    S%W( 8,17) = -ONE
    S%W( 9,17) =  ONE
    S%W(12,17) = -ONE
    S%W(13,17) =  ONE
    S%W(14,17) =  ONE
    S%W(15,17) = -ONE
    cubic%wspin(17) = 1
    cubic%irrep(17) = 9

    !18:9 f_{yx^2-yz^2}
    S%W( 6,18) = -ONE
    S%W( 7,18) =  ONE
    S%W( 8,18) = -ONE
    S%W( 9,18) =  ONE
    S%W(16,18) =  ONE
    S%W(17,18) = -ONE
    S%W(18,18) = -ONE
    S%W(19,18) =  ONE
    cubic%wspin(18) = 1
    cubic%irrep(18) = 9

    !19:9 f_{zx^2-zy^2}
    S%W(12,19) =  ONE
    S%W(13,19) = -ONE
    S%W(14,19) =  ONE
    S%W(15,19) = -ONE
    S%W(16,19) = -ONE
    S%W(17,19) =  ONE
    S%W(18,19) = -ONE
    S%W(19,19) =  ONE
    cubic%wspin(19) = 1
    cubic%irrep(19) = 9

    
    ! Normalize the wave matrix
    do i = 1, nwave
       norm = sum(S%W(:,i)**2)
       S%W(:,i) = S%W(:,i) / sqrt(norm)
    end do

    ! label for each wave function
    write(S%wlabel( 1),"(a20)") "           s-wave : "
    write(S%wlabel( 2),"(a20)") "          s*-wave : "
    write(S%wlabel( 3),"(a20)") "   d_{x2-y2}-wave : "
    write(S%wlabel( 4),"(a20)") "      d_{z2}-wave : "
    write(S%wlabel( 5),"(a20)") "         p_x-wave : "    
    write(S%wlabel( 6),"(a20)") "         p_y-wave : "    
    write(S%wlabel( 7),"(a20)") "         p_z-wave : "
    write(S%wlabel( 8),"(a20)") "         s**-wave : "
    write(S%wlabel( 9),"(a20)") "  d_{x2-y2}*-wave : "
    write(S%wlabel(10),"(a20)") "     d_{z2}*-wave : "
    write(S%wlabel(11),"(a20)") "      d_{xy}-wave : "
    write(S%wlabel(12),"(a20)") "      d_{zx}-wave : "
    write(S%wlabel(13),"(a20)") "      d_{yz}-wave : "
    write(S%wlabel(14),"(a20)") "        p_x*-wave : "    
    write(S%wlabel(15),"(a20)") "        p_y*-wave : "    
    write(S%wlabel(16),"(a20)") "        p_z*-wave : "
    write(S%wlabel(17),"(a20)") " f_{xy2-xz2}-wave : "
    write(S%wlabel(18),"(a20)") " f_{yz2-yx2}-wave : "
    write(S%wlabel(19),"(a20)") " f_{zx2-zy2}-wave : "

    S%checklist(STRUCT_WAVE) = .true.

  end subroutine cubic_init_wave

  !---------------------------------------------------------------------!

  subroutine cubic_init_ckb(S, lattice)

     use DQMC_struct
     use DQMC_latt

     implicit none

     type(lattice_t), intent(in) :: lattice
     type(struct), intent(inout) :: S

     integer :: ht, i, j, k, n, id
     integer :: ckb(lattice%nsites, lattice%nsites)
     logical :: vv(lattice%nsites)

     real*8 :: dx(3,3)

     n = lattice%nsites

     dx = 0.d0
     do i = 1, 3
        dx(i,i) = 1.d0
     enddo

     ckb = 0

     !checkerboard class of bonds alternates
     ! along x (id=1) as {1, 4, 1, 4 ....}
     ! along y (id=2) as {2, 5, 2, 5 ....}
     ! along z (id=3) as {3, 6, 3, 6 ....}

     do id = 1, 3
        vv = .false.
        !Initialize bond type
        ht = id
        do i = 1, n 
           !Check if site was already visited
           if ( vv(i) ) cycle
           vv(i) = .true.
           k = i
           do
              !assign bond type
              j = hoptowho(k-1, dx(:,id), 0, lattice) + 1
              ckb( j, k) = ht
              ckb( k, j) = ht
              !alternate between bond types along a direction
              ht = 2 * id + 3 - ht
              !If we are back to the beginning, exit.
              if( j == i ) exit
              vv(j) = .true.
              k = j
           enddo
        enddo
     enddo

     !Printing for visual check
     !do i = 1, n
     !   do j = i, n
     !      ht = ckb(i,j)
     !      if( ht /= 0 )then
     !        if(abs(lattice%cartpos(1,i-1)-lattice%cartpos(1,j-1)).gt.1.01)then
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,i-1)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,i-1)-0.5*dx(:,1)
     !           write(10+ht,*)
     !           write(10+ht,*)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,j-1)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,j-1)+0.5*dx(:,1)
     !           write(10+ht,*)
     !           write(10+ht,*)
     !        elseif(abs(lattice%cartpos(2,i-1)-lattice%cartpos(2,j-1)).gt.1.01)then
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,i-1)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,i-1)-0.5*dx(:,2)
     !           write(10+ht,*)
     !           write(10+ht,*)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,j-1)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,j-1)+0.5*dx(:,2)
     !           write(10+ht,*)
     !           write(10+ht,*)
     !        elseif(abs(lattice%cartpos(3,i-1)-lattice%cartpos(3,j-1)).gt.1.01)then
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,i-1)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,i-1)-0.5*dx(:,3)
     !           write(10+ht,*)
     !           write(10+ht,*)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,j-1)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,j-1)+0.5*dx(:,3)
     !           write(10+ht,*)
     !           write(10+ht,*)
     !        else
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,i-1)
     !           write(10+ht,'(3f10.5)') lattice%cartpos(:,j-1)
     !           write(10+ht,*)
     !           write(10+ht,*)
     !        endif
     !      endif
     !   enddo
     !enddo

     call DQMC_CCS_Compress(n, -1, ckb, S%ckb)

  end subroutine cubic_init_ckb

!------------------------------------------------------------!

end module cubic_module
