  program run_cubic

    use cubic_module
    use dqmc_cfg
    use dqmc_hubbard
 
    implicit none
  
    type(cubic_t) :: cubic
    type(config)  :: cfg
    type(hubbard) :: Hub

    integer :: i, j, nBin, nIter
    integer, parameter :: IPT=40, OPT=50, OPT2=60
    real :: t1, t2

    call cpu_time(t1)

    !Read input
    call cubic_read_input(cfg, cubic)

    !Initialize cubic, S and Hub
    call cubic_struct_init(cubic, Hub%S, cfg)
    call DQMC_Hub_Config(Hub, cfg)
  
    !Reset Phy2 to store spatial dependent pairing
    call Phy2_realloc(Hub%P2, Hub%S)
  
    ! Warmup sweep
    do i = 1, Hub%nWarm
       ! The second parameter means no measurement should be made
       write(*,'(A,i6,1x,i3)')' Warmup Sweep, nwrap  : ', i, Hub%G_up%nwrap
       call DQMC_Hub_Sweep(Hub, NO_MEAS0)
       call DQMC_Hub_Sweep2(Hub, Hub%nTry)
    end do
  
    !Set nwrap to the minimum value that keeps the number of 
    !recalculations unchanged
    i=Hub%L/Hub%G_up%nwrap+1
    Hub%G_up%nwrap=ceiling(dble(Hub%L)/i+1.d-8)
    Hub%G_dn%nwrap=Hub%G_up%nwrap
   
    ! We divide all the measurement into nBin,
    ! each having nPass/nBin pass.
    nBin   = Hub%P0%nBin
    nIter  = Hub%nPass/nBin
    do i = 1, nBin
       do j = 1, nIter
          write(*,'(A,i6)')' Simulation Sweep  : ', j+(i-1)*nIter
          call DQMC_Hub_Sweep(Hub, Hub%nMeas)
          call DQMC_Hub_Sweep2(Hub, Hub%nTry)
       end do
       ! Accumulate results for each bin
       call DQMC_Phy0_Avg(Hub%P0)
       ! Call local routine for P2 average
       call Phy2_Local_Avg(Hub%P2, Hub%S, cubic)
    end do
  
    ! Get average result
    call DQMC_Phy0_GetErr(Hub%P0)
    call DQMC_Phy2_GetErr(Hub%P2)
  
    !call Pair_Connected(Hub, cubic)

    !Open output files
    open(unit=OPT,file=trim(cubic%ofile))

    ! Print computed results using local routines
    call DQMC_Hub_OutputParam(Hub, OPT)
    call DQMC_Phy0_Print(Hub%P0, Hub%S, OPT)
    call Phy2_Print_local(Hub%P2, Hub%S, OPT)
    call Phy2_Print_Irrep(Hub%P2, cubic, Hub%S, OPT)

    ! Fourier transform
    call FT_print_header(cubic, OPT)
    call FT_phy0(Hub%P0, Hub%S, cubic, OPT)
    call FT_phy2(Hub%P2, Hub%S, cubic, OPT)
    call FT_Irrep(Hub%P2, Hub%S, cubic, OPT)

    ! Clean up the used storage
    call DQMC_Hub_Free(Hub)                                                                                                   
    call DQMC_Config_Free(cfg)               

    call cpu_time(t2)
    t2 = t2 - t1
    write(OPT,*) "Running time:",  t2, "(second)"
    t2 = t2 / 60 
    write(OPT,*) "Running time:",  t2, "(minute)"
    t2 = t2 / 60 
    write(OPT,*) "Running time:",  t2, "(hour)"
    t2 = t2 / 24 
    write(OPT,*) "Running time:",  t2, "(day)"

  end program

  !--------------------------------------------------------------------!
  
  subroutine Phy2_realloc(P2, S)
    !
    ! Purpose
    ! =======
    ! This subroutine reallocate some of the variable in Phy2 that are
    ! now containing position dependent pairing instead of the 
    ! usual zero-momentum case
    !
    use dqmc_struct
    use dqmc_phy2
    implicit none

    type(Phy2), intent(inout) :: P2          ! Phy2 to be initialized
    type(Struct), intent(in)  :: S

    ! ... Local Vars ...
    integer :: nb, nWave, nBin

    ! ... Executable ...

     nWave = S%nWave
     nb    = S%n_b
     nBin  = P2%nBin

     deallocate(P2%M1, P2%M2, P2%M3, P2%T)
       
     ! Allocate storages
     allocate(P2%M1(nb,nb))
     allocate(P2%M2(nb,nb))
     allocate(P2%M3(nWave*S%nclass,nBin+2))
     allocate(P2%T(S%nClass_b,1))
     
     ! Initialize 
     P2%M2  = ZERO
     P2%M3  = ZERO
     P2%sgn = ZERO

  end subroutine Phy2_realloc

!------------------------------------------------------------!

  subroutine Phy2_Local_Avg(P2, S, cubic)
    !
    ! Purpose
    ! =======
    !    This subroutine averges the pair measurements.
    !    which is stored in P2%M2. It is specific for distance
    !    dependent pairing. 
    !
    use dqmc_phy2
    use dqmc_struct
    use cubic_module
    implicit none

    type(Phy2), intent(inout) :: P2                 ! phy2
    type(Struct), intent(in)  :: S
    type(cubic_t), target, intent(in)  :: cubic

    ! ... local scalar ...
    integer :: ic, i, j, k, iw, idx, nc, iwc
    integer, pointer :: cb(:,:), outx(:), inx(:), outy(:), iny(:), wspin(:)
    real(wp) :: factor, spinpair(0:1), tmp(2)
    real(wp), pointer :: T(:)

    ! ... Executable ...

    if (P2%compute) then

       idx    = P2%idx
       nc     = S%nClass

       cb => S%class_b
       T => P2%T(:,1)
       wspin => cubic%wspin

       !symmetrize and temporarily store in T
       T = 0.d0
       do i = 1, P2%nb
         do j = 1, P2%nb
           k = cb(i,j)
           T(k) = T(k) + P2%M2(i,j)
         enddo
       enddo
       
       !Multiply by nSite since dqmc_phy2_meas erroneously divide by it.
       factor = dble(S%nsite)/dble(P2%cnt)
       T = factor * T / S%size_b

       !For each inequivalent distance compute pairing correlation
       do k = 1, S%nsite

          !pointers to bond ID
          outx => cubic%b_out(:, 0)
          inx  => cubic%b_in(:, 0)
          outy => cubic%b_out(:, k-1)
          iny  => cubic%b_in(:, k-1)

          ic = S%D(k,1)
          factor = 0.5d0 / S%F(ic)

          do i = 1, numneig
             do j = 1, numneig

                tmp(1) = T( cb( outx(i), outy(j) ) ) + T( cb(  inx(i),  iny(j) ) )
                tmp(2) = T( cb( outx(i),  iny(j) ) ) + T( cb(  inx(i), outy(j) ) )

                !Spin singlet(0) and triplet(1)
                spinpair(0) = factor * ( tmp(1) + tmp(2) )
                spinpair(1) = factor * ( tmp(1) - tmp(2) )

                !Sum up the amplitudes to construct symmetrized pairing function
                do iw = 1, P2%nwave
                   iwc = (iw-1) * nc + ic
                   P2%M3(iwc, idx) = P2%M3(iwc, idx)    &
                           + S%W(i, iw) * S%W(j, iw) * spinpair(wspin(iw))
                enddo

             enddo
          enddo
       enddo
       
       P2%sgn(idx) = P2%sgn(idx) / P2%cnt
    
       ! Reset counter and change bins
       P2%idx = P2%idx + 1
       P2%cnt = 0
       P2%M2  = ZERO
    end if

  end subroutine Phy2_Local_Avg

  !--------------------------------------------------------------------!

  subroutine Pair_Connected(Hub, cubic)
    !
    ! Purpose
    ! =======
    !    This subroutine computed the contribution to the pairing
    !    which is due to indepedent propagation of quasi-particles.
    !
    use dqmc_hubbard
    use cubic_module
    implicit none
    type(Hubbard), intent(inout) :: Hub    ! Hubbard model
    type(cubic_t), intent(in) :: cubic

    ! ... local scalar ...
    integer  :: i, j, k, avg, n
    real*8 :: G_up(Hub%n,Hub%n)
    real*8 :: G_dn(Hub%n,Hub%n)
    real*8, pointer :: tmpM3(:)
    real*8 :: tmpsgn

    ! ... Executable ...

    avg = Hub%P0%avg
    n   = Hub%n

    do i= 1, n
       do j= 1, n
          k=Hub%S%D(i,j)
          G_up(i,j)=Hub%P0%G_fun(k,avg)
          G_dn(i,j)=Hub%P0%G_fun(k,avg)
       enddo
    enddo

    call DQMC_Phy2_Pair(n, Hub%P2%M2, G_up, G_dn, Hub%S%B)

    !Save content of first bin
    i = size(Hub%P2%M3,1)
    allocate(tmpM3(i))
    tmpM3  = Hub%P2%M3(:,1)
    tmpsgn = Hub%P2%sgn(1)

    !Reuse first bin to store unconnected part
    Hub%P2%idx = 1
    Hub%P2%cnt = 1
    Hub%P2%M3(:,1) = 0.d0

    !Divide by n as done in phy2_meas
    Hub%P2%M2 = Hub%P2%M2 / n

    call Phy2_Local_Avg(Hub%P2, Hub%S, cubic)

    !Define connected part as the difference between full and disconnected
    !Assumes that the statistical fluctuations in the disconnected part 
    !are negligible compared to the fluctuations in the full pairing function. 
    tmpM3 = tmpM3 - Hub%P2%M3(:,1)
    do i = 2, avg
       Hub%P2%M3(:,i) = Hub%P2%M3(:,i) - Hub%P2%M3(:,1)
    enddo
    Hub%P2%M3(:,1) = tmpM3
    Hub%P2%sgn(1) = tmpsgn

    deallocate(tmpM3)

  end subroutine Pair_Connected

  !------------------------------------------------------------!

  subroutine Phy2_Print_Irrep(P2, cubic, S, OPT)
    !
    ! Purpose
    ! =======
    !    This subroutine prints position dependent pairing.
    !
    use dqmc_phy2
    use dqmc_struct
    use cubic_module
    implicit none

    type(Phy2), intent(inout)     :: P2   ! Phy2
    type(Struct), intent(in)      :: S    ! Underline lattice structure
    type(cubic_t), intent(inout)  :: cubic    ! Underline lattice structure
    integer, intent(in)           :: OPT  ! Output file handle

    ! ... Local scalar ...
    integer  :: nClass, avg, err, iw, ir, i, j, ii, jj, n
    integer  :: irrepdim(nirrep)

    real(wp) :: tmp(P2%nbin), sgn(P2%nBin), sum_sgn, y(P2%nBin)
    real(wp), pointer :: wtr(:,:), tmp2(:,:)

    ! ... Executable ...

    nClass = size(P2%M3,1)/S%nWave
    avg    = P2%avg
    err    = P2%err
    n      = P2%nBin

    allocate(cubic%wtrace(nirrep*nClass, err))
    wtr => cubic%wtrace

    wtr = 0.d0
    irrepdim = 0
    !Construct the trace for each irrep
    do iw = 1, S%nwave
       ir = cubic%irrep(iw)
       i  = (ir-1) * nClass + 1
       j  = ir * nClass
       ii = (iw-1)*nClass+1
       jj = iw*nClass
       wtr(i:j,:) = wtr(i:j,:) + P2%M3(ii:jj,:)
       irrepdim(ir) = irrepdim(ir) + 1
    enddo
    do ir = 1, nirrep
       i  = (ir-1) * nClass + 1
       j  = ir * nClass
       wtr(i:j,:) = wtr(i:j,:) / irrepdim(ir)
    enddo

    
    !Compute errorbars on irreps
    tmp = P2%sgn(1:n)
    call DQMC_JackKnife(n, P2%sgn(avg), P2%sgn(err), tmp, &
         y, sgn, sum_sgn)
    
    do i = 1, nirrep*nClass
       tmp = wtr(i, 1:n)
       call DQMC_SignJackKnife(n, wtr(i, avg), wtr(i, err), &
            tmp, y, sgn, sum_sgn)
    end do
       
    !Print irreps
    do ir = 1, nirrep
       tmp2 => wtr( (ir-1)*nClass+1:ir*nClass, : )
       call DQMC_Print_RealArray(0, nClass, cubic%irreplab(ir), &
          S%clabel, tmp2(:, avg:avg), tmp2(:, err:err), OPT)
    enddo

  end subroutine Phy2_Print_Irrep

  !------------------------------------------------------------!

  subroutine Phy2_Print_Local(P2, S, OPT)
    !
    ! Purpose
    ! =======
    !    This subroutine prints position dependent pairing.
    !
    use dqmc_phy2
    use dqmc_struct
    implicit none

    type(Phy2), intent(in)    :: P2   ! Phy2
    type(Struct), intent(in)  :: S    ! Underline lattice structure
    integer, intent(in)       :: OPT  ! Output file handle

    ! ... Local scalar ...
    integer :: nClass, avg, err, iw

    real(wp), pointer :: tmp(:,:)

    ! ... Executable ...

    nClass = size(P2%M3,1)/S%nWave
    avg    = P2%avg
    err    = P2%err

    ! Scalar terms
    do iw = 1, S%nwave
       
       tmp => P2%M3( (iw-1)*nClass+1:iw*nClass, : )

       call DQMC_Print_RealArray(0, nClass, S%wlabel(iw), &
            S%clabel, tmp(:, avg:avg), tmp(:, err:err), OPT)

    enddo
    

  end subroutine Phy2_Print_Local

  !------------------------------------------------------------!

  subroutine FT_print_header(cubic, OPT)
    !
    ! Purpose
    ! =======
    !   This subroutine prints the list of k-points divided in
    !   classes for the two reciprocal lattices
    !
     use dqmc_reclatt
     use cubic_module
     implicit none

     type(cubic_t), intent(in), target :: cubic
     integer, intent(in)               :: OPT

     integer :: i, ik, j, k
     logical :: firstline
     character(len=45)  :: strgrid
     type(recip_lattice_t), pointer    :: Rlptr

     integer, parameter :: ndim = 3

     Rlptr => cubic%Klatt
     strgrid = ' Grid for Green''s function'

     !Print header for FT. Grids and classes of k-points.
     do k = 1, 2

        write(OPT,'(A)') strgrid
        write(OPT,'(A)')'  K-points'
        write(OPT,'(A)')'  Class'

        do i = 1, Rlptr%nclass_k
           firstline = .true.
           do ik = 1, Rlptr%nkpts
              if( Rlptr%myclass_k(ik) == i )then
                 if ( firstline ) then
                    write(OPT,'(2x,i3,6x,3(f10.5))')i,(Rlptr%klist(ik,j), j=1,ndim)
                    firstline = .false.
                 else
                    write(OPT,'(11x,3(f10.5))')(Rlptr%klist(ik,j), j=1,ndim)
                 endif
              endif
           enddo
           write(OPT,*)
        enddo
        write(OPT,FMT_DBLINE)

        Rlptr => cubic%Glatt
        strgrid = ' Grid for spin/charge/pairing correlations'
     enddo
 
     call flush(OPT)

  end subroutine

  !------------------------------------------------------------!

  subroutine FT_phy0(P0, S, cubic, OPT)
    !
    ! Purpose
    ! =======
    !   This subroutine computes and prints the fourier transform
    !   of the green's function and spin and charge correlations.
    !
     use dqmc_phy0
     use dqmc_struct
     use cubic_module
     implicit none


     type(Phy0), intent(inout) :: P0
     type(struct), intent(in)  :: S
     type(cubic_t), intent(in), target :: cubic
     integer, intent(in)       :: OPT

     integer :: n, nk(2)
     integer :: i, j
     character(len=30), pointer :: clabel(:,:)
     complex*16, pointer :: FCK(:,:), FCG(:,:)

     !alias
     n     =  cubic%n
     nk(1) =  cubic%Klatt%nclass_k
     nk(2) =  cubic%Glatt%nclass_k
     FCK   => cubic%Klatt%FourierC
     FCG   => cubic%Glatt%FourierC

     allocate(clabel(maxval(nk),2))
     do j = 1, 2
        do i = 1, nk(j)
           write(clabel(i,j),'(i4)') i
        enddo
     enddo
     
     !Compute the FT
     call DQMC_phy0_Get_FT(P0, S%D, S%gf_phase, FCK, FCG, nk(1), nk(2), 1, n)

     !Print FT for Green's function and spin and charge correlations
     call DQMC_Phy0_Print_FT(P0, 1, nk(1), nk(2), clabel(:,1), clabel(:,2), OPT)

  end subroutine
     
  !------------------------------------------------------------!
  
  subroutine FT_phy2(P2, S, cubic, OPT)
    !
    ! Purpose
    ! =======
    !   This subroutine computes and prints the fourier transform
    !   of pairing correlation functions.
    !
    use dqmc_phy2 
    use dqmc_struct
    use cubic_module
    implicit none

    type(Phy2), intent(in)    :: P2
    type(struct), intent(in)  :: S
    type(cubic_t), intent(in) :: cubic
    integer, intent(in)       :: OPT

    ! ... Local variables ...
    real(wp), pointer :: F(:,:), G(:,:)
    real(wp) :: sgn(P2%avg), sum_sgn
    integer :: ibin, nBin
    integer :: avg, err, ik, i, it, nk, jt, n, nClass, iw
    complex*16, pointer ::  ft_wgt(:,:), jkbin(:)
    character(len=30) :: title
    character(len=30), pointer :: clabel(:)

    nBin = P2%nBin
    avg  = P2%avg
    err  = P2%err
    n    = cubic%n
    nk   = cubic%Glatt%nclass_k
    nClass = size(P2%M3,1) / S%nWave
    ft_wgt  => cubic%Glatt%FourierC

    !Fill JackKnived bins for Sign
    sum_sgn     = sum(P2%sgn(1:nBin))
    sgn(1:nBin) = sum_sgn - P2%sgn(1:nBin)
    sgn(avg)    = sum_sgn - 1.d0

    allocate(G(nk,err))
    allocate(jkbin(nClass))
    allocate(clabel(nk))

    do i = 1, nk
       write(clabel(i),'(i4)') i
    enddo

    !Loop over properties to Fourier transform
    do iw = 1, nwave

       F => P2%M3( (iw-1)*nClass+1:iw*nClass, : )

       !Initialize Fourier Transform
       G = 0.d0

       !Fourier transform each bin (after JackKnife)
       do ibin = avg, 1, -1

          !Form JK bin estimate (first bin is the average).
          do i = 1, nClass
             jkbin(i) = F(i,avg) * sum_sgn - F(i,ibin)
          enddo
          jkbin = jkbin / sgn(ibin)

          !Construct full FT matrix for ik-th k-point
          do ik = 1, nk
             !sum over translations
             do it = 1, n
                do jt = 1, n
                   !Find atom which is the translation of "ja" by "it"
                   i = S%D(it, jt)
                   !Use class that corresponds to the (i,j) pair 
                   G(ik, ibin) = G(ik, ibin) + jkbin(i) * ft_wgt(it,ik) / ft_wgt(jt,ik)
                enddo
             enddo
          enddo
          G(:, ibin) = G(:, ibin) / n

       enddo ! Loop over bins

       do i = 1, nk
           G(i,err) = sqrt( (nBin-1) * sum((G(i,1:nBin)-G(i,avg))**2) / nBin )
       enddo

       title = "FT of "//adjustl(S%wlabel(iw))
       call DQMC_Print_RealArray(0, nk, title, &
         clabel, G(:, avg:avg), G(:, err:err), OPT)
       
    enddo ! Loop over wave

    deallocate(G, jkbin, clabel)
  
 end subroutine FT_Phy2
     
  !------------------------------------------------------------!
  
 subroutine FT_Irrep(P2, S, cubic, OPT)
    !
    ! Purpose
    ! =======
    !   This subroutine computes and prints the fourier transform
    !   of pairing correlation functions.
    !
    use dqmc_phy2 
    use dqmc_struct
    use cubic_module
    implicit none

    type(Phy2), intent(in)    :: P2
    type(struct), intent(in)  :: S
    type(cubic_t), intent(in) :: cubic
    integer, intent(in)       :: OPT

    ! ... Local variables ...
    real(wp), pointer :: F(:,:), G(:,:)
    real(wp) :: sgn(P2%avg), sum_sgn
    integer :: ibin, nBin
    integer :: avg, err, ik, i, it, nk, jt, n, nClass, iw
    complex*16, pointer ::  ft_wgt(:,:), jkbin(:)
    character(len=30) :: title
    character(len=30), pointer :: clabel(:)

    nBin = P2%nBin
    avg  = P2%avg
    err  = P2%err
    n    = cubic%n
    nk   = cubic%Glatt%nclass_k
    nClass = size(P2%M3,1) / S%nWave
    ft_wgt  => cubic%Glatt%FourierC

    !Fill JackKnived bins for Sign
    sum_sgn     = sum(P2%sgn(1:nBin))
    sgn(1:nBin) = sum_sgn - P2%sgn(1:nBin)
    sgn(avg)    = sum_sgn - 1.d0

    allocate(G(nk,err))
    allocate(jkbin(nClass))
    allocate(clabel(nk))

    do i = 1, nk
       write(clabel(i),'(i4)') i
    enddo

    !Loop over properties to Fourier transform
    do iw = 1, nirrep

       F => cubic%wtrace( (iw-1)*nClass+1:iw*nClass, : )

       !Initialize Fourier Transform
       G = 0.d0

       !Fourier transform each bin (after JackKnife)
       do ibin = avg, 1, -1

          !Form JK bin estimate (first bin is the average).
          do i = 1, nClass
             jkbin(i) = F(i,avg) * sum_sgn - F(i,ibin)
          enddo
          jkbin = jkbin / sgn(ibin)

          !Construct full FT matrix for ik-th k-point
          do ik = 1, nk
             !sum over translations
             do it = 1, n
                do jt = 1, n
                   !Find atom which is the translation of "ja" by "it"
                   i = S%D(it, jt)
                   !Use class that corresponds to the (i,j) pair 
                   G(ik, ibin) = G(ik, ibin) + jkbin(i) * ft_wgt(it,ik) / ft_wgt(jt,ik)
                enddo
             enddo
          enddo
          G(:, ibin) = G(:, ibin) / n

       enddo ! Loop over bins

       do i = 1, nk
           G(i,err) = sqrt( (nBin-1) * sum((G(i,1:nBin)-G(i,avg))**2) / nBin )
       enddo

       title = "FT of "//adjustl(cubic%irreplab(iw))
       call DQMC_Print_RealArray(0, nk, title, &
         clabel, G(:, avg:avg), G(:, err:err), OPT)
       
    enddo ! Loop over wave

    deallocate(G, jkbin, clabel)
  
end subroutine FT_Irrep
