program dqmc_ggeom

  use DQMC_Cfg
  use DQMC_Geom_Wrap
  use DQMC_Hubbard
  use dqmc_mpi

  implicit none

  real                :: t1, t2
  type(config)        :: cfg
  type(Hubbard)       :: Hub
  type(GeomWrap)      :: Gwrap
  character(len=slen) :: gfile
  logical             :: tformat
  integer             :: na, nt, nkt, nkg, i, j, k, slice, nhist
  integer             :: nbin, nIter, ntausk, msxx, fullg, exactb
  character(len=30), allocatable   :: clabelt(:), clabelg(:)
  character(len=30)   :: jobname
  complex*16, pointer :: GFC(:,:), RFC(:,:)
  integer :: IPT, OPT, FOP

  call cpu_time(t1)  

  !Count the number of processors
  call DQMC_MPI_Init(qmc_sim, PLEVEL_1)
  
  !Open input/output files
  call DQMC_IO_Open(jobname, IPT, OPT)

  !Read input (IPT unit)
  call DQMC_Read_Config(cfg, IPT)
   
  !Locally store number of sweeps between measurements
  call CFG_Get(cfg, "tausk",ntausk)

  !Save whether to use refinement for G used in measurements.
  call CFG_Get(cfg, "msxx", msxx)
  call CFG_Get(cfg, "fullg", fullg)
  call CFG_Get(cfg, "exactb", exactb)
  call CFG_Get(cfg, "nhist", nhist)
  call CFG_Get(cfg, "gfile", gfile)

  if (nhist .gt. 0) then
     call DQMC_open_file(adjustl(trim(jobname))//'.HSF.stream','unknown', FOP)
  endif

  !Determins type of geometry file
  call DQMC_Geom_Read_Def(Hub%S, gfile, tformat)
  if (.not.tformat) then
     !If free format fill gwrap
     call DQMC_Geom_Fill(Gwrap,gfile,cfg)
     !Transfer info in Hub%S
     call DQMC_Geom_Init(Gwrap,Hub%S,cfg)
  endif

  ! Initialize the rest data
  call DQMC_Hub_Config(Hub, cfg)
 
  ! Warmup sweep
  do i = 1, Hub%nWarm
     ! The second parameter means no measurement should be made
     !write(*,'(A,i6,1x,i3)')' Warmup Sweep, nwrap  : ', i, Hub%G_up%nwrap
     call DQMC_Hub_Sweep(Hub, NO_MEAS0)
     call DQMC_Hub_Sweep2(Hub, Hub%nTry)
  end do

  ! We divide all the measurement into nBin,
  ! each having nPass/nBin pass.
  nBin   = Hub%P0%nBin
  nIter  = Hub%nPass / nBin / ntausk
  if (nIter > 0) then
     do i = 1, nBin
        do j = 1, nIter
           do k = 1, ntausk
              !write(*,'(A,i6,1x,i3)')' Simulation Sweep, nwrap : ', & 
              ! &  k+(j-1)*ntausk+(i-1)*nIter*ntausk, Hub%G_up%nwrap
              call DQMC_Hub_Sweep(Hub, Hub%nMeas)
              call DQMC_Hub_Sweep2(Hub, Hub%nTry)
           enddo
           !Measure on a random time slice
           slice = 0
           call DQMC_Hub_Meas(Hub, msxx, fullG, exactb, slice)
           if (nhist .gt. 0) then
              !Write fields 
              call DQMC_Hub_Output_HSF(Hub, .false., slice, FOP)
           endif
        end do
        ! Accumulate results for each bin
        call DQMC_Phy0_Avg(Hub%P0)
        if (Hub%meas2) then
           if(Hub%P2%diagonalize)then
             call DQMC_Phy2_Avg(Hub%P2, Hub%S)
           else
             call DQMC_Phy2_Avg(Hub%P2, Hub%S%W)
           endif
        end if
     end do
  endif

  !Read configurations from file if no sweep was perfomed
  if (Hub%nWarm + Hub%nPass .eq. 0) then
     call DQMC_count_records(Hub%npass, FOP)
     nIter = Hub%npass / nbin
     do i = 1, nBin
        do j = 1, nIter / qmc_sim%aggr_size
           call DQMC_Hub_Input_HSF(Hub, .false., slice, FOP)
           call DQMC_Hub_Init_Vmat(Hub)
           call DQMC_Hub_Meas(Hub, msxx, fullG, exactb, slice)
        enddo
        call DQMC_Phy0_Avg(Hub%P0)
        if (Hub%meas2) then
           if(Hub%P2%diagonalize)then
             call DQMC_Phy2_Avg(Hub%P2, Hub%S)
           else
             call DQMC_Phy2_Avg(Hub%P2, Hub%S%W)
           endif
        end if
     enddo
  endif

  !Compute average and error
  call DQMC_Phy0_GetErr(Hub%P0)
  if (Hub%meas2) then
     call DQMC_Phy2_GetErr(Hub%P2)
  end if

  ! Print computed results
  call DQMC_Hub_OutputParam(Hub, OPT)

  call DQMC_Phy0_Print(Hub%P0, Hub%S, OPT)
  
  ! Fill the matrix of fourier weights
  call DQMC_Fill_FourierC(Gwrap%RecipLattice, Gwrap%Lattice)
  call DQMC_Fill_FourierC(Gwrap%GammaLattice, Gwrap%Lattice)

  !Aliases for Fourier transform
  na  =  Gwrap%lattice%natom
  nt  =  Gwrap%lattice%ncell
  nkt =  Gwrap%RecipLattice%nclass_k
  nkg =  Gwrap%GammaLattice%nclass_k
  RFC => Gwrap%RecipLattice%FourierC
  GFC => Gwrap%GammaLattice%FourierC

  !Compute Fourier transform
  call DQMC_phy0_GetFt(Hub%P0, Hub%S%D, Hub%S%gf_phase, &
     GFC, GFC, nkt, nkg, na, nt)
  call DQMC_Phy0_GetErrFt(Hub%P0)

  allocate(clabelt(na*nkt),clabelg(na*nkg))  

  !Print info on k-points and construct clabel
  call DQMC_Print_HeaderFT(Gwrap,OPT,clabelt,.true.)
  call DQMC_Print_HeaderFT(Gwrap,OPT,clabelg,.false.)

  !Print computed reciprocal space properties
  call DQMC_Phy0_PrintFt(Hub%P0, na, nkt, nkg, clabelt, clabelg, OPT)

  if(Hub%P2%compute)then
     if(Hub%P2%diagonalize)then
        !Obtain waves from diagonalization
        call DQMC_Phy2_GetIrrep(Hub%P2, Hub%S)
        !Get error for waves
        call DQMC_Phy2_GetErrIrrep(Hub%P2, Hub%P0%G_fun, Hub%S)
        !Analyze symmetry of pairing modes
        call DQMC_Phy2_WaveSymm(Hub%S,Hub%P2,Gwrap%SymmOp)
        !Print Pairing info
        call dqmc_phy2_PrintSymm(Hub%S, Hub%P2, OPT)
     else
        call dqmc_phy2_print(Hub%P2, Hub%S%wlabel, OPT)
     endif
  endif
  
  ! Clean up the used storage
  call DQMC_Hub_Free(Hub)
  call DQMC_Config_Free(cfg)
  
  call cpu_time(t2)
  call DQMC_MPI_Final(qmc_sim)
  write(STDOUT,*) "Running time:",  t2-t1, "(second)"

end program dqmc_ggeom

