#ifdef QMC_MPI

subroutine meas2
end subroutine meas2

#else

subroutine meas2

  use DQMC_GEOM_WRAP
  use DQMC_HUBBARD
  use DQMC_CFG
  use DQMC_GTAU

  implicit none
  
  type(config)          :: cfg
  type(Hubbard)         :: Hub 
  type(GTAU)            :: tau
  type(GeomWrap)        :: Gwrap
  integer               :: nx, ny, nz, bcx, bcy, bcz, n, L, nxy, nzl, mm, iseed
  integer               :: nBin, nIter, ntausk, nData, ivtmp(3), nprevbin, nline
  integer               :: iBin, cnt, avg, err, naccept, nreject, icount, ncount
  integer               :: i, j, k, val(8), irand, ios, HSF, iunit, first_unit, nbc
  real(wp)              :: gamma
  real(wp), pointer     :: Gavg(:,:), Gacc(:,:) 
  real(wp), pointer     :: data(:,:), data_acc(:,:)
  real(wp), pointer     :: bc_phase(:)
  real                  :: tt1, tt2
  real(wp), allocatable :: AveSpi(:), ErrSpi(:), pdata(:), gdata(:)
  integer               :: ierr, count, NIHSF
  integer, allocatable  :: HSFV(:), ff(:), HSFM(:,:)

  integer, parameter    :: f1=13, f2=15
  character(len=slen)   :: fname, fopt, gfile
  character(len=200)    :: line
  character(len=3)      :: string
  logical               :: vname, restart, used, maketaumeas, tformat
  logical, allocatable  :: ex(:)

  call CPU_TIME(tt1)

  ! read parameters from config file
  call GetArg(1, fname)
  open (unit = f1, file = fname)
  cfg%hasDef=.false.
  call DQMC_Read_Config(cfg, f1)
  close(f1)

  ! Initialize geometry
  call CFG_Get(cfg, "gfile", gfile)
  call DQMC_Geom_Read_Def(Hub%S, gfile, tformat)
  if(.not.tformat)then
     call DQMC_Geom_Fill(Gwrap,gfile,cfg)
     nxy = Gwrap%lattice%ncell
     n   = Gwrap%lattice%nsites
     nz  = n / nxy
     call build_ckb( nz, Gwrap%lattice, Hub%S)
     call DQMC_Geom_Init(Gwrap,Hub%S,cfg)
  endif

  !Look for preexisting HSF set generated using mpi
  call Configure

  !Initialize time dependent Green's function
  call DQMC_Gtau_Init2(Hub%n, Hub%L, TAU_BOTH, &
       Hub%SB%nOrth, Hub%G_up%nWrap, tau, Hub%B, Hub%WS)
  !call DQMC_Gtau_Init2(Hub%n, Hub%L, TAU_T0, &
  !     Hub%SB%nOrth, Hub%G_up%nWrap, tau, Hub%B, Hub%WS)

  !Initialize local variables
  L    = Hub%L
  !nzl  = nz*(L+1) MODIFIED TO DO FULL G
  nzl = Hub%S%nclass*(L+1)
  call CFG_Get(cfg, "nbin", nBin)
  allocate(Gavg(nzl+1, nBin+2))
  iBin = 1
  cnt  = 0
  avg  = nBin + 1
  err  = nBin + 2
  Gavg = ZERO

  ! Warm up sweeps
  do i = 1, Hub%nWarm
     ! The second parameter means no measurement should be made.
     call DQMC_Hub_Sweep(Hub, NO_MEAS0)
  end do

  ! We divide all the measurement into nBin, each having nPass/nBin pass.
  maketaumeas = .true.
  call CFG_Get(cfg, "tausk", ntausk)
  if (ntausk <= 0) then
    maketaumeas = .false.
    ntausk = 1
  endif
  nIter  = Hub%nPass / nBin / ntausk
  Hub%meas2 = .false.

  !ncount : number of sweeps in a bin
  ncount = Hub%nPass / nBin
  allocate(HSFM(NIHSF,ncount))

  !Initialize binary files where bins are written
  call Init_Flush

  !Start measurement sweeps
  do i = 1, nBin

     icount = 0
     do j = 1, nIter

        do k = 1, ntausk

           icount = icount + 1
           !Sweep through the field
           call DQMC_Hub_Sweep(Hub, Hub%nMeas)
           call DQMC_Hub_Sweep2(Hub, Hub%ntry)
           !Save fields at the end of the sweep
           call Save_HSFields()

        end do

        !Measure the time-dependent Gfun
        !call Measure() COMMENTED TO DO FULL G
        call Measure_all()

     end do

     !Average basic measurement and gfun for bin 'i"
     call DQMC_Phy0_Avg(Hub%P0)
     call Average() 

     !Flush basic measurement and gfun for bin "i"
     call Flush_Phy0
     call Flush_Gavg

     !Flush fields produced by each sweep in bin 'i"
     call Flush_History

     !Flush current set of fields (for restart)
     call Flush_HSFields

     !Say something about where you are!
     write(*,*)'Done with bin ',i

  end do

  !Close all units where binned measurements were flushed
  call Close_Flush

  ! Open output units and write header on them
  call Init_output_units()
  
  ! Write Gloc computed on root node 
  !call Dump_result() COMMENTED TO DO FULL G
  call Dump_result_all()

  ! Write basic parameters used in the simulation
  call DQMC_Hub_OutputParam(Hub, f1)

  ! Write a few more parameters (not written by default)
  write(f1,'(A,3(f6.3))')  " Boundary conditions        : ", (bc_phase(j), j=1,nbc)
  write(f1,'(A,i5)')     " Number of sweeps in a bin  : ", Hub%nPass/nBin
  write(f1,'(A,i5)')     " Number of bins             : ", nBin
  write(f1,'(A,i5)')     " Skipped sweeps in tau dependent measurement : ", ntausk
  write(f1, FMT_DBLINE) 
  
  !Print Basic measurements
  call DQMC_Phy0_GetErr(Hub%P0)
  call DQMC_Phy0_Print(Hub%P0, Hub%S, f1)

  !Compute Layer by Layer S(pi,pi)
  mm = nz * (nz + 1) / 2
  allocate(AveSpi(mm),ErrSpi(mm))
  call LayerByLayer(Hub%P0, Hub%S, AveSpi, ErrSpi, mm, nz)
  call Print_LbyL()
  write(f1, FMT_DBLINE) 

  ! Print G 
  call GetError()
  call PrintOut(f1, Gavg, "sheet, tau, Gavg(sheet, tau) +- error")
  call cpu_time(tt2)

  ! Write exec time and close units
  write(f1, *) "Execution time=", tt2-tt1
  close(f1)
  do i=1,nz
     close(ff(i))
  enddo

  ! Clean up the used storage
  call DQMC_Hub_Free(Hub)
  call DQMC_Config_Free(cfg)

contains

  ! ---------------------------------------------------------------- !
  !                contained supporting functions                    !
  ! -----------------------------------------------------------------!

  subroutine Measure_all()
    ! Purpose
    ! =======
    !    compute G(i, tau) for time dependent G_up and down
    !
    ! ... Local variables ...
    integer :: il, jl, kl, gi, idx, L1, ii, jj
    real(wp) ::factor, sgn, wgtt0, wgt0t

    ! ... Executable ...
    ! compute the first row of G tau

    L1 = L + 1
    sgn = Hub%G_up%sgn * Hub%G_dn%sgn
    factor = HALF * sgn / nxy
    Gavg(:, avg) = ZERO

    if(maketaumeas)then

      ii    = Hub%G_up%ilb
      jj    = ii
      wgtt0 = 1.d0
      wgt0t = 0.d0

      !Initialize Gtau at equal time
      tau%upt0 = Hub%G_up%G
      tau%up0t = Hub%G_up%G
      tau%dnt0 = Hub%G_dn%G
      tau%dn0t = Hub%G_dn%G
      tau%ii   = ii
      tau%ib   = jj
      tau%sfc  = max (tau%nwrap - (Hub%G_up%nWrap - Hub%G_up%wps), 1)
      
      ! For each time slice
      do il = 1, L

         ! Compute new Gtau if not equal time
         if (il > 1) then
            ii = jj + il - 1
            if ( ii >  L ) ii = ii - L  
            !wgtt0 = sign ( 1.0d0, dble(ii - jj) )
            wgtt0 = sign ( 0.5d0, dble(ii - jj) )
            wgt0t = -wgtt0
            call DQMC_MakeGtau2(tau, Hub%G_up, Hub%G_dn, ii, jj)
         endif

         ! For each class
         do jl = 1, Hub%S%nSite
            ! For each site k on sheet j 
            do kl = 1, hub%S%nSite
               hl = Hub%S%D(jl,kl)
               gi = il + (hl-1) * L1
               Gavg(gi, avg)  = Gavg(gi, avg)  + &
                   wgtt0 * ( tau%upt0(jl,kl) + tau%dnt0(jl,kl) )
               gi = hl * L1 + 1 - il
               Gavg(gi, avg)  = Gavg(gi, avg)  + &
                    wgt0t * ( tau%up0t(jl,kl) + tau%dn0t(jl,kl) )
         end do

      end do
      
      ! special case for L+1
      do il = L1, nzl, L1
         Gavg(il,avg) = Gavg(il,avg)  + 2*nxy - Gavg(il-L1+1,avg)
      end do

    endif
    
    ! average
    Gavg(1:nzl, iBin) = Gavg(1:nzl, iBin) +  (Gavg(1:nzl,avg)*factor)
    Gavg(nzl+1, iBin) = Gavg(nzl+1, iBin) + sgn 
  
    cnt = cnt + 1
    
  end subroutine Measure_all

  ! ----------------------------------------------------------------- !

  subroutine Measure()
    ! Purpose
    ! =======
    !    compute G(i, tau) for time dependent G_up and down
    !
    ! ... Local variables ...
    integer :: il, jl, kl, gi, idx, L1, ii, jj
    real(wp) ::factor, sgn, wgtt0, wgt0t

    ! ... Executable ...
    ! compute the first row of G tau

    L1 = L + 1
    sgn = Hub%G_up%sgn * Hub%G_dn%sgn
    factor = HALF * sgn / nxy
    Gavg(:, avg) = ZERO

    if(maketaumeas)then

      ii    = Hub%G_up%ilb
      jj    = ii
      wgtt0 = 1.d0
      wgt0t = 0.d0

      !Initialize Gtau at equal time
      tau%upt0 = Hub%G_up%G
      tau%up0t = Hub%G_up%G
      tau%dnt0 = Hub%G_dn%G
      tau%dn0t = Hub%G_dn%G
      tau%ii   = ii
      tau%ib   = jj
      tau%sfc  = max (tau%nwrap - (Hub%G_up%nWrap - Hub%G_up%wps), 1)
      
      ! For each time slice
      do il = 1, L

         ! Compute new Gtau if not equal time
         if (il > 1) then
            ii = jj + il - 1
            if ( ii >  L ) ii = ii - L  
            !wgtt0 = sign ( 1.0d0, dble(ii - jj) )
            wgtt0 = sign ( 0.5d0, dble(ii - jj) )
            wgt0t = -wgtt0
            call DQMC_MakeGtau2(tau, Hub%G_up, Hub%G_dn, ii, jj)
         endif

         ! For each sheet j
         do jl = 1, nz
            gi = il + (jl-1) * L1
            ! For each site k on sheet j 
            do kl = jl, n, nz
               Gavg(gi, avg)  = Gavg(gi, avg)  + &
                   wgtt0 * ( tau%upt0(kl,kl) + tau%dnt0(kl,kl) )
            end do
            gi = jl * L1 + 1 - il
            ! For each site k on sheet j 
            do kl = jl, n, nz
               Gavg(gi, avg)  = Gavg(gi, avg)  + &
                    wgt0t * ( tau%up0t(kl,kl) + tau%dn0t(kl,kl) )
            end do
         end do

      end do
      
      ! special case for L+1
      do il = L1, nzl, L1
         Gavg(il,avg) = Gavg(il,avg)  + 2*nxy - Gavg(il-L1+1,avg)
      end do

    endif
    
    ! average
    Gavg(1:nzl, iBin) = Gavg(1:nzl, iBin) +  (Gavg(1:nzl,avg)*factor)
    Gavg(nzl+1, iBin) = Gavg(nzl+1, iBin) + sgn 
  
    cnt = cnt + 1
    
  end subroutine Measure

  ! ----------------------------------------------------------------- !

  subroutine Average()
    ! Purpose
    ! =======
    !    partially averge Green's function
    !
    ! ... Local variables ...
    real(wp) :: factor
    integer  :: i
    real(wp) :: noise(nz)
    real(wp), parameter :: ratio = 10D-4

    ! ... Executable ...
    
    ! average all the values
    factor = ONE/cnt
    Gavg(:, iBin) = Gavg(:, iBin) * factor

    ! reset counter and adavence to next bin
    cnt = 0
    iBin = iBin + 1

  end subroutine Average

  ! ----------------------------------------------------------------- !

  subroutine Init_output_units

     allocate(ff(nz), ex(0:nz))

     ! open files for Gloc for each separate layer
     first_unit = f1 + 1
     do i = 1, nz

       !Find an unused unit
       do iunit = first_unit, 99
         inquire(unit=iunit, opened=used)
         if(.not.used)exit
       enddo

       !assign the unit to layer "i"
       ff(i) = iunit
       first_unit = iunit + 1

       !Check whether file exists
       write(string,'(i3)')i 
       string = adjustl(string)
       fopt = trim(fname)//"."//trim(string)//".opt"
       inquire(file=fopt, exist=ex(i))

       if(restart.and.ex(i))then

         !If continuining simulation, update header with proper number of bins
         iunit = ff(i) + 1
         open(unit=ff(i), file=fopt, action="read", status='old')
         open(unit=iunit, status='scratch')

         !First : copy file into scratch with updated bin number
         nline = 0
         do 
           read(ff(i),'(A)',iostat=ios)line
           if(ios /= 0) exit
           nline = nline + 1
           if(nline == 2)then
             write(iunit, *) L, n, (nBin+nprevbin), ntausk, nIter
           else
             write(iunit, '(A)') trim(line)
           endif
         enddo
         close(ff(i))
         rewind(iunit)

         !Second : reopen file into write mode and copy scratch into it
         open(unit=ff(i), file=fopt, action="write", status="replace" )
         do j=1,nline 
           read(iunit,'(A)')line
           write(ff(i),'(A)')trim(line)
         enddo
         close(iunit)

       else
        
         !Simply open file if the simulation is new
         open(unit=ff(i), file=fopt, action="write" )

       endif

     enddo

     !Print header if the simulation is new
     do i=1,nz
       if(.not.(restart.and.ex(i)))then
         write(ff(i), *) 'Results for Nl,Nc,Nbin,Nskip,Nmeas, NProc= '
         write(ff(i), *) L, n, nBin, ntausk, nIter
         write(ff(i), *) 't1,t2,t3,mu1,mu2,u1,u2,beta,filling = '
         write(ff(i), "(9f8.4)") Hub%t(:), Hub%mu(:), Hub%U(:), Hub%dtau*L, ONE
       endif
     enddo

     !Create or append main output
     fopt=trim(fname)//".BASIC"
     inquire(file=fopt,exist=ex(0))
     if(restart.and.ex(0))then
       open(unit=f1, file=fopt,action="write",status='old',position='append' )
       do i=1,5
         write(f1,*)'=============================='
       enddo
     else
       open(unit=f1, file=trim(fname)//".BASIC", action="write" )
     endif

  end subroutine Init_output_units

  ! ----------------------------------------------------------------- !

  subroutine Dump_result_all()
    ! Purpose
    ! =======
    !   Output results
    !

    ! ... Local variables ...
    integer :: i, j, ioff
    character(len=60) :: wfmt

    ! ... Executable ...

    wfmt = "(1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8)"

    ! output results to files
    do i = 1, nBin
       do j = 1, nz
         write(ff(j), *) ' '
         write(string,'(i3)')j; string=adjustl(string)
         write(ff(j), *) ' G(tau,',trim(string),') for bin ', i+nprevbin
         ioff = j*nxy*(L+1)
         ioff=Hub%S%D(ioff, ioff)
         write(ff(j), wfmt) Gavg(ioff-L:ioff-1, i), Gavg(nzl+1, i)
       enddo
    end do

  end subroutine Dump_result_all

  ! ----------------------------------------------------------------- !

  subroutine Dump_result()
    ! Purpose
    ! =======
    !   Output results
    !

    ! ... Local variables ...
    integer :: i, j, ioff
    character(len=60) :: wfmt

    ! ... Executable ...

    wfmt = "(1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8,1x,f11.8)"

    ! output results to files
    do i = 1, nBin
       do j=1,nz
         write(ff(j), *) ' '
         write(string,'(i3)')j; string=adjustl(string)
         write(ff(j), *) ' G(tau,',trim(string),') for bin ', i+nprevbin
         ioff=j*(L+1)
         write(ff(j), wfmt) Gavg(ioff-L:ioff-1, i), Gavg(nzl+1, i)
       enddo
    end do

  end subroutine Dump_result

  ! ----------------------------------------------------------------- !

  subroutine GetError()
    ! Purpose
    ! =======
    !    Compute std err and average
    !

    ! ... Local variables ...
    integer  :: i, sgn_idx
    real(wp) :: sum_sgn, z(nBin), y(nBin), factor

    ! ... Executable ...
    sgn_idx = nzl + 1
    
    call DQMC_JackKnife(nBin, Gavg(sgn_idx, avg), Gavg(sgn_idx, err), &
         Gavg(sgn_idx, 1:nBin), y, z, sum_sgn)

    do i = 1, nzl
       call DQMC_SignJackKnife(nBin, Gavg(i, avg), Gavg(i, err), &
            Gavg(i, 1:nBin), y, z, sum_sgn)
    end do

  end subroutine GetError

  ! ----------------------------------------------------------------- !
  
  subroutine PrintOut(OPT, G, title)
    ! Purpose
    ! =======
    !    Print out values
    !
    ! Argument
    ! ========
    integer, intent(in)  :: OPT
    real(wp), intent(in) :: G(:,:)
    character(*), intent(in) :: title

    ! ... Local variables ...
    integer :: i, j, idx
    character(*), parameter :: OPTFMT = "(i5,i5,f20.16,'   +- ', f20.16)"

    write(OPT,*) title
    idx = 1
    do j = 1, nz
       do i = 0, L
          write(OPT,OPTFMT) j, i, G(idx,avg), G(idx,err)
          idx = idx + 1
       end do
    end do
    write(OPT, FMT_DBLINE)

  end subroutine PrintOut

  ! ----------------------------------------------------------------- !
  
  subroutine Print_LbyL()
    integer :: im, i, j
    character(len=50) :: wfmt

    wfmt = "(4x,i3,7x,i3,3x,f10.6,' +-',f10.6)"

    write(f1,'(A)')' Layer by Layer Correlations'

    write(f1,'(A)')'    Layer1    Layer2     S(pi,pi) '
    im=0
    do i=1,nz
       do j=i,nz
          im=im+1
          write(f1,wfmt)i,j,AveSpi(im),ErrSpi(im)
       enddo
    enddo
  end subroutine Print_LbyL

  ! ----------------------------------------------------------------- !

   subroutine Save_HSFields
   
     integer :: ii, kk, jj, inew, ivecpos, ibitpos

     !Switch bits on and off as a representation of the HS Fields
     HSFM(:,icount) = 0
     ii = -1
     do kk = 1, L
        do jj = 1, n
           ii = ii + 1
           ivecpos = ii/32 + 1
           ibitpos = ii - (ivecpos-1)*32
           if (Hub%HSF(jj,kk )== 1) then
             inew = ibset(HSFM(ivecpos,icount),ibitpos)
           else
             inew = ibclr(HSFM(ivecpos,icount),ibitpos)
           endif
           HSFM(ivecpos,icount) = inew
        enddo
     enddo

  end subroutine

  ! ----------------------------------------------------------------- !

   subroutine Flush_History
   
     integer :: ii, kk, jj, inew, ivecpos, ibitpos

        write(41)((HSFM(jj,kk),jj=1,NIHSF),kk=1,ncount)

        call flush(41)
   
   end subroutine

  ! ----------------------------------------------------------------- !

   subroutine Flush_HSFields
     !The current fields are converted and written to output
     !together with the state of the rng. Although field conversion
     !and printing are already done in Save_HSF/Flush_history
     !redoing it here as the advantage to make this routine 
     !completely independent of the previous two : removing
     !or modifying them does not alter the behaviour of this one.
   
     integer :: ii, kk, jj, inew, ivecpos, ibitpos

     !Switch bits on and off as a representation of the HS Fields

     HSFV=0
     ii=-1
     do kk=1,L
        do jj=1,n
           ii=ii+1
           ivecpos=ii/32+1
           ibitpos=ii-(ivecpos-1)*32
           if(Hub%HSF(jj,kk)==1)then
             inew=ibset(HSFV(ivecpos),ibitpos)
           else
             inew=ibclr(HSFV(ivecpos),ibitpos)
           endif
           HSFV(ivecpos)=inew
        enddo
     enddo
   
     rewind(40)

     write(40)n, L
   
     write(40) Hub%gamma,Hub%naccept,Hub%nreject
     write(40) Hub%seed
     write(40) (HSFV(jj),jj=1,NIHSF)
   
     call flush(40)
   
   end subroutine

  ! ----------------------------------------------------------------- !

   subroutine Flush_Phy0
  
    integer :: i1, i2

     i1   = 1
     i2   = i1 + Hub%P0%nMeas - 1
     pdata(i1:i2) =  Hub%P0%meas(1:Hub%P0%nMeas,iBin)
     i1   = i2 + 1
     i2   = i2 + 3
     pdata(i1:i2) =  Hub%P0%sign(1:3,iBin)
     i1   = i2 + 1
     i2   = i2 + Hub%P0%nClass
     pdata(i1:i2) =  Hub%P0%G_fun(1:Hub%P0%nClass,iBin)
     i1   = i2 + 1
     i2   = i2 + Hub%P0%nClass
     pdata(i1:i2) =  Hub%P0%SpinXX(1:Hub%P0%nClass,iBin)
     i1   = i2 + 1
     i2   = i2 + Hub%P0%nClass
     pdata(i1:i2) =  Hub%P0%SpinZZ(1:Hub%P0%nClass,iBin)
     i1   = i2 + 1
     i2   = i2 + Hub%P0%nClass
     pdata(i1:i2) =  Hub%P0%Den0(1:Hub%P0%nClass,iBin)
     i1   = i2 + 1
     i2   = i2 + Hub%P0%nClass
     pdata(i1:i2) =  Hub%P0%Den1(1:Hub%P0%nClass,iBin)

     write(38)nprevbin+iBin,pdata(1:ndata)
     call flush(38)

   end subroutine Flush_Phy0

  ! ----------------------------------------------------------------- !

   subroutine Flush_Gavg
   
     real(wp) :: factor

     factor = ONE/cnt

     pdata(1:nzl+1)=Gavg(1:nzl+1,iBin)*factor
     write(39) nprevbin+iBin,pdata(1:(nzl+1))
     call flush(39)

   end subroutine Flush_Gavg

  ! ----------------------------------------------------------------- !

   subroutine Init_Flush
   
      integer :: bufsize, nbin_phy0, nbin_gloc, jj, jbin, nbin_hist
  
 
      nData = Hub%P0%nmeas+5*Hub%P0%nClass+3
      bufsize=max(nzl+1,ndata)
      allocate(pdata(bufsize))
      call cfg_get(cfg,'bcond', nbc, bc_phase)

      if(restart)then 

         ! Simulation restarted : find out number of bins 
         open(file=trim(fname)//'.phy0.bins',status='old', unit=38, form='unformatted')
         open(file=trim(fname)//'.gloc.bins',status='old', unit=39, form='unformatted')
         open(file=trim(fname)//'.history',status='old', unit=41, form='unformatted')
     
         ! Read the header ( phy0 has a 2-line header)
         read(38); read(38)
         read(39)
         read(41)
         nbin_phy0=0; nbin_gloc=0; nbin_hist=0
         do 
           read(38,iostat=ios)jj
           if(ios/=0)exit
           nbin_phy0=nbin_phy0+1
         enddo
         do 
           read(39,iostat=ios)jj
           if(ios/=0)exit
           nbin_gloc=nbin_gloc+1
         enddo
         do 
           read(41,iostat=ios)jj
           if(ios/=0)exit
           nbin_hist=nbin_hist+1
         enddo

         ! Number of bin may differ due to sudden job termination. Set to minimum.
         nprevbin=min(nbin_phy0, nbin_gloc, nbin_hist)
         write(*,*)'Number of bins detected',nprevbin

         !Rewind and position each unit at the same bin
         rewind(38); rewind(39); rewind(41)
         read(38); read(38)
         read(39)
         read(41)
         do jbin = 1, nprevbin
            read(38)
            read(39)
            read(41)
         enddo

      else

        ! New simulation: write out the header.
        nprevbin = 0

        open(file=trim(fname)//'.phy0.bins',status='unknown', unit=38, form='unformatted')
        write(38)Hub%P0%nmeas, Hub%S%nClass, Hub%n_U, Hub%U, Hub%n_t, Hub%t, Hub%n_mu, Hub%mu,             &
         &  Hub%L, Hub%dtau, Hub%nWarm,Hub%nMeas, Hub%nPass/nBin, Hub%idum, Hub%G_up%nWrap, Hub%SB%north,  &
         &  Hub%B%name, nxy, nz, (bc_phase(jj), jj=1, nbc)
        write(38)Hub%S%clabel(1:Hub%S%nClass)

        open(file=trim(fname)//'.gloc.bins',status='unknown', unit=39, form='unformatted')
        write(39)nz,L,n,ntausk,nIter, Hub%n_t, Hub%n_mu, Hub%n_U, Hub%t(:), Hub%mu(:), Hub%U(:), Hub%dtau

        open(file=trim(fname)//'.history',status='unknown', unit=41, form='unformatted')
        write(41)n,L

      endif

      open(file=trim(fname)//'.HSF.out',status='unknown', unit=40, form='unformatted')
      

   end subroutine Init_Flush

  ! ----------------------------------------------------------------- !

   subroutine Close_Flush

      deallocate(pdata, HSFV)
      close(38)
      close(39)
      close(40)
      close(41)

   end subroutine Close_Flush

  ! ----------------------------------------------------------------- !

   subroutine Configure

      integer :: ii, jj, kk, ivecpos, ibitpos, seed(4)

      call CFG_Get(cfg, "seed", iseed)
      
      !L is still not set at this point. So read it.
      call CFG_Get(cfg, "L", L)

      !HS Fields are stored as bits in an integer vector. Compute length of such a vector
      NIHSF=(n*L)/32+1
      allocate(HSFV(NIHSF))

      !Check if $fname.HSF.in file exists.
      inquire(file=trim(fname)//'.HSF.in',exist=restart)

      !If $fname.HSF.in exists, restart the simulation from there.
      if(restart)then

        !If a file fname.HSF.in exists try to read HS fields from there.
        open(unit=38,file=trim(fname)//'.HSF.in',status='old',form='unformatted') 

        !Read number of sites and of time slices
        read(38) j, k

        !Make sure number of sites and slices in $fname.HSF.in is compatible with input.
        if(j/=n .or. k/=L)then
          write(*,*)'Values of n or L in HSF.in are not consistent with main input',j,k
          stop
        endif

        !Load the fields and unpack the random generator status (if seed/=0)
        allocate(Hub%HSF(n,L))
        read(38)gamma,naccept,nreject

        !Read random buffer
        read(38) (seed(j), j=1,4)
        !Read HS fields...
        read(38) (HSFV(j), j=1,NIHSF)
        !...and convert from bit representation to integer
        ii = -1
        Hub%HSF = -1
        do kk = 1, L
           do jj = 1, n
              ii = ii + 1
              ivecpos = ii/32 + 1
              ibitpos = ii - (ivecpos-1) * 32
              if (btest(HSFV(ivecpos),ibitpos)) Hub%HSF(jj,kk) = 1
           enddo
        enddo

        HSF = 0

        !Set HSF flag so that fields will not be randomly overwritten later.
        call CFG_Set(cfg,"HSF",HSF)

        !We are done with restarting the simulation
        close(38)

      endif

      ! Initialize the rest data
      call DQMC_Hub_Config(Hub, cfg)
      Hub%meas2 = .false.

      !Overwrite gamma, naccept/nreject and seed using previous run data if available.
      if(restart)then
         Hub%gamma   = gamma
         Hub%naccept = naccept
         Hub%nreject = nreject
         Hub%seed    = seed
      endif

   end subroutine Configure

end subroutine meas2

  ! ---------------------------------------------------------------- !
  !                external supporting functions                     !
  ! -----------------------------------------------------------------!
  
   subroutine LayerByLayer(P0,S,AveSpi,ErrSpi,mm,nl)
    use DQMC_Phy0
    type(Phy0),   intent(in) :: P0
    type(Struct), intent(in) :: S
    integer     , intent(in) :: mm,nl
    real(wp), intent(out) :: AveSpi(mm), ErrSpi(mm)

    real(wp) :: sum_sgn, sgn(P0%nBin), y(P0%nBin), xdata(P0%nBin), Spi(P0%nBin, mm)
    real(wp) :: pifact, sumpi(mm)

    integer :: i, il, ix, iy, iz, im, ibin
    real(wp) :: xx, yy, zz

    xdata = P0%sign(P0_SGN, 1:P0%nBin)
    !Note: AveSpi and ErrSpi are used as dummy variables here
    call DQMC_JackKnife(P0%nBin, AveSpi(1), ErrSpi(1), xdata, y, sgn, sum_sgn)

     Spi=0.d0
     sumpi=0.d0
     do i=1,S%nClass
       read(S%clabel(i),*)il,xx,yy,zz
       ix=nint(xx)
       iy=nint(yy)
       iz=nint(zz)
       pifact=S%F(i)*(-1.d0)**(ix+iy+iz)
       im=1+iz+il*nl-il*(il-1)/2
       sumpi(im)=sumpi(im)+nl*S%F(i)/S%nSite
       do ibin=1,P0%nBin
         Spi(ibin,im)=Spi(ibin,im)+(2.d0*P0%SpinXX(i,ibin)+P0%SpinZZ(i,ibin))*pifact
       enddo
     enddo
     do im=1,mm
       xdata=Spi(:,im)
       call DQMC_SignJackKnife(P0%nbin, AveSpi(im), ErrSpi(im), xdata, y, sgn, sum_sgn)
     enddo
     AveSpi=AveSpi/(3.d0*sumpi)
     ErrSpi=ErrSpi/(3.d0*sumpi)
    
   end subroutine LayerByLayer
   
   subroutine build_ckb( nz, latt, S)
      use DQMC_struct
      use DQMC_latt

      implicit none

      integer, intent(in) :: nz
      type(lattice_t), intent(in) :: latt
      type(struct), intent(inout) :: S

      integer :: ht, at, i, j, k, n, id ,iz
      integer :: ckb(latt%nsites, latt%nsites)
      logical :: vv(latt%nsites)

      real*8 :: dx(3, 2)

      n = latt%nsites
      dx = 0.d0
      dx(1, 1) = 1.d0
      dx(2, 2) = 1.d0

      ckb = 0

      !Assign label to bonds along x (id=1) and y (id=2)
      do id = 1, 2
         vv = .false.
         !Initialize bond type
         ht = 2 * id - 1
         do i = 1, n 
            !Check if site was already visited
            if ( vv(i) ) cycle
            vv(i) = .true.
            !Find atom type
            at = mod( i-1, nz)
            k = i
            do
                !assign bond type
                j = hoptowho(k-1, dx(:,id), at, latt) + 1
                ckb( j, k) = ht
                ckb( k, j) = ht
                !alternate between bond types along a direction
                ht = 4 * id - 1 - ht
                !If we are back to the beginning, exit.
                if( j == i ) exit
                vv(j) = .true.
                k = j
            enddo
         enddo
      enddo

      do i = 1, n, nz 
          k = i
          ht = 5
          do iz = 1, nz - 1
             j = k + 1
             ckb( j, k) = ht
             ckb( k, j) = ht
             write(*,*) j, k, ht
             ht = 11 - ht
             k = j
          enddo
      enddo

      select case(nz)
      case (1)
         S%nckb = 4
      case (2)
         S%nckb = 5
      case default
         S%nckb = 6
      end select

      !do i = 1, n
      !   at=nint(latt%cartpos(3,i-1))+1
      !   do j = i, n
      !      if(ckb(i,j) /=0)then
      !        ht = ckb(i,j)
      !        if(abs(latt%cartpos(1,i-1)-latt%cartpos(1,j-1)).gt.1.01)then
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,i-1)
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,i-1)-0.5*dx(:,1)
      !           write(10*(at+3)+ht,*)
      !           write(10*(at+3)+ht,*)
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,j-1)
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,j-1)+0.5*dx(:,1)
      !           write(10*(at+3)+ht,*)
      !           write(10*(at+3)+ht,*)
      !        elseif(abs(latt%cartpos(2,i-1)-latt%cartpos(2,j-1)).gt.1.01)then
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,i-1)
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,i-1)-0.5*dx(:,2)
      !           write(10*(at+3)+ht,*)
      !           write(10*(at+3)+ht,*)
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,j-1)
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,j-1)+0.5*dx(:,2)
      !           write(10*(at+3)+ht,*)
      !           write(10*(at+3)+ht,*)
      !        else
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,i-1)
      !           write(10*(at+3)+ht,'(3f10.5)') latt%cartpos(:,j-1)
      !           write(10*(at+3)+ht,*)
      !           write(10*(at+3)+ht,*)
      !        endif
      !      endif
      !   enddo
      !enddo


      call DQMC_CCS_Compress(n, -1, ckb, S%ckb)

   end subroutine

#endif
