module DQMC_SHEET
#include "dqmc_include.h"

  use DQMC_UTIL
  use DQMC_CFG
  use DQMC_STRUCT
  use DQMC_HUBBARD
  implicit none 

  ! 
  ! This module defines subroutines to initialize the data structure of 
  ! a 3-dimensional sheet rectangular lattice.
  !     

contains
  
  !---------------------------------------------------------------------!

  subroutine DQMC_Init_Sheet(nx, ny, nz, S)    
    !
    ! Purpose
    ! =======
    !    This subroutine constuctures data structure for a 3D sheet
    !    rectangular lattice. Each sheet  
    !
    ! Details
    ! =======
    !    For a Nx*Ny*Nz 3D rectangular sheet lattice.
    !
    ! Arguments
    ! =========
    !
    integer, intent(in)         :: nx, ny, nz  ! dimension of the lattice
    type(Struct), intent(inout) :: S           ! Struct

    ! ... local scalar ...
    integer  :: n,nxy, nxy_nx    ! Order of matrix T and D 
    integer  :: i, j, k          ! Loop iterator
    integer  :: idx, zoff        ! 
    integer  :: long, short      ! Used in computing nClass
    integer  :: ix, iy, iz, jx, jy, jz, dx, dy, dz
    real(wp) :: sgn

    ! L1 is a hash table, whose indices are for (dx, dy, dz, layer)
    integer  :: L1(0:nx/2, 0:ny/2, 0:nz-1, 1:nz)
    integer  :: T(nx*ny*nz, nx*ny*nz)
    real(wp), pointer :: cord(:,:)

    ! ... parameters ...
    character(*), parameter :: FMT_DIST3  = "(x,'L',x,i1,i3,i3,i3)"


    ! ... Executable ...

    n   = nx*ny*nz
    nxy = nx*ny
    nxy_nx = nxy - nx
    S%nSite = n
    write(S%name,'(A,I3,A,I3,A,I3,A,I5)') &
         "3D Sheet Lattice (Nx=", &
         nx, ", Ny=", ny, ", Nz=", nz, ") total sites=", S%nSite
    
    ! In default, we construct nz+1 band

    ! --------------------------------------------------------------
    ! memory allocation
    S%n_t = 3

    ! Build adjacent matrix
    ! build the matrix layer by layer
    ! 
    zoff = 0
    T = 0
    do k = 1, nz/2
       ! build links for each layer
       ! horizontal links
       do j = 1, ny
          do i = 1, nx-1
             idx = (j-1)*nx+i+zoff
             T(idx, idx+1) = 1
             T(idx+1, idx) = 1
          end do
          ! right most node
          T((j-1)*nx+1+zoff, j*nx+zoff) = 1
          T(j*nx+zoff, (j-1)*nx+1+zoff) = 1
       end do

       ! vertical links
       do j = 1, ny-1
          do i = 1, nx
             idx = (j-1)*nx+i+zoff
             T(idx, idx+nx) = 1
             T(idx+nx, idx) = 1
          end do
       end do
       do i = 1, nx
          T(i+zoff, i+nxy_nx+zoff) = 1
          T(i+nxy_nx+zoff, i+zoff) = 1
       end do

       ! another layers
       zoff = zoff + nxy
    end do

    do k = nz/2+1, nz
       ! build links for each layer
       ! horizontal links
       do j = 1, ny
          do i = 1, nx-1
             idx = (j-1)*nx+i+zoff
             T(idx, idx+1) = 2
             T(idx+1, idx) = 2
          end do
          ! right most node
          T((j-1)*nx+1+zoff, j*nx+zoff) = 2
          T(j*nx+zoff, (j-1)*nx+1+zoff) = 2
       end do

       ! vertical links
       do j = 1, ny-1
          do i = 1, nx
             idx = (j-1)*nx+i+zoff
             T(idx, idx+nx) = 2
             T(idx+nx, idx) = 2
          end do
       end do
       do i = 1, nx
          T(i+zoff, i+nxy_nx+zoff) = 2
          T(i+nxy_nx+zoff, i+zoff) = 2
       end do

       ! another layers
       zoff = zoff + nxy
    end do

    ! build inter layer links
    zoff = 0
    do k = 1, nz-1
       do j = zoff+1, zoff+nxy
          T(j, j+nxy) = 3          ! up link
          T(j+nxy, j) = 3         ! down link
       end do
       zoff = zoff + nxy
    end do
    
    call DQMC_CCS_Compress(n, 6*n-2*nxy, T, S%T)

    ! --------------------------------------------------------------
    ! build up the distance matrix.
    long = max(nx, ny)
    short = min(nx, ny)
    S%nClass = (nz+1)*nz*(short/2+1)*(long-short/2+2)/4

    allocate(S%D(n,n))
    allocate(S%F(S%nClass))
    allocate(S%clabel(S%nClass))
    allocate(cord(S%nClass,4))

    ! set nClass as the default
    ! site i to i is a special case, do not compute it

    !! using lookup table (maybe hashing table)
    idx = 0
    S%F = 0
    L1  = 0
    do i = 1, n
       !! compute the index of i
       ix = mod(i-1, nx)+1
       iy = mod(i-1,nxy)/nx + 1
       iz = (i-1)/nxy+1 

       do j = i, n

          jx = mod(j-1, nx)+1
          jy = mod(j-1,nxy)/nx + 1
          jz = (j-1)/nxy+1 

          !! compute the distance
          dx = abs(ix-jx)
          dx = min(dx,nx-dx)
          dy = abs(iy-jy)
          dy = min(dy,ny-dy)
          dz = abs(iz-jz)
          long = max(dx, dy)
          short = min(dx, dy)
          
          ! not found
          if (L1(long,short, dz, iz) .eq. 0) then
             
             idx = idx + 1             
             L1(long, short, dz, iz) = idx
             S%D(i,j) = idx
             write(S%clabel(idx),FMT_DIST3) iz, long, short, dz
             cord(idx, 1) = long
             cord(idx, 2) = short
             cord(idx, 3) = dz
             cord(idx, 4) = iz
             
          else ! found
             
             S%D(i,j) = L1(long, short, dz, iz)
             
          end if
          
          if (i .ne. j) then
             S%F(S%D(i,j)) = S%F(S%D(i,j)) + 2
          else
             S%F(S%D(i,j)) = S%F(S%D(i,j)) + 1
          end if

          ! matrix D is symmetric
          S%D(j,i) = S%D(i,j)
       end do ! end of j
    end do

    ! --------------------------------------------------------------
    ! Other data structure
    S%nWave = 0
    allocate(S%P(n))

    S%P(1:nx:2) = -1
    S%P(2:nx:2) = 1
    do i = 2, ny, 2
       S%P((i-1)*nx+1:i*nx) =  -S%P(1:nx)
    end do

    do i = 3, ny, 2
       S%P((i-1)*nx+1:i*nx) = S%P(1:nx)
    end do

    sgn = -ONE
    do i = 1, nz-1
       S%P(i*nxy+1:(i+1)*nxy) = sgn*S%P(1:nxy)
       sgn = -sgn
    end do

    S%B%nnz = 0
    nullify(S%W)
    allocate(S%dim(3))
    S%dim(1) = nx
    S%dim(2) = ny
    S%dim(3) = nz

    ! setup mu and U 
    allocate(S%map(n))

    S%Map(1:n/2)   = 1
    S%Map(n/2+1:n) = 2

    ! enable the flag
    S%checklist = .true.
    S%checklist(STRUCT_WAVE) = .false.
    S%checklist(STRUCT_BOND) = .false.
    S%checklist(STRUCT_FT) = .false.
    deallocate(cord)

  end subroutine DQMC_INIT_SHEET


!!$  !---------------------------------------------------------------------!
!!$
!!$  subroutine DQMC_Sort_SpinSpin(S, path, npath, lpath, Lmaxpath)
!!$
!!$    type(Struct), intent(in) :: S              ! Struct
!!$    integer, intent(in)      :: npath, Lmaxpath
!!$    integer, intent(inout)   :: path(npath,Lmaxpath) 
!!$    integer, intent(in)      :: lpath(npath)
!!$
!!$    integer :: i, j
!!$
!!$    do i = 1, S%nClass
!!$        if (S%cord(i,2) .eq. 0 .and. S%cord(i,3) .eq. 0 .and. S%cord(i,4).eq.1) then
!!$	    path(1, S%cord(i,1)+1) = i
!!$	end if
!!$ 
!!$        if (S%cord(i,1) .eq. lpath(1)-1 .and. S%cord(i,3) .eq. 0 .and. S%cord(i,4).eq.1) then
!!$	    path(2, S%cord(i,2)+1) = i
!!$	end if
!!$
!!$        if (S%cord(i,1) .eq. S%cord(i,2) .and. S%cord(i,3) .eq. 0 .and. S%cord(i,4).eq.1) then
!!$	    path(3, lpath(3)-S%cord(i,1)) = i
!!$	end if
!!$    end	do	
!!$    
!!$  end subroutine    
!!$
!!$  !---------------------------------------------------------------------!
!!$
!!$  subroutine DQMC_Print_SpinSpin(S, P0, path, npath, lpath, Lmaxpath, OPT)
!!$    type(Struct), intent(in) :: S              ! Struct
!!$    type(Phy0), intent(in)   :: P0
!!$    integer, intent(in)      :: npath, Lmaxpath, OPT   
!!$    integer, intent(in)      :: lpath(npath)  !Lmaxpath=max(lpath(:));
!!$    integer, intent(in)      :: path(npath,Lmaxpath)        
!!$
!!$    integer :: i, j, idx, ii
!!$
!!$    write(OPT,'(a)') "# Spin-Spin correlation function along selected paths:"
!!$
!!$    ii=0
!!$    do i = 1, npath
!!$       do j = 1, lpath(i)
!!$          idx = path(i, j)
!!$          write(OPT, "(i4,a,f20.15,f20.15)") ii, S%clabel(idx), P0%SpinXX(idx, P0%avg), &
!!$                P0%SpinXX(idx, P0%err)
!!$          ii=ii+1
!!$       end do
!!$       ii=ii-1
!!$    end	do
!!$
!!$  end subroutine    

end module DQMC_SHEET
 
