module trap_module

use DQMC_HAMILT
use DQMC_SYMM
use DQMC_LATT
use DQMC_RECLATT
use DQMC_BONDS
use DQMC_Cfg
use DQMC_STRUCT

implicit none

!#neighbors of each site (= #lattice edges)
integer, parameter :: numneig = 4

!#bonds of each site (for pairing measurements)
integer, parameter :: numbond = 5 !FIXME: should be 007


type trap_t

   !This class contains info that are very peculiar of a
   !trapped system and/or that are needed 1)to keep track of the
   !spatial dependence of pairing correlations and 2)
   !to use in the checkerboard decomposition.

   !Number of total sites and of sites along trap diameter
    integer :: nsites, n       

   !Trap curvature
    real*8 :: Vtrap    

   !Trap off-diagonal curvature
    real*8 :: ttrap    
 
   !center of the trap
    real*8  :: trap_center(2)
    real*8  :: latt_center(2)
    logical :: excentric

   !edges are trimmed?
    logical :: trimmed

   !computes gtau ?
    logical :: gtau

   !Interaction
    real*8 :: U

   !Hopping
    real*8 :: t

   !Chemical potential
    real*8 :: mu

   !mapping from site to cartesian coordinates
    real*8, pointer :: pos(:,:)

   !mapping from integer coordinates (0:n-1, 0:n-1) to site
    integer, pointer :: map(:,:)   

   !Bonds around a site are labeled as
   !     2     
   !     |
   ! 3 - 5 - 1  (5: self-bond)
   !     |
   !     4
   !     
   !and they are "outward" when they describe the creation of an up
   !spin on the site and a down spin on i=1,2,3,4 and "inward" in the
   !opposite case.

   !b_out(i,j) returns the i-th outward bonds (i=1,2,3,4,5) of site j
    integer, pointer :: b_out(:,:)

   !b_in(i,j) returns the i-th inward bonds (i=1,2,3,4,5) of site j
    integer, pointer :: b_in(:,:)

   !neig(i,j) returns the i-th neighbor of site j
    integer, pointer :: neig(:,:)

   !ckbtype(i) returns the type of the i-th site in the ckb decomposition
    integer, pointer :: ckbtype(:)

   !class_head(i,j) contains a pair of sites (i=1,2) belonging to class j
    integer, pointer :: class_head(:,:)

   !class_head_bond(i,j) contains a pair of bonds (i=1,2) belonging to class j
    integer, pointer :: class_head_bond(:,:)

   !number of classes for singlets
    integer :: nsinglcl

   !member of the singlet classes
    integer, pointer :: singlcl(:,:)

   !wgt of a singlet class
    real(wp), pointer :: wgtcl(:)

   !Additional info for time dependent Green's function
    real(wp), pointer :: gfactor(:), Gavg(:,:)

   !Label for singlet classes
    character(len=32), pointer :: labcl(:)

    character(len=30) :: ofile

end type trap_t

contains

!------------------------------------------------------------!

subroutine trap_read_input(cfg, trap)

    ! Gather input parameters that define the trap

    type(config), intent(inout) :: cfg
    type(trap_t), target, intent(inout) :: trap

    integer :: i, n
    character(len=256) :: ifile
    real(wp), pointer :: tcfg(:)

    integer, parameter :: IPT = 40, JPT = 41
    character(len=30), parameter :: cfgfmt = '(A7, 1x, i1, 1x, L, 1x, A8)'

    integer :: status
    call get_command_argument(1, ifile, STATUS=status)

    if (status > 0) then
       call DQMC_Error("failed to retrieve input file argument", 0)
    elseif (status == -1) then
       call DQMC_Error("buffer 'ifile' is too small to hold input file name, recompile me with a larger ifile!", 0)
    end if

    open(unit=IPT,file=ifile,status='old',action='read')

    ! Create temporary file with fields in CFG 
    open(file="config.def", unit=JPT)

    ! First : write the default parameters
    do i = 1, N_Param 
       write(JPT,cfgfmt) PARAM_NAME(i), PARAM_TYPE(i), PARAM_ARRAY(i), PARAM_DVAL(i)
    enddo

    ! Second : add trap parameters
    write(JPT, cfgfmt) "ofile  ", TYPE_STRING,  .false., "ofile   "
    write(JPT, cfgfmt) "Vtrap  ", TYPE_REAL,    .false., "0.d0    "
    write(JPT, cfgfmt) "ttrap  ", TYPE_REAL,    .false., "0.d0    "
    write(JPT, cfgfmt) "trim   ", TYPE_INTEGER, .false., "0       "
    write(JPT, cfgfmt) "gtau   ", TYPE_INTEGER, .false., "0       "
    write(JPT, cfgfmt) "center ", TYPE_REAL,    .false., "0.d0    "

    ! Done with writing the file
    close(JPT)

    ! Read in CFG
    call DQMC_Read_Config(cfg, IPT)

    ! Remove temp definition file
    call system("rm config.def")

    !Set trap diameter
    call CFG_Get(cfg, 'nx', n)
    trap%n = n

    !Set chemical potential mu
    call CFG_Get(cfg, 'mu_up', n, tcfg)
    trap%mu = tcfg(1)

    call CFG_Get(cfg, 'mu_dn', n, tcfg)
    if (trap%mu .ne. tcfg(1)) then
       write(*,*) "mu_up have to be equal to mu_dn in trap"
    endif

    !Set interaction U
    call CFG_Get(cfg, 'U', n, tcfg)
    trap%U = tcfg(1)

    !Set hopping matrix element
    call CFG_Get(cfg, 't_up', n, tcfg)
    trap%t = tcfg(1)

    call CFG_Get(cfg, 't_dn', n, tcfg)
    if (trap%t .ne. tcfg(1)) then
       write(*,*) "t_up have to be equal to t_dn in trap"
    endif

    !Set output fil name
    call CFG_Get(cfg, 'ofile', trap%ofile)

    !Set Vtrap
    call CFG_Get(cfg, 'Vtrap', trap%Vtrap)

    !Set ttrap
    call CFG_Get(cfg, 'ttrap', trap%ttrap)

    !Set trimming
    call CFG_Get(cfg, 'trim', i)
    if(i > 0) then
       trap%trimmed = .true.
    else
       trap%trimmed = .false.
    endif

    !Switch on computation of G_tau
    call CFG_Get(cfg, 'gtau', i)
    if(i > 0) then
       trap%gtau = .true.
    else
       trap%gtau = .false.
    endif

    trap%latt_center(1) = 0.5*real(trap%n-1)
    trap%latt_center(2) = 0.5*real(trap%n-1)

    call CFG_Get(cfg, 'center', tcfg(1))
    trap%trap_center(1) = trap%latt_center(1) + 0.92388*tcfg(1)
    trap%trap_center(2) = trap%latt_center(2) + 0.38268*tcfg(1)
    trap%excentric = tcfg(1) /= 0

end subroutine trap_read_input

!------------------------------------------------------------!

subroutine trap_struct_init(trap, S, cfg)

   !  This routine fills out trap and S.
   !  It internally generates the info that could be otherwise
   ! read using functions in the  general geomtery interface (GGI). 
   ! This is convenient here as the GGI was designed for systems 
   ! having translational invariance, which is not the
   ! case when one has a trap.
   !  GGI routines are then called to group distances, bonds and
   ! matrix elements in equivalent classes.
   !  Elements of S are then explicitely initialized using
   ! the GGI classes.
  
   type(trap_t), intent(inout) :: trap
   type(struct), intent(out) :: S
   type(config), intent(inout) :: cfg

   type(lattice_t) :: lattice
   type(recip_lattice_t) :: Rlattice
   type(Hamiltonian_t) :: hamilt
   type(Symm_Operations) :: symmOp
   type(bonds_t) :: bonds
   type(pairing) :: pairs

   integer :: i, j, k, n, idx
   integer, allocatable :: tmp(:,:) 
   real*8 :: xi, yi
   real*8, pointer :: clab(:,:), tupvalue(:), tdnvalue(:)
   logical, allocatable :: class_visited(:)

   character(len=5) :: string
   
   integer, dimension(4,4) :: ckbmap
   !Portability issue with some version gfortran with this
   !integer, parameter, dimension(4,4) :: &
   !  ckbmap=reshape((/ 1, 2, 3, 4,  3, 2, 1, 4,  1, 4, 3, 2,  3, 4, 1, 2/),(/4, 4/))

  !Define ckbmap
   ckbmap(1, 1) = 1; ckbmap(1, 2) = 3; ckbmap(1, 3) = 1; ckbmap(1,4) = 3
   ckbmap(2, 1) = 2; ckbmap(2, 2) = 2; ckbmap(2, 3) = 4; ckbmap(2,4) = 4
   ckbmap(3, 1) = 3; ckbmap(3, 2) = 1; ckbmap(3, 3) = 3; ckbmap(3,4) = 1
   ckbmap(4, 1) = 4; ckbmap(4, 2) = 4; ckbmap(4, 3) = 2; ckbmap(4,4) = 2
  
  !Fill basic info about lattice
   call trap_init_latt(lattice, trap)
   call construct_lattice(lattice)

  !Find neighbors for ckb decomposition
   call trap_init_ckb(lattice, trap)

  !Fill reciprocal lattice
   call trap_init_rlatt(Rlattice, lattice)
   call construct_recip_lattice(Rlattice)

  !Fill hamiltonian using parameters in trap
   call trap_init_hamilt(hamilt, lattice, trap)
 
  !Define the symmetry of the trap
   call trap_init_symm(symmOp, trap)

  !Defined the site-to-site mapping of all symmetries
   call map_symm_lattice(symmOp,lattice,hamilt)
   call map_symm_recip_lattice(symmOp,Rlattice,.false.)

  !Construct equivalence classes
   call construct_lattice_classes(symmOp,lattice)
   call construct_recip_lattice_classes(symmOp,Rlattice,.false.)
 
   call count_hop_class(lattice,hamilt)
   call count_local_classes(lattice,hamilt)
   !Assign phase for Green's function
   call assign_gf_phase(lattice,Rlattice%ktwist)

  !Fill basic info about bonds and pairs. The distinction between
  !the two is minimal here since, for a trap, the concept of
  !primitive- and super- cell is identical.
   call trap_init_pair(bonds, pairs, trap)

  !Define the bond-to-bond mapping of all syymetries
   call map_symm_bonds(bonds,symmOp,lattice)
  !Construct equivalent classes for bonds
   call construct_bond_classes(bonds,symmOp)
  
  !Define the pair-to-pair mapping of all syymetries
   call map_symm_pairs(Pairs, SymmOp)
  !Construct classes for pairs
   call construct_pair_classes(Bonds, Pairs)

  !========= Transfer info on S ==========!

   n = trap%nsites
   S%nsite = n
   S%ncell = 1
   write(string,'(i5)')trap%n
   S%Name = 'Trap, Nx = '//adjustl(string)
   write(string,'(i5)')n
   S%Name = trim(S%Name)//', total sites = '//adjustl(string)

   allocate(tmp(n,n))

   !Fill S%T : hopping classes in a compact storage
   call group_hopping(hamilt, n, S%n_t, tmp, tupvalue, tdnvalue)
   call DQMC_CCS_Compress(n, -1, tmp, S%T)
   S%checklist(STRUCT_ADJ) = .true.

   S%nClass = lattice%nclass
   !Save distance classes in S
   allocate(S%D(n,n))
   allocate(S%F(S%nClass))
   allocate(S%clabel(S%nClass))
   S%D(1:n,1:n) = lattice%myclass(0:n-1,0:n-1)
   S%F(:) = Lattice%class_size(:)
   clab => Lattice%class_label
   do i = 1, S%nClass
      idx = nint(clab(i,4))
      xi = lattice%cartpos(1,idx) - trap%latt_center(1)
      yi = lattice%cartpos(2,idx) - trap%latt_center(2)
      write(S%clabel(i),'(4(f7.2))') xi, yi, (clab(i,j), j = 1, 2)
   end do

   !Store local mapping
   S%nGroup = hamilt%nlocclass
   allocate(S%map(n))
   S%map(1:n) = Hamilt%mylocclass(0:n-1)

   !Store Green's function phase
   allocate(S%gf_phase(n,n))
   do i = 1, n
      do j = 1, n
         S%gf_phase(i,j) = lattice%gf_phase(i-1,j-1)
      enddo
   enddo
   S%checklist(STRUCT_CLASS) = .true.    

   !Store checkerboard classes
   tmp=0
   S%nckb = numneig
   do i = 0, n-1
      idx = trap%ckbtype(i)
      do j = 1, numneig
         k = trap%neig(j,i)
         tmp(k+1, i+1) = ckbmap(j,idx)
      enddo
   enddo

   !do i = 1, n
   !   do j = i, n
   !      k = tmp(i,j)
   !      if(k /= 0)then
   !        if(abs(lattice%cartpos(1,i-1)-lattice%cartpos(1,j-1)).gt.1.01)then
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,i-1)
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,i-1)-0.5*(/1.0, 0.0, 0.0/)
   !           write(10+k,*)
   !           write(10+k,*)
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,j-1)
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,j-1)+0.5*(/1.0, 0.0, 0.0/)
   !           write(10+k,*)
   !           write(10+k,*)
   !        elseif(abs(lattice%cartpos(2,i-1)-lattice%cartpos(2,j-1)).gt.1.01)then
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,i-1)
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,i-1)-0.5*(/0.0, 1.0, 0.0/)
   !           write(10+k,*)
   !           write(10+k,*)
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,j-1)
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,j-1)+0.5*(/0.0, 1.0, 0.0/)
   !           write(10+k,*)
   !           write(10+k,*)
   !        else
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,i-1)
   !           write(10+k,'(3f10.5)') lattice%cartpos(:,j-1)
   !           write(10+k,*)
   !           write(10+k,*)
   !        endif
   !      endif
   !   enddo
   !enddo

   call DQMC_CCS_Compress(n, -1, tmp, S%ckb)

   !Store bonds in S
   tmp = 0
   S%n_b = Pairs%nbond
   do i = 1, Pairs%nbondv(0)
      k = Pairs%bond_origin(i,0) + 1
      j = Pairs%bond_end(i,0) + 1
      tmp(k,j) = Pairs%bond_number(i,0)
   end do
   call DQMC_CCS_Compress(n, -1, tmp, S%B)
   deallocate(tmp)

   !Store symmetry in S
   allocate(S%class_b(S%n_b ,S%n_b))
   allocate(S%size_b(pairs%nclass_p))
   S%nClass_b = Pairs%nclass_p
   S%size_b = Pairs%class_size_p
   do j = 1, S%n_b
      do i = 1, S%n_b
        S%class_b(i,j) = Pairs%myclass_p(i,j)
      enddo
   enddo
   S%checklist(STRUCT_BOND) = .true.

   !Determine the class representative (class_head)
   allocate(class_visited(S%nClass_b), trap%class_head_bond(2,S%nclass_b))
   class_visited = .false.
   do i = 1, S%n_b
      do j = 1, S%n_b
         if (.not.class_visited(S%class_b(i,j))) then
            trap%class_head_bond(1,S%class_b(i,j)) = i
            trap%class_head_bond(2,S%class_b(i,j)) = j
            class_visited(S%class_b(i,j)) = .true.
         endif
      enddo
   enddo
   deallocate(class_visited)

   call trap_init_singlet(S, Pairs, trap)

   !Determine the class representative (class_head)
   allocate(class_visited(S%nClass), trap%class_head(2,S%nclass))
   class_visited=.false.
   do i = 1, n
      do j = 1, n
         if (.not.class_visited(S%D(i,j))) then
            trap%class_head(1,S%D(i,j)) = i
            trap%class_head(2,S%D(i,j)) = j
            class_visited(S%D(i,j)) = .true.
         endif
      enddo
   enddo
   deallocate(class_visited)

   if (abs(trap%Vtrap)<1.d-8.and.abs(trap%mu)<1.d-8) then
      allocate(S%P(n))
      call trap_init_phase(lattice, S%P)
      S%checklist(STRUCT_PHASE) = .true.
   endif

   !Define wave properties
   S%nWave = 3
   allocate(S%wlabel(S%nWave), S%W(numbond, S%nWave))
   ! S*-wave
   S%wlabel(1) = 'S*-wave'
   S%W(1,1) = 0.5  
   S%W(2,1) = 0.5  
   S%W(3,1) = 0.5 
   S%W(4,1) = 0.5
   S%W(5,1) = 0.0
   ! D-wave
   S%wlabel(2) = 'D-wave'
   S%W(1,2) = -0.5  
   S%W(2,2) =  0.5  
   S%W(3,2) = -0.5 
   S%W(4,2) =  0.5
   S%W(5,2) =  0.0
   ! S-wave
   S%wlabel(3) = 'S-wave'
   S%W(1,3) = 0  
   S%W(2,3) = 0
   S%W(3,3) = 0
   S%W(4,3) = 0
   S%W(5,3) = sqrt(.5)
   ! Set wave
   S%checklist(STRUCT_WAVE) = .true.

   !Set variables that are otherwise read from main input
   call CFG_Set(cfg, "n", n)
   call CFG_Set(cfg, "t_up", S%n_t, tupvalue)
   call CFG_Set(cfg, "t_dn", S%n_t, tdnvalue)
   call CFG_Set(cfg, "U", S%nGroup, Hamilt%Uvalue)
   call CFG_Set(cfg, "mu_up", S%nGroup, Hamilt%muupvalue)
   call CFG_Set(cfg, "mu_dn", S%nGroup, Hamilt%mudnvalue)

   S%checklist(STRUCT_INIT) = .true.

   !There is some conflict between lattice and pairs when
   !deallocating. The first that deallocate is fine. The
   !second complain about deallocating somthing not allocated.
   call free_lattice(lattice)
   call free_reclatt(rlattice)
   call free_hamilt(Hamilt)
   call free_symm(SymmOp)
   call free_bonds(Bonds)
   call free_pairs(Pairs)

   deallocate(tupvalue)
   deallocate(tdnvalue)

end subroutine trap_struct_init

!------------------------------------------------------------!

subroutine trap_init_latt(lattice, trap)

    type(lattice_t), intent(out) :: lattice
    type(trap_t), intent(inout) :: trap
   
    integer :: natom, nx, i, j, k
  
    real(wp) :: d2, trimr2

    nx = trap%n

    !Compute trimming radius squared
    trimr2 = 0.5*nx*nx

    if (trap%trimmed) trimr2 = (real(nx) - 1. - trap%trap_center(1))**2

    !Count the sites within a trimming radius from the center
    natom = 0
    do i = 0, nx - 1
       do j = 0, nx - 1
          d2 = (dble(i)-trap%latt_center(1))**2 + (dble(j)-trap%latt_center(2))**2
          if( d2 < trimr2 ) natom = natom + 1
       enddo
    enddo

    trap%nsites = natom

    lattice%natom = natom
    lattice%nsites = natom
    lattice%ndim = 2
    lattice%ncell = 1
    
    lattice%sc = 0
    lattice%ac = 0.d0
    lattice%scc = 0.d0

    do i = 1, 2
       lattice%sc(i,i) = 1
       lattice%ac(i,i) = dble(nx)
       lattice%scc(i,i) = dble(nx)
    enddo

    i = 3
    lattice%sc(i,i) = 1
    lattice%ac(i,i) = 1.d0
    lattice%scc(i,i) = 1.d0

    allocate(lattice%olabel(0:natom-1), lattice%xat(rdim,0:natom-1))
    allocate(trap%map(0:nx-1,0:nx-1))
    allocate( trap%pos(2, 0:natom-1) )

    lattice%olabel = "s0"

    !populate the tables that, given a site position, returns its label
    !and its inverse, that given the label returns the position. In the
    !latter case, if a position is given that does not belong to the 
    !cluster, -1 is returned.
    k = 0
    trap%map = -1
    do j = 0, nx-1
       do i = 0, nx-1
          d2 = (dble(i)-trap%latt_center(1))**2 + (dble(j)-trap%latt_center(2))**2
          if( d2 < trimr2 ) then 
             trap%pos(1,k)= real(i)-trap%latt_center(1)
             trap%pos(2,k)= real(j)-trap%latt_center(2)
             lattice%xat(1,k) = dble(i) / nx
             lattice%xat(2,k) = dble(j) / nx
             lattice%xat(3,k) = 0.d0
             trap%map(i,j) = k
             k = k + 1
          endif
       enddo
    enddo

    nullify(lattice%phase)

    lattice%initialized = .true.

end subroutine trap_init_latt

!------------------------------------------------------------!

subroutine trap_init_ckb(lattice, trap)

   !Find the label of the 4 neighbors surrounding a site

   type(lattice_t), intent(in) :: lattice
   type(trap_t), intent(inout) :: trap

   integer  :: nx, i, j, k, h, nneig, ieo, jeo
   real*8 :: x(2)
   real*8, pointer :: pos(:,:)

   integer, dimension(2,2) :: stype 

   real*8, dimension(2,2) :: dx 

   !Portability issues with some version of gfortran
   !integer, parameter, dimension(2,2) :: stype = reshape((/ 1, 2, 3, 4 /), (/2, 2/))
   !real*8, parameter, dimension(2,2) :: dx = reshape((/ 1.0, 0.0, 0.0, 1.0 /), (/2, 2/))

   stype(1, 1) = 1; stype(1, 2) = 3
   stype(2, 1) = 2; stype(2, 2) = 4

   dx(1, 1) = 1.d0; dx(1, 2) = 0.d0
   dx(2, 1) = 0.d0; dx(2, 2) = 1.d0

   allocate(trap%neig(numneig,0:lattice%nsites-1))

   pos => lattice%cartpos
   nx = trap%n

   do j=0,lattice%nsites-1
      nneig=0
      !i= 1 => +dx, +dy  ;  i=-1 => -dx, -dy  
      do i=1,-1,-2
         !k=0 => dx ; k=1 => dy
         do k=0,1  
            nneig=nneig+1
            !Find the new position
            x(:) = pos(1:2,j) + i * dx(:,k+1) 
            h = map(nint(x(1)),nint(x(2)))
            if(h >= 0)then 
               ! Found
               trap%neig(nneig,j) = h
            else
              ! Finding the new position involves wrapping around the boundary
              x(1) = k * pos(1,j) + (1-k) * (nx-1-pos(1,j))
              x(2) = (1-k) * pos(2,j) + k * (nx-1-pos(2,j))
              trap%neig(nneig,j) = map(nint(x(1)),nint(x(2)))
            endif
         enddo
      enddo
    enddo

    !label sites as 1,2,3,4 depending on position (even/odd x and y)
    !  3 -- 4   
    !  |    |      
    !  1 -- 2
    allocate(trap%ckbtype(0:lattice%nsites-1))
    do j = 0, nx - 1
       jeo = mod(j, 2) + 1
       do i = 0, nx - 1
          ieo = mod(i, 2) + 1
          k = trap%map(i,j)
          if (k>=0) trap%ckbtype(k) = stype(ieo,jeo)
       enddo
    enddo

    contains
  
    integer function map(i1,i2)

       integer i1,i2

        if( i1>=0 .and. i1<trap%n .and. i2>=0 .and. i2<trap%n ) then
           map = trap%map(i1,i2)
        else
           map = -1
        endif

    end function map

end subroutine trap_init_ckb

!------------------------------------------------------------!

subroutine trap_init_phase(lattice, phase)

   type(lattice_t), intent(in) :: lattice
   real(wp), intent(inout)     :: phase(lattice%nsites)
   
   integer  :: i, j
   real*8, pointer :: pos(:,:)

   pos => lattice%cartpos

   do i = 0, lattice%nsites-1
     j = nint(pos(1,i)+pos(2,i))
     phase(i+1) = 2*abs(mod(j,2)) - 1
   enddo

end subroutine trap_init_phase

!------------------------------------------------------------!

subroutine trap_init_rlatt(Rlattice, lattice)

   type(lattice_t), intent(in) :: lattice
   type(recip_lattice_t), intent(out) :: Rlattice

   real*8, parameter :: pi = 3.141592653589793d0

   integer :: i

   Rlattice%ktwist=0.d0
   Rlattice%kpoint=0.d0

   Rlattice%kc=0.d0
   Rlattice%ks=0.d0

   do i=1,3
     Rlattice%kc(i,i)=2.d0*pi/lattice%ac(i,i)
     Rlattice%ks(i,i)=1.d0
   enddo

   Rlattice%kcs=Rlattice%kc

   Rlattice%ndim=lattice%ndim

   Rlattice%nkpts=lattice%ncell

   Rlattice%initialized=.true.

   nullify(Rlattice%FourierC)

end subroutine trap_init_rlatt

!------------------------------------------------------------!

subroutine trap_init_hamilt(hamilt, lattice, trap)
   
   type(lattice_t), intent(in) :: lattice
   type(trap_t), intent(in) :: trap
   type(Hamiltonian_t), intent(out) :: hamilt

   integer :: i, j, k, nsites
   real*8 :: dist, eta, db2, dh(2)
   real*8, pointer :: pos(:,:)
   real*8, parameter, dimension(2) :: dx=(/ 1.0, 0.0 /), dy=(/ 0.0, 1.0 /)
  

   nsites = lattice%nsites
   allocate( hamilt%hopup(0:nsites-1,0:nsites-1) )
   allocate( hamilt%hopdn(0:nsites-1,0:nsites-1) )
   allocate( hamilt%Uv(0:nsites-1,0:nsites-1) )
   allocate( hamilt%Jv(0:nsites-1,0:nsites-1) )
   allocate( hamilt%phase(0:nsites-1))

   hamilt%hopup = 0.d0
   hamilt%hopdn = 0.d0
   hamilt%Uv = 0.d0
   hamilt%mu_up = trap%mu
   hamilt%mu_dn = trap%mu

   do i = 0, nsites-1
      hamilt%Uv(i,i) = trap%U
   enddo

   pos => lattice%cartpos

   !get the exponent for the gaussian decay : exp(-eta*d^2)
   if(trap%ttrap > 1.d-12)then 
      eta = -4 * log(trap%ttrap/trap%t) / (trap%n)**2
   else
      eta = 0.d0
   endif

   do j = 0, lattice%nsites-1
      do i = 1, numneig
         k = trap%neig(i,j)
         dh(1:2) = pos(1:2,k) - pos(1:2,j)
         dist = sqrt(sum(dh**2))
         if(dist > 1.001 ) then
           dh = -dh / dist
           db2 = sum((pos(1:2,j) + 0.5d0*dh(1:2) - trap%latt_center)**2)
         else
           db2 = 0.25*sum((pos(1:2,j)+pos(1:2,k)-2*trap%latt_center)**2)
         endif
         hamilt%hopup(j,k) = trap%t * exp(-db2*eta)
         hamilt%hopdn(j,k) = trap%t * exp(-db2*eta)
      enddo
   enddo

   do j = 0, lattice%nsites-1
      hamilt%hopup(j,j) = trap%Vtrap * sum((pos(1:2,j)-trap%trap_center)**2)
      hamilt%hopdn(j,j) = trap%Vtrap * sum((pos(1:2,j)-trap%trap_center)**2)
   enddo

   hamilt%constructed = .true.

   call find_neighbors(hamilt)
   hamilt%neig_found = .true.

end subroutine trap_init_hamilt

!------------------------------------------------------------!

subroutine trap_init_symm(symmOp, trap)

    type(symm_operations), intent(out) :: symmOp
    type(trap_t), intent(in) :: trap
    integer :: nsymm

    if(trap%excentric) then 
       nsymm=0
    else
       nsymm=4
    end if

    ! Set up symmetry operations
    symmOp%nsymm = nsymm
    symmOp%ntotsymm = nsymm

    ! Allocate
    allocate(symmOp%symmangle(nsymm))
    allocate(symmOp%symmpoint(3,nsymm))
    allocate(symmOp%symmaxis(3,nsymm))
    allocate(symmOp%symmlabel(nsymm))
    nullify(symmOp%map_symm_k)

    if(nsymm>0) then
       ! Set labels
       symmOp%symmlabel = (/'D', 'D', 'C', 'I'/)

       ! Set point of symmetry
       SymmOp%symmpoint=0.d0
       SymmOp%symmpoint(1:2,1:nsymm)=trap%latt_center(1)

       ! Set symmetry axes
       symmOp%symmaxis = 0.d0
       symmOp%symmaxis(2,1) = 1.d0
       symmOp%symmaxis(1,2) = 1.d0
       symmOp%symmaxis(3,3) = 1.d0

       symmOp%symmangle = 0.d0
       symmOp%symmangle(3) = acos(0.d0)

       symmOp%addTimeRev = .false.
       symmOp%initialized = .false.
    endif

end subroutine trap_init_symm

!------------------------------------------------------------!

subroutine trap_init_pair(bonds, pairs, trap)

    type(trap_t), intent(inout) :: trap
    type(bonds_t), intent(out) :: bonds
    type(pairing), intent(out) :: pairs

    integer :: nbond, ibond, ilab, j, i, k, h, bo, bt, bno, bni

    nbond=numbond*trap%nsites
 
    Bonds%ntotbond=nbond
    allocate(Bonds%bond_label(nbond))
    allocate(Bonds%bond_origin(nbond))
    allocate(Bonds%bond_target(nbond))
    allocate(Bonds%xxbond(3,nbond))

    allocate(trap%b_out(numbond,0:trap%nsites-1), trap%b_in(numbond,0:trap%nsites-1))
  
    Bonds%xxbond = 0.d0

    ibond=0
    ilab=0
    do j = 0, trap%nsites-1
       do i = 1, 2
          k = trap%neig(i,j)
          ilab = ilab + 1
          do h = 0, 1
             ibond = ibond + 1
             bo = (1-h)*j + h*k       !bond origin
             bt = (1-h)*k + h*j       !bond target
             bno = i + 2*h            !bond number out
             bni = i - 2*(h-1)        !bond number in
             Bonds%bond_label(ibond) = ilab*(-1)**h
             Bonds%bond_origin(ibond) = bo
             Bonds%bond_target(ibond) = bt
             trap%b_out(bno,bo) = ibond
             trap%b_in(bni,bt) = ibond
             Bonds%xxbond(1:2,ibond) = (-1)**h*(trap%pos(:,k)-trap%pos(:,j))
          enddo
       enddo


       !the fifth bond
       ibond = ibond+1

       bo  = j
       bt  = j
       bno = 5
       bni = 5

       Bonds%bond_label (ibond) = 2*trap%nsites+ilab
       Bonds%bond_origin(ibond) = bo
       Bonds%bond_target(ibond) = bt
       trap%b_out(bno,bo)       = ibond
       trap%b_in (bni,bt)       = ibond
       Bonds%xxbond(1:2,ibond)  = 0
    enddo

    Bonds%initialized = .true.
    Found_Field(BONDS_F) = .true.
    
    allocate(Pairs%nbondv(0:0))
    allocate(Pairs%bond_origin(nbond,0:0))
    allocate(Pairs%bond_end(nbond,0:0))
    allocate(Pairs%bond_map(nbond))
    allocate(Pairs%pair_map(nbond))
    allocate(Pairs%bond_number(nbond,0:0))
    nullify(Pairs%wave_label)
    nullify(Pairs%bond_wgt)


    Pairs%nbond = nbond
    Pairs%nbondv(0) = nbond
    Pairs%bond_origin(:,0) = Bonds%bond_origin(:)
    Pairs%bond_end(:,0) = Bonds%bond_target(:)
    do i =1, nbond
      Pairs%bond_map(i) = i
      Pairs%pair_map(i) = i
      Pairs%bond_number(i,0) = i
    enddo
    Pairs%nwave = 2

end subroutine trap_init_pair

!------------------------------------------------------------!

subroutine trap_init_singlet(S, Pairs, trap)

   type(struct), intent(in) :: S
   type(pairing), intent(in) :: Pairs
   type(trap_t), intent(inout) :: trap

   integer :: i, ij, ib, jb, ibr, jbr, nsinglcl, ic, ibo, ibt
   integer :: im(4), singlcl(4,S%nClass_b), bondmat(0:S%nsite-1, 0:S%nsite-1)
   integer, pointer :: bo(:), bt(:)

   real*8  :: fact, wgt(S%nclass_b), xb(2,2)

   logical :: visited(S%nclass_b)

   !Initialize
   visited = .false.
   nsinglcl = 0
   bo => Pairs%bond_origin(:,0) 
   bt => Pairs%bond_end(:,0) 

   !Fill bond matrix : given two sites returns bond index
   call dqmc_CCS_fill(S%nsite, bondmat, S%B)

   do ij = 1, S%nclass_b

      if(visited(ij))cycle
 
      ib = trap%class_head_bond(1,ij)
      jb = trap%class_head_bond(2,ij)

      !First Check
      if(bondmat(bo(ib), bt(ib))/=ib .or. bondmat(bo(jb),bt(jb))/=jb) &
        & stop 'Problem with bond origin or target'

      !Reverse bonds
      ibr = bondmat(bt(ib), bo(ib))
      jbr = bondmat(bt(jb), bo(jb))

      !The four classes making up a singlet
      im(1)  = S%class_b(ib,jb)
      im(2)  = S%class_b(ibr,jb)
      im(3)  = S%class_b(ib,jbr)
      im(4)  = S%class_b(ibr,jbr)

      nsinglcl = nsinglcl + 1
      singlcl(1:4,nsinglcl) = im(1:4)

      do i = 1, 4
         visited(im(i))   = .true.
      enddo

      !Check whether the bonds involve creation on the same site
      fact = 0.5d0
      if(bo(ib)==bt(ib)) fact = fact * sqrt(0.5d0)
      if(bo(jb)==bt(jb)) fact = fact * sqrt(0.5d0)

      wgt(nsinglcl) = fact

   enddo
 
   allocate(trap%singlcl(4,nsinglcl))
   allocate(trap%wgtcl(nsinglcl))
   allocate(trap%labcl(nsinglcl))

   trap%nsinglcl = nsinglcl
   trap%singlcl(1:4,1:nsinglcl) = singlcl(1:4,1:nsinglcl)
   trap%wgtcl(1:nsinglcl) = wgt(1:nsinglcl)

   !Assign label to the singlet classes
   do i = 1, nsinglcl
     ib = trap%class_head_bond(1,singlcl(1,i))
     jb = trap%class_head_bond(2,singlcl(1,i))
     im(1) = ib
     im(2) = jb
     do ij = 1, 2 
        !Assign coordinate of bond to average amongst site position
        ibo = bo(im(ij))
        ibt = bt(im(ij))
        xb(1:2,ij) = 0.5*(trap%pos(1:2,ibo) + trap%pos(1:2,ibt))
        !Check whether the bond winds around
        if( sum( (xb(1:2,ij)-trap%pos(1:2,ibo))**2 ) > 0.251 )then
           ic = 1
           !Bond winds : check whether x-coordinate of bo and bt line up
           if(abs( trap%pos(ic,ibo)-trap%pos(ic,ibt) ) < 0.01) ic = 2
           xb(ic,ij) = max( trap%pos(ic,ibo),trap%pos(ic,ibt) ) + 0.5d0
        endif
     enddo
       
     write(trap%labcl(i),'(2(2f7.2))') xb(1:2,1), xb(1:2,2)-xb(1:2,1)
   enddo

end subroutine trap_init_singlet

end module trap_module
