  program run_trap

    use trap_module
    use dqmc_cfg
    use dqmc_hubbard
    use dqmc_gtau
    use dqmc_mpi
 
    implicit none
  
    type(trap_t)    :: trap
    type(config)    :: cfg
    type(hubbard)   :: Hub
    type(gtau)      :: tdgf

    integer :: i, j, k, nBin, nIter, ntausk, msxx, fullg, exactb, slice
    integer, parameter :: IPT=40, OPT=50, OPT2=60, OPT3=70
    integer :: t1, t2, t3, t4, rate

    call system_clock(t1, rate)

    !Count the number of processors
    call DQMC_MPI_Init(qmc_sim, PLEVEL_1)
  
    !Read input
    call trap_read_input(cfg, trap)

    !Initialize trap, S and Hub
    call trap_struct_init(trap, Hub%S, cfg)
    call DQMC_Hub_Config(Hub, cfg)

    !Reset Phy2 to store spatial dependent pairing
    call Phy2_realloc(Hub%P2, Hub%S, trap%nsinglcl)

    !Locally store number of sweeps between measurements
    call CFG_Get(cfg, "tausk", ntausk)
    call CFG_Get(cfg, "msxx", msxx)
    call CFG_Get(cfg, "fullg", fullG)
    call CFG_Get(cfg, "exactb", exactb)

    !Set the number of sweeps to skip before performing a new measurement
    call Gtau_init(Hub, tdgf, trap)
  
    call system_clock(t3)
#ifdef DQMC_PROFILE
    gfun_profile = .true.
    !matb_profile = .true.
    !call profile_enable()
#endif

    ! Warmup sweep
    do i = 1, Hub%nWarm
       ! The second parameter means no measurement should be made
       write(*,'(A,i6,1x,i3)')' Warmup Sweep, nwrap  : ', i, Hub%G_up%nwrap
       call DQMC_Hub_Sweep(Hub, NO_MEAS0)
       call DQMC_Hub_Sweep2(Hub, Hub%nTry)
    end do

    call system_clock(t4)

    ! We divide all the measurement into nBin,
    ! each having nPass/nBin pass.
    nBin   = Hub%P0%nBin
    nIter  = Hub%nPass / nBin / ntausk
    do i = 1, nBin
       do j = 1, nIter
          do k = 1, ntausk
             write(*,'(A,i6,1x,i3)')' Simulation Sweep, nwrap : ', & 
              &  k+(j-1)*ntausk+(i-1)*nIter*ntausk, Hub%G_up%nwrap
             call DQMC_Hub_Sweep(Hub, Hub%nMeas)
             call DQMC_Hub_Sweep2(Hub, Hub%nTry)
          enddo
          slice = 0
          call DQMC_Hub_Meas(Hub, msxx, fullg, exactb, slice)
          call Gtau_Measure(Hub, tdgf, trap, i)
       end do
       ! Accumulate results for each bin
       call DQMC_Phy0_Avg(Hub%P0)
       ! Call local routine for P2 average
       call Phy2_Local_Avg(Hub%P2, Hub%S, trap)
       ! Call local routine for G(tau) average
       call Gtau_Local_Avg(trap, i)
    end do
  
    ! Get average result
    call DQMC_Phy0_GetErr(Hub%P0)
    call DQMC_Phy2_GetErr(Hub%P2)
    call Gtau_GetErr(trap, qmc_sim%size)
  
    call Pair_No_Vertex(Hub, trap)

    call system_clock(t2)

    !Open output files
    open(unit=OPT,file=trap%ofile)
    open(unit=OPT2,file="fn_"//adjustl(trap%ofile))
    open(unit=OPT3, file="geom_"//adjustl(trap%ofile))

    write(OPT3, "('nx = ', i3)"), trap%n
    do i=1, Hub%S%nClass
       write(OPT3, "('  ', i3, a30)"), Hub%S%F(i), Hub%S%clabel(i)
    end do

    ! Print computed results using local routines
    call OutputParam(Hub, trap, OPT)
    call Phy0_Print(Hub%P0, Hub%S, trap%pos, OPT, OPT2)
    call Phy2_Print_local(Hub%P2, Hub%S, trap, OPT2)
    call Gtau_Print(trap, Hub)
  
    ! Clean up the used storage
    call DQMC_Hub_Free(Hub)
    call DQMC_Config_Free(cfg)               

#ifdef DQMC_PROFILE
    call gfun_print()
    !call matb_print()
    call profile_print()
#endif

    t4 = t4 - t3                ! warmup
    t3 = t3 - t1                ! init
    t2 = t2 - t1                ! total

    write(OPT,*) "Total time:",  t2 / REAL(rate), "(second)"

    write(OPT,*) "Init time:",  t3 / REAL(rate), "(second)"

    write(OPT,*) "Warmup time:",  t4 / REAL(rate), "(second)"

  end program

!------------------------------------------------------------!

  subroutine Phy2_Local_Avg(P2, S, trap)
    !
    ! Purpose
    ! =======
    !    This subroutine averges the pair measurements.
    !    which is stored in P2%M2.
    !
    ! Arguments
    ! =========
    !
    use dqmc_phy2
    use dqmc_struct
    use trap_module
    implicit none

    type(Phy2), intent(inout) :: P2                 ! phy2
    type(Struct), intent(in)  :: S
    type(trap_t), target, intent(in)  :: trap

    ! ... local scalar ...
    integer :: ic, i, j, x, y, iw
    integer, pointer :: cb(:,:) 
    integer, pointer :: outx(:), outy(:), inx(:), iny(:)
    real(wp) :: factor, singlet
    real(wp), pointer :: T(:)

    ! ... Executable ...

    if (P2%compute) then

       factor = 1.d0/P2%cnt

       cb => S%class_b
       T => P2%T(:,1)

       !symmetrize and temporarily store in T
       T = 0.d0
       do i = 1, P2%nb
         do j = 1, P2%nb
           x = cb(i,j)
           T(x) = T(x) + P2%M2(i,j)
         enddo
       enddo

       T = factor * T / S%size_b

       !For each inequivalent distance compute pairing correlation
       do ic = 1, trap%nsinglcl

          do i = 1, 4
             P2%M3(ic,P2%idx) = P2%M3(ic,P2%idx) + T(trap%singlcl(i,ic))
          enddo
          P2%M3(ic,P2%idx) = P2%M3(ic,P2%idx) * trap%wgtcl(ic)
  
       enddo
       
       !For each inequivalent distance compute pairing correlation
       do ic = 1, S%nClass

          !(x,y) = Representative pair of class "ic"
          x = trap%class_head(1,ic) - 1
          y = trap%class_head(2,ic) - 1

          !pointers to bond ID
          outx => trap%b_out(:,x)
          inx  => trap%b_in(:,x)
          outy => trap%b_out(:,y)
          iny  => trap%b_in(:,y)

          do i = 1, numbond
             do j = 1, numbond

                !Amplitude of a singlet from bond (x,i) to bond (y,j)
                singlet = T( cb( outx(i), outy(j) ) ) + T( cb ( outx(i), iny(j) ) )  &
                        + T( cb(  inx(i), outy(j) ) ) + T( cb (  inx(i), iny(j) ) )

                !Sum up the amplitudes to construct symmetrized pairing function
                do iw = 0, P2%nwave-1
                   P2%M3(trap%nsinglcl+iw*S%nclass+ic, P2%idx) =       &
                      P2%M3(trap%nsinglcl+iw*S%nclass+ic, P2%idx)      &
                      + S%W(i, iw+1) * S%W(j, iw+1) * singlet * 0.5d0
                enddo
             enddo
          enddo
       enddo
       
       P2%sgn(P2%idx) = P2%sgn(P2%idx) * factor
    
       ! Reset counter and change bins
       P2%idx = P2%idx + 1
       P2%cnt = 0
       P2%M2  = ZERO
    end if

  end subroutine Phy2_Local_Avg

  !--------------------------------------------------------------------!
  
  subroutine Phy2_realloc(P2, S, nclass)

    use dqmc_struct
    use dqmc_phy2
    implicit none

    type(Phy2), intent(inout) :: P2          ! Phy2 to be initialized
    type(Struct), intent(in)  :: S
    integer, intent(in)       :: nclass

    ! ... Local Vars ...
    integer :: nb, nWave, nBin

    ! ... Executable ...

     nWave = S%nWave
     nb    = S%n_b
     nBin  = P2%nBin

     deallocate(P2%M1, P2%M2, P2%M3, P2%T)
       
     ! Allocate storages
     allocate(P2%M1(nb,nb))
     allocate(P2%M2(nb,nb))
     allocate(P2%M3(nWave*S%nclass+nclass,nBin+2))
     allocate(P2%T(S%nClass_b,1))
     
     ! Initialize 
     P2%M2  = ZERO
     P2%M3  = ZERO
     P2%sgn = ZERO

  end subroutine Phy2_realloc

  !--------------------------------------------------------------------!

  subroutine Pair_No_Vertex(Hub, trap)
    !
    use dqmc_hubbard
    use trap_module
    implicit none
    type(Hubbard), intent(inout) :: Hub    ! Hubbard model
    type(trap_t), intent(in) :: trap

    ! ... local scalar ...
    integer  :: i, j, k, avg, n
    real*8 :: G_up(Hub%n,Hub%n)
    real*8 :: G_dn(Hub%n,Hub%n)

    ! ... Executable ...

   avg=Hub%P0%avg
   n = Hub%n

    do i= 1, n
       do j= 1, n
          k=Hub%S%D(i,j)
          G_up(i,j)=Hub%P0%G_fun(k,avg)
          G_dn(i,j)=Hub%P0%G_fun(k,avg)
       enddo
    enddo

    call DQMC_Phy2_Pair(n, Hub%P2%M2, G_up, G_dn, Hub%S%B)

    !Reuse first bin to store unconnected part
    Hub%P2%idx=1
    Hub%P2%cnt=1
    Hub%P2%M3(:,1)=0.d0

    call Phy2_Local_Avg(Hub%P2, Hub%S, trap)

  end subroutine Pair_No_Vertex

  !------------------------------------------------------------!

  subroutine Phy2_Print_Local(P2, S, trap, OPT)

    use dqmc_mpi
    use dqmc_struct
    use dqmc_phy2
    use trap_module

    implicit none
    type(Phy2), intent(in)    :: P2   ! Phy2
    type(trap_t), intent(in)  :: trap
    type(struct), intent(in)  :: S
    integer, intent(in)       :: OPT  ! Output file handle

    ! ... Local var
    integer  :: i, avg, err, nclass, iw, init, final
    real*8   :: Pkzero
    real*8, pointer   :: localM3(:,:)
    character(len=30) :: label

    ! ... Executable ...

    if (P2%compute .and. qmc_sim%rank.eq.0) then

       avg = P2%avg
       err = P2%err

       nclass = trap%nsinglcl

       allocate(localM3(nclass,1))
       !Print full pairing function
       label = "Pairing"
       call DQMC_Print_RealArray(0, nclass, label, &
           trap%labcl, P2%M3(1:nclass, avg:avg), P2%M3(1:nclass, err:err),OPT)

       !Print connected
       label = "Pairing (Connected)"
       localM3(:,1) = P2%M3(1:nclass,avg) - P2%M3(1:nclass,1)
       !Print connected
       call DQMC_Print_RealArray(0, nclass, label, &
           trap%labcl, localM3(1:nclass,1:1), P2%M3(1:nclass, err:err),OPT)
       deallocate(localM3)

       !Resize
       nclass = S%nclass
       allocate(localM3(nclass,1))

       do iw = 1, P2%nWave

          init  = trap%nsinglcl + (iw-1)*nclass + 1
          final = trap%nsinglcl +  iw*nclass
          !Print full pairing function
          call DQMC_Print_RealArray(0, nClass, S%wlabel(iw), &
              S%clabel, P2%M3(init:final, avg:avg), P2%M3(init:final, err:err),OPT)

          !Define local label and matrix for connected pairing function
          label = trim(S%wlabel(iw))//"(Connected)"
          localM3(:,1) = P2%M3(init:final,avg) - P2%M3(init:final,1)
          !Print connected
          call DQMC_Print_RealArray(0, nClass, label, &
              S%clabel, localM3(1:nclass,1:1), P2%M3(init:final, err:err),OPT)

          !Compute k=0 correlation function
          Pkzero = 0.d0
          do i = 1, nclass
            Pkzero = Pkzero + P2%M3(init+i-1,avg)*S%F(i)
          enddo
          write(OPT,*)'Sum : ', Pkzero/2.d0

       enddo
       deallocate(localM3)

    end if

  end subroutine Phy2_Print_Local

  !------------------------------------------------------------!

  subroutine Phy0_Print(P0, S, pos, OPT, OPT2)
    !
    ! Derived from DQMC_Phy0_Print in dqmc_phy0.F90
    !
    use dqmc_mpi
    use dqmc_phy0
    use dqmc_struct
    implicit none

    type(Phy0), intent(in)    :: P0   ! Phy0
    type(Struct), intent(in)  :: S    ! Underline lattice structure
    real*8, intent(in)        :: pos(2,*)
    integer, intent(in)       :: OPT, OPT2  ! Output file handle

    ! ... Local scalar ...
    integer :: nClass, nSite, avg, err, i, j, idx

    real*8 :: tmp(S%nGroup,2) 

    logical :: visited(S%nGroup)

    character(30) :: loclabel(S%nGroup)

    ! ... Executable ...
    if (qmc_sim%rank .ne. 0) return

    nClass = P0%nClass
    nSite  = S%nSite
    avg    = P0%avg
    err    = P0%err

    ! Scalar terms
    call DQMC_Print_RealArray(0, 3, "Sign of equal time measurements:", &
         P0_SIGN_STR, P0%sign(:,avg:avg), P0%sign(:,err:err), OPT)
    
    call DQMC_Print_RealArray(0, P0%nmeas, "Equal time measurements:", &
         P0_STR, P0%meas(:,avg:avg), P0%meas(:,err:err), OPT)

    !Define label for local classes
    visited = .false.
    do i = 1, S%nsite
       j = S%map(i)
       if(visited(j))cycle
       write(loclabel(j),'(3f9.3)') pos(1:2,i),sqrt(sum(pos(1:2,i)**2))
       visited(j)=.true.
    enddo

    !Fill tmp with local density
    do i = 1, S%nsite
       j = S%map(i)
       idx = S%D(i, i)
       tmp(j,1) = P0%Den0(idx,avg)
       tmp(j,2) = P0%Den0(idx,err)
    enddo

    call DQMC_Print_RealArray(0, S%nGroup, "Local up-spin density:", &
         loclabel, tmp(:,1:1), tmp(:,2:2), OPT)
    
    call DQMC_Print_RealArray(0, S%nGroup, "Local down-spin density:", &
         loclabel, tmp(:,1:1), tmp(:,2:2), OPT)
    
    tmp=2.d0*tmp

    call DQMC_Print_RealArray(0, S%nGroup, "Local density:", &
         loclabel, tmp(:,1:1), tmp(:,2:2), OPT)
    
    !Fill tmp with double occupancy
    do i = 1, S%nsite
       j = S%map(i)
       idx = S%D(i, i)
       tmp(j,1) = P0%Den1(idx,avg)
       tmp(j,2) = P0%Den1(idx,err)
    enddo

    call DQMC_Print_RealArray(0, S%nGroup, "Double occupancy:", &
         loclabel, tmp(:,1:1), tmp(:,2:2), OPT)
    
    !Fill tmp with local number fluctuation
    do i = 1, S%nsite
       j = S%map(i)
       idx = S%D(i, i)
       tmp(j,1) = 2.d0 * ( P0%Den0(idx,avg) + P0%Den1(idx,avg) - 2*P0%Den0(idx,avg)**2 )
       tmp(j,2) = 2.d0 * P0%Den1(idx, err)
    enddo

    call DQMC_Print_RealArray(0, S%nGroup, "Number fluctuation:", &
         loclabel, tmp(:,1:1), tmp(:,2:2), OPT)

    ! Function terms
    call DQMC_Print_RealArray(0, nClass, "Equal time Green's function:", &
         S%clabel, P0%G_fun(:, avg:avg), P0%G_fun(:, err:err), OPT2)
    
    call DQMC_Print_RealArray(0, nClass, &
         "Density-density correlation fn: (up-up)", &
         S%clabel, P0%Den0(:, avg:avg), P0%Den0(:, err:err), OPT2)
    
    call DQMC_Print_RealArray(0, nClass, &
         "Density-density correlation fn: (up-dn)", &
         S%clabel, P0%Den1(:, avg:avg), P0%Den1(:, err:err), OPT2)
    
    call DQMC_Print_RealArray(0, nClass, "XX Spin correlation function:", &
         S%clabel, P0%SpinXX(:, avg:avg), P0%SpinXX(:, err:err), OPT2)
    
    call DQMC_Print_RealArray(0, nClass, "ZZ Spin correlation function:", &
         S%clabel, P0%SpinZZ(:, avg:avg), P0%SpinZZ(:, err:err), OPT2)

    call DQMC_Print_RealArray(0, nClass, "Average Spin correlation function:", &
         S%clabel, P0%AveSpin(:, avg:avg), P0%AveSpin(:, err:err), OPT2)

    
  end subroutine Phy0_Print

  !---------------------------------------------------------------------!

  subroutine OutputParam(Hub, trap, OPT)

    use dqmc_mpi
    use dqmc_util
    use dqmc_hubbard
    use trap_module
    !
    ! Derived from DQMC_Hub_OutputParam in dqmc_hubbard.F90
    !
    implicit none
    type(Hubbard), intent(in) :: Hub     ! Hubbard model
    type(trap_t),  intent(in) :: trap     ! Hubbard model
    integer, intent(in)       :: OPT     ! output handle

    ! ... Local ...
    logical       :: lex
    logical, parameter :: restore = .true.
    integer, parameter :: slice = 0

    ! ... Executable ....
    if (qmc_sim%rank .ne. 0) return

    write(OPT,*)  Hub%S%Name(:)
    write(OPT,FMT_STRDBL)  "                          t : ", trap%t
    write(OPT,FMT_STRDBL)  "                          U : ", trap%U
    write(OPT,FMT_STRDBL)  "                         mu : ", trap%mu
    write(OPT,FMT_STRDBL)  "                      Vtrap : ", trap%Vtrap
    write(OPT,FMT_STRDBL)  "                      ttrap : ", trap%ttrap
    write(OPT,FMT_STR2BL)  "                Trap center : ", trap%trap_center(:)-trap%latt_center(:)
    write(OPT,FMT_STRINT)  "             Time slice (L) : ", Hub%L
    write(OPT,FMT_STRDBL)  "                       dtau : ", Hub%dtau
    write(OPT,FMT_STRDBL)  "              beta (dtau*L) : ", Hub%dtau*Hub%L
    write(OPT,FMT_STRINT)  "     Number of warmup sweep : ", Hub%nWarm
    write(OPT,FMT_STRINT)  "Number of measurement sweep : ", Hub%nPass
    write(OPT,FMT_STRINT)  "   Frequency of measurement : ", Hub%nMeas
    write(OPT,FMT_STRINT)  "                Random seed : ", Hub%idum
    write(OPT,FMT_STRINT)  " Frequency of recomputing G : ", Hub%G_up%nWrap
    write(OPT,FMT_STRINT)  "Global move number of sites : ", Hub%nTry
    write(OPT,FMT_STRINT)  "               Accept count : ", Hub%naccept
    write(OPT,FMT_STRINT)  "               Reject count : ", Hub%nreject
    write(OPT,FMT_STRDBL)  "    Approximate accept rate : ", &
         dble(Hub%naccept)/dble(Hub%naccept+Hub%nreject)
    write(OPT,FMT_STRDBL)  "                      gamma : ", Hub%gamma
    write(OPT,FMT_STRINT)  "   Global move accept count : ", Hub%nAcceptGlobal
    write(OPT,FMT_STRINT)  "   Global move reject count : ", Hub%nRejectGlobal
    write(OPT,FMT_STRDBL)  "   Global move accept rate : ", &
         dble(Hub%nAcceptGlobal)/dble(Hub%nAcceptGlobal+Hub%nRejectGlobal)
    write(OPT,*)           "          Type of matrix B : ", Hub%B_up%name
    if (Hub%HSFtype .eq. HSF_DISC) then
       write(OPT,*)        "        Type of matrix HSF : 0/1"
    else
       write(OPT,*)        "        Type of matrix HSF : continuous"
       write(OPT,*)        "                   delta 1 : ", Hub%delta1
       write(OPT,*)        "                   delta 2 : ", Hub%delta2
    end if
    write(OPT, FMT_DBLINE)

    ! Check if the file is valid.
    if (Hub%outputHSF) then
       inquire(UNIT=HSF_OUTPUT_UNIT, EXIST=lex)
       if (lex) then
          call DQMC_Hub_Output_HSF(Hub, restore, slice, HSF_OUTPUT_UNIT)
       else
          call DQMC_Warning("HSF output file does not initialized.", 1)
       end if
    end if

  end subroutine OutputParam
  
  !---------------------------------------------------------------------!

  subroutine Gtau_init(Hub, tdgf, trap)
 
    use dqmc_hubbard
    use dqmc_cfg
    use dqmc_gtau
    use trap_module
    implicit none
 
    type(trap_t), intent(inout) :: trap
    type(Gtau), intent(out)     :: tdgf
    type(hubbard), intent(in)   :: Hub

    integer :: i, j, nbin

    if (trap%gtau) then

       !Initialize time dependent Green's function
       call DQMC_Gtau_Init2(Hub%n, Hub%L, TAU_BOTH, &
            Hub%SB_up%nOrth, Hub%G_up%nWrap, tdgf, Hub%B_up, Hub%B_dn, Hub%WS)

       !Auxiliary quantities for time-dependent G
       allocate(trap%gfactor(Hub%S%nGroup))
       trap%gfactor = 0.d0
       do i = 1, Hub%n
          j = Hub%S%map(i)
          trap%gfactor(j) = trap%gfactor(j) + 1.d0
       enddo
       trap%gfactor = 0.5 / trap%gfactor

       !Container for binned Green's function
       nbin = Hub%P0%nbin
       allocate(trap%Gavg(Hub%S%ngroup*Hub%L+1,nBin+2))
       trap%Gavg = 0.d0

    endif

  end subroutine Gtau_init

  !---------------------------------------------------------------------!

  subroutine Gtau_Measure(Hub, tdgf, trap, iBin)
     ! Purpose
     ! =======
     !    compute G(i, tau) for time dependent G_up and down
     !
     use dqmc_hubbard
     use dqmc_gtau
     use trap_module
     implicit none

     type(hubbard), intent(inout) :: Hub
     type(gtau), intent(inout)    :: tdgf
     type(trap_t), intent(inout)  :: trap
     integer, intent(in)          :: iBin

     ! ... Local variables ...
     integer  :: i, j, L, ii, jj, kk, hh, ll, avg, err
     real(wp) :: sgn, wgtt0, wgt0t
     real(wp), pointer :: gfct(:), Gavg(:,:)

     ! ... Executable ...
     ! compute the first row of G tau

     if(trap%gtau)then

        L     =  Hub%L 
        sgn   =  Hub%G_up%sgn * Hub%G_dn%sgn
        Gavg  => trap%Gavg
        gfct  => trap%gfactor
        err   =  size(Gavg, 2)
        avg   =  err - 1

        ii    = Hub%G_up%ilb
        jj    = ii
  
        !Initialize Gtau at equal time
        tdgf%upt0 = Hub%G_up%G
        tdgf%up0t = Hub%G_up%G
        tdgf%dnt0 = Hub%G_dn%G
        tdgf%dn0t = Hub%G_dn%G
        tdgf%ii   = ii
        tdgf%ib   = jj
        tdgf%sfc  = max (tdgf%nwrap - (Hub%G_up%nWrap - Hub%G_up%wps), 1)
        Gavg(:, avg) = 0.d0

        !Equal time measurement
        do j = 1, Hub%n
           kk = Hub%S%map(j)
           Gavg(kk, avg)  = Gavg(kk, avg)  + tdgf%upt0(j,j) + tdgf%dnt0(j,j) 
        end do

        
        ! For each time slice
        do i = 2, L
           ! Compute new Gtau
           ii = jj + i - 1
           if ( ii >  L ) ii = ii - L  
           wgtt0 = sign ( 0.5d0, dble(ii - jj) )
           wgt0t = -wgtt0
           call DQMC_MakeGtau2(tdgf, Hub%G_up, Hub%G_dn, ii, jj)

           hh = (i-1) * Hub%S%nGroup
           ll = (L-i+1) * Hub%S%nGroup

           do j = 1, Hub%n
              kk = hh + Hub%S%map(j)
              Gavg(kk, avg)  = Gavg(kk, avg)  + &
                   wgtt0 * ( tdgf%upt0(j,j) + tdgf%dnt0(j,j) )
              kk = ll + Hub%S%map(j)
              Gavg(kk, avg)  = Gavg(kk, avg)  + &
                   wgt0t * ( tdgf%up0t(j,j) + tdgf%dn0t(j,j) )
           end do
        end do

        ! average
        kk = 0
        Gavg(:, iBin) = 0.d0
        do j = 1, L
           do i = 1, Hub%S%nGroup
              kk = kk + 1
              Gavg(kk, iBin) = Gavg(kk, iBin) + Gavg(kk, avg) * sgn * gfct(i)
           enddo
        enddo
        Gavg(kk+1, iBin) = Gavg(kk+1, iBin) + sgn

        ! Store number of measurement in last element of Gavg
        Gavg(kk+1, err) = Gavg(kk+1, err) + 1.d0

     endif
  
  end subroutine Gtau_Measure

  ! ----------------------------------------------------------------- !

  subroutine Gtau_Local_Avg(trap, iBin)

     use trap_module
     implicit none

     integer, intent(in) :: iBin
     type(trap_t), intent(inout) :: trap
     integer :: isgn, err
     real(wp) :: factor
  
     if (trap%gtau) then

        isgn = size(trap%Gavg, 1)
        err = size(trap%Gavg, 2)
        factor = 1.d0 / trap%Gavg(isgn, err)
        trap%Gavg(:, iBin) = trap%Gavg(:, iBin) * factor
        trap%Gavg(isgn, err) = 0.d0

     endif

  end subroutine Gtau_Local_Avg

  ! ----------------------------------------------------------------- !

  subroutine Gtau_GetErr(trap, nproc)
    ! Purpose
    ! =======
    !    Compute std err and average
    !
    use dqmc_mpi
    use trap_module
    implicit none

    type(trap_t) :: trap
    integer, intent(in) :: nproc

    ! ... Local variables ...
    integer  :: i, sgn_idx, avg, err, nbin
    real(wp) :: sum_sgn
    real(wp), pointer :: z(:), y(:), Gavg(:,:)

    if(trap%gtau)then

       ! ... Executable ...
       Gavg    => trap%Gavg

       err     = size(Gavg,2)
       avg     = err - 1
       nbin    = avg - 1
       sgn_idx = size(Gavg,1)

       if(nproc == 1)then
 
          allocate(z(nbin), y(nbin))
          
          call DQMC_JackKnife(nBin, Gavg(sgn_idx, avg), Gavg(sgn_idx, err), &
               Gavg(sgn_idx, 1:nBin), y, z, sum_sgn)

          do i = 1, sgn_idx - 1
             call DQMC_SignJackKnife(nBin, Gavg(i, avg), Gavg(i, err), &
                  Gavg(i, 1:nBin), y, z, sum_sgn)
          end do

          !Store Jackknife in bins
          do i = 1, nbin
             Gavg(sgn_idx,i) = (nbin*Gavg(sgn_idx,avg) - Gavg(sgn_idx,i)) / dble(nbin-1)
             Gavg(1:sgn_idx-1,i) = (sum_sgn*Gavg(1:sgn_idx-1,avg) - Gavg(1:sgn_idx-1,i))/dble(nbin-1)
             Gavg(1:sgn_idx-1,i) = Gavg(1:sgn_idx-1,i) / Gavg(sgn_idx,i)
          enddo

          deallocate(z, y)

       else

#         ifdef QMC_MPI

             i = sgn_idx - 1

             !Average properties
             call mpi_allreduce(Gavg(:,1), Gavg(:,avg), sgn_idx, mpi_double, &
                mpi_sum, mpi_comm_world, mpi_err)

             !Compute average over n-1 processors
             Gavg(:,1) = (Gavg(:,avg) - Gavg(:,1)) / dble(nproc - 1)

             !Store average amongst all processors
             Gavg(1:i,avg) = Gavg(1:i,avg) / Gavg(sgn_idx,avg) 

             !Store jackknife in the processor bin
             Gavg(1:i,1) = Gavg(1:i,1) / Gavg(sgn_idx,1) 

             !Compute error
             call mpi_allreduce(Gavg(1:i,1)**2, Gavg(1:i,err), sgn_idx, mpi_double, &
                mpi_sum, mpi_comm_world, mpi_err)
             Gavg(:,err) = Gavg(:,err) / dble(nproc) - Gavg(:,avg)**2 
             Gavg(:,err) = sqrt(Gavg(:,err) * dble(nproc-1))

#         endif

      endif


    endif

  end subroutine Gtau_GetErr

  ! ----------------------------------------------------------------- !

  subroutine Gtau_Print(trap, Hub)
    use dqmc_mpi
    use trap_module
    use dqmc_hubbard
    ! Purpose
    ! =======
    !    Print out values
    !
    ! Argument
    ! ========
    implicit none

    type(trap_t), intent(in) :: trap    
    type(hubbard), intent(in) :: Hub

    ! ... Local variables ...
    integer :: i, j, k, idx, nbin, avg, err
    character(*), parameter :: OPTFMT = "(f7.4,1x,f20.16,'  ', f20.16)"
    character(len=9) xc, yc, cn
    character(len=300), pointer :: label(:)
    logical, pointer :: visited(:)
    integer, parameter :: OPT3 = 70

    if(trap%gtau .and. qmc_sim%rank==0)then

       err     = size(trap%Gavg,2)
       avg     = err - 1
       nbin    = avg - 1

       ! ... Executable ...
       allocate(label(Hub%S%nGroup))
       allocate(visited(Hub%S%nGroup))

       visited = .false.
       do i = 1, Hub%n
          !Define string with x- and y- coordinates
          write(xc,'(f9.1)') trap%pos(1,i-1)
          xc = adjustl(xc)
          write(yc,'(f9.1)') trap%pos(2,i-1)
          yc = adjustl(yc)
          !Define 
          j = Hub%S%map(i)
          if(.not.visited(j))then
             write(cn,'(i9)') j-1
             cn = adjustl(cn)
             write(label(j),'("#Class",1x,A,":",1x,A,1x,A)') trim(cn), trim(xc), trim(yc)
             visited(j)=.true.
          else
             write(label(j),'(A,",",1x,A,1x,A)') trim(label(j)), trim(xc), trim(yc)
          endif
       enddo

       !Write "plottable" file
       open(unit=OPT3, file="gtau_"//adjustl(trap%ofile))
       visited = .false.
       do i = 1, Hub%n
          j = Hub%S%map(i)
          if(visited(j))cycle
          write(OPT3,'(A)') trim(label(j))
          idx = 0
          do k = j, Hub%S%nGroup*Hub%L, Hub%S%nGroup
             write(OPT3,OPTFMT) idx*Hub%dtau, trap%Gavg(k,avg), trap%Gavg(k,err)
             idx = idx + 1
          end do
          write(OPT3,OPTFMT) idx*Hub%dtau, 1.d0-trap%Gavg(j,avg), trap%Gavg(j,err)
          visited(j) = .true.
          write(OPT3,*)
          write(OPT3,*)
       enddo
       close(OPT3)

       !Write binary file with all bins.
       open(unit=OPT3, file="gtau_"//trim(adjustl(trap%ofile))//".bin", form='unformatted')
       write(OPT3) Hub%S%nGroup, Hub%L
       do i = 1, nbin
          write(OPT3)trap%Gavg(:,i)
       enddo
       close(OPT3)

       deallocate(label, visited)

    endif

  end subroutine Gtau_Print

  ! ----------------------------------------------------------------- !
