module DQMC_WRAPPER1

  use DQMC_Config
  use DQMC_HUBBARD
  use DQMC_3BAND
  use DQMC_TDM
  use DQMC_GTAU

  implicit none 
  ! 
  ! This module defines a set of wrapper functions for 
  ! other programs. It computes all the equal time +
  ! uneqaul time physical measurements on a two dimensional
  ! rectangular lattice.
  !  
  ! Data Type
  ! =========
  !
  type Wrapper1
     type(Config)  :: cfg
     type(hubbard) :: Hub             ! Hubbard model
     type(TDM)     :: Tm              ! time dependent measurement 1
     integer       :: ntausk
  end type Wrapper1

  !
  ! Parameters
  ! ==========
  !
  ! output results
  integer, parameter :: EQ_UP_SPIN      =     1
  integer, parameter :: EQ_DN_SPIN      =     2
  integer, parameter :: EQ_P_ENERGY     =     3
  integer, parameter :: EQ_K_ENERGY     =     4
  integer, parameter :: EQ_ENERGY       =     5
  integer, parameter :: EQ_DENSITY      =     6
  integer, parameter :: EQ_XX_FERR      =     7
  integer, parameter :: EQ_ZZ_FERR      =     8
  integer, parameter :: EQ_XX_AFER      =     9
  integer, parameter :: EQ_ZZ_AFER      =    11
                                                               
  integer, parameter :: EQ_G_FUN        =   101
  integer, parameter :: EQ_D_UPUP       =   102
  integer, parameter :: EQ_D_UPDN       =   103
  integer, parameter :: EQ_XX_SPIN      =   104
  integer, parameter :: EQ_ZZ_SPIN      =   105
                                                
  integer, parameter :: EQ_P_MASK       =    20
  integer, parameter :: EQ_P_S          =    21
  integer, parameter :: EQ_P_SX         =    22
  integer, parameter :: EQ_P_D          =    23
  integer, parameter :: EQ_P_SXX        =    24
  integer, parameter :: EQ_P_DXX        =    25
  integer, parameter :: EQ_P_PX         =    26
  integer, parameter :: EQ_P_PY         =    27
  integer, parameter :: EQ_P_PXY        =    28
  integer, parameter :: EQ_P_PYX        =    29
                                               
  integer, parameter :: UNEQ_F_MASK     =   100 
  integer, parameter :: UNEQ_G_NL       =   303 
  integer, parameter :: UNEQ_G_QL       =   306
  integer, parameter :: UNEQ_G_NW       =   309
  integer, parameter :: UNEQ_G_QW       =   312
                                                            
  integer, parameter :: UNEQ_CHI_NL     =   503
  integer, parameter :: UNEQ_CHI_QL     =   506
  integer, parameter :: UNEQ_CHI_NW     =   509
  integer, parameter :: UNEQ_CHI_QW     =   512
                                                               
  integer, parameter :: UNEQ_P_S_L      =   221
  integer, parameter :: UNEQ_P_SX_L     =   222
  integer, parameter :: UNEQ_P_D_L      =   223
  integer, parameter :: UNEQ_P_SXX_L    =   224
  integer, parameter :: UNEQ_P_DXX_L    =   225
  integer, parameter :: UNEQ_P_PX_L     =   226
  integer, parameter :: UNEQ_P_PY_L     =   227
  integer, parameter :: UNEQ_P_PXY_L    =   228
  integer, parameter :: UNEQ_P_PYX_L    =   229
                                                               
  integer, parameter :: UNEQ_P_S_W      =   421 
  integer, parameter :: UNEQ_P_SX_W     =   422
  integer, parameter :: UNEQ_P_D_W      =   423
  integer, parameter :: UNEQ_P_SXX_W    =   424
  integer, parameter :: UNEQ_P_DXX_W    =   425
  integer, parameter :: UNEQ_P_PX_W     =   426
  integer, parameter :: UNEQ_P_PY_W     =   427
  integer, parameter :: UNEQ_P_PXY_W    =   428
  integer, parameter :: UNEQ_P_PYX_W    =   429

  integer, parameter :: UNEQ_NV_P_S_L   =   621
  integer, parameter :: UNEQ_NV_P_SX_L  =   622
  integer, parameter :: UNEQ_NV_P_D_L   =   623
  integer, parameter :: UNEQ_NV_P_SXX_L =   624
  integer, parameter :: UNEQ_NV_P_DXX_L =   625
  integer, parameter :: UNEQ_NV_P_PX_L  =   626
  integer, parameter :: UNEQ_NV_P_PY_L  =   627
  integer, parameter :: UNEQ_NV_P_PXY_L =   628
  integer, parameter :: UNEQ_NV_P_PYX_L =   629
                                                               
  integer, parameter :: UNEQ_NV_P_S_W   =   821 
  integer, parameter :: UNEQ_NV_P_SX_W  =   822
  integer, parameter :: UNEQ_NV_P_D_W   =   823
  integer, parameter :: UNEQ_NV_P_SXX_W =   824
  integer, parameter :: UNEQ_NV_P_DXX_W =   825
  integer, parameter :: UNEQ_NV_P_PX_W  =   826
  integer, parameter :: UNEQ_NV_P_PY_W  =   827
  integer, parameter :: UNEQ_NV_P_PXY_W =   828
  integer, parameter :: UNEQ_NV_P_PYX_W =   829

contains

  !--------------------------------------------------------------------

  subroutine DQMC_Wrap1_ReadBoth(wrap1, n, L, fname)
    !
    ! This subroutine reads in parameters standard input. The format 
    ! can be ether the legacy format or the new input format.
    !
    ! Arguments
    ! =========
    type(Wrapper1), intent(inout)   :: wrap1
    integer, intent(in)             :: n, L
    character(30), intent(inout)    :: fname
    
    ! ... Local variables...
    character(len=80)  :: str
    character, parameter   :: STARTCHAR = "<"
    integer            :: ios, pos

    ! ... Executable ...

    read (unit=STDIN, FMT="(a)", iostat=ios)  str
    str = trim(str)

    pos = scan(str, STARTCHAR)
    if (pos .gt. 0) then
       print *, str
       call DQMC_Wrap1_Init(wrap1, STDIN)
    else
       print *, "<Legacy input format>"
       fname = trim(str)
       call DQMC_Wrap1_Readin(wrap1, n, L, fname, .false.)
    end if

  end subroutine DQMC_Wrap1_ReadBoth

  !-------------------------------------------------------------------- 

  subroutine DQMC_Wrap1_Readin(wrap1, n, L, fname, readfname)
    !
    ! This subroutine reads in parameters from standard input.
    !
    ! Arguments
    ! =========
    type(Wrapper1), intent(inout)   :: wrap1
    integer, intent(in)             :: n, L
    character(30), intent(inout)    :: fname
    logical, intent(in)             :: readfname
    
    ! ... Local variables ...
    real(wp) r1, r2, r3, r4, pr1(1)
    integer  i1, i2, i3, i4

    integer, parameter :: nBin   = 10
    integer, parameter :: nHist  = 50
    integer, parameter :: tausk  = 10
    integer, parameter :: nItvl  = 4

    ! ... Executable ...

    call DQMC_Config_default(wrap1%cfg)

    ! Setup default values
    call CFG_Set(wrap1%cfg, PARAM_NX, n)
    call CFG_Set(wrap1%cfg, PARAM_NY, n)
    call CFG_Set(wrap1%cfg, PARAM_L,  L)

    call CFG_Set(wrap1%cfg, PARAM_NBIN,  nBin)
    call CFG_Set(wrap1%cfg, PARAM_NHIST, nHist)

    call CFG_Set(wrap1%cfg, PARAM_TAUSK, tausk)
    call CFG_Set(wrap1%cfg, PARAM_NITVL, nitvl)

    ! Read in parameters from stdin
    if (readfname) then
       write(STDOUT,*) "Enter output file name:"
       read (STDIN, *) fname
    end if
    call CFG_Set(wrap1%cfg, PARAM_FNAME, fname)


    write(STDOUT,*) "Enter random seed:" 
    read (STDIN, *) i1
    call CFG_Set(wrap1%cfg, PARAM_IDUM, i1)

    write(STDOUT,*) "Enter t mu delmu dtau:"
    read (STDIN, *) r1, r2, r3, r4
    pr1(1) = r1
    call CFG_Set(wrap1%cfg, PARAM_t, 1, pr1)
    pr1(1) = r2
    call CFG_Set(wrap1%cfg, PARAM_MU, 1, pr1)
    call CFG_Set(wrap1%cfg, PARAM_DTAU, r4)
    
    write(STDOUT,*) "Enter nwarm  npass:"
    read (STDIN, *) i1, i2
    call CFG_Set(wrap1%cfg, PARAM_NWARM, i1)
    call CFG_Set(wrap1%cfg, PARAM_NPASS, i2)

    write(STDOUT,*) "Enter U:"
    read (STDIN, *) r1
    pr1(1) = r1
    call CFG_Set(wrap1%cfg, PARAM_U, 1, pr1)

    write(STDOUT,*) "Enter nwrap err1 err2:"
    read (STDIN, *) i1, r1, r2
    call CFG_Set(wrap1%cfg, PARAM_NWRAP, i1)
    call CFG_Set(wrap1%cfg, PARAM_ERRAT, r1)
    call CFG_Set(wrap1%cfg, PARAM_DIFF,  r2)

    write(STDOUT,*) "Enter doauto  orthlen  eorth   dopair  numpair:"
    read (STDIN, *) i1, i2, r1, i3, i4
    call CFG_Set(wrap1%cfg, PARAM_NORTH,  i2)
    call CFG_Set(wrap1%cfg, PARAM_NMEAS,  i4)
    
    write(STDOUT,*) "Enter startype (999=readin old HS field values)"
    read (STDIN, *) i1

    if (i1 .eq. 999) then  ! read in HSF now
       call DQMC_Reshape(n, L, wrap1%Hub%HSF)
       call DQMC_Hub_Input_HSF(n, L, wrap1%Hub%HSF, STDIN)
       wrap1%cfg%I(PARAM_HSF) = 0
    else
       ! randomly generate HSF
       wrap1%cfg%I(PARAM_HSF) = -1
    end if

    ! Initialize the model
    wrap1%ntausk = CFG_Get(wrap1%cfg, PARAM_TAUSK)
    call DQMC_Init_3Band(n, n, wrap1%Hub%S, IMP_RECTANGLE)
    call DQMC_Hub_Config(wrap1%Hub, wrap1%cfg)
    call DQMC_TDM_Init(TDM_ALL, wrap1%Tm, wrap1%Hub%S, &
         wrap1%Hub%WS, wrap1%Hub%B, wrap1%cfg)

  end subroutine DQMC_Wrap1_Readin

  !--------------------------------------------------------------------

  subroutine DQMC_Wrap1_Init(wrap1, IPT)
    !
    ! This subroutine initializes some parameters of wrap1
    !
    ! Arguments
    ! =========
    type(Wrapper1), intent(inout) :: wrap1
    integer, intent(in)           :: IPT

    ! ... local variables ...
    integer :: nx, ny, nitvl

    ! ... Executable ...

    call DQMC_Config_Read(wrap1%cfg, STDIN, CONFIG_CONFIG)
    
    ! Read parameters from config
    wrap1%ntausk = CFG_Get(wrap1%cfg, PARAM_TAUSK)
    nx =           CFG_Get(wrap1%cfg, PARAM_NX)
    ny =           CFG_Get(wrap1%cfg, PARAM_NY)
    nitvl =        CFG_Get(wrap1%cfg, PARAM_NITVL)

    ! Initialize the Hubbard model
    call DQMC_Init_3Band(nx, ny, wrap1%Hub%S, IMP_RECTANGLE)
    call DQMC_Hub_Config(wrap1%Hub, wrap1%cfg)
    call DQMC_TDM_Init(TDM_ALL, wrap1%Tm, wrap1%Hub%S, &
         wrap1%Hub%WS, wrap1%Hub%B, wrap1%cfg)
    
  end subroutine DQMC_Wrap1_Init

  !--------------------------------------------------------------------

  subroutine DQMC_Wrap1_PrintParam(wrap1, OPT)
    !
    ! This subroutine initializes some parameters of wrap1
    !
    ! Arguments
    ! =========
    type(Wrapper1), intent(inout) :: wrap1
    integer, intent(in)           :: OPT

    ! ... Executable ...

    call DQMC_Hub_OutputParam(wrap1%Hub, OPT)  

  end subroutine DQMC_Wrap1_PrintParam

  !--------------------------------------------------------------------

  subroutine DQMC_Wrap1_OutputHSF(wrap1, fno)
    !
    ! This subroutine initializes some parameters of wrap1
    !
    ! Arguments
    ! =========
    type(Wrapper1), intent(inout) :: wrap1
    integer, intent(in)           :: fno

    ! ... Executable ...

    call DQMC_Hub_Output_HSF(wrap1%Hub%n, wrap1%Hub%L, wrap1%Hub%HSF, fno)

  end subroutine DQMC_Wrap1_OutputHSF

  !--------------------------------------------------------------------

  subroutine DQMC_Wrap1_Free(wrap1)
    !
    ! This subroutine santizes storage.
    !
    ! Arguments
    ! =========
    type(Wrapper1), intent(inout) :: wrap1

    ! ... Executable ...

    call DQMC_Hub_Free(wrap1%Hub)
    call DQMC_TDM_Free(wrap1%Tm)
    call DQMC_Config_Free(wrap1%cfg)

  end subroutine DQMC_Wrap1_Free

  !--------------------------------------------------------------------

  subroutine DQMC_Wrap1_Compute(wrap1)
    !
    ! This subroutine santizes storage.
    !
    ! Arguments
    ! =========
    type(Wrapper1), intent(inout) :: wrap1
    
    ! ... Local variables ...
    integer :: i, j, k, nBin, nIter, ntausk

    ! ... Executable ...
    nBin   = wrap1%Hub%P0%nBin 
    ntausk = wrap1%ntausk    
    nIter  = wrap1%Hub%nPass/nBin/ntausk
    
    ! Warmup sweep
    do i = 1, wrap1%Hub%nWarm
       ! The second parameter means no measurement should be made.
       call DQMC_Hub_Sweep(wrap1%Hub, NO_MEAS0)
    end do
 
    ! We divide all the measurement into nBin,
    ! each having nPass/nBin pass.
    
    do i = 1, nBin
       do j = 1, nIter
          do k = 1, ntausk
             call DQMC_Hub_Sweep(wrap1%Hub, wrap1%Hub%nMeas)
          end do
          call DQMC_TDM_Meas(wrap1%tm, wrap1%Hub%G_up, wrap1%Hub%G_dn)
       end do
       
       ! Accumulate results for each bin
       call DQMC_TDM_Avg(wrap1%Tm)
       ! Accumulate results for each bin
       call DQMC_Phy0_Avg(wrap1%Hub%P0)
       call DQMC_Phy2_Avg(wrap1%Hub%P2, wrap1%Hub%S%W)
    end do
    
    ! Get average result
    call DQMC_Phy0_GetErr(wrap1%Hub%P0)
    call DQMC_Phy2_GetErr(wrap1%Hub%P2)
    call DQMC_TDM_GerErr(wrap1%Tm)

  end subroutine DQMC_Wrap1_Compute

  !--------------------------------------------------------------------

  subroutine DQMC_Wrap1_Get(wrap1, which, m, n, avg, err)    
    !
    ! This subroutine returns computed results.
    !
    ! Arguments
    ! =========
    type(Wrapper1), intent(inout)  :: wrap1
    integer, intent(in)            :: which
    integer, intent(out)           :: m, n
    real(wp), pointer, intent(out) :: avg(:,:), err(:,:)
    
    ! ... Local variables ...
    integer :: idx, avgi, erri

    ! ... Executable ...
    
    
    select case (which)
    case (EQ_UP_SPIN:EQ_ZZ_AFER)
       m = 1
       n = 1
       idx = mod(which, P0_N)
       avgi = wrap1%Hub%P0%avg
       erri = wrap1%Hub%P0%err
       avg => wrap1%Hub%P0%S(idx:idx, avgi:avgi)
       err => wrap1%Hub%P0%S(idx:idx, erri:erri)

    case (EQ_G_FUN)
       m = wrap1%Hub%S%nClass
       n = 1
       avg => wrap1%Hub%P0%G_fun(1:m, wrap1%Hub%P0%avg:wrap1%Hub%P0%avg)
       err => wrap1%Hub%P0%G_fun(1:m, wrap1%Hub%P0%err:wrap1%Hub%P0%err)

    case (EQ_D_UPUP)
       m = wrap1%Hub%S%nClass
       n = 1
       avg => wrap1%Hub%P0%Den0(1:m, wrap1%Hub%P0%avg:wrap1%Hub%P0%avg)
       err => wrap1%Hub%P0%Den0(1:m, wrap1%Hub%P0%err:wrap1%Hub%P0%err)

    case (EQ_D_UPDN)
       m = wrap1%Hub%S%nClass
       n = 1
       avg => wrap1%Hub%P0%Den1(1:m, wrap1%Hub%P0%avg:wrap1%Hub%P0%avg)
       err => wrap1%Hub%P0%Den1(1:m, wrap1%Hub%P0%err:wrap1%Hub%P0%err)


    case (EQ_XX_SPIN)
       m = wrap1%Hub%S%nClass
       n = 1
       avg => wrap1%Hub%P0%SpinXX(1:m, wrap1%Hub%P0%avg:wrap1%Hub%P0%avg)
       err => wrap1%Hub%P0%SpinXX(1:m, wrap1%Hub%P0%err:wrap1%Hub%P0%err)

    case (EQ_ZZ_SPIN)
       m = wrap1%Hub%S%nClass
       n = 1
       avg => wrap1%Hub%P0%SpinZZ(1:m, wrap1%Hub%P0%avg:wrap1%Hub%P0%avg)
       err => wrap1%Hub%P0%SpinZZ(1:m, wrap1%Hub%P0%err:wrap1%Hub%P0%err)

    case (EQ_P_S:EQ_P_PYX)
       m = 1
       n = 1
       idx = mod(which, EQ_P_MASK)
       avg => wrap1%Hub%P2%M3(idx:idx, wrap1%Hub%P2%avg:wrap1%Hub%P2%avg)
       err => wrap1%Hub%P2%M3(idx:idx, wrap1%Hub%P2%err:wrap1%Hub%P2%err)

       ! unequal time Green's function
       
    case (UNEQ_G_NL)
       m = wrap1%Hub%S%nClass
       n = wrap1%Hub%L + 1

       avg => wrap1%Tm%T1%gnl(1:m, 1:n,  wrap1%Tm%T1%avg)
       err => wrap1%Tm%T1%gnl(1:m, 1:n,  wrap1%Tm%T1%err)

    case (UNEQ_G_QL)
       m = wrap1%Hub%S%nClass
       n = wrap1%Hub%L + 1

       avg => wrap1%Tm%T1%gql(1:m, 1:n,  wrap1%Tm%T1%avg)
       err => wrap1%Tm%T1%gql(1:m, 1:n,  wrap1%Tm%T1%err)


    case (UNEQ_G_NW)
       m = wrap1%Hub%S%nClass
       n = wrap1%Hub%L + 2

       avg => wrap1%Tm%T1%gnw(1:m, 1:n,  wrap1%Tm%T1%avg)
       err => wrap1%Tm%T1%gnw(1:m, 1:n,  wrap1%Tm%T1%err)

    case (UNEQ_G_QW)
       m = wrap1%Hub%S%nClass
       n = wrap1%Hub%L + 2

       avg => wrap1%Tm%T1%gqw(1:m, 1:n,  wrap1%Tm%T1%avg)
       err => wrap1%Tm%T1%gqw(1:m, 1:n,  wrap1%Tm%T1%err)

       ! unequal time chi 
    case (UNEQ_CHI_NL)
       m = wrap1%Hub%S%nClass
       n = wrap1%Hub%L + 1

       avg => wrap1%Tm%T1%chinl(1:m, 1:n, wrap1%Tm%T1%avg)
       err => wrap1%Tm%T1%chinl(1:m, 1:n, wrap1%Tm%T1%err)

    case (UNEQ_CHI_QL)
       m = wrap1%Hub%S%nClass
       n = wrap1%Hub%L + 1

       avg => wrap1%Tm%T1%chiql(1:m, 1:n, wrap1%Tm%T1%avg)
       err => wrap1%Tm%T1%chiql(1:m, 1:n, wrap1%Tm%T1%err)


    case (UNEQ_CHI_NW)
       m = wrap1%Hub%S%nClass
       n = wrap1%Hub%L + 1

       avg => wrap1%Tm%T1%chinw(1:m, 1:n, wrap1%Tm%T1%avg)
       err => wrap1%Tm%T1%chinw(1:m, 1:n, wrap1%Tm%T1%err)

    case (UNEQ_CHI_QW)
       m = wrap1%Hub%S%nClass
       n = wrap1%Hub%L + 1

       avg => wrap1%Tm%T1%chiqw(1:m, 1:n, wrap1%Tm%T1%avg)
       err => wrap1%Tm%T1%chiqw(1:m, 1:n, wrap1%Tm%T1%err)

       ! unequal time pair measurements
    case (UNEQ_P_S_L:UNEQ_P_PYX_L)
       m = 1
       n = wrap1%Hub%L + 1
       idx = mod(which, EQ_P_MASK)
       avg => wrap1%Tm%T2%BPair(idx:idx, 1:n, wrap1%Tm%T2%avg)
       err => wrap1%Tm%T2%BPair(idx:idx, 1:n, wrap1%Tm%T2%err)


    case (UNEQ_P_S_W:UNEQ_P_PYX_W)
       m = 1
       n = wrap1%Hub%L + 2
       idx = mod(which, EQ_P_MASK)
       avg => wrap1%Tm%T2%BPairFT(idx:idx, 1:n, wrap1%Tm%T2%avg)
       err => wrap1%Tm%T2%BPairFT(idx:idx, 1:n, wrap1%Tm%T2%err)

    case (UNEQ_NV_P_S_L:UNEQ_NV_P_PYX_L)
       m = 1
       n = wrap1%Hub%L + 1
       idx = mod(which, EQ_P_MASK)
       avg => wrap1%Tm%T2%NPair(idx:idx, 1:n, wrap1%Tm%T2%avg)
       err => wrap1%Tm%T2%NPair(idx:idx, 1:n, wrap1%Tm%T2%err)


    case (UNEQ_NV_P_S_W:UNEQ_NV_P_PYX_W)
       m = 1
       n = wrap1%Hub%L + 2
       idx = mod(which, EQ_P_MASK)
       avg => wrap1%Tm%T2%NPairFT(idx:idx, 1:n, wrap1%Tm%T2%avg)
       err => wrap1%Tm%T2%NPairFT(idx:idx, 1:n, wrap1%Tm%T2%err)

    case default
       stop "No such measurement !!"
    end select
    
  end subroutine DQMC_Wrap1_Get

end module DQMC_WRAPPER1
