module DQMC_3BAND
#include "dqmc_include.h"

  use DQMC_UTIL
  use DQMC_CONFIG
  use DQMC_STRUCT
  use DQMC_HUBBARD
  implicit none 

  ! 
  ! This module defines subroutines to initialize the data structure of 
  ! a two-dimensional periodic rectangular lattice (2DPerl).
  !     
  !  Subroutine List
  !  ===============
  !    DQMC_Read2PERL(Hub, IPT, OPT) : read in data for 2D rectangular 
  !                                    lattice.
  !    DQMC_2DREC(nx, ny, S) : construct data for a 2D rectangular lattice.   
  !

  integer, parameter :: IMP_TRIANGLE  = 1
  integer, parameter :: IMP_RECTANGLE = 2
  
contains
  
  !---------------------------------------------------------------------!

  subroutine DQMC_Comp_3Band(IPT, OPT, cfg_type)
    !
    ! Purpose
    ! =======
    !    This subroutine reads in data for a 2D rectangular lattice,
    !    and the calls DQMC_2DREC to construct the lattice structure.
    !    Basically, the only parameters needed are Nx and Ny, which
    !    the size of the lattice along the x-cord and the y-cord.
    !
    ! Arguments
    ! =========
    !
    integer, intent(in) :: IPT, OPT     ! Input/output handle
    integer, intent(in) :: cfg_type     ! config file format

    ! ... Local scalar ...
    type(config)  :: cfg
    type(Hubbard) :: Hub          ! Hubbard model

    ! ... Executable ...

    call DQMC_Config_Read(cfg, IPT, cfg_type)

    ! Initialize the geometry
    call DQMC_Init_3Band(CFG_Get(cfg, PARAM_NX), CFG_Get(cfg, PARAM_NY), &
         Hub%S, IMP_TRIANGLE)

    ! Initialize the rest data
    call DQMC_Hub_Config(Hub, cfg)

    ! Execution MC loop
    call DQMC_Hub_Run(Hub)

    ! Print computed results
    call DQMC_Hub_Print(Hub, OPT)

    ! Clean up the used storage
    call DQMC_Hub_Free(Hub)
    call DQMC_Config_Free(cfg)
    
  end subroutine DQMC_Comp_3Band
  
  !---------------------------------------------------------------------!

  subroutine DQMC_Init_3Band(nx, ny, S, IMP)    
    !
    ! Purpose
    ! =======
    !    This subroutine constuctures data structure for a 2D 
    !    periodic rectangular lattice.  
    !
    ! Details
    ! =======
    !    For a Nx*Ny 2D rectangular lattice.
    !
    !    1. The sites are numbered from 1, which is the site on the 
    !       south-west corner. The numbering is row major, which means
    !       it increases along x direction first (left to right) and 
    !       then y-direction (bottom up).
    !
    !    2. Adjacency (T) has exact 4 elements per site: left
    !       right, up and down. Since the adjacency is cyclic, 
    !       the code has had spacial treat for boundary sites.
    !       *** The way it computed here is not satisfied the 
    !           'checkboard' order. should change it later.
    !
    !    3. The number of unique distance is computed as follows.
    !       Let long = max(Nx, Ny) and short = min(Nx, Ny).
    !       The distinct distance sites form a trapezoid
    !       The bottom is (long/2+1), the top is ((long-short)/2+1)
    !       and the height is (short/2+1). Therefore, 
    !       
    !           nClass = (short/2+1)*(long-short/2+2)/2
    !       
    !    4. The phase (P) is computed in the rules that
    !       (a) adjacent sites have opposite phase.
    !       (b) site 1 is phased +.
    !
    ! Arguments
    ! =========
    !
    integer, intent(in)         :: nx, ny  ! dimension of the lattice
    type(Struct), intent(inout) :: S       ! Struct
    integer, intent(in)         :: IMP  

    ! ... local scalar ...
    integer  :: n                ! Order of matrix T and D 
    integer  :: i, j             ! Loop iterator
    integer  :: idx              ! 

    ! ... local array ...
    real(wp) :: mphase(nx*ny)    ! Used in phase computation
    integer  :: sitemap(nx, ny)

    ! ... parameters ...
    integer, parameter :: NADJ  = 4  ! Number of adjacencies
    integer, parameter :: NNBR  = 9  ! Number of neighbors
    
    integer, parameter :: NE    = 1  ! index of neighbor  
    integer, parameter :: E_    = 2
    integer, parameter :: SE    = 3
    integer, parameter :: N_    = 4
    integer, parameter :: C_    = 5
    integer, parameter :: S_    = 6
    integer, parameter :: NW    = 7
    integer, parameter :: W_    = 8
    integer, parameter :: SW    = 9   

    ! ... Executable ...

    if (nx .le. 2 .or. ny .le. 2) then
       call DQMC_Error("3 Band lattice size must be larger than 2", 1)
    end if

    n   = nx*ny
    S%nSite = n
    write(S%name,'(A,I3,A,I3,A,I3)') &
         "3 Band 2D Periodic Lattice (Nx=", &
         nx, ", Ny=", ny, ") total sites=", S%nSite
    
    ! Compute all distinct sites
    call DQMC_Reshape(n,3,S%BS)
    S%BS = NADJ
    S%nBand = 3
    S%max_nbr = NNBR

    ! memory allocation
    call DQMC_Reshape(n, NADJ*3, S%T)
    call DQMC_Reshape(NNBR, n, S%N)
    call DQMC_Reshape(9, 9, S%W)
    call DQMC_Reshape(2, S%dim)
    call DQMC_Reshape(n, S%uMap)
    call DQMC_Reshape(n, S%muMap)

    S%dim(1) = nx
    S%dim(2) = ny
    S%uMap   = 1
    S%muMap  = 1

    ! Build adjacent matrix
    ! Assign index of lattice
    do j = 1, ny
       do i = 1, nx
          sitemap(i,j) = i + (j-1)*nx
       end do
    end do


    ! Fill the T table
    idx = 0
    do j = 1, ny
       do i = 1, nx
          idx = idx + 1

          ! BAND 1 BAND 1 BAND 1 BAND 1 BAND 1 BAND 1
          ! (1, 0)
          S%T(idx, 1) = sitemap(mod(i,nx)+1, j)

          ! (-1, 0)
          
          if (i .eq. 1) then
             S%T(idx, 2) = sitemap(nx, j)
          else
             S%T(idx, 2) = sitemap(i-1, j)
          end if

          ! (0, 1)
          S%T(idx, 3) = sitemap(i, mod(j,ny)+1)

          ! (0, -1)
          if (j .eq. 1) then
             S%T(idx, 4) = sitemap(i, ny)
          else
             S%T(idx, 4) = sitemap(i, j-1)
          end if

          ! BAND 2 BAND 2 BAND 2 BAND 2 BAND 2 BAND 2
          ! (1, 1)
          if (i .eq. nx) then
             if (j .eq. ny) then
                S%T(idx, 5) = sitemap(1, 1)
             else
                S%T(idx, 5) = sitemap(1, j+1)
             end if
          elseif (j .eq. ny) then
             S%T(idx, 5) = sitemap(i+1, 1)
          else
             S%T(idx, 5) = sitemap(i+1, j+1)
          end if


          ! (1, -1)
          if (i .eq. nx) then
             if (j .eq. 1) then
                S%T(idx, 6) = sitemap(1, ny)
             else
                S%T(idx, 6) = sitemap(1, j-1)
             end if
          elseif (j .eq. 1) then
             S%T(idx, 6) = sitemap(i+1, ny)
          else
             S%T(idx, 6) = sitemap(i+1, j-1)
          end if

          ! (-1, 1)
          if (i .eq. 1) then
             if (j .eq. ny) then
                S%T(idx, 7) = sitemap(nx, 1)
             else
                S%T(idx, 7) = sitemap(nx, j+1)
             end if
          elseif (j .eq. ny) then
             S%T(idx, 7) = sitemap(i-1, 1)
          else
             S%T(idx, 7) = sitemap(i-1, j+1)
          end if

          ! (-1, -1)
          if (i .eq. 1) then
             if (j .eq. 1) then
                S%T(idx, 8) = sitemap(nx, ny)
             else
                S%T(idx, 8) = sitemap(nx, j-1)
             end if
          elseif (j .eq. 1) then
             S%T(idx, 8) = sitemap(i-1, ny)
          else
             S%T(idx, 8) = sitemap(i-1, j-1)
          end if

          ! BAND 3 BAND 3 BAND 3 BAND 3 BAND 3 BAND 3
          ! (2, 0)
          S%T(idx, 9) = sitemap(mod(i+1,nx)+1, j)

          ! (-2, 0)
          if (i .le. 2) then
             S%T(idx, 10) = sitemap(nx+i-2, j)
          else
             S%T(idx, 10) = sitemap(i-2, j)
          end if

          ! (0, 2)
          S%T(idx, 11) = sitemap(i, mod(j+1,ny)+1)

          ! (0, -2)
          if (j .le. 2) then
             S%T(idx, 12) = sitemap(i, ny+j-2)
          else
             S%T(idx, 12) = sitemap(i, j-2)
          end if          

       end do
    end do

    ! build up the distance matrix.
    if (IMP .eq. IMP_TRIANGLE) then
       call DQMC_2DPerl_DC_Imp1(n, nx, ny, S)
    else
       call DQMC_2DPerl_DC_Imp2(n, nx, ny, S)
    end if

    ! Build neighboring matrix from the adjacent matrix
    S%N(E_,1:n) = S%T(1:n, 1)
    S%N(W_,1:n) = S%T(1:n, 2)   
    S%N(N_,1:n) = S%T(1:n, 3)
    S%N(S_,1:n) = S%T(1:n, 4) 
    S%N(NE,1:n) = S%T(1:n, 5) 
    S%N(SE,1:n) = S%T(1:n, 6) 
    S%N(NW,1:n) = S%T(1:n, 7) 
    S%N(SW,1:n) = S%T(1:n, 8) 
    do i = 1, n
       S%N(C_,i) = i
    end do

    ! Initialize phase matrix
    call DQMC_Reshape(n, n, S%P)
    mphase(1:nx:2) = -1
    mphase(2:nx:2) = 1
    do i = 2, ny, 2
       mphase((i-1)*nx+1:i*nx) =  -mphase(1:nx)
    end do

    do i = 3, ny, 2
       mphase((i-1)*nx+1:i*nx) = mphase(1:nx)
    end do

    do i = 1,n
       do j = 1,n
          S%P(i,j) = mphase(i)*mphase(j)
       end do
    end do
    

    S%nWave = 9
    ! Make wave matrix
    S%W(:,1) = (/ ZERO, ZERO, ZERO, ZERO,  ONE, ZERO, ZERO, ZERO, ZERO/)
    S%W(:,2) = (/ ZERO, HALF, ZERO, HALF, ZERO, HALF, ZERO, HALF, ZERO/)
    S%W(:,3) = (/ ZERO,-HALF, ZERO, HALF, ZERO, HALF, ZERO,-HALF, ZERO/)
    S%W(:,4) = (/ HALF, ZERO, HALF, ZERO, ZERO, ZERO, HALF, ZERO, HALF/)
    S%W(:,5) = (/-HALF, ZERO, HALF, ZERO, ZERO, ZERO, HALF, ZERO,-HALF/)
    S%W(:,6) = (/ ZERO, ZERO, ZERO,-HALF, ZERO, HALF, ZERO, ZERO, ZERO/)
    S%W(:,7) = (/ ZERO,-HALF, ZERO, ZERO, ZERO, ZERO, ZERO, HALF, ZERO/)
    S%W(:,8) = (/-HALF, ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, HALF/)
    S%W(:,9) = (/ ZERO, ZERO,-HALF, ZERO, ZERO, ZERO, HALF, ZERO, ZERO/)

    ! label for each wave function
    allocate(S%wlabel(9))
    S%wlabel(1) = "  S-Wave : "
    S%wlabel(2) = " SX-Wave : "
    S%wlabel(3) = "  D-Wave : "
    S%wlabel(4) = "SXX-Wave : "
    S%wlabel(5) = "DXX-Wave : "
    S%wlabel(6) = " PX-Wave : "    
    S%wlabel(7) = " PY-Wave : "    
    S%wlabel(8) = "PXY-Wave : "    
    S%wlabel(9) = "PYX-Wave : "

    ! enable the flag
    S%init = .true.

  end subroutine DQMC_INIT_3BAND

  !---------------------------------------------------------------------!

  subroutine DQMC_2DPerl_DC_Imp1(n, nx, ny, S)    
    !
    ! Purpose
    ! =======
    !    This subroutine implements distance classification, which
    !    use compact stragety
    !
    ! Arguments
    ! =========
    !
    integer, intent(in)         :: n, nx, ny  ! dimension of the lattice
    type(Struct), intent(inout) :: S          ! Struct

    ! ... local vars ...
    integer  :: L(nx*ny,nx*ny)   ! distance table
    integer  :: long, short      ! Used in computing nClass
    integer  :: i, j, ix, iy, jx, jy, dx, dy, idx


    ! site i to i is a special case, do not compute it
    long = max(nx, ny)
    short = min(nx, ny)
    S%nClass = (short/2+1)*(long-short/2+2)/2

    call DQMC_Reshape(n, n, S%D)
    call DQMC_Reshape(S%nClass, S%F)
    allocate(S%label(S%nClass))
    call DQMC_Reshape(S%nClass, 2, S%cord)

    S%D = 0
    S%F = 0

    S%F(1) = n
    write(S%label(1),FMT_INTPAR) 0, 0

    !! using lookup table (maybe hashing table )
    L = 0
    idx = 1 ! the first one is a special one
    do i = 1, n
       !! compute the index of i
       ix = mod(i-1, nx)+1
       iy = (i-1)/nx + 1

       !! initial the index of j
       do j = i+1, n
          !! compute the index of j
          jx = mod(j-1, nx)+1
          jy = (j-1)/nx + 1

          !! compute the distance
          dx = abs(ix-jx)
          dx = min(dx,nx-dx)
          dy = abs(iy-jy)
          dy = min(dy,ny-dy)
          long = max(dx, dy) + 1
          short = min(dx, dy) + 1
          
          ! not found
          if (L(long,short) .eq. 0) then

             idx = idx + 1             
             L(long, short) = idx
             S%D(i,j) = idx
             write(S%label(idx),FMT_INTPAR) long-1, short-1
             S%cord(idx, 1) = long  - 1
             S%cord(idx, 2) = short - 1

          else ! found

             S%D(i,j) = L(long, short)

          end if

          ! matrix D is symmetric
          S%D(j,i) = S%D(i,j)

          ! increase count by 2
          S%F(S%D(i,j)) = S%F(S%D(i,j)) + 2

       end do

       ! site i to i
       S%D(i,i) = 1
    end do

  end subroutine DQMC_2DPERL_DC_IMP1

  !---------------------------------------------------------------------!

  subroutine DQMC_2DPerl_DC_Imp2(n, nx, ny, S)    
    !
    ! Purpose
    ! =======
    !    In this implementation of distance classication, the classes
    !    are in a rectangular form. 
    !
    ! Arguments
    ! =========
    !
    integer, intent(in)         :: n, nx, ny  ! dimension of the lattice
    type(Struct), intent(inout) :: S          ! Struct

    ! ... parameters ...
    real(wp), parameter :: TWOPI = 6.283185307179586

    ! ... local vars ...
    integer  :: L(n,n)                        ! distance table
    integer  :: i, j                          ! loop iterators
    integer  :: ix, iy, jx, jy, dx, dy, idx   ! indices

    ! site i to i is a special case, do not compute it
    S%nClass = (nx/2+1)*(ny/2+1)

    call DQMC_Reshape(n, n, S%D)

    call DQMC_Reshape(S%nClass,S%F)
    allocate(S%label(S%nClass))
    call DQMC_Reshape(S%nClass, 2, S%cord)

    S%D  = 0
    S%F  = 0

    S%F(1) = n
    write(S%label(1),FMT_INTPAR) 0, 0

    !! using lookup table (maybe hashing table )
    L = 0
    idx = 1 ! the first one is a special one
    S%cord(1,1) = ZERO
    S%cord(1,2) = ZERO
    do i = 1, n
       !! compute the index of i
       ix = mod(i-1, nx)+1
       iy = (i-1)/nx + 1

       !! initial the index of j
       do j = i+1, n
          !! compute the index of j
          jx = mod(j-1, nx)+1
          jy = (j-1)/nx + 1

          !! compute the distance
          dx = abs(ix-jx)
          dx = min(dx,nx-dx)+1
          dy = abs(iy-jy)
          dy = min(dy,ny-dy)+1
          
          ! not found
          if (L(dx,dy) .eq. 0) then

             ! Creat a new node
             idx = idx + 1             
             L(dx, dy) = idx
             S%D(i,j) = idx
             write(S%label(idx),FMT_INTPAR) dx-1, dy-1

             ! Build a new row of COS table
             S%cord(idx, 1) = dx - 1
             S%cord(idx, 2) = dy - 1
          else ! found

             S%D(i,j) = L(dx, dy)

          end if

          ! matrix D is symmetric
          S%D(j,i) = S%D(i,j)

          ! increase count by 2
          S%F(S%D(i,j)) = S%F(S%D(i,j)) + 2

       end do

       ! site i to i
       S%D(i,i) = 1

    end do

  end subroutine DQMC_2DPERL_DC_IMP2

  !---------------------------------------------------------------------!
end module DQMC_3BAND
 
