! Declares global variables
module global
  implicit none
  save

  integer :: Nx, Ny ! Dimensions of lattice
  integer :: Nclass, Nsites ! Total number of classes and sites
  integer, dimension(:,:), allocatable :: indices ! Holds the definition of 
                                                  ! of the different class 
                                                  ! types

end module global

! Determines the neighbor table, phase table, and measurement class 
! table for a rectangular lattice geometry and writes each to the 
! following files:
!    neigh.table
!    phase.table
!    class.table
program rect_neigh
  use global
  implicit none

  ! Integer
  integer :: i, j, i2, j2 ! x and y indices
  integer :: a, b, c, d ! Neighbors
  integer :: site ! Site label
  integer :: info ! Dummy variable for checking whether allocations occurred.
  integer :: dx, dy, iy ! Class indicators
  integer :: count ! Class counter
  integer :: flag ! Indicates current class

  ! Array containing site labels and relation to x and y coords.
  integer, dimension(:,:), allocatable :: sindex

  ! Read size of lattice
  write(*,*) "Enter Nx, Ny:"
  read(*,*) Nx, Ny

  ! Open table files for writing
  open(unit=11,file="rect.neigh",status='replace',action='WRITE')
  open(unit=12,file="rect.phase",status='replace',action='WRITE')
  open(unit=13,file="rect.class",status='replace',action='WRITE')
  open(unit=14,file="rect.label",status='replace',action='WRITE')

  ! Determine number of classes and sites
  Nclass = 0
  do j=Ny,1,-2
     Nclass = Nclass + j 
  end do
  Nclass = Nclass * (Nx / 2 + 1)
  Nsites = Nx * Ny

  ! Write header for neigh.table and class.table
  write(11,'(2I4)') Nsites, 1
  write(12,'(I4)')  Nsites
  write(13,'(2I4)') Nsites, Nclass
  write(14,'(I4)')  Nclass


  ! Allocate sindex
  allocate(sindex(Nx,Ny), STAT = info )
  if(info.ne.0) then
     write(*,*) 'sindex not allocated' 
     stop
  end if

  ! Allocate indices
  allocate(indices(Nclass,3), STAT = info )
  if(info.ne.0) then
     write(*,*) 'indices not allocated' 
     stop
  end if


  ! Label sites and generate phase table
  site = 0
  do j = 1, Ny
     do i = 1, Nx
        site = site + 1
        sindex(i,j) = site
        if(mod(site,2).eq.0) write(12,'(I5,I3)') site, -1
        if(mod(site,2).eq.1) write(12,'(I5,I3)') site, 1
     end do
  end do

  ! Generate neighbor table
  do j = 1, Ny
     do i = 1, Nx
        if( j.eq.1 ) then ! Bottom edge
           a = i + 1 ! Right
           b = i - 1 ! Left
           c = j + 1 ! Up
           if( a.gt.Nx ) a = a - Nx ! Check PBC in x
           if( b.lt.1 ) b = b + Nx
           write(11,'(2I3," 1")') sindex(i,j), sindex(a,j), &
                sindex(i,j), sindex(b,j),&
                sindex(i,j), sindex(i,c)
        else if( j.eq.Ny ) then ! Top edge
           a = i + 1 ! Right
           b = i - 1 ! Left
           c = j - 1 ! Down
           if( a.gt.Nx ) a = a - Nx ! Check PBC in x
           if( b.lt.1 ) b = b + Nx
           write(11,'(2I3," 1")') sindex(i,j), sindex(a,j), &
                sindex(i,j), sindex(b,j),&
                sindex(i,j), sindex(i,c)
        else ! Inside
           a = i + 1 ! Right
           b = i - 1 ! Left
           c = j + 1 ! Up
           d = j - 1 ! Down
           if( a.gt.Nx ) a = a - Nx ! Check PBC in x
           if( b.lt.1 ) b = b + Nx
           write(11,'(2I3," 1")') sindex(i,j), sindex(a,j), &
                sindex(i,j), sindex(b,j), &
                sindex(i,j), sindex(i,c),&
                sindex(i,j), sindex(i,d)
        end if
     end do
  end do

  ! Generate measurement class table
  count = 0
  ! First site
  do j = 1, Ny
  do i = 1, Nx
     ! Second site
     do j2 = 1, Ny
     do i2 = 1, Nx
        dx = min( abs(i-i2), abs(abs(i-i2)-Nx) ) ! Determine closest x
                                                 ! distance between pair
        dy = abs(j2 - j) ! Vertical distance
        iy = min(j, j2, Ny - j+1, Ny - j2 + 1) ! 1 if edge site involved
                                               ! 2 if only interior sites

        ! Record different classes and check to see if current pair
        ! fits in a class.
        if( count.eq.0 ) then ! First class
           count = count + 1
           indices(count,:) = (/dx, dy, iy/)
           write(14,'(3I3)') dx, dy, iy
           write(13,'(3I3)') sindex(i,j), sindex(i2,j2), count
        else
           call check_index(dx,dy,iy,flag)
           if( flag.eq.0 ) then ! Not a recorded class
              count = count + 1
              indices(count,:) = (/dx, dy, iy/)
              write(14,'(3I3)') dx, dy, iy
              write(13,'(3I3)') sindex(i,j), sindex(i2,j2), count
           else ! Class recorded
              write(13,'(3I3)') sindex(i,j), sindex(i2,j2), flag
           end if
        end if

     end do
     end do
  end do
  end do

  ! Check to make sure correct number of classes found.
  if( count.ne.Nclass ) then
     write(*,*) "Number of classes not equal to predicted value."
     write(*,*) Nclass, count
  end if

!  do i=1,count
!     write(14,'(4I3)') i, indices(i,:)
!  end do
!  write(*,'(I3)') count

  ! Deallocate matrices
  deallocate(sindex)
  deallocate(indices)

  ! Close tables
  close(11)
  close(12)
  close(13)

end program rect_neigh

! This subroutine checks to see if a given combination of dx, dy, and
! iy has been recorded. flag returns 0 if it has not been recorded and
! returns the location (class number) of the input if it exists.
subroutine check_index(dx, dy, iy, flag)
  use global
  implicit none

  integer, intent(in) :: dx, dy, iy
  logical, intent(out) :: flag

  integer :: i

  do i = 1, Nclass
     if( dx.eq.indices(i,1) .AND. dy.eq.indices(i,2) &
          .AND. iy.eq.indices(i,3) ) then
        flag = i ! Class exists in record
        return
     end if
  end do

  flag = 0 ! Class not found
  return
end subroutine check_index
