#!/bin/sh

###############################################################################
## Add directories containing the Bowtie, Tophat, and R executables to PATH: ##
###############################################################################

####################### MUST ADD PATH TO SAMTOOLS ON VICO HERE: ###################################
## Please uncomment the following two lines, if you are running the pipeline
## on vico.genomecenter.ucdavis.edu
#PATH=${PATH}:/share/apps/bowtie-0.12.2:/share/apps/tophat-1.0.12/bin:/share/apps/R-2.11.0/bin
#export PATH
###################################################################################################

## Please uncomment the following two lines, if you are running the pipeline
## on bruno.cs.ucdavis.edu
PATH=${PATH}:/share/apps/bwa:/share/apps/python/bin:/share/apps/R/bin:/share/apps/samtools:/share/apps/bowtie:/share/apps/tophat:/share/apps/stampy:/share/apps/bin
export PATH

TOPHAT=tophat
BWA=bwa

# input variables:
#	read_alignment_method
check_if_specified_read_alignment_method_is_valid () {
	if [ ${read_alignment_method} == ${TOPHAT} ]; then
		## do nothing
		:
	elif [ ${read_alignment_method} == ${BWA} ]; then
		## do nothing
		:
	else
		echo Specified read_alignment_method \'${read_alignment_method}\' is not implemented
		exit
	fi
}

create_directories_in_directory_list () {
	for directory in ${directory_list[*]}; do
		if [ ! -e $directory ]; then
			mkdir $directory
		fi
	done
}


# input variables:
#	reads_file
#	sam_hits_file
#	tophat_output_directory
#	gene_model_annotations_GFF3_file
#	bowtie_index_basename
run_tophat_on_reads_file_to_generate_sam_hits_file () {
	gzipped_sam_hits_file=${sam_hits_file}.gz
	sam_file_generated_by_tophat=${tophat_output_directory}/accepted_hits.sam
	bam_file_generated_by_tophat=${tophat_output_directory}/accepted_hits.bam
	## if sam hits file does not exist
	if [ ! -e ${sam_hits_file} ]; then
		## but gzipped sam hits file exists
		if [ -e ${gzipped_sam_hits_file} ]; then
			## then unzip it to get sam hits file
			gunzip ${gzipped_sam_hits_file}
		else
			## generate sam hits file
			run_tophat_on_reads_file
			if [ ! -e $sam_file_generated_by_tophat ]; then
				if [ -e $bam_file_generated_by_tophat ]; then
					## convert bam file to sam file
					samtools view -h $bam_file_generated_by_tophat > $sam_file_generated_by_tophat 
					
					## remove bam file
					rm ${bam_file_generated_by_tophat}
				else
					echo Error: Expected tophat to generate either a sam or a bam file
					exit
				fi
			fi
			mv $sam_file_generated_by_tophat ${sam_hits_file}
		fi
	fi
}

# input variables:
#	reads_file
#	bam_hits_file
#	tophat_output_directory
#	gene_model_annotations_GFF3_file
#	bowtie_index_basename
run_tophat_on_reads_file_to_generate_bam_hits_file () {
	gzipped_bam_hits_file=${bam_hits_file}.gz
	bam_file_generated_by_tophat=${tophat_output_directory}/accepted_hits.bam
	sam_file_generated_by_tophat=${tophat_output_directory}/accepted_hits.sam
	## if bam hits file does not exist
	if [ ! -e ${bam_hits_file} ]; then
		## but gzipped bam hits file exists
		if [ -e ${gzipped_bam_hits_file} ]; then
			## then unzip it to get bam hits file
			gunzip ${gzipped_bam_hits_file}
		else
			## generate bam hits file
			run_tophat_on_reads_file
			if [ ! -e $bam_file_generated_by_tophat ]; then
				if [ -e $sam_file_generated_by_tophat ]; then
					## if reference file does not exist, then generate it
					ref_file=bowtie_indices/${bowtie_index_basename}.ebwt/${bowtie_index_basename}.fa
					if [ ! -e ${ref_file} ]; then
						pushd bowtie_indices/${bowtie_index_basename}.ebwt
						bowtie-inspect ${bowtie_index_basename} > ../${ref_file}
						popd
					fi

					## convert sam file to bam file
					samtools faidx ${ref_file}
					samtools view -bt ${ref_file}.fai -o ${bam_file_generated_by_tophat} ${sam_file_generated_by_tophat}

					## remove sam file
					rm ${sam_file_generated_by_tophat}
				else
					echo Error: Expected tophat to generate either a bam or a sam file
					exit
				fi
			fi
			mv $bam_file_generated_by_tophat ${bam_hits_file}
		fi
	fi
}

run_tophat_on_reads_file () {
	tophat -o $tophat_output_directory -G $gene_model_annotations_GFF3_file $bowtie_index_basename ${reads_file}
}

run_bwa_on_reads_file () {
	if [ ! -e ${genomic_reference_fasta_file}.bwt ]; then
		bwa index -a bwtsw ${genomic_reference_fasta_file}
	fi
	bwa aln ${genomic_reference_fasta_file} ${reads_file} > aln_sa.sai
	bwa samse ${genomic_reference_fasta_file} aln_sa.sai ${reads_file} > ${sam_file_generated_by_bwa}
}

run_stampy_on_reads_file () {
	if [ ! -e ${bowtie_index_basename}.stidx ]; then
		stampy.py -G $bowtie_index_basename ${genomic_reference_fasta_file}
	fi
	if [ ! -e ${bowtie_index_basename}.sthash ]; then
		stampy.py -g $bowtie_index_basename -H $bowtie_index_basename
	fi
	if [ ! -e ${genomic_reference_fasta_file}.bwt ]; then
		bwa index -a bwtsw ${genomic_reference_fasta_file}
	fi
	stampy.py --bwaoptions="-q10 ${genomic_reference_fasta_file}" -g $bowtie_index_basename -h $bowtie_index_basename -M ${reads_file} -f sam -o ${sam_file_generated_by_stampy}
}

# input variables:
#	reads_file
#	sam_hits_file
#	bwa_output_directory
#	genomic_reference_fasta_file
run_bwa_on_reads_file_to_generate_sam_hits_file () {
	gzipped_sam_hits_file=${sam_hits_file}.gz
	sam_file_generated_by_bwa=${bwa_output_directory}/accepted_hits.sam
	bam_file_generated_by_bwa=${bwa_output_directory}/accepted_hits.bam
	## if sam hits file does not exist
	if [ ! -e ${sam_hits_file} ]; then
		## but gzipped sam hits file exists
		if [ -e ${gzipped_sam_hits_file} ]; then
			## then unzip it to get sam hits file
			gunzip ${gzipped_sam_hits_file}
		else
			## generate sam hits file
			run_bwa_on_reads_file
			if [ ! -e $sam_file_generated_by_bwa ]; then
				if [ -e $bam_file_generated_by_bwa ]; then
					## convert bam file to sam file
					samtools view -h $bam_file_generated_by_bwa > $sam_file_generated_by_bwa 
					
					## remove bam file
					rm ${bam_file_generated_by_bwa}
				else
					echo Error: Expected bwa to generate either a sam or a bam file
					exit
				fi
			fi
			mv $sam_file_generated_by_bwa ${sam_hits_file}
		fi
	fi
}

# input variables:
#	reads_file
#	sam_hits_file
#	stampy_output_directory
#	bowtie_index_basename
#	genomic_reference_fasta_file
run_stampy_on_reads_file_to_generate_sam_hits_file () {
	echo RETURN here
	gzipped_sam_hits_file=${sam_hits_file}.gz
	sam_file_generated_by_stampy=${stampy_output_directory}/accepted_hits.sam
	bam_file_generated_by_stampy=${stampy_output_directory}/accepted_hits.bam
	## if sam hits file does not exist
	if [ ! -e ${sam_hits_file} ]; then
		## but gzipped sam hits file exists
		if [ -e ${gzipped_sam_hits_file} ]; then
			## then unzip it to get sam hits file
			gunzip ${gzipped_sam_hits_file}
		else
			## generate sam hits file
			run_stampy_on_reads_file
			if [ ! -e $sam_file_generated_by_stampy ]; then
				if [ -e $bam_file_generated_by_stampy ]; then
					## convert bam file to sam file
					samtools view -h $bam_file_generated_by_stampy > $sam_file_generated_by_stampy 
					
					## remove bam file
					rm ${bam_file_generated_by_stampy}
				else
					echo Error: Expected stampy to generate either a sam or a bam file
					exit
				fi
			fi
			mv $sam_file_generated_by_stampy ${sam_hits_file}
		fi
	fi
}
