/*
**++
**  FACILITY:  University of California, Davis
**              Computer Science Department
**              Theory Lab
**
**  PRGRAM NAME:        XPARAL
**
**  MODULE DESCRIPTION:
**
**      This module contains all of the code for using optimal
**      alignment information to find regions of common
**      optimality (which are convex polygons).
**
*/
#include "header.h"
#include "globals_s.h"

extern void     draw_dark_poly(Poly*);
extern Funct    *find_max(Point*, int);
extern Poly     *get_poly_for(Point*);


/*********************************************************************/
/*************************  Debugging tools  *************************/
/*********************************************************************/

void    print_fun(      Funct   *fun)
{
    fprintf(log_fp, "Funct: match=%g mis=%g inordel=%g",
           fun->match_val, fun->mis_val, fun->inordel_val);
    if  (opt_function >= 4)
        fprintf(log_fp, " gaps=%g", fun->gaps_val);
    fprintf(log_fp, "\n");
}


static
void    print_pt(       Point   *pt)
{
    fprintf(log_fp, "Point: x=%lf y=%lf v=%lf w=%lf\n",
           pt->x, pt->y, pt->v, pt->w);
}


static
void    print_line(     Line    *line)
{
    if  (line->m_is_inf)
        fprintf(log_fp, "Line: m=Infinity  b=%lf \n", line->b * Global_v);
    else
        fprintf(log_fp, "Line: m=%lf  b=%lf \n", line->m, line->b * Global_v);
}


/********************************************************************/
/*                Intersection of two planes                       **/
/* Many different routines for each different geometry             **/
/********************************************************************/

#define MA1 (fun1->match_val)
#define MI1 (fun1->mis_val)
#define ID1 (fun1->inordel_val)
#define GA1 (fun1->gaps_val)
#define MA2 (fun2->match_val)
#define MI2 (fun2->mis_val)
#define ID2 (fun2->inordel_val)
#define GA2 (fun2->gaps_val)

inline
Line    *intersect_planes(      Funct   *fun1,
                                Funct   *fun2)
{
    double      m_denom;
    Line        *new_line;

/* **   fprintf(log_fp, "intersect_planes: begin\n");
/* */
    new_line = new Line();
    switch (opt_function)
    {
        case 1: if  ((m_denom = ID2-ID1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = Global_v * (MA2-MA1) / (MI1-MI2);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m = (MI1-MI2) / m_denom;
                    new_line->b = Global_v * (MA1-MA2) / m_denom;
                }
                break;
        case 2: if  ((m_denom = ID2-ID1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = Global_v * (MI2-MI1) / (MA1-MA2);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m = (MA1-MA2) / m_denom;
                    new_line->b = Global_v * (MI1-MI2) / m_denom;
                }
                break;
        case 3: if  ((m_denom = MI2-MI1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = Global_v * (ID2-ID1) / (MA1-MA2);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m = (MA1-MA2) / m_denom;
                    new_line->b = Global_v * (ID1-ID2) / m_denom;
                }
                break;
        case 4: if  ((m_denom = GA2-GA1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = (Global_v * (MA1-MA2)
                                 + Global_w * (MI1-MI2)) / (ID2-ID1);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m =  (ID1-ID2) / m_denom;
                    new_line->b = (Global_v * (MA1-MA2)
                                 + Global_w * (MI1-MI2)) / m_denom;
                }
                break;
        case 5: if  ((m_denom = GA2-GA1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = (Global_v * (MA1-MA2) 
                                 + Global_w * (ID1-ID2)) / (MI2-MI1);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m = (MI1-MI2) / m_denom;
                    new_line->b = (Global_v * (MA1-MA2)
                                 + Global_w * (ID1-ID2)) / m_denom;
                }
                break;
        case 6: if  ((m_denom = ID2-ID1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = (Global_v * (MA1-MA2)
                                 + Global_w * (GA1-GA2)) / (MI2-MI1);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m = (MI1-MI2) / m_denom;
                    new_line->b = (Global_v * (MA1-MA2)
                                 + Global_w * (GA1-GA2)) / m_denom;
                }
                break;
        case 7: if  ((m_denom = GA2-GA1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = (Global_v * (MI1-MI2)
                                 + Global_w * (ID1-ID2)) / (MA2-MA1);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m = (MA1-MA2) / m_denom;
                    new_line->b = (Global_v * (MI1-MI2)
                                 + Global_w * (ID1-ID2)) / m_denom;
                }
                break;
        case 8: if  ((m_denom = ID2-ID1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = (Global_v * (MI1-MI2)
                                 + Global_w * (GA1-GA2)) / (MA2-MA1);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m = (MA1-MA2) / m_denom;
                    new_line->b = (Global_v * (MI1-MI2)
                                 + Global_w * (GA1-GA2)) / m_denom;
                }
                break;
        case 9: if  ((m_denom = MI2-MI1) == 0.0)
                {
                    new_line->m_is_inf = 1;
                    new_line->m = INFINITY;
                    new_line->b = (Global_v * (ID1-ID2)
                                 + Global_w * (GA1-GA2)) / (MA2-MA1);
                }
                else
                {
                    new_line->m_is_inf = 0;
                    new_line->m = (MA1-MA2) / m_denom;
                    new_line->b = (Global_v * (ID1-ID2)
                                 + Global_w * (GA1-GA2)) / m_denom;
                }
                break;
    }
/* **   fprintf(log_fp, "Intersect planes gives "); print_line(new_line);
/* */
/* **   fprintf(log_fp, "intersect_planes: end\n");
/* */
    return new_line;
}

#undef MA1
#undef MI1
#undef ID1
#undef GA1
#undef MA2
#undef MI2
#undef ID2
#undef GA2


/********************************************************************/
/*  Checks the lists of functions (poly_list, todo_list, fun_list)  */
/*  to see if the passed function is in the lists.                  */
/*  If it is not there, then it adds it to fun_list.                */
/*  If it is there, then it frees the memory for the new function.  */
/*  and sets the passed pointer to the function that is already     */
/*  there.                                                          */
/********************************************************************/

void    check_add_fun(  Funct   **new_fun)
{
    Poly        *curr_poly;
    Todo        *curr_todo;
    Funct_list  *curr_fun,
                *prev_fun;

/* **   fprintf(log_fp, "check_add_fun: begin\n");
/* */
/*
**  First look for new function in the function lists.
*/
    curr_poly = poly_list;
    while  (curr_poly != NULL)
    {
        if  (curr_poly->fun->equal(*new_fun, opt_function))
        {
/* **       fprintf(log_fp, "delete call 001\n");
/* */
            delete *new_fun;
            *new_fun = curr_poly->fun;
            return;
        }
        else
            curr_poly = curr_poly->next;
    }
/*
**  There is no polygon for the function
*/
    curr_todo = todo_list;
    while  (curr_todo != NULL)
    {
        if  (curr_todo->fun->equal(*new_fun, opt_function))
        {
/* **       fprintf(log_fp, "delete call 002\n");
/* */
            delete *new_fun;
            *new_fun = curr_todo->fun;
            return;
        }
        else
            curr_todo = curr_todo->next;
    }
/*
**  There is no todo for the function
*/
    prev_fun = NULL;
    curr_fun = fun_list;
    while  (curr_fun != NULL)
    {
        if  (curr_fun->fun->equal(*new_fun, opt_function))
        {
/* **       fprintf(log_fp, "delete call 003\n");
/* */
            delete *new_fun;
            *new_fun = curr_fun->fun;
            return;
        }
        else
        {
            prev_fun = curr_fun;
            curr_fun = curr_fun->next;
        }
    }
/*
**  There is no fun_list entry for the function --
**  Add it to the list
*/
    curr_fun = new Funct_list;
    curr_fun->fun = *new_fun;
    curr_fun->next = NULL;
    if  (fun_list == NULL)
        fun_list = curr_fun;
    else
        prev_fun->next = curr_fun;
/* **   fprintf(log_fp, "check_add_fun: end\n");
/* */
}


/********************************************************************/
/*  Checks for the passed function in fun_list.                     */
/*  If it finds the function, then it removes it.                   */
/********************************************************************/

void    check_remove_fun(       const Funct     *remove_fun)
{
    Funct_list  *curr_fun,
                *prev_fun;
/*
**    Check the fun_list for the passed function.
**    If the function is there, then remove it.
*/
    prev_fun = NULL;
    curr_fun = fun_list;
    while  (curr_fun != NULL)
    {
        if  (curr_fun->fun->equal(remove_fun, opt_function))
        {
            if  (prev_fun == NULL)
                fun_list = fun_list->next;
            else
                prev_fun->next = curr_fun->next;
            if  (curr_fun->fun == remove_fun)
                curr_fun->fun = NULL;
/* **       fprintf(log_fp, "delete call 005\n");
/* */
            delete curr_fun;
	    curr_fun = NULL;
            return;
        }
        else
        {
            prev_fun = curr_fun;
            curr_fun = curr_fun->next;
        }
    }
    return;
}


/********************************************************************/
/*  Adds the passed todo information to the todo_list               */
/********************************************************************/
inline
static void     add_todo(       Todo    *new_todo)
{
    Todo        *curr_todo,
                *prev_todo;

/* **   fprintf(log_fp, "add_todo: begin\n");
/* */
/*
**  Find where to put the function in the list
**  (maintain the list in order of increasing distance).
*/
    if  (todo_list == NULL)
        todo_list = new_todo;
    else if  (   (new_todo->diff_from_input < todo_list->diff_from_input)
              || (   (new_todo->diff_from_input == todo_list->diff_from_input)
                  && (new_todo->distance_sq < todo_list->distance_sq))
              || (   (new_todo->diff_from_input == todo_list->diff_from_input)
                  && (new_todo->distance_sq == todo_list->distance_sq)
                  && (new_todo->start->y <= todo_list->start->y)))
    {
        new_todo->next = todo_list;
        todo_list = new_todo;
    }
    else
    {
        prev_todo = todo_list;
        curr_todo = todo_list->next;
        while  (curr_todo != NULL)
        {
            if  (   (new_todo->diff_from_input < curr_todo->diff_from_input)
                 || (   (new_todo->diff_from_input == curr_todo->diff_from_input)
                     && (new_todo->distance_sq < curr_todo->distance_sq))
                 || (   (new_todo->diff_from_input == curr_todo->diff_from_input)
                     && (new_todo->distance_sq == curr_todo->distance_sq)
                     && (new_todo->start->y <= curr_todo->start->y)))
                break;
            else
            {
                prev_todo = curr_todo;
                curr_todo = curr_todo->next;
            }
        }
        new_todo->next = curr_todo;
        prev_todo->next = new_todo;
    }
/* **   fprintf(log_fp, "add_todo: end\n");
/* */
}


/********************************************************************/
/*  Checks the lists of functions (poly_list, todo_list, fun_list)  */
/*  to see if the function for the passed todo information is in    */
/*  the lists.                                                      */
/*  If it is in the poly_list,                                      */
/*  then it frees the memory for the passed structure.              */
/*  If it is in the todo_list,                                      */
/*  then it keeps the one with the corner point with the smallest   */
/*      distance value (closest to the lower left corner            */
/*      (minval_x, minval_y)).                                      */
/*  Otherwise it adds it to the list.                               */
/*  If it is in the fun_list,                                       */
/*  then it removes it.                                             */
/********************************************************************/
inline
static void     check_add_todo( Todo    *new_todo)
{
    Poly        *curr_poly;
    Todo        *curr_todo,
                *prev_todo;
    Funct_list  *curr_fun,
                *prev_fun;

/* **   fprintf(log_fp, "check_add_todo: begin\n");
/* */
/*
**  First look for new function in the poly_list.
*/
    curr_poly = poly_list;
    while  (curr_poly != NULL)
    {
        if  (curr_poly->fun->equal(new_todo->fun, opt_function))
        {
            if  (curr_poly->fun == new_todo->fun)
                new_todo->fun = NULL;
/* **       fprintf(log_fp, "delete call 009\n");
/* */
            delete new_todo;
	    new_todo = NULL;
            return;
        }
        else
            curr_poly = curr_poly->next;
    }
/*
**  The function is not in the poly_list.
*/
    prev_todo = NULL;
    curr_todo = todo_list;
    while  (curr_todo != NULL)
    {
        if  (curr_todo->fun->equal(new_todo->fun, opt_function))
        {
/*
**          Found the function
*/
            if  (   (curr_todo->diff_from_input < new_todo->diff_from_input)
                 || (   (curr_todo->diff_from_input == new_todo->diff_from_input)
                     && (curr_todo->distance_sq < new_todo->distance_sq))
                 || (   (curr_todo->diff_from_input == new_todo->diff_from_input)
                     && (curr_todo->distance_sq == new_todo->distance_sq)
                     && (curr_todo->start->y <= new_todo->start->y)))
            {
/*
**              Keep the old todo information
*/
                if  (new_todo->fun == curr_todo->fun)
                    new_todo->fun = NULL;
/* **           fprintf(log_fp, "delete call 011\n");
/* */
                delete new_todo;
		new_todo = NULL;
            }
            else
            {
/*
**              Keep the new todo information
*/
                if  (prev_todo == NULL)
                    todo_list = curr_todo->next;
                else
                    prev_todo->next = curr_todo->next;
                if  (new_todo->fun != curr_todo->fun)
                {
/* **               fprintf(log_fp, "delete call 014\n");
/* */
                    delete new_todo->fun;
                    new_todo->fun = curr_todo->fun;
                }
                curr_todo->fun = NULL;
                delete curr_todo;
		curr_todo = NULL;
                add_todo(new_todo);
            }
            return;
        }
        else
        {
            prev_todo = curr_todo;
            curr_todo = curr_todo->next;
        }
    }
/*
**  The new function is not in the todo list --
**  Add the new todo information to the todo list
*/
    add_todo(new_todo);
    check_remove_fun(new_todo->fun);
/* **   fprintf(log_fp, "check_add_todo: end\n");
/* */
}


/********************************************************************/
/*  This function receives the information for finding the next     */
/*  corner and calculates the point on edgeline1 that is on the     */
/*  edge of the partial plane being explored in the appropriate     */
/*  direction.                                                      */
/*  It updates find_info so that corner2 points to the point and    */
/*  edgeline2 points to the appropriate edge line of the planar     */
/*  section.                                                        */
/********************************************************************/
inline
static void     find_extreme_pt(        Find_info       *find_info)
{

/* **   fprintf(log_fp, "find_extreme_pt: begin\n");
/* */
/*
**  Create a point structure, and initialize it to the furthest
**  possible point along the edgeline1 (at the quadrant section edge).
*/
    find_info->corner2 = new Point(Global_v, Global_w);
    if  (find_info->edgeline1->m_is_inf)
    {
        find_info->corner2->x = find_info->corner1->x;
        if  (find_info->incr1)
        {
            find_info->corner2->y = maxval_y;
            find_info->edgeline2 = new Line(0, 0.0, maxval_y);
            if  (find_info->above1)
            {
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else
            {
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
        }
        else /* =>  (find_info->incr1) */
        {
            find_info->corner2->y = minval_y;
            find_info->edgeline2 = new Line(0, 0.0, minval_y);
            if  (find_info->above1)
            {
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else
            {
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
        }
    }
    else
    {
        if  (find_info->incr1)
        {
            find_info->corner2->x = maxval_x;
            find_info->corner2->y = maxval_x * find_info->edgeline1->m +
                                find_info->edgeline1->b;
            if  (find_info->corner2->y == minval_y)
            {
                if  (find_info->above1)
                {
                    find_info->edgeline2 = new Line(1, INFINITY, maxval_x);
                    find_info->above2 = 0;
                    find_info->incr2 = 1;
                }
                else
                {
                    find_info->edgeline2 = new Line(0, 0.0, minval_y);
                    find_info->above2 = 1;
                    find_info->incr2 = 0;
                }
            }
            else if  (find_info->corner2->y == maxval_y)
            {
                if  (find_info->above1)
                {
                    find_info->edgeline2 = new Line(0, 0.0, maxval_y);
                    find_info->above2 = 0;
                    find_info->incr2 = 0;
                }
                else
                {
                    find_info->edgeline2 = new Line(1, INFINITY, maxval_x);
                    find_info->above2 = 0;
                    find_info->incr2 = 0;
                }
            }
            else /* =>  (   (find_info->corner2->y != maxval_y)   */
                 /*      && (find_info->corner2->y != minval_y))  */
                 /* =>  treat as though                           */
                 /*     (   (find_info->corner2->y < maxval_y)    */
                 /*      && (find_info->corner2->y > minval_y))   */
                 /*     (Will fix below if outside this range).   */
            {
                find_info->edgeline2 = new Line(1, INFINITY, maxval_x);
                if  (find_info->above1)
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 1;
                }
                else
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 0;
                }
            }
        }
        else /* =>  (!find_info->incr1) */
        {
            find_info->corner2->x = minval_x;
            find_info->corner2->y = minval_x * find_info->edgeline1->m +
                                find_info->edgeline1->b;
            if  (find_info->corner2->y == minval_y)
            {
                if  (find_info->above1)
                {
                    find_info->edgeline2 = new Line(1, INFINITY, minval_x);
                    find_info->above2 = 1;
                    find_info->incr2 = 1;
                }
                else
                {
                    find_info->edgeline2 = new Line(0, 0.0, minval_y);
                    find_info->above2 = 1;
                    find_info->incr2 = 1;
                }
            }
            else if  (find_info->corner2->y == maxval_y)
            {
                if  (find_info->above1)
                {
                    find_info->edgeline2 = new Line(0, 0.0, maxval_x);
                    find_info->above2 = 0;
                    find_info->incr2 = 1;
                }
                else
                {
                    find_info->edgeline2 = new Line(1, INFINITY, minval_x);
                    find_info->above2 = 1;
                    find_info->incr2 = 0;
                }
            }
            else /* =>  (   (find_info->corner2->y != maxval_y)   */
                 /*      && (find_info->corner2->y != minval_y))  */
                 /* =>  treat as though                           */
                 /*     (   (find_info->corner2->y < maxval_y)    */
                 /*      && (find_info->corner2->y > minval_y))   */
                 /*     (Will fix below if outside this range).   */
            {
                find_info->edgeline2 = new Line(1, INFINITY, minval_x);
                if  (find_info->above1)
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 1;
                }
                else
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 0;
                }
            }
        }
/*
**      Now fix the ones where y was out of range
**      (I.e., < minval_y or > maxval_y).
*/
        if  (find_info->corner2->y > maxval_y)
        {
            find_info->corner2->x = (maxval_y - find_info->edgeline1->b)/
                                        find_info->edgeline1->m;
            find_info->corner2->y = maxval_y;
            find_info->edgeline2->m_is_inf = 0;
            find_info->edgeline2->m = 0.0;
            find_info->edgeline2->b = maxval_y;
            if  (find_info->incr1)
            {
                if  (find_info->above1)
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 0;
                }
                else
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 1;
                }
            }
            else /* =>  (!find_info->incr1) */
            {
                if  (find_info->above1)
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 1;
                }
                else
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 0;
                }
            }
        }
        else if  (find_info->corner2->y < minval_y)
        {
            find_info->corner2->x = (minval_y - find_info->edgeline1->b)/
                                        find_info->edgeline1->m;
            find_info->corner2->y = minval_y;
            find_info->edgeline2->m_is_inf = 0;
            find_info->edgeline2->m = 0.0;
            find_info->edgeline2->b = minval_y;
            if  (find_info->incr1)
            {
                if  (find_info->above1)
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 1;
                }
                else
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 0;
                }
            }
            else /* =>  (!find_info->incr1) */
            {
                if  (find_info->above1)
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 0;
                }
                else
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 1;
                }
            }
        }
    }
/* **   fprintf(log_fp, "find_extreme_pt: end\n");
/* */
}


/********************************************************************/
/*  This function is used when a function gives the previous        */
/*  corner when looking for a new corner.                           */
/*  It checks the line generated by intersecting the function       */
/*  with the search function and accepts the new line if it is      */
/*  "inside" the space defined by edgeline0 and edgeline1.          */
/*  (Based on the values of above0 and incr0).                      */
/********************************************************************/
inline
static void     eval_corner1_line(      Find_info       *find_info,
                                        Point           *new_pt,
                                        Line            *new_line,
                                        Funct           *new_fun)
{
/* **   fprintf(log_fp, "eval_corner1_line: begin\n");
/* */
/*
**  Check if new_line is outside the polygon limits
**  as defined by edgeline0 and edgeline1.
*/
    if  (find_info->edgeline0->m_is_inf)
    {
        if  (new_line->m < find_info->edgeline1->m)
        {
            if  (   (find_info->incr0)
                 && (   (find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (new_line->m < find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 018\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 019\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else if  (   (!find_info->incr0)
                      && (   (!find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (new_line->m < find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 020\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 021\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m >                    */
                 /*                          find_info->edgeline2->m))))  */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal        */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m >                    */
                 /*                          find_info->edgeline2->m))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 022\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 023\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (new_line->m > find_info->edgeline1->m) */
        {
            if  (   (find_info->incr0)
                 && (   (!find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (new_line->m > find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 024\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 025\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr0)
                      && (   (find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (new_line->m > find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 026\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 027\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal        */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m <                    */
                 /*                          find_info->edgeline2->m))))  */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal        */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m <                    */
                 /*                          find_info->edgeline2->m))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 028\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 029\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
    else if  (find_info->edgeline1->m_is_inf)
    {
        if  (new_line->m < find_info->edgeline0->m)
        {
            if  (   (find_info->incr0)
                 && (   (find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (new_line->m > find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 030\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 031\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr0)
                      && (   (!find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (new_line->m > find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 032\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 033\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal        */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m <                    */
                 /*                          find_info->edgeline2->m))))  */
                 /*                          find_info->edgeline2->m))))) */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m <                    */
                 /*                          find_info->edgeline2->m))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 034\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 035\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (new_line->m > find_info->edgeline0->m) */
        {
            if  (   (find_info->incr0)
                 && (   (!find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (new_line->m < find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 036\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 037\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr0)
                      && (   (find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (new_line->m < find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 038\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 039\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m >                    */
                 /*                          find_info->edgeline2->m))))  */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m >                    */
                 /*                          find_info->edgeline2->m))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 040\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 041\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
    else if  (new_line->m_is_inf)
    {
        if  (find_info->edgeline0->m < find_info->edgeline1->m)
        {
            if  (   (find_info->incr0)
                 && (   (find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (find_info->edgeline2->m > 0.0))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 042\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 043\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else if  (   (!find_info->incr0)
                      && (   (!find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (find_info->edgeline2->m > 0.0))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 044\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 045\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (find_info->edgeline2->m <        */
                 /*                          0.0))))                      */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (find_info->edgeline2->m <        */
                 /*                          0.0)))))                     */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 046\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 047\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (find_info->edgeline0->m > find_info->edgeline1->m) */
        {
            if  (   (find_info->incr0)
                 && (   (!find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (find_info->edgeline2->m < 0.0))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 048\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 049\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr0)
                      && (   (find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (find_info->edgeline2->m < 0.0))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 050\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 051\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (find_info->edgeline2->m >        */
                 /*                          0.0))))                      */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (find_info->edgeline2->m >        */
                 /*                          0.0)))))                     */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 052\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 053\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
/*
**  None of edgeline0, edgeline1, or newline is vertical.
*/
    else if  (find_info->edgeline0->m < find_info->edgeline1->m)
    {
        if  (   (new_line->m > find_info->edgeline0->m)
             && (new_line->m < find_info->edgeline1->m))
        {
            if  (   (find_info->incr0)
                 && (   (!find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (new_line->m < find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 054\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 055\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr0)
                      && (   (find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (new_line->m < find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 056\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 057\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m >                    */
                 /*                          find_info->edgeline2->m))))  */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m >                    */
                 /*                          find_info->edgeline2->m))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 058\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 059\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else if  (new_line->m < find_info->edgeline0->m)
        {
            if  (   (find_info->incr0)
                 && (   (find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (new_line->m > find_info->edgeline2->m)
                         || (find_info->edgeline2->m >
                                 find_info->edgeline1->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 060\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 061\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr0)
                      && (   (!find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (new_line->m > find_info->edgeline2->m)
                              || (find_info->edgeline2->m >
                                      find_info->edgeline1->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 062\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 063\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m <                    */
                 /*                          find_info->edgeline2->m)     */
                 /*                  && (find_info->edgeline2->m <        */
                 /*                          find_info->edgeline0->m))))  */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m <                    */
                 /*                          find_info->edgeline2->m)     */
                 /*                  && (find_info->edgeline2->m <        */
                 /*                          find_info->edgeline0->m))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 064\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 065\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (new_line->m > find_info->edgeline1->m)  */
        {
            if  (   (find_info->incr0)
                 && (   (find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (   (new_line->m > find_info->edgeline2->m)
                             && (find_info->edgeline2->m >
                                     find_info->edgeline1->m)))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 066\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 067\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else if  (   (!find_info->incr0)
                      && (   (!find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (   (new_line->m > find_info->edgeline2->m)
                                  && (find_info->edgeline2->m >
                                          find_info->edgeline1->m)))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 068\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 069\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else /* =>  (   (   (find_info->incr0)                         */
                 /*          && (   (!find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(        */
                 /*                               find_info->corner2))     */
                 /*                  && (   (new_line->m <                 */
                 /*                          find_info->edgeline2->m)      */
                 /*                      || (find_info->edgeline2->m <     */
                 /*                          find_info->edgeline0->m)))))  */
                 /*      || (   (!find_info->incr0)                        */
                 /*          && (   (find_info->above0)                    */
                 /*              || (   (find_info->corner1->equal(        */
                 /*                               find_info->corner2))     */
                 /*                  && (   (new_line->m <                 */
                 /*                          find_info->edgeline2->m)      */
                 /*                      || (find_info->edgeline2->m <     */
                 /*                          find_info->edgeline0->m)))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 070\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 071\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
    else /* =>  (find_info->edgeline0->m > find_info->edgeline1->m) */
    {
        if  (   (new_line->m < find_info->edgeline0->m)
             && (new_line->m > find_info->edgeline1->m))
        {
            if  (   (find_info->incr0)
                 && (   (find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (new_line->m > find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 072\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 073\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr0)
                      && (   (!find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (new_line->m > find_info->edgeline2->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 074\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 075\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m <                    */
                 /*                          find_info->edgeline2->m))))  */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m <                    */
                 /*                          find_info->edgeline2->m))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 076\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 077\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else if  (new_line->m < find_info->edgeline1->m)
        {
            if  (   (find_info->incr0)
                 && (   (!find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (   (new_line->m < find_info->edgeline2->m)
                             && (find_info->edgeline2->m <
                                     find_info->edgeline1->m)))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 078\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 079\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else if  (   (!find_info->incr0)
                      && (   (find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (   (new_line->m < find_info->edgeline2->m)
                                  && (find_info->edgeline2->m <
                                          find_info->edgeline1->m)))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 080\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 081\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else /* =>  (   (   (find_info->incr0)                         */
                 /*          && (   (!find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(        */
                 /*                               find_info->corner2))     */
                 /*                  && (   (new_line->m >                 */
                 /*                          find_info->edgeline2->m)      */
                 /*                      || (find_info->edgeline2->m >     */
                 /*                          find_info->edgeline0->m)))))  */
                 /*      || (   (!find_info->incr0)                        */
                 /*          && (   (!find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(        */
                 /*                               find_info->corner2))     */
                 /*                  && (   (new_line->m >                 */
                 /*                          find_info->edgeline2->m)      */
                 /*                      || (find_info->edgeline2->m >     */
                 /*                          find_info->edgeline0->m)))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 082\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 083\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (new_line->m > find_info->edgeline0->m) */
        {
            if  (   (find_info->incr0)
                 && (   (!find_info->above0)
                     && (   (!find_info->corner1->equal(find_info->corner2))
                         || (new_line->m < find_info->edgeline2->m)
                         || (find_info->edgeline2->m <
                                     find_info->edgeline1->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 084\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 085\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr0)
                      && (   (find_info->above0)
                          && (   (!find_info->corner1->equal(find_info->corner2))
                              || (new_line->m < find_info->edgeline2->m)
                              || (find_info->edgeline2->m <
                                          find_info->edgeline1->m))))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 086\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 087\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr0)                        */
                 /*          && (   (find_info->above0)                   */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m >                    */
                 /*                          find_info->edgeline2->m)     */
                 /*                  && (find_info->edgeline2->m >        */
                 /*                          find_info->edgeline0->m))))  */
                 /*      || (   (!find_info->incr0)                       */
                 /*          && (   (!find_info->above0)                  */
                 /*              || (   (find_info->corner1->equal(       */
                 /*                               find_info->corner2))    */
                 /*                  && (new_line->m >                    */
                 /*                          find_info->edgeline2->m)     */
                 /*                  && (find_info->edgeline2->m >        */
                 /*                          find_info->edgeline0->m))))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 088\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 089\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
/* **   fprintf(log_fp, "eval_corner1_line: end\n");
/* */
}


/********************************************************************/
/*  This function is used when two functions give the same          */
/*  intersection point when looking for a new corner.               */
/*  It compares the lines generated by intersecting the functions   */
/*  with the search function and picks the one that is "closest"    */
/*  to the polygon being found.                                     */
/********************************************************************/
inline
static void     eval_corner2_lines(     Find_info       *find_info,
                                        Point           *new_pt,
                                        Line            *new_line,
                                        Funct           *new_fun)
{
/* **   fprintf(log_fp, "eval_corner2_line: begin\n");
/* */
/*
**  Check if new_line is outside the polygon limits
**  as defined by edgeline0 and edgeline2.
*/
    if  (find_info->edgeline1->m_is_inf)
    {
        if  (new_line->m < find_info->edgeline2->m)
        {
            if  (   (find_info->incr1)
                 && (find_info->above1))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 090\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 091\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else if  (   (!find_info->incr1)
                      && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 092\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 093\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else /* =>  (   (   (find_info->incr1)    */
                 /*          && (!find_info->above1)) */
                 /*      || (   (!find_info->incr1)   */
                 /*          && (find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 094\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 095\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (new_line->m > find_info->edgeline2->m) */
        {
            if  (   (find_info->incr1)
                 && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 096\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 097\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else  if  (   (!find_info->incr1)
                       && (find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 098\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 099\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr1)     */
                 /*          && (find_info->above1))   */
                 /*      || (   (!find_info->incr1)    */
                 /*          && (!find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 100\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 101\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
    else if  (find_info->edgeline2->m_is_inf)
    {
        if  (new_line->m < find_info->edgeline1->m)
        {
            if  (   (find_info->incr1)
                 && (find_info->above1))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 102\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 103\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr1)
                      && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 104\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 105\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr1)    */
                 /*          && (!find_info->above1)) */
                 /*      || (   (!find_info->incr1)   */
                 /*          && (find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 106\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 107\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (new_line->m > find_info->edgeline1->m) */
        {
            if  (   (find_info->incr1)
                 && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 108\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 109\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr1)
                      && (find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 110\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 111\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr1)     */
                 /*          && (find_info->above1))   */
                 /*      || (   (!find_info->incr1)    */
                 /*          && (!find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 112\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 113\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
    else if  (new_line->m_is_inf)
    {
        if  (find_info->edgeline1->m < find_info->edgeline2->m)
        {
            if  (   (find_info->incr1)
                 && (find_info->above1))
            {
/*
**              Use new_pt and new_line
*/
/* **           fprintf(log_fp, "delete call 114\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 115\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else if  (   (!find_info->incr1)
                      && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 116\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 117\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else /* =>  (   (   (find_info->incr1)    */
                 /*          && (!find_info->above1)) */
                 /*      || (   (!find_info->incr1)   */
                 /*          && (find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 118\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 119\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (find_info->edgeline1->m > find_info->edgeline2->m) */
        {
            if  (   (find_info->incr1)
                 && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 120\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 121\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr1)
                      && (find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 122\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 123\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr1)     */
                 /*          && (find_info->above1))   */
                 /*      || (   (!find_info->incr1)    */
                 /*          && (!find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 124\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 125\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
/*
**  None of edgeline1, edgeline2, or newline is vertical.
*/
    else if  (find_info->edgeline1->m < find_info->edgeline2->m)
    {
        if  (   (new_line->m > find_info->edgeline1->m)
             && (new_line->m < find_info->edgeline2->m))
        {
            if  (   (find_info->incr1)
                 && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 126\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 127\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr1)
                      && (find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 128\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 129\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr1)     */
                 /*          && (find_info->above1))   */
                 /*      || (   (!find_info->incr1)    */
                 /*          && (!find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 130\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 131\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (   (new_line->m < find_info->edgeline1->m)  */
             /*      || (new_line->m > find_info->edgeline2->m)) */
        {
            if  (   (find_info->incr1)
                 && (find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 132\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 133\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                if  (new_line->m < find_info->edgeline1->m)
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 0;
                }
                else /* =>  (new_line->m > find_info->edgeline2->m) */
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 1;
                }
            }
            else if  (   (!find_info->incr1)
                      && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 134\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 135\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 0;
                if  (new_line->m < find_info->edgeline1->m)
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 1;
                }
                else /* =>  (new_line->m > find_info->edgeline2->m) */
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 0;
                }
            }
            else /* =>  (   (   (find_info->incr1)    */
                 /*          && (!find_info->above1)) */
                 /*      || (   (!find_info->incr1)   */
                 /*          && (find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 136\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 137\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
    else /* =>  (find_info->edgeline1->m > find_info->edgeline2->m) */
    {
        if  (   (new_line->m < find_info->edgeline1->m)
             && (new_line->m > find_info->edgeline2->m))
        {
            if  (   (find_info->incr1)
                 && (find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 138\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 139\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 0;
                find_info->incr2 = 0;
            }
            else if  (   (!find_info->incr1)
                      && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 140\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 141\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                find_info->above2 = 1;
                find_info->incr2 = 1;
            }
            else /* =>  (   (   (find_info->incr1)    */
                 /*          && (!find_info->above1)) */
                 /*      || (   (!find_info->incr1)   */
                 /*          && (find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 142\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 143\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
        else /* =>  (   (new_line->m > find_info->edgeline1->m)  */
             /*      || (new_line->m < find_info->edgeline2->m)) */
        {
            if  (   (find_info->incr1)
                 && (!find_info->above1))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 144\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 145\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                if  (new_line->m > find_info->edgeline1->m)
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 0;
                }
                else /* =>  (new_line->m < find_info->edgeline2->m) */
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 1;
                }
            }
            else if  (   (!find_info->incr0)
                      && (find_info->above0))
            {
/*
**              Use new_pt and new_line.
*/
/* **           fprintf(log_fp, "delete call 146\n");
/* */
                delete find_info->corner2;
                find_info->corner2 = new_pt;
/* **           fprintf(log_fp, "delete call 147\n");
/* */
                delete find_info->edgeline2;
                find_info->edgeline2 = new_line;
                find_info->fun2 = new_fun;
                if  (new_line->m > find_info->edgeline1->m)
                {
                    find_info->above2 = 0;
                    find_info->incr2 = 1;
                }
                else /* =>  (new_line->m < find_info->edgeline2->m) */
                {
                    find_info->above2 = 1;
                    find_info->incr2 = 0;
                }
            }
            else /* =>  (   (   (find_info->incr1)    */
                 /*          && (find_info->above1)) */
                 /*      || (   (!find_info->incr1)   */
                 /*          && (!find_info->above1))) */
            {
/*
**              Don't use new_pt.
*/
/* **           fprintf(log_fp, "delete call 148\n");
/* */
                delete new_pt;
		new_pt = NULL;
/* **           fprintf(log_fp, "delete call 149\n");
/* */
                delete new_line;
		new_line = NULL;
            }
        }
    }
/* **   fprintf(log_fp, "eval_corner2_line: end\n");
/* */
}


/********************************************************************/
/*  This function intersects the function for the polygon being     */
/*  found with a given function to get a line, and intersects       */
/*  that line with edgeline1.                                       */
/*  If  the new point is between corner1 and corner2 then it        */
/*  discards corner2 and replaces it with new_pt.                   */
/*  If new_pt coincides with corner2, then it saves the function    */
/*  whose intersection line is "closest" to the polygon.            */
/********************************************************************/

static void     check_function( Find_info       *find_info,
                                Funct           *test_fun)
{
    Point       *new_pt;
    Line        *new_line;
    Funct       *new_fun;

/* **   fprintf(log_fp, "check_function: begin\n");
/* */
/*
**  Do nothing if test_fun is the same as one of the
**  previous functions.
*/
    if  (test_fun->equal(find_info->fun, opt_function))
    {
/* **   fprintf(log_fp, "check_function: end\n");
/* */
        return;
    }
    if  (   (find_info->fun1 != NULL)
         && (test_fun->equal(find_info->fun1, opt_function)))
    {
/* **   fprintf(log_fp, "check_function: end\n");
/* */
        return;
    }
    if  (   (find_info->fun2 != NULL)
         && (test_fun->equal(find_info->fun2, opt_function)))
    {
/* **   fprintf(log_fp, "check_function: end\n");
/* */
        return;
    }

/*
**  First intersect the functions to get a line,
**  make sure the lines are not the same, and
**  intersect the lines to get a point.
*/
    new_line = intersect_planes(find_info->fun, test_fun);
    if  (   (find_info->edgeline0 != NULL)
         && (new_line->equal(find_info->edgeline0)))
    {
/* **   fprintf(log_fp, "delete call 150\n");
/* */
        delete new_line;
	new_line = NULL;
/* **   fprintf(log_fp, "check_function: end\n");
/* */
        return;
    }
    if  (   (new_line->m_is_inf)
         && (find_info->edgeline1->m_is_inf))
    {
/* **   fprintf(log_fp, "delete call 151\n");
/* */
        delete new_line;
	new_line = NULL;
/* **   fprintf(log_fp, "check_function: end\n");
/* */
        return;
    }
    if  (   (!new_line->m_is_inf)
         && (!find_info->edgeline1->m_is_inf)
         && (equal(new_line->m, find_info->edgeline1->m)))
    {
/* **   fprintf(log_fp, "delete call 152\n");
/* */
        delete new_line;
	new_line = NULL;
/* **   fprintf(log_fp, "check_function: end\n");
/* */
        return;
    }
    new_pt = new Point(new_line, find_info->edgeline1, Global_v, Global_w);
    if  (new_pt->equal(find_info->corner1))
    {
/*
**      new_pt and corner1 coincide,
**      so decide what to do based on the lines.
**
**      edgeline0 could be null (if corner1 is on outside boundary).
**      if it is, then discard new_line and new_pt.
*/
        if  (find_info->edgeline0 == NULL)
        {
/* **       fprintf(log_fp, "delete call 153\n");
/* */
            delete new_line;
	    new_line = NULL;
/* **       fprintf(log_fp, "delete call 154\n");
/* */
            delete new_pt;
	    new_pt = NULL;
/* **       fprintf(log_fp, "check_function: end\n");
/* */
            return;
        }
        else
            eval_corner1_line(find_info, new_pt, new_line, test_fun);
    }
    else if  (new_pt->equal(find_info->corner2))
    {
/*
**      new_pt and corner2 coincide,
**      so pick the one to keep based on the lines.
*/
        eval_corner2_lines(find_info, new_pt, new_line, test_fun);
    }
    else if  (!find_info->corner1->equal(find_info->corner2))
    {
/*
**      Check if new_pt is in the correct direction.
**      If it is, then of new_pt and corner2
**      keep the one that is closer to corner1,
*/
        if  (find_info->edgeline1->m_is_inf)
        {
            if  (find_info->corner1->y < find_info->corner2->y)
            {
/*
**              (find_info->incr1)
*/
                if  (   (new_pt->y < find_info->corner2->y)
                     && (new_pt->y > find_info->corner1->y))
                {
/*
**                  new_pt between corner1 and corner2
**                  so save new_pt
*/
/* **               fprintf(log_fp, "delete call 155\n");
/* */
                    delete find_info->corner2;
                    find_info->corner2 = new_pt;
/* **               fprintf(log_fp, "delete call 156\n");
/* */
                    delete find_info->edgeline2;
                    find_info->edgeline2 = new_line;
                    find_info->fun2 = test_fun;
                    if  (find_info->above1)
                    {
                        find_info->above2 = 0;
                        find_info->incr2 = 1;
                    }
                    else /* =>  (!find_info->above) */
                    {
                        find_info->above2 = 0;
                        find_info->incr2 = 0;
                    }
                }
                else /* =>  (   (new_pt->y >                 */
                     /*              find_info->corner2->y)  */
                     /*      || (new_pt->y <                 */
                     /*              find_info->corner1->y)) */
                {
/*
**                  new_pt outside corner1 and corner2
**                  so discard new_pt
*/
/* **               fprintf(log_fp, "delete call 157\n");
/* */
                    delete new_pt;
		    new_pt = NULL;
/* **               fprintf(log_fp, "delete call 158\n");
/* */
                    delete new_line;
		    new_line = NULL;
                }
            }
            else /* =>  (find_info->corner1->y >    */
                 /*          find_info->corner2->y) */
            {
/*
**              (!find_info->incr1)
*/
                if  (   (new_pt->y < find_info->corner1->y)
                     && (new_pt->y > find_info->corner2->y))
                {
/*
**                  new_pt between corner1 and corner2
**                  so save new_pt
*/
/* **               fprintf(log_fp, "delete call 159\n");
/* */
                    delete find_info->corner2;
                    find_info->corner2 = new_pt;
/* **               fprintf(log_fp, "delete call 160\n");
/* */
                    delete find_info->edgeline2;
                    find_info->edgeline2 = new_line;
                    find_info->fun2 = test_fun;
                    if  (find_info->above1)
                    {
                        find_info->above2 = 1;
                        find_info->incr2 = 1;
                    }
                    else /* =>  (!find_info->above) */
                    {
                        find_info->above2 = 1;
                        find_info->incr2 = 0;
                    }
                }
                else /* =>  (   (new_pt->y >                 */
                     /*              find_info->corner1->y)  */
                     /*      || (new_pt->y <                 */
                     /*              find_info->corner2->y)) */
                {
/*
**                  new_pt outside corner1 and corner2
**                  so discard new_pt
*/
/* **               fprintf(log_fp, "delete call 161\n");
/* */
                    delete new_pt;
		    new_pt = NULL;
/* **               fprintf(log_fp, "delete call 162\n");
/* */
                    delete new_line;
		    new_line = NULL;
                }
            }
        }
        else /* =>  (!find_info->edgeline1->m_is_inf) */
        {
            if  (find_info->corner1->x < find_info->corner2->x)
            {
/*
**              (find_info->incr1)
*/
                if  (   (new_pt->x < find_info->corner2->x)
                     && (new_pt->x > find_info->corner1->x))
                {
/*
**                  new_pt between corner1 and corner2
**                  so save new_pt
*/
/* **               fprintf(log_fp, "delete call 163\n");
/* */
                    delete find_info->corner2;
                    find_info->corner2 = new_pt;
/* **               fprintf(log_fp, "delete call 164\n");
/* */
                    delete find_info->edgeline2;
                    find_info->edgeline2 = new_line;
                    find_info->fun2 = test_fun;
                    if  (new_line->m_is_inf)
                    {
                        if  (find_info->above1)
                        {
                            find_info->above2 = 0;
                            find_info->incr2 = 1;
                        }
                        else /* =>  (!find_info->above) */
                        {
                            find_info->above2 = 0;
                            find_info->incr2 = 0;
                        }
                    }
                    else if  (new_line->m < find_info->edgeline1->m)
                    {
                        if  (find_info->above1)
                        {
                            find_info->above2 = 0;
                            find_info->incr2 = 0;
                        }
                        else /* =>  (!find_info->above) */
                        {
                            find_info->above2 = 0;
                            find_info->incr2 = 1;
                        }
                    }
                    else /* =>  (new_line->m >                */
                         /*          find_info->edgeline->m)  */
                    {
                        if  (find_info->above1)
                        {
                            find_info->above2 = 1;
                            find_info->incr2 = 1;
                        }
                        else /* =>  (!find_info->above) */
                        {
                            find_info->above2 = 1;
                            find_info->incr2 = 0;
                        }
                    }
                }
                else /* =>  (   (new_pt->x >                 */
                     /*              find_info->corner2->x)  */
                     /*      || (new_pt->x <                 */
                     /*              find_info->corner1->x)) */
                {
/*
**                  new_pt outside corner1 and corner2
**                  so discard new_pt
*/
/* **               fprintf(log_fp, "delete call 165\n");
/* */
                    delete new_pt;
		    new_pt = NULL;
/* **               fprintf(log_fp, "delete call 166\n");
/* */
                    delete new_line;
		    new_line = NULL;
                }
            }
            else /* =>  (find_info->corner1->x >    */
                 /*          find_info->corner2->x) */
            {
/*
**              (!find_info->incr1)
*/
                if  (   (new_pt->x < find_info->corner1->x)
                     && (new_pt->x > find_info->corner2->x))
                {
/*
**                  new_pt between corner1 and corner2
**                  so save new_pt
*/
/* **               fprintf(log_fp, "delete call 167\n");
/* */
                    delete find_info->corner2;
                    find_info->corner2 = new_pt;
/* **               fprintf(log_fp, "delete call 168\n");
/* */
                    delete find_info->edgeline2;
                    find_info->edgeline2 = new_line;
                    find_info->fun2 = test_fun;
                    if  (new_line->m_is_inf)
                    {
                        if  (find_info->above1)
                        {
                            find_info->above2 = 1;
                            find_info->incr2 = 1;
                        }
                        else /* =>  (!find_info->above) */
                        {
                            find_info->above2 = 1;
                            find_info->incr2 = 0;
                        }
                    }
                    else if  (new_line->m < find_info->edgeline1->m)
                    {
                        if  (find_info->above1)
                        {
                            find_info->above2 = 1;
                            find_info->incr2 = 0;
                        }
                        else /* =>  (!find_info->above) */
                        {
                            find_info->above2 = 1;
                            find_info->incr2 = 1;
                        }
                    }
                    else /* =>  (new_line->m >                */
                         /*          find_info->edgeline->m)  */
                    {
                        if  (find_info->above1)
                        {
                                find_info->above2 = 0;
                                find_info->incr2 = 1;
                        }
                        else /* =>  (!find_info->above) */
                        {
                                find_info->above2 = 0;
                                find_info->incr2 = 0;
                        }
                    }
                }
                else /* =>  (   (new_pt->x >                 */
                     /*              find_info->corner1->x)  */
                     /*      || (new_pt->x <                 */
                     /*              find_info->corner2->x)) */
                {
/*
**                  new_pt outside corner1 and corner2
**                  so discard new_pt
*/
/* **               fprintf(log_fp, "delete call 169\n");
/* */
                    delete new_pt;
		    new_pt = NULL;
/* **               fprintf(log_fp, "delete call 170\n");
/* */
                    delete new_line;
		    new_line = NULL;
                }
            }
        }
    }
/* **   fprintf(log_fp, "check_function: end\n");
/* */
}


/********************************************************************/
/*  This function finds a first candidate point for the next        */
/*  corner by first finding the extreme point of edgeline1 on       */
/*  the partial plane, and then checking all the functions that     */
/*  have been identified so far to get the one that generates       */
/*  the closest point to the previous corner by intersecting        */
/*  the polygon function, the test function, and the edge line.     */
/********************************************************************/
inline
static void     find_candidate_pt(      Find_info       *find_info)
{
    Point       *new_pt;
    Line        *new_line;
    Poly        *curr_poly;
    Todo        *curr_todo;
    Funct_list  *curr_fun;

/* **   fprintf(log_fp, "find_candidate_pt: begin\n");
/* */
/*
**  First calculate the extreme point on the partial plane that is
**  on the edge line in the appropriate direction.
*/
    find_extreme_pt(find_info);
/*
**  Now for each function already found, check if it generates a
**  candidate point "closer" to the previous corner.
*/
    curr_poly = poly_list;
    while  (curr_poly != NULL)
    {
        check_function(find_info, curr_poly->fun);
        curr_poly = curr_poly->next;
    }

    curr_todo = todo_list;
    while  (curr_todo != NULL)
    {
        check_function(find_info, curr_todo->fun);
        curr_todo = curr_todo->next;
    }

    curr_fun = fun_list;
    while  (curr_fun != NULL)
    {
        if  (curr_fun->fun->is_2D_opt)
            check_function(find_info, curr_fun->fun);
        curr_fun = curr_fun->next;
    }

/* **   fprintf(log_fp, "find_candidate_pt: end\n");
/* */
}


/********************************************************************/
/*  This function receives the information for setting up to find   */
/*  the second corner and completes the set up so the standard      */
/*  routine for finding corners (find_next_corner) can be used.     */
/********************************************************************/

void    get_corner2_info(       Find_info       *find_info)
{
    Point       *new_pt1,
                *new_pt2;
    Line        *new_line1,
                *new_line2;
    Funct       *new_fun;

/* **   fprintf(log_fp, "get_corner2_info: begin\n");
/* */
    while  (find_info->corner2 == NULL)
    {
/*
**      First get the candidate for corner2
*/
        find_candidate_pt(find_info);
/*
**      Now find the optimal function for corner2 and evaluate it.
**      If it is not cooptimal with the function for the polygon, then
**      then intersect with the polygon function to get a new point
**      and try again.
*/
        for(;;)
        {
/* **       fprintf(log_fp, "find_max 1 in get_corner2_info\n");
/* */
            new_fun = find_max(find_info->corner2, 0);
            if  (new_fun->equal(find_info->fun, opt_function))
            {
/* **           fprintf(log_fp, "delete call 171\n");
/* */
                delete new_fun;
                new_fun = find_info->fun;
                break;
            }
            else if  (   (find_info->fun1 != NULL)
                      && (new_fun->equal(find_info->fun1, opt_function)))
            {
/* **           fprintf(log_fp, "delete call 172\n");
/* */
                delete new_fun;
                new_fun = find_info->fun1;
            }
            else
                check_add_fun(&new_fun);

            if  (equal(find_info->corner2->max_val,
                        find_info->corner2->eval(find_info->fun, opt_function)))
                break;
/*
**          Don't have a real corner yet -
**          have to intersect and try again.
**          Use check_function to do this to make sure
**          incr and above flags get set properly.
*/
            check_function(find_info, new_fun);
        }
/*
**      Have a real corner
**
**      If the new corner2 is the same as corner1,
**      then replace corner1 info with the corner2 info and start over.
*/
        if  (find_info->corner1->equal(find_info->corner2))
        {
/* **       fprintf(log_fp, "corners evaluated as equal:\n    ");
            print_pt(find_info->corner1);
            fprintf(log_fp, "    ");
            print_pt(find_info->corner2);
            fprintf(log_fp, "    x diff=%lf  y diff=%lf\n",
                   find_info->corner1->x - find_info->corner2->x,
                   find_info->corner1->y - find_info->corner2->y);
/* */
/* **       fprintf(log_fp, "delete call 173\n");
/* */
            delete find_info->corner1;
/* **       fprintf(log_fp, "delete call 174\n");
/* */
            delete find_info->edgeline1;
            find_info->corner1 = find_info->corner2;
            find_info->fun1 = find_info->fun2;
            find_info->edgeline1 = find_info->edgeline2;
            find_info->above1 = find_info->above2;
            find_info->incr1 = find_info->incr2;
            find_info->corner2 = NULL;
            find_info->fun2 = NULL;
            find_info->edgeline2 = NULL;
        }
    }
/* **   fprintf(log_fp, "get_corner2_info: end\n");
/* */
}


/********************************************************************/
/* This function uses the passed information to get the next corner */
/* for a polygon, and updates the find_info information to reflect  */
/* the correct values for the new corner.                           */
/* It also updates the function list and todo_list based on what it */
/* learns.                                                          */
/********************************************************************/

Point   *find_next_corner(      Find_info       *find_info,
                                int             build_todo)
{
    Funct       *new_fun;
    Point       *new_pt1,
                *new_pt2;
    Todo        *new_todo;

/* **   fprintf(log_fp, "find_next_corner: begin\n");
/* */
/*
**  Get rid of unneeded data and
**  copy the corner2 info to corner1, set corner2 to NULL.
*/
/* **   fprintf(log_fp, "delete call 175\n");
/* */
    delete find_info->edgeline0;

    find_info->edgeline0 = find_info->edgeline1;
    find_info->incr0 = find_info->incr1;
    find_info->above0 = find_info->above1;
    find_info->corner1 = find_info->corner2;
    find_info->fun1 = find_info->fun2;
    find_info->edgeline1 = find_info->edgeline2;
    find_info->incr1 = find_info->incr2;
    find_info->above1 = find_info->above2;
    find_info->corner2 = NULL;
    find_info->edgeline2 = NULL;
    find_info->fun2 = NULL;
/*
**  Get the new corner2 info.
*/
    get_corner2_info(find_info);
/*
**  The info for corner1 is now finalized,
**  because corner2 is distinct from corner1.
**  So, can now add corner1, fun1, etc. to todo_list.
*/
    if  (   (build_todo)
         && (find_info->fun1 != NULL))
        if  (!find_info->fun1->equal(find_info->fun))
        {
            new_todo = new Todo(find_info);

            check_add_todo(new_todo);
        }
/*
**  All that's left is to return the corner.
*/
/* **   fprintf(log_fp, "find_next_corner: end\n");
/* */
    return find_info->corner1;
}


/********************************************************************/
/* This routine finds the polygon for the poly_todo information     */
/* passed to it.  That information includes:                        */
/*     - the optimal function for the polygon                       */
/*     - the first corner point                                     */
/*     - the slope of the edge to traverse first                    */
/*     - whether the polygon is above or below the edge line        */
/*     - whether the edge leaves the point in increasing (in (x,y)) */
/*       or decreasing direction.                                   */
/* This routine builds a new polygon structure for the polygon and  */
/* returns it to the calling routine.                               */
/* It also updates the todo_list and the function list with new     */
/* information it learns.                                           */
/********************************************************************/

Poly    *find_poly(     Todo    *poly_todo)
{
    Funct       *new_fun;
    Find_info   *find_info;
    Point       *opt_pt,
                *curr_pt,
                *interior_pt,
                *last_corner,
                *prev_corner;
    Line        *curr_line,
                *new_line;
    Poly        *curr_poly,
                *new_poly,
                *prev_poly;
    Todo        *curr_todo;
    int         num_corners_min_diff;

/* **   fprintf(log_fp, "find_poly: begin\n");
/* */
    find_info = new Find_info(poly_todo);

/* **   fprintf(log_fp, "delete call 176\n");
/* */
    poly_todo->fun = NULL;
    poly_todo->start = NULL;
    poly_todo->edgeline = NULL;
    delete poly_todo;
    poly_todo = NULL;

    new_poly = new Poly;
    new_poly->fun = find_info->fun;
    new_poly->corners = find_next_corner(find_info, 1);
    new_poly->next = NULL;

    last_corner = new_poly->corners;
    if  (   (last_corner->x == minval_x)
         && (last_corner->y == minval_y))
        have_minval_poly = 1;
    do
    {
        last_corner->next = find_next_corner(find_info, 1);
        prev_corner = last_corner;
        last_corner = last_corner->next;
        number_corners++;
        if  (   (last_corner->x == minval_x)
             && (last_corner->y == minval_y))
            have_minval_poly = 1;
    } while  (!last_corner->equal(new_poly->corners));

/* **    if  (   (last_corner->x != new_poly->corners->x)
         || (last_corner->y != new_poly->corners->y)
         || (last_corner->v != new_poly->corners->v)
         || (last_corner->w != new_poly->corners->w))
    {
        fprintf(log_fp, "last corner != first\n");
    }
/* */

/*
**  We have the polygon,
**  So now we free all memory used for find_info.
*/
    prev_corner->next = NULL;
/* **   fprintf(log_fp, "delete call 177\n");
/* */
    delete last_corner;
    last_corner = NULL;

    find_info->fun = NULL;
    find_info->fun1 = NULL;
    find_info->fun2 = NULL;
// GDBROWN: set corner1 to NULL, to avoid duplicate freeing
    find_info->corner1 = NULL;
/* **   fprintf(log_fp, "delete call 182\n");
/* */
    delete find_info;
    find_info = NULL;

/*
**  Now we reorder the points so that the first is the one
**  with the smallest difference from the input function.
*/
    opt_pt = new_poly->corners;
    num_corners_min_diff = 1;

    curr_pt = opt_pt->next;
    while (curr_pt != NULL)
    {
        if  (curr_pt->diff_val < opt_pt->diff_val)
        {
            opt_pt = curr_pt;
            num_corners_min_diff = 1;
        }
        else if  (curr_pt->diff_val == opt_pt->diff_val)
        {
            num_corners_min_diff++;
            if  (   (curr_pt->dist_sq_val < opt_pt->dist_sq_val)
                 || (   (curr_pt->dist_sq_val == opt_pt->dist_sq_val)
                     && (curr_pt->y <= opt_pt->y)))
            {
                opt_pt = curr_pt;
            }
        }
        curr_pt = curr_pt->next;
    }
    if  (opt_pt != new_poly->corners)
    {
/*
**      The opt_pt is not the first one, so reorder the list.
**      Put opt_pt first,
**      then tack the front end of the list at the end.
**      Also need to break the chain so it doesn't circle back to opt_pt.
*/
        curr_pt = new_poly->corners;
        new_poly->corners = opt_pt;
        last_corner = opt_pt;
        while (last_corner->next != NULL)
            last_corner = last_corner->next;
        last_corner->next = curr_pt;
        while (curr_pt->next != opt_pt)
            curr_pt = curr_pt->next;
        curr_pt->next = NULL;
    }

    new_poly->diff_opt_pt = opt_pt->diff_val;
    new_poly->dist_sq_opt_pt = opt_pt->dist_sq_val;

    if  (input_fun == NULL)
        new_poly->min_diff_region_dim = 0;
    else  // (input_fun != NULL)
    {
        if  (num_corners_min_diff == 1)
            new_poly->min_diff_region_dim = 0;
        else if  (num_corners_min_diff == 2)
            new_poly->min_diff_region_dim = 1;
        else // ((num_corners_min_diff > 2)
            new_poly->min_diff_region_dim = 2;
    }

/*
**  Get function one last time to get string alignment.
**  To ensure that we get the correct optimal function,
**  we use an interior point obtained by averaging the
**  coordinates for the first three corners.
*/
    interior_pt = new Point(new_poly->corners);
    curr_pt = interior_pt->next;
    interior_pt->x += curr_pt->x;
    interior_pt->y += curr_pt->y;
    curr_pt = curr_pt->next;
    interior_pt->x += curr_pt->x;
    interior_pt->y += curr_pt->y;
    interior_pt->x /= 3.0;
    interior_pt->y /= 3.0;

/* **   fprintf(log_fp, "find_max 1 in find_poly\n");
/* */
    new_fun = find_max(interior_pt, 1);
/* **   fprintf(log_fp, "delete call 183\n");
/* */
    delete interior_pt;
    interior_pt = NULL;

    if  (!new_poly->fun->equal(new_fun, opt_function))
    {
        fprintf(log_fp, "Error getting function for polygon\n");
        fprintf(log_fp, "     polygon function is: %d %d %d %d\n",
                new_poly->fun->match_ct,
                new_poly->fun->mis_ct,
                new_poly->fun->inordel_ct,
                new_poly->fun->gaps_ct);
        fprintf(log_fp, "                          %g %g %g %g\n",
                new_poly->fun->match_val,
                new_poly->fun->mis_val,
                new_poly->fun->inordel_val,
                new_poly->fun->gaps_val);
        fprintf(log_fp, "     interior point is: x: %g  y: %g\n",
                interior_pt->x,
                interior_pt->y);
        fprintf(log_fp, "     found   function is: %d %d %d %d\n",
                new_fun->match_ct,
                new_fun->mis_ct,
                new_fun->inordel_ct,
                new_fun->gaps_ct);
        fprintf(log_fp, "                          %g %g %g %g\n",
                new_fun->match_val,
                new_fun->mis_val,
                new_fun->inordel_val,
                new_fun->gaps_val);
    }

/* **   fprintf(log_fp, "delete call 184\n");
/* */
    delete new_poly->fun;
    new_poly->fun = new_fun;

    number_polys++;
    fprintf(log_fp,
            "Number polys: %4d  Number corners: %4d  Number opts: %4d  Number opt alignments: ",
            number_polys, number_corners, number_opts);
    if  (new_fun->opt_align_ct <= 10000)
        fprintf(log_fp, "%4d\n", new_fun->opt_align_ct);
    else
        fprintf(log_fp, "> 10,000");

/* **   fprintf(log_fp, "find_poly: end\n");
/* */

/*
**  Now add the new polygon to the polygon list.
**
**  The polygon list is in order of
**      decreasing min_diff_region_dim and
**      increasing (diff_opt_pt, dist_sq_opt_pt).
*/
    prev_poly = NULL;
    curr_poly = poly_list;
    while (   (curr_poly != NULL)
           && (   (new_poly->min_diff_region_dim < curr_poly->min_diff_region_dim)
               || (   (new_poly->min_diff_region_dim == curr_poly->min_diff_region_dim)
                   && (new_poly->diff_opt_pt > curr_poly->diff_opt_pt))
               || (   (new_poly->min_diff_region_dim == curr_poly->min_diff_region_dim)
                   && (new_poly->diff_opt_pt == curr_poly->diff_opt_pt)
                   && (new_poly->dist_sq_opt_pt > curr_poly->dist_sq_opt_pt))
               || (   (new_poly->min_diff_region_dim == curr_poly->min_diff_region_dim)
                   && (new_poly->diff_opt_pt == curr_poly->diff_opt_pt)
                   && (new_poly->dist_sq_opt_pt == curr_poly->dist_sq_opt_pt)
                   && (new_poly->corners->y > curr_poly->corners->y))))
    {
        prev_poly = curr_poly;
        curr_poly = curr_poly->next;
    }
    if  (prev_poly == NULL)
        poly_list = new_poly;
    else
        prev_poly->next = new_poly;
    new_poly->next = curr_poly;
    if  (new_poly->next == NULL)
        last_poly = new_poly;

    return new_poly;
}


/********************************************************************/
/*  This function receives the information for setting up to find   */
/*  the second corner and completes the set up so the standard      */
/*  routine for finding corners (find_next_corner) can be used.     */
/********************************************************************/

void    get_corner2_with_coopt( Find_info       *find_info)
{
    Point       *new_pt1,
                *new_pt2;
    Line        *new_line1,
                *new_line2;
    Funct       *new_fun;

/* **   fprintf(log_fp, "get_corner2_info: begin\n");
/* */
    while  (find_info->corner2 == NULL)
    {
/*
**      First get the candidate for corner2
*/
        find_candidate_pt(find_info);
/*
**      Now find the optimal function for corner2 and evaluate it.
**      First check if it is cooptimal with fun at corner2.
**      If it is, then make sure corner1 and corner2 are distinct.
**      Otherwise check if it is cooptimal with fun at corner1.
**      If it is, then keep it instead of fun and return (since
**      corner1 and corner2 must be distinct.
**      If neither function is optimal at both points then intersect
**      with the polygon function to get a new point and try again.
*/
        for(;;)
        {
/* **       fprintf(log_fp, "find_max 1 in get_corner2_info\n");
/* */
            new_fun = find_max(find_info->corner2, 0);
            if  (new_fun->equal(find_info->fun, opt_function))
            {
/* **           fprintf(log_fp, "delete call 171\n");
/* */
                delete new_fun;
                new_fun = find_info->fun;
                break;
            }
            else if  (   (find_info->fun1 != NULL)
                      && (new_fun->equal(find_info->fun1, opt_function)))
            {
/* **           fprintf(log_fp, "delete call 172\n");
/* */
                delete new_fun;
                new_fun = find_info->fun1;
            }
            else
                check_add_fun(&new_fun);

            if  (equal(find_info->corner2->max_val,
                        find_info->corner2->eval(find_info->fun, opt_function)))
                break;
 
            if  (equal(find_info->corner1->max_val,
                        find_info->corner1->eval(new_fun, opt_function)))
            {
                find_info->fun = new_fun;
                new_fun = NULL;
                break;
            }
/*
**          Don't have a real corner yet -
**          have to intersect and try again.
**          Use check_function to do this to make sure
**          incr and above flags get set properly.
*/
            check_function(find_info, new_fun);
        }
/*
**      Have a real corner
**
**      If the new corner2 is the same as corner1,
**      then replace corner1 info with the corner2 info and start over.
*/
        if  (find_info->corner1->equal(find_info->corner2))
        {
/* **       fprintf(log_fp, "corners evaluated as equal:\n    ");
            print_pt(find_info->corner1);
            fprintf(log_fp, "    ");
            print_pt(find_info->corner2);
            fprintf(log_fp, "    x diff=%lf  y diff=%lf\n",
                   find_info->corner1->x - find_info->corner2->x,
                   find_info->corner1->y - find_info->corner2->y);
/* */
/* **       fprintf(log_fp, "delete call 173\n");
/* */
            delete find_info->corner1;
/* **       fprintf(log_fp, "delete call 174\n");
/* */
            delete find_info->edgeline1;
            find_info->corner1 = find_info->corner2;
            find_info->fun1 = find_info->fun2;
            find_info->edgeline1 = find_info->edgeline2;
            find_info->above1 = find_info->above2;
            find_info->incr1 = find_info->incr2;
            find_info->corner2 = NULL;
            find_info->fun2 = NULL;
            find_info->edgeline2 = NULL;
        }
    }
/* **   fprintf(log_fp, "get_corner2_info: end\n");
/* */
}


/*******************************************************************/
/* This routine processes a whole decomposition for a given set of */
/* settings. It does this by calculating a polygon and remembering */
/* information about the surrounding regions. The surrounding data */
/* is use to calculate new points to use to initiate new polygons. */
/*******************************************************************/

/* ARGSUSED */
void    complete_decomp(        int     all_polys)
{
    Arg         wargs[1];
    char        buf[256];
    Funct       *minval_fun;
    Point       *minval_pt,
                *wk_pt;
    Poly        *new_poly;
    Todo        *poly_todo;

/* **   fprintf(log_fp, "complete_decomp: begin\n");
/* */
    XDefineCursor(XtDisplay(toplevel), XtWindow(toplevel), hourglass);
    XFlush(my_data->disp);

/*
**  First, make sure we have a polygon that includes the lower left
**  corner (minval_x, minval_y).
**  Then remove the todo structures one at a time
**  and find the corresponding polygons.
*/
    if  (!have_minval_poly)
    {
        if  (display_pt != NULL)
        {
            delete display_pt;
            display_pt = NULL;
            delete display_fun_pt;
            display_fun_pt = NULL;
            delete display_fun;
            display_fun = NULL;
        }
        minval_pt = new Point(Global_v, Global_w, minval_x, minval_y);
        new_poly = get_poly_for(minval_pt);

        display_pt = new Point(new_poly->corners);
        display_pt->next = NULL;
        display_fun_pt = new Point(display_pt);
        wk_pt = new_poly->corners->next;
        display_fun_pt->x += wk_pt->x;
        display_fun_pt->y += wk_pt->y;
        wk_pt = wk_pt->next;
        display_fun_pt->x += wk_pt->x;
        display_fun_pt->y += wk_pt->y;
        wk_pt = NULL;
        display_fun_pt->x /= 3.0;
        display_fun_pt->y /= 3.0;

        display_fun = find_max(display_pt,0);
        delete display_fun;
        display_fun = find_max(display_fun_pt,1);

        draw_dark_poly(new_poly);

        if  (input_fun == NULL)
        {
            sprintf(buf, inputaligndiff_noinput,
                    display_pt->x, display_pt->y,
                    display_pt->max_val);
        }
        else
        {
            sprintf(buf, inputaligndiff,
                    display_pt->x, display_pt->y,
                    display_pt->max_val,
                    display_pt->input_fun_val,
                    display_pt->diff_val);
        }
        XtSetArg(wargs[0], XtNlabel, buf);
        XtSetValues(input_align_diff_label, wargs, 1);

        XFlush(my_data->disp);
    }
/*
**  Now find the polygons.
*/
    while (todo_list != NULL)
    {
        if  (   (!all_polys)
             && (todo_list->diff_from_input > poly_list->diff_opt_pt)
             && (!equal(todo_list->diff_from_input, poly_list->diff_opt_pt)))
            break;
        poly_todo = todo_list;
        todo_list = todo_list->next;
/* **   fprintf(log_fp, "Doing point x:%lf y:%lf\n",
                poly_todo->start->x, poly_todo->start->y);
/* */
        new_poly = find_poly(poly_todo);

        delete display_pt;
        display_pt = NULL;
        delete display_fun_pt;
        display_fun_pt = NULL;
        delete display_fun;
        display_fun = NULL;

        display_pt = new Point(new_poly->corners);
        display_pt->next = NULL;
        display_fun_pt = new Point(display_pt);
        wk_pt = new_poly->corners->next;
        display_fun_pt->x += wk_pt->x;
        display_fun_pt->y += wk_pt->y;
        wk_pt = wk_pt->next;
        display_fun_pt->x += wk_pt->x;
        display_fun_pt->y += wk_pt->y;
        wk_pt = NULL;
        display_fun_pt->x /= 3.0;
        display_fun_pt->y /= 3.0;

        display_fun = find_max(display_pt,0);
        delete display_fun;
        display_fun = find_max(display_fun_pt,1);

        draw_dark_poly(new_poly);

        if  (input_fun == NULL)
        {
            sprintf(buf, inputaligndiff_noinput,
                    display_pt->x, display_pt->y,
                    display_pt->max_val);
        }
        else
        {
            sprintf(buf, inputaligndiff,
                    display_pt->x, display_pt->y,
                    display_pt->max_val,
                    display_pt->input_fun_val,
                    display_pt->diff_val);
        }
        XtSetArg(wargs[0], XtNlabel, buf);
        XtSetValues(input_align_diff_label, wargs, 1);

        XFlush(my_data->disp);
    }
/*
**  Now clean up the special settings and we're done.
*/
    XUndefineCursor(XtDisplay(toplevel), XtWindow(toplevel));
    XFlush(my_data->disp);
/* **   fprintf(log_fp, "complete_decomp: end\n");
/* */
}
