/* 
**++
**  FACILITY:   University of California, Davis
**              Computer Science Department
**              Theory Lab
**
**  PROGRAM NAME:       XPARAL
**
**  MODULE DESCRIPTION:
**
**      This is the main module for the X program that does
**      PARameterized string ALignments (XPARAL).
**      In addition to overall flow control this module contains
**      all the code pertaining to the graphics area (canvas).
**
**  AUTHORS:
**      Paul F. Stelling
**      Jim Knight
**      Justin Bronder
**      Douglas Mayfield
**      K. Balasubramanian
**
**  CREATION DATE:      04/25/1991
**
**  MODIFICATION HISTORY:
**
**      05/02/91        J. Bronder      use Motif widgets
**      1992-1995       P. Stelling     corrected and improved
**                                        algorithmic approach
**                                        (including new/revised data structures)
**                                      read strings from file
**                                      reverse strings
**                                      specify region
**                                      output performance statistics (log)
**                                      save alignment/polygon info to a file
**      1994            J. Knight       remove Motif -- use all Xt Intrinsics
**      1995            P. Stelling     read in a reference alignment
**                                      compute values for input alignment
**                                        and display them
**      2010            M. James        Lots of clean-up work. Far fewer crashes.
**                                      (None currently known, but that's just
**                                      asking for it!) built against latest
**                                      Xaw libraries for stability; close icon
**                                      on windows and dialog boxes can now be
**                                      clicked upon; now the named scoring
**                                      matrices are embedded, so users don't
**                                      have to search for files, now includes
**                                      BLOSUM 80, 62, and 45 scoring matrics;
**                                      etc.
**
*/

#include "header.h"
#include "globals_m.h"
#include <assert.h>
#include <X11/Shell.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Sme.h>
#include <X11/Xaw/SmeBSB.h>

#include <X11/Xaw/SmeLine.h>
#include <X11/Xatom.h>

#include <stdarg.h>

extern void     calc_fun_from_path(Funct*);
extern void     calc_align_from_path(Max_node*, Funct*);
extern void     fill_gap_table(int);

/************************************************************************/
/*                                                                      */
/*  These values make sure the scrollbars created inside each text      */
/*  widget are of a certain size, and that the text widget is sized     */
/*  large enough for them.                                              */
/*                                                                      */
/*  There's no other way to set this size, and there's no way to find   */
/*  out when the scrollbar is created under the                         */
/*  XawTextscrollWhenNeeded mode (since I found from looking at the     */
/*  Xaw Text widget source code that its creation does NOT cause a      */
/*  resize of the text window).                                         */
/*                                                                      */
/*  The value of scrollbar size must four pixels bigger than the value  */
/*  of scrollbar thickness.                                             */
/*                                                                      */
/************************************************************************/

#define SCROLLBAR_SIZE         12
#define SCROLLBAR_THICKNESS     8

#define changed_NO              0
#define changed_ALIGN           1
#define changed_PARMS           2
#define changed_STRINGS         4
#define changed_ALIGN_PARMS     3
#define changed_ALIGN_STRINGS   5
#define changed_PARMS_STRINGS   6

int             changed_something = changed_NO;

char            align_io_function;
#define align_io_function_ALL_POLYS    'A'
#define align_io_function_DARK_POLY    'D'
#define align_io_function_LOG          'L'
#define align_io_function_MIN_D_POLY   'M'
#define align_io_function_POLY_ALIGNS  'P'

char            file_io_function;
#define file_io_function_LOAD    'L'
#define file_io_function_SAVE    'S'

struct stat     input_align_buf;
char            *input_align_fname = NULL;
FILE            *input_align_fp = NULL;

/************************************************************************/
/*                                                                      */
/*  Set up the string matching global variables                         */
/*                                                                      */
/************************************************************************/

unsigned char   *input_align_str1 = NULL,
                *input_align_str2 = NULL;
unsigned char   *weight_alphabet = NULL,
                weight_alphabet_char_max,
                weight_alphabet_char_min;
char            *weight_matrix_name = NULL;
int             cursor_flag = 1;
int             weight_alphabet_max_as_int,
                weight_alphabet_min_as_int,
                weight_alphabet_size,
                weight_case_sensitive = 0,
                weight_matrix_array_size;

int             string_index;
int             shade_polys = 0;

double          side_x = 0.0,
                side_y = 0.0;

Pixmap          dark_shade,
                light_shade;
XColor		gray, black;

GC              bggc,
                fggc;
int             depth,
                screen;
Window          root;
Display         *dpy = NULL;
XtAppContext    app;
XFontStruct     *fixed_font = NULL;
int             char_height,
                char_width;

Widget          align_ct_label,
                align_disp,
                align_label,
                align_next,
                align_prev,
                b[4],
                bb,
                canvas,
                command,
                input_align_title_label,
                input_align_info_label,
                menu,
                string[2],
                tracker,
                weight_label;

static int      Xaxis[] = {1,0,0,2,1,1,0,0,0};
static int      Yaxis[] = {2,2,1,3,3,2,3,2,1};

static char     *axis_label[] =
                      { "Score of a match   \0",
                        "Score of a mismatch\0",
                        "Score of an indel / gap extension \0",
                        "Score of a gap initiation\0"};

static char     *gapsetting[] =
                      { "c1=%4.2f         ",
                        "c1=%4.2f  c2=%4.2f"};

static char     *scoring_method_setting[] =
                      { "Opt: Global                   ",
                        "Opt: End-gaps Free            ",
                        "Opt: Local                    ",
                        "Opt: 1st as sub-string of 2nd "};

static char     *varformatstr[] =
                      { "Var: c1(%3d)-x(%3d)-y(%3d)            ",
                        "Var: x(%3d)-c1(%3d)-y(%3d)            ",
                        "Var: x(%3d)-y(%3d)-c1(%3d)            ",
                        "Var: c1(%3d)-c2(%3d)-x(%3d)-y(%3d)    ",
                        "Var: c1(%3d)-x(%3d)-c2(%3d)-y(%3d)    ",
                        "Var: c1(%3d)-x(%3d)-y(%3d)-c2(%3d)    ",
                        "Var: x(%3d)-c1(%3d)-c2(%3d)-y(%3d)    ",
                        "Var: x(%3d)-c1(%3d)-y(%3d)-c2(%3d)    ",
                        "Var: x(%3d)-y(%3d)-c1(%3d)-c2(%3d)    ",
                        "Var: c1(%3d)+x(%3d)-y(%3d)            ",
                        "Var: x(%3d)+c1(%3d)-y(%3d)            ",
                        "Var: x(%3d)+y(%3d)-c1(%3d)            ",
                        "Var: c1(%3d)+c2(%3d)-x(%3d)-y(%3d)    ",
                        "Var: c1(%3d)+x(%3d)-c2(%3d)-y(%3d)    ",
                        "Var: c1(%3d)+x(%3d)-y(%3d)-c2(%3d)    ",
                        "Var: x(%3d)+c1(%3d)-c2(%3d)-y(%3d)    ",
                        "Var: x(%3d)+c1(%3d)-y(%3d)-c2(%3d)    ",
                        "Var: x(%3d)+y(%3d)-c1(%3d)-c2(%3d)    "};

static char     *varsetting[] =
                      { "Var: c1(mat)-x(mis)-y(i/d)            ",
                        "Var: x(mat)-c1(mis)-y(i/d)            ",
                        "Var: x(mat)-y(mis)-c1(i/d)            ",
                        "Var: c1(mat)-c2(mis)-x(i/d)-y(gap)    ",
                        "Var: c1(mat)-x(mis)-c2(i/d)-y(gap)    ",
                        "Var: c1(mat)-x(mis)-y(i/d)-c2(gap)    ",
                        "Var: x(mat)-c1(mis)-c2(i/d)-y(gap)    ",
                        "Var: x(mat)-c1(mis)-y(i/d)-c2(gap)    ",
                        "Var: x(mat)-y(mis)-c1(i/d)-c2(gap)    ",
                        "Var: c1(mat)+x(mis)-y(i/d)            ",
                        "Var: x(mat)+c1(mis)-y(i/d)            ",
                        "Var: x(mat)+y(mis)-c1(i/d)            ",
                        "Var: c1(mat)+c2(mis)-x(i/d)-y(gap)    ",
                        "Var: c1(mat)+x(mis)-c2(i/d)-y(gap)    ",
                        "Var: c1(mat)+x(mis)-y(i/d)-c2(gap)    ",
                        "Var: x(mat)+c1(mis)-c2(i/d)-y(gap)    ",
                        "Var: x(mat)+c1(mis)-y(i/d)-c2(gap)    ",
                        "Var: x(mat)+y(mis)-c1(i/d)-c2(gap)    "};

static char     *weightoptionconst =
                        "Constant Costs (No Scores)";

static char     *align_ct_blank =
                        "                    ";
static char     *align_ct_disp =
                        "%4d of %4d        ";
static char     *align_ct_overlim_disp =
                        "%4d of > 10,000";

static char     *inputaligntitle =
                        "    Input Alignment Information      ";

static char     *inputaligninfonone =
                        "           (None Input)            \n"
                        " \n"
                        " \n"
                        " \n"
                        " \n"
                        " \n"
                        " \n"
                        " \n"
                        " \n"
                        " ";
static char     *inputaligninfo =
                        "Counts:                              \n"
                        "  Matches:        %8d       \n"
                        "  Mismatches:     %8d       \n"
                        "  In/Dels:        %8d       \n"
                        "  Gaps:           %8d       \n"
                        "Values:                          \n"
                        "  Match:          %15.8f\n"
                        "  Mismatch:       %15.8f\n"
                        "  In/Del (Gap Len):%14.8f\n"
                        "  Gap Init:       %15.8f";

extern void     analyze_alignment_c(    Funct*);
extern void     analyze_alignment_w(    Funct*);
extern void     analyze_alignment_convex_c(    Funct*);
extern void     analyze_alignment_convex_w(    Funct*);
extern void     calc_fun_from_path(Funct*);
extern void     calc_align_from_path(Max_node*, Funct*);
extern void     check_add_fun(Funct**);
extern void     check_remove_fun(const Funct*);
extern void     complete_decomp(int);
extern Funct    *find_max(Point*, int);
extern Funct    *find_max_count(Point*, int);
extern Point    *find_next_corner(Find_info*, int);
extern Poly     *find_poly(Todo*);
extern void     get_corner2_with_coopt(Find_info*);
extern void     get_first_align_path(Max_node*);
extern int      get_next_align_path(Max_node*);
extern int      get_prev_align_path(Max_node*);
extern void     print_fun(Funct*);

static Pixmap   intGetDarkPixmap(Display*, Window);
static Pixmap   intGetLightPixmap(Display*, Window);
//static Pixmap intGetWhitePixmap(Display*, Window);

static char     *intTextGetString(Widget);
static void     intTextSetString(Widget, char*);

Widget          create_command( char*, Widget, Widget, Widget,
                                void (*)(Widget, char*, XtPointer),
                                XtPointer, Arg*, int),
                create_form(char*, Widget, Widget, Widget, int, int, Arg*, int),
                create_label(char*, Widget, Widget  left, Widget, int, Arg*, int),
                create_menu(char*, Widget, Widget, Widget, int_menu_struct, Arg*, int),
                create_text(char*, Widget, Widget, Widget, char*, int, int, Arg*, int);

static void     process_change(int);

inline static int   weight_io_load(char *filename, struct stat  *filebuf, int isPreloaded);
static void	    get_mult_polys(Widget w, char *ch, XtPointer call_data);

/************************************************************************/
/*  Updates the row of info widgets                                     */
/************************************************************************/
inline
static void     update( Funct   *fun)
{
    Arg         wargs[1];
    char        buf[64];

/* **   fprintf(log_fp, "update: begin\n");
/* */

    if  (fun)
    {
        if  (input_fun == NULL)
        {
            if  (!with_gaps)
                sprintf(buf, varformatstr[opt_function - 1],
                        fun->match_ct, fun->mis_ct, fun->inordel_ct);
            else
                sprintf(buf, varformatstr[opt_function - 1],
                        fun->match_ct, fun->mis_ct, fun->inordel_ct,
                        fun->gaps_ct);
        }
        else
        {
            if  (!with_gaps)
                sprintf(buf, varformatstr[opt_function + 8],
                        fun->match_ct, fun->mis_ct, fun->inordel_ct);
            else
                sprintf(buf, varformatstr[opt_function + 8],
                        fun->match_ct, fun->mis_ct, fun->inordel_ct,
                        fun->gaps_ct);
        }
        XtSetArg(wargs[0], XtNlabel, buf);
        XtSetValues(b[2], wargs, 1);
    }
/* **   fprintf(log_fp, "update: end\n");
/* */
}


/*
** fills in all info needed to call XLib functions
*/
inline
static void     get_my_info(    Widget          canvas,
                                Image_data      *my_data)
{
/*    XGCValues dummy; */
    Arg         wargs[5];
    Dimension   height,
                width;
    int         n = 0;

/* **   fprintf(log_fp, "get_my_info: begin\n");
/* */
    my_data->disp = XtDisplay(canvas);
    my_data->win = XtWindow(canvas);

    width = height = 0;
    XtSetArg(wargs[n], XtNwidth, &width); n++;
    XtSetArg(wargs[n], XtNheight, &height); n++;
    XtGetValues(canvas, wargs, n);

    /* Casting needed because of different versions of type Dimension */
    my_data->width = (Dimension)width;
    my_data->height = (Dimension)height;
/* **   fprintf(log_fp, "Width %d\n", my_data->width);
    fprintf(log_fp, "Height %d\n", my_data->height);
/* */
    if  (cursor_flag)
    {
        hourglass = XCreateFontCursor(my_data->disp, XC_watch);
        dark_shade = intGetDarkPixmap(my_data->disp, my_data->win);
        light_shade = intGetLightPixmap(my_data->disp, my_data->win);
        cursor_flag = 0;
    }
/* **   fprintf(log_fp, "get_my_info: end\n");
/* */
}


/************************************************************************/
/* Draws lines, hash marks, and titles along the borders                */
/************************************************************************/
inline
static void     draw_axes(      Widget          canvas,
                                Image_data      *my_data)
{
    int         i,
                len1,
                len2;
    char        *axis_x_string,
                *axis_y_string;
    char        region_string[200];
    double      hash_loc,
                hash_step;

/* **   fprintf(log_fp, "draw_axes: begin\n");
/* */
/*
**  clear the canvas, then draw the x and y axes
*/
    XClearArea(my_data->disp, my_data->win, 0, 0, 0, 0, FALSE);
    XSetFillStyle(my_data->disp, fggc, FillSolid);
    XSetLineAttributes(my_data->disp, fggc, 2, LineSolid, CapButt, JoinMiter);
    XDrawLine(my_data->disp, my_data->win, fggc,
                20, my_data->height - 20,
                my_data->width - 20, my_data->height - 20);
    XDrawLine(my_data->disp, my_data->win, fggc,
                20, 20,
                20, my_data->height - 20);
/*
**  draw the hash marks based on:
**      minimum and maximum values for x and y,
**      size of the display area on the screen.
*/
    hash_step = ((double)((int)side_x + 1) /
                 (double)((my_data->width - 40) / 10));
    i  =  1;
    while  (hash_step > (double)i)
        i *= 10;
    hash_step = (double)i;

    hash_loc = minval_x - (double)((int)minval_x);
    for  (; hash_loc <= side_x; hash_loc += hash_step)
        XDrawLine(my_data->disp, my_data->win, fggc,
                20 +
                  (int)((hash_loc * (double)(my_data->width - 40))/
                        side_x),
                my_data->height - 14,
                20 +
                  (int)((hash_loc * (double)(my_data->width - 40))/
                        side_x),
                my_data->height - 20);


    hash_step = ((double)((int)side_y + 1) /
                 (double)((my_data->height - 40) / 10));
    i  =  1;
    while  (hash_step > (float)i)
        i *= 10;
    hash_step = (float)i;

    hash_loc = minval_y - (double)((int)minval_y);
    for  (; hash_loc <= side_y; hash_loc += hash_step)
        XDrawLine(my_data->disp, my_data->win, fggc,
                14,
                my_data->height - 20 -
                  (int)((hash_loc * (double)(my_data->height - 40))/
                        side_y),
                20,
                my_data->height - 20 -
                  (int)((hash_loc * (double)(my_data->height - 40))/
                        side_y));
/*
**  display the titles for the axes
*/
/* **   fprintf(log_fp, "drew hash marks\n");
/* */
    sprintf(region_string, "Region: X: %f to %f  Y: %f to %f",
            minval_x, maxval_x, minval_y, maxval_y);
    XDrawString(my_data->disp, my_data->win, fggc, 20, 10,
                region_string, strlen(region_string));

    axis_x_string = axis_label[Xaxis[opt_function - 1]];
    axis_y_string = axis_label[Yaxis[opt_function - 1]];
    len1 = strlen(axis_x_string);
    len2 = strlen(axis_y_string);
    XDrawString(my_data->disp, my_data->win, fggc,
                (my_data->width / 2) - (4 * strlen(axis_x_string)),
                my_data->height - 4, 
                axis_x_string, strlen(axis_x_string));
  /* hack, draw each character separately along, appears like it goes down */
    for (i = 0; i < len2; i++)
        XDrawString(my_data->disp, my_data->win, fggc,
                    4,
                    (my_data->height / 2) - (4 * strlen(axis_y_string)) + (i * 10),
                    axis_y_string + i, 1);

/* **   fprintf(log_fp, "draw_axes: end\n");
/* */
}


/************************************************************************/
/*  Draw a single polygon                                               */
/************************************************************************/

static void     draw_onepoly(   Widget          canvas,
                                Image_data      *my_data,
                                Poly            *polygon,
                                Pixmap          *shade)
{
    int         i = 0;
    int         n_pts = 0;
    Point       *corner;
    XPoint      *pointlist;

/* **   fprintf(log_fp, "draw_onepoly: begin\n");
/* */
/*
**  get the number of corners in the polygon
*/
    corner = polygon->corners;
    while (corner != NULL)
    {
        n_pts++;
        corner = corner->next;
    }
    n_pts++;
    pointlist = new XPoint[n_pts];
/*
** do for each point in the polygon
*/
    corner = polygon->corners;
    while (corner != NULL)
    {
        pointlist[i].x = 20 +
                         (int)((corner->x - minval_x) *
                               (double)(my_data->width - 40) / side_x);
        pointlist[i].y = my_data->height - 20 - 
                         (int)((corner->y - minval_y) *
                               (double)(my_data->height - 40) / side_y);
/* **   fprintf(log_fp,
                "draw_onepoly - x:%.16f y:%.16f\n", corner->x, corner->y);
        fprintf(log_fp,
                "draw_onepoly - x:%d y:%d\n", pointlist[i].x, pointlist[i].y);
/* */
        i++;
        corner = corner->next;
    }
/*
**  Put the first one in one last time to close the polygon.
*/
    corner = polygon->corners;
    pointlist[i].x = 20 +
                         (int)((corner->x - minval_x) *
                               (double)(my_data->width - 40) / side_x);
    pointlist[i].y = my_data->height - 20 -
                         (int)((corner->y - minval_y) *
                               (double)(my_data->height - 40) / side_y);
/* **   fprintf(log_fp,
                "draw_onepoly - x:%.16f y:%.16f\n", corner->x, corner->y);
    fprintf(log_fp,
            "draw_onepoly - x:%d y:%d\n\n", pointlist[i].x, pointlist[i].y);
/* */
    XSetLineAttributes(my_data->disp, fggc, 2, LineSolid, CapButt, JoinBevel);
    XSetFillStyle(my_data->disp, fggc, FillTiled);
    XSetTile(my_data->disp, fggc, *shade);
    if  (   (shade_polys)
         || (shade == &dark_shade))
        XFillPolygon(my_data->disp, my_data->win, fggc,
                     pointlist, n_pts - 1, Convex, CoordModeOrigin);
    else if  (!shade_polys || shade != &dark_shade) {
        XSetFillStyle(my_data->disp, bggc, FillSolid);
        XFillPolygon(my_data->disp, my_data->win, bggc,
                     pointlist, n_pts - 1, Convex, CoordModeOrigin);
    }

    XSetFillStyle(my_data->disp, fggc, FillSolid);
    XDrawLines(my_data->disp, my_data->win, fggc, pointlist, n_pts, CoordModeOrigin);
/*    XSetLineAttributes(my_data->disp, fggc, 0, LineSolid, CapButt, JoinBevel); */
    delete pointlist;
    pointlist = NULL;
/* **   fprintf(log_fp, "draw_onepoly: end\n");
/* */
}


/************************************************************************/
/*  This routine takes a polygon and displays it dark after             */
/*  making the previously dark polygon (if any) light.                  */
/************************************************************************/

void    draw_dark_poly( Poly    *poly_to_draw)
{
    Arg         wargs[1];
    char        buf[32],
                *dstr;
    int         align1_len,
                align2_len;
/*
**  Unshade the previous polygon (if any)
*/
    if  (   (dark_poly != NULL)
         && (dark_poly != poly_to_draw))
    {
/* **   fprintf(log_fp, "unshade the poly:\n");
/* */
        draw_onepoly(canvas, my_data, dark_poly, &light_shade);
    }
    dark_poly = poly_to_draw;
/*
**  Output the string alignment for the new polygon
*/
    align1_len = strlen(display_fun->align1);
    align2_len = strlen(display_fun->align2);
    dstr = new char[align1_len + align2_len + 2];
    strcpy(dstr, display_fun->align1);
    dstr[align1_len] = '\n';
    strcpy(dstr+align1_len+1, display_fun->align2);

/*    intTextSetMaxLength(align_disp, align1_len+align2_len+1); */
    intTextSetString(align_disp, dstr);
    delete dstr;
    dstr = NULL;

    if  (display_fun->opt_align_ct <= 10000)
        sprintf(buf, align_ct_disp, align_number, display_fun->opt_align_ct);
    else
        sprintf(buf, align_ct_overlim_disp, align_number);
    XtSetArg(wargs[0], XtNlabel, buf);
    XtSetValues(align_ct_label, wargs, 1);

    update(display_fun);
/*
**  Shade the polygon
*/
/* **   fprintf(log_fp, "shade the poly:\n");
/* */

    if  (poly_to_draw != NULL)
    {
        draw_onepoly(canvas, my_data, poly_to_draw, &dark_shade);
    }

    XFlush(my_data->disp);
    XSetFillStyle(my_data->disp, fggc, FillSolid);

}


/************************************************************************/
/*  Draw all of the polygons                                            */
/************************************************************************/
inline
static void     draw_polygons(  Widget          canvas,
                                Image_data      *my_data)
{
    Arg         wargs[1];
    Poly        *poly_curr;
    char        buf[256];

/* **   fprintf(log_fp, "draw_polygons: begin\n");
/* */
/*
** do for each polygon in the list
*/
    poly_curr = poly_list;
    while (poly_curr != NULL)
    {
        draw_onepoly(canvas, my_data, poly_curr, &light_shade);
        poly_curr = poly_curr->next;
    }
    if  (dark_poly != NULL)
    {
        draw_dark_poly(dark_poly);

        if  (input_fun == NULL)
        {
            if  (display_pt != NULL)
            {
                sprintf(buf, inputaligndiff_noinput,
                        display_pt->x, display_pt->y,
                        display_pt->max_val);
            }
            else
            {
                sprintf(buf, inputaligndiff_noinput,
                        dark_poly->corners->x, dark_poly->corners->y,
                        dark_poly->corners->max_val);
            }
        }
        else
        {
            if  (display_pt != NULL)
            {
                sprintf(buf, inputaligndiff,
                        display_pt->x, display_pt->y,
                        display_pt->max_val,
                        display_pt->input_fun_val,
                        display_pt->diff_val);
            }
            else
            {
                sprintf(buf, inputaligndiff,
                        dark_poly->corners->x, dark_poly->corners->y,
                        dark_poly->corners->max_val,
                        dark_poly->corners->input_fun_val,
                        dark_poly->corners->diff_val);
            }
        }
    }
    else
    {
        sprintf(buf, "%s", inputaligndiff_blank);
    }
    XtSetArg(wargs[0], XtNlabel, buf);
    XtSetValues(input_align_diff_label, wargs, 1);
/* **   fprintf(log_fp, "draw_polygons: end\n");
/* */
}


/************************************************************************/
/*  Redraw the polygons                                                 */
/************************************************************************/
/* ARGSUSED */
static void     redraw( Widget          w,
                        XtPointer       client_data,
                        XtPointer       call_data)
{
/* **   fprintf(log_fp, "redraw: begin\n");
/* */
    if  (XtIsRealized(canvas))
    {
        get_my_info(canvas, my_data);
        draw_axes(canvas, my_data);
        if  (poly_list != NULL)
	    {
            draw_polygons(canvas, my_data);
	    }
    }
/* **   fprintf(log_fp, "redraw: end\n");
/* */
}


/************************************************************************/
/*  Initialize all global variables                                     */
/************************************************************************/
inline
static void init_globals()
{
/* **   fprintf(log_fp, "init_globals: begin\n");
/* */
    len1 = 0;
    len2 = 0;
    with_gaps = 0;
    opt_function = 1;
    scoring_method = GLOBAL;
    Global_v = 1.0; 
    Global_w = 0.0;
    minval_x = 0.0;
    maxval_x = 10.0;
    side_x = maxval_x - minval_x;
    minval_y = 0.0;
    maxval_y = 10.0;
    side_y = maxval_y - minval_y;
    my_data = &hack_data;
    weights_in_array = 0;
/* **   fprintf(log_fp, "init_globals: end\n");
/* */
}


/************************************************************************/
/*  Clear and delete the input function from the input alignment --     */
/*    one of the strings has been changed.                              */
/************************************************************************/

static void     check_delete_input_fun_align()
{
    Arg         wargs[1];

    if  (input_fun != NULL)
    {
        delete input_fun;
        input_fun = NULL;

        XtSetArg(wargs[0], XtNlabel, inputaligninfonone);
        XtSetValues(input_align_info_label, wargs, 1);

        XtSetArg(wargs[0], XtNlabel, " ");
        XtSetValues(input_align_diff_label, wargs, 1);
    }
}


/************************************************************************/
/*  Clear all polygons and functions, freeing all associated memory     */
/************************************************************************/
inline
static void     clear_polys()
{
    Poly        *poly_curr;
    Todo        *todo_curr;
    Funct_list  *fun_curr;

/* **   fprintf(log_fp, "clear_polys: begin\n");
/* */
    dark_poly = NULL;
    while (poly_list != NULL)
	{
        poly_curr = poly_list;
        poly_list = poly_list->next;
        delete poly_curr;
	}
    poly_curr = NULL;
    have_minval_poly = 0;
    while (todo_list != NULL)
	{
        todo_curr = todo_list;
        todo_list = todo_list->next;
        delete todo_curr;
	}
    todo_curr = NULL;
    while (fun_list != NULL)
	{
        fun_curr = fun_list;
        fun_list = fun_list->next;
        delete fun_curr;
	}
    fun_curr = NULL;
    number_corners = 0;
    number_opts = 0;
    number_polys = 0;
/* **   fprintf(log_fp, "\n\nPolygons cleared\n\n");
/* */
/* **   fprintf(log_fp, "clear_polys: end\n");
/* */
}



char    *tracker_not_in_area = "Not in Area         ",
        *tracker_format = "X:%14.6f  \nY:%14.6f  ";

/************************************************************************/
/*  Clear the mouse pointer display                                     */
/************************************************************************/
/* ARGSUSED */
static void     clear_pos(      Widget  w,
                                Widget  tracker,
                                XEvent  event)
{
    Arg         wargs[1];

    XtSetArg(wargs[0], XtNlabel, tracker_not_in_area);
    XtSetValues(tracker, wargs, 1);
}


/************************************************************************/
/*  Track the mouse pointer                                             */
/************************************************************************/
/* ARGSUSED */
static void     track_pos(      Widget  w,
                                Widget  tracker,
                                XEvent  *event)
{
    Arg         wargs[1];
    char        buf[64];
    double      norm_pos_x,
                norm_pos_y;

    norm_pos_x = minval_x +
                ((double)(event->xbutton.x - 20) /
                 (double)(my_data->width - 40)) *
                  side_x;
    norm_pos_y = minval_y +
                ((double)(my_data->height - 20 - event->xbutton.y)/
                 (double)(my_data->height - 40)) *
                  side_y;

    if  (   (norm_pos_x <= maxval_x)
         && (norm_pos_x >= minval_x) 
         && (norm_pos_y <= maxval_y)
         && (norm_pos_y >= minval_y))
    {
        sprintf(buf, tracker_format, norm_pos_x, norm_pos_y);
        XtSetArg(wargs[0], XtNlabel, buf);
    }
    else
    {
        XtSetArg(wargs[0], XtNlabel, tracker_not_in_area);
    }
    XtSetValues(tracker, wargs, 1);
}


/************************************************************************/
/*  This routine searches poly_list for a polygon whose function        */
/*  is the same as the one passed as a parameter.                       */
/*  If it finds the polygon,                                            */
/*  then it returns a pointer to the found polygon                      */
/*  otherwise it returns a NULL pointer.                                */
/************************************************************************/

static Poly     *search_poly_list(      const Funct     *search_fun,
                                        const Point     *search_pt,
                                        const int       match_function_exactly)
{
    Point       search_pt_wk = search_pt;
    Poly        *curr_poly;

/* **   fprintf(log_fp, "search_poly_list: begin\n");
/* */
/*
**  First look for the passed function
*/
    curr_poly = poly_list;
    while  (curr_poly != NULL)
    {
        if  (search_fun->equal(curr_poly->fun, opt_function))
        {
/* **       fprintf(log_fp, "search_poly_list: end\n");
/* */
            return curr_poly;
        }
        else
        {
            curr_poly = curr_poly->next;
        }
    }
    if  (match_function_exactly)
        return NULL;
/*
**  Didn't find the function.
**  Now look for a function that is co-optimal with
**  the passed function at the passed point.
*/
    curr_poly = poly_list;
    while  (curr_poly != NULL)
    {
        if  (equal(search_pt_wk.max_val,
                   search_pt_wk.eval(curr_poly->fun, opt_function)))
        {
/* **       fprintf(log_fp, "search_poly_list: end\n");
/* */
            return curr_poly;
        }
        else
        {
            curr_poly = curr_poly->next;
        }
    }
/*
**  Didn't find the function.
**  Return NULL pointer.
*/
/* **   fprintf(log_fp, "search_poly_list: end\n");
/* */
    return curr_poly;
}


/************************************************************************/
/*  This routine searches todo_list for polygon to do information       */
/*  where the function is the same as the one passed as a               */
/*   parameter.                                                         */
/*  If it finds the to do information,                                  */
/*  then it removes the found to do information from the list and       */
/*       returns a pointer to it                                        */
/*  otherwise it returns a NULL pointer.                                */
/************************************************************************/

static Todo     *get_fr_todo_list(      const Funct     *search_fun,
                                        const Point     *search_pt)
{
    Point       search_pt_wk = search_pt;
    Todo        *curr_todo,
                *prev_todo;

/* **   fprintf(log_fp, "get_fr_todo_list: begin\n");
/* */
/*
**  First look for the passed function
*/
    prev_todo = NULL;
    curr_todo = todo_list;
    while  (curr_todo != NULL)
    {
        if  (search_fun->equal(curr_todo->fun, opt_function))
        {
            if  (prev_todo == NULL)
                todo_list = curr_todo->next;
            else
                prev_todo->next = curr_todo->next;
/* **       fprintf(log_fp, "get_fr_todo_list: end\n");
/* */
            return curr_todo;
        }
        else
        {
            prev_todo = curr_todo;
            curr_todo = curr_todo->next;
        }
    }
/*
**  Didn't find the function.
**  Now look for a function that is co-optimal with
**  the passed function at the passed point.
*/
    prev_todo = NULL;
    curr_todo = todo_list;
    while  (curr_todo != NULL)
    {
        if  (equal(search_pt_wk.max_val,
                   search_pt_wk.eval(curr_todo->fun, opt_function)))
        {
            if  (prev_todo == NULL)
                todo_list = curr_todo->next;
            else
                prev_todo->next = curr_todo->next;
/* **       fprintf(log_fp, "get_fr_todo_list: end\n");
/* */
            return curr_todo;
        }
        else
        {
            prev_todo = curr_todo;
            curr_todo = curr_todo->next;
        }
    }
/*
**  Didn't find the function.
**  Return NULL pointer.
*/
/* **   fprintf(log_fp, "get_fr_todo_list: end\n");
/* */
    return curr_todo;
}


/************************************************************************/
/*  Find an interior point of a polygon that includes the passed        */
/*  point (try the passed function as most likely).                     */
/************************************************************************/
inline
static Point    *find_interior_pt(      Point   *passed_pt,
                                        Funct   *passed_fun)
{
    Find_info   *temp_find = NULL;
    Point       *first_pt = NULL;
    double      slope,
                step;

/* **   fprintf(log_fp, "find_interior_pt: begin\n");
/* */
/*
**  Find a point distinct from passed_pt that shares an optimal
**  function with passed_pt.
**  Start from a point that is on the line between passed_pt
**  and the lower left corner such that max(x difference, y difference)
**  is min(0.01, side_x * 0.01, side_y * 0.01).
**  The point will be closer to the lower left corner if there is room,
**  otherwise further.
*/
    if  (side_x <= side_y)
        if  (side_x >= 1.0)
            step = 0.01;
        else
            step = side_x * 0.01;
    else
        if  (side_y >= 1.0)
            step = 0.01;
        else
            step = side_y * 0.01;

    temp_find = new Find_info;
    temp_find->fun = passed_fun;
    temp_find->corner1 = passed_pt;
    if  (equal(passed_pt->y, minval_y))
        temp_find->edgeline1 = new Line(0, 0.0, minval_y);
    else if  (equal(passed_pt->x, minval_x))
        temp_find->edgeline1 = new Line(1, INFINITY, minval_x);
    else
    {
        slope = (passed_pt->y - minval_y) / (passed_pt->x - minval_x);
        temp_find->edgeline1 = new Line(0, slope, minval_y - (slope * minval_x));
    }

    if  (   (temp_find->corner1->x - minval_x > step)
         || (temp_find->corner1->y - minval_y > step))
        temp_find->incr1 = 0;
    else
        temp_find->incr1 = 1;
    if  (temp_find->edgeline1->m_is_inf)
        if  (temp_find->corner1->x - minval_x > step)
            temp_find->above1 = 0;
        else
            temp_find->above1 = 1;
    else
        if  (temp_find->corner1->y - minval_y > step)
            temp_find->above1 = 0;
        else
            temp_find->above1 = 1;

    get_corner2_with_coopt(temp_find);
/*
**  corner2 is distinct from corner1 and they share an
**  optimality function.
**  If the shared optimality function was found for
**  corner2, then it has replaced the previous temp_find->fun.
**
**  Save the first co-optimal point that we found as first_pt.
**  temp_find->fun is optimal over the line segment between
**  passed_pt and first_pt, so there can be no
**  polygon corners between them.
**  Further, if we take a perpendicular to edgeline1 from
**  the point halfway between passed_pt and first_pt then
**  we must find a solution that is cooptimal for the
**  midpoint, and as a result cooptimal for three
**  non-linear points -- i.e., that is optimal in a three
**  dimensional region (a polygon).
*/
    temp_find->fun1 = NULL;

    first_pt = temp_find->corner2;
    temp_find->corner2 = NULL;
    temp_find->fun2 = NULL;

    temp_find->corner1 = new Point(Global_v, Global_w,
                                   (passed_pt->x + first_pt->x) / 2.0,
                                   (passed_pt->y + first_pt->y) / 2.0);
    temp_find->corner1->set_vals(temp_find->fun, opt_function,
                                 minval_x, minval_y,
                                 input_fun);

/*
**  Now set temp_find->line1 to be a line that is perpendicular to 
**  the old temp_find->line1 and passes through the new temp_find->corner1.
*/
    if  (temp_find->edgeline1->m_is_inf)
    {
        temp_find->edgeline1->m_is_inf = 0;
        temp_find->edgeline1->m = 0.0;
        temp_find->edgeline1->b = temp_find->corner1->y;
        temp_find->incr1 = temp_find->above1;
    }
    else if  (temp_find->edgeline1->m == 0.0)
    {
        temp_find->edgeline1->m_is_inf = 1;
        temp_find->edgeline1->m = INFINITY;
        temp_find->edgeline1->b = temp_find->corner1->x;
        temp_find->incr1 = temp_find->above1;
    }
    else
    {
        //  temp_find->edgeline1->m_is_inf = 0;
        temp_find->edgeline1->m = -(1.0 / temp_find->edgeline1->m);
        temp_find->edgeline1->b = temp_find->corner1->y
                            - (temp_find->edgeline1->m * temp_find->corner1->x);
        if  (temp_find->edgeline1->m > 0)
            temp_find->incr1 = 1 - temp_find->above1;
        else
            temp_find->incr1 = temp_find->above1;
    }
//  above1 can be set to either 0 or 1.
    temp_find->above1 = 0;

    get_corner2_with_coopt(temp_find);

/*
**  Now passed_pt, first_pt, and second_pt1, are all
**  distinct and not on a single line.
**  Only need to free the allocated structures and
**  average the three points and return the averaged point.
*/
    first_pt->x += passed_pt->x + temp_find->corner2->x;
    first_pt->x /= 3.0;
    first_pt->y += passed_pt->y + temp_find->corner2->y;
    first_pt->y /= 3.0;

    temp_find->fun = NULL;
    temp_find->fun1 = NULL;
    temp_find->fun2 = NULL;
    delete temp_find;
    temp_find = NULL;

/* **   fprintf(log_fp, "find_interior_pt: end\n");
/* */
    return first_pt;
}


/************************************************************************/
/*  This routine receives a function for which it must find and         */
/*  format the appropriate todo information.                            */
/*  First it looks for the function in the fun_list.                    */
/*  If it finds it there,                                               */
/*  then it removes it.                                                 */
/*  Next it sets up a find_info structure using the interior            */
/*  point passed to it.                                                 */
/*  Finally, it uses find_next_corner twice to ensure that it has       */
/*  a real corner and edgeline to start from.                           */
/*  It then formats todo structure for the polygon and                  */
/*  returns a pointer to it.                                            */
/************************************************************************/
inline
static Todo     *set_up_todo(   Funct   *start_fun,
                                Point   *start_pt)
{
    Find_info   *temp_find_info;
    Funct_list  *curr_fun,
                *prev_fun,
                *new_fun1,
                *new_fun2,
                *new_fun3;
    Point       *new_pt1,
                *new_pt2,
                *new_pt3;
    Line        *new_line1,
                *new_line2,
                *new_line3;
    Todo        *new_todo;
    int         found = 0;

/* **   fprintf(log_fp, "set_up_todo: begin\n");
/* */
    check_remove_fun(start_fun);
/*
**  Allocate and set up a find_info structure for temporary use.
*/
    temp_find_info = new Find_info;
    temp_find_info->fun = start_fun;
    temp_find_info->corner2 = new Point(start_pt);
    temp_find_info->edgeline2 = new Line(0, 0.0, start_pt->y);
    temp_find_info->above2 = 1;
    temp_find_info->incr2 = 1;
/*
**  Use find_next_corner three times to ensure that it has
**  a real corner and edgeline to start from.
*/
    new_pt1 = find_next_corner(temp_find_info, 0);
    new_pt2 = find_next_corner(temp_find_info, 0);
    new_pt3 = find_next_corner(temp_find_info, 0);
/*
**  Format the todo structure,
**  clean up unused memory,
**  and return a pointer to the todo structure.
*/
    delete new_pt1;
    delete new_pt2;
    new_pt1 = NULL;
    new_pt2 = NULL;

    new_todo = new Todo(start_fun, new_pt3, temp_find_info->edgeline0,
                        temp_find_info->above0, 1 - temp_find_info->incr0);

    temp_find_info->fun = NULL;
    temp_find_info->edgeline0 = NULL;
    temp_find_info->corner1 = NULL;
    temp_find_info->fun1 = NULL;
    temp_find_info->fun2 = NULL;
    delete temp_find_info;
    temp_find_info= NULL;

/* **   fprintf(log_fp, "set_up_todo: end\n");
/* */
    return new_todo;
}


/************************************************************************/
/*  This function is passed a point and                                 */
/*  returns a pointer to a polygon that includes the point.             */
/*  First it finds an optimal function for the point.                   */
/*  Then it checks for the function (or one that is co-optimal          */
/*  with it at the passed point) in the polygon list.                   */
/*  If it finds one, then it returns a pointer to it.                   */
/*  Otherwise it checks for a todo structure for the function (or       */
/*  one that is co-optimal at the passed point) in the todo list.       */
/*  If it doesn't find one, then it finds a point that must be on       */
/*  the interior of a polygon whose function is optimal for the         */
/*  passed point, and sets up a todo structure to find the polygon.     */
/*  Then it calls find_poly to find the polygon,                        */
/*  adds it to the polygon list,                                        */
/*  and returns a pointer to it.                                        */
/************************************************************************/

Poly    *get_poly_for(  Point   *input_pt)
{
    Funct       *new_fun,
                *start_fun;
    Point       *poly_pt,
                *start_pt;
    Poly        *curr_poly,
                *new_poly;
    Todo        *curr_todo;

/* **   fprintf(log_fp, "get_poly_for: begin\n");
/* */
/*
**  Find an optimal function for poly_pt.
*/
/* **   fprintf(log_fp, "find_max 1 in get_poly_for\n");
/* */
    if ( max_gap_size < ( (len1 > len2) ? len1 : len2) ) {
	fill_gap_table( ( (len1 > len2) ? len1 : len2) );
    }
    new_fun = find_max(input_pt, 0);
    poly_pt = new Point(input_pt);
/*
**  Check the poly_list for a polygon for that point.
*/
    curr_poly = search_poly_list(new_fun, poly_pt, 0); 

    if  (curr_poly == NULL)
    {
/*
**      Didn't find a polygon,
**      look for a todo structure
*/
        curr_todo = get_fr_todo_list(new_fun, poly_pt);
        if  (curr_todo == NULL)
        {
/*
**          Didn't find a todo structure,
**          so first make sure new_fun is in the function list.
*/
            check_add_fun(&new_fun);
/*
**          Copy new_fun so can delete it later
*/
            new_fun = new Funct(new_fun);
/*
**          Now find a point on the interior of a polygon
**          whose function is optimal for poly_pt.
*/
            start_pt = find_interior_pt(poly_pt, new_fun);
/* **       fprintf(log_fp, "find_max 2 in get_poly_for\n");
/* */
            start_fun = find_max(start_pt, 0);
/* **       fprintf(log_fp, "optimal function from interior pt: "); print_fun(start_fun);
/* */
/*
**          Check the poly_list for a polygon for start point.
**          (Shouldn't be necessary, just being careful).
*/
            curr_poly = search_poly_list(start_fun, start_pt, 0); 

            if  (curr_poly == NULL)
            {
/*
**              Didn't find a polygon,
**              look for a todo structure
*/
                curr_todo = get_fr_todo_list(start_fun, start_pt);
                if  (curr_todo != NULL)
                {
/* **               fprintf(log_fp, "found todo 2nd try in get_poly_for\n");
/* */
                    delete start_fun;
                    start_fun = NULL;
                    delete start_pt;
                    start_pt = NULL;
                }
                else
                {
                    curr_todo = set_up_todo(start_fun, start_pt);
		    // NOTE: ***Do NOT*** Add delete start_fun; here.
		    // set_up_todo() must not do a deep copy!
                    start_fun = NULL;
                    delete start_pt;
                    start_pt = NULL;
                }
            }
        }
    }
    if  (curr_poly == NULL)
    {
/*
**      Find the polygon (curr_todo is all set up).
*/
        curr_poly = find_poly(curr_todo);
    }
/*
**  Now make sure that all polygons adjacent at input_pt have been found.
**  (I.e., if the point is on an edge then we want to make sure that we
**    have already found the polygons on both sides of the edge, and
**    if the point is at a vertex, then we want to make sure that we
**    have already found and drawn all polygons that share the point as a
**    vertex.)
*/
    while ((curr_todo = get_fr_todo_list(new_fun, poly_pt)) != NULL)
        curr_poly = find_poly(curr_todo);

    curr_poly = search_poly_list(new_fun, poly_pt, 0);
    delete new_fun;
    delete poly_pt;
    new_fun = NULL;
    poly_pt = NULL;

/* **   fprintf(log_fp, "get_poly_for: end\n");
/* */
    return curr_poly;
}


/********************************************************************/
/*  These routines handle the mouse events occurring in the canvas. */
/*                                                                  */
/*       set_strings  -  Gets the two sequences.                    */
/*       explore      -  Draws the polygons on the screen.          */
/*                       Activated by the left and right mouse      */
/*                       buttons when in the canvas.                */
/*       resize       -  handles resize events.                     */
/*       expose       -  handles exposure events.                   */
/********************************************************************/


void set_strings(       int     change_type)
{
    char        *temp1,
                *temp2;
    int         processed_a_change = 0;

/* **   fprintf(log_fp, "get_strings: begin\n");
/* */
    temp1 = intTextGetString(string[0]);
    temp2 = intTextGetString(string[1]);
    if  (   (str1 == NULL)
         || (strcmp(temp1, (char *)str1) != 0))
    {
        len1 = strlen(temp1);
        delete str1;
        str1 = new unsigned char[len1 + 1];
        strcpy((char *)str1, temp1);
        processed_a_change = 1;
    }
    if  (   (str2 == NULL)
         || (strcmp(temp2, (char *)str2) != 0))
    {
        len2 = strlen(temp2);
        delete str2;
        str2 = new unsigned char[len2 + 1];
        strcpy((char *)str2, temp2);
        processed_a_change = 1;
    }

    if ((len1 != 0) || (len2 != 0))
	{
	if  ((processed_a_change) || (change_type & changed_ALIGN))
	    {
	    if (change_type & changed_STRINGS)
		{
		check_delete_input_fun_align();
		}
	    delete array;
	    array = new Max_node[(len1+1)*(len2+1)];
	    assert (array != NULL);
	    delete align_path;
	    align_path = new char[len1+len2+3];
	// Was mixing malloc and delete. Bad! Very bad!!
	    delete fpt;
	    fpt = new int[len2 + 1];
	    assert (fpt != NULL);

	    delete elist;
	    elist = new lpoint[len2+1];
	    assert (elist != NULL);
	    if (flist != NULL)
		{
	      for ( int i=1; i<=len2; i++) 
		    {
		    delete flist[i];
		    }
		delete flist;
		}

	    flist = new lpoint *[len2 + 1];
	    assert(flist !=  NULL);

	    for (int i=1; i<=len2; i++) 
		{
		flist[i] = new lpoint[len1 + 1];
		}
	    assert (flist[len2]!=NULL);
	    }
	}

    if ((len1 == 0) && (len2 == 0))
	{
	if (str1)
	    {
	    delete str1;
	    str1 = NULL;
	    }
	if (str2)
	    {
	    delete str2;
	    str2 = NULL;
	    }
	}
    changed_something = changed_NO;

/* **   fprintf(log_fp, "get_strings: end\n");
/* */
}

/* ARGSUSED */
static void     explore(        Widget          w,
                                XtPointer       info,
                                XEvent          *event,
                                int             n)
{
    Arg         wargs[1];
    char        buf[256];
    double      norm_pos_x,
                norm_pos_y;
    Point       *click_pt,
                *wk_pt;
    Poly        *curr_poly;

    if  ((changed_something & changed_STRINGS) > 0)
        set_strings(changed_STRINGS);

//--changed
    // If both strings are NULL, there are no sequences, and thus no polygons.
    // In this case, nothing should be done. Capturing this event prevents a
    // segment fault.
    // Mars
    if ((str1 == NULL) && (str2 == NULL))
	{
	return;
	}

    norm_pos_x = minval_x +
                 ((double)(event->xbutton.x - 20)/
                  (double)(my_data->width - 40)) *
                  side_x;
    norm_pos_y = minval_y +
                ((double)(my_data->height - 20 - event->xbutton.y) /
                 (double)(my_data->height - 40)) *
                 side_y;
    if  (   (norm_pos_x <= maxval_x)
         && (norm_pos_x >= minval_x) 
         && (norm_pos_y <= maxval_y)
         && (norm_pos_y >= minval_y))
    {
        if  (display_pt != NULL)
        {
            delete display_pt;
            display_pt = NULL;
            delete display_fun_pt;
            display_fun_pt = NULL;
            delete display_fun;
            display_fun = NULL;
        }
        click_pt = new Point(Global_v, Global_w);
        click_pt->x = norm_pos_x;
        click_pt->y = norm_pos_y;
        XDefineCursor(XtDisplay(toplevel), XtWindow(toplevel), hourglass);
        XFlush(my_data->disp);
        curr_poly = get_poly_for(click_pt);
        display_fun = find_max(click_pt,0);

        if  (event->xbutton.button == 2)
        {
            display_pt = new Point(click_pt);
            display_pt->next = NULL;
            display_fun_pt = new Point(click_pt);
            display_fun_pt->next = NULL;
        }
        else
        {
            display_pt = new Point(curr_poly->corners);
            display_pt->next = NULL;
            display_fun_pt = new Point(display_pt);
            wk_pt = curr_poly->corners->next;
            display_fun_pt->x += wk_pt->x;
            display_fun_pt->y += wk_pt->y;
            wk_pt = wk_pt->next;
            display_fun_pt->x += wk_pt->x;
            display_fun_pt->y += wk_pt->y;
            wk_pt = NULL;
            display_fun_pt->x /= 3.0;
            display_fun_pt->y /= 3.0;
        }
        delete click_pt;
	click_pt = NULL;

        display_fun = find_max_count(display_fun_pt,2);

        draw_dark_poly(curr_poly);

        if  (input_fun == NULL)
        {
            sprintf(buf, inputaligndiff_noinput,
                    display_pt->x, display_pt->y,
                    display_pt->max_val);
        }
        else
        {
            sprintf(buf, inputaligndiff,
                    display_pt->x, display_pt->y,
                    display_pt->max_val,
                    display_pt->input_fun_val,
                    display_pt->diff_val);
        }
        XtSetArg(wargs[0], XtNlabel, buf);
        XtSetValues(input_align_diff_label, wargs, 1);

        XUndefineCursor(XtDisplay(toplevel), XtWindow(toplevel));
        XFlush(my_data->disp);
    }
/* **   fprintf(log_fp, "explore: end\n");
/* */
}


/* ARGSUSED */
static void     resize( Widget          w,
                        XtPointer       client_data,
                        XEvent          event,
                        int             n)
{
/* **   fprintf(log_fp, "resize: begin\n");
/* */
    redraw(NULL, NULL, NULL);
/* **   fprintf(log_fp, "resize: end\n");
/* */
}


static void     expose( Widget          w,
                        XtPointer       data,
                        XEvent          *event,
                        int             n)
{
    if  (event->xexpose.count == 0)
        redraw(NULL, NULL, NULL);
}


// Sets the app to receive notification when the specified dialog box's close
// icon is clicked.
//
void TrapForDialogCloseBoxHit(Widget widget)
{
    Display	*dpy = XtDisplay(widget);
    Atom	wmDeleteMessage = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    XChangeProperty(dpy, XtWindow(widget),
		    XInternAtom(dpy, "WM_PROTOCOLS", False), XA_ATOM,
		    32, PropModeAppend, (unsigned char *)&wmDeleteMessage, 1);
}


/************************************************/
/*  alignment routines (cooptimals)             */
/************************************************/
//
// Common code for the handling of the previous and next buttons.
//
static void do_align_show_prev_next()
{
    Arg         wargs[1];
    char        buf[256];
    Poly        *curr_poly;

    calc_fun_from_path(display_fun);
    calc_align_from_path(array, display_fun);
    curr_poly = search_poly_list(display_fun, display_fun_pt, 0);
    draw_dark_poly(curr_poly);

    if  (input_fun == NULL)
    {
        sprintf(buf, inputaligndiff_noinput,
                display_pt->x, display_pt->y,
                display_pt->max_val);
    }
    else
    {
        sprintf(buf, inputaligndiff,
                display_pt->x, display_pt->y,
                display_pt->max_val,
                display_pt->input_fun_val,
                display_pt->diff_val);
    }
    XtSetArg(wargs[0], XtNlabel, buf);
    XtSetValues(input_align_diff_label, wargs, 1);

    XUndefineCursor(XtDisplay(toplevel), XtWindow(toplevel));
    XFlush(my_data->disp);
}

static void align_show_next(Widget w, char *client_data, XtPointer call_data)
{
    if  ((align_number == 0) || (display_fun == NULL)
    || (align_number >= display_fun->opt_align_ct))
	{
        XBell(dpy, 50);
	}
    else
	{
	// Get the next alignment.
	get_next_align_path(array);
	do_align_show_prev_next();
	}
}

//change me!
static void align_show_prev( Widget w, char *client_data, XtPointer call_data)
{
    if  (align_number <= 1)
        {
        XBell(dpy, 50);
        }
    else
	{
/*      Get previous alignment  */
        get_prev_align_path(array);
	do_align_show_prev_next();
	}
}



/************************************************/
/*  plane routines                              */
/************************************************/

typedef struct  {
                char    *label_c1,
                        *label_c2;
                } const_plane_struct;

const_plane_struct const_plane_labels[] =
                { {"                ", "                "},
                  {"   Match Coeff. c1 = ", NULL},
                  {"Mismatch Coeff. c1 = ", NULL},
                  {"  In/Del Coeff. c1 = ", NULL},
                  {"   Match Coeff. c1 = ", "Mismatch Coeff. c2 = "},
                  {"   Match Coeff. c1 = ", "  In/Del Coeff. c2 = "},
                  {"   Match Coeff. c1 = ", "     Gap Coeff. c2 = "},
                  {"Mismatch Coeff. c1 = ", "  In/Del Coeff. c2 = "},
                  {"Mismatch Coeff. c1 = ", "     Gap Coeff. c2 = "},
                  {"  In/Del Coeff. c1 = ", "     Gap Coeff. c2 = "}
                };

Widget          const_plane,
                const_plane_button,
                const_plane_label_c1,
                const_plane_label_c2,
                const_plane_edit_c1,
                const_plane_edit_c2;

static void 
const_plane_hide(Widget w, char *client_data, XtPointer call_data)
{
    XtPopdown(const_plane);
}


static void
const_plane_show(Widget w, char *client_data, XtPointer call_data)
{
    Arg         wargs[10];
    char        buf[32];
    Dimension   height,
                twidth,
                width;
    int         n;
    Position    absx,
                absy,
                x,
                y;

/* **   fprintf(log_fp, "const_plane_show: begin\n");
/* */

    n = 0;
    XtSetArg(wargs[n], XtNx, &absx); n++;
    XtSetArg(wargs[n], XtNy, &absy); n++;
    XtGetValues(toplevel, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, &x); n++;
    XtSetArg(wargs[n], XtNy, &y); n++;
    XtSetArg(wargs[n], XtNwidth, &width); n++;
    XtSetArg(wargs[n], XtNheight, &height); n++;
    XtGetValues(const_plane_button, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNwidth, &twidth); n++;
    XtGetValues(const_plane, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, absx + x + (width / 2) - (twidth / 2)); n++;
    XtSetArg(wargs[n], XtNy, absy + y + height + 3); n++;
    XtSetArg(wargs[n], XtNinput, (XtArgVal)TRUE); n++;
    XtSetValues(const_plane, wargs, n);

    assert (   (opt_function >= 1)
            && (opt_function <= 9));

    XtSetArg(wargs[0], XtNlabel, const_plane_labels[opt_function].label_c1);
    XtSetValues(const_plane_label_c1, wargs, 1);

    sprintf(buf, "%.2f", Global_v);
    intTextSetString(const_plane_edit_c1, buf);

    if  (const_plane_labels[opt_function].label_c2 == NULL)
    {
        XawTextDisplayCaret(const_plane_edit_c2, FALSE);
        XtSetArg(wargs[0], XtNlabel, const_plane_labels[0].label_c2);
        XtSetValues(const_plane_label_c2, wargs, 1);

        intTextSetString(const_plane_edit_c2, NULL);
    }
    else
    {
        XawTextDisplayCaret(const_plane_edit_c2, TRUE);
        XtSetArg(wargs[0], XtNlabel, const_plane_labels[opt_function].label_c2);
        XtSetValues(const_plane_label_c2, wargs, 1);

        sprintf(buf, "%.2f", Global_w);
        intTextSetString(const_plane_edit_c2, buf);
    }

    // Trap for hits to the const_plane dialog's close icon.
    TrapForDialogCloseBoxHit(const_plane);

    XtPopup(const_plane, XtGrabExclusive);

/* **   fprintf(log_fp, "const_plane_show: end\n");
/* */
}


static void
const_plane_ok(Widget w, char *client_data, XtPointer call_value)
{
    float       new_v,
                new_w;

/* **   fprintf(log_fp, "const_plane_ok: begin\n");
/* */

    if  (   (sscanf(intTextGetString(const_plane_edit_c1), "%f", &new_v) != 1)
         || (   (const_plane_labels[opt_function].label_c2 != NULL)
             && (sscanf(intTextGetString(const_plane_edit_c2), "%f", &new_w) != 1)))
    {
        XBell(dpy, 50);
        return;
    }

    Global_v = new_v;
    if  (const_plane_labels[opt_function].label_c2 != NULL)
        Global_w = new_w;

    XtPopdown(const_plane);

    process_change(changed_PARMS);

/* **   fprintf(log_fp, "const_plane_ok: end\n");
/* */
}


static Widget   const_plane_create(Widget  parent)
{
    Arg         wargs[10];
    int         n;
    Widget      command,
                form,
                label,
                left,
                top;


/* **   fprintf(log_fp, "const_plane_create: begin\n");
/* */

    const_plane_button = parent;
    const_plane = XtCreatePopupShell("const_plane", transientShellWidgetClass,
                                     parent, NULL, 0);

    form = create_form("form", const_plane, NULL, NULL, 0, 2, wargs, 0);
    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    top = create_label("  Enter the constant(s):", form, NULL, NULL, 0,
                       wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    label = const_plane_label_c1 = create_label("Mismatch Coeff. c1 = ", form,
                                                NULL, top, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    top = const_plane_edit_c1 = create_text("const_plane_edit_c1", form, label,
                                            top, "", 1, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    label = const_plane_label_c2 = create_label("Mismatch Coeff. c2 = ", form,
                                                NULL, top, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    top = const_plane_edit_c2 = create_text("const_plane_edit_c2", form, label,
                                            top, "", 1, 0, wargs, n);

    command = create_form("const_plane_command", form, NULL, top, 0, 4,
                          wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    left = create_command("Cancel", command, NULL, NULL,
                          const_plane_hide, NULL, wargs, n);
    create_command("Set Values", command, left, NULL, const_plane_ok, NULL,
                   wargs, n);

    XtRealizeWidget(const_plane);

/* **   fprintf(log_fp, "const_plane_create: end\n");
/* */
}


/**********************************************************************/
/*  get_polys_for_pt routines                                         */
/*    These routines bring up a window that allows the user           */
/*    to enter a specific point for which all polygons and            */
/*    (co-)optimal alignments are to be shown.                        */
/**********************************************************************/

Widget          get_polys_pt,
                get_polys_pt_button,
                get_polys_pt_x,
                get_polys_pt_y;

static void     get_polys_pt_hide(      Widget          w,
                                        char            *client_data,
                                        XtPointer       call_data)
{
    XtPopdown(get_polys_pt);
}


static void     get_polys_pt_show(  Widget          w,
                                    char            *client_data,
                                    XtPointer       call_data)
{
    Arg         args[10];
    char        buf[30];
    Dimension   height,
                twidth,
                width;
    Position    absx,
                absy,
                x,
                y;
    int         n;

/* **   fprintf(log_fp, "get_polys_pt_show: begin\n");
/* */

    n = 0;
    XtSetArg(args[n], XtNx, &absx); n++;
    XtSetArg(args[n], XtNy, &absy); n++;
    XtGetValues(toplevel, args, n);

    n = 0;
    XtSetArg(args[n], XtNx, &x); n++;
    XtSetArg(args[n], XtNy, &y); n++;
    XtSetArg(args[n], XtNwidth, &width); n++;
    XtSetArg(args[n], XtNheight, &height); n++;
    XtGetValues(get_polys_pt_button, args, n);

    n = 0;
    XtSetArg(args[n], XtNwidth, &twidth); n++;
    XtGetValues(get_polys_pt, args, n);

    n = 0;
    XtSetArg(args[n], XtNx, absx + x + (width / 2) - (twidth / 2)); n++;
    XtSetArg(args[n], XtNy, absy + y + height + 3); n++;
    XtSetArg(args[n], XtNinput, (XtArgVal)TRUE); n++;
    XtSetValues(get_polys_pt, args, n);

    // Trap for hits to the get_polys_pt dialog's close icon.
    TrapForDialogCloseBoxHit(get_polys_pt);

    XtPopup(get_polys_pt, XtGrabExclusive);
/* **   fprintf(log_fp, "get_polys_pt_show: end\n");
/* */
}


static void     get_polys_pt_ok(Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
/* **   fprintf(log_fp, "get_polys_pt_ok: begin\n");
/* */
    Arg         wargs[10];
    char        buf[256];
    int         got_valid_pt;
    float       pt_x,
                pt_y;
    Poly        *curr_poly;

    if  (   (sscanf(intTextGetString(get_polys_pt_x), "%f", &pt_x) != 1)
         || (pt_x < minval_x)
         || (pt_x > maxval_x)
         || (sscanf(intTextGetString(get_polys_pt_y), "%f", &pt_y) != 1)
         || (pt_y < minval_y)
         || (pt_y > maxval_y))
    {
        XBell(dpy, 50);
        return;
    }

    XtPopdown(get_polys_pt);

    if  (changed_something & changed_STRINGS)
        set_strings(changed_STRINGS);

    if  (display_pt != NULL)
    {
        delete display_pt;
        display_pt = NULL;
        delete display_fun_pt;
        display_fun_pt = NULL;
        delete display_fun;
        display_fun = NULL;
    }

    display_pt = new Point(Global_v, Global_w, pt_x, pt_y);
    curr_poly = get_poly_for(display_pt);

    display_fun_pt = new Point(display_pt);
    display_fun = find_max(display_fun_pt,1);
    draw_dark_poly(curr_poly);

    if  (input_fun == NULL)
    {
        sprintf(buf, inputaligndiff_noinput,
                display_pt->x, display_pt->y,
                display_pt->max_val);
    }
    else
    {
        sprintf(buf, inputaligndiff,
                display_pt->x, display_pt->y,
                display_pt->max_val,
                display_pt->input_fun_val,
                display_pt->diff_val);
    }
    XtSetArg(wargs[0], XtNlabel, buf);
    XtSetValues(input_align_diff_label, wargs, 1);

    XUndefineCursor(XtDisplay(toplevel), XtWindow(toplevel));
    XFlush(my_data->disp);

/* **   fprintf(log_fp, "get_polys_pt_ok: end\n");
/* */
}


static Widget   get_polys_pt_create(Widget  parent)
{
    Arg         wargs[10];
    int         n = 0;
    Widget      command,
                form,
                label,
                left,
                top;

    get_polys_pt_button = parent;
    get_polys_pt = XtCreatePopupShell("get_polys_pt", transientShellWidgetClass,
                                   parent, NULL, 0);

    form = create_form("form", get_polys_pt, NULL, NULL, 0, 2, wargs, 0);
    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    top = create_label("Please enter the desired point:", form, NULL,
                       NULL, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    label = create_label(" X value:", form, NULL, top, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    top = get_polys_pt_x = create_text("get_polys_pt_x", form, label,
                                       top, "", 1, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    label = create_label(" Y value:", form, NULL, top, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    top = get_polys_pt_y = create_text("get_polys_pt_y", form, label,
                                       top, "", 1, 0, wargs, n);

    command = create_form("get_polys_pt_command", form, NULL, top, 0, 4,
                          wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    left = create_command("Cancel", command, NULL, NULL,
                          get_polys_pt_hide, NULL, wargs, n);
    create_command("Use Entered Values", command, left, NULL, get_polys_pt_ok, NULL,
                   wargs, n);

    XtRealizeWidget(get_polys_pt);

/* **   fprintf(log_fp, "get_polys_pt_create: end\n");
/* */
}

/**********************************************************************/
/*  xy_region routines                                                */
/**********************************************************************/

Widget          xy_region,
                xy_region_button,
                xy_region_edit_max_x,
                xy_region_edit_max_y,
                xy_region_edit_min_x,
                xy_region_edit_min_y;


static void xy_region_hide(Widget w, char *client_data, XtPointer call_data)
{
    XtPopdown(xy_region);
}


static void     xy_region_show( Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    Arg         args[10];
    char        buf[30];
    Dimension   height,
                twidth,
                width;
    Position    absx,
                absy,
                x,
                y;
    int         n;

/* *   fprintf(log_fp, "xy_region_show: begin\n");
/* */

    n = 0;
    XtSetArg(args[n], XtNx, &absx); n++;
    XtSetArg(args[n], XtNy, &absy); n++;
    XtGetValues(toplevel, args, n);

    n = 0;
    XtSetArg(args[n], XtNx, &x); n++;
    XtSetArg(args[n], XtNy, &y); n++;
    XtSetArg(args[n], XtNwidth, &width); n++;
    XtSetArg(args[n], XtNheight, &height); n++;
    XtGetValues(xy_region_button, args, n);

    n = 0;
    XtSetArg(args[n], XtNwidth, &twidth); n++;
    XtGetValues(xy_region, args, n);

    n = 0;
    XtSetArg(args[n], XtNx, absx + x + (width / 2) - (twidth / 2)); n++;
    XtSetArg(args[n], XtNy, absy + y + height + 3); n++;
    XtSetArg(args[n], XtNinput, (XtArgVal)TRUE); n++;
    XtSetValues(xy_region, args, n);

    sprintf(buf, "%.2f", minval_x);
    intTextSetString(xy_region_edit_min_x, buf);
  
    sprintf(buf, "%.2f", maxval_x);
    intTextSetString(xy_region_edit_max_x, buf);
  
    sprintf(buf, "%.2f", minval_y);
    intTextSetString(xy_region_edit_min_y, buf);

    sprintf(buf, "%.2f", maxval_y);
    intTextSetString(xy_region_edit_max_y, buf);

    TrapForDialogCloseBoxHit(xy_region);

    XtPopup(xy_region, XtGrabExclusive);
/* **   fprintf(log_fp, "xy_region_show: end\n");
/* */
}


static void     xy_region_ok(   Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
/* **   fprintf(log_fp, "xy_region_ok: begin\n");
/* */
    int         xy_region_changed;
    float       new_maxx,
                new_maxy,
                new_minx,
                new_miny;

    if  (   (sscanf(intTextGetString(xy_region_edit_min_x), "%f", &new_minx) != 1)
         || (new_minx < 0.0)
         || (sscanf(intTextGetString(xy_region_edit_max_x), "%f", &new_maxx) != 1)
         || (new_maxx < 0.0)
         || (new_minx >= new_maxx)
         || (sscanf(intTextGetString(xy_region_edit_min_y), "%f", &new_miny) != 1)
         || (new_miny < 0.0)
         || (sscanf(intTextGetString(xy_region_edit_max_y), "%f", &new_maxy) != 1)
         || (new_maxy < 0.0)
         || (new_minx >= new_maxx))
    {
        XBell(dpy, 50);
        return;
    }

    if  (new_minx != minval_x)
    {
        minval_x = new_minx;
        xy_region_changed = 1;
    }
    if  (new_maxx != maxval_x)
    {
        maxval_x = new_maxx;
        xy_region_changed = 1;
    }
    if  (new_miny != minval_y)
    {
        minval_y = new_miny;
        xy_region_changed = 1;
    }
    if  (new_maxy != maxval_y)
    {
        maxval_y = new_maxy;
        xy_region_changed = 1;
    }

    if  (xy_region_changed)
    {
        side_x = maxval_x - minval_x;
        side_y = maxval_y - minval_y;

        xy_region_changed = 0;
        process_change(changed_PARMS);
    }

    XtPopdown(xy_region);
/* **   fprintf(log_fp, "xy_region_ok: end\n");
/* */
}


static Widget   xy_region_create(Widget  parent)
{
    Arg         wargs[10];
    int         n = 0;
    Widget      command,
                form,
                label,
                left,
                top;

    xy_region_button = parent;
    xy_region = XtCreatePopupShell("xy_region", transientShellWidgetClass,
                                   parent, NULL, 0);

    form = create_form("form", xy_region, NULL, NULL, 0, 2, wargs, 0);
    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    top = create_label("Please set the desired region values:", form, NULL,
                       NULL, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    label = create_label("Min X value:", form, NULL, top, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    top = xy_region_edit_min_x = create_text("xy_region_edit_min_x", form, label,
                                             top, "", 1, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    label = create_label("Max X value:", form, NULL, top, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    top = xy_region_edit_max_x = create_text("xy_region_edit_max_x", form, label,
                                             top, "", 1, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    label = create_label("Min Y value:", form, NULL, top, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    top = xy_region_edit_min_y = create_text("xy_region_edit_min_y", form, label,
                                             top, "", 1, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    label = create_label("Max Y value:", form, NULL, top, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    top = xy_region_edit_max_y = create_text("xy_region_edit_max_y", form, label,
                                             top, "", 1, 0, wargs, n);

    command = create_form("xy_region_command", form, NULL, top, 0, 4,
                          wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    left = create_command("Cancel", command, NULL, NULL,
                          xy_region_hide, NULL, wargs, n);
    create_command("Set Values", command, left, NULL, xy_region_ok, NULL,
                   wargs, n);

    XtRealizeWidget(xy_region);

/* **   fprintf(log_fp, "xy_region_create: end\n");
/* */
}


/********************************************************************/
/*  io error messages                                               */
/********************************************************************/

char    *io_msg_Cannot_open_file =
                        "Cannot open file.  Please reenter:",
        *io_msg_Enter_Filename =
                        "Enter Filename:",
        *io_msg_Enter_Filename_for_Save =
                        "Enter Filename for Save:",
        *io_msg_Enter_Filename_to_Load =
                        "Enter Filename to Load:",
        *io_msg_Enter_Filename_for_Scoring_Matrix =
                        "Enter Filename for Scoring Matrix:",
        *io_msg_Enter_Log_Filename =
                        "Enter Log Filename:",
        *io_msg_Enter_String1_Label =
                        "Enter String #1 Label:",
        *io_msg_Enter_String2_Label =
                        "Enter String #2 Label:",
        *io_msg_File_not_correct_format =
                        "File not correct format.  Please reenter:",
        *io_msg_File_incorrect_type =
                        "File incorrect type.  Please reenter:",
        *io_msg_File_not_found =
                        "File not found.  Please reenter:",
        *io_msg_File_not_found_or_invalid =
                        "File not found or invalid.  Please reenter:",
        *io_msg_Invalid_filename =
                        "Invalid filename.  Please reenter:",
        *io_msg_Label_for_String1_not_found =
                        "Label for String #1 not Found.  Please reenter:",
        *io_msg_Label_for_String2_not_found =
                        "Label for String #2 not Found.  Please reenter:",
        *io_msg_Missing_string_name =
                        "Missing string name.  Please reenter:",
        *io_msg_String_alignments_different_lengths =
                        "String alignments different lengths.  Please reenter:",
        *io_msg_String_alignments_invalid =
                        "String alignments invalid.  Please reenter:",
        *io_msg_String_not_found_or_invalid =
                        "String not found or invalid.  Please reenter:",
        *io_msg_Strings_must_be_different =
                        "Strings must be different.  Please reenter:";

/********************************************************************/
/*  align_io routines                                               */
/********************************************************************/


Widget          align_io,
                align_io_button,
                align_io_edit,
                align_io_label;

static void     align_io_ok(Widget, char*, XtPointer);
static void     align_io_hide(Widget, char*, XtPointer);

static String   align_io_translations =
                        "Ctrl<Key>J: align_io_ok() \n"
                        "Ctrl<Key>M: align_io_ok() \n"
                        "<Key>Linefeed: align_io_ok() \n"
                        "<Key>Return: align_io_ok()\n"
			"Ctrl<Key>.: align_io_hide()";

static XtActionsRec     align_io_actions[] =
                { {"align_io_ok", (XtActionProc) align_io_ok},
		  {"align_io_hide", (XtActionProc) align_io_hide},
                };


static void     align_io_hide(  Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    XtPopdown(align_io);
}


static void     align_io_show(  Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    Arg         wargs[10];
    char        buf[30];
    Dimension   h;
    int         n;
    Position    absx,
                absy,
                x,
                y;

/* **   fprintf(log_fp, "align_io_show: begin\n");
/* */

    align_io_function = client_data[0];

    assert (   (align_io_function == align_io_function_ALL_POLYS)
            || (align_io_function == align_io_function_DARK_POLY)
            || (align_io_function == align_io_function_LOG)
            || (align_io_function == align_io_function_MIN_D_POLY)
            || (align_io_function == align_io_function_POLY_ALIGNS));

    if  (   (   (align_io_function == align_io_function_ALL_POLYS)
             && (poly_list == NULL))
         || (   (align_io_function == align_io_function_DARK_POLY)
             && (dark_poly == NULL))
         || (   (align_io_function == align_io_function_MIN_D_POLY)
             && (input_fun == NULL))
         || (   (align_io_function == align_io_function_POLY_ALIGNS)
             && (dark_poly == NULL)))
    {
        XBell(dpy, 50);
        return;
    }

    n = 0;
    XtSetArg(wargs[n], XtNx, &absx); n++;
    XtSetArg(wargs[n], XtNy, &absy); n++;
    XtGetValues(toplevel, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, &x); n++;
    XtSetArg(wargs[n], XtNy, &y); n++;
    XtSetArg(wargs[n], XtNheight, &h); n++;
    XtGetValues(align_io_button, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, absx + x + 3); n++;
    XtSetArg(wargs[n], XtNy, absy + y + h + 3); n++;
    XtSetArg(wargs[n], XtNinput, (XtArgVal)TRUE); n++;
    XtSetValues(align_io, wargs, n);

    if  (align_io_function == align_io_function_LOG)
        XtSetArg(wargs[0], XtNlabel, io_msg_Enter_Log_Filename);
    else
        XtSetArg(wargs[0], XtNlabel, io_msg_Enter_Filename_for_Save);
    XtSetValues(align_io_label, wargs, 1);

    // Trap for hits to the align_io dialog's close icon.
    TrapForDialogCloseBoxHit(align_io);

    XtPopup(align_io, XtGrabExclusive);

/* **   fprintf(log_fp, "align_io_show: end\n");
/* */
}


static void     align_io_output_align(  char    *out_align1,
                                        char    *out_align2,
                                        FILE    *out_fp)
{
    int         str_length;

    fprintf(out_fp, "Alignment:\n");
    str_length = strlen(out_align1);
    while (str_length >= 50)
    {
        fprintf(out_fp, "\t%.50s\n\t%.50s\n\n", out_align1, out_align2);
        out_align1 += 50;
        out_align2 += 50;
        str_length -= 50;
    }
    if  (str_length > 0)
    {
        fprintf(out_fp, "\t%s\n\t%s\n\n", out_align1, out_align2);
    }
}


static void     align_io_output_poly(   Poly    *out_poly,
                                        FILE    *out_fp)
{
    Funct       *out_fun;
    Point       *out_pt,
                *eval_pt;
    char        *out_str1,
                *out_str2;
    int         str_length;

    fprintf(out_fp, "Polygon corners are:\n");
    out_pt = out_poly->corners;
    while (out_pt != NULL)
    {
        fprintf(out_fp, "\tPoint:\t   x=%24.16le\t   y=%24.16le\n",
                out_pt->x, out_pt->y);
        fprintf(out_fp, "\t\tc1=v=%24.16le\tc2=w=%24.16le\n",
                out_pt->v, out_pt->w);
        fprintf(out_fp, "\t\tValue of Optimal Alignment: %24.16le\n",
                out_pt->max_val);
        if  (input_fun != NULL)
        {
            fprintf(out_fp, "\t\tValue of   Input Alignment: %24.16le\n",
                    out_pt->input_fun_val);
            fprintf(out_fp, "\t\tDifference:                 %24.16le\n",
                    out_pt->diff_val);
        }
        fprintf(out_fp, "\n");
        out_pt = out_pt->next;
    }
    out_pt = out_poly->corners;
    fprintf(out_fp, "\tPoint:\t   x=%24.16le\t   y=%24.16le\n",
            out_pt->x, out_pt->y);
    fprintf(out_fp, "\t\tc1=v=%24.16le\tc2=w=%24.16le\n\n",
            out_pt->v, out_pt->w);

    eval_pt = new Point(out_pt);
    eval_pt->next = NULL;
    out_pt = out_pt->next;
    eval_pt->x += out_pt->x;
    eval_pt->y += out_pt->y;
    out_pt = out_pt->next;
    eval_pt->x += out_pt->x;
    eval_pt->y += out_pt->y;
    eval_pt->x /= 3.0;
    eval_pt->y /= 3.0;

    out_fun = find_max(eval_pt, 1);
    get_first_align_path(array);

    do {
        calc_fun_from_path(out_fun);
        calc_align_from_path(array, out_fun);

        fprintf(out_fp, "\n");
        if  (align_io_function == align_io_function_POLY_ALIGNS)
        {
            fprintf(out_fp, "Alignment %4d of %4d:\n",
                    align_number, out_fun->opt_align_ct);
            fprintf(out_fp, "    %s\n    %s\n",
                    out_fun->align1, out_fun->align2);
        }
        else
        {
            fprintf(out_fp, "    Function for Alignment %4d of %4d:\n",
                    align_number, out_fun->opt_align_ct);
            fprintf(out_fp, "\tMatches:        %10d\tValue: %24.16le\n",
                    out_fun->match_ct, out_fun->match_val);
            fprintf(out_fp, "\tMismatches:     %10d\tValue: %24.16le\n",
                    out_fun->mis_ct, out_fun->mis_val);
            fprintf(out_fp, "\tInsert/Deletes: %10d\tValue: %24.16le\n",
                    out_fun->inordel_ct, out_fun->inordel_val);
            if  (with_gaps)
                fprintf(out_fp, "\tGaps:           %10d\tValue: %24.16le\n",
                        out_fun->gaps_ct, out_fun->gaps_val);

            fprintf(out_fp, "\n");
            align_io_output_align(out_fun->align1, out_fun->align2, out_fp);

        }
    } while (get_next_align_path(array));

    fprintf(out_fp, "\n");

    delete out_fun;
    delete eval_pt;
    out_fun = NULL;
    eval_pt = NULL;
}


static int      align_io_log(   char    *filename)
{
    struct stat sbuf;

/* **   fprintf(log_fp, "align_io_log: begin\n");
/* */

    if  (*filename == '\0')
    {
        if  (log_fname != NULL)
        {
            delete log_fname;
            log_fname = NULL;
            fclose(log_fp);
            log_fp = stdout;
        }
    }
    else
    {
        if  (log_fname != NULL)
            if  (strcmp(filename, log_fname))
            {
                delete log_fname;
                log_fname = NULL;
                fclose(log_fp);
            }
        if  (log_fname == NULL)
        {
            log_fname = new char[strlen(filename) + 1];
            strcpy(log_fname, filename);
            if  (stat(log_fname, &sbuf) > -1)
            {
	      /* ANSI standard file test */
	      if (S_ISREG(sbuf.st_mode))
                    log_fp = fopen(log_fname, "w");
            }
            else
                if  (errno == ENOENT)
                    log_fp = fopen(log_fname, "w");
        }
    }

    if  (log_fp == NULL)
    {
        delete log_fname;
        log_fname = NULL;
        log_fp = stdout;
        fprintf(stderr, "Error in opening file %s\n", filename);
        return 1;
    }

/* **   fprintf(log_fp, "align_io_log: end\n");
/* */
    return 0;
}


static int      align_io_save(  char    *filename)
{
    char        *out_str;
    int         str_length;
    int         need_to_close_io_file = 0;
    Poly        *io_poly;
    struct stat sbuf;
    FILE        *fp = NULL;

/* **   fprintf(log_fp, "align_io_save: begin\n");
/* */
/* **   fprintf(log_fp, "Saving Current Alignment to file %s\n", filename);
/* */
    if  (*filename == '\0')
    {
        fp = stdout;
    }
    else
    {
        if  (log_fname != NULL)
        {
            if  (!strcmp(filename, log_fname))
            /*  => file names are the same */
            {
                fp = log_fp;
            }
        }
        if  (fp == NULL)
        {
            need_to_close_io_file = 1;
            if  (stat(filename, &sbuf) > -1)
            {
	      /* ANSI standard file test */
                if (S_ISREG(sbuf.st_mode))
                    fp = fopen(filename,"a");
            }
            else
            {
                if  (errno == ENOENT)
                    fp = fopen(filename,"a");
            }
        }
    }

    if  (fp == NULL)
    {
        fprintf(stderr, "Error in opening file %s\n", filename);
        return 1;
    }
    fprintf(fp, "Alignment information:\n\n");

    fprintf(fp, "String 1:\n");
    out_str = intTextGetString(string[0]);
    str_length = strlen(out_str);
    while (str_length > 50)
    {
        fprintf(fp, "\t%.50s\n", out_str);
        out_str += 50;
        str_length -= 50;
    }
    if  (str_length > 0)
    {
        fprintf(fp, "\t%s\n", out_str);
    }

    fprintf(fp, "String 2:\n");
    out_str = intTextGetString(string[1]);
    str_length = strlen(out_str);
    while (str_length > 50)
    {
        fprintf(fp, "\t%.50s\n", out_str);
        out_str += 50;
        str_length -= 50;
    }
    if  (str_length > 0)
    {
        fprintf(fp, "\t%s\n", out_str);
    }
    fprintf(fp, "\n");

    if  (weights_in_array)
        fprintf(fp, "%s\n\n", weight_matrix_name);
    else
        fprintf(fp, "%s\n\n", weightoptionconst);

    if  (input_fun == NULL)
    {
        fprintf(fp, "No Input Alignment\n\n");
    }
    else
    {
        fprintf(fp, "Input Alignment Function:\n");
        fprintf(fp, "\tMatches:        %10d\tValue: %24.16le\n",
                input_fun->match_ct, input_fun->match_val);
        fprintf(fp, "\tMismatches:     %10d\tValue: %24.16le\n",
                input_fun->mis_ct, input_fun->mis_val);
        fprintf(fp, "\tInsert/Deletes: %10d\tValue: %24.16le\n",
                input_fun->inordel_ct, input_fun->inordel_val);
        if  (with_gaps)
            fprintf(fp, "\tGaps:           %10d\tValue: %24.16le\n",
                    input_fun->gaps_ct, input_fun->gaps_val);
        fprintf(fp, "\n");

        fprintf(fp, "Input ");
        align_io_output_align(input_fun->align1, input_fun->align2, fp);
    }
    fprintf(fp, "\n");

    fprintf(fp, "Alignment Options:  %s\n", scoring_method_setting[scoring_method-1]);
    if  (input_fun == NULL)
    {
        fprintf(fp, "                    %s\n", varsetting[opt_function - 1]);
    }
    else
    {
        fprintf(fp, "                    %s\n", varsetting[opt_function + 8]);
    }
    fprintf(fp, "                    ");
    fprintf(fp, gapsetting[with_gaps], Global_v, Global_w);
    fprintf(fp, "\n\n");

    fprintf(fp, "X-Y Region: X: %g to %g\n", minval_x, maxval_x);
    fprintf(fp, "            Y: %g to %g\n\n", minval_y, maxval_y);

    if  (   (align_io_function == align_io_function_DARK_POLY)
         || (align_io_function == align_io_function_POLY_ALIGNS))
    {
        io_poly = dark_poly;
        fprintf(fp, "\n");
        align_io_output_poly(io_poly, fp);
    }
    else if  (align_io_function == align_io_function_MIN_D_POLY)
    {
        if  (poly_list != NULL)
        {
            io_poly = poly_list;
            while  (   (io_poly != NULL)
                    && (equal(io_poly->diff_opt_pt, poly_list->diff_opt_pt)))
            {
                fprintf(fp, "\n");
                align_io_output_poly(io_poly, fp);
                io_poly = io_poly->next;
            }
        }
    }
    else if  (align_io_function == align_io_function_ALL_POLYS)
    {
        io_poly = poly_list;
        while  (io_poly != NULL)
        {
            fprintf(fp, "\n");
            align_io_output_poly(io_poly, fp);
            io_poly = io_poly->next;
        }
    }

    fprintf(fp, "\n");

    if  (need_to_close_io_file)
        fclose(fp);

/* **   fprintf(log_fp, "align_io_save: end\n");
/* */
    return 0;
}


static void     align_io_ok(    Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    Arg         wargs[1];
    int         len,
                io_error;
    char        *io_filename,
                fnamebuf[1024];
    struct stat sbuf;

/* **   fprintf(log_fp, "align_io_ok: begin\n");
/* */
    io_filename = intTextGetString(align_io_edit);

    if  (io_filename[0] != '~')
      strcpy(fnamebuf, io_filename);
    else {
      strcpy(fnamebuf, getenv("HOME"));
      len = strlen(fnamebuf);
      fnamebuf[len++] = '/';
      strcpy(fnamebuf + len, io_filename + 1);
    }

    if  (align_io_function == align_io_function_LOG)
        io_error = align_io_log(fnamebuf);
    else
        io_error = align_io_save(fnamebuf);

    if  (!io_error)
        align_io_hide(NULL, NULL, NULL);
    else
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_Cannot_open_file);
        XtSetValues(align_io_label, wargs, 1);

        XBell(dpy, 50);
    }

/* **   fprintf(log_fp, "align_io_ok: end\n");
/* */
}


static void     align_io_create(        Widget  parent)
{
    Arg         wargs[15];
    int         n;
    Widget      command,
                form,
                left;

/* **   fprintf(log_fp, "align_io_create: begin\n");
/* */

    align_io_button = parent;
    align_io = XtCreatePopupShell("align_io", transientShellWidgetClass, parent,
                             NULL, 0);

    form = create_form("form", align_io, NULL, NULL, 0, 2, wargs, 0);
    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    align_io_label = create_label("Cannot open file.  Please reenter:", form,
                                  NULL, NULL, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNhorizDistance, 4), n++;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    XtSetArg(wargs[n], XtNwidth, 250); n++;
    align_io_edit = create_text("align_io_edit", form, NULL, align_io_label,
                                "", 1, 0, wargs, n);

    XtAppAddActions(app, align_io_actions, XtNumber(align_io_actions));
    XtOverrideTranslations(align_io_edit,
                           XtParseTranslationTable(align_io_translations));

    command = create_form("align_io_command", form, NULL, align_io_edit, 0, 4,
                          wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    left = create_command("Cancel", command, NULL, NULL, align_io_hide, NULL,
                          wargs, n);
    create_command("OK", command, left, NULL, align_io_ok, NULL, wargs, n);

    XtRealizeWidget(align_io);

/* **   fprintf(log_fp, "align_io_create: end\n");
/* */
}


/********************************************************************/
/*  input_align_io routines                                         */
/********************************************************************/

Widget          input_align_io,
                input_align_io_button,
                input_align_io_edit1,
                input_align_io_edit2,
                input_align_io_edit3,
                input_align_io_label1,
                input_align_io_label2,
                input_align_io_label3;

static void     input_align_io_ok(Widget, char*, XtPointer);
static void     input_align_io_hide(Widget, char*, XtPointer);

static String   input_align_io_translations =
                        "Ctrl<Key>J: input_align_io_ok() \n"
                        "Ctrl<Key>M: input_align_io_ok() \n"
                        "<Key>Linefeed: input_align_io_ok() \n"
                        "<Key>Return: input_align_io_ok()\n"
			"Ctrl<Key>.: input_align_io_hide()";

static XtActionsRec     input_align_io_actions[] =
                        { {"input_align_io_ok", (XtActionProc) input_align_io_ok},
			  {"input_align_io_hide", (XtActionProc) input_align_io_hide}
                        };



static void     input_align_io_hide(    Widget          w,
                                        char            *client_data,
                                        XtPointer       call_data)
{
/* **   fprintf(log_fp, "input_align_io_hide: begin\n");
/* */

    XtPopdown(input_align_io);

/* **   fprintf(log_fp, "input_align_io_hide: end\n");
/* */
}


static void     input_align_io_show(    Widget          w,
                                        char            *client_data,
                                        XtPointer       call_data)
{
    Arg         wargs[10];
    char        buf[30];
    Dimension   h;
    int         n;
    Position    absx,
                absy,
                x,
                y;

/* **   fprintf(log_fp, "input_align_io_show: begin\n");
/* */

    n = 0;
    XtSetArg(wargs[n], XtNx, &absx); n++;
    XtSetArg(wargs[n], XtNy, &absy); n++;
    XtGetValues(toplevel, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, &x); n++;
    XtSetArg(wargs[n], XtNy, &y); n++;
    XtSetArg(wargs[n], XtNheight, &h); n++;
    XtGetValues(input_align_io_button, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, absx + x + 3); n++;
    XtSetArg(wargs[n], XtNy, absy + y + h + 3); n++;
    XtSetArg(wargs[n], XtNinput, (XtArgVal)TRUE); n++;
    XtSetValues(input_align_io, wargs, n);

    XtSetArg(wargs[0], XtNlabel, io_msg_Enter_Filename);
    XtSetValues(input_align_io_label1, wargs, 1);

    XtSetArg(wargs[0], XtNlabel, io_msg_Enter_String1_Label);
    XtSetValues(input_align_io_label2, wargs, 1);

    XtSetArg(wargs[0], XtNlabel, io_msg_Enter_String2_Label);
    XtSetValues(input_align_io_label3, wargs, 1);

    // Trap for hits to the input_align_io dialog's close icon.
    TrapForDialogCloseBoxHit(input_align_io);

    XtPopup(input_align_io, XtGrabExclusive);

/* **   fprintf(log_fp, "input_align_io_show: end\n");
/* */
}


static int      input_align_io_get_str_len(     char    *str_name,
                                                char    *input_text)
{
    int         str_name_len = strlen(str_name);
    int         str_length = 0;
    int         str_has_characters = 0;

/* **   fprintf(log_fp, "input_align_io_get_str_len: begin\n");
/* */

    while (*input_text != '\0')
    {
/*
**      This is the start of a new line -- check for string name
*/
        while (*input_text == ' ')
            input_text++;
        if  (*input_text == '\0')
            break;
        if  (   (strncasecmp(input_text, str_name, str_name_len) == 0)
             && (   ((*(input_text + str_name_len)) == ' ')
                 || ((*(input_text + str_name_len)) == '\"')))
        {
            input_text += str_name_len;
/*
**          We got a string name match -- get the length of this segment
**          First get to the beginning of the segment (or end of the line)
*/
            while (   (*input_text != '\"')
                   && (*input_text != '\n')
                   && (*input_text != '\0'))
                input_text++;

            if  (*input_text == '\"')
            {
/*
**              Found the beginning of a segment -- process it
*/
                input_text++;
                while (   (*input_text != '\"')
                       && (*input_text != '\n')
                       && (*input_text != '\0'))
                {
                    str_length++;
                    input_text++;
                    if  (   (*input_text != ' ')
                         && (*input_text != '[')
                         && (*input_text != ']'))
                        str_has_characters = 1;
                }
            }
        }
/*
**      Get to the end of the line if not already there.
*/
        while (   (*input_text != '\n')
               && (*input_text != '\0'))
            input_text++;
/*
**      Now make sure at the beginning of the next line
*/
        while (*input_text == '\n')
            input_text++;
    }
/*
**  Return 0 if no characters found, otherwise return length
*/
    if  (!str_has_characters)
        return 0;

/* **   fprintf(log_fp, "input_align_io_get_str_len: end\n");
/* */

    return str_length;
}


static char     *input_align_io_get_str(        char    *str_name,
                                                int     str_len,
                                                char    *input_text)
{
    int         str_name_len = strlen(str_name);
    char        *new_align_str,
                *new_align_str_wk;

/* **   fprintf(log_fp, "input_align_io_get_str: begin\n");
/* */

    new_align_str_wk = new_align_str = new char[str_len + 1];
    while (*input_text != '\0')
    {
/*
**      This is the start of a new line -- check for string name
*/
        while (*input_text == ' ')
            input_text++;
        if  (*input_text == '\0')
            break;
        if  (   (strncasecmp(input_text, str_name, str_name_len) == 0)
             && (   ((*(input_text + str_name_len)) == ' ')
                 || ((*(input_text + str_name_len)) == '\"')))
        {
            input_text += str_name_len;
/*
**          We got a string name match -- get the length of this segment
**          First get to the beginning of the segment (or end of the line)
*/
            while (   (*input_text != '\"')
                   && (*input_text != '\n')
                   && (*input_text != '\0'))
                input_text++;

            if  (*input_text == '\"')
            {
/*
**              Found the beginning of a segment -- process it
*/
                input_text++;
                while (   (*input_text != '\"')
                       && (*input_text != '\n')
                       && (*input_text != '\0'))
                {
                    *new_align_str_wk = *input_text;
                    new_align_str_wk++;
                    input_text++;
                }
            }
        }
/*
**      Get to the end of the line if not already there.
*/
        while (   (*input_text != '\n')
               && (*input_text != '\0'))
            input_text++;
/*
**      Now make sure at the beginning of the next line
*/
        while (*input_text == '\n')
            input_text++;
    }
/*
**  Mark the end of the string and return it
*/
    *new_align_str_wk = '\0';

/* **   fprintf(log_fp, "input_align_io_get_str: end\n");
/* */

    return new_align_str;
}

inline
static int      input_align_io_get_align_len(   char    *str1,
                                                char    *str2)
{
    int         align_length = 0;
    int         found_local_alignment_start = 0,
                found_local_alignment_end = 0;

/* **   fprintf(log_fp, "input_align_io_get_align_len: begin\n");
/* */

    while (*str1 != '\0')
    {
        if  (   (*str1 != ' ')
             || (*str2 != ' '))
        {
/*
**          At least one of the strings has a non-space equivalent
**          Make sure local alignment is ok
*/
            if  (*str1 == '[')
            {
                if  (   (found_local_alignment_start)
                     || (*str2 != '['))
/*                  can't have 2 local alignments */
/*                  and local alignment has to match up */
                    return 0;
                found_local_alignment_start = 1;
            }
            if  (*str1 == ']')
            {
                if  (   (!found_local_alignment_start)
                     || (found_local_alignment_end)
                     || (*str2 != ']'))
/*                  local alignment can't end before it starts */
/*                  and local alignment can't have two ends */
/*                  and local alignment has to match up */
                    return 0;
                found_local_alignment_end = 1;
            }
            align_length++;
        }
        str1++;
        str2++;
    }

/* **   fprintf(log_fp, "input_align_io_get_align_len: end\n");
/* */

    return align_length;
}

inline
static void         input_align_io_compact_align(       char    *instr1,
                                                        char    *instr2,
                                                        char    *outstr1,
                                                        char    *outstr2,
                                                        int     have_local)
{
    char        *instr1_wk = instr1;

/* **   fprintf(log_fp, "input_align_io_compact_align: begin\n");
/* */

    if  (!have_local)
    {
        *outstr1 = '[';
        *outstr2 = '[';
        outstr1++;
        outstr2++;
    }
    while (*instr1 != '\0')
    {
        if  (   (*instr1 != ' ')
             || (*instr2 != ' '))
        {
            *outstr1 = *instr1;
            *outstr2 = *instr2;
            outstr1++;
            outstr2++;
        }
        instr1++;
        instr2++;
    }
    if  (!have_local)
    {
        *outstr1 = ']';
        *outstr2 = ']';
        outstr1++;
        outstr2++;
    }
    *outstr1 = '\0';
    *outstr2 = '\0';

/* **   fprintf(log_fp, "input_align_io_compact_align: end\n");
/* */
}


static void     input_align_io_compact_str(     char    *str)
{
    char        *str_wk;

/* **   fprintf(log_fp, "input_align_io_compact_str: begin\n");
/* */

    while (   (*str != ' ')
           && (*str != '[')
           && (*str != '\0'))
        str++;
    if  (*str != '\0')
    {
        str_wk = str;
        do {
            str_wk++;
            if  (   (*str_wk != ' ')
                 && (*str_wk != '[')
                 && (*str_wk != ']'))
            {
                *str = *str_wk;
                str++;
            }
        } while (*str_wk != '\0');
    }

/* **   fprintf(log_fp, "input_align_io_compact_str: end\n");
/* */
}

inline
static int      input_align_io_load(    char            *filename,
                                        struct stat     *filebuf,
                                        char            *str1_name,
                                        char            *str2_name)
{
    int         have_local = 0,
                line_string_number,
                str1_align_len = 0,
                str2_align_len = 0,
                tempstr1_len = 0,
                tempstr2_len = 0;
    char        *file_io_str,
                *tempstr1,
                *tempstr1_wk,
                *tempstr2;
    FILE        *fp;
    size_t	result;

/* **   fprintf(log_fp, "input_align_io_load: begin\n");
/* */
/* **   fprintf(log_fp, "Loading Strings %s and %s from file %s\n",
                str1_name, str2_name, filename);
/* */

    if  ((fp = fopen(filename,"r")) == NULL)
    {
        fprintf(log_fp,
	    "Error in opening file %s to load aligned strings\n",
	    filename);
        return 1;
    }
    file_io_str = new char[filebuf->st_size + 1];
    result = fread(file_io_str, 1, filebuf->st_size, fp);
    if (result != filebuf->st_size)
	{
	fprintf(log_fp,
	    "Error reading info from file. Expected %ld bytes, retrieved %ld bytes\n",
	    filebuf->st_size, result);
	fclose(fp);
	return -1;
	}
    fclose(fp);
    file_io_str[filebuf->st_size] = '\0';

/*
**  Pre-process the string so all in/dels are ' '
**  and all line changes are '\n'
*/
    tempstr1_wk = file_io_str;
    while (*tempstr1_wk != '\0')
    {
        if  (*tempstr1_wk != '\n')
        {
            if  (   (*tempstr1_wk == '\f')
                 || (*tempstr1_wk == '\r'))
                *tempstr1_wk = '\n';
            else if  (   (isspace(*tempstr1_wk))
                      || (*tempstr1_wk == '-')
                      || (*tempstr1_wk == '_')
                      || (*tempstr1_wk == '~'))
                *tempstr1_wk = ' ';
        }
        tempstr1_wk++;
    }
/*
**  Now get the lengths of the two strings, then get the alignment
*/
    str1_align_len = input_align_io_get_str_len(str1_name, file_io_str);
    str2_align_len = input_align_io_get_str_len(str2_name, file_io_str);

    if  (str1_align_len == 0)
    {
        fprintf(log_fp,
                "String %s not found or invalid in file %s\n",
                str1_name, filename);
        return 2;
    }
    if  (str2_align_len == 0)
    {
        fprintf(log_fp,
                "String %s not found or invalid in file %s\n",
                str2_name, filename);
        return 3;
    }
    if  (str1_align_len != str2_align_len)
    {
        fprintf(log_fp,
                "Alignments of strings %s and %s not same length\n",
                str1_name, str2_name);
        return 4;
    }
/*
**  Now have the lengths of the aligned strings,
**  so allocate them and get them.
*/
    tempstr1 = input_align_io_get_str(str1_name, str1_align_len, file_io_str);
    tempstr2 = input_align_io_get_str(str2_name, str2_align_len, file_io_str);

    delete file_io_str;
    file_io_str = NULL;

/*
**  Now check the alignments for validity and pairwise length
**  (validity is to ensure that at most one local alignment in the
**   same positions in both strings)
*/
    str1_align_len = input_align_io_get_align_len(tempstr1, tempstr2);

    if  (str1_align_len == 0)
    {
        fprintf(log_fp,
                "Local alignments for strings %s and %s invalid\n",
                str1_name, str2_name);
        return 5;
    }

/*
**  Have valid alignment -- save it
**  First check if the alignment has a local designation --
**  If not we will use global as local
*/
    tempstr1_wk = tempstr1;
    while (*tempstr1_wk != '\0')
    {
        if  (*tempstr1_wk == '[')
        {
            have_local = 1;
            break;
        }
        tempstr1_wk++;
    }
    if  (!have_local)
        str1_align_len += 2;

    if  (input_fun != NULL)
    {
        input_fun->match_ct = 0;
        input_fun->mis_ct = 0;
        input_fun->inordel_ct = 0;
        input_fun->gaps_ct = 0;
        input_fun->match_val = 0.0;
        input_fun->mis_val = 0.0;
        input_fun->inordel_val = 0.0;
        input_fun->gaps_val = 0.0;
        input_fun->is_2D_opt = 0;

        delete input_fun->align1;
        delete input_fun->align2;
	input_fun->align1 = NULL;
	input_fun->align2 = NULL;
    }
    else
    {
        input_fun = new Funct;
    }
    input_fun->align1 = new char[str1_align_len + 1];
    input_fun->align2 = new char[str1_align_len + 1];

    input_align_io_compact_align(tempstr1, tempstr2,
                               input_fun->align1, input_fun->align2,
                               have_local);

/*
**  Now reset tempstr1 and tempstr2 to hold the alignment strings
**  (minus the ' ','[', and ']')
*/
    input_align_io_compact_str(tempstr1);
    input_align_io_compact_str(tempstr2);

/*
**  Set the strings
*/
    intTextSetString(string[0], tempstr1);
    intTextSetString(string[1], tempstr2);

    delete str1;
    len1 = strlen(tempstr1);
    str1 = new unsigned char[len1 + 1];
    strcpy((char *)str1, tempstr1);

    delete str2;
    len2 = strlen(tempstr2);
    str2 = new unsigned char[len2 + 1];
    strcpy((char *)str2, tempstr2);

    delete tempstr1;
    delete tempstr2;
    tempstr1 = NULL;
    tempstr2 = NULL;

    process_change(changed_ALIGN);

    return 0;

/* **   fprintf(log_fp, "input_align_io_load: end\n");
/* */
}


static void     input_align_io_ok(      Widget          w,
                                        char            *client_data,
                                        XtPointer       call_data)
{
    Arg         wargs[1];
    int         len,
                io_error;
    char        *io_filename,
                fnamebuf[1024];
    char        *io_str1_name,
                *io_str2_name;
    struct stat sbuf;

/* **   fprintf(log_fp, "input_align_io_ok: begin\n");
/* */

    io_filename = intTextGetString(input_align_io_edit1);
    io_str1_name = intTextGetString(input_align_io_edit2);
    io_str2_name = intTextGetString(input_align_io_edit3);

    if  (io_filename[0] == '\0')
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_Invalid_filename);
        XtSetValues(input_align_io_label1, wargs, 1);

        XBell(dpy, 50);
        return;
    }

    if  (io_filename[0] != '~')
        strcpy(fnamebuf, io_filename);
    else
    {
        strcpy(fnamebuf, getenv("HOME"));
        len = strlen(fnamebuf);
        fnamebuf[len++] = '/';
        strcpy(fnamebuf + len, io_filename + 1);
    }

    if  (stat(fnamebuf, &sbuf) == -1)
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_File_not_found);
        XtSetValues(input_align_io_label1, wargs, 1);

        XBell(dpy, 50);
        return;
    }
    /* ANSI standard file test */
    else if (!S_ISREG(sbuf.st_mode))
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_File_incorrect_type);
        XtSetValues(input_align_io_label1, wargs, 1);

        XBell(dpy, 50);
        return;
    }

    if  (io_str1_name[0] == '\0')
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_Missing_string_name);
        XtSetValues(input_align_io_label2, wargs, 1);

        XBell(dpy, 50);
        return;
    }
    if  (io_str2_name[0] == '\0')
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_Missing_string_name);
        XtSetValues(input_align_io_label3, wargs, 1);

        XBell(dpy, 50);
        return;
    }
    if  (strcmp(io_str1_name, io_str2_name) == 0)
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_Strings_must_be_different);
        XtSetValues(input_align_io_label2, wargs, 1);

        XtSetArg(wargs[0], XtNlabel, io_msg_Enter_String2_Label);
        XtSetValues(input_align_io_label3, wargs, 1);

        XBell(dpy, 50);
        return;
    }

    io_error = input_align_io_load(fnamebuf, &sbuf, io_str1_name, io_str2_name);

    if  (!io_error)
        input_align_io_hide(NULL, NULL, NULL);
    else
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_Enter_Filename);
        XtSetValues(input_align_io_label1, wargs, 1);

        XtSetArg(wargs[0], XtNlabel, io_msg_Enter_String1_Label);
        XtSetValues(input_align_io_label2, wargs, 1);

        XtSetArg(wargs[0], XtNlabel, io_msg_Enter_String2_Label);
        XtSetValues(input_align_io_label3, wargs, 1);


        switch (io_error)
        {
            case 1:
                XtSetArg(wargs[0], XtNlabel, io_msg_File_not_found_or_invalid);
                XtSetValues(input_align_io_label1, wargs, 1);
                break;
            case 2:
                XtSetArg(wargs[0], XtNlabel, io_msg_String_not_found_or_invalid);
                XtSetValues(input_align_io_label2, wargs, 1);
                break;
            case 3:
                XtSetArg(wargs[0], XtNlabel, io_msg_String_not_found_or_invalid);
                XtSetValues(input_align_io_label3, wargs, 1);
                break;
            case 4:
                XtSetArg(wargs[0], XtNlabel,
                         io_msg_String_alignments_different_lengths);
                XtSetValues(input_align_io_label2, wargs, 1);
                break;
            case 5:
                XtSetArg(wargs[0], XtNlabel, io_msg_String_alignments_invalid);
                XtSetValues(input_align_io_label2, wargs, 1);
                break;
            default:
                ;
        }

        XBell(dpy, 50);
    }

/* **   fprintf(log_fp, "input_align_io_ok: end\n");
/* */
}

inline
static Widget   input_align_io_create(  Widget  parent)
{
    Arg         wargs[10];
    int         n;
    Widget      command,
                form,
                left;

/* **   fprintf(log_fp, "input_align_io_create: begin\n");
/* */

    input_align_io_button = parent;
    input_align_io = XtCreatePopupShell("input_align_io", transientShellWidgetClass, parent,
                                 NULL, 0);

    form = create_form("form", input_align_io, NULL, NULL, 0, 2, wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    input_align_io_label1 = create_label(io_msg_File_incorrect_type,
                                 form, NULL, NULL, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNhorizDistance, 4), n++;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    XtSetArg(wargs[n], XtNwidth, 500); n++;
    input_align_io_edit1 = create_text("input_align_io_edit1", form, NULL,
                                       input_align_io_label1,
                                       "", 1, 0, wargs, n);

    XtAppAddActions(app, input_align_io_actions, XtNumber(input_align_io_actions));
    XtOverrideTranslations(input_align_io_edit1,
                           XtParseTranslationTable(input_align_io_translations));

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    input_align_io_label2 = create_label(io_msg_Label_for_String1_not_found,
                                         form, NULL, input_align_io_edit1, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNhorizDistance, 4), n++;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    XtSetArg(wargs[n], XtNwidth, 500); n++;
    input_align_io_edit2 = create_text("input_align_io_edit2", form, NULL,
                                       input_align_io_label2,
                                       "", 1, 0, wargs, n);

/*  XtAppAddActions(app, input_align_io_actions, XtNumber(input_align_io_actions)); */
    XtOverrideTranslations(input_align_io_edit2,
                           XtParseTranslationTable(input_align_io_translations));

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    input_align_io_label3 = create_label(io_msg_Label_for_String2_not_found,
                                 form, NULL, input_align_io_edit2, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNhorizDistance, 4), n++;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    XtSetArg(wargs[n], XtNwidth, 500); n++;
    input_align_io_edit3 = create_text("input_align_io_edit3", form, NULL,
                                       input_align_io_label3,
                                       "", 1, 0, wargs, n);

/*  XtAppAddActions(app, input_align_io_actions, XtNumber(input_align_io_actions)); */
    XtOverrideTranslations(input_align_io_edit3,
                           XtParseTranslationTable(input_align_io_translations));

    command = create_form("input_align_io_command3", form, NULL, input_align_io_edit3, 0, 4,
                          wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    left = create_command("Cancel", command, NULL, NULL, input_align_io_hide, NULL,
                          wargs, n);
    create_command("OK", command, left, NULL, input_align_io_ok, NULL, wargs, n);

    XtRealizeWidget(input_align_io);

/* **   fprintf(log_fp, "input_align_io_create: end\n");
/* */
}


/********************************************************************/
/*  file_io routines                                                */
/********************************************************************/

Widget          file_io,
                file_io_button,
                file_io_edit,
                file_io_label;

static void     file_io_ok(Widget, char*, XtPointer);
static void     file_io_hide(Widget, char*, XtPointer);

static String   file_io_translations =
                        "Ctrl<Key>J: file_io_ok() \n"
                        "Ctrl<Key>M: file_io_ok() \n"
                        "<Key>Linefeed: file_io_ok() \n"
                        "<Key>Return: file_io_ok() \n"
			"Ctrl<Key>.: file_io_hide()";

static XtActionsRec     file_io_actions[] =
                        { {"file_io_ok", (XtActionProc) file_io_ok},
			  {"file_io_hide", (XtActionProc) file_io_hide}
                        };



static void     file_io_hide(   Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
/* **   fprintf(log_fp, "file_io_hide: begin\n");
/* */

    XtPopdown(file_io);

/* **   fprintf(log_fp, "file_io_hide: end\n");
/* */
}


static void     file_io_show(   Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    Arg         wargs[10];
    char        buf[30];
    Dimension   h;
    int         n;
    Position    absx,
                absy,
                x,
                y;

/* **   fprintf(log_fp, "file_io_show: begin\n");
/* */

    string_index = atoi(client_data);
    file_io_function = client_data[1];

    assert (   (string_index == 1)
            || (string_index == 2));
    assert (   (file_io_function == file_io_function_LOAD)
            || (file_io_function == file_io_function_SAVE));

    n = 0;
    XtSetArg(wargs[n], XtNx, &absx); n++;
    XtSetArg(wargs[n], XtNy, &absy); n++;
    XtGetValues(toplevel, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, &x); n++;
    XtSetArg(wargs[n], XtNy, &y); n++;
    XtSetArg(wargs[n], XtNheight, &h); n++;
    XtGetValues(file_io_button, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, absx + x + 3); n++;
    XtSetArg(wargs[n], XtNy, absy + y + h + 3); n++;
    XtSetArg(wargs[n], XtNinput, (XtArgVal)TRUE); n++;
    XtSetValues(file_io, wargs, n);

    if  (file_io_function == file_io_function_LOAD)
        XtSetArg(wargs[0], XtNlabel, io_msg_Enter_Filename_to_Load);
    else
        XtSetArg(wargs[0], XtNlabel, io_msg_Enter_Filename_for_Save);

    XtSetValues(file_io_label, wargs, 1);
    TrapForDialogCloseBoxHit(file_io);
    XtPopup(file_io, XtGrabExclusive);

/* **   fprintf(log_fp, "file_io_show: end\n");
/* */
}

inline
static int      file_io_load(   char            *filename,
                                struct stat     *filebuf)
{
    int         len = 0;
    char        *file_io_str,
                *s,
                *t;
    FILE        *fp;
    size_t	result;

/* **   fprintf(log_fp, "file_io_load: begin\n");
/* */
/* **   fprintf(log_fp, "Loading String #%d from file %s\n", string_index, filename);
/* */

    if  ((fp = fopen(filename,"r")) == NULL)
    {
        fprintf(log_fp,
                "Error in opening file %s to load string #%d\n",
                filename, string_index);
        return 1;
    }
    file_io_str = new char[filebuf->st_size + 1];
    result = fread(file_io_str, 1, filebuf->st_size, fp);
    if (result != filebuf->st_size)
	{
	fprintf(log_fp,
	    "Error reading info from file. Expected %ld bytes, retrieved %ld bytes\n",
	    filebuf->st_size, result);
	fclose(fp);
	return -1;
	}
    fclose(fp);
    file_io_str[filebuf->st_size] = '\0';

/*
**  Remove all newline characters.
*/
    for (s=file_io_str,t=s; *s; s++)
    {
        if  (*s != '\n')
        {
            if  (t != s)
                *t = *s;
                t++;
        }
    }
    *t = '\0';

    intTextSetString(string[string_index-1], file_io_str);
    delete file_io_str;
    file_io_str = NULL;

    check_delete_input_fun_align();

    process_change(changed_STRINGS);

    return 0;

/* **   fprintf(log_fp, "file_io_load: end\n");
/* */
}

inline
static int      file_io_save(   char            *filename,
                                struct stat     *filebuf)
{
    char        *savestr;
    FILE        *fp = NULL;

/* **   fprintf(log_fp, "file_io_save: begin\n");
/* */
/* **   fprintf(log_fp, "Saving String #%d\n to file %s", string_index, filename);
/* */
    fp = fopen(filename,"w");

    if  (fp == NULL)
    {
        fprintf(log_fp, "Error in opening file %s\n", filename);
        return 1;
    }
    savestr = intTextGetString(string[string_index - 1]);
    fprintf(fp, "%s\n", savestr);
    fclose(fp);

/* **   fprintf(log_fp, "file_io_save: end\n");
/* */
    return 0;
}


static void     file_io_ok(     Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    Arg         wargs[1];
    int         len, io_error;
    char        *io_filename, fnamebuf[1024];
    struct stat sbuf;

/* **   fprintf(log_fp, "file_io_ok: begin\n");
/* */

    io_filename = intTextGetString(file_io_edit);

    if  (io_filename[0] == '\0')
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_Invalid_filename);
        XtSetValues(file_io_label, wargs, 1);

        XBell(dpy, 50);
        return;
    }

    if  (io_filename[0] != '~')
        strcpy(fnamebuf, io_filename);
    else
    {
        strcpy(fnamebuf, getenv("HOME"));
        len = strlen(fnamebuf);
        fnamebuf[len++] = '/';
        strcpy(fnamebuf + len, io_filename + 1);
    }

    if  (file_io_function == file_io_function_LOAD)
    {
        if  (stat(fnamebuf, &sbuf) == -1)
        {
            XtSetArg(wargs[0], XtNlabel, io_msg_File_not_found);
            XtSetValues(file_io_label, wargs, 1);

            XBell(dpy, 50);
            return;
        }
	/* ANSI standard file test */
        else if (!S_ISREG(sbuf.st_mode))
        {
            XtSetArg(wargs[0], XtNlabel, io_msg_File_incorrect_type);
            XtSetValues(file_io_label, wargs, 1);

            XBell(dpy, 50);
            return;
        }
    }
    else
    {
        if  (stat(fnamebuf, &sbuf) > -1) {
	  /* ANSI standard file test */
	  if (!S_ISREG(sbuf.st_mode))
            {
	      XtSetArg(wargs[0], XtNlabel, io_msg_File_incorrect_type);
	      XtSetValues(file_io_label, wargs, 1);
	      
	      XBell(dpy, 50);
	      return;
            }
        }
	else if  (errno != ENOENT) {
	  XtSetArg(wargs[0], XtNlabel, io_msg_File_incorrect_type);
	  XtSetValues(file_io_label, wargs, 1);
	  
	  XBell(dpy, 50);
	  return;
	}
    }

    if  (file_io_function == file_io_function_LOAD)
        io_error = file_io_load(fnamebuf, &sbuf);
    else
    {
        assert (file_io_function == file_io_function_SAVE);
        io_error = file_io_save(fnamebuf, &sbuf);
    }

    if  (!io_error)
        file_io_hide(NULL, NULL, NULL);
    else
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_Cannot_open_file);
        XtSetValues(file_io_label, wargs, 1);

        XBell(dpy, 50);
    }

/* **   fprintf(log_fp, "file_io_ok: end\n");
/* */
}

inline
static Widget   file_io_create( Widget  parent)
{
    Arg         wargs[10];
    int         n;
    Widget      command,
                form,
                left;

/* **   fprintf(log_fp, "file_io_create: begin\n");
/* */

    file_io_button = parent;
    file_io = XtCreatePopupShell("file_io", transientShellWidgetClass, parent,
                                 NULL, 0);

    form = create_form("form", file_io, NULL, NULL, 0, 2, wargs, 0);
    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    file_io_label = create_label(io_msg_File_incorrect_type,
                                 form, NULL, NULL, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNhorizDistance, 4), n++;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    XtSetArg(wargs[n], XtNwidth, 250); n++;
    file_io_edit = create_text("file_io_edit", form, NULL, file_io_label,
                                "", 1, 0, wargs, n);

    XtAppAddActions(app, file_io_actions, XtNumber(file_io_actions));
    XtOverrideTranslations(file_io_edit,
                           XtParseTranslationTable(file_io_translations));

    command = create_form("file_io_command", form, NULL, file_io_edit, 0, 4,
                          wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    left = create_command("Cancel", command, NULL, NULL, file_io_hide, NULL,
                          wargs, n);
    create_command("OK", command, left, NULL, file_io_ok, NULL, wargs, n);

    XtRealizeWidget(file_io);

/* **   fprintf(log_fp, "file_io_create: end\n");
/* */
}


/************************************************/
/* weight_io routines */
/************************************************/

Widget          weight_io,
                weight_io_button,
                weight_io_label,
                weight_io_edit;

static void     weight_io_ok(Widget, char*, XtPointer);
static void     weight_io_hide(Widget, char*, XtPointer);


static String   weight_io_translations =
                        "Ctrl<Key>J: weight_io_ok() \n\
                        Ctrl<Key>M: weight_io_ok() \n\
                        <Key>Linefeed: weight_io_ok() \n\
                        <Key>Return: weight_io_ok()\n\
			Ctrl<Key>.: weight_io_hide()";


static XtActionsRec weight_io_actions[] =
                        { {"weight_io_ok", (XtActionProc) weight_io_ok},
			  {"weight_io_hide", (XtActionProc) weight_io_hide},
                        };


static void     weight_io_hide( Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
/* **   fprintf(log_fp, "weight_io_hide: begin\n");
/* */

    XtPopdown(weight_io);

/* **   fprintf(log_fp, "weight_io_hide: end\n");
/* */
}


static void     weight_io_show( Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    Arg         wargs[10];
    char        buf[30];
    Dimension   h;
    int         n;
    Position    absx,
                absy,
                x,
                y;
    int		score;

/* **   fprintf(log_fp, "weight_io_show: begin\n");
/* */
    /* convert client_data to a number */

    score = atoi(client_data); 

    n = 0;
    XtSetArg(wargs[n], XtNx, &absx); n++;
    XtSetArg(wargs[n], XtNy, &absy); n++;
    XtGetValues(toplevel, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, &x); n++;
    XtSetArg(wargs[n], XtNy, &y); n++;
    XtSetArg(wargs[n], XtNheight, &h); n++;
    XtGetValues(weight_io_button, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNx, absx + x + 3); n++;
    XtSetArg(wargs[n], XtNy, absy + y + h + 3); n++;
    XtSetArg(wargs[n], XtNinput, (XtArgVal)TRUE); n++;
    XtSetValues(weight_io, wargs, n);

    /* Set the default filename */
    switch  (score)
	{
      case 0:
			// User will enter filename.
	XtSetArg(wargs[0], XtNlabel, io_msg_Enter_Filename_for_Scoring_Matrix);
	XtSetValues(weight_io_label, wargs, 1);
	TrapForDialogCloseBoxHit(weight_io);
	XtPopup(weight_io, XtGrabExclusive);
	break;
      case 1:		// W_CONSTANT : Use a constant weighting scheme.
        if  (weights_in_array)
	    {
            delete weight_matrix_name;
            weight_matrix_name = NULL;
            delete weight_alphabet;
            weight_alphabet = NULL;
            delete weight_matrix_array;
            weight_matrix_array = NULL;

            process_change(changed_PARMS);
	    }
        weights_in_array = 0;
        XtSetArg(wargs[0], XtNlabel, weightoptionconst);
        XtSetValues(weight_label, wargs, 1);
        break;
			// Predefined weight/scoring matrices.
      case 2:	    // PAM250_MAT : the PAM 250 matrix
      case 3:	    // PAM250_8_MAT : the normalized PAM 250 matrix
      case 4:	    // BLOSUM80_MAT
      case 5:	    // BLOSUM62_MAT
      case 6:	    // BLOSUM45_MAT
      case 7:	    // GONNET_MAT
      case 8:	    // GRIBSKOW_MAT
      case 9:	    // MCCLURE_MAT
      case 10:	    // TAYLOR_MAT
	struct	    stat sbuf;
	char	    identifier[2];

	identifier[0] = score;
	identifier[1] = '\0';
	weight_io_load(identifier, &sbuf, TRUE);
        break;
	}

/* **   fprintf(log_fp, "weight_io_show: end\n");
/* */
}


static void     weight_io_bad_format(   char    *filename)
{
    fprintf(log_fp, "Scoring matrix file %s has invalid format\n\n", filename);
    fprintf(log_fp, "Correct format is:\n");
    fprintf(log_fp, "    Any line starting with # is a comment\n");
    fprintf(log_fp, "    For lines not starting with #:\n");
    fprintf(log_fp,
            "    Line 1: 'Name: X', where X is weight table name, e.g., 'Name: PAM 250'\n");
    fprintf(log_fp,
            "    Line 2: 'Case Sensitive: Y' or 'Case Sensitive: N'\n");
    fprintf(log_fp,
            "            N means no  -- 'A' and 'a' are the same\n");
    fprintf(log_fp,
            "            Y means yes -- 'A' and 'a' are different\n");
    fprintf(log_fp,
            "            N is automatically duplicated to opposite case\n");
    fprintf(log_fp,
            "            (Note that if Case Sensitive is No then just one of the capitol\n");
    fprintf(log_fp,
            "             and lowercase versions of the character can be in the file.)\n");
    fprintf(log_fp,
            "    Line 3: 'Alphabet: XY' where XY are the characters in the alphabet\n");
    fprintf(log_fp,
            "    Line 4-end:\n");
    fprintf(log_fp,
            "            weight matrix, one row per line\n");
    fprintf(log_fp,
            "            each row is for one character\n");
    fprintf(log_fp,
            "            (i,j) entry is j-th value of i-th row\n");
    fprintf(log_fp,
            "            Either full rows or just bottom left triangle can be provided\n\n");
    fprintf(log_fp,
            "            If just those below the diagonal are present then the values\n");
    fprintf(log_fp,
            "            above the diagonal are assumed to be symmetric.\n");
    fprintf(log_fp,
            "            Each row can be of one of two forms\n");
    fprintf(log_fp,
            "                First form:\n");
    fprintf(log_fp,
            "                    Values separated by spaces or commas\n");
    fprintf(log_fp,
            "                    (i-th row of values used for i-th character of alphabet\n");
    fprintf(log_fp,
            "                Second form:\n");
    fprintf(log_fp,
            "                    \"X:\" followed by values separated by spaces or commas\n");
    fprintf(log_fp,
            "                    where X is the character for the row\n");
    fprintf(log_fp,
            "            In both cases the i-th character must have either i values\n");
    fprintf(log_fp,
            "            or alphabet_size values\n\n");
}

inline static int 
weight_io_load(char *filename, struct stat  *filebuf, int isPreloaded)
{
    Arg         wargs[1];
    char        *wk_str1;
    unsigned
        char    *next_char,
                *str1,
                *str2,
                *weight_io_data,
                *weight_next_line,
                wk_char,
                wk_char_lower;
    int         i,
                i_char_row_num,
                i_lower_row_num,
                j,
                j_char_col_num,
                j_lower_col_num,
                k,
                k_char_row_num,
                found;
    int         *have_char_row;
    int         num_this_row;
    int         read_full_rows;
    FILE        *fp;
    size_t	result;

/* **   fprintf(log_fp, "weight_io_load: begin\n");
/* */
/* **   fprintf(log_fp, "Loading Scoring Matrix from file %s\n", filename);
/* */

    if (isPreloaded)
	{
	const char	*strPtr;

	switch(*filename)
	    {
	    case 2:	// PAM250_MAT
		strPtr = PAM250_MAT;
		break;
	    case 3:	// PAM250_8_MAT
		strPtr = PAM250_8_MAT;
		break;
	    case 4:	// BLOSUM80_MAT
		strPtr = BLOSUM80_MAT;
		break;
	    case 5:	// BLOSUM62_MAT
		strPtr = BLOSUM62_MAT;
		break;
	    case 6:	// BLOSUM45_MAT
		strPtr = BLOSUM45_MAT;
		break;
	    case 7:	// GONNET_MAT
		strPtr = GONNET_MAT;
		break;
	    case 8:	// GRIBSKOW_MAT
		strPtr = GRIBSKOW_MAT;
		break;
	    case 9:	// MCCLURE_MAT
		strPtr = MCCLURE_MAT;
		break;
	    case 10:	// TAYLOR_MAT
		strPtr = TAYLOR_MAT;
		break;
	    }
	weight_io_data = new unsigned char[strlen(strPtr) + 1];
	strcpy((char *)weight_io_data, strPtr);
	}
    else
	{
	if  ((fp = fopen(filename,"r")) == NULL)
	{
	    fprintf(log_fp,
		    "Error in opening file %s to load weight matrix\n",
		    filename, string_index);
	    return 1;
	}
	weight_io_data = new unsigned char[filebuf->st_size + 1];
	result = fread(weight_io_data, 1, filebuf->st_size, fp);
	if (result != filebuf->st_size)
	    {
	    fprintf(log_fp,
		"Error reading info from file. Expected %ld bytes, retrieved %ld bytes\n",
		filebuf->st_size, result);
	    fclose(fp);
	    return -1;
	    }
	fclose(fp);
	weight_io_data[filebuf->st_size] = '\0';
	}
/*
**    Scoring Matrix file format is:
**      Any line starting with # is a comment
**      Among the other lines:
**      Line 1: 'Name: X', where X is weight table name, e.g., 'Name: PAM 250'
**      Line 2: 'Case Sensitive: Y' or 'Case Sensitive: N'
**              N means no  -- 'A' and 'a' are the same
**              Y means yes -- 'A' and 'a' are different
**              N is automatically duplicated to opposite case
**      Line 3: 'Alphabet: XY' where XY are the characters in the alphabet
**      Line 4-end:
**              weight matrix, one row per line
**              each row is for one character
**              matrix values may be separated by any number of
**                      commas, spaces, and tabs
**              Either the full matrix or just the values on and
**              below the (top left) to (bottom right) diagonal
**              can be entered.  If just those below the diagonal
**              are present then the program assumes that the
**              values above the diagonal are symmetric.
**              (Note that if case sensitive is No then just one
**              the capitol and lowercase versions of the character
**              can be in the file.)
*/
/*
**    Skip commented lines
*/
    weight_next_line = weight_io_data;
    while (*weight_next_line == '#')
    {
        do
        {
            weight_next_line++;
        }  while (   (*weight_next_line != '\n')
                  && (*weight_next_line != '\0'));
        if  (*weight_next_line != '\0')
            weight_next_line++;
    }
/*
**    Check for and get the name
*/
    if  (strncasecmp((const char *)weight_next_line, "Name:", 5))
    {
        weight_io_bad_format(filename);
        fprintf(log_fp,
                "*****  Line 1  (weight matrix name) missing or invalid  *****\n\n");
        delete weight_io_data;
	weight_io_data = NULL;
        return 1;
    }
    weight_next_line += 5;
    while  (   (*weight_next_line == ' ')
            || (*weight_next_line == '\t'))
        weight_next_line++;
    str1 = weight_next_line;
    while  (   (*str1 != '\n')
            && (*str1 != '\0'))
        str1++;
    delete weight_matrix_name;
    weight_matrix_name = new char[str1 - weight_next_line + 1];
    wk_str1 = weight_matrix_name;
    while  (   (*weight_next_line != '\n')
            && (*weight_next_line != '\0'))
    {
        *wk_str1 = (char)*weight_next_line;
        wk_str1++;
        weight_next_line++;
    }
    *wk_str1 = '\0';
    if  (*weight_next_line == '\n')
        weight_next_line++;
/*
**          Skip commented lines
*/
    while (*weight_next_line == '#')
    {
        do
        {
            weight_next_line++;
        }  while (   (*weight_next_line != '\n')
                  && (*weight_next_line != '\0'));
        if  (*weight_next_line != '\0')
            weight_next_line++;
    }
/*
**    Check for case sensitivity
*/
    if  (strncasecmp((char *)weight_next_line, "Case Sensitive:", 15))
    {
        weight_io_bad_format(filename);
        fprintf(log_fp,
                "*****  Line 2 (case sensitivity) missing or invalid  *****\n\n");
        delete weight_io_data;
        delete weight_matrix_name;
	weight_io_data = NULL;
        weight_matrix_name = NULL;
        return 1;
    }
    weight_next_line += 15;
    while  (   (*weight_next_line == ' ')
            || (*weight_next_line == '\t'))
        weight_next_line++;
    if  (   (*weight_next_line == 'n')
         || (*weight_next_line == 'N'))
        weight_case_sensitive = 0;
    else
    if  (   (*weight_next_line == 'y')
         || (*weight_next_line == 'Y'))
        weight_case_sensitive = 1;
    else
    {
        weight_io_bad_format(filename);
        fprintf(log_fp, "*****  Line 2 (case sensitivity) invalid  *****\n\n");
        delete weight_io_data;
        delete weight_matrix_name;
	weight_io_data = NULL;
        weight_matrix_name = NULL;
        return 1;
    }
    while  (   (*weight_next_line != '\n')
            && (*weight_next_line != '\0'))
    {
        weight_next_line++;
    }
    if  (*weight_next_line != '\0')
        weight_next_line++;
/*
**    Skip commented lines
*/
    while (*weight_next_line == '#')
    {
        do
        {
            weight_next_line++;
        }  while (   (*weight_next_line != '\n')
                  && (*weight_next_line != '\0'));
        if  (*weight_next_line != '\0')
            weight_next_line++;
    }
/*
**    Find out the alphabet (valid characters for strings),
**    while finding the smallest and largest characters.
*/
    if  (strncasecmp((char *)weight_next_line, "Alphabet:", 9))
    {
        weight_io_bad_format(filename);
        fprintf(log_fp, "*****  Line 3 (alphabet) missing or invalid  *****\n\n");
        delete weight_io_data;
        delete weight_matrix_name;
	weight_io_data = NULL;
        weight_matrix_name = NULL;
        return 1;
    }
    weight_next_line += 9;
/*
**    First find out how many characters are in the alphabet
**    and allocate the space for them.
*/
    if  (weight_alphabet != NULL)
    {
        delete weight_alphabet;
        weight_alphabet = NULL;
    }
    weight_alphabet_size = 0;
    str1 = weight_next_line;
    while (   (*str1 != '\n')
           && (*str1 != '\0'))
    {
        if  (   (*str1 != ' ')
             && (*str1 != '\t'))
            weight_alphabet_size++;
        str1++;
    }
    weight_alphabet = new unsigned char[weight_alphabet_size + 1];
/*
**    Now copy the alphabet to the alphabet array,
**    keeping track of the smallest and largest characters encountered.
*/
    weight_alphabet_char_max = '\0';
    weight_alphabet_char_min = '\377';
    next_char = weight_alphabet;
    while (   (*weight_next_line != '\n')
           && (*weight_next_line != '\0'))
    {
        if  (   (*weight_next_line != ' ')
             && (*weight_next_line != '\t'))
        {
            *next_char = *weight_next_line;
            if  (   (!weight_case_sensitive)
                 && (*next_char >= 'a')
                 && (*next_char <= 'z'))
                *next_char = (unsigned char)((int)(*next_char)
                           - ((int)('a') - (int)('A')));
            if  (*next_char < weight_alphabet_char_min)
                weight_alphabet_char_min = *next_char;
            if  (   (!weight_case_sensitive)
                 && (*next_char >= 'A')
                 && (*next_char <= 'Z'))
            {
                wk_char_lower = (unsigned char)((int)(*next_char)
                              + ((int)('a') - (int)('A')));
                if  (wk_char_lower > weight_alphabet_char_max)
                    weight_alphabet_char_max = wk_char_lower;
            }
            if  (*next_char > weight_alphabet_char_max)
                weight_alphabet_char_max = *next_char;
            next_char++;
        }
        weight_next_line++;
    }
    weight_alphabet_min_as_int = (int)weight_alphabet_char_min;
    weight_alphabet_max_as_int = (int)weight_alphabet_char_max;
    next_char = '\0';
    weight_next_line++;
/*
**    Make sure that no two characters are the same.
*/
    for (i = 0; i < weight_alphabet_size; i++)
        for (j = i+1; j <= weight_alphabet_size; j++)
            if  (weight_alphabet[i] == weight_alphabet[j])
            {
                weight_io_bad_format(filename);
                fprintf(log_fp, "*****  Line 3 (Alphabet) invalid  *****\n");
                fprintf(log_fp, "*****  (char %c multiply defined)  *****\n\n",
                        weight_alphabet[i]);
                delete weight_io_data;
                delete weight_matrix_name;
                delete weight_alphabet;
		weight_io_data = NULL;
		weight_matrix_name = NULL;
                weight_alphabet = NULL;
                return 1;
            };
/*
**    Now allocate, initialize, and read in the weight matrix.
*/
    have_char_row = new int[weight_alphabet_size];
    for (i = 0; i < weight_alphabet_size; i++)
        have_char_row[i] = 0;

    weight_matrix_array_size = weight_alphabet_max_as_int
                             - weight_alphabet_min_as_int + 1;
    weight_matrix_array =
                new double[weight_matrix_array_size * weight_matrix_array_size];
    for (i = 0; i < weight_matrix_array_size; i++)
    {
        for (j = 0; j < weight_matrix_array_size; j++)
            weight_matrix_array[i*weight_matrix_array_size+j] = 0.0;
    }

    read_full_rows = -1;
    for (i = 0; i < weight_alphabet_size; i++)
    {
/*
**      Skip commented lines
*/
        while  (*weight_next_line == '#')
        {
            do
            {
                weight_next_line++;
            }  while (   (*weight_next_line != '\n')
                      && (*weight_next_line != '\0'));
            if  (*weight_next_line != '\0')
                weight_next_line++;
        }
/*
**      Figure out which row this is for
*/
        if  (*(weight_next_line+1) == ':')
        {
            /*  The first character on the line is the     */
            /*  alphabet character for the row.            */
            /*  Find it and set up to read the row.        */
            /*  Case insensitive characters are capitals.  */
            if  (   (!weight_case_sensitive)
                 && (*weight_next_line >= 'a')
                 && (*weight_next_line <= 'z'))
                wk_char = (unsigned char)((int)(*weight_next_line)
                                          - ((int)('a') - (int)('A')));
            else
                wk_char = *weight_next_line;
            k = 0;
            found = 0;
            while  (   (wk_char != weight_alphabet[k])
                    && (k < weight_alphabet_size))
                k++;
            if  (k >= weight_alphabet_size)
            {
                weight_io_bad_format(filename);
                fprintf(log_fp,
                        "*****  Line %d of matrix values invalid     *****\n", i);
                fprintf(log_fp,
                        "*****  (lead character %c not in alphabet)  *****\n\n",
                        *weight_next_line);
                delete weight_io_data;
                delete weight_matrix_name;
                delete weight_alphabet;
                delete have_char_row;
                delete weight_matrix_array;
		weight_io_data = NULL;
		weight_matrix_name = NULL;
		weight_alphabet = NULL;
		have_char_row = NULL;
                weight_matrix_array = NULL;
                return 1;
            }
            weight_next_line += 2;
        }
        else
            k = i;
        k_char_row_num = (int)weight_alphabet[k] - weight_alphabet_min_as_int;
        num_this_row = 0;
/*
**      Skip ahead to '\n' and change to '\0' for scanning
**      (will change it back after the scan)
*/
        str1 = weight_next_line;
        while  (   (*str1 != '\n')
                && (*str1 != '\0'))
            str1++;
        if  (*str1 == '\n')
            *str1 = '\0';
        else
            str1 = NULL;
/*
**      Now get the values
*/
        j = 0;
        while  (*weight_next_line != '\0')
        {
            while  (   (*weight_next_line != '+')
                    && (*weight_next_line != '-')
                    && (!(   (*weight_next_line >= '0')
                          && (*weight_next_line <= '9')))
                    && (*weight_next_line != '\0'))
                weight_next_line++;
            if  (*weight_next_line != '\0')
            {
                j_char_col_num = (int)weight_alphabet[j] - weight_alphabet_min_as_int;
                weight_matrix_array[k_char_row_num*weight_matrix_array_size
                                    +j_char_col_num] =
                        atof((char *)weight_next_line);
                while  (   (*weight_next_line != ' ')
                        && (*weight_next_line != '\t')
                        && (*weight_next_line != ',')
                        && (*weight_next_line != '\0'))
                    weight_next_line++;
                j++;
            }
        }
        if  (read_full_rows == -1)
        {
            if  (j == k+1)
                read_full_rows = 0;
            else
                read_full_rows = 1;
        }
        if  (read_full_rows)
        {
            if  (j != weight_alphabet_size)
            {
                weight_io_bad_format(filename);
                fprintf(log_fp,
                        "*****  Line %d of matrix values invalid     *****\n", i);
                fprintf(log_fp,
                        "*****  %d values in row %d, not %d  *****\n",
                        j, k, weight_alphabet_size);
                delete weight_io_data;
                delete weight_matrix_name;
                delete weight_alphabet;
                delete have_char_row;
                delete weight_matrix_array;
		weight_io_data = NULL;
		weight_matrix_name = NULL;
		weight_alphabet = NULL;
		have_char_row = NULL;
                weight_matrix_array = NULL;
                return 1;
            }
        }
        else  /*  (!read_full_rows)  */
        {
            if  (j != k+1)
            {
                weight_io_bad_format(filename);
                fprintf(log_fp,
                        "*****  Line %d of matrix values invalid  *****\n", i);
                fprintf(log_fp,
                        "*****  %d values in row %d, not %d  *****\n",
                        j, k, k);
                delete weight_io_data;
                delete weight_matrix_name;
                delete weight_alphabet;
                delete have_char_row;
                delete weight_matrix_array;
		weight_io_data = NULL;
                weight_matrix_name = NULL;
                weight_alphabet = NULL;
		have_char_row = NULL;
                weight_matrix_array = NULL;
                return 1;
            }
        }
        have_char_row[k] = 1;
        if  (str1 != NULL)
        {
            *str1 = '\n';
            weight_next_line = str1 + 1;
        }
    }
    for (i = 0; i < weight_alphabet_size; i++)
    {
        if  (!have_char_row[i])
        {
            weight_io_bad_format(filename);
            fprintf(log_fp,
                    "*****  Line %d of matrix values for char %c missing  *****\n",
                    i, weight_alphabet[i]);
            delete weight_io_data;
            delete weight_matrix_name;
            delete weight_alphabet;
            delete have_char_row;
            delete weight_matrix_array;
	    weight_io_data = NULL;
            weight_matrix_name = NULL;
	    weight_alphabet = NULL;
	    have_char_row = NULL;
            weight_matrix_array = NULL;
            return 1;
        }
    }
    delete have_char_row;
    have_char_row = NULL;
/*
**    If read in lower left triangle, then fill in upper right values
*/
    if  (!read_full_rows)
    {
        for (i = 1; i < weight_alphabet_size; i++)
            for (j = 0; j < i; j++)
                weight_matrix_array[j*weight_matrix_array_size+i] =
                        weight_matrix_array[i*weight_matrix_array_size+j];
    }
/*
**    If not case sensitive, copy capital to lowercase, rows then columns
*/
    if  (!weight_case_sensitive)
    {
        for (i = 0; i < weight_alphabet_size; i++)
        {
            if  (   (weight_alphabet[i] >= 'A')
                 && (weight_alphabet[i] <= 'Z'))
            {
                i_char_row_num = (int)weight_alphabet[i]
                               - weight_alphabet_min_as_int;
                i_lower_row_num = i_char_row_num
                               + ((int)((unsigned char)('a')))
                               - ((int)((unsigned char)('A')));
                for (j = 0; j < weight_matrix_array_size; j++)
                    weight_matrix_array[i_lower_row_num*weight_matrix_array_size + j] =
                        weight_matrix_array[i_char_row_num*weight_matrix_array_size + j];
            }
        }
        for (j = 0; j < weight_alphabet_size; j++)
        {
            if  (   (weight_alphabet[j] >= 'A')
                 && (weight_alphabet[j] <= 'Z'))
            {
                j_char_col_num = (int)weight_alphabet[j]
                               - weight_alphabet_min_as_int;
                j_lower_col_num = j_char_col_num
                               + ((int)((unsigned char)('a')))
                               - ((int)((unsigned char)('A')));
                for (i = 0; i < weight_matrix_array_size; i++)
                    weight_matrix_array[i*weight_matrix_array_size + j_lower_col_num] =
                        weight_matrix_array[i*weight_matrix_array_size + j_char_col_num];
            }
        }
    }
/*
**    Everything is fine, so go ahead and set to use the array.
**    Also, clean up by deallocating work fields.
*/
    weights_in_array = 1;
    XtSetArg(wargs[0], XtNlabel, weight_matrix_name);
    XtSetValues(weight_label, wargs, 1);

    delete weight_io_data;
    weight_io_data = NULL;

    process_change(changed_PARMS);

    // Make a fake event corresponding to a mouse click in the main canvas.
    // This will cause the outline of the polygon to be drawn, and,
    // importantly, the number of cooptimal polygons to be calculated,
    // displayed and accessible via the "Prev" and "Next" buttons. --Mars
    XEvent	    event = {0};

    event.xbutton.x = 40;
    event.xbutton.y = 40;
    event.xbutton.button = 1;
    explore(NULL, NULL, &event, 0);

    return 0;

/* **   fprintf(log_fp, "weight_io_load: end\n");
/* */
}


static void     weight_io_ok(   Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    Arg         wargs[1];
    int         len, io_error;
    char        *io_filename, fnamebuf[1024];
    struct stat sbuf;

/* **   fprintf(log_fp, "weight_io_ok: begin\n");
/* */
    io_filename = intTextGetString(weight_io_edit);

    /*  Need to set up for a weight matrix  */
    if  (io_filename[0] != '~')
        strcpy(fnamebuf, io_filename);
    else
    {
        strcpy(fnamebuf, getenv("HOME"));
        len = strlen(fnamebuf);
        fnamebuf[len++] = '/';
        strcpy(fnamebuf + len, io_filename + 1);
    }
    if  (stat(fnamebuf, &sbuf) == -1)
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_File_not_found);
        XtSetValues(weight_io_label, wargs, 1);

        XBell(dpy, 50);
        return;
    }
    /* ANSI standard file test */
    else if (!S_ISREG(sbuf.st_mode))
    {
        XtSetArg(wargs[0], XtNlabel, io_msg_File_incorrect_type);
        XtSetValues(weight_io_label, wargs, 1);

        XBell(dpy, 50);
        return;
    }
    weights_in_array = 0;
    delete weight_matrix_name;
    weight_matrix_name = NULL;
    delete weight_alphabet;
    weight_alphabet = NULL;
    delete weight_matrix_array;
    weight_matrix_array = NULL;
    io_error = weight_io_load(fnamebuf, &sbuf, FALSE);
    if  (!io_error)
        weight_io_hide(NULL, NULL, NULL);
    else
	{
        XtSetArg(wargs[0], XtNlabel, io_msg_File_not_correct_format);
        XtSetValues(weight_io_label, wargs, 1);

        XBell(dpy, 50);
	}
/* **   fprintf(log_fp, "weight_io_ok: end\n");
/* */
}

inline
static Widget   weight_io_create(       Widget  parent)
{
    int         n;
    Arg         wargs[10];
    Widget      command,
                form,
                left;

/* **   fprintf(log_fp, "weight_io_create: begin\n");
/* */

    weight_io_button = parent;
    weight_io = XtCreatePopupShell("weight_io", transientShellWidgetClass, parent,
                                   NULL, 0);

    form = create_form("form", weight_io, NULL, NULL, 0, 2, wargs, 0);
    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, 4); n++;
    XtSetArg(wargs[n], XtNjustify, XtJustifyLeft); n++;
    weight_io_label = create_label(io_msg_File_incorrect_type,
                                   form, NULL, NULL, 0, wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNhorizDistance, 4), n++;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    XtSetArg(wargs[n], XtNwidth, 400); n++;
    weight_io_edit = create_text("weight_io_edit", form, NULL, weight_io_label,
                                 "", 1, 0, wargs, n);

    XtAppAddActions(app, weight_io_actions, XtNumber(weight_io_actions));
    XtOverrideTranslations(weight_io_edit,
                           XtParseTranslationTable(weight_io_translations));

    command = create_form("weight_io_command", form, NULL, weight_io_edit, 0, 4,
                          wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2); n++;
    left = create_command("Cancel", command, NULL, NULL, weight_io_hide, NULL,
                          wargs, n);
    create_command("OK", command, left, NULL, weight_io_ok, NULL, wargs, n);

    XtRealizeWidget(weight_io);

/* **   fprintf(log_fp, "weight_io_create: end\n");
/* */
}


/************************************************/
/*  Other routines */
/************************************************/

/* ARGSUSED */
static void     do_nothing(     Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
}


static void     get_mult_polys(Widget w, char *ch, XtPointer call_data)
{
    Arg         wargs[1];
    Point       *wk_pt;
    char        buf[256];
    int         all_polys;

    all_polys = atoi(ch);
    if  ((all_polys == 0) && (input_fun == NULL))
	{
        XBell(dpy, 50);
	}
    else
    {
        if  (changed_something & changed_STRINGS)
            set_strings(changed_STRINGS);

//--changed
    // If, after calling set_string(), both strings are NULL, then there are no
    // sequences, and thus no polygons. In this case, nothing should be done.
    // Capturing this event prevents a segmentation fault when clicking in the
    // middle of the window when there are no sequences.
    // Mars
	if ((str1 == NULL) && (str2 == NULL))
	    {
	    XBell(dpy, 50);
	    return;
	    }

        complete_decomp(all_polys);

        if  (display_pt != NULL)
        {
            delete display_pt;
            display_pt = NULL;
            delete display_fun_pt;
            display_fun_pt = NULL;
            delete display_fun;
            display_fun = NULL;
        }
        display_pt = new Point(poly_list->corners);
        display_pt->next = NULL;
        display_fun_pt = new Point(display_pt);
        wk_pt = poly_list->corners->next;
        display_fun_pt->x += wk_pt->x;
        display_fun_pt->y += wk_pt->y;
        wk_pt = wk_pt->next;
        display_fun_pt->x += wk_pt->x;
        display_fun_pt->y += wk_pt->y;
        wk_pt = NULL;
        display_fun_pt->x /= 3.0;
        display_fun_pt->y /= 3.0;

        display_fun = find_max(display_pt,0);
        delete display_fun;
        display_fun = find_max(display_fun_pt,1);

        draw_dark_poly(poly_list);

        if  (input_fun == NULL)
        {
            sprintf(buf, inputaligndiff_noinput,
                            display_pt->x, display_pt->y,
                            display_pt->max_val);
        }
        else
        {
            sprintf(buf, inputaligndiff,
                            display_pt->x, display_pt->y,
                            display_pt->max_val,
                            display_pt->input_fun_val,
                            display_pt->diff_val);
        }
        XtSetArg(wargs[0], XtNlabel, buf);
        XtSetValues(input_align_diff_label, wargs, 1);
    }
}


static void     quit(   Widget          w,
                        char            *ch,
                        XtPointer       call_data)
{
    if  (log_fname != NULL)
        fclose(log_fp);

    exit(0); 
}


static void     set_opt(        Widget          w,
                                char            *ch,
                                XtPointer       call_data)
{
/* **   fprintf(log_fp, "set_opt: begin\n");
/* */
    scoring_method = atoi(ch);
    Global_v = 1.0;
    Global_w = 1.0;
    process_change(changed_PARMS);
/* **   fprintf(log_fp, "set_opt: end\n");
/* */
}


static void     set_gaptype(    Widget          w,
                                char            *ch,
                                XtPointer       call_data)
{
    convex = atoi(ch);
    max_gap_size=0;
    redraw(NULL, NULL, NULL);
    process_change(changed_PARMS);
}


static void     set_var(        Widget          w,
                                char            *ch,
                                XtPointer       call_data)
{
/* **   fprintf(log_fp, "set_var: begin\n");
/* */
    opt_function = atoi(ch);
    with_gaps = (opt_function < 4) ? 0 : 1;
    Global_v = 1.0;
    if  (with_gaps)
        Global_w = 1.0;
    else
        Global_w = 0.0;
    process_change(changed_PARMS);
/* **   fprintf(log_fp, "set_var: end\n");
/* */
}

inline
static void     set_input_fun_vals()
{
    Arg         wargs[1];
    char        buf[512];
    char        *input_align1,
                *input_align2;
    int         str_w_gap;

/* **   fprintf(log_fp, "set_input_fun_vals: begin\n");
/* */

    if  (input_fun == NULL)
    {
/*      No input alignment, so make sure we display that  */

        XtSetArg(wargs[0], XtNlabel, inputaligninfonone);
        XtSetValues(input_align_info_label, wargs, 1);

        XtSetArg(wargs[0], XtNlabel, " ");
        XtSetValues(input_align_diff_label, wargs, 1);
    }
    else
    {
/*      Have an input alignment, so must get its counts, values    */

        if  (!weights_in_array) {
	    if (!convex) {
            	analyze_alignment_c(input_fun);
	    }
	    else {
            	analyze_alignment_convex_c(input_fun);
	    }
	}
        else {
	    if (!convex) {
            	analyze_alignment_w(input_fun);
	    }
	    else {
            	analyze_alignment_convex_w(input_fun);
	    }
	}

/*      Now display the new alignment information  */

        sprintf(buf, inputaligninfo,
                input_fun->match_ct, input_fun->mis_ct,
                input_fun->inordel_ct, input_fun->gaps_ct,
                input_fun->match_val, input_fun->mis_val,
                input_fun->inordel_val, input_fun->gaps_val);
        XtSetArg(wargs[0], XtNlabel, buf);
        XtSetValues(input_align_info_label,  wargs, 1);

        XtSetArg(wargs[0], XtNlabel, " ");
        XtSetValues(input_align_diff_label, wargs, 1);
    }

/* **   fprintf(log_fp, "set_input_fun_vals: end\n");
/* */
}


static void process_change(int     change_type)
{
    Arg         wargs[1];
    Poly        *poly_curr;
    char        buf[32];
    int         i;

/* **   fprintf(log_fp, "process_change: begin\n");
/* */
    if  (poly_list != NULL)
    {
        delete display_pt;
        display_pt = NULL;
        delete display_fun_pt;
        display_fun_pt = NULL;
        delete display_fun;
        display_fun = NULL;

        clear_polys();

        XtSetArg(wargs[0], XtNlabel, scoring_method_setting[scoring_method-1]);
        XtSetValues(b[1], wargs, 1);

        if  (input_fun == NULL)
        {
            XtSetArg(wargs[0], XtNlabel, varsetting[opt_function - 1]);
        }
        else
        {
            XtSetArg(wargs[0], XtNlabel, varsetting[opt_function + 8]);
        }
        XtSetValues(b[2], wargs, 1);

        sprintf(buf, gapsetting[with_gaps], Global_v, Global_w);
        XtSetArg(wargs[0], XtNlabel, buf);
        XtSetValues(b[3], wargs, 1);

        intTextSetString(align_disp, NULL);
        redraw(NULL, NULL, NULL);
    }
    else if  (change_type & changed_PARMS)
    {
        XtSetArg(wargs[0], XtNlabel, scoring_method_setting[scoring_method-1]);
        XtSetValues(b[1], wargs, 1);

        if  (input_fun == NULL)
        {
            XtSetArg(wargs[0], XtNlabel, varsetting[opt_function - 1]);
        }
        else
        {
            XtSetArg(wargs[0], XtNlabel, varsetting[opt_function + 8]);
        }
        XtSetValues(b[2], wargs, 1);

        sprintf(buf, gapsetting[with_gaps], Global_v, Global_w);
        XtSetArg(wargs[0], XtNlabel, buf);
        XtSetValues(b[3], wargs, 1);

        redraw(NULL, NULL, NULL);
    }

    if  (change_type & changed_ALIGN_PARMS)
    {
        if  (change_type & changed_ALIGN)
            set_strings(changed_ALIGN);
        set_input_fun_vals();
        changed_something &= changed_STRINGS;
    }
    else  //  (change_type & changed_STRINGS)
    {
        check_delete_input_fun_align();
        changed_something = changed_STRINGS;
    } 

/* **   fprintf(log_fp, "process_change: end\n");
/* */
}


static void     string_reverse( Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    char        a,
                *s,
                *temp;
    int         i,
                j,
                n;
/* **   fprintf(log_fp, "string_reverse: begin\n");
/* */
    string_index = atoi(client_data);
    if  (string_index == 1)
    {
        delete str1;
        s = intTextGetString(string[0]);
        n = strlen(s);
        str1 = new unsigned char[n + 1];
	assert(str1 != NULL);
        for (i=0; i < n; i++)
            str1[i] = s[n-i-1];
        str1[n] = '\0';
        intTextSetString(string[0], (char *)str1);
    }
    else if  (string_index == 2)
    {
        delete str2;
        s = intTextGetString(string[1]);
        n = strlen(s);
        str2 = new unsigned char[n + 1];
	assert(str2 != NULL);
        for (i=0; i < n; i++)
            str2[i] = s[n-i-1];
        str2[n] = '\0';
        intTextSetString(string[1], (char *)str2);
    }

    check_delete_input_fun_align();
    process_change(changed_PARMS_STRINGS);

/* **   fprintf(log_fp, "string_reverse: end\n");
/* */
}


/*
 *  Additional functions needed in order to remove the Motif stuff.
 */


Widget          create_form(    char    *name,
                                Widget  parent,
                                Widget  left,
                                Widget  top,
                                int     border_width,
                                int     int_distance,
                                Arg     *args,
                                int     n)
{
    XtSetArg(args[n], XtNborderWidth, border_width); n++;
    XtSetArg(args[n], XtNdefaultDistance, int_distance); n++;
    if  (left != NULL)
    {
        XtSetArg(args[n], XtNfromHoriz, left); n++;
    }
    if  (top != NULL)
    {
        XtSetArg(args[n], XtNfromVert, top); n++;
    }

    return XtCreateManagedWidget(name, formWidgetClass, parent, args, n);
}


Widget          create_command( char            *name,
                                Widget          parent,
                                Widget          left,
                                Widget          top,
                                void            (*callback)(Widget, char*, XtPointer),
                                XtPointer       data,
                                Arg             *args,
                                int             n)
{
    Widget      command;

    if  (left != NULL)
    {
        XtSetArg(args[n], XtNfromHoriz, left); n++;
    }
    if  (top != NULL)
    {
        XtSetArg(args[n], XtNfromVert, top); n++;
    }

    command = XtCreateManagedWidget(name, commandWidgetClass, parent, args, n);
    XtAddCallback(command, XtNcallback, (callback_type) callback, data);

    return command;
}


Widget          create_label(   char    *name,
                                Widget  parent,
                                Widget  left,
                                Widget  top,
                                int     border_width,
                                Arg     *args,
                                int     n)
{
    XtSetArg(args[n], XtNborderWidth, border_width); n++;

    if  (left != NULL)
    {
        XtSetArg(args[n], XtNfromHoriz, left); n++;
    }
    if  (top != NULL)
    {
        XtSetArg(args[n], XtNfromVert, top); n++;
    }

    return XtCreateManagedWidget(name, labelWidgetClass, parent, args, n);
}


static String   text_translations =
                        "Ctrl<Key>J:  no-op(RingBell)\n"
                        "Ctrl<Key>M:  no-op(RingBell)\n"
                        "Ctrl<Key>N:  no-op(RingBell)\n"
                        "Ctrl<Key>P:  no-op(RingBell)\n"
                        "Ctrl<Key>V:  no-op(RingBell)\n"
                        "Ctrl<Key>Z:  no-op(RingBell)\n"
                        "Meta<Key>I:  no-op(RingBell)\n"
                        "Meta<Key>V:  no-op(RingBell)\n"
                        "Meta<Key>Z:  no-op(RingBell)\n"
                        "<Key>Down:  no-op(RingBell)\n"
                        "<Key>Up:  no-op(RingBell)\n"
                        "<Key>Linefeed:  no-op(RingBell)\n"
                        "<Key>Return:  no-op(RingBell)\n";

Widget          create_text(    char    *name,
                                Widget  parent,
                                Widget  left,
                                Widget  top,
                                char    *string,
                                int     edit_flag,
                                int     height,
                                Arg     *args,
                                int     n)
{
    Widget      text;

    XtSetArg(args[n], XtNstring, (string != NULL ? string : "")); n++;

    if  (left != NULL)
    {
        XtSetArg(args[n], XtNfromHoriz, left); n++;
    }
    if  (top != NULL)
    {
        XtSetArg(args[n], XtNfromVert, top); n++;
    }

    if  (edit_flag)
    {
        XtSetArg(args[n], XtNeditType, XawtextEdit); n++;
    }
    else
    {
        XtSetArg(args[n], XtNdisplayCaret, 0); n++;
    }

    if  (height == 0)
        height = char_height + SCROLLBAR_SIZE;
    XtSetArg(args[n], XtNheight, height); n++;

    XtSetArg(args[n], XtNscrollHorizontal, XawtextScrollWhenNeeded);  n++;
    text = XtCreateManagedWidget(name, asciiTextWidgetClass, parent, args, n);

    XtOverrideTranslations(text, XtParseTranslationTable(text_translations));

    return text;
}


Widget          create_menu(    char            *name,
                                Widget          parent,
                                Widget          left,
                                Widget          top,
                                int_menu_struct *menu_data,
                                Arg             *args,
                                int             n)
{
    Arg         iargs[10];
    int         i;
    Widget      button,
                entry,
                menu;

    if  (left != NULL)
    {
        XtSetArg(args[n], XtNfromHoriz, left); n++;
    }
    if  (top != NULL)
    {
        XtSetArg(args[n], XtNfromVert, top); n++;
    }

    button = XtCreateManagedWidget(name, menuButtonWidgetClass, parent, args, n);

    menu = XtCreatePopupShell("menu", simpleMenuWidgetClass, button, NULL, 0);

    for (i=0; menu_data[i].title != NULL; i++)
    {
	// For the separator lines to be interpreted as proper Xaw separator
	// lines the menu item's title must consist of a single dash. Not
	// enabled because the default separator height is not large enough,
	// the default color is black (rather than gray), and I do not know
	// how to change either. [It's via resources, but so what?]
	if ((*menu_data[i].title == '-') && (strlen(menu_data[i].title) == 1))
	    entry = XtCreateManagedWidget(menu_data[i].title, smeLineObjectClass, menu, NULL, 0);
	else
	    entry = XtCreateManagedWidget(menu_data[i].title, smeBSBObjectClass, menu, NULL, 0);

        XtAddCallback(entry, XtNcallback, (callback_type)(menu_data[i]).callback,
                      menu_data[i].arg);
    }
    return button;
}


static void     string_change(  Widget          w,
                                char            *client_data,
                                XtPointer       call_data)
{
    process_change(changed_STRINGS);
}


inline
static Pixmap   intGetDarkPixmap(       Display *dpy,
                                        Window  win)
{
    Pixmap      pm;

    pm = XCreatePixmap(dpy, root, 1, 1, depth);
    XSetForeground(dpy, fggc, gray.pixel);
	XDrawPoint(dpy, pm, fggc, 0, 0);
    XSetForeground(dpy, fggc, black.pixel);

    return pm;
}


/*
 * This function is not used, but was not removed on the off chance
 * that someone wanted a shade between dark and light.
 */
inline
static Pixmap   intGetMediumPixmap(     Display *dpy,
                                        Window  win)
{
    Pixmap      pm;

    pm = XCreatePixmap(dpy, root, 4, 4, depth);

    XDrawPoint(dpy, pm, fggc, 0, 0);
    XDrawPoint(dpy, pm, bggc, 1, 0);
    XDrawPoint(dpy, pm, bggc, 0, 1);
    XDrawPoint(dpy, pm, fggc, 1, 1);

    XDrawPoint(dpy, pm, fggc, 2, 0);
    XDrawPoint(dpy, pm, bggc, 3, 0);
    XDrawPoint(dpy, pm, bggc, 2, 1);
    XDrawPoint(dpy, pm, fggc, 3, 1);

    XDrawPoint(dpy, pm, fggc, 0, 2);
    XDrawPoint(dpy, pm, bggc, 0, 3);
    XDrawPoint(dpy, pm, bggc, 1, 2);
    XDrawPoint(dpy, pm, fggc, 1, 3);

    XDrawPoint(dpy, pm, fggc, 2, 2);
    XDrawPoint(dpy, pm, bggc, 2, 3);
    XDrawPoint(dpy, pm, bggc, 3, 2);
    XDrawPoint(dpy, pm, fggc, 3, 3);

    return pm;
}

inline
static Pixmap   intGetLightPixmap(Display *dpy, Window win)
{
    Pixmap      pm;

    pm = XCreatePixmap(dpy, root, 4, 4, depth);

    XDrawPoint(dpy, pm, bggc, 0, 0);
    XDrawPoint(dpy, pm, bggc, 1, 0);
    XDrawPoint(dpy, pm, bggc, 0, 1);
    XDrawPoint(dpy, pm, bggc, 1, 1);	// was fggc

    XDrawPoint(dpy, pm, bggc, 2, 0);
    XDrawPoint(dpy, pm, bggc, 3, 0);
    XDrawPoint(dpy, pm, bggc, 2, 1);
    XDrawPoint(dpy, pm, bggc, 3, 1);	// was fggc

    XDrawPoint(dpy, pm, bggc, 0, 2);
    XDrawPoint(dpy, pm, bggc, 0, 3);
    XDrawPoint(dpy, pm, bggc, 1, 2);
    XDrawPoint(dpy, pm, bggc, 1, 3);	// was fggc

    XDrawPoint(dpy, pm, bggc, 2, 2);
    XDrawPoint(dpy, pm, bggc, 2, 3);
    XDrawPoint(dpy, pm, bggc, 3, 2);
    XDrawPoint(dpy, pm, bggc, 3, 3);	// was fggc

    return pm;
}


static char     *intTextGetString(      Widget  w)
{
    Arg         args[1];
    char        *s;
  
    XtSetArg(args[0], XtNstring, &s);
    XtGetValues(w, args, 1);

    return s;
}


static void     intTextSetString(       Widget  w,
                                        char    *s)
{
    Arg         args[1];

    if  (s == NULL)
        XtSetArg(args[0], XtNstring, "");
    else
        XtSetArg(args[0], XtNstring, s);

    XtSetValues(w, args, 1);
    XawTextSetInsertionPoint(w, (s == NULL ? 0 : strlen(s)));
}


/*
 * The data structures for each of the menus.
 */

int_menu_struct file_menu_data[] =
                        { {"Load Aligned Strings", input_align_io_show, NULL},
                          {"Load String #1", file_io_show, "1L"},
                          {"Load String #2", file_io_show, "2L"},
			  // A separator line composed of dash characters. If
			  // you want a proper SmeLine separator, change the
			  // menu title from multiple dashes to a single dash.
			  // See create_menu() below.
			  {"---------------------------",do_nothing," "},
                          {"Reverse String #1", string_reverse, "1"},
                          {"Reverse String #2", string_reverse, "2"},
			  {"---------------------------",do_nothing," "},
                          {"Save String #1", file_io_show, "1S"},
                          {"Save String #2", file_io_show, "2S"},
                          {"Save All Polygons", align_io_show, "A"},
                          {"Save Min Distance Polygon", align_io_show, "M"},
                          {"Save Dark Polygon", align_io_show, "D"},
                          {"Save Polygon Alignments", align_io_show, "P"},
			  {"---------------------------",do_nothing," "},
                          {"New Log File", align_io_show, "L"},
                          /* {"Scoring Matrix", weight_io_show, NULL}, */
			  {"---------------------------",do_nothing," "},
                          {"Quit", quit, "Quit"},
                          {NULL, NULL, NULL}
                        };

static int_menu_struct  alignment_menu_data[] =
                        { {"  Global ", set_opt, "1"},
                          {"  End-gaps free", set_opt, "2"},
                          {"  Local", set_opt, "3"},
                          {"  1st as sub-string of 2nd", set_opt, "4"},
                          {NULL, NULL, NULL}
                        };

static int_menu_struct  opt_function_menu_data[] =
        { {"-------------------- NO GAPS ---------------------",do_nothing," "},
          {"  (#matches)-X*(#mismatches)-Y*(#indels)", set_var, "1"},
          {"  X*(#matches)-(#mismatches)-Y*(#indels)", set_var, "2"},
          {"  X*(#matches)-Y*(#mismatches)-(#indels)", set_var, "3"},
          {"--------------------- GAPS -----------------------",do_nothing," "},
          {"  (#matches)-(#mismatches)-X*(#indels)-Y*(#gaps)", set_var, "4"},
          {"  (#matches)-X*(#mismatches)-(#indels)-Y*(#gaps)", set_var, "5"},
          {"  (#matches)-X*(#mismatches)-Y*(#indels)-(#gaps)", set_var, "6"},
          {"  X*(#matches)-(#mismatches)-(#indels)-Y*(#gaps)", set_var, "7"},
          {"  X*(#matches)-(#mismatches)-Y*(#indels)-(#gaps)", set_var, "8"},
          {"  X*(#matches)-Y*(#mismatches)-(#indels)-(#gaps)", set_var, "9"},
          {NULL, NULL, NULL}
        };

static int_menu_struct  opt_function_menu_data_w[] =
        { {"---------------------- NO GAPS  ------------------------",do_nothing," "},
          {"  (match val)+X*(mismatch val)-Y*(indel val)", set_var, "1"},
          {"  X*(match val)+(mismatch val)-Y*(indel val)", set_var, "2"},
          {"  X*(match val)+Y*(mismatch val)-(indel val)", set_var, "3"},
          {"------------------------ GAPS --------------------------",do_nothing," "},
          {"  (match val)+(mismatch val)-X*(indel val)-Y*(gap val)", set_var, "4"},
          {"  (match val)+X*(mismatch val)-(indel val)-Y*(gap val)", set_var, "5"},
          {"  (match val)+X*(mismatch val)-Y*(indel val)-(gap val)", set_var, "6"},
          {"  X*(match val)+(mismatch val)-(indel val)-Y*(gap val)", set_var, "7"},
          {"  X*(match val)+(mismatch val)-Y*(indel val)-(gap val)", set_var, "8"},
          {"  X*(match val)+Y*(mismatch val)-(indel val)-(gap val)", set_var, "9"},
          {NULL, NULL, NULL}
        };

static int_menu_struct  gaptype_menu_data[] =
                        { {"  g = N ", set_gaptype, "0"},
                          /*{"  g = N'", set_gaptype, "1"},*/
			  /*{"  g = log(N)", set_gaptype, "2"},*/
                          {"  g = log(N)+1 ", set_gaptype, "3"},
                          {"  g = log(N+1) ", set_gaptype, "4"},
                          {"  g = sqrt(N) ", set_gaptype, "5"},
                          {"  g = sqrt(N)+1 ", set_gaptype, "6"},
                          {"  g = sqrt(N+1) ", set_gaptype, "7"},
                          {NULL, NULL, NULL}
                        };

static int_menu_struct  scoretype_menu_data[] =
                        {
			  {"No Scoring Matrix",weight_io_show, "1"},
			  {"-----------------------------------",do_nothing," "},
			  {"PAM 250 ", weight_io_show, "2"},
			  {"PAM 250 + 8 ", weight_io_show, "3"},
			  {"-----------------------------------",do_nothing," "},
			  {"BLOSUM 80 ", weight_io_show, "4"},
			  {"BLOSUM 62 ", weight_io_show, "5"},
			  {"BLOSUM 45 ", weight_io_show, "6"},
			  {"-----------------------------------",do_nothing," "},
			  {"Gonnet", weight_io_show, "7"},
			  {"Gribskow", weight_io_show, "8"},
			  {"McClure", weight_io_show, "9"},
			  {"Taylor ", weight_io_show, "10"},
			  {"-----------------------------------",do_nothing," "},
			  {"Other Scoring Matrix ", weight_io_show, "0"},
			  {NULL, NULL, NULL}
                        };

static int_menu_struct  get_polygons_menu_data[] =
                        { {"  Find All Polygons               ", get_mult_polys, "1"},
                          {"  Find Closest to Input Alignment ", get_mult_polys, "0"},
                          {"  Find Polygon(s) for Point       ", get_polys_pt_show, NULL},
                          {NULL, NULL, NULL}
                        };
/*static int_menu_struct	get_polygons_menu_data[4];
	//int_menu_strct *ptr = get_polygons_menu_data;
	get_polygons_menu_data[0].title = "  Find All Polygons               ";
	get_polygons_menu_data[0].callback = get_mult_polys;
	get_polygons_menu_data[0].arg = "1";
	get_polygons_menu_data[1].title = "  Find All Polygons               ";
	get_polygons_menu_data[1].callback = get_mult_polys;
	get_polygons_menu_data[1].arg = "1";
	get_polygons_menu_data[2].title = "  Find All Polygons               ";
	get_polygons_menu_data[2].callback = get_mult_polys;
	get_polygons_menu_data[2].arg = "1";
	get_polygons_menu_data[3].title = NULL;
	get_polygons_menu_data[3].callback = NULL;
	get_polygons_menu_data[3].arg = NULL;*/


/********** MAIN *******************/


main(   int     argc,
        char    *argv[])
{
    Arg         wargs[15];
    char        *button1 = "Opt: Global                   ";
    char        *button2 = "Var: c1(mat)-y(mis)-x(i/d)            ";
    char        *button3 = "c1 = 1.00       ";
    char        buf[40],
                **new_argv;
    int         canvas_height,
                canvas_width,
                str_width;
    int         i,j,k,
                n,
                new_argc,
                x,
                y;
    XGCValues   gcvals;
    Widget      left,
                new_top,
                top;
    char        answer;

/*
**  Initialize some global variables.
*/
    log_fname = NULL;
    log_fp = stdout;

    array = NULL;
    str1 = NULL;
    str2 = NULL;

/*
 *  Add arguments to the command line to guarantee that the thickness of
 *  all of the horizontal scrollbars are of size SCROLLBAR_THICKNESS.
 *
 *  This seems to be the only way (other than implementing resource
 *  managing routines) to set this value inside the text widgets.
 */
    new_argv = new char*[argc + 3];
    for (new_argc=0; new_argc < argc; new_argc++)
        new_argv[new_argc] = argv[new_argc];
    new_argv[new_argc++] = "-xrm";
    sprintf(buf, "*thickness: %d", SCROLLBAR_THICKNESS);
    new_argv[new_argc++] = buf;
    new_argv[new_argc] = NULL;

/*
 *      The Legal Disclaimer
 */
    printf("\n");
    printf("\n");
    printf("\n");
    printf("XPARAL 2.0\nParametric Sequence Alignment\n");
    printf("\n\nUniversity of California, Davis\nComputer Science Department\n");
    printf("\n");
    printf("\n");
    /* Busy Wait */
    for ( i=1; i < 1000; i++) {
      for (j=1; j < 1000; j++) {
	printf("");
      }
    }
      
    /*    printf("Do you accept? (y/n): ");
	  while(1) {
	  answer=getchar();
	  if (answer=='y' || answer=='Y') {
	  break;
	  }
	  else if (answer=='n' || answer=='N') {
	  exit(0);
	  }
	  }
    */
      
   

/*
**  Set up and go.
*/
    toplevel = XtOpenApplication(&app, "XParal", NULL, 0, &new_argc, new_argv,
	    NULL, applicationShellWidgetClass, NULL, 0);
    XtSetArg(wargs[0], XtNinput, (XtArgVal)TRUE);
    XtSetValues(toplevel, wargs, 1);
    dpy = XtDisplay(toplevel);

/*
 * Create the GC's used when drawing to the canvas.
 *
 *    This use of fixed Black and White should be changed to use the 
 *    resource manager to find the foreground and background colors.
 */
    screen = DefaultScreen(dpy);
    root = RootWindow(dpy, screen);
    depth = DefaultDepth(dpy, screen);
 
    Colormap	colors;
    char	grayRGB[] = "#AAAAAA";
    char	blackRGB[] = "#000000";
 
    colors = DefaultColormap(dpy, 0);

    gcvals.foreground = BlackPixel(dpy, screen);
    gcvals.background = WhitePixel(dpy, screen);
    fggc = XCreateGC(dpy, root, GCForeground|GCBackground, &gcvals);
 
    XParseColor(dpy, colors, grayRGB, &gray);
    XAllocColor(dpy, colors, &gray);
    XParseColor(dpy, colors, blackRGB, &black);
    XAllocColor(dpy, colors, &black);

    i = gcvals.background;
    gcvals.background = gcvals.foreground;
    gcvals.foreground = i;
    bggc = XCreateGC(dpy, root, GCForeground|GCBackground, &gcvals);

/*  Create form area */
    bb = create_form("bb", toplevel, NULL, NULL, 0, 2, wargs, 0);

/*  Create the menu buttons */
    menu = create_form("menubar", bb, NULL, NULL, 1, 0, wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 0);  n++;
    XtSetArg(wargs[n], XtNhighlightThickness, 0);  n++;
    XtSetArg(wargs[n], XtNinternalWidth, 10);  n++;
    XtSetArg(wargs[n], XtNinternalHeight, 10);  n++;
    left = create_menu("File ", menu, NULL, NULL, file_menu_data, wargs, n);

/*
 * Retrieve the width and height of the default font used by the 
 * text, command and label widgets.
 *
 * Had to delay this computation until after one of the widgets was 
 * created (there doesn't seem to be another way to find out the 
 * default font).
 *
 * And, just in case even this doesn't work, use the dimensions of 
 * the 'fixed' font.
 */

    fixed_font = NULL;
    XtSetArg(wargs[0], XtNfont, &fixed_font);
    XtGetValues(left, wargs, 1);

    if  (fixed_font != NULL)
    {
        char_width = XTextWidth(fixed_font, "x", 1);
        char_height = fixed_font->ascent + fixed_font->descent;
    }
    else
    {
        fixed_font = XLoadQueryFont(dpy, "fixed");
        assert (fixed_font != NULL);

        char_width = XTextWidth(fixed_font, "x", 1) + 1;
        char_height = fixed_font->ascent + fixed_font->descent + 1;
    }

    canvas_width = 800;
    canvas_height = 650 - 4 * char_height - 48;
    str_width = canvas_width - 6 * char_width - 8;

/*  Resume the construction of the menus (and resetting wargs) */

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 0);  n++;
    XtSetArg(wargs[n], XtNhighlightThickness, 0);  n++;
    XtSetArg(wargs[n], XtNinternalWidth, 10);  n++;
    XtSetArg(wargs[n], XtNinternalHeight, 10);  n++;

    input_align_io_create(left);
    file_io_create(left);
    align_io_create(left);
    weight_io_create(left);

    left = create_menu("Alignment ", menu, left, NULL, alignment_menu_data,
                       wargs, n);

    left = create_menu("Opt Function ", menu, left, NULL, opt_function_menu_data,
                       wargs, n);

    left = create_command("Constants ", menu, left, NULL, const_plane_show,
                          NULL, wargs, n);
    const_plane_create(left);

    left = create_command("Ranges ", menu, left, NULL, xy_region_show, NULL,
                          wargs, n);
    xy_region_create(left);

    left = create_menu("Scoring System ", menu, left, NULL,
		       scoretype_menu_data, wargs, n);

    get_polys_pt_create(left);

    left = create_menu("Gap Type ", menu, left, NULL, gaptype_menu_data,
                       wargs, n);


    left = create_menu("Get Polygons ", menu, left, NULL, get_polygons_menu_data,
                       wargs, n);

/*  Create string[0] area */
    top = create_form("string1form", bb, NULL, menu, 0, 0, wargs, 0);

    left = create_label("Seq 1:", top, NULL, NULL, 0, wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2);  n++;
    XtSetArg(wargs[n], XtNwidth, str_width); n++;
    string[0] = create_text("string1", top, left, NULL, "", 1, 0, wargs, n);

    XtSetArg(wargs[0], XtNtextSource, &left);
    XtGetValues(string[0], wargs, 1);
    XtAddCallback(left, XtNcallback, (XtCallbackProc) string_change, NULL);

/*  Create string[1] area */
    top = create_form("string2form", bb, NULL, top, 0, 0, wargs, 0);

    left = create_label("Seq 2:", top, NULL, NULL, 0, wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2);  n++;
    XtSetArg(wargs[n], XtNwidth, str_width); n++;
    string[1] = create_text("string2", top, left, NULL, "", 1, 0, wargs, n);

    XtSetArg(wargs[0], XtNtextSource, &left);
    XtGetValues(string[1], wargs, 1);
    XtAddCallback(left, XtNcallback, (XtCallbackProc) string_change, NULL);


/*  Create align_disp area */
    align_label = create_label("Align:\n    ", bb, NULL, top, 0, wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2);  n++;
    XtSetArg(wargs[n], XtNwidth, canvas_width);  n++;
    align_disp = create_text("align_disp", bb, align_label, top, "", 0, 
                             2 * char_height + 4 + SCROLLBAR_SIZE, wargs, n);

    align_ct_label = create_label(align_ct_blank, bb, align_disp, top, 0, wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNborderWidth, 2);  n++;
    left = create_command(" Prev ", bb, align_disp, align_ct_label,
                          align_show_prev, NULL, wargs, n);
    left = create_command(" Next ", bb, left, align_ct_label,
                          align_show_next, NULL, wargs, n);

/*  Create command area */
/* **   fprintf(log_fp, "Get to create command widget\n");
/* */
    b[1] = create_label(button1, bb, NULL, align_disp, 2, wargs, 0);
    b[2] = create_label(button2, bb, b[1], align_disp, 2, wargs, 0);
    b[3] = create_label(button3, bb, b[2], align_disp, 2, wargs, 0);

    weight_label = create_label(weightoptionconst, bb, b[3], align_disp, 2, wargs, 0);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, char_height); n++;
    tracker = create_label("Position          ", bb, b[3], weight_label,
                           0, wargs, n);

/* **   fprintf(log_fp, "get to creating canvas\n");
/* */

/*  Create the canvas graphics area */
    n = 0;
    XtSetArg(wargs[n], XtNfromVert, b[2]);  n++;
    XtSetArg(wargs[n], XtNhorizDistance, 10);  n++;
    XtSetArg(wargs[n], XtNvertDistance, 10);  n++;
    XtSetArg(wargs[n], XtNborderWidth, 0);  n++;
    XtSetArg(wargs[n], XtNwidth, canvas_width); n++;
    XtSetArg(wargs[n], XtNheight, canvas_height); n++;
    canvas = XtCreateManagedWidget("canvas", simpleWidgetClass, bb, wargs, n);

    XtAddEventHandler(canvas, PointerMotionMask, FALSE, (XtEventHandler) track_pos, tracker);
    XtAddEventHandler(canvas, LeaveWindowMask, FALSE, (XtEventHandler) clear_pos, tracker);
    XtAddEventHandler(canvas, ButtonPressMask, FALSE, (XtEventHandler) explore, tracker);
    XtAddEventHandler(canvas, StructureNotifyMask, FALSE, (XtEventHandler) resize, NULL);
    XtAddEventHandler(canvas, ExposureMask, FALSE, (XtEventHandler) expose, NULL);

/*  Create input alignment information area  */
    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, char_height); n++;
    input_align_title_label = create_label(inputaligntitle, bb,
                                           canvas, tracker, 0,
                                           wargs, n);

    n = 0;
    input_align_info_label = create_label(inputaligninfonone, bb,
                                          canvas, input_align_title_label, 0,
                                          wargs, n);

    n = 0;
    XtSetArg(wargs[n], XtNinternalHeight, char_height); n++;
    XtSetArg(wargs[n], XtNlabel, inputaligndiff_blank); n++;
    input_align_diff_label = create_label((char *)inputaligndiff, bb,
                                          canvas, input_align_info_label, 0,
                                          wargs, n);

/* **   fprintf(log_fp, "Realizing widgets\n");
/* */

    XtRealizeWidget(toplevel);

    Atom	wmDeleteMessage = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    Status	status = XSetWMProtocols(dpy, XtWindow(toplevel), &wmDeleteMessage, 1);

    if (status == 0)
	{
	fprintf(log_fp, "There was an error, Elvis.\n");
	if (status == BadAlloc)
	    {
	    fprintf(log_fp, "BadAlloc error, Elvis.\n");
	    }
	else if (status == BadWindow)
	    {
	    fprintf(log_fp, "BadWindow error, Elvis.\n");
	    }
	}

/*  Main stuff */
    init_globals();
    process_change(changed_PARMS);

/* **   fprintf(log_fp, "main loop\n");
/* */
   while (true)
    {
    XEvent event;
    XNextEvent(dpy, &event);

    if (event.type == ClientMessage && event.xclient.data.l[0] == wmDeleteMessage)
	{
	// Prevents an error message from being displayed when the Window Manager
	// closes the window / allows dialog boxes to be closed rather than
	// crashing out of the application.
	//
	// Trap for clicks on the close boxes of each of the dialog boxes and if
	// found, close them rather than quitting the entire application.
	if (event.xclient.window == XtWindow(xy_region))
	    XtPopdown(xy_region);
	else if (event.xclient.window == XtWindow(const_plane))
	    XtPopdown(const_plane);
	else if (event.xclient.window == XtWindow(get_polys_pt))
	    XtPopdown(get_polys_pt);
	else if (event.xclient.window == XtWindow(align_io))
	    XtPopdown(align_io);
	else if (event.xclient.window == XtWindow(file_io))
	    XtPopdown(file_io);
	else if (event.xclient.window == XtWindow(weight_io))
	    XtPopdown(weight_io);
	// This will close the entire app now. Consider displaying an
	// "Are you sure?" dialog box.
	else
	    break;
	}
    XtDispatchEvent(&event);
    }


XtDestroyApplicationContext(app);

return 0;
}

