// $Id: Util.java,v 1.12 2003/09/22 21:31:57 hchen Exp $

import java.io.*;
import java.util.*;
import gnu.getopt.*;

/**
 * Common tools
 */
public class Util
{
  /**
   * This part deals with writing logs and outputs
   */
  public static void warn(int severity, int source, String message,
			  String filename, int lineno)
  {
    if (getVerbosity() >= severity)
      printError(severity, source, formatMessage(message, filename, lineno));
  }

  public static void warn(int severity, int source, String message,
			  String filename)
  {
    warn(severity, source, message, filename, -1);
  }
  
  public static void warn(int severity, int source, String message)
  {
    warn(severity, source, message, null, -1);
  }

  public static void die(int source, String message, String filename, int lineno)
  {
    printError(ERROR, source, formatMessage(message, filename, lineno));
    System.exit(1);
  }

  public static void die(int source, String message, String filename)
  {
    die(source, message, filename, -1);
  }
			     
  public static void die(int source, String message)
  {
    die(source, message, null, -1);
  }
			   
  static void printError(int severity, int source, String message)
  {
    final int NO_POSITION = 1, LAST_POSITION = 2, CALL_STACK = 3;
    int printLocation;

    stderr.print(severityLabels[severity]);
    stderr.print(": ");
    stderr.print(programName);
    stderr.print(": ");
    stderr.print(sourceLabels[source]);
    stderr.print(": ");
    stderr.print(message);
    // decide which location information to print
    switch(source)
    {
      case INTERNAL:
	if (isDebug())
	  printLocation = CALL_STACK;
	else
	  printLocation = LAST_POSITION;
	break;
	
      default:
	if (isDebug())
	  printLocation = LAST_POSITION;
	else
	  printLocation = NO_POSITION;
    }
    switch(printLocation)
    {
      case CALL_STACK:
	stderr.println();
	stderr.print(getLocation(true));
	stderr.flush();
	break;

      case LAST_POSITION:
	stderr.print(" at ");
	stderr.println(getLocation(false));
	// because stderr is autoflush,
	// so calling stderr.flush() is unnecessary
	break;

      case NO_POSITION:
      default:
	stderr.println();
	break;
    }
  }

  public static boolean isDebug()
  {
    return verbosity == DEBUG;
  }

  public static void setVerbosity(int verbosity)
  {
    Util.verbosity = verbosity;
  }

  public static int getVerbosity()
  {
    return verbosity;
  }
  
  /**
   * Get the last program point on the call stack before calling a
   * function in this class
   */
  private static String getLocation(boolean fullStack)
  {
    StringBuffer sb;
    int i, begin, length;
    String str, prefix;
    Throwable throwable;

    // Get class and method names at the call site by parsing
    // Throwable messages
    throwable = new Throwable();
    prefix = "\tat Util.";
    throwable.printStackTrace(pw);
    sb = sw.getBuffer();
    length = sb.length();
    // Find the highest call frame that is not in Util
    for (i = 0; i < length; i++)
      if (sb.charAt(i) == '\t')
      {
	if (sb.indexOf(prefix, i) != i)
	  break;
      }

    if (i >= length)
    {
      Util.stderr.println("\nCannot find \\t from the exception message in Util.getLocation");
      System.exit(1);
    }
    if (fullStack)
    {
      str = sb.substring(i);
      sb.setLength(0);
      return str;
    }
    
    begin = i + 4;
    for (i = begin; i < length; i++)
      if (sb.charAt(i) == '\n')
	break;
    str = sb.substring(begin, i);
    sb.setLength(0);
    return str;
  }

  private static String getProgramName()
  {
    StringBuffer sb;
    int index, begin, end;
    Throwable throwable;
    String str;

    throwable = new Throwable();
    throwable.printStackTrace(pw);
    sb = sw.getBuffer();
    index = sb.lastIndexOf("\t");
    if (index == -1 || (begin = sb.indexOf(" ", index + 1)) == -1 ||
	(end = sb.indexOf(".", begin + 1)) == -1)
    {
      sb.setLength(0);
      return "(Cannot get program name)";
    }
    else
    {
      str = sb.substring(begin + 1, end);
      sb.setLength(0);
      return str;
    }
  }
  
  private static String formatMessage(String message, String filename, int lineno)
  {
    StringBuffer str;

    str = new StringBuffer(message);
    if (lineno >= 0)
      str.append(" on line " + lineno);
    if (filename != null)
      str.append(" in file " + filename);

    return str.toString();
  }

  // Message severity
  static final int ERROR = 0, WARNING = 1, INFO = 2, DEBUG = 3;
  static final String[] severityLabels =
  { "ERROR", "WARNING", "INFO", "DEBUG" };
  // Message source
  static final int INTERNAL = 0, FILE_FORMAT = 1, FILE_IO = 2, EXTERNAL = 3;
  static final String[] sourceLabels =
  { "Internal", "File format", "File I/O", "External" };
  
  public static PrintWriter stdout, stderr;
  static StringWriter sw;
  static PrintWriter pw;
  static int verbosity;
  static String programName;

  /**
   * This part deals with processing command line options
   */
  public static void processOption(int ch, Getopt opt, String usage,
				   boolean isStrict)
    throws IOException
  {
    int i;
    String label;
    
    switch(ch)
    {
      case 'h':
	stdout.println(usage);
	System.exit(0);

      case 'v':
	label = opt.getOptarg();
	for (i = 0; i < severityLabels.length; i++)
	{
	  if (label.equalsIgnoreCase(severityLabels[i]))
	    break;
	}
	if (i < severityLabels.length)
	  setVerbosity(i);
	else
	  warn(ERROR, FILE_FORMAT,
	       "Invalid parameter to the -" + (char)ch + " option: " + label);
	break;
 
      case 'E':
	label = opt.getOptarg();
	if (label.equals("-"))
	  stderr = new PrintWriter(new OutputStreamWriter(System.err), true);
	else
	  stderr = new PrintWriter(new BufferedWriter
				   (new FileWriter(label, true)), true);
	break;

      case 'O':
	label = opt.getOptarg();
	stdout = new PrintWriter(new BufferedWriter(openWriter(label, true)),
				 true);
	break;
	
      case 'V':
	printVersion();
	System.exit(0);

      case '?':
	// opt.getopt() has already printed a warning message
	if (isStrict)
	{
	  stderr.println(usage);
	  System.exit(1);
	}
	break;

      default:
	die(FILE_FORMAT, "Unknown command line option -" + (char)ch);
    }
  }

  public static void processOption(int ch, Getopt opt, String usage)
    throws IOException
  {
    processOption(ch, opt, usage, true);
  }
  
  public static final String getOptionString()
  {
    return optionString;
  }

  public static final String getOptionUsage()
  {
    return optionUsage;
  }

  public static final LongOpt[] getLongOpts()
  {
    return longOpts;
  }
  
  private static final String optionString = "hv:E:O:V";

  private static final String optionUsage =
    "  -h\t\tDisplay help\n" +
    "  -v <level>\tSet verbosity level.  <level> can be: error warning info debug\n" +
    "  --help\tSame as -h\n" +
    "  -E <file>\tRedirect log messages from stderr to <file>\n" +
    "  -O <file>\tRedirect output from stdout to <file>\n" +
    "  -V\t\tDisplay version\n";

  private static final LongOpt[] longOpts = new LongOpt[] {
      new LongOpt("help", LongOpt.NO_ARGUMENT, null, 'h')};

  /**
   * This part deals with opening files while correctly handles "-"
   */
  public static final InputStream openInputStream(String filename)
    throws IOException
  {
    if (filename.equals("-"))
      return System.in;
    else
      return new FileInputStream(filename);
  }

  public static final Reader openReader(String filename)
    throws IOException
  {
    if (filename.equals("-"))
      return new InputStreamReader(System.in);
    else
      return new FileReader(filename);
  }

  public static final OutputStream openOutputStream(String filename,
						    boolean append)  
    throws IOException
  {
    if (filename.equals("-"))
      return System.out;
    else
      return new FileOutputStream(filename, append);
  }

  public static final OutputStream openOutputStream(String filename)  
    throws IOException
  {
    return openOutputStream(filename, false);
  }

  public static final Writer openWriter(String filename, boolean append)  
    throws IOException
  {
    if (filename.equals("-"))
      return new OutputStreamWriter(System.out);
    else
      return new FileWriter(filename, append);
  }

  public static final Writer openWriter(String filename)
    throws IOException
  {
    return openWriter(filename, false);
  }
  
  /**
   * This part is for miscellaneous functions
   */
  public static void readEntryFunctions(String filename, HashMap entryFunctions)
    throws IOException
  {
    BufferedReader reader;
    String str;
    int lineno;
    
    reader = new BufferedReader(openReader(filename));
    for (lineno = 1;
	 (str = reader.readLine()) != null;
	 lineno++)
    {
      if (entryFunctions.containsKey(str))
      {
	warn(WARNING, FILE_IO, "redundant entry function " + str,
		  filename, lineno);
      }
      else
      {
	entryFunctions.put(str, null);
      }
    }
    reader.close();
  }

  public static String readString(Input reader, String errorMessage)
    throws IOException
  {
    String str;
    
    reader.nextToken();
    str = reader.stringValue();
    if (!reader.isOk())
      Util.die(Util.FILE_FORMAT, errorMessage, reader.getFileName(),
	       reader.getLineNumber());
    return str;
  }
  
  /**
   * Assume that divisor is positive, but divident may be negative
   */
  public static final int remainder(int divident, int divisor)
  {
    if (divident < 0)
      divident += divisor * ((-divident + divisor - 1) / divisor);
    return divident % divisor;
  }
  
  public static void dijkstra(Vector nodes, boolean addNewNodes,
			      Vector unreachableNodes)
  {
    Vector outEdges;
    int i;
    PriorityQueue pq;
    ShortestPathNode node, node2;
    EdgeBase edge;
    
    pq = new PriorityQueue(nodes);
    node = null;
    while (pq.getSize() > 0)
    {
      node = (ShortestPathNode)pq.extract();
      if (node.getDistance() < 0)
	break;
      outEdges = node.getOutEdges();
      for (i = 0; i < outEdges.size(); i++)
      {
	edge = (EdgeBase)outEdges.get(i);
	node2 = (ShortestPathNode)edge.getDstNode();
	if (!pq.contains(node2) && !addNewNodes)
	  continue;
	if (node.getDistance() >= 0 &&
	    node2.compareTo(node.getDistance() + 1) > 0)
	{
	  node2.setDistance(node.getDistance() + 1);
	  node2.setParent(edge);
	  if (pq.contains(node2))
	    pq.decrease(node2);
	  else
	    pq.add(node2);
	}
      }
      node = null;
    }

    if (unreachableNodes != null)
    {
      pq.getAll(unreachableNodes);
      if (node != null)
	unreachableNodes.add(node);
    }
  }
  
  public static void printVersion()
  {
    stdout.println(version);
  }

  static final String version =
    "MOPS version XXX: MOdelchecking Programs for Security violations";

  static
  {
    int i;

    // Must set up stdout and stderr first
    stdout = new PrintWriter(System.out, true);
    stderr = new PrintWriter(System.err, true);
    sw = new StringWriter();
    pw = new PrintWriter(sw);
    setVerbosity(WARNING);
    programName = "java " + getProgramName();
  }
}
