// $Id: CfgMerge.java,v 1.15 2003/09/22 21:31:56 hchen Exp $

import java.io.*;
import java.util.*;
import gnu.getopt.*;

/**
 * End user application to merge several CFGs
 */
public class CfgMerge
{
  static final String optionString = "a:l:L:o:t:r" + Util.getOptionString();
  
  static final LongOpt[] longOpts = new LongOpt[] {
      new LongOpt("library-path", LongOpt.REQUIRED_ARGUMENT, null, 'L')};

  static final String optionUsage =
    "Usage: CfgMerge [options] file ...\n" +
    "Options:\n" +
    "  -a <file>\tUse <file> as the archiving program (ar)\n" +
    "  -l <file>\tLink the CFGs stored in the archive file lib<file>.a\n" +
    "  -L <dir>\tAdd <dir> to the list of directories to be searched for -l\n" +
    "  --library-path=<dir>\tSame as -L <dir>\n" +
    "  -o <file>\tPlace the output into <file>\n" +
    "  -t <dir>\tUse <dir> as the temporary directory for storing files extracted from an archive\n" +
    "  -r\t\tIgnore wrong command line arguments (do not abort)\n" +
    Util.getOptionUsage();

  public static void main(String args[]) throws IOException, InterruptedException
  {
    new CfgMerge().merge(args);
  }

  public void merge(String args[]) throws IOException, InterruptedException
  {
    Cfg cfg;
    String outputFileName = "-", name;
    File file;
    Getopt opt;
    int i, ch;
    boolean isStrict;
    ByteArrayOutputStream baos;
    BinaryOutput writer;
    Vector libFileNames, libSearchDirs;
    Vector cfgs;
    IntegerVar maxAddress;
    LongOpt[] combinedLongOpts;
    

    // create magic string for .a files
    archiveMagicBytes = "!<arch>".getBytes();
    // create magic string for CFG files
    baos = new ByteArrayOutputStream();
    writer = new BinaryOutput(new BufferedOutputStream(baos));
    writer.writeString(Constants.CFG_MAGIC_STRING);
    writer.close();
    cfgMagicBytes = baos.toByteArray();
    magicBytesLength =
      StrictMath.max(archiveMagicBytes.length, cfgMagicBytes.length);

    // parse command line arguments
    isStrict = true;
    tempDirName = null;
    arFileName = null;
    libFileNames = new Vector();
    libSearchDirs = new Vector();
    combinedLongOpts = new LongOpt[Util.getLongOpts().length + longOpts.length];
    for (i = 0; i < longOpts.length; i++)
      combinedLongOpts[i] = longOpts[i];
    for (i = 0; i < Util.getLongOpts().length; i++)
      combinedLongOpts[i + longOpts.length] = Util.getLongOpts()[i];
    
    opt = new Getopt("CfgMerge", args, optionString, combinedLongOpts);
    while ((ch = opt.getopt()) != -1)
    {
      switch(ch)
      {
	case 'r':
	  isStrict = false;
	  break;
	  
	case 'l':
	  libFileNames.add(opt.getOptarg());
	  break;

	case 't':
	  tempDirName = opt.getOptarg();
	  break;

	case 'a':
	  arFileName = opt.getOptarg();
	  break;
	  
	case 'L':
	  libSearchDirs.add(opt.getOptarg());
	  break;

	case 'o':
	  outputFileName = opt.getOptarg();
	  break;

	default:
	  Util.processOption(ch, opt, optionUsage, false);
      }
    }
    /*
    for (i = 0; i < Util.isPrintWarnings.length; i++)
      Util.stdout.println(Util.warningLabels[i] + ": " + Util.isPrintWarnings[i]);
    */
    if (arFileName == null)
    {
      arFileName = "ar";
    }
    if (tempDirName == null)
    {
      tempDirName = "MOPSTEMP";
    }
    /*
    if (outputFileName == null)
    {
      Util.stderr.println(optionUsage);
      System.exit(1);
    }
    */

    cfgs = new Vector();
    for (i = opt.getOptind(); i < args.length; i++)
    {
      file = new File(args[i]);
      switch(getFileType(file))
      {
	case CFG:
	  readCfg(file, cfgs);
	  break;

	case ARCHIVE:
	  readArchive(file, cfgs);
	  break;

	default:
	  Util.warn(Util.WARNING, Util.FILE_FORMAT, "Ignore file " + file +
			     ": neither a CFG nor an archive file");
      }
    }

    for (i = 0; i < libFileNames.size(); i++)
    {
      name = (String)libFileNames.get(i);
      file = resolveLibFile(name, libSearchDirs);
      if (file == null)
      {
	Util.warn(Util.WARNING, Util.FILE_IO, "Cannot locate library " + name);
      }
      else
      {
	if (getFileType(file) == ARCHIVE)
	{
	  //Util.stdout.println("enter readArchive " + file);
	  readArchive(file, cfgs);
	  //Util.stdout.println("exit readArchive");
	}
	else
	{
	  Util.warn(Util.WARNING, Util.FILE_FORMAT,
		    "Ignore file " + file + ": not an archive file");
	}
      }
    }

    if (cfgs.size() == 0)
    {
      Util.stderr.println(optionUsage);
      System.exit(1);
    }
    cfg = Cfg.merge(cfgs);
    maxAddress = new IntegerVar();
    maxAddress.data = 0;
    cfg.write(outputFileName, true, maxAddress);
  }

  File resolveLibFile(String libName, Vector libpath)
  {
    String libFileName;
    File file;
    int i;
    
    libFileName = "lib" + libName + ".a";
    for (i = 0; i < libpath.size(); i++)
    {
      file = new File((String)libpath.get(i), libFileName);
      if (file.canRead())
	return file;
    }
    return null;
  }
  
  boolean startsWith(byte[] array, int count, byte[] prefix)
  {
    int i;

    if (count < prefix.length)
      return false;
    
    for (i = 0; i < prefix.length; i++)
      if (array[i] != prefix[i])
	return false;

    return true;
  }
  
  int getFileType(File file) throws IOException
  {
    BufferedInputStream bis;
    byte[] buffer;
    int count;

    bis = new BufferedInputStream(new FileInputStream(file));
    buffer = new byte[magicBytesLength];
    count = bis.read(buffer, 0, buffer.length);
    bis.close();
    if (startsWith(buffer, count, cfgMagicBytes))
      return CFG;
    else if (startsWith(buffer, count, archiveMagicBytes))
      return ARCHIVE;
    else
      return UNKNOWN;
  }

  void readCfg(File file, Vector cfgs) throws IOException
  {
    Cfg cfg;

    // Util.stderr.println("Reading CFG " + file);
    cfg = new Cfg();
    cfg.read(file.getAbsolutePath(), true);
    cfgs.add(cfg);
  }

  void readArchive(File libFile, Vector cfgs)
    throws IOException, InterruptedException
  {
    Process process;
    String[] cmdArray;
    File tempDir;
    BufferedReader reader = null;
    String str, tempFilename;
    File objFile, tempFile;
    Timer timer;
    boolean interrupted;
    long length, oldLength;

    // Warning: platform-specific trick
    if (libFile.getAbsolutePath().indexOf("/usr/lib") == 0)
    {
      Util.warn(Util.WARNING, Util.FILE_IO,
		"Ignore system-wide library " + libFile);
      return;
    }
    // Util.stderr.println("Reading archive " + libFile);
    tempDir = new File(tempDirName);
    tempDir.deleteOnExit();
    if (tempDir.exists())
    {
      if (tempDir.isDirectory())
      {
	Util.warn(Util.WARNING, Util.FILE_IO, "The temporary directory " + tempDir +
			   " already exists");
      }
      else
      {
	Util.die(Util.FILE_IO, tempDir + " exists but is not a directory");
      }
    }
    else if (!tempDir.mkdirs())
    {
      Util.die(Util.FILE_IO, "Cannot create the temporary directory " + tempDir);
    }
    tempFilename = "ARCHIVELIST.TMP";
    tempFile = new File(tempDir, tempFilename);
    tempFile.delete();
    cmdArray = new String[]
      { "/bin/sh", "-c",
	arFileName + " xv " + libFile.getAbsolutePath() + " >" + tempFilename
        + " 2>/dev/null </dev/null"};
    //Util.stderr.println(arFileName + " xv " + libFile.getAbsolutePath() + " > " + tempFilename);
    process = Runtime.getRuntime().exec(cmdArray, null, tempDir);
    process.getInputStream().close();
    process.getOutputStream().close();
    process.getErrorStream().close();
    // This is an alternative hack to circumvent java's broken
    // Process.waitFor()
    process.waitFor();
    /*
    Util.stdout.print("Press return to continue");
    new BufferedReader(new InputStreamReader(System.in)).readLine();
    Thread.currentThread().sleep(1000);
    for (oldLength = 0;
	 (length = tempFile.length()) > oldLength;
	 Thread.currentThread().sleep(1000))
    {
      oldLength = length;
    }
    */
    //Util.stderr.println(tempFile + ": " + length);
    reader = new BufferedReader(new FileReader(tempFile));
    while ((str = reader.readLine()) != null)
    {
      //      objFileName = MetaFsa.getAbsoluteFileName(
      //tempDirName, str.substring(str.lastIndexOf(' ') + 1));
      // Util.stderr.println(str);
      objFile = new
	File(tempDirName, str.substring(str.lastIndexOf(' ') + 1));
      if (getFileType(objFile) != CFG)
      {
	Util.warn(Util.WARNING, Util.FILE_FORMAT, objFile + " is not a CFG file");
      }
      else
      {
	readCfg(objFile, cfgs);
      }
      if (!objFile.delete())
      {
	Util.warn(Util.WARNING, Util.FILE_IO, "Cannot delete " + objFile);
      }
    }
    reader.close();
    if (!tempFile.delete())
    {
      Util.warn(Util.WARNING, Util.FILE_IO, "Cannot delete the temporary file " + tempFilename);
    }
    if (!tempDir.delete())
    {
      Util.warn(Util.WARNING, Util.FILE_IO, "Cannot delete the temporary directory " + tempDir);
    }
    //Util.stdout.print("Press return to continue");
    //new BufferedReader(new InputStreamReader(System.in)).readLine();
  }

  /*
  boolean deleteAll(File dir) throws IOException
  {
    File[] files;
    int i;
    boolean ok;

    ok = true;
    files = dir.listFiles();
    if (files != null)
    {
      for (i = 0; i < files.length; i++)
      {
	if (!files[i].delete())
	{
	  ok = false;
	  Util.stderr.println("Cannot delete " + files[i]);
	}
	else
	{
	  Util.stderr.println("Deleted " + files[i]);
	}
      }
    }

    return ok;
  }
  */
  
  byte[] cfgMagicBytes, archiveMagicBytes;
  int magicBytesLength;
  String tempDirName, arFileName;
  static final int CFG = 0, ARCHIVE = 1, UNKNOWN = 2;
}

  /*
  static class MyTimerTask extends TimerTask
  {
    Timer timer;
    Thread thread;

    public MyTimerTask(Timer timer, Thread thread)
    {
      this.timer = timer;
      this.thread = thread;
    }
    
    public void run()
    {
      timer.cancel();
      Thread[] threads = new Thread[100];

      Thread.enumerate(threads);
      for (int i = 0; i < threads.length; i++)
	if (threads[i] == null)
	  break;
        else
	{
	  Util.stderr.println(threads[i].getName());
	  if (threads[i].getName().equals("process reaper"))
	  {
	    threads[i].interrupt();
	    Util.stderr.println("interrupting");
	  }
	}
      thread.interrupt();
    }
  };
  */
  
    /*
    timer = new Timer();
    timer.schedule(new MyTimerTask(timer, Thread.currentThread()), 10000);
    interrupted = false;
    try
    {
      if (process.waitFor() != 0)
      {
	Util.warn(Util.WARNING, Util.FILE_IO, arFileName + " exits abnormally");
      }
    }
    catch(InterruptedException e)
    {
      Util.warn(Util.WARNING, Util.FILE_IO, arFileName + " xv " + libFile.getAbsolutePath()
		+ " runs too long and was interrupted");
      interrupted = true;
    }
    Util.stderr.println("here");
    timer.cancel();
    if (interrupted)
    {
      if (!tempDir.delete())
      {
	Util.warn(Util.WARNING, Util.FILE_IO, "Cannot delete the temporary directory " + tempDir);
      }
      return;
    }
    reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
    */
