import java.io.*;
import gnu.getopt.*;
import java.util.*;

/**
 * User application to transform the output from check of a compacted
 * CFG into an uncompacted CFG
 */
public class Transform
{
  static final String optionString = "c:i:o:p:" + Util.getOptionString(),
    optionUsage =
    "Usage: Transform [options] -c CFG_file\n" +
    "  -c <file>\tRead CFG from <file>\n" +
    "  -i <file>\tRead the input from <file>\n" +
    "  -o <file>\tWrite the output to <file>\n" +
    "  -p <olddir:newdir>\tReplace directory prefix <olddir> with <newdir> in the output(trace)\n" +
    Util.getOptionUsage();
  
  public static void main(String[] args) throws IOException
  {
    Cfg cfg = new Cfg();
    String cfgFilename = null, inputFilename = "-", outputFilename = "-",
      str, filename, mfsaFileName, mfsaLabel;
    boolean isTrace, hasTrace;
    Getopt opt;
    int ch, counter, token, index;
    TextInput reader;
    PrintWriter writer, indexWriter;
    Vector dirPrefixes;

    dirPrefixes = new Vector();
    opt = new Getopt("Transform", args, optionString, Util.getLongOpts());
    while ((ch = opt.getopt()) != -1)
    {
      switch(ch)
      {
	case 'c':
	  cfgFilename = opt.getOptarg();
	  break;
	  
	case 'i':
	  inputFilename = opt.getOptarg();
	  break;
	  
	case 'o':
	  outputFilename = opt.getOptarg();
	  break;

	case 'p':
	  str = opt.getOptarg();
	  if ((index = str.indexOf(':')) < 0)
	  {
	    Util.die(Util.FILE_FORMAT,
		     "Missing ':' in the argument to the -p option");
	  }
	  dirPrefixes.add(new StringPair(str.substring(0, index),
					 str.substring(index + 1)));
	  break;
	  
	default:
	  Util.processOption(ch, opt, optionUsage);
      }
    }
    if (cfgFilename == null)
    {
      Util.stderr.println(optionUsage);
      System.exit(1);
    }
    reader = new TextInput(new BufferedReader(Util.openReader(inputFilename)), inputFilename);
    reader.nextToken();
    str = reader.stringValue();
    if (str == null)
    {
      Util.stderr.println(inputFilename + " is not a MOPS trace or program point file.");
      System.exit(0);
      // not reachable, but make javac happy
      isTrace = true;
    }
    if (str.equals(Check.RAW_TRACE_MAGIC_STRING))
    {
      isTrace = true;
    }
    else if (str.equals(Check.RAW_PROGRAM_POINT_MAGIC_STRING))
    {
      isTrace = false;
    }
    else
    {
      Util.stderr.println(inputFilename + " is not a MOPS trace or program point file.");
      System.exit(0);
      // not reachable, but make javac happy
      isTrace = true;
    }
    if (reader.skipToEOL() > 1)
    {
      Util.warn(Util.WARNING, Util.FILE_FORMAT, "Ignore extra words at the end of line",
		inputFilename, reader.getLineNumber());
    }
    mfsaFileName = Util.readString(reader, "Expect an MFSA file name");
    reader.skipToEOL();
    mfsaLabel = Util.readString(reader, "Expect an MFSA file label");
    reader.skipToEOL();
    cfg.read(cfgFilename, true);
    if (isTrace)
    {
      indexWriter = new PrintWriter(new BufferedWriter(
				    Util.openWriter(outputFilename)));
      indexWriter.println("mfsa: " + mfsaFileName);
      indexWriter.println("mfsalabel: " + mfsaLabel);
      indexWriter.println("cfg: " + cfgFilename);
      hasTrace = false;
      reader.nextToken();
      for (counter = 1, str = reader.stringValue();
	   reader.isOk();
	   counter++, str = reader.stringValue())
      {
	if (!reader.isOk() || !str.equals(Check.RAW_TRACE_SEPARATOR))
	{
	  break;
	}
	reader.skipToEOL();
	if (!hasTrace)
	{
	  hasTrace = true;
	  Util.stdout.print("Writing to ");
	}
	filename = outputFilename + "." + counter;
	Util.stdout.print(filename);
	Util.stdout.print(' ');
	Util.stdout.flush();
	indexWriter.println(filename);
	writer = new PrintWriter(new BufferedWriter(Util.openWriter(filename)));
	cfg.transformPath(reader, writer, dirPrefixes);
	writer.close();
      }
      indexWriter.close();
      if (hasTrace)
      {
	Util.stdout.println();
      }
      token = reader.tokenValue();
      if (!reader.isOk() || token != Constants.EOF)
      {
	Util.stderr.println("Unexpected end of file " + inputFilename);
	System.exit(1);
      }
    }
    else
    {
      writer = new PrintWriter(new BufferedWriter(Util.openWriter(outputFilename)));
      // input is a list of nodes (output from Fsa.writeInitialTransitions())
      cfg.transformNodes(reader, writer);
      writer.close();
    }
    reader.close();
  }
}
