#include "cell.h"

Cell::Cell(Eve_parameters* _global, int _ID, double _fitness, int random_flag)
{

  // copy a pointer to the static instance of the gloabal variables
  global=_global;

  double _sparcity = 0.21 - 0.2*my_rand(global->param.globals.idum, global->param.globals.itemp);
  
  sd.n_nodes = 6+ 3*MAX(0,ROUND((MAX_NODES/3 -2)*my_rand(global->param.globals.idum, global->param.globals.itemp)));
  setn_nodes(sd.n_nodes);

  setID(_ID);
  setfitness(_fitness);
  setenergy(0,1); //energy reset 
  setenergy(global->param.energy.initial_energy, 0); //energy initialization to initial_energy
  setsparcity(_sparcity); //initial sparcity 
  setmutation(0,0,0,0,0); //mutation reset readme-acquired values will be set in the eve21.cpp code
  setinput(0, -3, global->param.general.n_signals, global->param.network.initial_sensor_prob); 

  setv(0,0,random_flag,0);
  setmiddle(0,0,random_flag,0);

  setslope(0,0,random_flag,0);
  setdeg(0,0,random_flag,0);
  setbasal(0,0, random_flag,0);
	
  setw(0,0,0,random_flag,0);
  setregslope(0,0,0,random_flag,0);
  setregmiddle(0,0,0,random_flag,0);

  setmutation_counter(-1,0,0);
  

#if _IF_MPI
  // create an MPI type for all static variables:
  // define MPI structure with all static data:
  int s_size=12; // number of blocks in the structure
  MPI_Aint _addresses[s_size]; // address of the begining of each block (of one element)
  MPI_Aint _offsets[s_size]; // offset from the begining (first block) -- will be calculated
  int _block_len[s_size]; // length of each block (=1)
  MPI_Datatype _old_types[s_size]; // types of elements in each block (only one element per block)
  
  MPI_Address(&sd.n_nodes,      &_addresses[0]);
  _old_types[0] = MPI_INT;
  
  MPI_Address(&sd.ID,      &_addresses[1]);
  _old_types[1] = MPI_INT;
  
  MPI_Address(&sd.fitness, &_addresses[2]);
  _old_types[2] = MPI_DOUBLE;
  
  MPI_Address(&sd.energy,  &_addresses[3]);
  _old_types[3] = MPI_DOUBLE;
  
  MPI_Address(&sd.sparcity,  &_addresses[4]);
  _old_types[4] = MPI_DOUBLE;
  
  MPI_Address(&sd.creation_cell,  &_addresses[5]);
  _old_types[5] = MPI_DOUBLE;
  
  MPI_Address(&sd.destruction_cell,  &_addresses[6]);
  _old_types[6] = MPI_DOUBLE;
  
  MPI_Address(&sd.mild_mutation_cell,  &_addresses[7]);
  _old_types[7] = MPI_DOUBLE;
  
  MPI_Address(&sd.strong_mutation_cell,  &_addresses[8]);
  _old_types[8] = MPI_DOUBLE;
  
  MPI_Address(&sd.evolvability_cell,  &_addresses[9]);
  _old_types[9] = MPI_DOUBLE;
	
  MPI_Address(&sd.coord,  &_addresses[10]);
  _old_types[10] = MPI_INT;
  
  MPI_Address(&sd.label,  &_addresses[11]);
  _old_types[11] = MPI_INT;
  
  // all blocks have length 1 (except few reassigned below)  
  for ( int i=0; i<s_size; i++)
    _block_len[i] = 1;
  // calculate offsets from the beginig
  for ( int i=0; i<s_size; i++)
    _offsets[i] = _addresses[i]-_addresses[0];
  // create cell_data_tMPI
  MPI_Type_struct( s_size, _block_len, _offsets, _old_types, &cell_static_data_tMPI );
  MPI_Type_commit( &cell_static_data_tMPI );
#endif
}

Cell::~Cell()
{
  int i;
  delete[] v;
  delete[] slope;
  delete[] middlepoint;
  delete[] basal;
  delete[] deg;
  delete[] input_sensor_protein;
  
  delete[] mild_mutations;
  delete[] strong_mutations;
	
  for (i=0;i<sd.n_nodes;i++)
    {
      delete[] w[i];
      delete[] regslope[i];
      delete[] regmiddle[i];
    }
  delete[] w;
  delete[] regslope;
  delete[] regmiddle;

 
}


void Cell::free_MPI_types() {
#if _IF_MPI
  //remove MPI data type
  MPI_Type_free( &cell_static_data_tMPI );
#endif
}


int Cell::getn_nodes() const { return sd.n_nodes; }

int Cell::getn_links() const { 
  int n_links=0;
  for (int i=0; i<sd.n_nodes; i++) {
    for (int j=0; j<sd.n_nodes; j++) {
      if ( w[i][j] != 0 )
	n_links++;
    }
  }
  return n_links; 
}

int Cell::getID() const { return sd.ID; }

double Cell::getfitness() const { return sd.fitness; }

double Cell::getenergy() const { return sd.energy; }

double Cell::getsparcity() const { return sd.sparcity; }

double Cell::getmutation(int selector) const
{
	if (selector ==1)
		return sd.creation_cell;
	else if (selector ==2)
		return sd.destruction_cell;
	else if (selector ==3)
		return sd.mild_mutation_cell;
	else if (selector ==4)
		return sd.strong_mutation_cell;
	else if (selector ==5)
		return sd.evolvability_cell;
	else 
		return 0;
}





int Cell::getinput(int i) const { return input_sensor_protein[i];}

int Cell::getcoord() const { return sd.coord; };

int Cell::getlabel() const { return sd.label; };

double Cell::getconnectprob(int input) const
{
	int i,temp=0;
	for (i=0;i<sd.n_nodes;i++)
	{
		if (input_sensor_protein[i] == input)
			temp = temp +1;
	}
	return exp(-(double)temp);
}

int Cell::getv(int position) const
{

	return v[position];
}

double Cell::getmiddle(int position) const
{
	return middlepoint[position];
}


double Cell::getslope(int position) const
{
	return slope[position];
}


double Cell::getdeg(int position) const
{
	return deg[position];
}

double Cell::getbasal(int position) const
{
	return basal[position];
}



double Cell::getw(const int i, const int j) const
{
	return w[i][j];
}

double Cell::getregmiddle(const int i, const int j) const
{
	return regmiddle[i][j];
}

double Cell::getregslope(const int i, const int j) const
{
	return regslope[i][j];
}

int Cell::getv_history(const int i, const int j) const
{
	return v_history[i][j];
}

int** Cell::getv_history_pointer() const
{
	return v_history;
}

int Cell::getmutation_counter(int mutation_type, int nodeID) const
{
	if (mutation_type ==1)
		return mild_mutations[nodeID];
	else if (mutation_type ==2)
		return strong_mutations[nodeID];
	else
		{
			cout << "Problem in getmutation_counter";
			return 0;
		}
}



void Cell::copy_from_another_cell(const Cell *other)
{ 
  int old_n_nodes = getn_nodes();

  setID(other->getID());
  setcoord(other->getcoord());
  setlabel(other->getlabel());

  setenergy(0,1);
  setenergy(other->getenergy(),0);
  setfitness(other->getfitness());
	
  setn_nodes(other->getn_nodes());
  setsparcity(other->getsparcity());
  setmutation(other->getmutation(1), 
	      other->getmutation(2), 
	      other->getmutation(3), 
	      other->getmutation(4), 
	      other->getmutation(5));
  
  setinput(0,-4,0,0);
  setv(0,0,0,4);
  setmiddle(0,0,0,4);
  setslope(0,0,0,4);
  setdeg(0,0,0,4);
  setbasal(0,0,0,4);
  setw(0,old_n_nodes,0,0,4);
  setregslope(0,old_n_nodes,0,0,4);
  setregmiddle(0,old_n_nodes,0,0,4);
  for (int temp1=0;temp1<other->getn_nodes();temp1++){
    setinput(temp1,other->getinput(temp1),0,0);
    setv(temp1,other->getv(temp1),0,5);
    setmiddle(temp1,other->getmiddle(temp1),0,5);
    setslope(temp1,other->getslope(temp1),0,5);
    setdeg(temp1,other->getdeg(temp1),0,5);
    setbasal(temp1,other->getbasal(temp1),0,5);
    for (int temp2=0;temp2<other->getn_nodes();temp2++){
      setw(other->getw(temp1,temp2),temp1 ,temp2,0,5);
      setregslope(other->getregslope(temp1,temp2),temp1 ,temp2,0,5);
      setregmiddle(other->getregmiddle(temp1,temp2),temp1 ,temp2,0,5);
    }
  }
  // Addition in mutation tracer. Careful, the mutation_counter is zeroed.
  setmutation_counter(-2,0,0); // delete mild_ and strong_ mutation arrays
  setmutation_counter(-1,0,0); // alocate arrays
}




void Cell::MPI_Send_cell_copy(int target)
{
#if _IF_MPI
  int tag = 239;
  MPI_Status status;

  // send static data 
  MPI_Send( &sd, 1, cell_static_data_tMPI, target, tag, MPI_COMM_WORLD );
 
  // send int*[n_nodes]
  MPI_Send( v,                    sd.n_nodes, MPI_INT, target, tag, MPI_COMM_WORLD );
  MPI_Send( input_sensor_protein, sd.n_nodes, MPI_INT, target, tag, MPI_COMM_WORLD );

  // send double*[n_nodes]
  MPI_Send( slope,       sd.n_nodes, MPI_DOUBLE, target, tag, MPI_COMM_WORLD );
  MPI_Send( middlepoint, sd.n_nodes, MPI_DOUBLE, target, tag, MPI_COMM_WORLD );
  MPI_Send( basal,       sd.n_nodes, MPI_DOUBLE, target, tag, MPI_COMM_WORLD );
  MPI_Send( deg,         sd.n_nodes, MPI_DOUBLE, target, tag, MPI_COMM_WORLD );

  // send double**[n_nodes][n_nodes]
  for (int i=0; i<sd.n_nodes; i++)
    {
      MPI_Send( &w[i][0],         sd.n_nodes, MPI_DOUBLE, target, tag, MPI_COMM_WORLD );
      MPI_Send( &regslope[i][0],  sd.n_nodes, MPI_DOUBLE, target, tag, MPI_COMM_WORLD );
      MPI_Send( &regmiddle[i][0], sd.n_nodes, MPI_DOUBLE, target, tag, MPI_COMM_WORLD );
    }
#endif
}



void Cell::MPI_Recv_cell_copy(int source)
{
#if _IF_MPI
  int old_n_nodes = getn_nodes();
  int tag = 239;
  MPI_Status status;

  // recv static data
  MPI_Recv( &sd, 1, cell_static_data_tMPI, source, tag, MPI_COMM_WORLD, &status  );

  setinput(0,-4,0,0);
  setv(0,0,0,4);
  setmiddle(0,0,0,4);
  setslope(0,0,0,4);
  setdeg(0,0,0,4);
  setbasal(0,0,0,4);
  setw(0,old_n_nodes,0,0,4);
  setregslope(0,old_n_nodes,0,0,4);
  setregmiddle(0,old_n_nodes,0,0,4);

  // recv int*[n_nodes]
  MPI_Recv( v,                    sd.n_nodes, MPI_INT, source, tag, MPI_COMM_WORLD, &status );
  MPI_Recv( input_sensor_protein, sd.n_nodes, MPI_INT, source, tag, MPI_COMM_WORLD, &status );
 
  // recv double*[n_nodes]
  MPI_Recv( slope,       sd.n_nodes, MPI_DOUBLE, source, tag, MPI_COMM_WORLD, &status );
  MPI_Recv( middlepoint, sd.n_nodes, MPI_DOUBLE, source, tag, MPI_COMM_WORLD, &status );
  MPI_Recv( basal,       sd.n_nodes, MPI_DOUBLE, source, tag, MPI_COMM_WORLD, &status );
  MPI_Recv( deg,         sd.n_nodes, MPI_DOUBLE, source, tag, MPI_COMM_WORLD, &status );

  // recv double**[n_nodes][n_nodes]
  for (int i=0; i<sd.n_nodes; i++)
    {
      MPI_Recv( &w[i][0],        sd.n_nodes, MPI_DOUBLE, source, tag, MPI_COMM_WORLD, &status );
      MPI_Recv( &regslope[i][0],  sd.n_nodes, MPI_DOUBLE, source, tag, MPI_COMM_WORLD, &status );
      MPI_Recv( &regmiddle[i][0], sd.n_nodes, MPI_DOUBLE, source, tag, MPI_COMM_WORLD, &status );
    }
  
  // Addition in mutation tracer. Careful, the mutation_counter is zeroed.
  setv_history(0,1,selection_period,0);
  setmutation_counter(-2,0,0); // delete mild_ and strong_ mutation arrays
  setmutation_counter(-1,0,0); // alocate arrays
#endif
}



void Cell::setn_nodes(int _n_nodes) { sd.n_nodes =_n_nodes; } 

void Cell::setID(int _ID) { sd.ID = _ID; }

void Cell::setfitness(double _fitness) {sd.fitness = _fitness;}



void Cell::setenergy(double add_energy, int if_reset) 
{
  if (if_reset ==1)
    sd.energy = 0;
  else
    sd.energy += add_energy;
}



void Cell::setsparcity(double _sparcity) {sd.sparcity = _sparcity;}

void Cell::setmutation(double _creation_cell, double _destruction_cell, double _mild_mutation_cell, double _strong_mutation_cell, double _evolvability_cell)
{
	sd.creation_cell = _creation_cell;
	sd.destruction_cell = _destruction_cell;
	sd.mild_mutation_cell = _mild_mutation_cell;
	sd.strong_mutation_cell = _strong_mutation_cell;
	sd.evolvability_cell = _evolvability_cell;
}

void Cell::setcoord(int _coord) {sd.coord=_coord;};

void Cell::setlabel(int _label) {sd.label=_label;};

void Cell::setinput(int which_node, int which_input, int n_signals, double global_probability) 
{ 
	
	int i;
	if (which_input == -3) //initialization
	{
		if (which_node!=-1)
			input_sensor_protein = new int [sd.n_nodes]; //this will not be executed when re-initializing
		for (i=0;i<sd.n_nodes;i++)
		{
			if (my_rand(global->param.globals.idum, global->param.globals.itemp) < global_probability && n_signals!=0)
			{
				double *temp = new double [n_signals];
				int j;
				double total=0;
				for (j=0; j<n_signals; j++)
				{
					temp[j] = this->getconnectprob(j+1);
					total = total +temp[j];
				}
				temp[0] = temp[0]/total;
				for (j=1; j<n_signals; j++)
					temp[j] = temp[j]/total + temp[j-1];
				
				total = my_rand(global->param.globals.idum, global->param.globals.itemp);
				for (j=0; j<n_signals; j++)
				{
					if (temp[j] > total)
					{
						input_sensor_protein[i] =  j+1;
						break;
					}
				}
				delete[] temp;
			}
			else
				input_sensor_protein[i] = 0;
			
		}
		
	}

	else if (which_input == -2) //input deletion
	{
		int *temp;
		temp = new int [sd.n_nodes];
		
		for (i=0;i<which_node;i++)
			temp[i] = input_sensor_protein[i];
		for (i=(which_node);i<(sd.n_nodes);i++)
			temp[i] = input_sensor_protein[i+3];
		delete[] input_sensor_protein;
		input_sensor_protein = new int [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			input_sensor_protein[i] = temp[i];
		delete[] temp;
	}
	else if (which_input == -1) //addition
	{
		int *temp;
		temp = new int [sd.n_nodes];

		for (i=0;i<(sd.n_nodes-3);i++)
			temp[i] = input_sensor_protein[i];
		temp[sd.n_nodes -3] = input_sensor_protein[which_node];
		temp[sd.n_nodes -2] = input_sensor_protein[which_node+1];
		temp[sd.n_nodes -1] = input_sensor_protein[which_node+2];

		delete[] input_sensor_protein;
		input_sensor_protein = new int [sd.n_nodes];
		for (i=0;i<(sd.n_nodes);i++)
			input_sensor_protein[i] = temp[i];
		delete[] temp;
	}
	else if (which_input ==-4) //reinitialization to 0.
	{
		delete[] input_sensor_protein;
		input_sensor_protein = new int [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			input_sensor_protein[i] = 0;
	}
	else
	{
		input_sensor_protein[which_node] = which_input;
	}
}
void Cell::setv(int position, int a, int random_flag, int change)
{
  int i;

  if (random_flag ==1)
    {
      v = new int [sd.n_nodes];
      for (int random_index=0;random_index <sd.n_nodes;random_index++)
	{
	  v[random_index] = ROUND(my_rand(global->param.globals.idum, global->param.globals.itemp)*30); //30 is an arbitrary upper bound	
	}
    }


  // vm: here is a new number_of_nodes
  else if (change == 0) 
    v[position] = a;	
  
  // adds a triplet (three nodes) to the end of v; values set to 0;
  else if (change == 1) 
    {
      int *v_temp;
      v_temp = new int [sd.n_nodes];
      for (i=0;i<sd.n_nodes -3;i++)
	v_temp[i] = v[i];
      v_temp[sd.n_nodes -3] = 0;  //v_temp[position];
      v_temp[sd.n_nodes -2] = 0;  //v_temp[position+1];
      v_temp[sd.n_nodes -1] = 0;  //v_temp[position+2];
      delete[] v;
      
      v = new int [sd.n_nodes];
      for (i=0;i<sd.n_nodes;i++)
	v[i] = v_temp[i];
      delete[] v_temp;
      
    }

  // delete a triplet (three nodes) from v starting "position"
  else if (change ==-1) {
    int *v_temp;
    v_temp = new int [sd.n_nodes];
    for (i=0;i<position;i++)
      v_temp[i] = v[i];
    for (i=position+3;i<sd.n_nodes+3;i++)
      v_temp[i-3] = v[i];
    delete[] v;
    v = new int [sd.n_nodes];
    for (i=0;i<sd.n_nodes;i++)
      v[i] = v_temp[i];
    delete[] v_temp;
  }

  // delete v, create new clean
  else if (change ==4) {
    delete[] v;
    v = new int [sd.n_nodes];
  }

  // REMOVE
  else if (change ==5) {
    v[position] = a;
  }
}

void Cell::setdeg(int position, double a, int random_flag, int change)
{
	double *deg_temp;
	double min_deg =0.05;
	int i;
	int random_index;
	if (random_flag ==1)
	{
		deg = new double [sd.n_nodes];
		for (random_index=0;random_index <sd.n_nodes;random_index++)
		{
			deg[random_index] = min_deg +(1-min_deg)*my_rand(global->param.globals.idum, global->param.globals.itemp);
		}
	}
	else if (change ==0)
		deg[position] = a;
	else if (change == 1)
	{
		deg_temp = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes -3;i++)
			deg_temp[i] = deg[i];
		deg_temp[sd.n_nodes -3] = deg_temp[position];
		deg_temp[sd.n_nodes -2] = deg_temp[position+1];
		deg_temp[sd.n_nodes -1] = deg_temp[position+2];
		delete[] deg;

		deg = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			deg[i] = deg_temp[i];
		delete[] deg_temp;
	}
	else if (change ==-1)
	{
		deg_temp = new double [sd.n_nodes];
		for (i=0;i<position;i++)
			deg_temp[i] = deg[i];
		for (i=position+3;i<sd.n_nodes+3;i++)
			deg_temp[i-3] = deg[i];
		delete[] deg;

		deg = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			deg[i] = deg_temp[i];
		delete[] deg_temp;
	}
	else if (change == 2) 
	{
	  deg[position] = deg[position]*(1 + +0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp)); //amount of change is 0.2*N(0,1)*previous_value
		deg[position] = MIN(1,MAX(deg[position],min_deg));
	}
	else if (change == 3)
	{
		deg[position] = min_deg +(1-min_deg)*my_rand(global->param.globals.idum, global->param.globals.itemp);
	}
	else if (change ==4)
	{
		delete[] deg;
		deg = new double [sd.n_nodes];
	}
		else if (change ==5)
	{
		deg[position] = a;
	}
}

void Cell::setbasal(int position, double a, int random_flag, int change)
{
	double *basal_temp;
	int i;
	int random_index;
	if (random_flag ==1)
	{
		basal =new double [sd.n_nodes];
		for (random_index=0;random_index <sd.n_nodes;random_index++)
		{
			basal[random_index] =0.5*my_rand(global->param.globals.idum, global->param.globals.itemp);  //change back to 0.5
		}
		
	}
	else if (change ==0)
		basal[position] = a;
	else if (change == 1)
	{
		basal_temp = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes -3;i++)
			basal_temp[i] = basal[i];
		basal_temp[sd.n_nodes -3] = basal_temp[position];
		basal_temp[sd.n_nodes -2] = basal_temp[position+1];
		basal_temp[sd.n_nodes -1] = basal_temp[position+2];
		delete[] basal;

		basal = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			basal[i] = basal_temp[i];
		delete[] basal_temp;
	}
	else if (change ==-1)
	{
		basal_temp = new double [sd.n_nodes];
		for (i=0;i<position;i++)
			basal_temp[i] = basal[i];
		for (i=position+3;i<sd.n_nodes+3;i++)
			basal_temp[i-3] = basal[i];
		delete[] basal;

		basal = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			basal[i] = basal_temp[i];
		delete[] basal_temp;
	}
	else if (change == 2) 
	{
		basal[position] = basal[position]*(1 + +0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
		basal[position] = MIN(1,MAX(basal[position],0.01));
	}
	else if (change == 3)
	{
		basal[position] =0.5*my_rand(global->param.globals.idum, global->param.globals.itemp);  //change back to 0.5
	}
	else if (change ==4)
	{
		delete[] basal;
		basal = new double [sd.n_nodes];
	}
		else if (change ==5)
	{
		basal[position] = a;
	}
}

void Cell::setmutation_counter(int mutation_type, int position, int value_to_add)
{
	int i;
	if (mutation_type == -2) //delete vectors
	{
		delete[] mild_mutations;
		delete[] strong_mutations;
	}
	else if (mutation_type == -1) //allocate vectors
	{
		mild_mutations = new int [sd.n_nodes];
		strong_mutations = new int [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
		{
			mild_mutations[i] = 0;
			strong_mutations[i] = 0;
		}
	}
	else if (mutation_type == 0) 
	{
		if (value_to_add ==-1) //node deletion
		{
			int *mild_temp = new int [sd.n_nodes]; //assume that the sd.n_nodes have already changed
			int *strong_temp = new int [sd.n_nodes];
			for (int i=0;i<position;i++)
			{
				mild_temp[i] = mild_mutations[i];
				strong_temp[i] = strong_mutations[i];
			}
			for (int i=position+3;i<(sd.n_nodes+3);i++)
			{
				mild_temp[i-3] = mild_mutations[i];
				strong_temp[i-3] = strong_mutations[i];
			}
			delete[] mild_mutations;
			delete[] strong_mutations;
			mild_mutations = new int [sd.n_nodes];
			strong_mutations = new int [sd.n_nodes];
			for (int i=0;i<sd.n_nodes;i++)
			{
				mild_mutations[i] = mild_temp[i];
				strong_mutations[i] = strong_temp[i]; 
			}
			delete[] mild_temp;
			delete[] strong_temp;
		}
		
		if (value_to_add == 1) //node addition
		{
			int *mild_temp = new int [sd.n_nodes]; //assume that the sd.n_nodes have already changed
			int *strong_temp = new int [sd.n_nodes];
			for (int i=0;i<sd.n_nodes-3;i++)
			{
				mild_temp[i] = mild_mutations[i];
				strong_temp[i] = strong_mutations[i];
			}
			delete[] mild_mutations;
			delete[] strong_mutations;
			mild_mutations = new int [sd.n_nodes];
			strong_mutations = new int [sd.n_nodes];
			for (int i=0;i<sd.n_nodes -3;i++)
			{
				mild_mutations[i] = mild_temp[i];
				strong_mutations[i] = strong_temp[i]; 
			}
			delete[] mild_temp;
			delete[] strong_temp;
			mild_mutations[sd.n_nodes-3] = 0;
			mild_mutations[sd.n_nodes-2] = 0;
			mild_mutations[sd.n_nodes-1] = 0;
			
			strong_mutations[sd.n_nodes-3] = 0;
			strong_mutations[sd.n_nodes-2] = 0;
			strong_mutations[sd.n_nodes-1] = 0;
						
		}
	}


	else if (mutation_type == 1)
	{
		mild_mutations[position] += value_to_add;
	}
	else if (mutation_type == 2)
	{
		strong_mutations[position] += value_to_add;
	}
}

void Cell::setmiddle(int position, double a, int random_flag, int change)
{
	int random_index;
	int i;
	double *middle_temp;
	if (random_flag ==1)
	{
		middlepoint=new double [sd.n_nodes];
		for (random_index=0;random_index <sd.n_nodes;random_index++)
		{
			middlepoint[random_index] = -1 + 2*my_rand(global->param.globals.idum, global->param.globals.itemp); 
		}
	}
	else if (change ==0)
		middlepoint[position] = a;
	else if (change == 1)
	{
		middle_temp = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes -3;i++)
			middle_temp[i] = middlepoint[i];
		middle_temp[sd.n_nodes -3] = middle_temp[position];
		middle_temp[sd.n_nodes -2] = middle_temp[position+1];
		middle_temp[sd.n_nodes -1] = middle_temp[position+2];
		delete[] middlepoint;

		middlepoint = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			middlepoint[i] = middle_temp[i];
		delete[] middle_temp;
	}
	else if (change ==-1)
	{
		middle_temp = new double [sd.n_nodes];
		for (i=0;i<position;i++)
			middle_temp[i] = middlepoint[i];
		for (i=position+3;i<sd.n_nodes+3;i++)
			middle_temp[i-3] = middlepoint[i];
		delete[] middlepoint;

		middlepoint = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			middlepoint[i] = middle_temp[i];
		delete[] middle_temp;
	}
	else if (change == 2) 
	{
		middlepoint[position] = middlepoint[position]*(1 + +0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
	}
	else if (change == 3)
	{
		middlepoint[position] = -1 + 2*my_rand(global->param.globals.idum, global->param.globals.itemp); 
	}
	else if (change ==4)
	{
		delete[] middlepoint;
		middlepoint = new double [sd.n_nodes];
	}
		else if (change ==5)
	{
		middlepoint[position] = a;
	}
		
}

void Cell::setslope(int position, double a, int random_flag, int change)
{
	int random_index;
	double *slope_temp;
	int i;
	if (random_flag ==1)
	{
		slope = new double [sd.n_nodes];
		for (random_index=0;random_index <sd.n_nodes;random_index++)
		{
			slope[random_index] = 1+ 3*my_rand(global->param.globals.idum, global->param.globals.itemp); 
		}
	}
	else if (change ==0)
		slope[position] = a;
	else if (change == 1)
	{
		slope_temp = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes -3;i++)
			slope_temp[i] = slope[i];
		slope_temp[sd.n_nodes -3] = slope_temp[position];
		slope_temp[sd.n_nodes -2] = slope_temp[position+1];
		slope_temp[sd.n_nodes -1] = slope_temp[position+2];
		delete[] slope;

		slope = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			slope[i] = slope_temp[i];
		delete[] slope_temp;
	}
	else if (change ==-1)
	{
		slope_temp = new double [sd.n_nodes];
		for (i=0;i<position;i++)
			slope_temp[i] = slope[i];
		for (i=position+3;i<sd.n_nodes+3;i++)
			slope_temp[i-3] = slope[i];
		delete[] slope;

		slope = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			slope[i] = slope_temp[i];
		delete[] slope_temp;
	}
	else if (change == 2) 
	{
		slope[position] = slope[position]*(1 + +0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
		slope[position] = MAX(slope[position],0.1);
	}
	else if (change == 3)
	{
		slope[position] = 1 + 3*my_rand(global->param.globals.idum, global->param.globals.itemp); 
	}
	else if (change ==4)
	{
		delete[] slope;
		slope = new double [sd.n_nodes];
	}
		else if (change ==5)
	{
		slope[position] = a;
	}
		
}


void Cell::setw(double value, int position_row, int position_col, int random_flag, int change)
{ 
	double **w_temp;
	double temp;
	int random_index, random_index2;
	if (random_flag ==1)
	{
		w = new double *[sd.n_nodes];
		for (random_index=0;random_index<sd.n_nodes;random_index++)
			w[random_index] = new double [sd.n_nodes];

		for (int i=0;i<sd.n_nodes;i++)
		{
			for (int j=0;j<sd.n_nodes;j++)
				w[i][j] = 0;
		}
		//cout << "sparcity: " << (sd.sparcity) << " " << ceil(sd.sparcity*sd.n_nodes*sd.n_nodes);
		for (int index=0; index<ceil(sd.sparcity*sd.n_nodes*sd.n_nodes); index++)
		{
			if (intraconnectivity_flag == 0)
			{
				//cout << "\nInitialization of Organism Population: No Intraconnectivity Matrix, equal connection probability.";
				random_index  = static_cast<int>(sd.n_nodes*my_rand(global->param.globals.idum, global->param.globals.itemp)); //early version when intraconnectivity prob. was uniform.
				random_index2 = static_cast<int>(sd.n_nodes*my_rand(global->param.globals.idum, global->param.globals.itemp));
				//Power law distribution - neg. and pos.
				w[random_index][random_index2] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.
			}
			else
			{
			  //When Connection Specificity Flag is up, then the probability of a node regulating nodes of multiple types is decreased
			  if (connection_specificity_flag ==1)
			    {
			      // select a source (regulator) triplet
			      random_index2 = 3*static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*sd.n_nodes/3); 

			      // add select source (regulator) node type [0, 1, or 2] using intraconnectivity_regulator probability distribution
			      temp = my_rand(global->param.globals.idum, global->param.globals.itemp); // target is selected for given source - regulator by intraconnectivity 
			      
			      if (temp <intraconnectivity_regulator[0])
				random_index2;
			      else if ( temp < (intraconnectivity_regulator[0] + intraconnectivity_regulator[1]) )
				random_index2 += 1; 
			      else 
				random_index2 += 2; 

			      double temp_intraconnectivity[3] = {intraconnectivity[0][random_index2%3],intraconnectivity[1][random_index2%3],intraconnectivity[2][random_index2%3]};
			      // normolize this row in the intraconnectivity matrix:
			      double sum = temp_intraconnectivity[0]+temp_intraconnectivity[1]+temp_intraconnectivity[2];

			      // only if this type of a node can actually regulate anything:
			      if (sum!=0)
				{
				  for (int i=0; i<3; i++)
				    temp_intraconnectivity[i] /= sum;

				  double counter[3] = {0};
				  sum = 0;
				  for (int i=0;i<sd.n_nodes;i++)
				    {
				      if (w[i][random_index2]!=0)
					counter[i%3] = counter[i%3] +1;
				    }
				  sum = counter[0]+counter[1]+counter[2];
			      
				  // rescale target/regulated probabilities (1 - relative frequency of connection type): if sum is 0 then no need for rescaling
				  if (sum!=0)
				    {
				      for (int i=0;i<3;i++)
					counter[i] = 1 - (counter[i]/sum);
				  
				      for (int i=0;i<3;i++)
					temp_intraconnectivity[i] = temp_intraconnectivity[i]*exp(-1*counter[i]);
				  
				      sum = temp_intraconnectivity[0] + temp_intraconnectivity[1]+ temp_intraconnectivity[2];
				      for (int i=0;i<3;i++)
					temp_intraconnectivity[i] = temp_intraconnectivity[i]/sum;
				    }
			      
				  // get a regulated node (target)
				  temp = my_rand(global->param.globals.idum, global->param.globals.itemp); 
				  if (temp <temp_intraconnectivity[0])
				    random_index = 0;
				  else if (temp < (temp_intraconnectivity[0] + temp_intraconnectivity[1]))
				    random_index =1; 
				  else 
				    random_index = 2; 
				  random_index  = 3*static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*sd.n_nodes/3) + random_index;
			      
				  //Power law distribution - neg. and pos.
				  w[random_index][random_index2] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.
				}
			    }

			  else //if connectivity_specificity_flag is 0, then there is no penalty for multitype connections
			    {
			      //Selection of target node based on the intraconnectivity matrix

			      // select source (regulator) triplet (random)
			      random_index2 = 3*static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*sd.n_nodes/3); 

			      // add select source (regulator) node type [0, 1, or 2] using intraconnectivity_regulator probability distribution
			      temp = my_rand(global->param.globals.idum, global->param.globals.itemp); // target is selected for given source - regulator by intraconnectivity 

			      if (temp <intraconnectivity_regulator[0])
				random_index2;
			      else if ( temp < (intraconnectivity_regulator[0] + intraconnectivity_regulator[1]) )
				random_index2 += 1; 
			      else 
				random_index2 += 2; 

			      //cout << "\nInitialization of Organism Population: Intraconnectivity Matrix without Multitype Penalization.";
			      temp = my_rand(global->param.globals.idum, global->param.globals.itemp); // target is selected for given source - regulator by intraconnectivity 
			      if (temp <intraconnectivity[0][random_index2%3])
				random_index = 0;
			      else if (temp < (intraconnectivity[0][random_index2%3] + intraconnectivity[1][random_index2%3]))
				random_index =1; 
			      else 
				random_index = 2; 
			      random_index  = 3*static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*sd.n_nodes/3) + random_index;
					
			      //Power law distribution - neg. and pos.
			      w[random_index][random_index2] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.

			    }
			} //enf of intraconnectivity_flag loop
		} //end of all nodes - loop
	} // end of random_flag loop
	else if (change == 0) //set specific value to specific node
		w[position_row][position_col] = value;
	else if (change == 1) //triplet duplication
	{
		this->setn_nodes(sd.n_nodes+3);
		w_temp = new double *[sd.n_nodes];
		for (int i=0;i<sd.n_nodes;i++)
			w_temp[i] = new double [sd.n_nodes];
		for (int i=0;i<sd.n_nodes - 3; i++)
		{
			for (int j=0;j<sd.n_nodes - 3;j++)
			{
				w_temp[i][j] = w[i][j];
			}
			w_temp[i][sd.n_nodes-3] =w[i][position_col];
			w_temp[i][sd.n_nodes-2] =w[i][position_col + 1];
			w_temp[i][sd.n_nodes-1] =w[i][position_col + 2];

			w_temp[sd.n_nodes-3][i] =w[position_row][i];
			w_temp[sd.n_nodes-2][i] =w[position_row + 1][i];
			w_temp[sd.n_nodes-1][i] =w[position_row + 2][i];
		}
		for (int i=0;i<sd.n_nodes-3;i++)
			delete[] w[i];
		delete[] w;

		
		w = new double *[sd.n_nodes];
		for (int i=0;i<sd.n_nodes;i++)
			w[i] = new double [sd.n_nodes];
		for (int i=0;i<sd.n_nodes; i++)
		{
			for (int j=0;j<sd.n_nodes;j++)
			{
				w[i][j] = w_temp[i][j];
			}
		}
		//for the leftover values in the new rows
		for (int i=sd.n_nodes-3;i<sd.n_nodes;i++)
		{
			w[sd.n_nodes-3][i] = w[position_row][i];
			w[sd.n_nodes-2][i] = w[position_row+1][i];
			w[sd.n_nodes-1][i] = w[position_row+2][i];
		}

		for (int i=0;i<sd.n_nodes;i++)
			delete[] w_temp[i];
		delete[] w_temp;
	}
	//triplet destruction
	else if (change ==-1){ 
	  this->setn_nodes(sd.n_nodes-3);
	  w_temp = new double *[sd.n_nodes];
	  for (int i=0;i<sd.n_nodes;i++)
	    w_temp[i] = new double [sd.n_nodes];

		for (int i=0;i<position_row; i++)
		{
			for (int j=0;j<position_col;j++)
			{
				w_temp[i][j] = w[i][j];
			}
			for (int j=position_col+3;j<sd.n_nodes+3;j++)
			{
				w_temp[i][j-3] = w[i][j];
			}
		}
		
		for (int i=position_row+3;i<sd.n_nodes+3; i++)
		{
			for (int j=0;j<position_col;j++)
			{
				w_temp[i-3][j] = w[i][j];
			}
			for (int j=position_col+3;j<sd.n_nodes+3;j++)
			{
				w_temp[i-3][j-3] = w[i][j];
			}
		}

		for (int i=0;i<sd.n_nodes + 3;i++)
			delete[] w[i];
		delete[] w;
		w = new double *[sd.n_nodes];
		for (int i=0;i<sd.n_nodes;i++)
			w[i] = new double [sd.n_nodes];
		for (int i=0;i<sd.n_nodes; i++)
		{
			for (int j=0;j<sd.n_nodes;j++)
			{
				w[i][j] = w_temp[i][j];
			}
		}
		for (int i=0;i<sd.n_nodes;i++)
			delete[] w_temp[i];
		delete[] w_temp;
	}
	
	//mild weight change
	else if (change == 2) 
	  {
	    //zzzz
		for (int i=0;i<sd.n_nodes;i++)
		{
			if (w[position_row][i] !=0)
				w[position_row][i] = w[position_row][i]*(1 + 0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
			if (w[i][position_col] !=0)
			  {
				w[i][position_col] = w[i][position_col]*(1 + 0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
			  }
		}
	}
	else if (change ==3) //strong weight change
	{
		int random_connection_number = 0;
		for (int i=0;i<sd.n_nodes;i++)
		{
			w[position_row][i] = 0;
			w[i][position_row] = 0;
		}
		if (intraconnectivity_flag == 0)
		{
			random_connection_number = static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*(sd.n_nodes+1));
			for (int i=0;i< random_connection_number;i++)
			{
				//cout << "\nInitialization of Organism Population: No Intraconnectivity Matrix, equal connection probability.";
				random_index  = static_cast<int>(sd.n_nodes*my_rand(global->param.globals.idum, global->param.globals.itemp)); //early version when intraconnectivity prob. was uniform.
				//Power law distribution - neg. and pos.
				w[position_row][random_index] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.
			}
			random_connection_number = static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*(sd.n_nodes+1));
			for (int i=0;i< random_connection_number;i++)
			{
				//cout << "\nInitialization of Organism Population: No Intraconnectivity Matrix, equal connection probability.";
				random_index  = static_cast<int>(sd.n_nodes*my_rand(global->param.globals.idum, global->param.globals.itemp)); //early version when intraconnectivity prob. was uniform.
				//Power law distribution - neg. and pos.
				w[random_index][position_row] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.
			}
		}
		else
		{
			//When Connection Specificity Flag is up, then the probability of a node regulating nodes of multiple types is decreased
			if (connection_specificity_flag ==1)
			{

			  // probabilities for a given node to be a source/regulator
			  double temp_intraconnectivity[3] = {intraconnectivity[0][position_row%3],intraconnectivity[1][position_row%3],intraconnectivity[2][position_row%3]};
			  double sum = temp_intraconnectivity[0]+temp_intraconnectivity[1]+temp_intraconnectivity[2];
			  // only if this node can regulate anything:
			  if (sum!=0)
			    {
			    for (int i=0; i<3; i++)
			      temp_intraconnectivity[i] /= sum;

			    double counter[3] = {0};
			    sum = 0;

			    //connection to a number of regulated nodes
			    random_connection_number = static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*(sd.n_nodes+1));
			    for (int index=0;index< random_connection_number;index++)
			      {
				for (int i=0;i<sd.n_nodes;i++)
				  {
				    if (w[i][position_row]!=0)
				      counter[i%3] = counter[i%3] +1;
				  }
				sum = counter[0]+counter[1]+counter[2];

				//calculating (1 - relative frequency of connection type): if sum is 0 then no need for rescaling
				if (sum!=0)
				  {
				    for (int i=0;i<3;i++)
				      counter[i] = 1 - (counter[i]/sum);
				    
				    for (int i=0;i<3;i++)
				      temp_intraconnectivity[i] = temp_intraconnectivity[i]*exp(-1*counter[i]);
				    
				    sum = temp_intraconnectivity[0] + temp_intraconnectivity[1]+ temp_intraconnectivity[2];
				    for (int i=0;i<3;i++)
							temp_intraconnectivity[i] = temp_intraconnectivity[i]/sum;
				  }

				//Calculate connection target
				temp = my_rand(global->param.globals.idum, global->param.globals.itemp); // target is selected for given source - regulator by intraconnectivity 
				if (temp <temp_intraconnectivity[0])
				  random_index = 0;
				else if (temp < (temp_intraconnectivity[0] + temp_intraconnectivity[1]))
				  random_index =1; 
				else 
				  random_index = 2; 
				random_index  = 3*static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*sd.n_nodes/3) + random_index;

				//Power law distribution - neg. and pos.
				w[random_index][position_row] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.
			      }
			    } // end if (sum!=0)

			  //connection to various regulator nodes
			  for (int index = 0; index<3;index++)
			      temp_intraconnectivity[index] = intraconnectivity[position_row%3][index];

			  sum = temp_intraconnectivity[0]+temp_intraconnectivity[1]+temp_intraconnectivity[2];

			  // if any type of a node can regulate this node:
			  if (sum!=0)
			    {
			      for (int i=0; i<3; i++)
				temp_intraconnectivity[i] /= sum;

			      double counter[3] = {0};

			      random_connection_number = static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*(sd.n_nodes+1));
			      for (int index=0;index< random_connection_number;index++)
				{
				  for (int i=0;i<sd.n_nodes;i++)
				    {
				      if (w[position_row][i]!=0)
					counter[i%3] = counter[i%3] +1;
				    }
				  sum = counter[0]+counter[1]+counter[2];

				  //calculating (1 - relative frequency of connection type): if sum is 0 then no need for rescaling
				  if (sum!=0)
				    {
				      for (int i=0;i<3;i++)
					counter[i] = 1 - (counter[i]/sum);
				      
				      for (int i=0;i<3;i++)
					temp_intraconnectivity[i] = temp_intraconnectivity[i]*exp(-1*counter[i]);
				      
				      sum = temp_intraconnectivity[0] + temp_intraconnectivity[1]+ temp_intraconnectivity[2];
				      for (int i=0;i<3;i++)
					temp_intraconnectivity[i] = temp_intraconnectivity[i]/sum;
				    }
				  
				  //Pick regulator
				  temp = my_rand(global->param.globals.idum, global->param.globals.itemp); // target is selected for given source - regulator by intraconnectivity 
				  if (temp <temp_intraconnectivity[0])
				    random_index = 0;
				  else if (temp < (temp_intraconnectivity[0] + temp_intraconnectivity[1]))
				    random_index =1; 
				  else 
				    random_index = 2; 
				  random_index  = 3*static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*sd.n_nodes/3) + random_index;
				  
				  //Power law distribution - neg. and pos.
				  w[position_row][random_index] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.
				}
			    } // end if (sum!=0)
			} //end connection_specificity flag if
			else //if connectivity_specificity_flag is 0, then there is no penalty for multitype connections
			{

			  // create random links FROM this node (with appropriate probability)
			  random_connection_number = static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*(sd.n_nodes+1));
			  for (int i=0; i< random_connection_number; i++)
			    {
			      // (with appropriate probability)
			      if ( my_rand(global->param.globals.idum, global->param.globals.itemp) < intraconnectivity_regulator[position_row%3] ) 
				{
				  temp = my_rand(global->param.globals.idum, global->param.globals.itemp); // target is selected for given source - regulator by intraconnectivity 
				  if (temp <intraconnectivity[0][position_row%3])
				    random_index = 0;
				  else if (temp < (intraconnectivity[0][position_row%3] + intraconnectivity[1][position_row%3]))
				    random_index = 1; 
				  else 
				    random_index = 2; 
				  random_index  = 3*static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*sd.n_nodes/3) + random_index;

				  //Power law distribution - neg. and pos.
				  w[random_index][position_row] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.
			      }
			    }
			  

			  // create random links TO this node (as regulator)
			  random_connection_number = static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*(sd.n_nodes+1));
			  for (int i=0;i< random_connection_number;i++)
			    {
			      
			      // (with appropriate probability)
			      if ( my_rand(global->param.globals.idum, global->param.globals.itemp) < intraconnectivity_regulated[position_row%3] ) 
				{
				  //rows in "interconnectivity" matrix are not normolized, so divide by "intraconnectivity_regulated[position_row%3]"; check that !=0;
				  if (intraconnectivity_regulated[position_row%3] > 0) {
				    temp = my_rand(global->param.globals.idum, global->param.globals.itemp); // target is selected for given source - regulator by intraconnectivity 
				    if (temp <intraconnectivity[position_row%3][0] / intraconnectivity_regulated[position_row%3])
				      random_index = 0;
				    else if (temp < (intraconnectivity[position_row%3][0] + intraconnectivity[position_row%3][1])/intraconnectivity_regulated[position_row%3] )
				      random_index = 1; 
				    else 
				      random_index = 2; 
				    random_index  = 3*static_cast<int>(my_rand(global->param.globals.idum, global->param.globals.itemp)*sd.n_nodes/3) + random_index;
				
				    //Power law distribution - neg. and pos.
				    w[position_row][random_index] = powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp);  //10% of weights will have 4 or above.
				  }
				}

			    }

			  
			} //end connection_specificity flag if else 
		} //end of intraconnectivity_flag loop
	}

	else if (change ==4) //dealllocation and reallocation of w
	{
	  for (int i=0;i<position_row;i++)
	    delete[] w[i];
	  delete[] w;
	  w = new double *[sd.n_nodes];
	  for (int i=0;i<sd.n_nodes;i++)
	    w[i] = new double [sd.n_nodes];
	}
	else if (change ==5) //redundant - may be removed
	{
		w[position_row][position_col] = value;
	}
		
}


void Cell::setregmiddle(double value,int position_row, int position_col, int random_flag, int change)
{ 
	int i,j;
	double **regmiddle_temp;
	int random_index, random_index2;
	if (random_flag ==1)
	{
		regmiddle = new double *[sd.n_nodes];
		for (random_index=0;random_index<sd.n_nodes;random_index++)
		{
			regmiddle[random_index] = new double [sd.n_nodes];
		}

		for (random_index=0;random_index <sd.n_nodes;random_index++)
		{
			for (random_index2=0;random_index2 <sd.n_nodes; random_index2 ++)
				regmiddle[random_index][random_index2] = 10*my_rand(global->param.globals.idum, global->param.globals.itemp);
		}
	}
	else if (change == 0)
		regmiddle[position_row][position_col] = value;
	else if (change == 1)
	{
		regmiddle_temp = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regmiddle_temp[i] = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes - 3; i++)
		{
			for (j=0;j<sd.n_nodes - 3;j++)
			{
				regmiddle_temp[i][j] = regmiddle[i][j];
			}
			regmiddle_temp[i][sd.n_nodes-3] =regmiddle[i][position_col];
			regmiddle_temp[i][sd.n_nodes-2] =regmiddle[i][position_col + 1];
			regmiddle_temp[i][sd.n_nodes-1] =regmiddle[i][position_col + 2];

			regmiddle_temp[sd.n_nodes-3][i] =regmiddle[position_row][i];
			regmiddle_temp[sd.n_nodes-2][i] =regmiddle[position_row + 1][i];
			regmiddle_temp[sd.n_nodes-1][i] =regmiddle[position_row + 2][i];
		}
		for (i=0;i<sd.n_nodes -3 ;i++)
			delete[] regmiddle[i];
		delete[] regmiddle;
		regmiddle = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regmiddle[i] = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes; i++)
		{
			for (j=0;j<sd.n_nodes;j++)
			{
				regmiddle[i][j] = regmiddle_temp[i][j];
			}
		}
		//for the leftover values in the new rows
		for (i=sd.n_nodes-3;i<sd.n_nodes;i++)
		{
			regmiddle[sd.n_nodes-3][i] = regmiddle[position_row][i];
			regmiddle[sd.n_nodes-2][i] = regmiddle[position_row+1][i];
			regmiddle[sd.n_nodes-1][i] = regmiddle[position_row+2][i];
		}


		for (i=0;i<sd.n_nodes;i++)
			delete[] regmiddle_temp[i];
		delete[] regmiddle_temp;
	}
	else if (change ==-1)
	{
		//cout << "sd.n_nodes: " << this->sd.n_nodes<< "\n after: ";
		//cout << this->sd.n_nodes << "\n";
		regmiddle_temp = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regmiddle_temp[i] = new double [sd.n_nodes];

		for (i=0;i<position_row; i++)
		{
			for (j=0;j<position_col;j++)
			{
				regmiddle_temp[i][j] = regmiddle[i][j];
			}
			for (j=position_col+3;j<sd.n_nodes+3;j++)
			{
				regmiddle_temp[i][j-3] = regmiddle[i][j];
			}
		}
		
		for (i=position_row+3;i<sd.n_nodes+3; i++)
		{
			for (j=0;j<position_col;j++)
			{
				regmiddle_temp[i-3][j] = regmiddle[i][j];
			}
			for (j=position_col+3;j<sd.n_nodes+3;j++)
			{
				regmiddle_temp[i-3][j-3] = regmiddle[i][j];
			}
		}
		
		for (i=0;i<sd.n_nodes+3;i++)
			delete[] regmiddle[i];
		delete[] regmiddle;
		regmiddle = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regmiddle[i] = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes; i++)
		{
			for (j=0;j<sd.n_nodes;j++)
			{
				regmiddle[i][j] = regmiddle_temp[i][j];
			}
		}
		for (i=0;i<sd.n_nodes;i++)
			delete[] regmiddle_temp[i];
		delete[] regmiddle_temp;
	}
	else if (change == 2)
	{
		for (i=0;i<sd.n_nodes;i++)
		{
			regmiddle[position_row   ][i] = regmiddle[position_row  ][i]*(1 + +0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
			regmiddle[i][position_row   ] = regmiddle[i][position_row  ]*(1 + +0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
		}
	}
	else if (change ==3)
	{
		for (i=0;i<sd.n_nodes;i++)
		{
			regmiddle[position_row   ][i] = 10*my_rand(global->param.globals.idum, global->param.globals.itemp);
			regmiddle[i][position_row   ] = 10*my_rand(global->param.globals.idum, global->param.globals.itemp);
		}
	}
	else if (change ==4)
	{
		// we use the value of the position_row variable as the number of nodes that we have to delete.
		for (i=0;i<position_row;i++)
			delete[] regmiddle[i];
		delete[] regmiddle;
		regmiddle = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regmiddle[i] = new double [sd.n_nodes];
	}
	else if (change ==5)
	{
		regmiddle[position_row][position_col] = value;
	}
}

void Cell::setregslope(double value, int position_row, int position_col, int random_flag, int change)
{ 
	int i,j;
	double **regslope_temp;
	if (random_flag ==1)
	{
		int random_index, random_index2;
		regslope = new double *[sd.n_nodes];
		for (random_index=0;random_index<sd.n_nodes;random_index++)
			regslope[random_index] = new double [sd.n_nodes];

		for (random_index=0;random_index <sd.n_nodes;random_index++)
		{
			for (random_index2=0;random_index2 <sd.n_nodes; random_index2 ++)
				regslope[random_index][random_index2] = 1 + 3*my_rand(global->param.globals.idum, global->param.globals.itemp);
		}
	}
	else if (change == 0)
		regslope[position_row][position_col] = value;
	else if (change == 1)
	{
		regslope_temp = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regslope_temp[i] = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes - 3; i++)
		{
			for (j=0;j<sd.n_nodes - 3;j++)
			{
				regslope_temp[i][j] = regslope[i][j];
			}
			regslope_temp[i][sd.n_nodes-3] =regslope[i][position_col];
			regslope_temp[i][sd.n_nodes-2] =regslope[i][position_col + 1];
			regslope_temp[i][sd.n_nodes-1] =regslope[i][position_col + 2];

			regslope_temp[sd.n_nodes-3][i] =regslope[position_row][i];
			regslope_temp[sd.n_nodes-2][i] =regslope[position_row + 1][i];
			regslope_temp[sd.n_nodes-1][i] =regslope[position_row + 2][i];
		}
		for (i=0;i<sd.n_nodes-3;i++)
			delete[] regslope[i];
		delete[] regslope;
		regslope = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regslope[i] = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes; i++)
		{
			for (j=0;j<sd.n_nodes;j++)
			{
				regslope[i][j] = regslope_temp[i][j];
			}
		}
		//for the leftover values in the new rows
		for (i=sd.n_nodes-3;i<sd.n_nodes;i++)
		{
			regslope[sd.n_nodes-3][i] = regslope[position_row][i];
			regslope[sd.n_nodes-2][i] = regslope[position_row+1][i];
			regslope[sd.n_nodes-1][i] = regslope[position_row+2][i];
		}

		for (i=0;i<sd.n_nodes;i++)
			delete[] regslope_temp[i];
		delete[] regslope_temp;
	}
	else if (change ==-1)
	{
		regslope_temp = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regslope_temp[i] = new double [sd.n_nodes];

		for (i=0;i<position_row; i++)
		{
			for (j=0;j<position_col;j++)
			{
				regslope_temp[i][j] = regslope[i][j];
			}
			for (j=position_col+3;j<sd.n_nodes+3;j++)
			{
				regslope_temp[i][j-3] = regslope[i][j];
			}
		}
		
		for (i=position_row+3;i<sd.n_nodes+3; i++)
		{
			for (j=0;j<position_col;j++)
			{
				regslope_temp[i-3][j] = regslope[i][j];
			}
			for (j=position_col+3;j<sd.n_nodes+3;j++)
			{
				regslope_temp[i-3][j-3] = regslope[i][j];
			}
		}
		
		for (i=0;i<sd.n_nodes+3;i++)
			delete[] regslope[i];
		delete[] regslope;
		regslope = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regslope[i] = new double [sd.n_nodes];
		for (i=0;i<sd.n_nodes; i++)
		{
			for (j=0;j<sd.n_nodes;j++)
			{
				regslope[i][j] = regslope_temp[i][j];
			}
		}
		for (i=0;i<sd.n_nodes;i++)
			delete[] regslope_temp[i];
		delete[] regslope_temp;
	}
	else if (change == 2)
	{
		for (i=0;i<sd.n_nodes;i++)
		{
			regslope[position_row   ][i] = regslope[position_row  ][i]*(1 + +0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
			regslope[i][position_row   ] = regslope[i][position_row   ]*(1 + +0.2*gaussian(0,1,global->param.globals.idum,global->param.globals.itemp));
			regslope[position_row   ][i] = MAX(regslope[position_row   ][i],0.1);
			regslope[i][position_row   ] = MAX(regslope[i][position_row   ],0.1);
		}
	}
	else if (change ==3)
	{
		for (i=0;i<sd.n_nodes;i++)
		{
			regslope[position_row   ][i] = 1 + 3*my_rand(global->param.globals.idum, global->param.globals.itemp);
			regslope[i][position_row   ] = 1 + 3*my_rand(global->param.globals.idum, global->param.globals.itemp);
		}
	}
	else if (change ==4)
	{

		for (i=0;i<position_row;i++)
			delete[] regslope[i];
		delete[] regslope;
		regslope = new double *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			regslope[i] = new double [sd.n_nodes];
	}
	else if (change ==5)
	{
		regslope[position_row][position_col] = value;
	}
		
}


void Cell::setv_history(int which_col, int initialize, int selection_period, int which_row)
{
	int ** v_history_temp;
	int i,j;
	if (initialize == 1)
	{
		v_history = new int *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			v_history[i] = new int [selection_period];
	
		for (i=0;i<sd.n_nodes;i++)
		{
			for (j=0;j<selection_period;j++)
				v_history[i][j] = -1;
		}
	}
	//addition of a node
	else if (initialize == 2) 
	{
		v_history_temp = new int *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			v_history_temp[i] = new int [selection_period];
		for (i=0;i<sd.n_nodes - 3; i++)
		{
			for (j=0;j<=which_col ;j++)
			{
				v_history_temp[i][j] = v_history[i][j];
				v_history_temp[sd.n_nodes-3][j] =-1; // a value of -1 indicates absence of value.
				v_history_temp[sd.n_nodes-2][j] =-1; 
				v_history_temp[sd.n_nodes-1][j] =-1;
			}
		}
				v_history_temp[sd.n_nodes-3][which_col] = 0;
				v_history_temp[sd.n_nodes-2][which_col] = 0;
				v_history_temp[sd.n_nodes-1][which_col] = 0;


		for (i=0;i<sd.n_nodes-3;i++)
			delete[] v_history[i];
		delete[] v_history;
 		v_history = new int *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			v_history[i] = new int [selection_period];;
		for (i=0;i<sd.n_nodes; i++)
		{
			for (j=0;j<selection_period;j++)
			{
				v_history[i][j] = v_history_temp[i][j];
			}
		}

		for (i=0;i<sd.n_nodes;i++)
			delete[] v_history_temp[i];
		delete[] v_history_temp;
	}

	else if (initialize == -1)
	{
		for (i=0;i<sd.n_nodes;i++)
			delete[] v_history[i];
		delete[] v_history;
	}
	//deletion of a node. which_col will indicate the triplet
	else if (initialize == -2) 
	{
		v_history_temp = new int *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			v_history_temp[i] = new int [selection_period];

		for (i=0;i<which_row; i++)
		{
			for (j=0;j<=which_col;j++)
				v_history_temp[i][j] = v_history[i][j];
		}

		for (i=which_row;i<sd.n_nodes;i++)
		{
			for (j=0;j<=which_col;j++)
				v_history_temp[i][j] = v_history[i+3][j];
		}
			
		
		for (i=0;i<sd.n_nodes+3;i++)
			delete[] v_history[i];
		delete[] v_history;
		v_history = new int *[sd.n_nodes];
		for (i=0;i<sd.n_nodes;i++)
			v_history[i] = new int [selection_period];
		for (i=0;i<sd.n_nodes; i++)
		{
			for (j=0;j<=which_col;j++)
			{
				v_history[i][j] = v_history_temp[i][j];
			}
		}
		for (i=0;i<sd.n_nodes;i++)
			delete[] v_history_temp[i];
		delete[] v_history_temp;
	}

	else
	{
		for (i=0;i<sd.n_nodes;i++)
			v_history[i][which_col] = v[i];
	}
}

#if _USE_HDF5
void Cell::save(int local_cell_number, int epochID, hid_t &outfile) {

	int rank;
	MPI_Comm_rank(MPI_COMM_WORLD, &rank);
	stringstream out_data;

	hid_t epoch_g, cell_g; // epoch and cell group handles
	hid_t dataset, dataspace, attribute; // data storage id's
	hid_t lapl; // link access property list, used to check existence of objects
	hsize_t dims1[1], dims2[2]; // dataset dimensions

	char epoch_name[MAX_H5_OBJECT_NAME_LENGTH],
	cell_name[MAX_H5_OBJECT_NAME_LENGTH];
	lapl = H5Pcreate(H5P_LINK_ACCESS);

	sprintf(epoch_name, "%s%d", H5_GROUP_EPOCH, epochID);
	sprintf(cell_name, "%s%d[%d]", H5_GROUP_CELL, rank, local_cell_number);

	// if epoch group does not exist
	if (!H5Lexists(outfile, epoch_name, lapl)) {
		// create epoch and cell groups
		epoch_g = H5Gcreate(outfile, epoch_name, H5P_DEFAULT, H5P_DEFAULT,
				H5P_DEFAULT);
		cell_g = H5Gcreate(epoch_g, cell_name, H5P_DEFAULT, H5P_DEFAULT,
				H5P_DEFAULT);
	} else {
		epoch_g = H5Gopen(outfile, epoch_name, H5P_DEFAULT);
		// now epoch group exists for sure, cell group could possibly exist
		// if cell group doesn't exist
		if (!H5Lexists(epoch_g, cell_name, lapl))
		cell_g = H5Gcreate(epoch_g, cell_name, H5P_DEFAULT, H5P_DEFAULT,
				H5P_DEFAULT);
		else
		cell_g = H5Gopen(epoch_g, cell_name, H5P_DEFAULT);
	}
	// now cell group exists for sure

	dataspace = H5Screate(H5S_SCALAR);

	// save fitness as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_FITNESS, H5T_NATIVE_DOUBLE,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, &sd.fitness);

	// save rank as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_RANK, H5T_NATIVE_INT,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_INT, &rank);

	// save n_nodes as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_N_NODES, H5T_NATIVE_INT,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_INT, &sd.n_nodes);

	// save coord as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_COORD, H5T_NATIVE_INT,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_INT, &sd.coord);

	// save creation_cell as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_CREATION, H5T_NATIVE_DOUBLE,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, &sd.creation_cell);

	// save destruction_cell as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_DESTRUCTION,
			H5T_NATIVE_DOUBLE, dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, &sd.destruction_cell);

	// save mild_mutation as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_MILD_MUTATION,
			H5T_NATIVE_DOUBLE, dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, &sd.mild_mutation_cell);

	// save strong_mutation as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_STRONG_MUTATION,
			H5T_NATIVE_DOUBLE, dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, &sd.strong_mutation_cell);

	// save evolvability_cell as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_EVOLVABILITY,
			H5T_NATIVE_DOUBLE, dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, &sd.evolvability_cell);

	// save energy as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_ENERGY, H5T_NATIVE_DOUBLE,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, &sd.energy);

	// set dims1 to denote the number of nodes, for saving vectors later
	dims1[0] = sd.n_nodes;
	H5Sset_extent_simple(dataspace, 1, dims1, NULL);

	// save input_sensor_protein as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_ISP, H5T_NATIVE_INT,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_INT, input_sensor_protein);

	// save slope as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_SLOPE, H5T_NATIVE_DOUBLE,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, slope);

	// save middlepoint as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_MIDDLEPOINT,
			H5T_NATIVE_DOUBLE, dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, middlepoint);

	// save basal as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_BASAL, H5T_NATIVE_DOUBLE,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, basal);

	// save degradation as an attribute
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_DEGRADATION,
			H5T_NATIVE_DOUBLE, dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, deg);

	// set dims2 to denote the number of nodes, for saving matrices later
	dims2[0] = sd.n_nodes;
	dims2[1] = sd.n_nodes;
	H5Sset_extent_simple(dataspace, 2, dims2, NULL);

	// save w as an attribute
	double *w_as_vector = convertMatrixToVector(w, dims2[0], dims2[1]);
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_WEIGHT, H5T_NATIVE_DOUBLE,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, w_as_vector);
	delete w_as_vector;

	// save regslope as an attribute
	double *regslope_as_vector = convertMatrixToVector(regslope, dims2[0],
			dims2[1]);
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_REGSLOPE, H5T_NATIVE_DOUBLE,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, regslope_as_vector);
	delete regslope_as_vector;

	// save regmiddle as an attribute
	double *regmiddle_as_vector = convertMatrixToVector(regmiddle, dims2[0],
			dims2[1]);
	attribute = H5Acreate(cell_g, H5_ATTRIB_CELL_REGMIDDLE, H5T_NATIVE_DOUBLE,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
	H5Awrite(attribute, H5T_NATIVE_DOUBLE, regmiddle_as_vector);
	delete regmiddle_as_vector;

	// cleanup
	H5Pclose(lapl);
	H5Aclose(attribute);
	H5Sclose(dataspace);
	H5Gclose(cell_g);
	H5Gclose(epoch_g);
}

#endif

void Cell::save_ascii(int local_cell_number, int epochID, std::string& out_str)
{
  std::stringstream out_data;

  out_data << "In the end of epoch " << epochID <<"\n";

  int rank=0;
#if _IF_MPI
  MPI_Comm_rank(MPI_COMM_WORLD,&rank); 
#endif

  out_data << "Cell: " << rank << "[" << local_cell_number << "]."<<sd.ID << " " << sd.label << "\n";
  out_data << "Number of nodes: " << getn_nodes() << "\n";
  out_data << "Fitness: " << setprecision(5) << getfitness() << "\n";
  out_data << "Energy: " << getenergy() << "\n";
  out_data << "Mutational bias: " << getmutation(1) << " " << getmutation(2) << " " << getmutation(3) << " " << getmutation(4) << " " << getmutation(5) << "\n";
  out_data << "Connection to any signal (input sensor protein):\n";
  for (int i=0;i<sd.n_nodes;i++)
    out_data << input_sensor_protein[i] << " ";

  out_data << "\nSlope:\n";
  for (int i=0;i<sd.n_nodes;i++)
    out_data << std::fixed << std::setprecision(4) << std::setw(8) << slope[i] << " ";

  out_data << "\nMiddlepoint:\n";
  for (int i=0;i<sd.n_nodes;i++)
    out_data << std::fixed << std::setprecision(4) << std::setw(8) << middlepoint[i] << " ";
	
  out_data << "\nBasal:\n";
  for (int i=0;i<sd.n_nodes;i++)
    out_data << std::fixed << std::setprecision(4) << std::setw(8) << basal[i] << " ";

  out_data << "\nDegradation:\n";
  for (int i=0;i<sd.n_nodes;i++)
    out_data << std::fixed << std::setprecision(4) << std::setw(8) << deg[i] << " ";

  out_data << "\nWeight matrix";
  for (int i=0;i<sd.n_nodes;i++)
    {
      out_data << "\n";
      for (int j=0;j<sd.n_nodes;j++)
	out_data << std::fixed << std::setprecision(4) << std::setw(8) << w[i][j] << " ";
    }

  out_data << "\nRegslope";
  for (int i=0;i<sd.n_nodes;i++){
    out_data << "\n";
    for (int j=0;j<sd.n_nodes;j++)
      out_data << std::fixed << std::setprecision(4) << std::setw(8) << regslope[i][j] << " ";
  }

  out_data << "\nRegmiddle";
  for (int i=0;i<sd.n_nodes;i++) {
    out_data << "\n";
    for (int j=0;j<sd.n_nodes;j++)
      out_data << std::fixed << std::setprecision(4) << std::setw(8) << regmiddle[i][j] << " ";
  }

  out_data << " \n\n";

  //return
  out_str=out_data.str();
}

void Cell::save_v_history_ascii(std::string& out_str, int local_cell_number, int epochID)
{
  std::stringstream out_data;

  int rank=0;
#if _IF_MPI
  MPI_Comm_rank(MPI_COMM_WORLD,&rank); 
#endif
  out_data <<rank<< "[" << local_cell_number << "]." << sd.ID<< "  ("
	   << "in the end of epoch " << epochID << ")"
	   << " n_nodes=" << sd.n_nodes 
	   << " selection_period=" << selection_period
	   << "\n";

  for (int i=0;i<sd.n_nodes;i++)
    {
      for (int j=0;j<selection_period;j++)
	out_data << v_history[i][j] << " ";
      out_data << "\n";
    }
  out_data << "\n";
  
  // return
  out_str=out_data.str();
}

void Cell::save_w_graphml(std::string& out_str)
{
  std::stringstream out_data;

  out_data.str("");
  out_data << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
  out_data << "<graphml xmlns=\"http://graphml.graphdrawing.org/xmlns\"\n";
  out_data << "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n";
  out_data << "xsi:schemaLocation=\"http://graphml.graphdrawing.org/xmlns\n";
  out_data << "http://graphml.graphdrawing.org/xmlns/1.0/graphml.xsd\">\n";

  // define edge-weight attribute
  out_data << "<key id=\"d1\" for=\"edge\"\n";
  out_data << "attr.name=\"weight\" attr.type=\"double\"/>\n";
  out_data << "<graph id=\"w\" edgedefault=\"directed\">\n";

  std::string node1_type, node2_type;
  for (int i=0; i<sd.n_nodes; i++){
    if (i%3==0)
      node1_type="mRNA";
    else if (i%3==1)
      node1_type="P";
    else if (i%3==2)
      node1_type="mP";

    out_data << "<node id=\""<< node1_type <<"_"<< i/3+1 <<"\"/>\n";
  }

  // count number of non-zero links
  int n_links=0;
  for (int i=0; i<sd.n_nodes; i++) {
    for (int j=0; j<sd.n_nodes; j++) {
      if ( getw(i,j) != 0 )
	n_links++;
    }
  }
    // now create a list of links:
    int *link_row, * link_col;
    link_row = new int [n_links];
    link_col = new int [n_links];
    int link_counter=0;
    for (int i=0; i<sd.n_nodes; i++) {
      for (int j=0; j<sd.n_nodes; j++) {
	if ( getw(i,j) != 0 ){
	  link_row[link_counter]=i;
	  link_col[link_counter]=j;
	  link_counter++;
	}
      }
    }

  for (int link_i=0; link_i < n_links; link_i++){

    if (link_row[link_i]%3==0)
      node1_type="mRNA";
    else if (link_row[link_i]%3==1)
      node1_type="P";
    else if (link_row[link_i]%3==2)
      node1_type="mP";
    
    if (link_col[link_i]%3==0)
      node2_type="mRNA";
    else if (link_col[link_i]%3==1)
      node2_type="P";
    else if (link_col[link_i]%3==2)
	node2_type="mP";

    out_data << " <edge id=\"e"<< link_i 
	     <<"\" source=\""<< node1_type <<"_"<< link_row[link_i]/3+1  
	     <<"\" target=\""<< node2_type <<"_"<< link_col[link_i]/3+1 <<"\">\n"
	     << "  <data key=\"d1\">"
	     <<std::fixed << std::setprecision(2) << getw(link_row[link_i], link_col[link_i]) 
	     <<"</data>\n</edge>\n";
    }
  out_data << "</graph>\n</graphml>\n";


  delete[] link_row;
  delete[] link_col;
  //return
  out_str=out_data.str();
}


#if _USE_HDF5
void Cell::load(hid_t &infile, int epochID, int proc_to_load, int cell_to_load, int next_cell_ID) {

char cell_name[MAX_H5_OBJECT_NAME_LENGTH],
epoch_name[MAX_H5_OBJECT_NAME_LENGTH];
hid_t cell_g, epoch_g, attribute, dataset;
hsize_t dims2[2];
herr_t status;

sprintf(epoch_name, "%s%d", H5_GROUP_EPOCH, epochID);
sprintf(cell_name, "%s%d[%d]", H5_GROUP_CELL, proc_to_load, cell_to_load);

epoch_g = H5Gopen(infile, epoch_name, H5P_DEFAULT);
cell_g = H5Gopen(epoch_g, cell_name, H5P_DEFAULT);

//int old_nnodes = sd.n_nodes;
sd.ID = next_cell_ID;

// load n_nodes
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_N_NODES, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_INT, &sd.n_nodes);

// load fitness
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_FITNESS, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, &sd.fitness);

// load energy
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_ENERGY, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, &sd.energy);

// load coord
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_COORD, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_INT, &sd.coord);

// load creation_cell
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_CREATION, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, &sd.creation_cell);

// load destruction_cell
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_DESTRUCTION, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, &sd.destruction_cell);

// load mild_mutation
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_MILD_MUTATION, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, &sd.mild_mutation_cell);

// load strong_mutation
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_STRONG_MUTATION, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, &sd.strong_mutation_cell);

// load evolvability_cell
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_EVOLVABILITY, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, &sd.evolvability_cell);

// load energy
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_ENERGY, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, &sd.energy);

  //setting random values and deleting previous allocation
  setv(0,0,1,1); 
  //MG (not a full functionality) assignV(0, 0);

// load input_sensor_protein
input_sensor_protein = new int[sd.n_nodes];
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_ISP, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_INT, input_sensor_protein);

// load slope
slope = new double[sd.n_nodes];
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_SLOPE, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, slope);

// load middlepoint
middlepoint = new double[sd.n_nodes];
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_MIDDLEPOINT, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, middlepoint);

// load basal
basal = new double[sd.n_nodes];
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_BASAL, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, basal);

// load degradation
deg = new double[sd.n_nodes];
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_DEGRADATION, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, deg);

// assign dimensional lengths to each value in dimensionality vector
dims2[0] = sd.n_nodes;
dims2[1] = sd.n_nodes;

// create vector for storing these values,
// which will later be converted into a matrix
double *temp_double_vector;
int *temp_int_vector;

// load w
temp_double_vector = new double[dims2[0] * dims2[1]];
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_WEIGHT, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, temp_double_vector);
w = convertVectorToMatrix(temp_double_vector, dims2[0], dims2[1]);
delete temp_double_vector;

// load regslope
temp_double_vector = new double[dims2[0] * dims2[1]];
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_REGSLOPE, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, temp_double_vector);
regslope = convertVectorToMatrix(temp_double_vector, dims2[0], dims2[1]);
delete temp_double_vector;

// load regmiddle
temp_double_vector = new double[dims2[0] * dims2[1]];
attribute = H5Aopen(cell_g, H5_ATTRIB_CELL_REGMIDDLE, H5P_DEFAULT);
status = H5Aread(attribute, H5T_NATIVE_DOUBLE, temp_double_vector);
regmiddle = convertVectorToMatrix(temp_double_vector, dims2[0], dims2[1]);
delete temp_double_vector;

// reassign proper values to dimensonality matrix for v_history
//dims2[1] = selection_period;

//// load v_history
//temp_int_vector = new int[dims2[0] * dims2[1]];
//dataset = H5Dopen(cell_g, H5_DATASET_CELL_V_HISTORY, H5P_DEFAULT);
//status = H5Dread(dataset, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT,
//		temp_int_vector);
//v_history = convertVectorToMatrix(temp_int_vector, dims2[0], dims2[1]);
//delete temp_int_vector;

// cleanup
//H5Dclose(dataset);
H5Aclose(attribute);
H5Gclose(cell_g);
H5Gclose(epoch_g);
}

#endif

void Cell::load_ascii(std::string file_name, int cell_number, std::string& out_str, int epochID, int next_cell_ID)
{
  //  std::stringstream out_data;

  int rank=0;
#if _IF_MPI
  MPI_Comm_rank(MPI_COMM_WORLD,&rank); 
#endif

  
  char *x = new char [100];
  char *y = new char [100];
  char *z = new char [100];
  char *o = new char [100];
  char *w = new char [100];

  std::ifstream in_file;
  in_file.open(file_name.c_str());
  if (!in_file)
    {
      cout << "\n\nPopulation ERROR: Nothing to import from " << file_name << " !\nExiting...\nERROR  rannk " << rank<<"\n\n";
      exit(1);   
    }

  for (int cell_i=0; cell_i<cell_number; cell_i++){

    in_file.ignore(500,'\n'); // [1st line -- comment]

    //[2nd line -- old cell id and label]
    // OLD format:  in_file.ignore(500,'\n');
    in_file >> x >> x >> x; 
    setlabel(atoi(x)); // preserve the label from the input

    setID(next_cell_ID);  

    in_file >> x >> x >> x >> x; 
    int old_numnodes = getn_nodes();
    setn_nodes(atoi(x));

    in_file >> x >> x;
    setfitness(atof(x));

    in_file >> x >> x;
    setenergy(0,1); //initialization of variable energy;
    setenergy(atof(x),0); //this assigns the real energy

    in_file >> x >> x;
    in_file >> x; //first mutational
    in_file >> y; //second
    in_file >> z; //third
    in_file >> o; //fourth
    in_file >> w; //evolvability
    setmutation(atof(x),atof(y),atof(z),atof(o),atof(w)); 
    in_file.ignore(100,'\n'); // eol
    in_file.ignore(100,'\n'); // Connection to any signal (input sensor protein)

    setmiddle(0,0,0,4);
    setslope(0,0,0,4);
    setdeg(0,0,0,4);
    setbasal(0,0,0,4);
    setw(0,old_numnodes,0,0,4);
    setregslope(0,old_numnodes,0,0,4);
    setregmiddle(0,old_numnodes,0,0,4);
    setinput(0,-4,0,0);
    setv(0,0,0,4);

    //input sensor
    for (int i=0;i<getn_nodes();i++) {
      in_file >> x;
      setinput(i,atoi(x),0,0);
    }

    in_file.ignore(100,'\n'); // eol
    in_file.ignore(100,'\n'); // slope
    for (int i=0;i<getn_nodes();i++) {
      in_file >> x;
      setslope(i,atof(x),0,5);
    }

    in_file.ignore(100,'\n'); // eol
    in_file.ignore(100,'\n'); // midpoint
    for (int i=0; i<getn_nodes(); i++){
      in_file >> x;
      setmiddle(i,atof(x),0,5);
    }

    in_file.ignore(100,'\n'); // eol
    in_file.ignore(100,'\n'); // basal
    for (int i=0; i<getn_nodes(); i++) {
      in_file >> x;
      setbasal(i,atof(x),0,5);
    }
	
    in_file.ignore(100,'\n'); // eol
    in_file.ignore(100,'\n'); // degradation
    for (int i=0; i<getn_nodes(); i++) {
      in_file >> x;
      setdeg(i,atof(x),0,5);
    }

    in_file.ignore(100,'\n'); // eol
    in_file.ignore(100,'\n'); //weight matrix
    for (int i=0;i<getn_nodes();i++) {
      for (int j=0;j<getn_nodes();j++) {
	in_file >> x;
	setw(atof(x),i,j,0,5);
      }
      in_file.ignore(100,'\n'); //eol
    }

    in_file.ignore(100,'\n');  //regslope
    for (int i=0; i<getn_nodes(); i++){
      for (int j=0; j<getn_nodes(); j++) {
	in_file >> x;
	setregslope(atof(x),i,j,0,5);
      }
      in_file.ignore(100,'\n');
    }

    in_file.ignore(100,'\n'); //redmiddle
    for (int i=0;i<getn_nodes();i++) {
      for (int j=0;j<getn_nodes();j++) {
	in_file >> x;
	setregmiddle(atof(x),i,j,0,5);
      }
      in_file.ignore(100,'\n');
    }
	
    in_file.ignore(100,'\n'); // skip last empty line
  }

  //setting random values and deleting previous allocation
  setv(0,0,1,1); 

  in_file.close();
    
  delete x;
  delete y;
  delete z;
  delete o;
  delete w;
	

  // empty return for now
  out_str="";
  
  //  out_str=out_data.str();
}















void Cell::output_information(const char * filename,int selection_period, int full_version)
{
	int i,j;
	ofstream myfile;
	myfile.open (filename, ios::out | ios::app);
if (full_version == 0)
{
  myfile << "Cell " << getID() <<"\n";
  myfile << "sd.n_nodes " << getn_nodes() << "\n";
  myfile << "fitness " << setprecision(5) << getfitness() << "\n";
  myfile << "energy " << getenergy() << "\n";
  myfile << "sparcity " << getsparcity() << " \n";
  myfile << "MutationalBias " << getmutation(1) << " " << getmutation(2) << " " << getmutation(3) << " " << getmutation(4) << " " << getmutation(5) << "\n";
  myfile << "Connection to any signal - Input Sensor Protein:\n";
	for (i=0;i<sd.n_nodes;i++)
		myfile << input_sensor_protein[i] << " ";
	

	myfile << "\nWeight Matrix";
	for (i=0;i<sd.n_nodes;i++)
	{
		myfile << "\n";
		for (j=0;j<sd.n_nodes;j++)
			myfile << w[i][j] << " ";
	}

	myfile << "\nv_history:";
	for (i=0;i<sd.n_nodes;i++)
	{
		myfile << "\n" << i+1 << ": ";
		for (j=0;j<selection_period;j++)
			myfile << v_history[i][j] << " ";
	}
	myfile << " \n\n";
	myfile.close();
}


if (full_version  == 1 )
{
	myfile << "Cell " << sd.ID <<"\n";
	myfile << "sd.n_nodes " << getn_nodes() << "\n";
	myfile << "fitness " << setprecision(5) << getfitness() << "\n";
	myfile << "energy " << getenergy() << "\n";
	myfile << "sparcity " << getsparcity() << " \n";
	myfile << "MutationalBias " << getmutation(1) << " " << getmutation(2) << " " << getmutation(3) << " " << getmutation(4) << " " << getmutation(5) << "\n";
	myfile << "Connection to any signal - Input Sensor Protein:\n";
	for (i=0;i<sd.n_nodes;i++)
		myfile << input_sensor_protein[i] << " ";

	myfile << "\nslope:\n";
	for (i=0;i<sd.n_nodes;i++)
		myfile << slope[i] << " ";

	myfile << "\nmiddlepoint:\n";
	for (i=0;i<sd.n_nodes;i++)
		myfile << middlepoint[i] << " ";
	
	myfile << "\nbasal:\n";
	for (i=0;i<sd.n_nodes;i++)
		myfile << basal[i] << " ";

	myfile << "\ndegradation:\n";
	for (i=0;i<sd.n_nodes;i++)
		myfile << deg[i] << " ";

	myfile << "\nWeight Matrix";
	for (i=0;i<sd.n_nodes;i++)
	{
		myfile << "\n";
		for (j=0;j<sd.n_nodes;j++)
			myfile << w[i][j] << " ";
	}

	myfile << "\nregslope";
	for (i=0;i<sd.n_nodes;i++)
	{
		myfile << "\n";
		for (j=0;j<sd.n_nodes;j++)
			myfile << regslope[i][j] << " ";
	}

	myfile << "\nregmiddle";
	for (i=0;i<sd.n_nodes;i++)
	{
		myfile << "\n";
		for (j=0;j<sd.n_nodes;j++)
			myfile << regmiddle[i][j] << " ";
	}

	myfile << "\nv_history:";
	for (i=0;i<sd.n_nodes;i++)
	{
		myfile << "\n" << i+1 << ": ";
		for (j=0;j<selection_period;j++)
			myfile << v_history[i][j] << " ";
	}
	myfile << " \n\n";
	myfile.close();
}

if (full_version == 2)
  {
    myfile << "CellID " << sd.ID << " Energy: " << getenergy() << " Fitness: " << setprecision(5) << getfitness() <<"\n";
  myfile.close();
 }
}

void Cell::reduce_to_module(int module_size)
{

  //----------------------------------------------------------------------
  // first copy metabolic protein to the end

  // add to the end of the triplet list:
  int  metabolic_tripl = 3; // metabolictriplet -- nodes (3,4,5) -- second triplet

  setw(0,metabolic_tripl ,metabolic_tripl,0,1);
  setregmiddle(0,metabolic_tripl ,metabolic_tripl,0,1);
  setregslope(0,metabolic_tripl ,metabolic_tripl,0,1);

  // adds new nodes, values set to zero
  setv(metabolic_tripl,0,0,1);

  setmiddle(metabolic_tripl ,0,0,1);
  setslope(metabolic_tripl ,0,0,1);
  setdeg(metabolic_tripl ,0,0,1);
  setbasal(metabolic_tripl ,0,0,1);

  setinput(metabolic_tripl, -1, global->param.general.n_signals, 0);

  //setv_history(select_index,2,selection_period,metabolic_tripl); 
  //setv_history(0,2,selection_period,metabolic_tripl); 

  // adds new mutation counters and set them to 0
  setmutation_counter(0,metabolic_tripl,1);

  //----------------------------------------------------------------------
  // now delete metabolic triplet
  int triplet_to_del=metabolic_tripl;

  setw(0,triplet_to_del ,triplet_to_del,0,-1);
  setregmiddle(0,triplet_to_del ,triplet_to_del,0,-1);
  setregslope(0,triplet_to_del ,triplet_to_del,0,-1);
  setv(triplet_to_del ,0,0,-1);
  setmiddle(triplet_to_del ,0,0,-1);
  setslope(triplet_to_del ,0,0,-1);
  setdeg(triplet_to_del ,0,0,-1);
  setbasal(triplet_to_del ,0,0,-1);
  setinput(triplet_to_del,-2,0,0);
  setmutation_counter(0,triplet_to_del,-1);

  //----------------------------------------------------------------------
  // delete as many random nodes to reduce this cell size to module_size*3 nodes (keep last triplet unchanged)
  while ( ROUND(getn_nodes()/3)>module_size+1 ) {
    triplet_to_del = 3* ROUND(((getn_nodes()/3)-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
    //while (triplet_to_del == 0 || triplet_to_del == 3)
    //  triplet_to_del = 3* ROUND(((getn_nodes()/3)-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
    
    setw(0,triplet_to_del ,triplet_to_del,0,-1);
    setregmiddle(0,triplet_to_del ,triplet_to_del,0,-1);
    setregslope(0,triplet_to_del ,triplet_to_del,0,-1);
    setv(triplet_to_del ,0,0,-1);
    setmiddle(triplet_to_del ,0,0,-1);
    setslope(triplet_to_del ,0,0,-1);
    setdeg(triplet_to_del ,0,0,-1);
    setbasal(triplet_to_del ,0,0,-1);
    /*
    //  setv_history(time_step_i,-2,selection_period, triplet_to_del);
    // empty v_history:
    setv_history(0,-2,selection_period, triplet_to_del);
    setv_history(0,1,selection_period,0);
    */
    setinput(triplet_to_del,-2,0,0);
    setmutation_counter(0,triplet_to_del,-1);
  }
}

void Cell::addmodule(Cell* module, double link_probability)
{

  int add_tripl;
 
  int old_n_nodes = getn_nodes();

  // add every triplet from module to this cell

  // do not copy last triplet (metabolic triplet)
  for (int tripl_i=0; tripl_i < ROUND( (module->getn_nodes()-1) /3); tripl_i++){
    //add_tripl = 3* ROUND(((getn_nodes()/3)-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));

    // add to the end of the triplet list:
    add_tripl = getn_nodes()-3;

    setw(0,add_tripl ,add_tripl,0,1);
    setregmiddle(0,add_tripl ,add_tripl,0,1);
    setregslope(0,add_tripl ,add_tripl,0,1);
    
    // adds new nodes, values set to zero
    setv(add_tripl,0,0,1);

    setmiddle(add_tripl ,0,0,1);
    setslope(add_tripl ,0,0,1);
    setdeg(add_tripl ,0,0,1);
    setbasal(add_tripl ,0,0,1);
  
   
    setinput(add_tripl, -1, global->param.general.n_signals, 0);

    //setv_history(select_index,2,selection_period,add_tripl); 
    setv_history(0,2,selection_period,add_tripl); 
    
    // adds new mutation counters and set them to 0
    setmutation_counter(0,add_tripl,1);
   }

  //----------------------------------------------------------------------
  // now copy alll data from the module (except the last triplet):
  int nodes_copy=module->getn_nodes()-3;

  for (int i=0; i<nodes_copy; i++){
    setmiddle(old_n_nodes+i, module->getmiddle(i), 0, 0);
    setslope(old_n_nodes+i, module->getslope(i), 0, 0);
    setdeg(old_n_nodes+i, module->getdeg(i), 0, 0);
    setbasal(old_n_nodes+i, module->getbasal(i), 0, 0);

    // setw with link_probability set to a random number
    for (int k=0; k<old_n_nodes; k++)
       if (link_probability>my_rand(global->param.globals.idum, global->param.globals.itemp))
	 //Power law distribution - neg. and pos.
	 setw(powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp), old_n_nodes+i, k, 0, 0);  //10% of weights will have 4 or above.
       else
	 setw(0, old_n_nodes+i, k, 0, 0);
    for (int j=0; j<nodes_copy; j++){
      for (int k=0; k<old_n_nodes; k++)
       if (link_probability>my_rand(global->param.globals.idum, global->param.globals.itemp))
	 //Power law distribution - neg. and pos.
	 setw(powerlaw(1.5, global->param.globals.idum, global->param.globals.itemp), old_n_nodes+i, k, 0, 0);  //10% of weights will have 4 or above.
       else
	 setw(0, k, old_n_nodes+j, 0, 0);
      setw(module->getw(i,j), old_n_nodes+i ,old_n_nodes+j, 0, 0);
    }

    // setregmiddle
    for (int k=0; k<old_n_nodes; k++)
      setregmiddle(10*my_rand(global->param.globals.idum, global->param.globals.itemp), old_n_nodes+i, k, 0, 0);
    for (int j=0; j<nodes_copy; j++){
      for (int k=0; k<old_n_nodes; k++)
	setregmiddle(10*my_rand(global->param.globals.idum, global->param.globals.itemp), k, old_n_nodes+j, 0, 0);
      setregmiddle(module->getregmiddle(i,j), old_n_nodes+i ,old_n_nodes+j, 0, 0);
    }

    // setregslope
    for (int k=0; k<old_n_nodes; k++)
      setregslope(1 + 3*my_rand(global->param.globals.idum, global->param.globals.itemp), old_n_nodes+i, k, 0, 0);
    for (int j=0; j<nodes_copy; j++){
      for (int k=0; k<old_n_nodes; k++)
	setregslope(1 + 3*my_rand(global->param.globals.idum, global->param.globals.itemp), k, old_n_nodes+j, 0, 0);
      setregslope(module->getregslope(i,j), old_n_nodes+i ,old_n_nodes+j, 0, 0);
    }

    // setinput
    setinput(old_n_nodes+i, module->getinput(i) , global->param.general.n_signals, 0);

  } // for i

  //----------------------------------------------------------------------
  // now, if requested, use wireing to old metabolic proten (last triplet in the module)

  if (global->param.lgt.if_coonect_metabolic){
 
    int metabolic_tripl = 3; // metabolictriplet -- nodes (3,4,5) -- second triplet
    
    int metabolic_start=module->getn_nodes()-3; 

    for (int i=0; i<nodes_copy; i++)
      for (int j=0; j<3; j++){
	setw(module->getw(i, metabolic_start+j), old_n_nodes+i ,metabolic_tripl+j, 0, 0);
	setw(module->getw(metabolic_start+j, i), metabolic_tripl+j, old_n_nodes+i, 0, 0);

	setregmiddle(module->getregmiddle(i, metabolic_start+j), old_n_nodes+i ,metabolic_tripl+j, 0, 0);
	setregmiddle(module->getregmiddle(metabolic_start+j, i), metabolic_tripl+j, old_n_nodes+i, 0, 0);

	setregslope(module->getregslope(i, metabolic_start+j), old_n_nodes+i ,metabolic_tripl+j, 0, 0);
	setregslope(module->getregslope(metabolic_start+j, i), metabolic_tripl+j, old_n_nodes+i, 0, 0);
      }
  }

}


//Merging the module with the cell network
void Cell::setmodule(Cell *module_ptr, int module_index, int selection_period, int mode)
{
	int old_node_size = sd.n_nodes;
	int new_node_size = sd.n_nodes  + module_ptr[module_index].getn_nodes();
	int index; 

	int *temp_int = new int [old_node_size]; //will be used to swap integer vector
	double *temp_double = new double [old_node_size]; // will be used to swap double vector

	//updating the number of nodes to its new value
	sd.n_nodes = new_node_size;
	
	//de- and re- allocating value vector
	for (index=0;index<old_node_size;index++)
		temp_int[index] = v[index];
	delete[] v;
	v = new int [new_node_size];
	for (index=0;index<old_node_size;index++)
		v[index]  = temp_int[index];

	for (index=old_node_size;index<new_node_size; index++)
		v[index] = 0; // with module_ptr[module_index].getv(index-old_node_size) I could get the molecule value from the module, but not realistic

	//de- and re- allocating signal association
	for (index=0;index<old_node_size;index++)
		temp_int[index] = input_sensor_protein[index];
	delete[] input_sensor_protein;
	input_sensor_protein = new int [new_node_size];
	for (index=0;index<old_node_size;index++)
		input_sensor_protein[index]  = temp_int[index];

	for (index=old_node_size;index<new_node_size; index++)
	  input_sensor_protein[index] = module_ptr[module_index].getinput(index-old_node_size);

	//de- and re- allocating mild mutation counter
	for (index=0;index<old_node_size;index++)
		temp_int[index] = mild_mutations[index];
	delete[] mild_mutations;
	mild_mutations = new int [new_node_size];
	for (index=0;index<old_node_size;index++)
		mild_mutations[index]  = temp_int[index];

	for (index=old_node_size;index<new_node_size; index++)
		mild_mutations[index] = 0;

	//de- and re- allocating strong mutation counter
	for (index=0;index<old_node_size;index++)
		temp_int[index] = strong_mutations[index];
	delete[] strong_mutations;
	strong_mutations = new int [new_node_size];
	for (index=0;index<old_node_size;index++)
		strong_mutations[index]  = temp_int[index];

	for (index=old_node_size;index<new_node_size; index++)
		strong_mutations[index] = 0;

	//de- and re- allocating main slope
	for (index=0;index<old_node_size;index++)
		temp_double[index] = slope[index];
	delete[] slope;
	slope = new double [new_node_size];
	for (index=0;index<old_node_size;index++)
		slope[index]  = temp_double[index];

	for (index=old_node_size;index<new_node_size; index++)
		slope[index] = module_ptr[module_index].getslope(index-old_node_size);

	//de- and re- allocating main middlepoint
	for (index=0;index<old_node_size;index++)
		temp_double[index] = middlepoint[index];
	delete[] middlepoint;
	middlepoint = new double [new_node_size];
	for (index=0;index<old_node_size;index++)
		middlepoint[index]  = temp_double[index];

	for (index=old_node_size;index<new_node_size; index++)
		middlepoint[index] = module_ptr[module_index].getmiddle(index-old_node_size);

	//de- and re- allocating basal expression vector
	for (index=0;index<old_node_size;index++)
		temp_double[index] = basal[index];
	delete[] basal;
	basal = new double [new_node_size];
	for (index=0;index<old_node_size;index++)
		basal[index]  = temp_double[index];

	for (index=old_node_size;index<new_node_size; index++)
		basal[index] = module_ptr[module_index].getbasal(index-old_node_size);

	//de- and re- allocating degradation vector
	for (index=0;index<old_node_size;index++)
		temp_double[index] = deg[index];
	delete[] deg;
	deg = new double [new_node_size];
	for (index=0;index<old_node_size;index++)
		deg[index]  = temp_double[index];

	for (index=old_node_size;index<new_node_size; index++)
		deg[index] = module_ptr[module_index].getdeg(index-old_node_size);

	delete[] temp_int;
	delete[] temp_double;


	//2D Vector DE- RE- allocation
	int index_col;
	double ** temp_2D = new double *[old_node_size];
	for (index=0;index<old_node_size;index++)
		temp_2D[index] = new double [old_node_size];

	//de- and re- allocation of weights
	//cout<<"\nInitial nodes: " << old_node_size <<" Module Nodes: " << module_ptr[module_index].getn_nodes() << " Final nodes:" << new_node_size;
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
			temp_2D[index][index_col] = w[index][index_col];
	}
	
	for (index=0;index<old_node_size;index++)
			delete[] w[index];
	delete[] w;
	w = new double *[new_node_size];
	for (index=0;index<new_node_size;index++)
		w[index] = new double [new_node_size];
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
			w[index][index_col] = temp_2D[index][index_col];
	}
	/*
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
			cout <<w[index][index_col] << " ";
	}*/

	for (index=old_node_size; index<new_node_size;index++)
	{
		for (index_col = old_node_size; index_col<new_node_size; index_col++)
		  w[index][index_col] = module_ptr[module_index].getw(index-old_node_size,index_col-old_node_size);
	}

	//filler for empty entries
	for (index=old_node_size; index<new_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
		{
			w[index][index_col] = 0;
			w[index_col][index] = 0;
		}
	}
	
	//de- and re- allocation of regslope
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
			temp_2D[index][index_col] = regslope[index][index_col];
	}
	
	for (index=0;index<old_node_size;index++)
			delete[] regslope[index];
	delete[] regslope;
	regslope = new double *[new_node_size];
	for (index=0;index<new_node_size;index++)
		regslope[index] = new double [new_node_size];
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
			regslope[index][index_col] = temp_2D[index][index_col];
	}
	for (index=old_node_size; index<new_node_size;index++)
	{
		for (index_col = old_node_size; index_col<new_node_size; index_col++)
		  regslope[index][index_col] = module_ptr[module_index].getregslope(index-old_node_size,index_col-old_node_size);
	}

	//filler for empty entries
	for (index=old_node_size; index<new_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
		{
			regslope[index][index_col] = 1 + 3*my_rand(global->param.globals.idum, global->param.globals.itemp);;
			regslope[index_col][index] = 1 + 3*my_rand(global->param.globals.idum, global->param.globals.itemp);;
		}
	}

	//de- and re- allocation of regmiddlepoint
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
			temp_2D[index][index_col] = regmiddle[index][index_col];
	}
	
	for (index=0;index<old_node_size;index++)
			delete[] regmiddle[index];
	delete[] regmiddle;
	regmiddle = new double *[new_node_size];
	for (index=0;index<new_node_size;index++)
		regmiddle[index] = new double [new_node_size];
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
			regmiddle[index][index_col] = temp_2D[index][index_col];
	}
	for (index=old_node_size; index<new_node_size;index++)
	{
		for (index_col = old_node_size; index_col<new_node_size; index_col++)
		  regmiddle[index][index_col] = module_ptr[module_index].getregmiddle(index-old_node_size,index_col-old_node_size);
	}

	//filler for empty entries
	for (index=old_node_size; index<new_node_size;index++)
	{
		for (index_col = 0; index_col<old_node_size; index_col++)
		{
			regmiddle[index][index_col] = 10*my_rand(global->param.globals.idum, global->param.globals.itemp);
			regmiddle[index_col][index] = 10*my_rand(global->param.globals.idum, global->param.globals.itemp);
		}
	}

	for (index=0;index<old_node_size;index++)
		delete[] temp_2D[index];
	delete[] temp_2D;



	//v_history needs a 2D temporary array of old_node_size x selection_period
	int **temp_int_2D = new int *[old_node_size];
	for (index=0;index<old_node_size;index++)
		temp_int_2D[index] = new int [selection_period];
	
	//de- and re- allocation of v_history
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<selection_period; index_col++)
			temp_int_2D[index][index_col] = v_history[index][index_col];
	}
	
	for (index=0;index<old_node_size;index++)
			delete[] v_history[index];
	delete[] v_history;
	
	v_history = new int *[new_node_size];
	for (index=0;index<new_node_size;index++)
		v_history[index] = new int [selection_period];
	
	for (index=0; index<old_node_size;index++)
	{
		for (index_col = 0; index_col<selection_period; index_col++)
			v_history[index][index_col] = temp_int_2D[index][index_col];
	}

	//Filler for the rest 
	for (index=old_node_size; index<new_node_size;index++)
	{
		for (index_col = 0; index_col<selection_period; index_col++)
			v_history[index][index_col] = -1;
	}

	for (index=0;index<old_node_size;index++)
		delete[] temp_int_2D[index];
	delete[] temp_int_2D;
	

}









void Cell::calculation_new_values(int select_index, double **signal_ptr)
{
  int j,k;
  double *reg2hyp;
  double regulation_prob, transcription_effectiveness;

  reg2hyp = new double [getn_nodes()];
  for (j=0;j<getn_nodes();j++) {
    reg2hyp[j] = 0;
    for (k=0;k<getn_nodes();k++) {
      if (getw(j,k) !=0)
	reg2hyp[j] =reg2hyp[j] + 0.1*getw(j,k) * 
	  (0.5 + 0.5*tanh(((double)getv(k) - getregmiddle(j,k))/getregslope(j,k)));
    }
    
    // careful with the arbitrary values like 4* etc.
    regulation_prob = getbasal(j) + (1 - getbasal(j))*(tanh((4*reg2hyp[j] - getmiddle(j))/getslope(j)));
    if (j %3 == 1 || j%3 ==2) {
      transcription_effectiveness = 0.5 + 0.5 * tanh(((double)getv(j-1) - 10)/4); //random again -10
      reg2hyp[j] = regulation_prob * transcription_effectiveness;
    }
    else 
      reg2hyp[j] = regulation_prob;
  }
  /* Special Case: Receptor modified protein prob. depends only on the input signal */
  for (j=0;j<getn_nodes();j=j+1) {
    for (k=1; k<=global->param.general.n_signals; k++) {
      if (getinput(j) == k)
	reg2hyp[j] = 0.5 + 0.5 * signal_ptr[k-1][select_index];
    }
  }

  // calculation of the new values

  // for Genes :					
  for (j=0;j<(getn_nodes()); j=j+3) {
    if (reg2hyp[j] > my_rand(global->param.globals.idum, global->param.globals.itemp) && getenergy() > 2 ) {
      setv(j, (getv(j) +1), 0 , 0);
      setenergy(-2,0);
    }
    
    // again random
    if ((1 - getdeg(j))*(0.5 - 0.5*tanh(((double)getv(j) - 50)/4)) < my_rand(global->param.globals.idum, global->param.globals.itemp) && getv(j)>0 ) {
      setv(j, (getv(j) -1), 0 , 0);

      if (getenergy() > 1)
	setenergy(-1,0);
    }
						
  }
						
  // For Proteins :					
  for (j=1;j<(getn_nodes()); j=j+3) {
    if (reg2hyp[j] > my_rand(global->param.globals.idum, global->param.globals.itemp) && getv(j-1)>0) {
      if (getenergy() > 2) {
	setv(j, (getv(j) +1), 0 , 0);
	setenergy(-2,0);
      }
    }
    
    // again random
    if ((1 - getdeg(j))*(0.5 - 0.5*tanh(((double)getv(j) - 50)/4)) < my_rand(global->param.globals.idum, global->param.globals.itemp) && getv(j)>0 )  {
      setv(j, (getv(j) -1), 0 , 0);
  
      if (getenergy() > 1)
	setenergy(-1,0);
    }
  }
						
  // for modified proteins
  for (j=2;j<(getn_nodes()); j=j+3) {
    if (reg2hyp[j] > my_rand(global->param.globals.idum, global->param.globals.itemp) && getv(j-1)>0) {
      if (j==5 && getenergy() > 10) {
	setv(j, (getv(j) +1), 0 , 0);
	setv(j-1, (getv(j-1) -1), 0 , 0);
	setenergy(-10,0);
      }
      else if (j!=5 && getenergy() > 2) {
	setv(j, (getv(j) +1), 0 , 0);
	setv(j-1, (getv(j-1) -1), 0 , 0);
	setenergy(-2,0);
      }
    }

    //10% chance of getting dephosphorilated - can be variable here
    else if (my_rand(global->param.globals.idum, global->param.globals.itemp) > dephosphorylation && getv(j)>0)   {
      setv(j-1, (getv(j-1) +1), 0 , 0);
      setv(j, (getv(j) -1), 0 , 0);
    }

    // again random
    if ((1 - getdeg(j))*(0.5 - 0.5*tanh(((double)getv(j) - 50)/4)) < my_rand(global->param.globals.idum, global->param.globals.itemp) && getv(j)>0 ) {
      setv(j, (getv(j) -1), 0 , 0);
      if (getenergy() > 1)
	setenergy(-1,0);
    }
  }
  delete[] reg2hyp;
}


//hmmm

int Cell::mutational_step(int select_index, int selection_period)
{
  int return_mutation_type = 0;

  int gene_nodecan, node;
  double creation_probability,destruction_probability, mild_mutation_probability, strong_mutation_probability;
  double random_number, random_temp;
  ofstream outfile;
  random_number = my_rand(global->param.globals.idum, global->param.globals.itemp);

  //calculating mutation probability -- wither it is pe genome or per cell
  if ( if_mutation_per_triplet==1 )
    node = getn_nodes()/3;
  else 
    node = 1;
  // mutation probabilities can go over 1.0 (!), but that is not a big deal
  creation_probability        = node*getmutation(1);
  destruction_probability     = node*getmutation(2);
  mild_mutation_probability   = node*getmutation(3);
  strong_mutation_probability = node*getmutation(4);

  // NORMOLIZE probabilities, if the sum of probabilities > 1.0
  double total_mut_probability = creation_probability + destruction_probability + mild_mutation_probability + strong_mutation_probability;
  if  (total_mut_probability > 1){
    creation_probability        /= total_mut_probability;
    destruction_probability     /= total_mut_probability;
    mild_mutation_probability   /= total_mut_probability;
    strong_mutation_probability /= total_mut_probability;
  }

  if (random_number < creation_probability){

    return_mutation_type = 1;

    gene_nodecan = 3* ROUND(((getn_nodes()/3)-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
    setw(0,gene_nodecan ,gene_nodecan,0,1);
    setregmiddle(0,gene_nodecan ,gene_nodecan,0,1);
    setregslope(0,gene_nodecan ,gene_nodecan,0,1);
    
    setv(gene_nodecan ,0,0,1);
    setmiddle(gene_nodecan ,0,0,1);
    setslope(gene_nodecan ,0,0,1);
    setdeg(gene_nodecan ,0,0,1);
    setbasal(gene_nodecan ,0,0,1);
    
    setinput(gene_nodecan, -1, global->param.general.n_signals, 0);
    setv_history(select_index,2,selection_period,gene_nodecan); 
    
    setmutation_counter(0,gene_nodecan,1);
  }
  else if (random_number < (creation_probability + destruction_probability) && getn_nodes() >6 ) {

    return_mutation_type = 2;

    gene_nodecan = 3* ROUND(((getn_nodes()/3)-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
    while (gene_nodecan == 0 || gene_nodecan == 3)
      gene_nodecan = 3* ROUND(((getn_nodes()/3)-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
    
    setw(0,gene_nodecan ,gene_nodecan,0,-1);
    setregmiddle(0,gene_nodecan ,gene_nodecan,0,-1);
    setregslope(0,gene_nodecan ,gene_nodecan,0,-1);
    setv(gene_nodecan ,0,0,-1);
    setmiddle(gene_nodecan ,0,0,-1);
    setslope(gene_nodecan ,0,0,-1);
    setdeg(gene_nodecan ,0,0,-1);
    setbasal(gene_nodecan ,0,0,-1);
    setv_history(select_index,-2,selection_period, gene_nodecan);
    setinput(gene_nodecan,-2,0,0);

    setmutation_counter(0,gene_nodecan,-1);
  }

  else if (random_number < creation_probability + destruction_probability + mild_mutation_probability)
    {
  
      return_mutation_type = 3;

      //first select the node that will be changed and then what feature.
      //The features are in the following groups (w,regmiddle,regslope),(middle,slope),(basal),(deg)
      gene_nodecan = ROUND((getn_nodes()-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
      random_temp = my_rand(global->param.globals.idum, global->param.globals.itemp);
      if (random_temp < 0.25) {
	setw(0,gene_nodecan ,gene_nodecan,0,2);
	setregmiddle(0,gene_nodecan ,gene_nodecan,0,2);
	setregslope(0,gene_nodecan ,gene_nodecan,0,2);
     }
       else if (random_temp < 0.5) {
	setmiddle(gene_nodecan ,0,0,2);
	setslope(gene_nodecan ,0,0,2);
      }
      else if (random_temp <0.75)
	setdeg(gene_nodecan ,0,0,2);
      else if (random_temp < 1)
	setbasal(gene_nodecan ,0,0,2);
		
      setmutation_counter(1,gene_nodecan,1);
      if (input_connection_mild > my_rand(global->param.globals.idum, global->param.globals.itemp) ) {
	//The options available are connection to any of the inputs OR the disconnection
	int j;
	double total =0;
	double *temp = new double [global->param.general.n_signals+1];
	temp[global->param.general.n_signals] = input_disconnection_probability; //prob to get disconnected as last index
	total = input_disconnection_probability;
	for (j=0; j<global->param.general.n_signals; j++) {
	  temp[j] = getconnectprob(j+1);
	  total = total +temp[j];
	}
	temp[0] = temp[0]/total;
	for (j=1; j<=global->param.general.n_signals; j++)
	  temp[j] = temp[j]/total + temp[j-1];
	
	total = my_rand(global->param.globals.idum, global->param.globals.itemp);
	for (j=0; j<=global->param.general.n_signals; j++) {
	  if (temp[j] > total)
	    {
	      setinput(gene_nodecan,j+1,0,0);
	      break;
	    }
	}
	//the j+1 is the 0 input;
	if (getinput(gene_nodecan) == global->param.general.n_signals +1)
				setinput(gene_nodecan,0,0,0);
	delete[] temp;
      }
    }

  else if (random_number < creation_probability + destruction_probability + mild_mutation_probability + strong_mutation_probability) {

    return_mutation_type = 4;

   //first select the node that will be changed and then what feature.
    //The features are in the following groups (w,regmiddle,regslope),(middle,slope),(basal),(deg)
    gene_nodecan = ROUND( (getn_nodes()-1) * my_rand(global->param.globals.idum, global->param.globals.itemp) );
    random_temp = my_rand(global->param.globals.idum, global->param.globals.itemp);
    if (random_temp < 0.25) {
      setw(0,gene_nodecan ,gene_nodecan,0,3);
      setregmiddle(0,gene_nodecan ,gene_nodecan,0,3);
      setregslope(0,gene_nodecan ,gene_nodecan,0,3);
    }
    else if (random_temp < 0.5) {
      setmiddle(gene_nodecan ,0,0,3);
      setslope(gene_nodecan ,0,0,3);
    }
    else if (random_temp <0.75)
      setdeg(gene_nodecan ,0,0,3);
    else if (random_temp < 1)
      setbasal(gene_nodecan ,0,0,3);

    setmutation_counter(2,gene_nodecan,1);
    if (input_connection_strong > my_rand(global->param.globals.idum, global->param.globals.itemp) ) {
      //The options available are a connection to any of the inputs OR the disconnection
      int j;
      double total=0;
      double *temp = new double [global->param.general.n_signals+1];
      temp[global->param.general.n_signals] = input_disconnection_probability; //prob to get disconnected as last index
      total = input_disconnection_probability;
      for (j=0; j<global->param.general.n_signals; j++) {
	temp[j] = getconnectprob(j+1);
	total = total +temp[j];
      }
      temp[0] = temp[0]/total;
      for (j=1; j<=global->param.general.n_signals; j++)
	temp[j] = temp[j]/total + temp[j-1];
      
      total = my_rand(global->param.globals.idum, global->param.globals.itemp);
      for (j=0; j<=global->param.general.n_signals; j++) {
	if (temp[j] > total) {
	  setinput(gene_nodecan,j+1,0,0);
	  break;
	}
      }

      //the j+1 is the 0 input;
      if (getinput(gene_nodecan) == global->param.general.n_signals +1)
	setinput(gene_nodecan,0,0,0);
      delete[] temp;
    }
  }

  return return_mutation_type;
}


void Cell::test_fitness(double** signals, double* food, int test_fitness_times, int ave_epochs, int selection_period,  double& F_ave_out, double& F_sd_out )
{

  Cell* cell=new Cell(global);
  cell->copy_from_another_cell(this);
  cell->setv_history(0,1,selection_period,0);

  // make a long food vector (over ave_epochs)
  double * food_long = new double[ave_epochs*selection_period];
  for (int ave_i=0; ave_i<ave_epochs; ave_i++){
    for (int ts_i=0; ts_i<selection_period; ts_i++){
      food_long[ave_i*selection_period+ts_i]=food[ts_i];
    }
  }
 
  // sckip THREE epochs
  for (int e_i = 0; e_i < 3; e_i++) {
    cell->setenergy(0,1);
    cell->setenergy(INT_MAX,0);
    for (int time_step_i = 0; time_step_i < selection_period; time_step_i++){
      //save v to v_history
      cell->setv_history(time_step_i,0,selection_period,0);
      cell->calculation_new_values(time_step_i, signals);
    }
  }

  // keep a log of mRP2 expression for ave_epochs
  int * vh_log = new int[ave_epochs*selection_period];
  // list of fitnessses over test_fitness_times repetitions
  double *fitness_set = new double [test_fitness_times];

  for (int atmpt_i=0; atmpt_i<test_fitness_times; atmpt_i++){
    cell->setenergy(0,1);
    cell->setenergy(INT_MAX,0);
    
    for (int ave_i = 0; ave_i < ave_epochs; ave_i++) {
      for (int ts_i = 0; ts_i < selection_period; ts_i++){
	// save v to v_history
	cell->setv_history(ts_i,0,selection_period,0);
	vh_log[ave_i*selection_period+ts_i] = cell->getv_history(5,ts_i);
	cell->calculation_new_values(ts_i, signals);
      }
    }

    fitness_set[atmpt_i] = pearson(vh_log, food_long, selection_period*ave_epochs);
  }

  //return
  F_ave_out = mean(fitness_set, test_fitness_times);
  F_sd_out = standart_deviation(fitness_set, ave_epochs, F_ave_out);

  delete [] vh_log;
  delete [] fitness_set;
  delete [] food_long;

  cell->setv_history(0,-1,selection_period,0); 
#if _IF_MPI
  cell->free_MPI_types();
#endif
  delete cell;

 
}



#if _USE_HDF5
int* Cell::convertMatrixToVector(int **matrix, int rows, int cols) {
int *matrix_as_vector = new int[rows * cols];
for (int i = 0; i < rows; i++)
	for (int j = 0; j < cols; j++)
		matrix_as_vector[i * cols + j] = matrix[i][j];
return matrix_as_vector;
}

int** Cell::convertVectorToMatrix(int *vector, int rows, int cols) {
int **vector_as_matrix = new int *[rows];
for (int i = 0; i < rows; i++) {
	vector_as_matrix[i] = new int[cols];
	for (int j = 0; j < cols; j++) {
		vector_as_matrix[i][j] = vector[i * cols + j];
	}
}
return vector_as_matrix;
}

double** Cell::convertVectorToMatrix(double *vector, int rows, int cols) {
double **vector_as_matrix = new double *[rows];
for (int i = 0; i < rows; i++) {
	vector_as_matrix[i] = new double[cols];
	for (int j = 0; j < cols; j++) {
		vector_as_matrix[i][j] = vector[i * cols + j];
	}
}
return vector_as_matrix;
}

double* Cell::convertMatrixToVector(double **matrix, int rows, int cols) {
double *matrix_as_vector = new double[rows * cols];
for (int i = 0; i < rows; i++)
	for (int j = 0; j < cols; j++)
		matrix_as_vector[i * cols + j] = matrix[i][j];
return matrix_as_vector;
}


#endif
