// ********************************************
// * CELL CLASS FOR EVE                       *
// ********************************************


#ifndef _CELL_H_
#define _CELL_H_

#define MAX_NODES 30			

#include "generic_header.h"
#include "eve_parameters.h"

#include "ilibrary.h"

#if _USE_HDF5
	#include "hdf5_constants.h"
#endif

#if _IF_MPI
        #include "mpi.h"
#endif


class Cell
{
 public:

  Cell(Eve_parameters* _global, int _ID=0, double _fitness=0, int random_flag=1);
  ~Cell(); 


  void free_MPI_types();

  /*! \name Cell related member functions */
  //@{ 
  int getID() const; ///< returns \ref ID if the cell

  int getn_nodes() const; //!< returns \ref n_nodes -- number of nodes in the cell

  int getn_links() const; //!< returns number of links in the cell (non-zero elements in w matrix)

  //! returns \ref mild_mutations [nodeID] (for mutation_type=1) or \ref strong_mutations [NodeID] (for mutation_type=2)
  int getmutation_counter(int mutation_type, int nodeID) const; 

  double getfitness() const;//!< returns \ref fitness

  double getenergy() const; //!< returns \ref energy

  double getsparcity() const; //!< returns \ref sparcity
 
  /*!
   * \brief returns one of the mutation parameters (see more)
   *
   * @param selector values:
   *        \li 1 creation_cell
   *        \li 2 destruction_cell
   *        \li 3 mild_mutation_cell
   *        \li 4 strong_mutation_cell
   *        \li 5 evolvability_cell
   */
  double getmutation(int selector) const; 

  /*!
   * \brief (see details on parameters) sets mutation counters: \ref mild_mutations and \ref strong_mutations vectors. 
   *        ACHTUNG! \ref n_nodes are assumed to be updated VM: mixed deletion/creation of nodes with additions to the counter??
	if (selector ==1)
		return creation_cell;
	else if (selector ==2)
		return destruction_cell;
	else if (selector ==3)
		return mild_mutation_cell;
	else if (selector ==4)
		return strong_mutation_cell;
	else if (selector ==5)
		return evolvability_cell;
	else 
		return 0;
  */
  int getinput(int i) const;

  //! returns a coordinate of this cell in 2D grid
  int getcoord() const;

  //! returns a lable of this cell
  int getlabel() const;

  /*!
   * \brief copies "other" cell from  into this Cell, 
   *         assigns next_new_cell_ID as ID 
   *	     ACHTUNG: v_history is not included (!) in the copy and should be handled outside.
   *
   * 2DO: eliminate global variable selection_period
   * 2DO: v_history -- copy/initialize??
   *
   * Everything is coppied (even ID), except: v_history, mild_ and strong_ mutations
   *
   * @param other source  \ref Cell
   */
  void copy_from_another_cell(const Cell *other);
   
  /*!
    * \brief send a copy of this Cell to MPI process "target"
    *
    *  Cell::MPI_Send_cell_copy()/Cell::MPI_Recv_cell_copy pair is same 
    *  as \ref Cell::copy_from_another_cell(), but copies  cell 
    *  from MPI process "destination" to "target"
    */
  void MPI_Send_cell_copy(int target);

  /*!
   * \brief recieve a copy of Cell from MPI process "destination"
   *
   *  Cell::MPI_Send_cell_copy()/Cell::MPI_Recv_cell_copy pair is same 
   *  as \ref Cell::copy_from_another_cell(), but copies  cell 
   *  from MPI process "destination" to "target"
   */
  void MPI_Recv_cell_copy(int source);







  //  void setID(int _processID, int _time_step); //!< sets \ref ID.processID and ID.time_step
  void setID(int _ID); //!< sets \ref ID

  void setn_nodes(int _n_nodes); //!< sets \ref n_nodes

  void setfitness(double _fitness); //!< sets \ref fitness

  //! adds "add_energy" to \ref energy; "if_reset"==1 sets \ref energy to 0;
  void setenergy(double add_energy, int if_reset); 

  void setsparcity(double _sparcity); //!< sets \ref sparcity

  //! sets \ref creation_cell, \ref destruction_cell, \ref mild_mutation_cell, \ref  strong_mutation_cell, \ref strong_mutation_cell, and \ref evolvability_cell
  void setmutation(double _creation_cell, double _destruction_cell, double _mild_mutation_cell, double _strong_mutation_cell, double _evolvability_cell);

  //! sets \ref coord to _coord
  void setcoord(int _coord);

  
  //! sets \ref sd.label to _label
  void setlabel(int _label);

  /*!
   * \brief XXX  
   *
   * @param which_node
   * @param which_input
   * @param n_signals number of signals in the enviroment
   * @param global_probability
   */
  void setinput(int which_node, int which_input, int n_signals,  double global_probability);


  /*!
   * \brief (see details on parameters) sets mutation counters: \ref mild_mutations and \ref strong_mutations vectors. 
   *        ACHTUNG! \ref n_nodes are assumed to be updated VM: mixed deletion/creation of nodes with additions to the counter??
   *
   * @param mutation_type type of mutation 
   *   \li -2 delete \ref mild_mutations and \ref strong_mutations vectors 
   *   \li -1 allocate \ref mild_mutations and \ref strong_mutations vectors(size=\ref n_nodes) and set them to zero 
   *   \li  0 add or delete one node at the "position" (determined by "value_to_add")
   *   \li +1 mild_mutations[position] += value_to_add;
   *   \li +2 strong_mutations[position] += value_to_add;
   * @param position node's number to modify the counter 
   * @param value_to_add -- "-1" node deletion; "+1" node addition
   */
  void setmutation_counter(int mutation_type, int position, int value_to_add);

  //@} 


  /*! \name Node related member functions */
  //@{ 
  int getv(int position) const; //!< returns \ref v [position]
 
  double getmiddle(int position) const; //!< returns \ref middlepoint [position]
  
  double getslope(int position) const; //!< returns \ref slope [position]
  
  double getdeg(int position) const; //!< returns \ref deg [position]
  
  double getbasal(int position) const; //!< returns \ref basal [position]
  
  double getconnectprob(int input) const; //!returns \f$ exp(-NumberOf[input.sensor.protein[i]==input] ) \f$
  
  double getw(const int i, const int j) const; //!< returns \ref w
  
  double getregmiddle(const int i, const int j) const; //!< returns \ref regmiddle 
  
  double getregslope(const int i, const int j) const; //!< returns \ref regslope
  
  int getv_history(const int i, const int j) const; //!< returns \ref v_history[i][j]
  int** getv_history_pointer() const; //!< returns a pointer to \ref v_history 

	


  /*! \brief updates v (see more)
   *
   * 2DO -- remove change=5 option
   * 2DO -- random_flag --> anothe change value?
   * 2DO -- add debug check -- change out of range;
   *
   * @param position 
   * @param a (used in change ==0, and change ==5)
   * @param random_flag if ==1, than  v[i] = ROUND(my_rand()*30); EVERYTHING ELSE IS IGNORED! (move to the change value??)
   * @param change values (note: no 2 or 3):
   *      \li -1 -- delete a triplet (three nodes) from v starting "position"
   *      \li  0 -- v[position] = a; 
   *      \li  1 -- adds a triplet (three nodes) to the end of v; sets three new values to 0;
   *      \li  4 -- delete v[ ];  v = new int [n_nodes];
   *      \li  5 -- v[position] = a; REMOVE THIS OPTION
   */
  void setv(int position, int a, int random_flag, int change);
  


  /*! \brief creates, deletes, and updates v_history (see more)
   *
   * 2DO -- add DEBUG check of initialize values
   *
   * @param which_col used in initialize ==-2, ==2, =="other"
   * @param initialize values:
   *        \li -2 -- deletion of a node; which_col will indicate the triplet
   *        \li -1 -- delete v_history[ ][ ] 
   *        \li  1 -- v_history = new int* [n_nodes] x [selection_period];    v_history[i][j] = -1;
   *        \li  2 -- addition of a node
   *        \li  "other" -- v_history[i][which_col] = v[i]
   * @param selection_period -- one of two dimentions of v_history, used whenever v_history is reallocated
   * @param which_row used in initialize ==-2
   */
  void setv_history(int which_col, int initialize, int selection_period, int which_row);


  /*! \brief vm
   *
   *
   * vm: more
   */
  void setmiddle(int position, double a, int random_flag, int change);
  
  /*! \brief vm
   *
   *
   * vm: more
   */
  void setslope(int position, double a, int random_flag, int change);

  /*! \brief vm
   *
   *
   * vm: more
   */
  void setdeg(int, double, int, int);

  /*! \brief vm
   *
   *
   * vm: more
   */
  void setbasal(int, double, int, int);

  /*! \brief sets and changes w matric (connectivity matrix)
   *
   *
   * @param position_col
   * @param random_flag
   * @param change possible values:
   *        \li  0 set specific value to specific link w[position_row][position_col] = value;
   *        \li  1 duplicate a triplet at position_row & position_col -- first row & column (of three) which 
   *               describe the triplet (both numbers should be be equal for a proper triplet description)
   *        \li -1 deletion/destruction of the triplet at position_row & position_col
   *        \li  2 mild weight change (for every weight in position_row & position_col *=(1 + 0.2*gaussian(0,1)) )
   *        \li  3 strong weight change
   *        \li
   * vm: more
   */
  void setw(double value, int position_row, int position_col, int random_flag, int change);

  /*! \brief vm
   *
   *
   * vm: more
   */
  void setregmiddle(double, int, int, int, int);

  /*! \brief vm
   *
   *
   * vm: more
   */
  void setregslope(double value, int position_row, int position_col, int random_flag, int change);





  /*!
   * \brief appends log file with current cell info
   *
   * Used to save in either HDF5 or plain ASCII, depending on _USE_HDF5 flag
   * @param output is void*, will be cast to either hid_t or string depending on _USE_HDF5 flag
   * @param local_cell_number cell number in the array of local cells in this mpi process
   * @param epochID current epoch number
   */
#if _USE_HDF5
  void save(int local_cell_number, int epochID, hid_t &outfile);
#else
  void save(int local_cell_number, int epochID, std::string& out_str){
    this->save_ascii(local_cell_number, epochID, out_str);
  }
#endif

  void save_ascii(int local_cell_number, int epochID, std::string& out_str);

  /*!
   * \brief appends log file with current v_history
   *
   * @param out_data is printed into this stringstream
   * @param local_cell_number cell number in the array of local cells in this mpi process
   * @param epochID current epoch number
   */
  void save_v_history_ascii(std::string& out_str, int local_cell_number, int epochID);

  /*!
   * \brief saves w weight matrix in graphML format
   *
   * @param out_data is printed into this stringstream
   */
  void save_w_graphml(std::string& out_str);

#if _USE_HDF5
  /*!
     * \brief fixes given matrix to work with single, continuous allocation format
     *
     * @param matrix is input matrix
     * @param x_dim is number of columns in matrix
     * @param y_dim is number of rows in matrix
     */
  int* convertMatrixToVector(int **matrix, int x_dim, int y_dim);

  /*!
     * \brief fixes given matrix to work with single, continuous allocation format
     *
     * @param matrix is input matrix
     * @param x_dim is number of columns in matrix
     * @param y_dim is number of rows in matrix
     */
  double* convertMatrixToVector(double **matrix, int x_dim, int y_dim);

  double** convertVectorToMatrix(double *vector, int rows, int cols);

  int** convertVectorToMatrix(int *vector, int rows, int cols);
#endif


  /*!
   * \brief loads a cell from the file
   *
   * @param file_name file with the cell info to load
   * @param cell_number which cell to load from file (use 1 if only one cell in the output)
   * @param out_data output messages are printed into this stringstream
   * @param epochID current epoch number
   * @param next_cell_ID for the newly loaded cell
   */
#if _USE_HDF5
  void load(hid_t &infile, int epochID, int proc_to_load, int cell_to_load, int next_cell_ID);
#else
  void load(std::string file_name, std::string& out_str, int epochID, int next_cell_ID){
    int cell_number = 1;
    this->load_ascii(file_name, cell_number, out_str, epochID, next_cell_ID);
  }

#endif
  void load_ascii(std::string file_name, int cell_number, std::string& out_str, int epochID, int next_cell_ID);

  /*! \brief vm
   *
   *
   * vm: more
   */
  void output_information(const char * filename,int selection_period, int full_version);


  /*! \brief reduces this cell to a random module (subnetwork) withe module_size triplets
   *         Always keep metabolic protein as a node module_size+1 
   *
   * @param module_size number of triplets in the module to return
   */
  void reduce_to_module(int module_size);

  /*! \brief merges module with the cell network
   *
   * @param module to incorporate into the cell network
   */
  void addmodule(Cell* module, double link_probability);


  // WILL BE DELETED
  void setmodule(Cell *module_ptr, int module_index, int selection_period, int mode);

  //@}



  /*! \brief vm
   *
   *
   * @param cell
   * @param select_index
   * @param signal_ptr
   */
  void calculation_new_values(int select_index, double **signal_ptr);

  /*! \brief vm
   *
   *
   * @return 0 - no mutation; 1 - creation; 2 - destruction; 3 - mild mutation; 4 - strong mutation;
   */
  int mutational_step(int select_index, int selection_period);

  

 /*! \brief test fitness of this sell over ave_apochs (SLOW FUNCTION)
   *
   * @param signals pointer to the environmental signals
   * @param food pointer to the food signal
   * @param test_fitness_times averages fitness over test_fitness_times runs, each ave_epochs long
   * @param ave_epochs calculates fitness over ave_epochs
   * @param selection_period
   * @param F_ave_out RETURNS average fitness over test_fitness_times runs
   * @param F_sd_out RETURNS StDev for fitness over test_fitness_times runs
   * 
   */
  void test_fitness(double** signals, double* food, int test_fitness_times, int ave_epochs, int selection_period,  double& F_ave_out, double& F_sd_out );




 private:

  //! a static pointer to an instance of singletone classs Eve_parameters, where all global variables are stored
  Eve_parameters* global;

  //! max nodes in the graph; per cell (i.e. max cell size);

  /*
  //! number of nodes in the cell;  vm?: should be at least two triplets? always 3N?
  int n_nodes;


  int ID; //!< ID number of the cell
  
  double fitness;

  double energy; 

  double sparcity; 

  double creation_cell; 

  double destruction_cell; 

  double mild_mutation_cell; 

  double strong_mutation_cell;

  double evolvability_cell; 
  */

  /*! \brief the cell's static data
   *
   * Variables:
   *   \li n_nodes -- number of nodes in the graph (cell)
   *   \li ID -- ID of the cell, sequential for the length of the experiment
   *   \li fitness --
   *   \li energy --
   *   \li sparcity -- random number [0.01 .. 0.21] defines the sparcity for initial random cells
   *   \li creation_cell -- probability of creation
   *   \li destruction_cell -- probability of destruction
   *   \li mild_mutation_cell -- probability of a mild mutation
   *   \li strong_mutation_cell -- probability of a strong mutation
   *   \li evolvability_cell -- 
   */
  struct cell_static_info_t{
    int n_nodes;
    int ID;
    double fitness;
    double energy;
    double sparcity;
    double creation_cell; //probability of triplet duplication
    double destruction_cell;  //probability of triplet delition 
    double mild_mutation_cell;  //probability mild mutation
    double strong_mutation_cell;  //probability stong mutation
    double evolvability_cell; // rate of mutation probability change
    //! cordinate of the cell in 2D grid (x,y) is transformed to one int as: coord = max_x * x + y
    int coord;
    //! generic numeric label
    int label;
  };

  //! structure for now; static data (sd)
  cell_static_info_t sd;

  //! MPI data type for all static varibales in the Cell class
#if _IF_MPI  
  MPI_Datatype cell_static_data_tMPI;
#endif

  /*! \brief vm
   *
   *
   * node values
   */
  int *v; 

  /*! \brief vm
   *
   *
   * connection to signals {0,1,2}
   */
  int *input_sensor_protein; 

  /*! \brief vm
   *
   *
   * history of v for each timestep within an epoch
   */
  int **v_history;



   /*! \brief vm
   *
   *
   * log of mild mutations
   */
  int *mild_mutations; 

  /*! \brief vm
   *
   *
   * log of strong mutations
   */
  int *strong_mutations; 



  /*! \brief vm
   *
   *
   * slope of a sigmode per node
   */
  double *slope;

  /*! \brief vm
   *
   *
   * mid.point of a sigmode per node
   */
  double *middlepoint; 

  /*! \brief vm
   *
   *
   * basal level of expression
   */
  double *basal;

  /*! \brief vm
   *
   *
   * degradation probability of the node value
   */
  double *deg; 

  /*! \brief wight matrix for the regulatary network
   *
   *  Weight matrix is nonsymmetrical, directional; nodes can be selfregulated. Positive values correspond to activation,
   *  negative values are inhibition. Wij and Wji are regulation of i-th node by j-th node and vice versa.
   */
  double **w; 

   /*! \brief vm
   *
   *
   * sigmoid slope for each link
   */
  double **regslope; 

   /*! \brief vm
   *
   *
   * sigmoid mid point for each link
   */
  double **regmiddle; 



};

#endif







