// ********************************************
// * EVE v2.1                                 *
// *                                          *
// * Vadim Mozhayskiy and Ilias Tagkopoulos   *
// *                                          *
// * All rights reserved (C) 2007-2012        *
// ********************************************

//////////////////////////////////////////////////////////////////////
// Execute format: 
// ./eve work_folder_name {input_fime_name {experiment#  {seed[unsigned long]}}} 
//////////////////////////////////////////////////////////////////////

#include "generic_header.h"

#include "my_mpi_types.h"


///////////////////////////////////////////////////////////////////////////////
// GLOBAL PARAMTERS
///////////////////////////////////////////////////////////////////////////////

// use one of the Cell classes:
#include "cell.h"

//! print time/growth rate benchmark if 1
#define if_print_benchmark 0 

#include <sys/time.h>

//! save v_history log if 1
#define if_save_v_history 0 

//! save detailed log of LGT events if 1
#define if_save_lgt_events_log 0

//! save detailed log of mutation events if 1
#define if_save_mutation_events_log 0

//! two halfs of population start in opposite coreners if 1
#define if_fight_mode 0

//! if 1, then all LGT events are from the evolved network only (stored in input_evolved_cell.in file in the work folder)
#define if_HGT_from_evolved_cells 0

//! print division history log
#define if_phylogenetic_print 1

//! write to outputs if 1
#define if_save_output 1

//! if print labels of all cells in the end of each epoch
#define IF_PRINT_LABELS 0

//! module size for transformation is generated between 0..MAX_MODULE_TRANSFORMATION, than probability is applied
#define MAX_MODULE_TRANSFORMATION 10
//! maximum size of the module tried (for probability density) -- can be large
#define MAX_MODULE 20

//! temporary plug
#define MAX_LGT_EVENTS_PER_TIMESTEP 1


//! temporary plug
#define FIRST_EPOCH_ID -1
#define FILE_NAME_RANK_SUFFIX "__rank"
#define FILE_NAME_PREFIX "cell_info"
#define FILE_NAME_LAST_PREFIX "last_"
#define TEXT_FILE_EXTENSION ".data"



//////////////////////////////////////////////////////////////////////
// default values:

//! syncronization frequency [units=time steps] (cells allowed to divide only at syncronization points)
int sync_freq=1;
  
//! save info of all cells every save_cell_info_freq epochs [units=epochs]; default 1
int save_cell_info_freq = 1;
  
//! above this energy cell tries to divide; default: 1,600,000
int cell_division_threshold = 1600000; // must be >initial_energy, so that new born cell do not try to divide in the same time step...

//! below this, cell dies and is replaced with the _default_ cell; default: 1 (i.e. if energy =0 cell dies)
int cell_death_threshold = 1;  
   
//! number of time points in one epoch (generation); default: 4500
int selection_period = 4500; 

//! overal number of time points requested = selection_period*number_of_epochs; default: 18,000,000=4000*4500;
int number_of_epochs = 4000;

//! number of cells per node; default =1
int n_cells_per_process=1;

///////////////////////////////////////////////////////////////////////////////


#include <vector>

//#include <omp.h>
#include <time.h>

#include "import_export.h"
#include "value_calculation.h"
#include "ilibrary.h"


type signal_type = PRESET_GATE;

//----------------------------------------------------------------------

//! if 1 then, regulator node is selected according to the intraconnectivity matrix, first regulated type is picked according the probability, then clustered (1-exp(-n) correection)
int connection_specificity_flag=1;

//! if "1": "intraconnectivity" matrix is used; otherwise equal connection probability between any two nodes
int intraconnectivity_flag=1;


//! [i][j] indeces are in general:  j-source(regulator, column's index), i-target (regulated, row's index); 

//! intraconnectivity matrix; [i][j]; i,j in {0=mRNA, 1=prot., 2=mod.prot.}; j-source(regulator, column's index), i-target (regulated, row's index); 
//! SUM withing each column must ne "1"; at least one non-zero entry;

// equal destribution:
//double intraconnectivity[3][3] = {0.33,0.33,0.33, 0.33,0.33,0.33, 0.33,0.33,0.33};

// no regulation by mRNA
double intraconnectivity[3][3] = { 0.00,0.33,0.33, 0.00,0.33,0.33, 0.00,0.33,0.33};

// no regulation by mRNA + lower probability of protein regulating (other) protein translation
// double intraconnectivity[3][3] = {0.00,0.40,0.40, 0.00,0.20,0.20, 0.00,0.40,0.40};



// calculated probability of picking one of the nodes within a triplet as a regulator (normolize sum over columns of intraconnectivity)
double intraconnectivity_regulator[3]; // DEFINED in the main() code
// calculated probability of picking one of the nodes within a triplet as a regulator (normolize sum over columns of intraconnectivity)
double intraconnectivity_regulated[3]; // DEFINED in the main() code

//----------------------------------------------------------------------


//! if 1, number of mutations is proportional to the number of nodes
int if_mutation_per_triplet = 0;
// cost~1/mutation rate
int node_cost_proportional_to_mutation = 0;

//----------------------------------------------------------------------

// vm: during mutation -- prob to connect to a signal (signal instead of input) 
// vm: tree(double linked list) for the history of evolution
double input_connection_mild =   0.5;
    
double input_connection_strong = 0.8;
    
double input_disconnection_probability = 0.3;
    
// vm prob modified-->nonmodified by chance
double dephosphorylation =0.1;
    
// signals -- function of the signal -- varies in time/shape
int random_appearance =0, random_duration = 0; 

// check if nasty (wheither loop per cell or per node)-- if mutation proportional 
// to the rate only or also to the size (should be one) 
int mutation_flag = 0;

// ?
int evolvability_energy_cor = 0;
//
int maintenance_cost=0;
//
int node_cost=0;
//
int food_per_metabolic=0;


// import a cell from [work_folder_name]/input_cell.in
int import_cells = 0;

// import a cell from [work_folder_name]/input_signals.in
int import_signals = 0;


//////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////


// used to exit the program cleanly
void finalize() {

#if _IF_MPI
  // syncronize all processes so that output is finished (?)
  MPI_Barrier(MPI_COMM_WORLD);
  // and it is over
  MPI_Finalize();
#endif

#if _USE_HDF5
  H5close();
#endif
}



#if _USE_HDF5
// detects number of processes and cells per process from last_cell_info.h5 file
// and determines a specific cell for the load() method of Cell
int* determineCellToLoad(hid_t &infile, int rank, int next_cell_ID, int cur_nprocs, int cur_ncells_per_proc) {
  char epoch_name[MAX_H5_OBJECT_NAME_LENGTH], cell_name[MAX_H5_OBJECT_NAME_LENGTH];
  sprintf(epoch_name, "%s%d", H5_GROUP_EPOCH, FIRST_EPOCH_ID);
  //hid_t lapl = H5Pcreate(H5P_LINK_ACCESS);

  // ensure the last_cell_info.h5 file has data before attempting to read data
  if (!H5Lexists(infile, epoch_name, H5P_DEFAULT))
    {
      printf("Error loading first epoch from last_cell_info.h5. Exitting...\n");
      finalize();
      exit(1);
    }

  hid_t epoch_g = H5Gopen(infile, epoch_name, H5P_DEFAULT);
  int proc_count = 0, cell_count = 0;

  // count how many processes and cells per process were actually saved from the last run
  for(sprintf(cell_name, "%s%s%s%s%d[%d]", H5_GROUP_ROOT, epoch_name, H5_GROUP_ROOT, H5_GROUP_CELL, proc_count, cell_count); H5Lexists(epoch_g, cell_name, H5P_DEFAULT); sprintf(cell_name, "%s%s%s%s%d[%d]", H5_GROUP_ROOT, epoch_name, H5_GROUP_ROOT, H5_GROUP_CELL, ++proc_count, cell_count));
  for(sprintf(cell_name, "%s%s%s%s%d[%d]", H5_GROUP_ROOT, epoch_name, H5_GROUP_ROOT, H5_GROUP_CELL, proc_count - 1, cell_count); H5Lexists(epoch_g, cell_name, H5P_DEFAULT); sprintf(cell_name, "%s%s%s%s%d[%d]", H5_GROUP_ROOT, epoch_name, H5_GROUP_ROOT, H5_GROUP_CELL, proc_count - 1, ++cell_count));
  H5Gclose(epoch_g);

  printf("Rank %d, cell %d loading ", rank, next_cell_ID);
  int *results = new int[2];
  // determine whether to load cell from equivalent rank/cell or a random rank/cell
  if(proc_count == cur_nprocs && cell_count == cur_ncells_per_proc)
    {
      printf("equivalent cell from last run.\n");
      results[0] = rank;
      results[1] = next_cell_ID;
    }
  else
    {
      srand(time(NULL) + 3141592L * (rank + 1));
      results[0] = rand() % proc_count;

      srand(time(NULL) - next_cell_ID);
      results[1] = rand() % cell_count;
      printf("Cell%d[%d] from last run.\n", results[0], results[1]);
    }
  return results;
}
#endif





int main(int argc, char *argv[])
{

  //! number of processes (cores?); default: single process
  int n_processes = 1; 

  //! rank of the master process (only the master process logs some debug info)
  int master_rank = 0; 

  //! rank of the current process -- default is only one process (master)
  int rank = master_rank;

  // get number of requested processes and the rank of the current process:
#if _IF_MPI
  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD,&n_processes); 
  MPI_Comm_rank(MPI_COMM_WORLD,&rank); 
#endif
  
  if (argc<2){
    if (rank==0)
      std::cout << "\n\nExecute format: eve work_folder_name {input_fime_name {experiment#  {seed[unsigned long]}}}\n ***EXITING***\n\n " << std::flush;
    finalize();
    exit(1);
  }

  // create an instance of the global parameters singletone (Eve_parameters* global):
  create_global_eve_parameters();

  // output data buffered in this streams
  std::stringstream out_data;
  std::string out_str;

  //! work folder name
  std::string work_folder_name;
  work_folder_name = argv[1];
  work_folder_name+= "/";

#if if_save_output    
  // main output file 
  std::stringstream output_fname;
  ofstream output_file;

  if (rank==master_rank){
    output_fname << work_folder_name << "output.data";
    output_file.open(output_fname.str().c_str(), std::ios::out | std::ios::trunc);

    out_data.str("");
    out_data << "Work folder: " << work_folder_name << "\n";
    output_file << out_data.str() << std::flush;
  }

  // global statistics data (average energy, max.fitness)
  std::stringstream statistics_fname;
  ofstream statistics_file;

  // 2D density of cells at the end of epoch
  std::stringstream cell_density_fname;
  ofstream cell_density_file;
#endif 

  int experiment_number = 1;
  if (argc>3)
    experiment_number = atoi(argv[3]);

  unsigned long base_seed;
  if (argc>4){
    base_seed=strtoul(argv[4],NULL,0);
  }
  else {
    time_t time_now = time(NULL);
    base_seed = (unsigned long)time_now;
#if _IF_MPI
    // propagate base seed to all MPI processosors 
    MPI_Bcast(&base_seed, 1, MPI_UNSIGNED_LONG, 0, MPI_COMM_WORLD);
#endif
  }
  my_srand( base_seed + 3141592L*(rank+1)*experiment_number, global->param.globals.idum );

#if if_save_output    
  if (rank==master_rank){
    out_data.str("");
    out_data << "----------------------------------------------------------------------\n"
	     << "Random generator base seed = " << base_seed << "\n"
	     << "Experiment number = " << experiment_number << "\n"
	     << "Total number of processes = " << n_processes << "\n"
	     << "----------------------------------------------------------------------\n";
    output_file << out_data.str() << std::flush;
  }
#endif 

  //----------------------------------------------------------------------
  // calculated probability of picking one of the nodes within a triplet as a regulator (normolize sum over columns of intraconnectivity)
  // intraconnectivity_regulator[3]:
  double norm_tmp = 0;
  for (int i=0; i<3; i++){
    intraconnectivity_regulator[i]=0;
    for (int j=0; j<3; j++)
      intraconnectivity_regulator[i]+=intraconnectivity[j][i];
    norm_tmp += intraconnectivity_regulator[i];
  }
  for (int i=0; i<3; i++)
    intraconnectivity_regulator[i] /= norm_tmp;

  // calculated probability of picking one of the nodes within a triplet as a regulated node (normolize sum over rows of intraconnectivity)
  // intraconnectivity_regulator[3]:
  norm_tmp = 0;
  for (int i=0; i<3; i++){
    intraconnectivity_regulated[i]=0;
    for (int j=0; j<3; j++)
      intraconnectivity_regulated[i]+=intraconnectivity[i][j];
    norm_tmp += intraconnectivity_regulated[i];
  }
  for (int i=0; i<3; i++)
    intraconnectivity_regulated[i] /= norm_tmp;
  //----------------------------------------------------------------------
  

  //Variable mutation rates disabled
  mutation_flag = 0;
 

  // Anticorrelation between energy and mutation rate change is disabled
  evolvability_energy_cor = 0; 
    
  // Anticorrelation between relative mutation rate (irt envir. mutation rate)
  // and node cost is enabled
  node_cost_proportional_to_mutation = 1; 
    
  //cost of metabolic protein
  maintenance_cost = 10;			
    
  //cost of maintaining the nodes;
  node_cost = 1;					
    
  //metabolic gain
  food_per_metabolic= 50;			
    
  /*
    if (random_duration ==1 && signal_type==VARIANCE_LOCK) 
    //an extra signal is needed in 'VARIANCE_LOCK' experiment type
    number_of_inputs =3;		
    else 
    //all other experiments need two signals
    number_of_inputs =2;		
  */

  //Signal and food appearance is stochastic
  random_appearance =	0; 
    
  //Signal and food duration is stochastic
  random_duration = 0; 		

   


  //----------------------------------------------------------------------
  // output files (master_rank output)

#if if_save_output
  if (rank==master_rank)
    {
      // global staistics file:
      statistics_fname << work_folder_name << "output_statistics.data";
      statistics_file.open(statistics_fname.str().c_str(), std::ios::out | std::ios::trunc);
      statistics_file << " epoch   ave. energy   ave.fitnes    max.fitness max.f.cell  mm   sm  ave.nodes  ave.m.rate max.m.rate max.m.r.cell m.t.of.max.F.cell\n-------------------------------------------------------------------------------\n";

      // file with 2d density of cells in the end of each epoch
      cell_density_fname << work_folder_name << "cell_density.data";
      cell_density_file.open(cell_density_fname.str().c_str(), std::ios::out | std::ios::trunc);


    }
#endif
  //----------------------------------------------------------------------


  // load the input file (second optional argument)
  std::stringstream input_fname;
  if (argc>2){
    input_fname << argv[2];
    load_input( input_fname.str(), global, out_str);
#if if_save_output    
    if(rank==master_rank)
      output_file << out_str << std::flush;
#endif
  }



  // number of cells in the experiment
  int n_cells = n_processes*n_cells_per_process;



#if if_save_output    
  if(rank==master_rank){
    // save all parameters in "input format"
    save_input(global, out_str);
    output_file << out_str << std::flush;

    out_data.str("");
    out_data << "Total number of cells: " 
	     << n_cells << " = "
	     << n_processes <<" (number of processes) x "
	     << n_cells_per_process <<" (number of cells/proc.)\n";
    std::cout << out_data.str() << std::flush;
    output_file << out_data.str() << std::flush;

  }
#endif


  //Time stamp definition
  time_t rawtime;
  struct tm * timeinfo;
  time (&rawtime);
  timeinfo = localtime (&rawtime);




  // ************************************
  // * 	IMPORTING CELLS FROM FILE *
  // ************************************
 
#if if_save_output    
  // history of mutation parameters and mild & strong mutations for cells at this MPI process 
  std::stringstream mutation_history_fname;
  mutation_history_fname << work_folder_name << "mutation_history__rank" << rank << ".data";
  ofstream mutation_history_file;
  mutation_history_file.open(mutation_history_fname.str().c_str(), std::ios::out | std::ios::trunc);

#if _USE_HDF5

  /* ****************************
   * HDF5 OUTPUT INITIALIZATION *
   * ****************************
   */

  H5open();
  hid_t outfile, fapl, dataspace, attribute;
  char filename[255];
  fapl = H5Pcreate(H5P_FILE_ACCESS);
  H5Pset_fapl_sec2(fapl);
  sprintf(filename, "%s%s%s%d%s", work_folder_name.c_str(), FILE_NAME_PREFIX, FILE_NAME_RANK_SUFFIX, rank, H5_FILE_EXTENSION);

  // check if output file exists
  fstream raw_cell_file(filename, fstream::in);
  if (!raw_cell_file.is_open())
    {
      // create output file if it does not exist
      outfile = H5Fcreate(filename, H5F_ACC_EXCL, H5P_DEFAULT, fapl);
    }
  else {
    printf("\nERROR: Cell info file from previous run exists. Please move it elsewhere.\n%s\nExitting...\n\n\n", filename);
    raw_cell_file.close();
    H5Pclose(fapl);
    finalize();
    return 1;
  }

  dataspace = H5Screate(H5S_SCALAR);

  // save n_cells as an attribute
  attribute = H5Acreate(outfile, H5_ATTRIB_ROOT_N_CELLS, H5T_NATIVE_INT,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
  H5Awrite(attribute, H5T_NATIVE_INT, &n_cells);

  // save n_cells_per_process as an attribute
  attribute = H5Acreate(outfile, H5_ATTRIB_ROOT_N_CELLS_PER_PROC,
			H5T_NATIVE_INT, dataspace, H5P_DEFAULT, H5P_DEFAULT);
  H5Awrite(attribute, H5T_NATIVE_INT, &n_cells_per_process);

  // save n_processes as an attribute
  attribute = H5Acreate(outfile, H5_ATTRIB_ROOT_N_PROCS, H5T_NATIVE_INT,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
  H5Awrite(attribute, H5T_NATIVE_INT, &n_processes);

  // save number_of_epochs as an attribute
  attribute = H5Acreate(outfile, H5_ATTRIB_ROOT_N_EPOCHS, H5T_NATIVE_INT,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
  H5Awrite(attribute, H5T_NATIVE_INT, &number_of_epochs);

  // save selection_period as an attribute
  attribute = H5Acreate(outfile, H5_ATTRIB_ROOT_N_TIMESTEPS_PER_EPOCH,
			H5T_NATIVE_INT, dataspace, H5P_DEFAULT, H5P_DEFAULT);
  H5Awrite(attribute, H5T_NATIVE_INT, &selection_period);

  int n_timesteps = number_of_epochs * selection_period;

  // save total timesteps as an attribute
  attribute = H5Acreate(outfile, H5_ATTRIB_ROOT_N_TIMESTEPS, H5T_NATIVE_INT,
			dataspace, H5P_DEFAULT, H5P_DEFAULT);
  H5Awrite(attribute, H5T_NATIVE_INT, &n_timesteps);

  // close opened hid_t types
  H5Aclose(attribute);
  H5Sclose(dataspace);

#else

  // cell_info file -- every cell in this MPI process is loged  
  // every "save_cell_info_freq" epoch (one file per process)
  std::stringstream cell_info_fname;
  cell_info_fname << work_folder_name << "cell_info__rank" << rank << ".data";
  ofstream cell_info_file;
  cell_info_file.open(cell_info_fname.str().c_str(), std::ios::out | std::ios::trunc);

#endif
#endif

#if if_save_v_history
  // node_values_history file -- log saved every save_cell_info_freq epoch
  // (one file per process)
  std::stringstream node_values_history_fname;
  node_values_history_fname << work_folder_name << "node_values_history__rank" << rank << ".data";
  ofstream node_values_history_file;
  node_values_history_file.open(node_values_history_fname.str().c_str(), std::ios::out | std::ios::trunc);
#endif

#if if_save_lgt_events_log
  std::stringstream lgt_events_log_fname;
  lgt_events_log_fname << work_folder_name << "lgt_events_log__rank" << rank << ".data";
  ofstream lgt_events_log_file;
  lgt_events_log_file.open(lgt_events_log_fname.str().c_str(), std::ios::out | std::ios::trunc);
#endif

#if if_save_mutation_events_log
  std::stringstream mutation_events_log_fname;
  mutation_events_log_fname << work_folder_name << "mutation_events_log__rank" << rank << ".data";
  ofstream mutation_events_log_file;
  mutation_events_log_file.open(mutation_events_log_fname.str().c_str(), std::ios::out | std::ios::trunc);
#endif

  //! ne coordinate of the cell in 2D grid (temporary variable) used for cell apoptosis
  int new_coordinate, new_coordinate_original;
  int old_coordinate;
  int old_cell_ID;
  //! cells will be divided into (x_shift, y_shift) relative to the current position (with gaussian distribution)
  int x_shift, y_shift;
  //! displacement (x_shift, y_shift) is first generated in polar coordinates
  double alpha, R;


  //----------------------------------------------------------------------
  // create spatial 2D grids with info on spatial cell density distribution 
  int* cell_density;
  int grid_n_points = global->param.grid.n * global->param.grid.n; 
  int grid_max = global->param.grid.n;
  cell_density = new int[grid_n_points];
  // used for synchronisation:
  //! local updates to the change matrix;
  int* cell_density_change_this;
  //! sum of cell_density_change_this matrices from all MPI processes
  int* cell_density_change;
  cell_density_change_this = new int[grid_n_points];
  cell_density_change = new int[grid_n_points];
  // set all matices to zero:
  for (int coord_i=0; coord_i<grid_n_points; coord_i++){
    cell_density[coord_i]=0;
    cell_density_change[coord_i]=0;
    cell_density_change_this[coord_i]=0;
  }

  // yyy 
  //----------------------------------------------------------------------
  //! array with all cell coordinates
  int* cell_coordinates;
  cell_coordinates = new int[n_cells];
  //! coordinates of cells local to this MPI process
  int* cell_coordinates_this;
  cell_coordinates_this = new int[n_cells_per_process];
  //! temporary, coordinates of cell with max/min energy
  int max_e_coord, min_e_coord;

  //----------------------------------------------------------------------
  // CREATE or LOAD CELLS LOCAL TO THIS MPI PROCESS


  // ID of the cell (local counter of newborn cells on this core)
  int next_cell_ID=0;

  std::vector<Cell*> cells;

  // create a vector of cells which are to be stored in this MPI process:
  for (int cell_i=0; cell_i< n_cells_per_process; cell_i++){
    Cell *new_cell = new Cell(global);
     
    if (import_cells==0) {//Initializate of a random population
      new_cell->setID(next_cell_ID);
      next_cell_ID++;

      // set a random coordinate
      new_coordinate = ROUND((grid_n_points-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));

#if if_fight_mode

      int x1, y1, x2, y2;

      // first quarter
      x1=y1=ROUND(global->param.grid.n / 4)-1;
      // third quarter
      x2=y2=ROUND(global->param.grid.n * 3/4);
      
      R = ( my_rand(global->param.globals.idum, global->param.globals.itemp) * global->param.grid.n / 4) - 1;
      alpha = 2*PI*my_rand(global->param.globals.idum, global->param.globals.itemp);

      // if this cell in the first half of the population
      //     if (n_cells_per_process*rank+cell_i < n_cells/2 ){
      if (cells.size() < n_cells_per_process/2 ){
	x_shift = ROUND( cos(alpha)*R )+x1;
	y_shift = ROUND( sin(alpha)*R )+y1;
      }
      // if this cell in the second half of the population
      else {
	x_shift = ROUND( cos(alpha)*R )+x2;
	y_shift = ROUND( sin(alpha)*R )+y2;
      }
 
      new_coordinate = x_shift + y_shift * grid_max;
#endif

      // set label as an experimet number
      new_cell->setlabel(experiment_number);

      new_cell->setcoord(new_coordinate);
      cell_coordinates_this[cell_i] = new_coordinate; 

      new_cell->setinput(-1, -3,
			 global->param.general.n_signals,
			 global->param.network.initial_sensor_prob);
      new_cell->setmutation(global->param.network.creation, 
			    global->param.network.destruction,
			    global->param.network.mild_mutation,
			    global->param.network.strong_mutation,
			    global->param.network.evolvability);
#if if_save_output    
      out_data.str("");
      out_data << "cell " << rank << "." << new_cell->getID() << " was created at ("
	       << new_cell->getcoord() % grid_max << ","<<  new_cell->getcoord() / grid_max <<") "
	       << "Label=" << new_cell->getlabel() 
	       << "\n";
      std::cout << out_data.str() << std::flush;
#endif

    }

    else if (import_cells==1) {//Initializate a population from one cell
      std::stringstream cell_input_fname;
      cell_input_fname << work_folder_name;

#if _USE_HDF5
      MPI_Status status;
      int temp = 0;
      cell_input_fname << FILE_NAME_LAST_PREFIX << FILE_NAME_PREFIX << H5_FILE_EXTENSION;

      // start of critical section
      if(0 != rank)
	MPI_Recv(&temp, 1, MPI_INT, rank - 1, 4958, MPI_COMM_WORLD, &status);
      hid_t infile = H5Fopen(cell_input_fname.str().c_str(), H5F_ACC_RDONLY, H5P_DEFAULT);
      int *results = determineCellToLoad(infile, rank, next_cell_ID, n_processes, n_cells_per_process);
      new_cell->load(infile, FIRST_EPOCH_ID, results[0], results[1], next_cell_ID);
      delete results;
      H5Fclose(infile);
      if(rank != n_processes -1)
	MPI_Send(&temp, 1, MPI_INT, rank + 1, 4958, MPI_COMM_WORLD);
      // end of critical section
#else
      cell_input_fname << "input_cell.in";
      new_cell->load(cell_input_fname.str(), out_str, FIRST_EPOCH_ID,
		     next_cell_ID);
#endif
      next_cell_ID++;

      // set a random coordinate
      new_coordinate = ROUND( (grid_n_points-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
#if if_fight_mode
      //FFF
      int x1, y1, x2, y2;

      // first quarter
      x1=y1=ROUND(global->param.grid.n / 4)-1;
      // third quarter
      x2=y2=ROUND(global->param.grid.n * 3/4);
      
      R = ( my_rand(global->param.globals.idum, global->param.globals.itemp) * global->param.grid.n / 4) - 1;
      alpha = 2*PI*my_rand(global->param.globals.idum, global->param.globals.itemp);

      // if this cell in the first half of the population
      if (cells.size() < n_cells_per_process/2 ){
	x_shift = ROUND( cos(alpha)*R )+x1;
	y_shift = ROUND( sin(alpha)*R )+y1;
      }
      // if this cell in the second half of the population
      else {
	x_shift = ROUND( cos(alpha)*R )+x2;
	y_shift = ROUND( sin(alpha)*R )+y2;
      }
 
      new_coordinate = x_shift + y_shift * grid_max;

#endif
      new_cell->setcoord(new_coordinate);
      cell_coordinates_this[cell_i] = new_coordinate;

      new_cell->setmutation(global->param.network.creation,
			    global->param.network.destruction,
			    global->param.network.mild_mutation,
			    global->param.network.strong_mutation,
			    global->param.network.evolvability);

#if if_save_output    
      out_data.str("");
      out_data<<"Cell "<<rank<<"."<<new_cell->getID()<<" was imported from " << cell_input_fname.str() 
	      << ". Coordinate is set to (" << new_cell->getcoord() % grid_max << ","<<  new_cell->getcoord() / grid_max << ") "
	      << "Label=" << new_cell->getlabel() 
	      <<"\n";
      std::cout << out_data.str() << std::flush;
#endif
    }

    else if (import_cells==11) {//Initializate a population from two cell (half&half)
      std::stringstream cell_input_fname;
      cell_input_fname << work_folder_name;

#if _USE_HDF5
      // pass;
#else
      if (cells.size() < n_cells_per_process/2 )
	cell_input_fname << "input_cell_1.in";
      else
	cell_input_fname << "input_cell_2.in";
	
      new_cell->load(cell_input_fname.str(), out_str, FIRST_EPOCH_ID,
		     next_cell_ID);
#endif
      next_cell_ID++;

      // set a random coordinate
      new_coordinate = ROUND( (grid_n_points-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
#if if_fight_mode

      int x1, y1, x2, y2;

      // first quarter
      x1=y1=ROUND(global->param.grid.n / 4)-1;
      // third quarter
      x2=y2=ROUND(global->param.grid.n * 3/4);
      
      R = ( my_rand(global->param.globals.idum, global->param.globals.itemp) * global->param.grid.n / 4) - 1;
      alpha = 2*PI*my_rand(global->param.globals.idum, global->param.globals.itemp);

      // if this cell in the first half of the population
      if (cells.size() < n_cells_per_process/2 ){
	x_shift = ROUND( cos(alpha)*R )+x1;
	y_shift = ROUND( sin(alpha)*R )+y1;
      }
      // if this cell in the second half of the population
      else {
	x_shift = ROUND( cos(alpha)*R )+x2;
	y_shift = ROUND( sin(alpha)*R )+y2;
      }
 
      new_coordinate = x_shift + y_shift * grid_max;

#endif
      new_cell->setcoord(new_coordinate);
      cell_coordinates_this[cell_i] = new_coordinate;

      new_cell->setmutation(global->param.network.creation,
			    global->param.network.destruction,
			    global->param.network.mild_mutation,
			    global->param.network.strong_mutation,
			    global->param.network.evolvability);

#if if_save_output    
      out_data.str("");
      out_data<<"Cell "<<rank<<"."<<new_cell->getID()<<" was imported from " << cell_input_fname.str() 
	      << ". Coordinate is set to (" << new_cell->getcoord() % grid_max << ","<<  new_cell->getcoord() / grid_max << ") "
	      << "Label=" << new_cell->getlabel() 
	      <<"\n";
      std::cout << out_data.str() << std::flush;
#endif
    }

    else if (import_cells==2) {//load a full population from a previous run
 
      // load a copy of the cell
      std::stringstream cell_input_fname;
      cell_input_fname << work_folder_name << "/input_population/last_cell_info__rank" << rank << ".data";
      new_cell->load_ascii(cell_input_fname.str(), cells.size()+1, out_str, -1, next_cell_ID);
      next_cell_ID++;

      // set a random coordinate OR load coordinates (?)
      new_coordinate = ROUND( (grid_n_points-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
#if if_fight_mode

      int x1, y1, x2, y2;

      // first quarter
      x1=y1=ROUND(global->param.grid.n / 4)-1;
      // third quarter
      x2=y2=ROUND(global->param.grid.n * 3/4);
      
      R = ( my_rand(global->param.globals.idum, global->param.globals.itemp) * global->param.grid.n / 4) - 1;
      alpha = 2*PI*my_rand(global->param.globals.idum, global->param.globals.itemp);

      // if this cell in the first half of the population
      if (cells.size() < n_cells_per_process/2 ){
	x_shift = ROUND( cos(alpha)*R )+x1;
	y_shift = ROUND( sin(alpha)*R )+y1;
      }
      // if this cell in the second half of the population
      else {
	x_shift = ROUND( cos(alpha)*R )+x2;
	y_shift = ROUND( sin(alpha)*R )+y2;
      }
 
      new_coordinate = x_shift + y_shift * grid_max;

#endif
      new_cell->setcoord(new_coordinate);
      cell_coordinates_this[cell_i] = new_coordinate;

      new_cell->setmutation(global->param.network.creation,
			    global->param.network.destruction,
			    global->param.network.mild_mutation,
			    global->param.network.strong_mutation,
			    global->param.network.evolvability);

#if if_save_output    
      out_data.str("");
      out_data<<"Cell "<<rank<<"."<<new_cell->getID()<<" was imported from " << cell_input_fname.str() 
	      << ". Coordinate is set to (" << new_cell->getcoord() % grid_max << ","<<  new_cell->getcoord() / grid_max << ") "
	      << "Label=" << new_cell->getlabel() 
	      << "\n";
      std::cout << out_data.str() << std::flush;
#endif
    }
    else if (import_cells==3) {//combune two populations from previous runs (first half from first run, second half from the second run)
 
      // load a copy of the cell
      std::stringstream cell_input_fname;
      if (cells.size() < n_cells_per_process/2)
	cell_input_fname << work_folder_name << "/input_population_1/last_cell_info__rank" << rank << ".data";
      else
	cell_input_fname << work_folder_name << "/input_population_2/last_cell_info__rank" << rank << ".data";
	
      // read first n_cells_per_process/2 cells from each population file (in case you whant to load more cells than population have, use import_cells==20)
      new_cell->load_ascii(cell_input_fname.str(), (cells.size() % (n_cells_per_process/2)) +1, out_str, -1, next_cell_ID);
      next_cell_ID++;

      // set a random coordinate OR load coordinates(?)
      new_coordinate = ROUND( (grid_n_points-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
#if if_fight_mode

      int x1, y1, x2, y2;

      // first quarter
      x1=y1=ROUND(global->param.grid.n / 4)-1;
      // third quarter
      x2=y2=ROUND(global->param.grid.n * 3/4);
      
      R = ( my_rand(global->param.globals.idum, global->param.globals.itemp) * global->param.grid.n / 4) - 1;
      alpha = 2*PI*my_rand(global->param.globals.idum, global->param.globals.itemp);

      // if this cell in the first half of the population
      if (cells.size() < n_cells_per_process/2 ){
	x_shift = ROUND( cos(alpha)*R )+x1;
	y_shift = ROUND( sin(alpha)*R )+y1;
      }
      // if this cell in the second half of the population
      else {
	x_shift = ROUND( cos(alpha)*R )+x2;
	y_shift = ROUND( sin(alpha)*R )+y2;
      }
 
      new_coordinate = x_shift + y_shift * grid_max;

#endif
      new_cell->setcoord(new_coordinate);
      cell_coordinates_this[cell_i] = new_coordinate;

      new_cell->setmutation(global->param.network.creation,
			    global->param.network.destruction,
			    global->param.network.mild_mutation,
			    global->param.network.strong_mutation,
			    global->param.network.evolvability);

#if if_save_output    
      out_data.str("");
      out_data<<"Cell "<<rank<<"."<<new_cell->getID()<<" was imported from " << cell_input_fname.str() 
	      << ". Coordinate is set to (" << new_cell->getcoord() % grid_max << ","<<  new_cell->getcoord() / grid_max 
	      <<")\n";
      std::cout << out_data.str() << std::flush;
#endif
    }
    else if (import_cells==20) {//load a full population from a previous run of 256 cells on 32 mpi proc
      
      // funny constatns -- load that from somewhere...
      int old_n_mpi = 32-1;
      int old_n_per_mpi = 8-1;

      // load a copy of the cell
      std::stringstream cell_input_fname;

      int rank_to_load = ROUND(my_rand(global->param.globals.idum, global->param.globals.itemp)*old_n_mpi); // starts from 0
      cell_input_fname << work_folder_name << "/input_population/last_cell_info__rank" << rank_to_load << ".data";
      int cell_to_load = ROUND(my_rand(global->param.globals.idum, global->param.globals.itemp)*old_n_per_mpi)+1; // starts from 1
      new_cell->load_ascii(cell_input_fname.str(), cell_to_load, out_str, -1, next_cell_ID);
      next_cell_ID++;

      // set a random coordinate OR load coordinates (?)
      new_coordinate = ROUND( (grid_n_points-1) * my_rand(global->param.globals.idum, global->param.globals.itemp));
#if if_fight_mode
      //FFF
      int x1, y1, x2, y2;

      // first quarter
      x1=y1=ROUND(global->param.grid.n / 4)-1;
      // third quarter
      x2=y2=ROUND(global->param.grid.n * 3/4);
      
      R = ( my_rand(global->param.globals.idum, global->param.globals.itemp) * global->param.grid.n / 4) - 1;
      alpha = 2*PI*my_rand(global->param.globals.idum, global->param.globals.itemp);

      // if this cell in the first half of the population
      if (cells.size() < n_cells_per_process/2 ){
	x_shift = ROUND( cos(alpha)*R )+x1;
	y_shift = ROUND( sin(alpha)*R )+y1;
      }
      // if this cell in the second half of the population
      else {
	x_shift = ROUND( cos(alpha)*R )+x2;
	y_shift = ROUND( sin(alpha)*R )+y2;
      }
 
      new_coordinate = x_shift + y_shift * grid_max;

#endif
      new_cell->setcoord(new_coordinate);
      cell_coordinates_this[cell_i] = new_coordinate;

      new_cell->setmutation(global->param.network.creation,
			    global->param.network.destruction,
			    global->param.network.mild_mutation,
			    global->param.network.strong_mutation,
			    global->param.network.evolvability);

#if if_save_output    
      out_data.str("");
      out_data<<"Cell "<<rank<<"."<<new_cell->getID()<<" was imported from " << cell_input_fname.str() 
	      << ". Coordinate is set to (" << new_cell->getcoord() % grid_max << ","<<  new_cell->getcoord() / grid_max << ") "
	      << "Label=" << new_cell->getlabel() 
	      << "\n";
      std::cout << out_data.str() << std::flush;
#endif
    }

    //######################################################################
    // NEXT CELL IS CREATED / LOADED at this point
    //######################################################################


    // COMMON OPERATION FOR A GENERATED/LOADED CELL  

    // The way the class is created, the v_history is not initialized through 
    // the constructor and has to be defined externally. 
    // Additionally, mutation_counters and energy are initialized.

    new_cell->setv_history(0,1,selection_period,0);

    new_cell->setmutation_counter(-2,0,0);
    new_cell->setmutation_counter(-1,0,0);

    new_cell->setenergy(0,1);
    new_cell->setenergy(global->param.energy.initial_energy, 0); 

#if if_save_output
#if _USE_HDF5
    new_cell->save(cell_i, FIRST_EPOCH_ID, outfile);
#else
    new_cell->save(cell_i, FIRST_EPOCH_ID, out_str);
    cell_info_file << out_str << std::flush;
#endif
#endif

    // +1 to local cell density distribution matrix at this cell's coordinate:
    cell_density_change_this[new_cell->getcoord()]++;

   cells.push_back(new_cell);
    new_cell = NULL;
  }

  //######################################################################
  // ALL CELLS ARE CREATED / LOADED AT THIS POINT 
  //######################################################################


#if if_HGT_from_evolved_cells
  std::stringstream evolved_cell_fname;
  Cell evolved_cell( global );
  evolved_cell_fname << work_folder_name << "input_evolved_cell.in";
  evolved_cell.load_ascii(evolved_cell_fname.str(), 1, out_str, -1, next_cell_ID);
#endif


  // collect changes to the density distribution matrix from all MPI processes
#if _IF_MPI
  MPI_Allreduce(cell_density_change_this, cell_density_change, grid_n_points, MPI_INT, MPI_SUM, MPI_COMM_WORLD);
#else
  // Serial version:
  for (int coord_i=0; coord_i<grid_n_points; coord_i++){
    cell_density_change[coord_i] = cell_density_change_this[coord_i];
  }
#endif

  for (int coord_i=0; coord_i<grid_n_points; coord_i++){
    cell_density[coord_i]+=cell_density_change[coord_i];
    // and reset local chage matrix to zero:
    cell_density_change_this[coord_i]=0;
  }
  
  // save 2D density of cells
  if (rank==master_rank){
    out_data.str("");
    save_grid(grid_max, grid_n_points/grid_max, cell_density, out_str);
    out_data << out_str << "\n";
    cell_density_file << out_data.str() << std::flush;
  }

  // collect coordinates of all cells
#if _IF_MPI
  MPI_Allgather(cell_coordinates_this, n_cells_per_process, MPI_INT, cell_coordinates, n_cells_per_process, MPI_INT, MPI_COMM_WORLD);
#else
  // Serial version:
  // same size of cell_coordinates (population) and cell_coordinates_this (this process)
  for (int coord_i=0; coord_i<n_cells; coord_i++){
    cell_coordinates[coord_i] = cell_coordinates_this[coord_i];
  }
#endif

  // LGT variables

  //! array with LGT fragments (modules) for temporary use
  Cell* lgt_module_tmp = new Cell(global);

  //! if lgt event was established for a given pair in given direction at a given time step
  bool if_lgt_event;

  //! temporary variable for normalization
  double tmp_norm;

  //! flag is used in LGT selection of the sonor cell
  bool if_source_selected;

  //! combined probability of transformation, conjugation, and transduction:
  double tf_p, cj_p, td_p;


  //! random module size & an acceptance parameter (acceptence-rejection sampling to generate a fragment size with a given distribution)
  int module_size;
  double module_acceptance;

  //! calculated inter-cell distance
  double intercell_distance;

  //! maximum and current number of LGT events per time step
  int lgt_events;

  //! array of MPI ranks for LGT from
  int* lgt_from;
  //! array of corresponding local indeces for LGT from tasks
  int* lgt_index_from;
  //! array of MPI ranks for LGT to
  int* lgt_to;
  //! array of corresponding local indeces for LGT to tasks
  int* lgt_index_to;
  //! array of module_sizes for each LGT event
  int* module_sizes;
  //! array of of probabilities to include module for each LGT event
  double*  lgt_link_probability;

  lgt_from = new int[MAX_LGT_EVENTS_PER_TIMESTEP];
  lgt_index_from = new int[MAX_LGT_EVENTS_PER_TIMESTEP];
  lgt_to = new int[MAX_LGT_EVENTS_PER_TIMESTEP];
  lgt_index_to = new int[MAX_LGT_EVENTS_PER_TIMESTEP];
  module_sizes = new int[MAX_LGT_EVENTS_PER_TIMESTEP];
  lgt_link_probability = new double[MAX_LGT_EVENTS_PER_TIMESTEP];
	

  // ***********************************
  // *  DECLARING SIGNALS AND RESOURCE *
  // ***********************************
	
  double *food = new double [selection_period+1];
  double **signal_ptr = new double *[global->param.general.n_signals];

  // we need selection_period + 1, because of the way signal_creation 
  // is defined (needs N+1 elements)
  for (int i=0;i<global->param.general.n_signals;i++) 
    signal_ptr[i] = new double [selection_period+1];

  int offset =0;
  if (signal_type == BISTATE)
    offset = 3; //declare what is the ratio between normal:random


  //Signals and food are generated or imported from 
  std::stringstream signal_fname;

  if (import_signals ==1){
    signal_fname << work_folder_name << "input_signals.in";
    load_signals(global->param.general.n_signals, 
		 food, signal_ptr, signal_fname.str(), out_str);

#if if_save_output   
    if (rank==master_rank)
      output_file << out_str << std::flush;
#endif
  }

  else if (import_signals ==2){
    signal_fname << work_folder_name << "input_signals.in";
    load_signals(global->param.general.n_signals, 
		 food, signal_ptr, signal_fname.str(), out_str);

#if if_save_output   
    if (rank==master_rank)
      output_file << out_str << std::flush;
#endif
  }

  else {
    signal_subroutine(global->param, food, signal_ptr,random_appearance,
		      random_duration, 0, selection_period);

#if if_save_output   
    if (rank==master_rank){
      out_str = "Signals created on " + GetTime();
      output_file << out_str << std::flush;
    }
#endif
  }


  // save created or imported signals into the output:
#if if_save_output   
  if (rank==master_rank){
    save_signals(global->param.general.n_signals, food,signal_ptr, out_str);
    output_file << out_str << std::flush;
  }
#endif

  //----------------------------------------------------------------------
  // VARIABLES USED IN SYNCHRONIZATION

  // used in search for MIN & MAX energies
  double_int max_ftnss_cell, max_ftnss_cell_this ;
  double_int *max_e_cell, *max_e_cell_this;
  max_e_cell=new double_int[cells.size()];
  max_e_cell_this=new double_int[cells.size()];
  double_int max_e; //! max cell in the local array of cells

  double_double_int *min_e_dist_cell, *min_e_dist_cell_this;
  min_e_dist_cell=new double_double_int[cells.size()];
  min_e_dist_cell_this=new double_double_int[cells.size()];
  double_double_int min_e_dist; //! min cell in the local array of cells

  int max_e_index, min_e_dist_index; //! index of the min/max cell in the local array of cells
 
  // used in divisions
  bool if_this_divide, if_this_replace;
  int rank_to_replace_cell, rank_to_divide_cell;
  int index_to_divide_cell, index_to_replace_cell; //! a copy of index of the min/max cell in the local array of cells

 
  //! prepare MPI datatypes and operatotors
#if _IF_MPI
  create_MPI_DOUBLE_DOUBLE_INT();
  create_MPI_MIN_E_DIST_LOC();
  create_MPI_MIN_E_NODIST_LOC();
#else
  // Serial version: 
  // nothing to do here 
#endif


  //----------------------------------------------------------------------
  // keep initial mutation rates - they may change if 
  // the experiment is variable mutation rate is 1
  double initial_creation = global->param.network.creation;
  double initial_destruction = global->param.network.destruction;
  double initial_mild_mutation = global->param.network.mild_mutation;
  double initial_strong_mutation = global->param.network.strong_mutation;


  //! normalisation is used in calculation of LGT probabilities
  double tf_dist_norm = 1.0 - tanh(-global->param.lgt.tf_dist_middle / global->param.lgt.tf_dist_slope);
  double tf_size_norm = 1.0 - tanh(-global->param.lgt.tf_size_middle / global->param.lgt.tf_size_slope);
  double cj_dist_norm = 1.0 - tanh(-global->param.lgt.cj_dist_middle / global->param.lgt.cj_dist_slope);
  double td_dist_norm = 1.0-  tanh(-global->param.lgt.td_dist_middle / global->param.lgt.td_dist_slope);


#ifdef if_print_benchmark 
  // array with times of cell update for each cell per time step within one epoch (no need to zerro it -- initialized on the fly)
  long *mtimes = new long[ cells.size()*selection_period ];
  // array with growth rates for each cell per time step within one epoch (no need to zerro it -- initialized on the fly)
  double *grates = new double[ cells.size()*selection_period ];
  int *sizes = new int[ cells.size()*selection_period ];
  int *links = new int[ cells.size()*selection_period ];
#endif

  // ****************************************************
  // *			MAIN CYCLE		        *
  // ****************************************************


  for (int epochID=0; epochID< number_of_epochs; epochID++) {

    int rate_of_change=1;

    // the duration, appearance, stochasticity of signals and 
    // resources may change depending on the value of some flags and 
    // signal_type.
    // How frequent this change will take place can be regulated by 
    // the modulo base rate_of_change value of epochID.
    if (     (random_duration ==1 || random_appearance ==1 
	      || signal_type == MULTIGATE || signal_type ==BISTATE) 
	     && epochID%rate_of_change == 0) {

      // Obsolete: in MULTIGATES the ratio between two different gate types 
      // was given by the offset,
      // while now it alternates. Code disabled.
      // if (signal_type == MULTIGATE)
      // offset = epochID; //for multigates - for BISTATE has been 
      // declared above, under the signal block

      signal_subroutine(global->param, food, signal_ptr,random_appearance,random_duration,
		        offset, selection_period);
    }


    // If variable mutation rate is on, individual mutation rates of 
    // organism change 
    if (mutation_flag ==1) {
      double tmp_creat, tmp_destr, tmp_mild_mut, tmp_strong_mut, 
	tmp_evolvability, tmp_mutation_coeff; 

      // for each local cell
      for (int i=0; i<cells.size(); i++){
	tmp_evolvability = cells[i]->getmutation(5);

	// Version A
	//tmp_mutation_coeff = MAX(0,gaussian(1,tmp_evolvability));
	//tmp_creat = cells[i]->getmutation(1)*tmp_mutation_coeff;
	//tmp_destr = cells[i]->getmutation(2)*tmp_mutation_coeff;
	//tmp_mild_mut = cells[i]->getmutation(3)*tmp_mutation_coeff;
	//tmp_strong_mut = cells[i]->getmutation(4)*tmp_mutation_coeff;

	// Version B
	tmp_creat = MAX(0, gaussian(cells[i]->getmutation(1), cells[i]->getmutation(1)*tmp_evolvability, global->param.globals.idum, global->param.globals.itemp));
	tmp_destr = MAX(0, gaussian(cells[i]->getmutation(2), cells[i]->getmutation(2)*tmp_evolvability, global->param.globals.idum, global->param.globals.itemp));
	tmp_mild_mut = MAX(0, gaussian(cells[i]->getmutation(3), cells[i]->getmutation(3)*tmp_evolvability, global->param.globals.idum, global->param.globals.itemp));
	tmp_strong_mut = MAX(0, gaussian(cells[i]->getmutation(4), cells[i]->getmutation(4)*tmp_evolvability, global->param.globals.idum, global->param.globals.itemp));

	cells[i]->setmutation(tmp_creat, tmp_destr, tmp_mild_mut, tmp_strong_mut, tmp_evolvability);
      } // END for each local cell
    } // END if mutation_flag


    // last epoch -- no mutations:
    if (epochID==number_of_epochs-1)
      for (int i=0; i<cells.size(); i++)
	cells[i]->setmutation(0.0, 0.0, 0.0, 0.0, 0.0);






    //**********************************************************************
    //**********************************************************************
    // for each time step
    for (int time_step_i = 0; time_step_i < selection_period; time_step_i++){



      //**********************************************************************
      // for each cell in this MPI process
      for (int cell_i=0; cell_i<cells.size(); cell_i++){
 
  
	//put current v values to history
	cells[cell_i]->setv_history(time_step_i,0,selection_period,0); 

#if  if_save_mutation_events_log
	double F_before_m, F_after_m, F_sd_m;
	// test fitness "before"
	Cell* cell_before = new Cell(global);
	cell_before->copy_from_another_cell(cells[cell_i]);
	cell_before->setv_history(0,1,selection_period,0);
#endif

#ifdef if_print_benchmark 
	struct timeval start, end;
	long mtime, seconds, useconds;    
	gettimeofday(&start, NULL);
#endif

	int mutation_type = cells[cell_i]->mutational_step(time_step_i, selection_period); 

#if  if_save_mutation_events_log
	if (mutation_type>0){

	  // test fitness "before"
	  cell_before->test_fitness(signal_ptr, food, 1, 5, selection_period, F_before_m, F_sd_m);

	  // save cell "before
	  cell_before->save_ascii(cell_i  , -1, out_str);
	  mutation_events_log_file << out_str;

	  // test fitness "after"
	  cells[ cell_i ]->test_fitness(signal_ptr, food, 1, 5, selection_period, F_after_m, F_sd_m);

	  // save cell "after"
	  cells[ cell_i ]->save_ascii(cell_i, 1, out_str);
	  mutation_events_log_file << out_str;

	  // print mutation event log line:
	  out_data.str("");
	  out_data << epochID << "." << time_step_i << ": "
		   << rank << "[" << cell_i <<"] " 
		   << "[[ mutation_type= " << mutation_type 
		   << " t: " << std::scientific << std::setprecision(4) << std::setw(12) <<  epochID + time_step_i*1.0/selection_period
		   << " F: " << std::scientific << std::setprecision(6) << std::setw(14) << F_before_m << " => "
		   << std::scientific << std::setprecision(6) << std::setw(14) << F_after_m <<" ]] \n";
	  std::cout << out_data.str();

	  out_data << "----------------------------------------------------------------------\n";
	  mutation_events_log_file << out_data.str() << std::flush;
	}

	cell_before->setv_history(0,-1,selection_period,0); 
#if _IF_MPI
	cell_before->free_MPI_types();
#endif
	delete cell_before;

#endif

	//Calculates the new values for all nodes of the cell i
	cells[cell_i]->calculation_new_values(time_step_i, signal_ptr);
	
#ifdef if_print_benchmark 

	gettimeofday(&end, NULL);

	seconds  = end.tv_sec  - start.tv_sec;
	useconds = end.tv_usec - start.tv_usec;
	mtime = ((seconds) * 1000000 + useconds);

	// save time for (one cell/one timestep) into the array (calculate averages and st.dev. later):
	mtimes[ cell_i*selection_period + time_step_i ] = mtime;
	
	// initial energy at this timestep
	double energy_ini=cells[cell_i]->getenergy();
#endif

	// This is the food intake for all cells
	double intake, temp, final_node_cost;

	// If node maintenance cost inversely proportional 
	// to the mutation rate of the organism 
	// (the more efficient the repair mechanism, i.e. lower the mutation rate, 
	// the more expensive it is to have it).
	if (node_cost_proportional_to_mutation==1){
	  final_node_cost = (double)node_cost /
	    ( 
	     (
	      global->param.network.creation + 
	      global->param.network.destruction +
	      global->param.network.mild_mutation + 
	      global->param.network.strong_mutation
	      )
	     / 
	     (initial_creation+initial_destruction +initial_mild_mutation + initial_strong_mutation)
	      );
	}
	else
	  final_node_cost = (double)node_cost;

	//total energy = prior_energy + energy_harvested - energy_burned.
	intake = MIN(food[time_step_i],
		     food_per_metabolic
		     *cells[cell_i]->getv_history(5,time_step_i));

	//temp has the total energy spend/harvested
	temp = intake 
	  - maintenance_cost*cells[cell_i]->getv_history(5,time_step_i)
	  -final_node_cost*cells[cell_i]->getn_nodes(); 

	// if temp<0 then it substracts it only if cell's energy is larger, 
	// if temp>0 then by default it adds it
	if (cells[cell_i]->getenergy()>-temp) 
	  cells[cell_i]->setenergy(temp,0);
	else
	  // assign zero to energy
	  cells[cell_i]->setenergy(0,1); 

#ifdef if_print_benchmark 
	// get delta-energy per time step )growth rate):
	double grate = cells[cell_i]->getenergy() - energy_ini;
	grates[ cell_i*selection_period + time_step_i ] = grate;
	// save current size
	sizes[ cell_i*selection_period + time_step_i ] = cells[cell_i]->getn_nodes();
	// save number of links
	links[ cell_i*selection_period + time_step_i ] = cells[cell_i]->getn_links();
#endif


	// Cell apoptosis - if no energy left and no metabolic
	// modified proteins, the Cell dies and a new, random one takes it place.
	if ( cells[cell_i]->getenergy() <= cell_death_threshold ){

	  new_coordinate_original=cells[cell_i]->getcoord();

	  /*

	  // move position with gaussian distribution (and withing a 2D grid boundary):
	  do {
	  R = gaussian(0,global->param.grid.division_copy_sdev); // returns positive and negative R-s
	  alpha = PI*my_rand(global->param.globals.idum, global->param.globals.itemp);
	  x_shift = ROUND( cos(alpha)*R );
	  y_shift = ROUND( sin(alpha)*R );
	  } while ( 
	  ((new_coordinate_original / grid_max) + y_shift)<0 or 
	  ((new_coordinate_original / grid_max) + y_shift)>=grid_max or
	  ((new_coordinate_original % grid_max) + x_shift)<0 or 
	  ((new_coordinate_original % grid_max) + x_shift)>=grid_max
	  );
	  new_coordinate = new_coordinate_original + x_shift + y_shift * grid_max;
	  */

	  cell_density_change_this[cells[cell_i]->getcoord()]--;
	  cell_density_change_this[new_coordinate] ++;

	  //int new_label=(1000000+my_rand(global->param.globals.idum, global->param.globals.itemp)*1000000);
	  int new_label=rand();
 
#if if_phylogenetic_print
	  out_data.str("");
	  out_data << epochID << "." << time_step_i <<": " << rank << "[" << cell_i<<"]."<< cells[cell_i]->getID()
		   << " (" << cells[cell_i]->getcoord() % grid_max << ',' << cells[cell_i]->getcoord() / grid_max
		   << ") -|- "
		   << rank  << "["<< cell_i <<"]."<< next_cell_ID 
		   << " (" << new_coordinate % grid_max << ',' << new_coordinate / grid_max << ") "
		   << "Label=" << new_label 
		   << "\n";
	  std::cout << out_data.str() << std::flush;
#endif
	  
	  // create a new temporary cell //Cell()
	  Cell cell_temp(global, 0, 0, 1); 
	  cell_temp.setmutation(global->param.network.creation,
				global->param.network.destruction,
				global->param.network.mild_mutation,
				global->param.network.strong_mutation,
				global->param.network.evolvability);

	  // set a random coordinate for a new cell:
	  cell_temp.setcoord( new_coordinate );
	  cell_coordinates_this[cell_i] = new_coordinate;

	  cell_temp.setlabel(new_label);

	  cells[cell_i]->setv_history( 0, -1, selection_period, 0 );


	  cells[cell_i]->copy_from_another_cell( &cell_temp);
	  cells[cell_i]->setv_history(0,1,selection_period,0);
	  cells[cell_i]->setID(next_cell_ID);
	  next_cell_ID++;
	} // end apoptosis
      

	// now apply random diffusion with the probaility
	if ( global->param.grid.diff_probability > my_rand(global->param.globals.idum, global->param.globals.itemp) ){
	  new_coordinate=cells[cell_i]->getcoord();
	  do {
	    R = gaussian(0,global->param.grid.diff_sdev, global->param.globals.idum, global->param.globals.itemp); // returns positive and negative R-s
	    alpha = PI*my_rand(global->param.globals.idum, global->param.globals.itemp);
	    x_shift = ROUND( cos(alpha)*R );
	    y_shift = ROUND( sin(alpha)*R );
	  } while ( 
		   ((new_coordinate / grid_max) + y_shift)<0 or 
		   ((new_coordinate / grid_max) + y_shift)>=grid_max or
		   ((new_coordinate % grid_max) + x_shift)<0 or 
		   ((new_coordinate % grid_max) + x_shift)>=grid_max
		    );
	  new_coordinate += x_shift + y_shift * grid_max;
	  
	  cell_density_change_this[ cells[cell_i]->getcoord() ]--;
	  cell_density_change_this[new_coordinate] ++;
	  cell_coordinates_this[cell_i] = new_coordinate; 


#if if_phylogenetic_print
	  // printout diffusion event: 
	  std::cout << epochID << "." << time_step_i
		    << ": " 
		    << rank<< "[" << cell_i <<"] (" 
		    << cells[cell_i]->getcoord() % grid_max 
		    << "," 
		    << cells[cell_i]->getcoord() / grid_max 
		    << ") ~~~ " 
		    << rank<< "[" << cell_i <<"] (" 
		    << new_coordinate % grid_max 
		    << "," 
		    << new_coordinate / grid_max 
		    << ")\n";
#endif
	  cells[cell_i]->setcoord(new_coordinate);
	}

      }// END for each cell in the population


      // the rest -- divisions, LGT, 2D update -- is ignored in the last epoch:
      if (epochID!=number_of_epochs-1){



	//----------------------------------------------------------------------
	// start synchronisations

	// update spatial information:
	if ((time_step_i % sync_freq)==0){


#if _IF_MPI
	  // now collect changes to the density distribution matrix from all MPI processes
	  MPI_Allreduce(cell_density_change_this, cell_density_change, grid_n_points, MPI_INT, MPI_SUM, MPI_COMM_WORLD);
#else
	  // Serial version:
	  for (int coord_i=0; coord_i<grid_n_points; coord_i++){
	    cell_density_change[coord_i] = cell_density_change_this[coord_i];
	  }
#endif

	  for (int coord_i=0; coord_i<grid_n_points; coord_i++){
	    cell_density[coord_i]+=cell_density_change[coord_i];
	    // and reset local chage matrix to zero:
	    cell_density_change_this[coord_i]=0;
	  }
	}

#if _IF_MPI
	// update cell coordinates, and reset
	MPI_Allgather(cell_coordinates_this, n_cells_per_process, MPI_INT, cell_coordinates, n_cells_per_process, MPI_INT, MPI_COMM_WORLD);
#else
	// Serial version:
	// same size of cell_coordinates (population) and cell_coordinates_this (this process)
	for (int coord_i=0; coord_i<n_cells; coord_i++){
	  cell_coordinates[coord_i] = cell_coordinates_this[coord_i];
	}
#endif


	// ----------------------------------------------------------------------
	// LGT events

	if (rank==master_rank){
	
	  lgt_events=0;

	  for (int lgt_event_i=0; lgt_event_i<MAX_LGT_EVENTS_PER_TIMESTEP; lgt_event_i++)
	    lgt_from[lgt_event_i]=lgt_index_from[lgt_event_i]=lgt_to[lgt_event_i]=lgt_index_to[lgt_event_i]=module_sizes[lgt_event_i]=-1;

	  if_lgt_event = false;
	  int lgt_cell_i =0;
	  int lgt_cell_j =0;

	  // try a transformation lgt event
	  if (global->param.lgt.tf_probability*n_cells > my_rand(global->param.globals.idum, global->param.globals.itemp)){
	    // select a random target cell
	    lgt_cell_i = ROUND( (n_cells-1)*my_rand(global->param.globals.idum, global->param.globals.itemp) );
	    // find a source cell
	    for (int cell_j_count=0; cell_j_count<n_cells; cell_j_count++)
	      if (not(if_lgt_event)){
		lgt_cell_j = ROUND( (n_cells-1)*my_rand(global->param.globals.idum, global->param.globals.itemp) );
		intercell_distance = calc_dist(grid_max, cell_coordinates[lgt_cell_i], cell_coordinates[lgt_cell_j]);

		// probability of acceptence is a function inter-cell distance
		tf_p = (1.0 - tanh(( intercell_distance  - global->param.lgt.tf_dist_middle) / global->param.lgt.tf_dist_slope)) / tf_dist_norm;
		if (tf_p > my_rand(global->param.globals.idum, global->param.globals.itemp)){
		  if_lgt_event = true;
		  // get the module size with a proper distribution

		  // // normalization 2m+s*ln(exp(-2*m/s)+1) on [0..inf) for the sigmpid function 1-tanh((s-m)/s)
		  tmp_norm = 2*global->param.lgt.tf_size_middle + global->param.lgt.tf_size_slope * log( exp(-2*global->param.lgt.tf_size_middle/global->param.lgt.tf_size_slope) + 1);

		  do {
		    module_size = ROUND( 20*my_rand(global->param.globals.idum, global->param.globals.itemp) ) + 1;
		    module_acceptance = my_rand(global->param.globals.idum, global->param.globals.itemp); // 0..1 (function is normolized)
		    // sigmoid 1-tanh((s-m)/s) function
		  } while ( (1 - tanh( ( (double)module_size - global->param.lgt.tf_size_middle) / global->param.lgt.tf_size_slope ))/tmp_norm < module_acceptance );
		  //} while (  < module_acceptance);
		  module_sizes[lgt_events]=module_size;
		  lgt_link_probability[lgt_events]=global->param.lgt.tf_link_probability;
		
		}
	      }
	  }

	  // avoid overflow of the static arrays:
	  if (if_lgt_event and (lgt_events<MAX_LGT_EVENTS_PER_TIMESTEP)){
	    lgt_from[lgt_events]=lgt_cell_i / n_cells_per_process;
	    lgt_index_from[lgt_events]=lgt_cell_i % n_cells_per_process;
	    lgt_to[lgt_events]=lgt_cell_j / n_cells_per_process;
	    lgt_index_to[lgt_events]=lgt_cell_j % n_cells_per_process;
	    lgt_events++;
	  }

	} // end master rank lgt event

  
	// share information on lgt pairs across all mpi processes
#if _IF_MPI
	MPI_Bcast(lgt_from, MAX_LGT_EVENTS_PER_TIMESTEP, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(lgt_index_from, MAX_LGT_EVENTS_PER_TIMESTEP, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(lgt_to, MAX_LGT_EVENTS_PER_TIMESTEP, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(lgt_index_to, MAX_LGT_EVENTS_PER_TIMESTEP, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(module_sizes, MAX_LGT_EVENTS_PER_TIMESTEP, MPI_INT, 0, MPI_COMM_WORLD);
	MPI_Bcast(lgt_link_probability, MAX_LGT_EVENTS_PER_TIMESTEP, MPI_DOUBLE, 0, MPI_COMM_WORLD);
#else
	// Serial version:
	// nothing to do here
#endif


	// check if this cell is a donor/acceptor of an LGT fragment:
	for (int lgt_event_i=0; lgt_event_i<MAX_LGT_EVENTS_PER_TIMESTEP; lgt_event_i++){
	  // local LGT event
	  if ((rank==lgt_from[lgt_event_i]) and (rank==lgt_to[lgt_event_i])){
#if if_phylogenetic_print
	    // printout LGT event: 
	    out_data.str("");
	    out_data << epochID << "." << time_step_i
		     << ": " 
		     << lgt_from[lgt_event_i]<< "[" << lgt_index_from[lgt_event_i] <<"] (" 
		     << cell_coordinates[ lgt_from[lgt_event_i]*n_cells_per_process + lgt_index_from[lgt_event_i] ] % grid_max 
		     << "," 
		     << cell_coordinates[ lgt_from[lgt_event_i]*n_cells_per_process + lgt_index_from[lgt_event_i] ] / grid_max 
		     << ") ->- " 
		     << lgt_to[lgt_event_i]<< "[" << lgt_index_to[lgt_event_i] <<"] (" 
		     << cell_coordinates[ lgt_to[lgt_event_i]*n_cells_per_process + lgt_index_to[lgt_event_i] ] % grid_max 
		     << "," 
		     << cell_coordinates[ lgt_to[lgt_event_i]*n_cells_per_process + lgt_index_to[lgt_event_i] ] / grid_max 
		     << "); module_size= " << module_sizes[lgt_event_i];
#endif

#if  if_save_lgt_events_log
	    double F_before, F_after, F_sd;
	    // test fitness "before"
	    cells[ lgt_index_to[lgt_event_i] ]->test_fitness(signal_ptr, food, 1, 5, selection_period, F_before, F_sd);
	    // save cell "before
	    cells[ lgt_index_to[lgt_event_i] ]->save_ascii(lgt_index_to[lgt_event_i], -1, out_str);
	    lgt_events_log_file << out_str;
#endif

	    // copy a module from another cell 
	    lgt_module_tmp->copy_from_another_cell( cells[ lgt_index_from[lgt_event_i] ] );
	    // from the evolved network
#if if_HGT_from_evolved_cells
	    lgt_module_tmp->copy_from_another_cell( &evolved_cell );
#endif

	    lgt_module_tmp->reduce_to_module(module_sizes[lgt_event_i]);
	    cells[ lgt_index_to[lgt_event_i] ]->addmodule(lgt_module_tmp, lgt_link_probability[lgt_event_i]);

#if  if_save_lgt_events_log

	    // save module
	    lgt_module_tmp->save_ascii(lgt_index_to[lgt_event_i], 0, out_str);
	    lgt_events_log_file << out_str;

	    // test fitness "after"
	    cells[ lgt_index_to[lgt_event_i] ]->test_fitness(signal_ptr, food, 1, 5, selection_period, F_after, F_sd);

	    // save cell "after"
	    cells[ lgt_index_to[lgt_event_i] ]->save_ascii(lgt_index_to[lgt_event_i], 1, out_str);
	    lgt_events_log_file << out_str;

	    // add "before/after" info to the LGT even line 
	    out_data 
	      << " [[t: " << std::scientific << std::setprecision(4) << std::setw(12) << epochID+time_step_i*1.0/selection_period 
	      << " F: " << std::scientific << std::setprecision(6) << std::setw(14) << F_before 
	      << " => " << std::scientific << std::setprecision(6) << std::setw(14) << F_after <<" ]] ";

#endif


#if if_phylogenetic_print
	    out_data << " (LOCAL)\n";
	    std::cout << out_data.str();
#endif

#if  if_save_lgt_events_log
	    // save a copy of LGT event line to the log file:
	    lgt_events_log_file << out_data.str() << "(ABOVE)\n";
	    lgt_events_log_file << "----------------------------------------------------------------------\n" << std::flush;
#endif

	  }

	  // this mpi process has an LGT donor
	  else if (rank==lgt_from[lgt_event_i]){
	    lgt_module_tmp->copy_from_another_cell( cells[ lgt_index_from[lgt_event_i] ] );

#if if_HGT_from_evolved_cells
	    lgt_module_tmp->copy_from_another_cell( &evolved_cell );
#endif

	    lgt_module_tmp->reduce_to_module(module_sizes[lgt_event_i]);
#if _IF_MPI
	    lgt_module_tmp->MPI_Send_cell_copy( lgt_to[lgt_event_i] );
#endif
	  } 

	  // this mpi process has an LGT acceptor
	  else if (rank==lgt_to[lgt_event_i]){
	  

#if  if_save_lgt_events_log
	    double F_before, F_after, F_sd;

	    // test fitness "before"
	    cells[ lgt_index_to[lgt_event_i] ]->test_fitness(signal_ptr, food, 1, 5, selection_period, F_before, F_sd);

	    // save cell "before"
	    cells[ lgt_index_to[lgt_event_i] ]->save_ascii(lgt_index_to[lgt_event_i], -1, out_str);
	    lgt_events_log_file << out_str;
#endif

#if _IF_MPI
	    lgt_module_tmp->MPI_Recv_cell_copy( lgt_from[lgt_event_i] );
#endif

	    cells[ lgt_index_to[lgt_event_i] ]->addmodule(lgt_module_tmp, lgt_link_probability[lgt_event_i]);

#if  if_save_lgt_events_log

	    // save module
	    lgt_module_tmp->save_ascii(lgt_index_to[lgt_event_i], 0, out_str);
	    lgt_events_log_file << out_str;

	    // test fitness "after
	    cells[ lgt_index_to[lgt_event_i] ]->test_fitness(signal_ptr, food, 1, 5, selection_period, F_after, F_sd);

	    // save cell after
	    cells[ lgt_index_to[lgt_event_i] ]->save_ascii(lgt_index_to[lgt_event_i], 1, out_str);
	    lgt_events_log_file << out_str;
#endif


#if if_phylogenetic_print
	    // printout LGT event: 
	    out_data.str("");
	    out_data << epochID << "." << time_step_i
		     << ": " 
		     << lgt_from[lgt_event_i]<< "[" << lgt_index_from[lgt_event_i] <<"] (" 
		     << cell_coordinates[ lgt_from[lgt_event_i]*n_cells_per_process + lgt_index_from[lgt_event_i] ] % grid_max 
		     << "," 
		     << cell_coordinates[ lgt_from[lgt_event_i]*n_cells_per_process + lgt_index_from[lgt_event_i] ] / grid_max 
		     << ") ->- " 
		     << lgt_to[lgt_event_i]<< "[" << lgt_index_to[lgt_event_i] <<"] (" 
		     << cell_coordinates[ lgt_to[lgt_event_i]*n_cells_per_process + lgt_index_to[lgt_event_i] ] % grid_max 
		     << "," 
		     << cell_coordinates[ lgt_to[lgt_event_i]*n_cells_per_process + lgt_index_to[lgt_event_i] ] / grid_max 
		     << "); module_size= " << module_sizes[lgt_event_i];
#if  if_save_lgt_events_log
	    out_data << " [[t: " << std::scientific << std::setprecision(4) << std::setw(12) << epochID+time_step_i*1.0/selection_period 
		     << " F: " << std::scientific << std::setprecision(6) << std::setw(14) << F_before 
		     << " => " << std::scientific << std::setprecision(6) << std::setw(14) << F_after <<" ]] ";
#endif

	    out_data << "\n";
	    std::cout << out_data.str();
#endif

#if  if_save_lgt_events_log
	    // save a copy of LGT event string into the log file -- to label cells
	    lgt_events_log_file << out_data.str() << "(ABOVE)\n";
	    lgt_events_log_file << "----------------------------------------------------------------------\n" << std::flush;
#endif
	  } // else if

	}// for all lgt events




	
	//----------------------------------------------------------------------
	// Cell Division VERSION D

	if ((time_step_i % sync_freq)==0){
	  // if _one_ of the cells in this MPI process need to be divided/replaced
	  if_this_divide=if_this_replace=false;

	  // set infor of local cells for MAX & MIN search
	  for (int cell_idx=0; cell_idx<cells.size(); cell_idx++){
	    min_e_dist_cell_this[cell_idx].value = 
	      max_e_cell_this[cell_idx].value = cells[cell_idx]->getenergy();
	    min_e_dist_cell_this[cell_idx].rank =
	      max_e_cell_this[cell_idx].rank = rank;
	  }

	  // _always_ check if the max. energy is above the division threshold, if not -- skip division
#if _IF_MPI
	  MPI_Allreduce( max_e_cell_this, max_e_cell, cells.size(), MPI_DOUBLE_INT, MPI_MAXLOC, MPI_COMM_WORLD );
#else
	  // Serial version:
	  for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	    {
	      max_e_cell[cell_idx].value = max_e_cell_this[cell_idx].value;
	      max_e_cell[cell_idx].rank  = max_e_cell_this[cell_idx].rank;
	    }
#endif

	  // another reduction -- find global max value (& rank) -- and put it into max_e, save original index
	  max_e_index = find_maxloc(max_e_cell, cells.size(), max_e);
 
	  // broadcast coordinate of the cell with maximum energy
	  max_e_coord = cells[max_e_index]->getcoord();
#if _IF_MPI
	  MPI_Bcast(&max_e_coord, 1, MPI_INT, max_e_cell[max_e_index].rank, MPI_COMM_WORLD);
#else
	  // Serial version:
	  // nothing to do here 
#endif

	  // update intercell distance (except cells that are excluded from the search by DBL_MAX)
	  for (int cell_idx=0; cell_idx<cells.size(); cell_idx++){
	    min_e_dist_cell_this[cell_idx].dist = calc_dist(grid_max, max_e_coord, cells[cell_idx]->getcoord());
	    // exlude dividing cell itself:
	    if ((max_e_index==cell_idx)and(max_e_cell[max_e_index].rank==rank))
	      min_e_dist_cell_this[cell_idx].dist = DBL_MAX;
	  }



	  // search for pairs until there is a cell to divide;
	  // only one division over mpi to/from any process (first pair)
	  // all local copies are done right away (unlimited number of local divisions)
	  while (max_e.value>cell_division_threshold) {
 

	    // find cell with a global min energy
	    if (global->param.grid.if_global_min==1){

#if _IF_MPI
	      MPI_Allreduce( min_e_dist_cell_this, min_e_dist_cell, cells.size(), MPI_DOUBLE_DOUBLE_INT, MPI_MIN_E_NODIST_LOC, MPI_COMM_WORLD);
#else
	      // Serial version:
	      for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
		{
		  min_e_dist_cell[cell_idx].value = min_e_dist_cell_this[cell_idx].value;
		  min_e_dist_cell[cell_idx].dist  = min_e_dist_cell_this[cell_idx].dist;
		  min_e_dist_cell[cell_idx].rank  = min_e_dist_cell_this[cell_idx].rank;
		}
#endif

	      // another reduction -- find global min value (& rank) -- and put it into min_e, save original index
	      min_e_dist_index = find_min_e_dist_loc(min_e_dist_cell, cells.size(), min_e_dist);

	    }
	    //else find cell usinf F(E,dist) function
	    else {
#if _IF_MPI
	      MPI_Allreduce( min_e_dist_cell_this, min_e_dist_cell, cells.size(), MPI_DOUBLE_DOUBLE_INT, MPI_MIN_E_DIST_LOC, MPI_COMM_WORLD);
#else
	  for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	    {
	      min_e_dist_cell[cell_idx].value = min_e_dist_cell_this[cell_idx].value;
	      min_e_dist_cell[cell_idx].dist  = min_e_dist_cell_this[cell_idx].dist;
	      min_e_dist_cell[cell_idx].rank  = min_e_dist_cell_this[cell_idx].rank;
	    }
#endif
	      // another reduction -- find global min value (& rank) -- and put it into min_e, save original index
	      min_e_dist_index = find_min_e_dist_loc(min_e_dist_cell, cells.size(), min_e_dist);

	    }


	    

	    //----------------------------------------------------------------------
	    // LOCAL DIVISION
	    if ((max_e.rank==rank) and (min_e_dist.rank==rank) ){

	      // coordinate of a new cell is the same as of the previous one
	      new_coordinate_original=cells[max_e_index]->getcoord();
	    
	      // get coordinates with gaussian distribution (and withing a 2D grid boundary):
	      do {
		R = gaussian(0,global->param.grid.division_copy_sdev, global->param.globals.idum, global->param.globals.itemp); // returns positive and negative R-s
		alpha = PI*my_rand(global->param.globals.idum, global->param.globals.itemp);
		x_shift = ROUND( cos(alpha)*R );
		y_shift = ROUND( sin(alpha)*R );
	      } while ( 
		       ((new_coordinate_original / grid_max) + y_shift)<0 or 
		       ((new_coordinate_original / grid_max) + y_shift)>=grid_max or
		       ((new_coordinate_original % grid_max) + x_shift)<0 or 
		       ((new_coordinate_original % grid_max) + x_shift)>=grid_max
			);
	      new_coordinate = new_coordinate_original + x_shift + y_shift * grid_max;

	      /*
	      // if there are too many cells at this location divide into a random point in 3x3 matrix centered around this location
	      if (cell_density[new_coordinate]>=global->param.grid.max_cells) {
	      // throw two dices x=0..2 and y=0..2 to get a coordinate in 3x3 matrix
	      x_shift = ROUND( my_rand(global->param.globals.idum, global->param.globals.itemp)*2 )-1;
	      y_shift = ROUND( my_rand(global->param.globals.idum, global->param.globals.itemp)*2 )-1;

	      // check that not past the boundaries
	      if ( ((new_coordinate % grid_max) + x_shift)<0 or ((new_coordinate % grid_max) + x_shift)>=grid_max )
	      x_shift*=-1;
	      if ( ((new_coordinate / grid_max) + y_shift)<0 or ((new_coordinate / grid_max) + y_shift)>=grid_max )
	      y_shift*=-1;
	      new_coordinate += x_shift + y_shift * grid_max;
	      }
	      */

	      cell_density_change_this[ cells[min_e_dist_index]->getcoord() ]--;
	      cell_density_change_this[new_coordinate] ++;

#if if_phylogenetic_print
	      out_data.str("");
	      out_data << epochID << "." << time_step_i << ": " << max_e.rank << "[" << max_e_index << "]."<<cells[max_e_index]->getID() 
		       << " (" << new_coordinate_original % grid_max << ',' << new_coordinate_original / grid_max
		       << ") --> "
		       << min_e_dist.rank << "[" << min_e_dist_index << "]." << next_cell_ID 
		       << " (" << new_coordinate % grid_max << ',' << new_coordinate / grid_max
		       << ") {replaced cell: "
		       << min_e_dist.rank <<"["<< min_e_dist_index<<"]." << cells[min_e_dist_index]->getID() << " ("
		       << cells[min_e_dist_index]->getcoord() % grid_max << "," << cells[min_e_dist_index]->getcoord() / grid_max
		       << ")} LOCAL\n";
	      std::cout << out_data.str() << std::flush;
#endif

	      cells[min_e_dist_index]->setv_history(0,-1,selection_period,0); 

	      cells[min_e_dist_index]->copy_from_another_cell( cells[max_e_index] );
	      cells[min_e_dist_index]->setv_history(0,1,selection_period,0); 
	      cells[min_e_dist_index]->setID(next_cell_ID);
	      next_cell_ID++; 
	      cells[min_e_dist_index]->setcoord(new_coordinate);
	      cell_coordinates_this[min_e_dist_index] = new_coordinate; 

						

	      // energy splits in half between the old and new cell:
	      double new_energy = cells[max_e_index]->getenergy()/2;
	      cells[min_e_dist_index]->setenergy(0,1);
	      cells[min_e_dist_index]->setenergy(new_energy, 0);
	      cells[max_e_index]->setenergy(0,1);
	      cells[max_e_index]->setenergy(new_energy, 0);

	      // exclude this pair from the next search:
	      max_e_cell_this[max_e_index].value=-1;
	      min_e_dist_cell_this[min_e_dist_index].value=DBL_MAX;
	      min_e_dist_cell_this[min_e_dist_index].dist=DBL_MAX;
	    }
	    // END LOCAL DIVISION
	    //----------------------------------------------------------------------
	    
	    // if this is a cell to divide (threshold is already verified)
	    else if (max_e.rank==rank){
	      if_this_divide=true;
	      rank_to_replace_cell=min_e_dist.rank;
	      index_to_replace_cell=min_e_dist_index;
	      index_to_divide_cell=max_e_index;
	      // exclude this process (all local cells) from the next MIN/MAXsearch
	      for (int i=0; i< cells.size(); i++){
		max_e_cell_this[i].value=-1;
		min_e_dist_cell_this[i].value=DBL_MAX; 
		min_e_dist_cell_this[i].dist=DBL_MAX; 
	      }
	    }

	    // if this cell to be replaced:
	    else if (min_e_dist.rank==rank){
	      if_this_replace=true;
	      rank_to_divide_cell=max_e.rank;
	      index_to_replace_cell=min_e_dist_index;
	      index_to_divide_cell=max_e_index;
	      // exclude this process (all local cells) from the next MIN/MAXsearch
	      for (int i=0; i< cells.size(); i++){
		max_e_cell_this[i].value=-1;
		min_e_dist_cell_this[i].value=DBL_MAX;
		min_e_dist_cell_this[i].dist=DBL_MAX;
	      }
	    }

	    // find next max energy
#if _IF_MPI
	    MPI_Allreduce( max_e_cell_this, max_e_cell, cells.size(), MPI_DOUBLE_INT, MPI_MAXLOC, MPI_COMM_WORLD);
#else
	    // Serial version:
	    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	      {
		max_e_cell[cell_idx].value = max_e_cell_this[cell_idx].value;
		max_e_cell[cell_idx].rank  = max_e_cell_this[cell_idx].rank;
	      }
#endif

	    // another reduction -- find global max value (& rank) -- and put it into max_e, save original index
	    max_e_index = find_maxloc(max_e_cell, cells.size(), max_e);
	    

	    // broadcast coordinate of the cell with maximum energy
	    max_e_coord = cells[max_e_index]->getcoord();
#if _IF_MPI
	    MPI_Bcast(&max_e_coord, 1, MPI_INT, max_e_cell[max_e_index].rank, MPI_COMM_WORLD);
#else
	    // Serial version:
	    // nothing to do here 
#endif

	    // update intercell distance (except cells that are excluded from the search by DBL_MAX)
	    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++){
	      if (min_e_dist_cell_this[cell_idx].dist!=DBL_MAX){
		min_e_dist_cell_this[cell_idx].dist = calc_dist(grid_max, max_e_coord, cells[cell_idx]->getcoord());
		// exlude dividing cell itself:
		if ((max_e_index==cell_idx)and(max_e_cell[max_e_index].rank==rank))
		  min_e_dist_cell_this[cell_idx].dist = DBL_MAX;
	      }
	    }
	  } // end while max energy > cell_division_threshold




	  if  (if_this_replace){
	    //delete v_history 
	    cells[index_to_replace_cell]->setv_history(0,-1,selection_period,0); 

	    // first part of the message
#if if_phylogenetic_print
	    out_data.str("");
	    out_data << epochID << "." << time_step_i << ": " << rank_to_divide_cell << "[" << index_to_divide_cell << "].";

	    old_coordinate = cells[index_to_replace_cell]->getcoord();
	    old_cell_ID = cells[index_to_replace_cell]->getID();
#endif

	    // update local cell density 1/2
	    cell_density_change_this[ cells[index_to_replace_cell]->getcoord() ]--;

	    // **** MPI copy ****
#if _IF_MPI
	    cells[index_to_replace_cell]->MPI_Recv_cell_copy(rank_to_divide_cell);
#else
	    // Serial version:
	    cells[index_to_replace_cell]->copy_from_another_cell( cells[ rank_to_divide_cell ] );
#endif

	    new_coordinate_original=cells[index_to_replace_cell]->getcoord();

	    // get coordinates with gaussian distribution (and withing a 2D grid boundary):
	    do {
	      R = gaussian(0,global->param.grid.division_copy_sdev, global->param.globals.idum, global->param.globals.itemp); // returns positive and negative R-s
	      alpha = PI*my_rand(global->param.globals.idum, global->param.globals.itemp);
	      x_shift = ROUND( cos(alpha)*R );
	      y_shift = ROUND( sin(alpha)*R );
	    } while ( 
		     ((new_coordinate_original / grid_max) + y_shift)<0 or 
		     ((new_coordinate_original / grid_max) + y_shift)>=grid_max or
		     ((new_coordinate_original % grid_max) + x_shift)<0 or 
		     ((new_coordinate_original % grid_max) + x_shift)>=grid_max
		      );
	    new_coordinate = new_coordinate_original + x_shift + y_shift * grid_max;

	    /*
	    // if there are too many cells at this location divide into a random point in 3x3 matrix centered around this location
	    if (cell_density[new_coordinate]>=global->param.grid.max_cells) {

	    // throw two dices x=0..2 and y=0..2 to get a coordinate in 3x3 matrix
	    x_shift = ROUND( my_rand(global->param.globals.idum, global->param.globals.itemp)*2 )-1;
	    y_shift = ROUND( my_rand(global->param.globals.idum, global->param.globals.itemp)*2 )-1;
	    // check that not past the boundaries
	    if ( ((new_coordinate % grid_max) + x_shift)<0 or ((new_coordinate % grid_max) + x_shift)>=grid_max )
	    x_shift*=-1;
	    if ( ((new_coordinate / grid_max) + y_shift)<0 or ((new_coordinate / grid_max) + y_shift)>=grid_max )
	    y_shift*=-1;
	    new_coordinate += x_shift + y_shift * grid_max;
	    }
	    */

	    cells[index_to_replace_cell]->setcoord(new_coordinate);
	    cell_coordinates_this[index_to_replace_cell] = new_coordinate; 

	    // update local cell density 2/2
	    cell_density_change_this[ new_coordinate ]++;

	    // second part of the message
#if if_phylogenetic_print
	    out_data << cells[index_to_replace_cell]->getID() // original ID from the dividing cell
		     << " (" << new_coordinate_original % grid_max << ',' << new_coordinate_original / grid_max
		     << ") --> "
		     << rank << "[" << index_to_replace_cell << "]." << next_cell_ID // next ID at this process
		     << " (" << cells[index_to_replace_cell]->getcoord() % grid_max << ',' << cells[index_to_replace_cell]->getcoord() / grid_max
		     << ") {replaced cell: " 
		     << rank << "[" << index_to_replace_cell << "]." << old_cell_ID << " (" // previous ID at this process
		     << old_coordinate % grid_max << "," << old_coordinate / grid_max
		     <<")}\n";
	    std::cout << out_data.str() << std::flush;
#endif
	  
	    // update ID
	    cells[index_to_replace_cell]->setID(next_cell_ID);
	    next_cell_ID++;
  
	    //allocate v_history 
	    cells[index_to_replace_cell]->setv_history(0,1,selection_period,0); 
	  
	    // energy splits in half between the old and new cell:
	    double new_energy = cells[index_to_replace_cell]->getenergy()/2;

	    cells[index_to_replace_cell]->setenergy(0,1);
	    cells[index_to_replace_cell]->setenergy(new_energy, 0);
	  } 
      

	  if  (if_this_divide){
	    // MPI copy
#if _IF_MPI
	    cells[index_to_divide_cell]->MPI_Send_cell_copy(rank_to_replace_cell);
#else
	    // Serial version:  
	    // nothing to do here
#endif
	    
	    // energy splits in half between the old and new cell (FOR BOTH LOCAL AN OVER-MPI DIVISIONS):
	    double new_energy = cells[index_to_divide_cell]->getenergy()/2;
	    cells[index_to_divide_cell]->setenergy(0,1);
	    cells[index_to_divide_cell]->setenergy(new_energy, 0);
	  }


	} // end if syncronize at this time step

      } // END if not the last epoch

    } //end of time_step_i 

#if if_print_benchmark 
    // after the end of the epoch calculate an average time for every cell

    // print performance statistics for each cell in the process (time ave & stdev; groth rate ave & stdev):
    out_data.str("");
    for(int cell_idx=0; cell_idx<cells.size(); cell_idx++){
      double time_ave=0, time_stdev=0, grate_ave=0, grate_stdev=0, size_ave=0, size_stdev=0, link_ave=0, link_stdev=0;
      //averages:
      for (int ts=0; ts<selection_period; ts++){
	time_ave += mtimes[cell_idx*selection_period + ts];
	grate_ave += grates[cell_idx*selection_period + ts];
	size_ave += sizes[cell_idx*selection_period + ts];
	link_ave += links[cell_idx*selection_period + ts];
      }
      time_ave /= selection_period;
      grate_ave /= selection_period;
      size_ave /= selection_period;
      link_ave /= selection_period;

      // st. deviations:
      for (int ts=0; ts<selection_period; ts++){
	time_stdev += pow((mtimes[cell_idx*selection_period + ts] - time_ave), 2);
	grate_stdev += pow((grates[cell_idx*selection_period + ts] - grate_ave), 2);
	size_stdev += pow((sizes[cell_idx*selection_period + ts] - size_ave), 2);
	link_stdev += pow((links[cell_idx*selection_period + ts] - link_ave), 2);
      }
      time_stdev = sqrt(time_stdev / selection_period);
      grate_stdev = sqrt(grate_stdev / selection_period);
      size_stdev = sqrt(size_stdev / selection_period);
      link_stdev = sqrt(link_stdev / selection_period);

      out_data << "BENCHMARK " 
	       << std::fixed << std::setprecision(2) << std::setw(10)
	       << epochID << " " << rank << "[" << cell_idx << "] " << time_ave << "(" << time_stdev << ") " << grate_ave << "(" << grate_stdev << ")  " 
	       << size_ave << "(" << size_stdev << ") "  << link_ave << "(" << link_stdev <<")\n";
    }
    std::cout << out_data.str() << std::flush;

#endif
 
    // update fitness and save statistics at the end of every epoch
#if if_save_output
    // update fitness:
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++)
      cells[cell_idx]->setfitness(  pearson(cells[cell_idx]->getv_history_pointer()[5], food, selection_period) );
    
    // Collect statistics across all MPI processes:
    out_data.str("");
    out_data << std::setw(6) << epochID;

    // total energy, fitness
    double *log_total_dbl_this, *log_total_dbl;
    // maximum fitness & process rank with maximum fitness
    double_int *log_max_dbl_this, *log_max_dbl;
    // total number of mild & strong mutations, total number of normal nodes
    int *log_total_int_this, *log_total_int;

    double reduced_dbl;
    double_int reduced_dbl_int;
    int reduced_int;

    log_total_dbl_this = new double[cells.size()];
    log_total_dbl = new double[cells.size()];
    log_max_dbl_this = new double_int[cells.size()];
    log_max_dbl = new double_int[cells.size()];
    log_total_int_this = new int[cells.size()];
    log_total_int = new int[cells.size()];

    // get total energy of all cells, save average energy
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++)
      log_total_dbl_this[cell_idx]=cells[cell_idx]->getenergy();
#if _IF_MPI
    MPI_Reduce( log_total_dbl_this, log_total_dbl, cells.size(), MPI_DOUBLE, MPI_SUM, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
      log_total_dbl[cell_idx]  = log_total_dbl_this[cell_idx];
#endif
    reduced_dbl = find_sum_dbl(log_total_dbl, cells.size());
    out_data  << std::scientific << std::setprecision(6) << std::setw(14)
	      << reduced_dbl/n_cells;

    // get total fitness of all cells, save average fitness
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++)
      log_total_dbl_this[cell_idx]=cells[cell_idx]->getfitness();
#if _IF_MPI
    MPI_Reduce( log_total_dbl_this, log_total_dbl, cells.size(), MPI_DOUBLE, MPI_SUM, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
      log_total_dbl[cell_idx]  = log_total_dbl_this[cell_idx];
#endif
    reduced_dbl = find_sum_dbl(log_total_dbl, cells.size());
    out_data  << std::scientific << std::setprecision(6) << std::setw(14)
	      << reduced_dbl/n_cells; 

    // get max fitness and rank, save max fitness and cell ID
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
      log_max_dbl_this[cell_idx].value=cells[cell_idx]->getfitness();
      log_max_dbl_this[cell_idx].rank=rank;
    }
#if _IF_MPI
    MPI_Reduce( log_max_dbl_this, log_max_dbl, cells.size(), MPI_DOUBLE_INT, MPI_MAXLOC, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
      {
	log_max_dbl[cell_idx].rank  = log_max_dbl_this[cell_idx].rank;
	log_max_dbl[cell_idx].value  = log_max_dbl_this[cell_idx].value;
      }
#endif
    reduced_int = find_maxloc(log_max_dbl, cells.size(), reduced_dbl_int);
    out_data  << std::scientific << std::setprecision(6) << std::setw(14)
	      << reduced_dbl_int.value
	      << std::fixed << std::setw(7)
	      << reduced_dbl_int.rank << "[" << reduced_int << "]"; 
    // save mild mutation rate of the fittest cell:

    double tmp_mut_rate;
#if _IF_MPI
    MPI_Status mpi_status;
#endif

    //broadcst (!) rank of the process with the fittest cell
#if _IF_MPI
    MPI_Bcast(&reduced_dbl_int.rank, 1, MPI_INT, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    // no need to do anything here
#endif
   
    // local
    if (master_rank==reduced_dbl_int.rank){
      tmp_mut_rate = cells[reduced_int]->getmutation(3);
    }
    else{ // non-local update
      if (rank==master_rank){
	// recieve the mild mutation rate of the fittest cell  
#if _IF_MPI
	MPI_Recv(&tmp_mut_rate, 1, MPI_DOUBLE, reduced_dbl_int.rank, 239, MPI_COMM_WORLD, &mpi_status);
#else
	// Serial version:
	// no need to do anything here
	
#endif
      }
      // if this process contain cell with the maximum fitness, send mild mutation rate:
      if (rank==reduced_dbl_int.rank){
	tmp_mut_rate = cells[reduced_int]->getmutation(3);
#if _IF_MPI
	MPI_Send(&tmp_mut_rate, 1, MPI_DOUBLE, master_rank, 239, MPI_COMM_WORLD);
#else
	// Serial version:
	// no need to do anything here
#endif
      }
    }

    // get and save total number of mild mutations
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
      log_total_int_this[cell_idx]=0;
      for (int node_i=0; node_i<cells[cell_idx]->getn_nodes(); node_i++)
	log_total_int_this[cell_idx]+=cells[cell_idx]->getmutation_counter(1, node_i);
    }

#if _IF_MPI
    MPI_Reduce( log_total_int_this, log_total_int, cells.size(), MPI_INT, MPI_SUM, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	log_total_int[cell_idx] = log_total_int_this[cell_idx];
#endif

    reduced_int = find_sum_int(log_total_int, cells.size());
    out_data  << std::fixed << std::setw(5)
	      << reduced_int; 


    // get and save total number of strong mutations
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
      log_total_int_this[cell_idx]=0;
      for (int node_i=0; node_i<cells[cell_idx]->getn_nodes(); node_i++)
	log_total_int_this[cell_idx]+=cells[cell_idx]->getmutation_counter(2, node_i);
    }
#if _IF_MPI
    MPI_Reduce( log_total_int_this, log_total_int, cells.size(), MPI_INT, MPI_SUM, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	log_total_int[cell_idx] = log_total_int_this[cell_idx];

#endif
    reduced_int = find_sum_int(log_total_int, cells.size());
    out_data  << std::fixed << std::setw(5)
	      << reduced_int; 
 

    // get total number of nodes for all cells, save average number of nodes per cell
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++)
      log_total_int_this[cell_idx]=cells[cell_idx]->getn_nodes();
#if _IF_MPI
    MPI_Reduce( log_total_int_this, log_total_int, cells.size(), MPI_INT, MPI_SUM, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	log_total_int[cell_idx] = log_total_int_this[cell_idx];
#endif
    reduced_dbl = find_sum_int(log_total_int, cells.size());
    out_data  << std::fixed << std::setprecision(1) << std::setw(9)
	      << reduced_dbl/n_cells; 

    // get average mild mutation rate
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++)
      log_total_dbl_this[cell_idx]=cells[cell_idx]->getmutation(3);
#if _IF_MPI
    MPI_Reduce( log_total_dbl_this, log_total_dbl, cells.size(), MPI_DOUBLE, MPI_SUM, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	log_total_dbl[cell_idx] = log_total_dbl_this[cell_idx];
#endif
    reduced_dbl = find_sum_dbl(log_total_dbl, cells.size());
    out_data  << std::scientific << std::setprecision(2) << std::setw(10)
	      << reduced_dbl/n_cells; 

    // get maximum mild mutation rate
    for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
      log_max_dbl_this[cell_idx].value=cells[cell_idx]->getmutation(3);
      log_max_dbl_this[cell_idx].rank=rank;
    }
#if _IF_MPI
    MPI_Reduce( log_max_dbl_this, log_max_dbl, cells.size(), MPI_DOUBLE_INT, MPI_MAXLOC, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	log_total_dbl[cell_idx] = log_total_dbl_this[cell_idx];
#endif
    reduced_int = find_maxloc(log_max_dbl, cells.size(), reduced_dbl_int);
    out_data  << std::scientific << std::setprecision(2) << std::setw(10)
	      << reduced_dbl_int.value
	      << std::fixed << std::setw(5)
	      << reduced_dbl_int.rank << "[" << reduced_int << "]"; 

    // print mild mutation rate of the fittest organism:
    out_data  << std::scientific << std::setprecision(2) << std::setw(10)
	      <<  tmp_mut_rate; 

    // end of line for this epoch
    out_data  << "\n";

    delete[] log_total_dbl_this;
    delete[] log_total_dbl;
    delete[] log_max_dbl_this;
    delete[] log_max_dbl;
    delete[] log_total_int_this;
    delete[] log_total_int;

    if (rank==master_rank)
      statistics_file << out_data.str() << std::flush;


    // ----------------------------------------------------------------------
    // gather and print all labels:
    
    int *labels = new int [n_cells];
    int *labels_this = new int [cells.size()];
    for (int i=0; i<cells.size(); i++)
      labels_this[i]=cells[i]->getlabel();

#if _IF_MPI    
    MPI_Gather(labels_this, cells.size(), MPI_INT, labels, cells.size(), MPI_INT, master_rank, MPI_COMM_WORLD);
#else
    // Serial version:
    for (int cell_idx=0; cell_idx<cells.size(); cell_idx++)
	labels[cell_idx] = labels_this[cell_idx];
#endif
 

#if IF_PRINT_LABELS
    if (rank==master_rank){
      out_data.str("");
      out_data << epochID << ".end: LABELS: ";
      // count unique labels
      std::vector<int> labels_v (labels, labels+n_cells);
      std::sort( labels_v.begin(), labels_v.end() );
      int counter=1;
      for(int i=1; i<n_cells; i++){
	if (labels_v[i]==labels_v[i-1])
	  counter++;
	else{
 	  // save previous group:
	  out_data << counter << "@" << labels_v[i-1] << " ";
	  counter=1;
	}
      }
      // save last group:
      out_data << counter << "@" << labels_v[n_cells-1] << "\n";
      std::cout << out_data.str() << std::flush;
    }
#endif

    delete[] labels;
    delete[] labels_this;
    // ----------------------------------------------------------------------

#endif


    //----------------------------------------------------------------------

    // save data every save_cell_info_freq, and the last epoch [units: epochs]
    if ((epochID%save_cell_info_freq == 0)or(epochID==number_of_epochs-1)) {

#if if_save_output
  
      // save 2D density of cells
      if (rank==master_rank){
	out_data.str("");
	save_grid(grid_max, grid_n_points/grid_max, cell_density, out_str);
	out_data << out_str << "\n";
	cell_density_file << out_data.str() << std::flush;
      }
 
      // save cell info (fitness is already updated)
      out_data.str("");
      for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
#if _USE_HDF5
	// save current cell into the string
	cells[cell_idx]->save(cell_idx, epochID, outfile);
#else
	cells[cell_idx]->save(cell_idx, epochID, out_str);
	out_data << out_str;
#endif
      }
#if !_USE_HDF5
      // save all local cell info to file
      cell_info_file << out_data.str() << std::flush;
#endif

      // save mutation history info:
      out_data.str("");
      for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
	// create a comment line:
	out_data << rank << "[" << cell_idx << "]." << cells[cell_idx]->getID() << "  ("
		 << "in the end of epoch " << epochID << ")"
		 << " n_nodes=" << cells[cell_idx]->getn_nodes() 
		 << "\n";
	
	// save mutation parameters
	out_data << std::scientific << std::setprecision(6)
		 << cells[cell_idx]->getmutation(1) << " " 
		 << cells[cell_idx]->getmutation(2) << " " 
		 << cells[cell_idx]->getmutation(3) << " "
		 << cells[cell_idx]->getmutation(4) << " " 
		 << cells[cell_idx]->getmutation(5) << "\n";
	
	// save history of mild mutataions
	for (int node_i=0; node_i<cells[cell_idx]->getn_nodes(); node_i++)
	  out_data << cells[cell_idx]->getmutation_counter(1, node_i) << " ";
	out_data << "\n";
	
	// save history of strong mutataions
	for (int node_i=0; node_i<cells[cell_idx]->getn_nodes(); node_i++)
	  out_data << cells[cell_idx]->getmutation_counter(2, node_i) << " ";
	out_data << "\n";

	// add an empty line between cells
	out_data << "\n";
      }
      mutation_history_file << out_data.str() << std::flush;
#endif

#if if_save_v_history
      // save current v_history (one file per process now)
      for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
	cells[cell_idx]->save_v_history_ascii(out_str, cell_idx, epochID);
	node_values_history_file << out_str << std::flush;
      }
#endif

    } // end save cell info


  } // end of epochID 


  //----------------------------------------------------------------------
  // save last states of cells:
#if _USE_HDF5
  H5Fflush(outfile, H5F_SCOPE_GLOBAL);
  H5Fclose(outfile);
  H5Pclose(fapl);

#else
  // update fitness and save last cell info:
  std::stringstream cell_info_lastfname;
  cell_info_lastfname << work_folder_name << "last_cell_info__rank" << rank << ".data";
  ofstream cell_info_lastfile;
  cell_info_lastfile.open(cell_info_lastfname.str().c_str(), std::ios::out | std::ios::trunc);
  out_data.str("");
  for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
    // update fitness
    cells[cell_idx]->setfitness(  pearson(cells[cell_idx]->getv_history_pointer()[5], food, selection_period)  );
    
    // save current cell into the string
    cells[cell_idx]->save_ascii(cell_idx, -1, out_str);
    out_data << out_str;
  }
  // save all local cell info to file
  cell_info_lastfile << out_data.str();
  cell_info_lastfile.flush();
  cell_info_lastfile.close();

  // save last v_history (one file per process now)
  std::stringstream node_values_history_lastfname;
  node_values_history_lastfname << work_folder_name << "last_node_values_history__rank" << rank << ".data";
  ofstream node_values_history_lastfile;
  node_values_history_lastfile.open(node_values_history_lastfname.str().c_str(), std::ios::out | std::ios::trunc);
  for (int cell_idx=0; cell_idx<n_cells_per_process; cell_idx++){
    cells[cell_idx]->save_v_history_ascii(out_str, cell_idx, -1);
    node_values_history_lastfile << out_str;
  }
  node_values_history_lastfile.flush();
  node_values_history_lastfile.close();    
  //----------------------------------------------------------------------
#endif


  for (int i=0;i<global->param.general.n_signals;i++)
    delete[] signal_ptr[i];
  delete[] signal_ptr;
      
  delete[] food;

#if if_save_output    
  // close local output files
#if !_USE_HDF5
  cell_info_file.flush();
  cell_info_file.close();
  mutation_history_file.flush();
  mutation_history_file.close();
#endif
  // close global output files
  if (rank==master_rank){
    output_file.flush();
    output_file.close();
    statistics_file.flush();
    statistics_file.close();
    cell_density_file.flush();
    cell_density_file.close();
  }
#endif

#if if_save_v_history
  node_values_history_file.flush();    
  node_values_history_file.close();    
#endif

#if  if_save_lgt_events_log
  lgt_events_log_file.flush();    
  lgt_events_log_file.close();
#endif

#if  if_save_mutation_events_log
  mutation_events_log_file.flush();    
  mutation_events_log_file.close();
#endif


    
  // delete arrays used in mpi_reduce functions in energy synchronization
  delete[] min_e_dist_cell;
  delete[] max_e_cell;
  delete[] min_e_dist_cell_this;
  delete[] max_e_cell_this;

  delete[] cell_density;
  delete[] cell_density_change;
  delete[] cell_density_change_this;

  delete[] cell_coordinates;
  delete[] cell_coordinates_this;

  delete lgt_module_tmp;
  delete [] lgt_from;
  delete [] lgt_index_from;
  delete [] lgt_to;
  delete [] lgt_index_to;
  delete [] module_sizes;
  delete [] lgt_link_probability;

#ifdef if_print_benchmark 
  delete [] mtimes;
  delete [] grates;
  delete [] sizes;
  delete [] links;
#endif

  // delete cells;
  for (int i=0; i<cells.size(); i++){
    cells[i]->setv_history(0,-1,selection_period,0); 
#if _IF_MPI
    cells[i]->free_MPI_types();
#endif
    delete cells[i];
  }
  cells.clear();

#if if_HGT_from_evolved_cells
  evolved_cell.setv_history(0,-1,selection_period,0); 
#if _IF_MPI
  evolved_cell.free_MPI_types();
#endif
#endif

  // delete global parameters singletone instance
  global->destroyInstance();

  finalize();

  return 0;
}
