/*! 
 * \brief Definition of MPI data types and operators for EVE
 */

#ifndef _MY_MPI_TYPES_
#define _MY_MPI_TYPES_

#include "generic_header.h"

#if _IF_MPI
     #include "mpi.h"
#endif

// an instance of the global parameters singleton
#include "eve_parameters.h"
Eve_parameters* global;
void create_global_eve_parameters()
{
  global = Eve_parameters::getInstance();
}

//! stores for example: energy, distance and rank
struct double_double_int{
  double value;
  double dist;
  int rank;
};


#if _IF_MPI
MPI_Datatype MPI_DOUBLE_DOUBLE_INT;

void create_MPI_DOUBLE_DOUBLE_INT()
{

  double_double_int st;
  
  int s_size=3; // number of blocks in the structure
  MPI_Aint _addresses[s_size]; // address of the begining of each block (of one element)
  MPI_Aint _offsets[s_size]; // offset from the begining (first block) -- will be calculated
  int _block_len[s_size]; // length of each block (=1)
  MPI_Datatype _old_types[s_size]; // types of elements in each block (only one element per block)
  
  MPI_Address(&st.value,      &_addresses[0]);
  _old_types[0] = MPI_DOUBLE;

  MPI_Address(&st.dist,      &_addresses[1]);
  _old_types[1] = MPI_DOUBLE;

  MPI_Address(&st.rank,      &_addresses[2]);
  _old_types[2] = MPI_INT;
	
  // all blocks have length 1 (except few reassigned below)  
  for ( int i=0; i<s_size; i++)
    _block_len[i] = 1;
  // calculate offsets from the beginig
  for ( int i=0; i<s_size; i++)
    _offsets[i] = _addresses[i]-_addresses[0];
  // create cell_data_tMPI
  MPI_Type_struct( s_size, _block_len, _offsets, _old_types, &MPI_DOUBLE_DOUBLE_INT );
  MPI_Type_commit( &MPI_DOUBLE_DOUBLE_INT );
}
#endif

//!returns a function energy (just energy)
double F_e_nodist(double e, double dist)
{
  // v.1: returns a mimial energy
  return e;
}

//!returns a function of distance and energy
double F_e_dist(double e, double dist)
{
  // v.2: returns min of normolized sqrt(e*dist); dist+1 is used to eliminate zero-distance automatic selection
  return ( ( e/cell_division_threshold ) *  pow((dist+1)/global->param.grid.dmax, 2) );

  
  //return dist;
}

#if _IF_MPI
//! returnes MIN between in & out using F_e() function is defined above
void min_e_nodist_loc(double_double_int *invec, double_double_int *inoutvec, int *len, MPI_Datatype *dtype)
{
  // operation must be commutitave and associative (ifcommute=1)
  for (int i=0; i<*len; i++){
    if ( F_e_nodist(invec[i].value, invec[i].dist) < F_e_nodist(inoutvec[i].value, inoutvec[i].dist) ){
      inoutvec[i].value = invec[i].value;
      inoutvec[i].dist = invec[i].dist;
      inoutvec[i].rank = invec[i].rank;
    }
  }
}

//! returnes MIN between in & out using F_e_dist() function is defined above
void min_e_dist_loc(double_double_int *invec, double_double_int *inoutvec, int *len, MPI_Datatype *dtype)
{
  // operation must be commutitave and associative (ifcommute=1)
  for (int i=0; i<*len; i++){

    // version 1: use function of energy&dist
    /*
    if ( F_e_dist(invec[i].value, invec[i].dist) < F_e_dist(inoutvec[i].value, inoutvec[i].dist) ){
      inoutvec[i].value = invec[i].value;
      inoutvec[i].dist = invec[i].dist;
      inoutvec[i].rank = invec[i].rank;
    }
    */

    // version 2: search for a min energy within global->param.grid.search_radius or the closest cell

    // if both cells far away or both are within a search radius -- pick one with the least energy
    if (((inoutvec[i].dist>global->param.grid.search_radius) and (invec[i].dist>global->param.grid.search_radius))
	or ((inoutvec[i].dist<global->param.grid.search_radius) and (invec[i].dist<global->param.grid.search_radius))  ){
      if (invec[i].value < inoutvec[i].value){
	inoutvec[i].value = invec[i].value;
	inoutvec[i].dist = invec[i].dist;
	inoutvec[i].rank = invec[i].rank;
      }
    }
    // else --only one of the cells is further than the search radius, pick the closest one
    else{
      if (invec[i].dist < inoutvec[i].dist){
	inoutvec[i].value = invec[i].value;
	inoutvec[i].dist = invec[i].dist;
	inoutvec[i].rank = invec[i].rank;
      }
    }

  }
}
#endif

//! same as MPI_MIN_E_DIST_LOC but for a loacal "array", "out" -- is the selected element, returns index of the selected element 
int find_min_e_nodist_loc(double_double_int *array, int size, double_double_int &out)
{
  int index=0;

  // reset to the first element
  out.value=array[0].value;
  out.dist=array[0].dist;
  out.rank=array[0].rank;
  index=0;

  for (int i=0; i<size; i++)
    if ( F_e_nodist(array[i].value, array[i].dist) < F_e_nodist(out.value, out.dist) ){
      out.value=array[i].value;
      out.dist=array[i].dist;
      out.rank=array[i].rank;
      index=i;
    }
  return index;
}

//! same as MPI_MIN_E_DIST_LOC but for a loacal "array", "out" -- is the selected element, returns index of the selected element 
int find_min_e_dist_loc(double_double_int *array, int size, double_double_int &out)
{
  int index=0;

  // reset to the first element
  out.value=array[0].value;
  out.dist=array[0].dist;
  out.rank=array[0].rank;
  index=0;

  for (int i=0; i<size; i++)
    if ( F_e_dist(array[i].value, array[i].dist) < F_e_dist(out.value, out.dist) ){
      out.value=array[i].value;
      out.dist=array[i].dist;
      out.rank=array[i].rank;
      index=i;
    }
  return index;
}

#if _IF_MPI
//! operator which is is used e_loc function for mpi reduce calls
MPI_Op MPI_MIN_E_NODIST_LOC;
void create_MPI_MIN_E_NODIST_LOC()
{
  MPI_Op_create( (MPI_User_function *)min_e_nodist_loc, 1, &MPI_MIN_E_NODIST_LOC); 
}

//! operator which is is used e_dist_loc function for mpi reduce calls
MPI_Op MPI_MIN_E_DIST_LOC;
void create_MPI_MIN_E_DIST_LOC()
{
  MPI_Op_create( (MPI_User_function *)min_e_dist_loc, 1, &MPI_MIN_E_DIST_LOC); 
}
#endif

#endif
