/* ====Descriptors.H=============================================================================
 *
 * Author: Patrice Koehl, March 2020
 * Department of Computer Science
 * University of California, Davis
 *
 * Computes 3D descriptors at each vertex of a mesh. Considered here:
 *      Heat Kernel Signatures (HKS)
 *      Wave Kernel Signatures (WKS)
 *
 * based on the papers:

 *	J. Sun, M. Ovsjanikov, and Leonidas Guibas, "A Concise and Provably Informative 
 *	Multi-scale Signature Based on Heat Diffusion", Proc. Eurographics Symposium on teometry Processing (SGP) 2009.
 *
 *	M. Aubry, U. Schlickewei, and D. Cremers, “The wave kernel signature: a quantum mechanical 
 *	approach to shape analyis,” in Proc. CVPR, 2011.

 =============================================================================================== */

#ifndef _SPECTRAL_H_
#define _SPECTRAL_H_

/*================================================================================================
 BLAS / LAPACK prototypes
================================================================================================== */

  extern "C" {

	void dsyevd_(char * JOBZ, char * UPLO, int *N, double *A, int *LDA, double *W, 
	double *WORK, int *LWORK, int *IWORK, int *LIWORK, int *INFO);

  }

  #define _USE_MATH_DEFINES // for M_PI

  #include <vector>
  #include <cmath>
  #include <tuple>

  #include "Adjacency2.h"
  #include "BelkinLaplace.h"

  Adjacency2 table2;
  BelkinLaplace belkin;

/*================================================================================================
 Definitions for multi-threading
================================================================================================== */

typedef struct LB_data {
	int Nvect;
	int N1;
	int N2;
	int Npoint;
	std::vector <std::tuple<int, int, double> > ListPair;
	double *Vect1;
	double *Vect2;
	double *Vect3;
	double *invArea;
} LB_data;

LB_data LBs[NUM_THREADS];

  #include "BlockChebDav.h"
  blockChebDav eigen;


  /* =============================================================================================
     Define class
   =============================================================================================== */

  class Spectral {

  public:
	// generate shape descriptors at each vertex of a mesh
	void genDescriptors(Mesh& mesh, int NE, int laplace_type, int desc_type, int nfeatures, 
		int flag_step, std::vector<double>& steps, std::vector<std::vector<double> >& descriptors, 
		int nthreads);

  private:
 
	// Compute Laplace Beltrami operator and perform eigen analysis
	void LaplaceBeltrami(Mesh& mesh, int etype, int NE, double *eigVect, double *eigVal, int nthreads);

	// Wave kernel signatures
	void WKS(int nv, double *eigVect, double *eigVal, std::vector<double>& e, 
		double sigma, int first, int last, std::vector<std::vector<double> >& descriptors);

	// Heat kernel signatures
	void HKS(int nv, double *eigVect, double *eigVal, std::vector<double>& time, 
		int first, int last, std::vector<std::vector<double> >& descriptors);

	// Initialize
	void init(Mesh& mesh, int etype, int nthreads);

	// Compute cotan Laplacian: full matrix
	void computeLaplacianL(Mesh& mesh);

	// Compute cotan Laplacian: sparse matrix
	void computeLaplacianU(Mesh& mesh);

	// Compute all eigenpairs of symmetric real matrix when matrix is "small"
	void fullEigen(int N, double *eigVect, double *eigVal);

	// Compute area associated with each vertex
	void computeArea(Mesh& mesh);

	// Compute mixed area at each vertex
	double mixedAreaMesh(Mesh& mesh, double *MixedArea);

	// Compute angles and cotans in a triangle
	void trigAngles(Vector& a, Vector& b, Vector& c, double *angs, double *cotans);

	// Computes surface area of a triangle
	double  trigArea(Vector& A, Vector& B, Vector& C);

	// Normalize descriptors
	void normalize(std::vector<std::vector<double> >& descriptors);

	// Load Laplace Beltrami operators for multithreading
	void loadLB(int nthreads);

   protected:

	int area_type = 1;
	int n_vertices, n_pairs;
	double *M   = NULL;
	double *invArea = NULL;
	double *L = NULL;
	double *Space = NULL;
	std::vector<std::tuple<int, int, double> > ListPair;

  };

  /* =============================================================================================
   Compute shape descriptors at each vertex of a mesh
   All descriptors are based on the eigen analysis of the Laplace Beltrami operator on the mesh
   Currently implemented: HKS, SIHKS, and WKS

   Input:
	mesh:	the triangular mesh data structure
	neigen: number of eigenpairs of LB operator considered
	nfeatures: # of features for the descriptor
   Output:
	descriptors:    matrix of descriptors of size nfeatures*N, where N is the number of vertices
   =============================================================================================== */

  void Spectral::genDescriptors(Mesh& mesh, int NE, int laplace_type, int desc_type, int nfeatures, 
	int flag_step, std::vector<double>& steps, std::vector<std::vector<double> >& descriptors, int nthreads)
  {
	int CUTOFF = 1000;
	n_vertices = mesh.vertices.size();
	int etype = 1;
	if(n_vertices <= CUTOFF || NE <= CUTOFF || NE == n_vertices) etype = 0;

	// Init all arrays
	init(mesh, etype, nthreads);

	// Compute Laplace Beltrami operator, its eigenvectors, and eigenvalues
	int neigen;
	if(etype==0) {
		neigen = n_vertices;
	} else {
		neigen = NE;
	}
	double *eigVect = new double[n_vertices*neigen];
	double *eigVal  = new double[neigen];

	if(laplace_type==0) {
		LaplaceBeltrami(mesh, etype, neigen, eigVect, eigVal, nthreads);
	} else {
		belkin.laplace(mesh, eigVect);
		belkin.laplaceEigen(n_vertices, eigVect, eigVal);
		for(int i = 0; i < 10; i++) {
			std::cout << "i = " << i << " eigen[i] = " << eigVal[i] << std::endl;
		}
		for(int i = n_vertices-10; i < n_vertices; i++) {
			std::cout << "i = " << i << " eigen[i] = " << eigVal[i] << std::endl;
		}
	}

	// Compute descriptors

	int first = 0;
	int last = NE;
	for(int i = 0; i < NE; i++) {
		if(std::abs(eigVal[i]) > 1.e-8) {
			first = i;
			break;
		}
	}
	std::cout << "# of components in mesh " << first << std::endl;
	if(desc_type==0 ) {
		if(flag_step==0) {
			steps.clear();
			double tmax = 4.0*std::log(10.)/eigVal[first];
			double tmin = 4.0*std::log(10.)/eigVal[last-1];
			double min_log = std::log(tmin);
			double max_log = std::log(tmax);
			int num_intervals = nfeatures-1;
			double log_increment = ( max_log - min_log ) / num_intervals ;
			double log_value = min_log ;
			steps.push_back(tmin);
			for( int it = 0 ; it < num_intervals ; it++ )
			{
				log_value += log_increment ;
				steps.push_back(std::exp(log_value)) ;
			}
		}
		HKS(n_vertices, eigVect, eigVal, steps, first, last, descriptors);
	} else if(desc_type==1) {
		double sigma = 0.0;
		if(flag_step==0) {
			double emin = std::log(eigVal[first]);
			double emax = std::log(eigVal[last-1]);
			double step=(emax-emin)/(nfeatures-1);
			sigma = 7*step;
			emin += 2*sigma;
			emax -= 2*sigma;
			for(int it = 0; it < nfeatures; it++) {
				steps.push_back(emin+it*step);
			}
		} else {
			int nsteps = steps.size();
			double step = (steps[nsteps-1]-steps[0])/(nsteps-1);
			sigma = 7*step;
		}
		WKS(n_vertices, eigVect, eigVal, steps, sigma, first, last, descriptors);
	}

	normalize(descriptors);

	delete [] eigVect; delete [] eigVal;
	delete [] M; delete [] invArea;
	if(L != NULL) { 
		delete [] L; L = NULL;
	}
	if(Space != NULL) {
		delete [] Space; Space = NULL;
	}

  }
		
  /* =============================================================================================
   Initialize all arrays
   =============================================================================================== */

  void Spectral::init(Mesh& mesh, int etype, int nthreads)
  {

  /* ==================================================================================================
	Allocate space for all arrays
   ==================================================================================================*/

	n_vertices = mesh.vertices.size();
	n_pairs = mesh.edges.size();

	M = new double[n_vertices];
	invArea = new double[n_vertices];

	if(etype==0) {
		L = new double[n_vertices*n_vertices];
	} else {
		ListPair.clear();
		table2.construct(mesh, ListPair);
		Space = new double[n_vertices*nthreads];
	}

  }

  /* ===== Cotan Laplacian          =================================================================
	Laplacian: full matrix
   ==================================================================================================*/

  void Spectral::computeLaplacianL(Mesh& mesh)
  {

	int idxA, idxB, idxC;

	Vector p1, p2, p3;

	double cotan_A, cotan_B, cotan_C;
	double cA, cB, cC;

	HalfEdgeIter hAB, hBC, hCA;
	VertexIter v1, v2, v3;

	memset(L, 0, n_vertices*n_vertices*sizeof(double));

	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		p1 = v1->position;
		p2 = v2->position;
		p3 = v3->position;

		idxA = v1->index;
		idxB = v2->index;
		idxC = v3->index;

		cotan_A = dot(p2-p1, p3-p1) / ( cross(p2-p1, p3-p1).norm());
		cotan_B = dot(p1-p2, p3-p2) / ( cross(p1-p2, p3-p2).norm());
		cotan_C = dot(p1-p3, p2-p3) / ( cross(p1-p3, p2-p3).norm());

		cA = 0.5*cotan_A; cB = 0.5*cotan_B; cC = 0.5*cotan_C;

		L[idxA*n_vertices+idxB] -= cC;
		L[idxB*n_vertices+idxA] -= cC;
		L[idxA*n_vertices+idxC] -= cB;
		L[idxC*n_vertices+idxA] -= cB;
		L[idxB*n_vertices+idxC] -= cA;
		L[idxC*n_vertices+idxB] -= cA;

		L[idxA*n_vertices+idxA] += cB + cC;
		L[idxB*n_vertices+idxB] += cA + cC;
		L[idxC*n_vertices+idxC] += cA + cB;

	}

  }

  /* ===== Cotan Laplacian          =================================================================
	Laplacian: sparse representation
   ==================================================================================================*/

  void Spectral::computeLaplacianU(Mesh& mesh)
  {

	int idx, idxA, idxB, idxC;

	Vector p1, p2, p3;

	double cotan_A, cotan_B, cotan_C;
	double cA, cB, cC;

	HalfEdgeIter hAB, hBC, hCA;
	VertexIter v1, v2, v3;

	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		hAB =f_iter->he;
		hBC =hAB->next;
		hCA =hBC->next;

		v1 = hAB->vertex;
		v2 = hBC->vertex;
		v3 = hCA->vertex;

		p1 = v1->position;
		p2 = v2->position;
		p3 = v3->position;

		idxA = v1->index;
		idxB = v2->index;
		idxC = v3->index;

		cotan_A = dot(p2-p1, p3-p1) / ( cross(p2-p1, p3-p1).norm());
		cotan_B = dot(p1-p2, p3-p2) / ( cross(p1-p2, p3-p2).norm());
		cotan_C = dot(p1-p3, p2-p3) / ( cross(p1-p3, p2-p3).norm());

		cA = 0.5*cotan_A; cB = 0.5*cotan_B; cC = 0.5*cotan_C;

		idx = table2.getIndex(idxA, idxB);
		std::get<2>(ListPair[idx]) += cC;
		idx = table2.getIndex(idxA, idxC);
		std::get<2>(ListPair[idx]) += cB;
		idx = table2.getIndex(idxB, idxC);
		std::get<2>(ListPair[idx]) += cA;

	}

  }

  /* ===============================================================================================
   trigArea: Computes surface area of a triangle given by 3 points.

         if the three points are A, B, and C,
	 S = norm( A-B) x (A-C) )/2
	 where x is the cross product

   Input:
	  A, B, C: the three vertices of the triangle
   Output:
	  Area: surface area of the triangle
   =============================================================================================== */

  double Spectral::trigArea(Vector& A, Vector& B, Vector& C)
  {
	double Area;
	Area = 0.5 * cross(A-B , A-C).norm() ;
	return Area;
  }

  /* ===============================================================================================
   trigAngles: Computes the three angles of a triangle and their cotans
   =============================================================================================== */

  void Spectral::trigAngles(Vector& a, Vector& b, Vector& c, double *ang, double *cotan)
  {
	double lab, lca, lbc;
	double val, alpha_a, alpha_b, alpha_c;
	double cot_a, cot_b, cot_c;

	lab = (a-b).norm();
	lca = (a-c).norm();
	lbc = (b-c).norm();

	val = (lab*lab + lca*lca - lbc*lbc)/(2.0*lab*lca);
	alpha_a = std::acos(val);
	cot_a   = 1.0/std::tan(alpha_a);
	val = (lab*lab + lbc*lbc - lca*lca)/(2.0*lab*lbc);
	alpha_b = std::acos(val);
	cot_b   = 1.0/std::tan(alpha_b);
	val = (lca*lca + lbc*lbc - lab*lab)/(2.0*lca*lbc);
	alpha_c = std::acos(val);
	cot_c   = 1.0/std::tan(alpha_c);

	ang[0] = alpha_a; ang[1] = alpha_b; ang[2] = alpha_c;
	cotan[0] = cot_a; cotan[1] = cot_b; cotan[2] = cot_c;
  }

  /* ===============================================================================================
   mixedAreaMesh: computes the mixed area of all vertices (i.e. area of the Voronoi region
		associated to a vertex)

   We use the procedure described in:
   M. Meyer, M. Desbrun, P. Schroeder, A. Barr. "Discrete differential geometry operators
   for triangulated 2-manifolds", in: VisMath, Vol 2, pp 35-57 (2002)

   Input:
	  mesh:	the mesh data structure (pointer to the structure)
   Output:
	  MixedArea: mized area of each vertex
	  Area     : total area of the mesh
   =============================================================================================== */

  double Spectral::mixedAreaMesh(Mesh& mesh, double *MixedArea)
  {
	double HalfPI = M_PI/2;

	HalfEdgeCIter he;
	VertexCIter va, vb, vc;
	Vector a, b, c;

	int idxa, idxb, idxc;

	double angs[3], cotans[3];

	double amix, bmix, cmix;
	double lab, lca, lbc;
	double Tarea;

/* 	==========================================================================================
	Initialize mixed area to 0
        ========================================================================================== */

	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		idxa = v->index;
		MixedArea[idxa] = 0;
	}

/* 	==========================================================================================
	Iterate over each triangle:
		For each triangle:
			Divide area into three parts, one for each vertex, using either:
			- Voronoi formula (see paper by Meyer et al) for non-obtuse triangles
			- simpler scheme for obtuse triangle (again, see paper by Meyer et al)
        ========================================================================================== */

	for (FaceCIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		if (f->isReal()) {
			he = f->he;
			va = he->vertex;
			vb = he->next->vertex;
			vc = he->prev->vertex;
			idxa = va->index;
			idxb = vb->index;
			idxc = vc->index;
			a = va->position;
			b = vb->position;
			c = vc->position;

			lab = (a-b).norm();
			lca = (c-a).norm();
			lbc = (b-c).norm();

			trigAngles(a, b, c, angs, cotans);
			Tarea = trigArea(a, b, c);

			if(angs[0] > HalfPI || angs[1] > HalfPI || angs[2] > HalfPI) {
				if(angs[0] > HalfPI) amix = Tarea/2;
				else amix = Tarea/4;
				if(angs[1] > HalfPI) bmix = Tarea/2;
				else bmix = Tarea/4;
				if(angs[2] > HalfPI) cmix = Tarea/2;
				else cmix = Tarea/4;
			}
			else {
				amix = (cotans[2]*lab*lab + cotans[1]*lca*lca)/8.0;
				bmix = (cotans[2]*lab*lab + cotans[0]*lbc*lbc)/8.0;
				cmix = (cotans[0]*lbc*lbc + cotans[1]*lca*lca)/8.0;
			}

			MixedArea[idxa] += amix;
			MixedArea[idxb] += bmix;
			MixedArea[idxc] += cmix;
		}

	}

	double Area = 0;
	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		idxa = v->index;
		Area += MixedArea[idxa];
	}

	return Area;
  }
  /* ===============================================================================================
   computeArea: computes the area of all vertices ( 1/3 of the area of each of its adjacent triangles)

   Input:
	  mesh:	the mesh data structure (pointer to the structure)
   =============================================================================================== */

  void Spectral::computeArea(Mesh& mesh)
  {
	HalfEdgeCIter he;
	VertexCIter va, vb, vc;
	Vector a, b, c;

	int idxa, idxb, idxc;

	double Tarea;

	memset(M, 0, n_vertices*sizeof(double));

	if(area_type == 0) {

/* 		====================================================================================
		Iterate over each triangle:
        	=================================================================================== */

		for (FaceCIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
		{
			he = f->he;
			va = he->vertex;
			vb = he->next->vertex;
			vc = he->prev->vertex;
			idxa = va->index;
			idxb = vb->index;
			idxc = vc->index;
			a = va->position;
			b = vb->position;
			c = vc->position;

			Tarea = trigArea(a, b, c);

			M[idxa] += Tarea/3;
			M[idxb] += Tarea/3;
			M[idxc] += Tarea/3;
		}
	} else if(area_type == 1) {
		double *mixed = new double[n_vertices];
		mixedAreaMesh(mesh, mixed);
		for(int i = 0; i < n_vertices; i++) {
			M[i] = mixed[i];
		}
		delete [] mixed;
	}

  }

  /* ===============================================================================================
   LaplaceBeltrami: build and diagonalize Laplace Beltrami operator

   Input:
	  mesh:	the mesh data structure (pointer to the structure)
   =============================================================================================== */

  void Spectral::LaplaceBeltrami(Mesh& mesh, int etype, int NE, double *eigVect, double *eigVal, int nthreads)
  {

	computeArea(mesh);
	for(int i = 0; i < n_vertices; i++) invArea[i] = 1.0/std::sqrt(M[i]);

/*================================================================================================
	Full eigendecomposition if LB operator is small
================================================================================================== */

	if(etype==0) {

		computeLaplacianL(mesh);

		// Compute A = M^{-1/2} L M^{1/2}

		int idx;
		for(int i = 0; i < n_vertices; i++) {
			for(int j = 0; j < n_vertices; j++) {
				idx = j + i*n_vertices;
				eigVect[idx] = L[idx]*invArea[i]*invArea[j];
			}
		}

		fullEigen(n_vertices, eigVect, eigVal);

/*================================================================================================
	Compute only top NE eigenpairs if n_vertices <= 500
================================================================================================== */

	} else {

		computeLaplacianU(mesh);
		loadLB(nthreads);

		int k_conv;
		eigen.eigenpairs(n_vertices, NE, eigVal, eigVect, &k_conv, nthreads);
		if(k_conv < NE) NE = k_conv;
	}

	for(int j = 0; j < NE; j++) {
		for(int i = 0; i < n_vertices; i++) {
			eigVect[i+j*n_vertices] *= invArea[i];
		}
	}

  }

  /* ===============================================================================================
   WKS: Wave kernel signature at each vertex

   Input:
	n : number of vertices
	eigVect: eigenvectors of LB operator
	eigVal : eigenvalues of LB operator
	e: array of energy levels for WKS
   Output:
	WKS descriptors, matrix of size n*ntimes
   =============================================================================================== */

  void Spectral::WKS(int n, double *eigVect, double *eigVal, std::vector<double>& e, 
	double sigma, int first, int last, std::vector<std::vector<double> >& descriptors)
  {

	for(int i = 0; i < n*last; i++) eigVect[i] = eigVect[i]*eigVect[i];

	int ne = e.size();

	double eval, val, x, s2;

	descriptors.clear();
	std::vector<double> desc;
	for(int i = 0; i < ne; i++) desc.push_back(0.0);
	for(int i = 0; i < n; i++) descriptors.push_back(desc);

	s2 = 1.0/(2*sigma*sigma);

	double *loge = new double[last];
	double *f = new double[last];
	double *d = new double[n];

	for(int j = first; j < last; j++) {
		loge[j] = -std::log(eigVal[j]);
	}

	int inc = 1;
	double alpha=1.0, beta=0;
	int nval = last - first;
	char NoTrans = 'N';
	for(int ie = 0; ie < ne; ie++) {
		eval = e[ie];
		val = 0;
		for(int j = first; j < last; j++) {
			x = eval +loge[j];
			x = std::exp(-x*x*s2);
			val += x;
			f[j-first] = x;
		}
		if(val > 1.e-8) {
			val = 1.0/val;
		} else {
			val = 1.0;
		}

		dscal_(&nval, &val, f, &inc);
		dgemv_(&NoTrans, &n, &nval, &alpha, &eigVect[first*n], &n,
		f, &inc, &beta, d, &inc);
		for(int i = 0; i < n; i++) descriptors[i][ie] = d[i];

	}
	delete [] loge;
	delete [] f;
	delete [] d;

  }
  /* ===============================================================================================
	Normalize descriptors such that sum over descriptors is 1
   =============================================================================================== */

   void Spectral::normalize(std::vector<std::vector<double> >& descriptors)
   {

	int nvertices = descriptors.size();
	int nfeatures = descriptors[0].size();
	double s;
	for(int j = 0; j < nvertices; j++)
	{
		s = 0;
		for(int i = 0; i < nfeatures; i++) s+= descriptors[j][i];
		s = 1./s;
		for(int i = 0; i < nfeatures; i++) descriptors[j][i] *=s;
	}

  }
/*================================================================================================
 LoadLB: set LBs struct to store information about LB
================================================================================================== */

void Spectral::loadLB(int nthreads)
{
	for(int i = 0; i < nthreads; i++) 
	{
		LBs[i].Npoint    = n_vertices;
		LBs[i].ListPair  = ListPair;
		LBs[i].Vect2     = &Space[i*n_vertices];
		LBs[i].invArea   = invArea;
	}
}

/*================================================================================================
 Compute all eigenpairs of a real symmetric matrix using LAPAK routine dsyevd
================================================================================================== */

  void Spectral::fullEigen(int N, double *eigVect, double *eigVal)
  {

/*================================================================================================
	Declare some variables
================================================================================================== */

	int lwork, info;
	int liwork, isizeopt;
	double sizeopt;
	char U = 'U';
	char V = 'V';

/*================================================================================================
	Create all temporary arrays
================================================================================================== */

	lwork = -1;
	liwork = -1;
	dsyevd_(&V, &U, &N, eigVect, &N, eigVal, &sizeopt, 
		&lwork, &isizeopt, &liwork, &info);

	lwork = (int) sizeopt;
	liwork = isizeopt;

	int *iwork = new int[liwork];
	double *dwork = new double[lwork];

/*================================================================================================
	Compute eigenvalues / eigenvectors
================================================================================================== */

	dsyevd_(&V, &U, &N, eigVect, &N, eigVal, dwork, 
		&lwork, iwork, &liwork, &info);

/*================================================================================================
	Clean up all temporary arrays
================================================================================================== */

	delete [] dwork;
//	delete [] iwork;
 }


  /* ===============================================================================================
   HKS: Heat kernel signature at each vertex

   Input:
	n : number of vertices
	eigVect: eigenvectors of LB operator
	eigVal : eigenvalues of LB operator
	time: array of times for HKS
	first: first eigenpair of LB operator considered
	last: first eigenpair of LB operator considered
   Output:
	HKS descriptors, matrix of size n*ntimes
   =============================================================================================== */

  void Spectral::HKS(int n, double *eigVect, double *eigVal, std::vector<double>& times, int first,
	int last, std::vector<std::vector<double> >& descriptors)
  {

	for(int i = 0; i < n*last; i++) eigVect[i] = eigVect[i]*eigVect[i];

	int ntimes = times.size();

	double t, val, x;

	double *f = new double[last];
	double *d = new double[n];

	descriptors.clear();
	std::vector<double> desc;
	for(int i = 0; i < ntimes; i++) desc.push_back(0.0);
	for(int i = 0; i < n; i++) descriptors.push_back(desc);

	int inc = 1;
	double alpha=1.0, beta=0;
	int nval = last - first;
	char NoTrans = 'N';

	for(int it = 0; it < ntimes; it++) {
		t = times[it];
		val = 0;
		for(int j = first; j < last; j++) {
			x = std::exp(-eigVal[j]*t);
			val += x;
			f[j-first] = x;
		}
		if(val > 1.e-8) {
			val = 1.0/val;
		} else {
			val = 1.0;
		}

		dscal_(&nval, &val, f, &inc);
		dgemv_(&NoTrans, &n, &nval, &alpha, &eigVect[first*n], &n,
		f, &inc, &beta, d, &inc);
		for(int i = 0; i < n; i++) descriptors[i][it] = d[i];
	}

	delete [] f;

  }

#endif
