/*================================================================================================
  WriteResults.h
  Version 1: 9/29/2019

Copyright (c) Patrice Koehl.

================================================================================================== */

#ifndef _WRITERESULTS_H_
#define _WRITERESULTS_H_

/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>
#include <cstdlib>
#include <string>
#include <fstream>

/*================================================================================================
 class   
================================================================================================== */

 class WriteResults {

	public:

		// write elastic network
		void writeEN(std::string filename, std::vector<Atoms>& atoms, 
		std::vector<Links> links);

		// Write atom-based kconst
		void writekAtom(std::string filename, std::vector<Atoms>& atoms);
		
		// Write eigenvalues
		void writeEigen(std::string filename, int nval, double *eigVal);
		
		// write computed B-factors
		void writeBfact(std::string filename, std::vector<Atoms>& atoms, double *bfact);

		// write computed anisotropic B-tensors
		void writeBtensor(std::string filename, std::vector<Atoms>& atoms, double *bval);

		// ovewrall results
		void writeOverlapb(std::string filename, int nmodes, int check_bfact, double *rms_bfact, 
		double *correl_bfact, double rms0, double *rms, double *overlap);

		// contribution of each mode to: overlap, rms
		void writeOverlap(std::string filename, int nmodes, double rms0, double *rms, double *overlap);

		// generate normal mode traj
		void genTraj(std::string filename, std::vector<Atoms>& atoms, int n1, int n2, int nc, 
		double *eigVal, double *eigVect);

		void writeNMD(std::string filename, std::vector<Atoms>& atoms, int n1, int n2, double *eigVal, double *eigVect);

		void writeCOVAR(std::string filename, int natoms, double *Covar);

		void writeCORREL(std::string filename, int natoms, double *Correl);

};

/*================================================================================================
 writeEigen: write eigenvalues of Hessian matrix
================================================================================================== */

  void WriteResults::writeEigen(std::string filename, int nval, double *eigVal)
  {
	std::ofstream output;
	output.open(filename);

	for(int i = 0; i < nval; i++) {
		output << eigVal[i] << std::endl;
	}
 
	output.close();
  }

/*================================================================================================
 writeBfact: computed and experimental B-factors
================================================================================================== */

void WriteResults::writeBfact(std::string filename, std::vector<Atoms>& atoms, double *bfact)
{
	std::ofstream output;
	output.open(filename);

	int natoms = atoms.size();
	for(int i = 0; i < natoms; i++)
	{
		output << atoms[i].line.substr(0,30) << "  " << std::setw(10) << atoms[i].bfact;
		output << "  " << std::setw(10) << bfact[i] ;
		output << "  " << std::setw(10) << atoms[i].kconst << std::endl;
	}

	output.close();
}
/*================================================================================================
 writeBtensor: computed and experimental B-tensor
================================================================================================== */

void WriteResults::writeBtensor(std::string filename, std::vector<Atoms>& atoms, double *bval)
{
	std::ofstream output;
	output.open(filename);

	int natoms = atoms.size();
	for(int i = 0; i < natoms; i++)
	{
		output << atoms[i].line.substr(0,30);
		for(int j = 0; j < 6; j++) {
			output << "  " << atoms[i].U[j];
		}
		output << std::endl;
		output << atoms[i].line.substr(0,30);
		for(int j = 0; j < 6; j++) {
			output << "  " << bval[6*i+j];
		}
		output << std::endl;
	}

	output.close();
}

/*================================================================================================
 writeKAtom: computed atom-based Kconst
================================================================================================== */

void WriteResults::writekAtom(std::string filename, std::vector<Atoms>& atoms) 
{

	std::ofstream output;
	output.open(filename);

	int natoms = atoms.size();
	for(int i = 0; i < natoms; i++)
	{
		output << atoms[i].line.substr(0,60) << std::setw(6) << std::fixed << std::setprecision(2) << atoms[i].kconst << std::endl;
	}

	output.close();
}

/*================================================================================================
 writeEN: write Elastic Network
================================================================================================== */

void WriteResults::writeEN(std::string filename, std::vector<Atoms>& atoms,
		std::vector<Links> links)
{
	std::ofstream output;
	output.open(filename);

	std::string str1, str2;

	int npairs = links.size();
	int i1, i2;
	output << "#============================================================#" << std::endl;
	output << "#          Atom 1               Atom 2             kconst" << std::endl;
	output << "#============================================================#" << std::endl;
	for(int i = 0; i < npairs; i++)
	{
		i1 = links[i].atm1;
		i2 = links[i].atm2;
		str1 = atoms[i1].line.substr(4,22);
		str2 = atoms[i2].line.substr(4,22);
		output << str1 << " " << str2 << "      " << links[i].kconst << std::endl;
//		output << links[i].atm1+1 << "  " << links[i].atm2+1 << " " << links[i].kconst << std::endl;
	}

	output.close();
}

/*================================================================================================
 writeOverlapb: contribution of each mode to: b-factor, overlap, rms
================================================================================================== */

void WriteResults::writeOverlapb(std::string filename, int nmodes, int check_bfact, double *rms_bfact, 
	double *correl_bfact, double rms0, double *rms, double *overlap)
{
	std::ofstream output;
	output.open(filename);

	int j = 0;
	double zero=0;
	output << j << "  ";
	if(check_bfact != 0) {
		output << std::setw(10) << zero << "  " << std::setw(10) << zero << "  ";
	}
	output << std::setw(10) << zero << "  " << std::setw(10) << rms0 << std::endl;
	for(int i = 0; i < nmodes; i++)
	{
		output << i +1 << "  ";
		if(check_bfact != 0) {
			output << std::setw(10) << rms_bfact[i] << "  " << std::setw(10) << correl_bfact[i] << "  ";
		}
		output << std::setw(10) << overlap[i] << "  " << std::setw(10) << rms[i] << std::endl;
	}

	output.close();
}

/*================================================================================================
 writeOverlap: contribution of each mode to: overlap, rms
================================================================================================== */

void WriteResults::writeOverlap(std::string filename, int nmodes, double rms0, double *rms, double *overlap)
{
	std::ofstream output;
	output.open(filename);

	double cumul = 0;
	for(int i = 0; i < nmodes; i++)
	{
		cumul += overlap[i]*overlap[i];
		output << i +1 << "  ";
		output << std::setw(10) << overlap[i] << "  " << std::setw(10) << std::sqrt(cumul);
		output << " " << std::setw(10) << rms[i] << std::endl;
	}

	output.close();
}

/*================================================================================================
 	writeTraj: write a trajectory for a subset of normal modes
 ================================================================================================== */

void WriteResults::genTraj(std::string filename, std::vector<Atoms>& atoms, int n1, int n2, int nc, 
	double *eigVal, double *eigVect)
{
	std::ofstream output;
	output.open(filename);

	double KB_AKMA = 0.00198719;
	double Temp    = 293.0;
	double fac = std::sqrt(2.0*KB_AKMA*Temp);
	double twopi = 2*M_PI;

/*================================================================================================
	take the period of the lowest frequency mode (i.e. mode
	# n1) as the reference period. The time increment to generate
	trajectories is this period divided by the number of points
	per period
	Note that delta_t is in pico seconds
 ================================================================================================== */

	double Tref, delta_t;
	int Nperiod;

	if(n1 > 5) {
		Tref = 33.35/eigVal[n1];
	} else {
		Tref = 33.35;
	}

        Nperiod = 4;
        delta_t = (Nperiod*Tref)/nc;

/*================================================================================================
	Compute scaling factor to have trajectory with a prescribed rsmd of 1
 ================================================================================================== */

	int natoms = atoms.size();

	double rms1, val;
	double rmsd=0.0;
	for(int i = 0; i < natoms; i++) {
		rms1 = 0.0;
		for(int j = n1; j <= n2; j++)
		{
			for(int k = 0; k < 3; k++) {
				val = eigVect[3*natoms*j + 3*i + j];
				rms1 += val*val;
			}
		}
		rms1 = std::sqrt(rms1);
		rmsd += rms1;
	}
	rmsd = rmsd/natoms;

/*================================================================================================
	Now generate trajectory stored into a PDB file
 ================================================================================================== */

	output << "REMARK" << std::endl;
	output << "REMARK  Normal modes considered : " << n1+1 << " - " << n2+1 << std::endl;

	double amp;
	double crd[3];
	double rms_target = 1;

	for(int ic = 0; ic <=nc; ic++)
	{
		output << "REMARK" << std::endl;
		output << "REMARK  Current time: " << ic*delta_t << std::endl;
		output << "REMARK" << std::endl;

		for(int iatm = 0; iatm < natoms; iatm++) {

			for(int k = 0; k < 3; k++) {
				crd[k] = atoms[iatm].coord[k];
				for(int i = n1; i <=n2; i++) {
					amp = (fac/std::sqrt(eigVal[i]))*(rms_target/rmsd);
					amp = std::min(100.0, amp);
					crd[k] += amp*eigVect[3*natoms*i +3*iatm+k]
						*std::sin(eigVal[i]*twopi*delta_t*ic/33.35);
				}
			}
			output << atoms[iatm].line.substr(0,30);
			output << std::fixed << std::setprecision(3) << std::setw(8) << crd[0];
			output << std::fixed << std::setprecision(3) << std::setw(8) << crd[1];
			output << std::fixed << std::setprecision(3) << std::setw(8) << crd[2];
			output << std::endl;

		}

		output << "TER" << std::endl;
		output << "ENDMDL" << std::endl;
	}

	output.close();
}
/*================================================================================================
 	writeNMD: write normal modes into NMD format
 ================================================================================================== */

void WriteResults::writeNMD(std::string filename, std::vector<Atoms>& atoms, int n1, int n2,
	double *eigVal, double *eigVect)
{
	int natoms = atoms.size();

	std::ofstream output;
	output.open(filename);

	size_t index=0;

	index = filename.find(".nmd", index);
	std::string name = filename.substr(0, index);

	output << "nmwiz_load " << filename << std::endl;
	output << "name " << name << std::endl;

	std::string atmname;
	output << "atomnames ";
	for(int i = 0; i < natoms; i++) {
		atmname = atoms[i].line.substr(12,4);
		output << atmname;
	}
	output << std::endl;

	std::string resname;
	output << "resnames ";
	for(int i = 0; i < natoms; i++) {
		resname = atoms[i].line.substr(17,3);
		output << resname << " ";
	}
	output << std::endl;

	output << "resids ";
	int resid;
	for(int i = 0; i < natoms; i++) {
		resid = atoi(atoms[i].line.substr(22,4).c_str());
		output << resid << " ";
	}
	output << std::endl;

	if(atoms[0].line.substr(21,1) != " ") {
		std::string chainname;
		output << "chainids ";
		for(int i = 0; i < natoms; i++) {
			chainname = atoms[i].line.substr(21,1);
			output << chainname << " ";
		}
		output << std::endl;
	}

	output << "bfactors ";
	for(int i = 0; i < natoms; i++) {
		output << atoms[i].bfact << " ";
	}
	output << std::endl;

	output << "coordinates ";
	for(int i = 0; i < natoms; i++) {
		output << atoms[i].coord[0] << " " << atoms[i].coord[1] << " " <<atoms[i].coord[2] << " ";
	}
	output << std::endl;

	for(int i = n1; i < n2; i++) {
		output << " mode " << i+1 << " ";
		output << 1./std::sqrt(eigVal[i]) << " ";
		for(int j = 0; j < 3*natoms; j++) {
			output << eigVect[3*natoms*i + j] << " ";
		}
		output << std::endl;
	}

	output.close();

  }

/*================================================================================================
 writeCOVAR: write covariance matrix
================================================================================================== */

  void WriteResults::writeCOVAR(std::string filename, int natoms, double *Covar)
  {
	std::ofstream output;
	output.open(filename);

	for(int i = 0; i < 3*natoms; i++) {
		for(int j = 0; j < 3*natoms; j++) {
			output<< Covar[i+3*natoms*j] << " ";
		}
		output << std::endl;
	}
 
	output.close();
  }

/*================================================================================================
 writeCORREL: write correlation matrix
================================================================================================== */

  void WriteResults::writeCORREL(std::string filename, int natoms, double *Correl)
  {
	std::ofstream output;
	output.open(filename);

	for(int i = 0; i < natoms; i++) {
		for(int j = 0; j < natoms; j++) {
			output<< Correl[i+natoms*j] << " ";
		}
		output << std::endl;
	}
 
	output.close();
  }

#endif
