/*================================================================================================
  Bfactor.h
  Version 1: 12/1/2017

Copyright (c) Patrice Koehl.

================================================================================================== */

#ifndef _BFACTOR_H_
#define _BFACTOR_H_

/*================================================================================================
 Includes
================================================================================================== */

#include <math.h>
#include <cstdlib>

#ifndef M_PI
    #define M_PI 3.14159265358979323846
#endif

/*================================================================================================
  Prototypes for BLAS
================================================================================================== */

extern "C" {

	void daxpy_(int * n ,double *alpha , double * X, int *incx, double * Y,int *incy);
	double dnrm2_(int * n, double * X, int *incx);
	void dscal_(int * n, double * alpha, double * X, int *incx);
	void dcopy_(int * n, double * X, int *incx, double * Y, int *incy);
	double ddot_(int * n, double * u, int * incu, double * v, int *incv);
}

/*================================================================================================
  class
================================================================================================== */

  class Bfactor {

	public:

		// Compute b-factor
		void computeBfact(std::vector<Atoms>& atoms, double *rigid, int nmode1, int nmode2, 
		double *eigVal, double *eigVect, double *bfact);

		// compare experimental and computed b-factors
		void compareBfact(std::vector<Atoms>& atoms, double *bfact, double *rms, double *correl);



	private:

  };

/*================================================================================================
 CompareBfact: compare computed and experimental B-factors
================================================================================================== */

void Bfactor::compareBfact(std::vector<Atoms>& atoms, double *bfact, double *rms, double *correl)
{
	double Se, See, Sc, Scc, Sec;
	double val_exp, val_cal;
	double val1, val2, val;

	Se  = 0;
	See = 0;
	Sc  = 0;
	Scc = 0;
	Sec = 0;

	int Natoms = atoms.size();

	for(int i = 0; i < Natoms; i++)
	{
		val_exp = atoms[i].bfact;
		val_cal = bfact[i];
		Se  = Se + val_exp;
		See = See + val_exp*val_exp;
		Sc  = Sc + val_cal;
		Scc = Scc + val_cal*val_cal;
		Sec = Sec + val_exp*val_cal;
	}

	if(See!=0 && Scc!=0) {
		*rms    = std::sqrt( (See+Scc-2*Sec)/Natoms );
		val1 = Natoms*See-Se*Se;
		val2 = Natoms*Scc-Sc*Sc;
		val = std::abs(val1*val2);
		val = std::sqrt(val);
		if(val==0) {
			*correl = 0;
		} else {
			*correl = (Natoms*Sec - Se*Sc)/val;
		}
	} else {
		*rms = 0.0;
		*correl = 0.0;
	}
}

/*================================================================================================
 ComputeBfact: compute B-factors and compare with experimental B-factors
================================================================================================== */

void Bfactor::computeBfact(std::vector<Atoms>& atoms, double *rigid, int nmode1, int nmode2, double *eigVal, 
	double *eigVect, double *bfact)
{
	int Natoms = atoms.size();
	memset(bfact, 0, Natoms*sizeof(double));

	double kT = 0.593;
	double facb = 8.0*kT*M_PI*M_PI/3.0;

	double x,y,z,val, val2;
	double rms_val, correl_val;

	double x[3];
	double val;
	for(int i = 0; i < natoms; i++) 
	{
		for(int k = 0; k < 3; k++) x[k] = atoms[i].coord[k];
		val = rigid[0] + rigid[1]*x[0] + rigid[2]*x[1] + rigid[3]*x[2];
		val += rigid[4]*x[0]*x[0] + rigid[5]*x[0]*x[1] + rigid[6]*x[0]*x[2];
		val += rigid[7]*x[1]*x[1] + rigid[8]*x[1]*x[2] + rigid[9]*x[2]*x[2];
		bfact[i] = facb*val;
	}
		

	for (int i = nmode1; i < nmode2;i++)
	{
		val = facb/eigVal[i];
		for(int j = 0; j < Natoms; j++)
		{
			x = eigVect[i*3*Natoms+3*j];
			y = eigVect[i*3*Natoms+3*j+1];
			z = eigVect[i*3*Natoms+3*j+2];
			val2 = (x*x+y*y+z*z)*val;
			bfact[j] = bfact[j]+val2;
		}
	}
}

#endif
