/* ====WillmoreFlow.h ===========================================================================
 *
 * Author: Patrice Koehl, July 2018
 * Department of Computer Science
 * University of California, Davis
 *
 * This file applies a conformalized Willmore Flow to parameterize a surface onto the sphere
 *
 * It is based on the papers:
 * 	Z. Ye, O. Diamanti, C. Tang, L. Guibas, and T. Hoffmann, "A unified framework for intrinsic
 *      and extrinsic Dirac operators for geometry processing, Computer Graphics Forum, 37, (2018)
 *      K. Crane, U. Pinkall, and P. Schroeder, "Robust fairing via conformal curvature flow"
 *       ACM Transactions on Graphics, Vol. 32, No. 4 (2013).
 * and on the code SpinXform_OpenGL from Keenan Crane (available at:
 *	https://www.cs.cmu.edu/~kmcrane/index.html#code
 =============================================================================================== */

#pragma once

  /* ===== INCLUDES AND TYPEDEFS =================================================================
   *
   =============================================================================================== */

#include <vector>
#include <cmath>
#include "Quaternion.h"
#include "ConformalError.h"
#include "SpinTransform.h"

/*================================================================================================
  Prototypes for BLAS and LAPACK
================================================================================================== */

extern "C" {

        void dscal_(int * n, double * alpha, double * X, int *incx);
        double ddot_(int * n, double * u, int * incu, double * v, int *incv);

        void dgemv_(char * trans, int * m, int * n, double * alpha, double *A,
                int *lda, double * X, int * incx, double *beta, double * Y, int * incy);

}

 Eigen::CholmodDecomposition<Eigen::SparseMatrix<double>> solverE;

 int nE = 0;

  SpinTransform spin;

  /* ===== The Willmore class    =================================================================
   *
   =============================================================================================== */

class Willmore
 {
  public:
	// initialize Willmore flow
	void initFlow(Mesh& mesh, double dt);

	// Perform one step of Willmore flow
	Vector solveOneStep(Mesh& mesh, int niter, double dt, int btype);

	// terminate Willmore flow to sphere
	void stopFlow(Mesh& mesh);

   protected:

	// Initial area of mesh
	double Area0;

	// curvature (one value per face)
	std::vector<double> fcurv;

	// curvature (one value per edge)
	std::vector<double> ecurv;

	// area (one value per face)
	std::vector<double> farea;

	// area (one value per vertex)
	std::vector<double> varea;

	// normal (one value per face)
	std::vector<Vector> Normals;

	// controls change in curvature (one value per face)
	std::vector<double> rho;

	// local similarity transformation (one value per face)
	Eigen::VectorXd lambda_f;

	// local similarity transformation (one value per vertex)
	Eigen::VectorXd lambda_v;

	Eigen::SparseMatrix<double> Adj; //Adjacency matrix (between faces and vertices
	Eigen::SparseMatrix<double> Dirac; //extrinsic Dirac operator matrix
	Eigen::SparseMatrix<double> DA; // Dirac*Adj
	Eigen::SparseMatrix<double> M1; // Matrix for eigenvector problem

	// temp vectors
	Eigen::VectorXd temp1, temp2;

	// compute current mean curvature at each edge and each face
	void meanCurvature(Mesh& mesh);

	// compute Willmore energy for current mesh
	double willmoreEnergy(Mesh& mesh);

	// Recenter and scale mesh
	void normalizeSolution(Mesh& mesh);

	// Compute Mesh info: area, vol, sphericity
	void computeAVS(Mesh& mesh, double *Area, double *Vol, double *S);

	// init Dirac matrix
	void initDirac(Mesh& mesh, Eigen::SparseMatrix<double> &Mat);

	// Build matrix for extrinsic Dirac operator
	void buildDirac(Mesh& mesh);

	// init adjacency matrix
	void initAdj(Mesh& mesh, Eigen::SparseMatrix<double> &Mat);

	// Build adjacency matrix
	void buildAdj(Mesh& mesh);

	// update Dirac matrix
	void updateMatrix(Eigen::SparseMatrix<double>& Mat, int i0, int j0, double Q[4][4]);

	// Add target curvature to Dirac operator
	void addTargetCurv(Mesh& mesh, double dt);

	// Align eigenvecctor phi
	void alignPhi();

	// Convert phi/vertex to phi/face
	void phiVertex2Face(Mesh& mesh);

	// Solve for phi
	void solveForPhi(Mesh& mesh);

	// Fit Sphere
	void fitSphere(Mesh& mesh);
};

  /* ================================================================================================
   * Init flow
   ==================================================================================================*/

  void Willmore::initFlow(Mesh& mesh, double dt)
  {
	int n_vertices = mesh.vertices.size();
	int n_faces    = mesh.faces.size();
	int n_edges    = mesh.edges.size();
	int nf4        = 4*n_faces;
	int nv4        = 4*n_vertices;

	// Create Eigen matrices and arrays needed to minimize curvatures

	Dirac.resize(nf4, nf4);
	Adj.resize(nf4, nv4);
	DA.resize(nf4, nv4);
	M1.resize(nv4, nv4);

	fcurv.resize(n_faces);
	ecurv.resize(n_edges);
	farea.resize(n_faces);
	varea.resize(n_vertices);
	Normals.resize(n_faces);
	rho.resize(n_faces);
	lambda_f.resize(nf4);
	lambda_v.resize(nv4);
	temp1.resize(nv4);
	temp2.resize(nv4);

	initDirac(mesh, Dirac);
	initAdj(mesh, Adj);

	// Initialize positions
	for(VertexIter v_iter = mesh.vertices.begin(); v_iter != mesh.vertices.end(); v_iter++)
	{
		v_iter->position2[0] = v_iter->position[0];
		v_iter->position2[1] = v_iter->position[1];
		v_iter->position2[2] = v_iter->position[2];
	}

	int zero = 0;
	double Area, Vol, S;
	computeAVS(mesh, &Area, &Vol, &S);
	Area0 = Area;
	Vector r = ConformalError::quasiConformalError(mesh);
	double quasi = r[2];
	meanCurvature(mesh);
	double wenergy = willmoreEnergy(mesh);

	std::cout << "        " << "=============================================================================================" << std::endl;
	std::cout << "        " << "       Iter       Step size        Area     Willmore energy    QuasiConf. ratio   Sphericity          " << std::endl;
        std::cout << "        " << "=============================================================================================" << std::endl;
        std::cout << "        " << "   " << std::setw(8)<< zero << "    " << std::setw(12) << dt;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << Area;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << wenergy;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << quasi;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << S << std::endl;

  }

  /* =============================================================================================
   *	Solve one step
   =============================================================================================== */

Vector Willmore::solveOneStep(Mesh& mesh, int niter, double dt, int btype)
{

	// Compute curvature
	meanCurvature(mesh);

	// Compute Dirac operator matrix
	buildDirac(mesh);

	// Compute Dirac operator matrix
	buildAdj(mesh);

	// Add target mean curvature
	addTargetCurv(mesh, dt);

	// Solve for phi
	solveForPhi(mesh);
	
	if(btype==0) {
		spin.solveForPositions_lsq(mesh, ecurv, lambda_f);
	} else {
		spin.solveForPositions_poisson(mesh, lambda_v);
	}

	normalizeSolution(mesh);

	double Area, Vol, S;
	computeAVS(mesh, &Area, &Vol, &S);
	Vector r = ConformalError::quasiConformalError(mesh);
	double quasi = r[2];
	meanCurvature(mesh);
	double wenergy = willmoreEnergy(mesh);

        std::cout << "        " << "   " << std::setw(8)<< niter << "    " << std::setw(12) << dt;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << Area;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << wenergy;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << quasi;
	std::cout << "      " << std::fixed << std::setprecision(6) << std::setw(8) << S << std::endl;

	return Vector(S, quasi, wenergy);

}

  /* ================================================================================================
   * End flow
   ==================================================================================================*/

  void Willmore::stopFlow(Mesh& mesh)
  {
        std::cout << "        " << "=============================================================================================" << std::endl;
	std::cout << " " << std::endl;

	fitSphere(mesh);
  }

  /* =================================================================================================
  * Mesh area, Vol, sphericity
   ==================================================================================================*/

  void Willmore::computeAVS(Mesh& mesh, double *Area, double *Vol, double *S)
  {
	double A, V;

	Vector p1, p2, p3;

	A = 0; V = 0;
	for(FaceIter f_iter = mesh.faces.begin(); f_iter != mesh.faces.end(); f_iter++)
	{
		p1 = f_iter->he->vertex->position2;
		p2 = f_iter->he->next->vertex->position2;
		p3 = f_iter->he->next->next->vertex->position2;

		A += cross(p1-p2,p1-p3).norm();
		V += dot(p1 , cross(p2,p3) );
	}

	A = 0.5*A;
	V = std::abs(V)/6.0;
	*Area = A;
	*Vol  = V;
	*S    = std::pow(M_PI,1.0/3.0) * std::pow(6.0*V,2.0/3.0) / A;

  }

  /* =============================================================================================
   *	Compute current mean curvature at each face
   =============================================================================================== */

  void Willmore::meanCurvature(Mesh& mesh)
  {

	for (FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++) {
		fcurv[f->index] = 0;
		farea[f->index] = f->area2();
		Normals[f->index] = f->normal2(true);
	}

	for(VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) {
		int idx = v->index;
		varea[idx] = 0;
		HalfEdgeIter he = v->he;
		HalfEdgeIter h = he;
		do {
			int f = h->face->index;
			varea[idx] += farea[f];
			h = h->flip->next;
		} while(h != he);
		varea[idx] /= 3.;
	}

	int f1, f2;
	double elength, tan_halftheta, curv;
	Vector n1, n2, v1, v2, v3, v4;
	HalfEdgeIter h12, h21;

	for(EdgeIter e_iter = mesh.edges.begin(); e_iter != mesh.edges.end(); e_iter++)
	{
		f1 = e_iter->he->face->index;
		f2 = e_iter->he->flip->face->index;

		n1 = Normals[f1];
		n2 = Normals[f2];
		
		h12 = e_iter->he;
		h21 = h12->flip;

		v1 = h12->vertex->position2;
		v2 = h21->vertex->position2;
		v3 = h12->prev->vertex->position2;
		v4 = h21->prev->vertex->position2;

		elength = (v1-v2).norm();

		// tan(theta/2) where theta is the angle between normals n1 and n2
		tan_halftheta = cross(n1,n2).norm()/(1.+dot(n1,n2));
		// if edge is concave, change sign
		if(dot(v4-v3,n1)>=0) tan_halftheta = -tan_halftheta;

		curv = elength*tan_halftheta;

		ecurv[e_iter->index] = curv;
		fcurv[f1] += curv;
		fcurv[f2] += curv;
	}

  }

  /* =============================================================================================
   *	Compute Willmore energy
   =============================================================================================== */

  double Willmore::willmoreEnergy(Mesh& mesh)
  {
	double e = 0;
	double val;
	for (FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		val = fcurv[f->index]*fcurv[f->index]/farea[f->index];
		e += val;
	}
	return e/4.;
  }

  /* =============================================================================================
   Init Dirac matrix
   =============================================================================================== */

  void Willmore::initDirac(Mesh& mesh, Eigen::SparseMatrix<double> &Mat)
  {

	// Initialize all values to 0

	int idx, jdx;
	std::vector<Triplet> Mat_coefficients;
	double zero = 0;

	for (FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		idx = f->index;
		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat_coefficients.push_back(Triplet(4*idx+u, 4*idx+v, zero));
			}
		}
	}

	HalfEdgeIter he, he2;
	
	for (EdgeIter e = mesh.edges.begin(); e != mesh.edges.end(); e++)
	{
		he = e->he;
		he2 = he->flip;

		idx = he->face->index;
		jdx = he2->face->index;

		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat_coefficients.push_back(Triplet(4*idx+u, 4*jdx+v, zero));
			}
		}
		for(int u = 0; u < 4; u++) {
			for(int v = 0; v < 4; v++) {
				Mat_coefficients.push_back(Triplet(4*jdx+u, 4*idx+v, zero));
			}
		}
	}

	Mat.setFromTriplets(Mat_coefficients.begin(), Mat_coefficients.end());

  }

  /* =============================================================================================
   Build Adjacency matrix
   =============================================================================================== */

  void Willmore::initAdj(Mesh& mesh, Eigen::SparseMatrix<double> &Mat)
  {

	// Initialize all values to 0

	int idx;
	int idv1, idv2, idv3;
	std::vector<Triplet> Mat_coefficients;
	double zero = 0;

	for (FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		idx = f->index;
		idv1 = f->he->vertex->index;
		idv2 = f->he->next->vertex->index;
		idv3 = f->he->next->next->vertex->index;
		for(int u = 0; u < 4; u++) {
			Mat_coefficients.push_back(Triplet(4*idx+u, 4*idv1+u, zero));
			Mat_coefficients.push_back(Triplet(4*idx+u, 4*idv2+u, zero));
			Mat_coefficients.push_back(Triplet(4*idx+u, 4*idv3+u, zero));
		}
	}

	Mat.setFromTriplets(Mat_coefficients.begin(), Mat_coefficients.end());

  }

  /* =============================================================================================
   Build Adjacency matrix
   =============================================================================================== */

  void Willmore::buildAdj(Mesh& mesh)
  {

	// Initialize all values to 0

	int idx;
	int idv1, idv2, idv3;
	double a1, a2, a3;

	for (FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		idx = f->index;
		idv1 = f->he->vertex->index;
		idv2 = f->he->next->vertex->index;
		idv3 = f->he->next->next->vertex->index;
		a1 = 1./std::sqrt(varea[idv1]);
		a2 = 1./std::sqrt(varea[idv2]);
		a3 = 1./std::sqrt(varea[idv3]);
		for(int u = 0; u < 4; u++) {
			Adj.coeffRef(4*idx+u, 4*idv1+u) = a1;
			Adj.coeffRef(4*idx+u, 4*idv2+u) = a2;
			Adj.coeffRef(4*idx+u, 4*idv3+u) = a3;
		}
	}

  }


  /* =============================================================================================
   * update real sparse matrix with a quaternion value
   =============================================================================================== */

  void Willmore::updateMatrix(Eigen::SparseMatrix<double>& Mat, int i0, int j0, double Q[4][4])
  {
	int idx, jdx;
	for(int u = 0; u < 4; u++) {
		for(int v = 0; v < 4; v++) {
			idx = 4*i0 + u;
			jdx = 4*j0 + v;
			Mat.coeffRef(idx, jdx) = Q[u][v];
		}
	}
   }

  /* =============================================================================================
   *	Build Dirac
   =============================================================================================== */

void Willmore::buildDirac(Mesh& mesh)
{

	HalfEdgeIter hAB, hBA;
	Vector vA, vB;
	int f1, f2;

	Quaternion q;
	double Q[4][4];

	// visit each edge and add Hyperedge to Dirac matrix
	for( EdgeIter e_iter = mesh.edges.begin(); e_iter != mesh.edges.end(); e_iter++ )
	{

		hAB =e_iter->he;
		hBA =hAB->flip;

		f1 = hAB->face->index;
		f2 = hBA->face->index;

		vA = hAB->vertex->position2;
		vB = hBA->vertex->position2;

		q = Quaternion(ecurv[e_iter->index], vB - vA);
		q.toMatrix(Q);
		updateMatrix(Dirac, f1, f2, Q);
		(~q).toMatrix(Q);
		updateMatrix(Dirac, f2, f1, Q);

	}

	// Now add mean curvature on the diagonal
	double eps = 1.e-7;
	for(FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++) {
		int id = f->index;
		for(int u = 0; u < 4; u++) {
			int idx = 4*id + u;
			Dirac.coeffRef(idx, idx) = -fcurv[id]+eps;
		}
	}
 }

  /* =============================================================================================
   *	Add (fraction of) target mean curvature to extrinsic Dirac operator
   =============================================================================================== */

  void Willmore::addTargetCurv(Mesh& mesh, double dt)
  {

	double sc = 0, sA = 0;
	int id;
	for(FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++) {
		id = f->index;
		rho[id] = fcurv[id];
		sc += rho[id];
		sA += farea[id];
	}

	int idx;
	for(FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++) {
		id = f->index;
		rho[id] = dt*( rho[id] - sc*farea[id]/sA);
		for(int u = 0; u < 4; u++) {
			idx = 4*id + u;
			Dirac.coeffRef(idx, idx) += rho[id];
		}
	}
  }
 

/*================================================================================================
 PowerMethod: finds the smallest eigenvalue of the generalized eigenvalue problem:
	A^T D D A \phi = \lambda M_v \phi
 where:
	A is the adjacency matrix (size : 4 |F| * 4 |V|
	D is the extrinsic Dirac operator matrix
	M_v is a diagonal vertex mass matrix
 Since we only need the smallest eigenvalue, we use an inverse Power method
================================================================================================== */

void Willmore::solveForPhi(Mesh& mesh)
{

	int iter_max = 20;

	// initialize vector to the identity
	temp1.setZero();
	for(int i = 0; i < temp1.size()/4; i++) temp1.coeffRef(4*i) = 1.0;
	temp1.normalize();

	// Build M1 = A^T D D A = (DA)^T DA and perform Cholesky factorization
	DA = Dirac*Adj;
	M1 = 0.5*DA.transpose()*DA;

	if(nE==0) solverE.analyzePattern(M1);
        nE++;
	solverE.factorize(M1);

 	// Power iterations

	double tol = 1.e-6;
	double eig0, eig1;
	eig0 = 0.0;

	for(int iter =0; iter < iter_max; iter++)
	{
		// Solve M1 * temp2 = Mv * temp1; temp2 will be the new estimate of the eigenvector
		temp2 = solverE.solve(temp1);

 		//Normalize new vector
		eig1 = temp1.dot(temp2);
		temp2.normalize();

 		// Check for convergence

		if(std::abs(eig1-eig0) < tol) {
			break;
		}

 		// Prepare for next iteration

		temp1 = temp2;
		eig0 = eig1;

	}

	temp1.normalize();
	for(int i = 0; i < temp1.size(); i++) {
		int idx = i/4;
		temp1[i] /= std::sqrt(varea[idx]);
	}

	alignPhi();

	phiVertex2Face(mesh);

}
	
  /* =============================================================================================
   *	Align eigenvector
   =============================================================================================== */

  void Willmore::alignPhi()
  {
	int nvertex = temp1.size()/4;
	std::vector<double> er(nvertex);
	std::vector<double> ei(nvertex);
	std::vector<double> ej(nvertex);
	std::vector<double> ek(nvertex);

	double norm_sq, val, s;
	double sr=0, si=0, sj=0, sk=0;
	double vr, vi, vj, vk;
	for(int i = 0; i < nvertex; i++) {
		er[i] = temp1[4*i];
		ei[i] = temp1[4*i+1];
		ej[i] = temp1[4*i+2];
		ek[i] = temp1[4*i+3];
		vr = er[i]; vi = ei[i]; vj = ej[i]; vk=ek[i];
		norm_sq = vr*vr + vi*vi + vj*vj + vk*vk;
		val = varea[i]/norm_sq;
		vr *= val;
		vi *= -val;
		vj *= -val;
		vk *= -val;
		sr += vr;
		si += vi;
		sj += vj;
		sk += vk;
	}
	
	s=std::sqrt(sr*sr + si*si + sj*sj + sk*sk);
	sr = sr/s; si = si/s; sj = sj/s; sk=sk/s;

	for(int i = 0; i < nvertex; i++) {
		temp1[4*i+0] = er[i]*sr - ei[i]*si - ej[i]*sj - ek[i]*sk;
		temp1[4*i+1] = er[i]*si + ei[i]*sr + ej[i]*sk - ek[i]*sj;
		temp1[4*i+2] = er[i]*sj - ei[i]*sk + ej[i]*sr + ek[i]*si;
		temp1[4*i+3] = er[i]*sk + ei[i]*sj - ej[i]*si + ek[i]*sr;
	}
  }

  /* =============================================================================================
   *	Convert vertex-based eigenvector to face-based eigenvector
   =============================================================================================== */

  void Willmore::phiVertex2Face(Mesh& mesh)
  {

	for(int i = 0; i < 4*mesh.vertices.size(); i++) lambda_v[i] = temp1[i];

	int id, idx, jdx, kdx;
	for(FaceIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		id = f->index;
		idx = f->he->vertex->index;
		jdx = f->he->next->vertex->index;
		kdx = f->he->next->next->vertex->index;

		lambda_f[4*id+0] = temp1[4*idx+0] + temp1[4*jdx+0] + temp1[4*kdx+0];
		lambda_f[4*id+1] = temp1[4*idx+1] + temp1[4*jdx+1] + temp1[4*kdx+1];
		lambda_f[4*id+2] = temp1[4*idx+2] + temp1[4*jdx+2] + temp1[4*kdx+2];
		lambda_f[4*id+3] = temp1[4*idx+3] + temp1[4*jdx+3] + temp1[4*kdx+3];

	}

  }

  /* =============================================================================================
   *	Normalize mesh: re-center, and scale so that points are at most at distance 1 from center
   =============================================================================================== */

  void Willmore::normalizeSolution(Mesh& mesh)
  {

	double Area, Vol, S;
	computeAVS(mesh, &Area, &Vol, &S);
	double Scale = std::sqrt(Area0/Area);

	Vector center;
	center[0] = 0; center[1] = 0; center[2]=0;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		v->position2 *= Scale;
		center[0] += v->position2[0];
		center[1] += v->position2[1];
		center[2] += v->position2[2];
	}
	int n_vertices = mesh.vertices.size();
	center /= n_vertices;

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		v->position2 -= center;
	}

  }

  /* ===============================================================================================
   FitSphere: Fit a sphere into the vertices of a mesh, and then translate the mesh such that
		the center of the sphere is at 0,0,0

   Input:
	  mesh:	 the mesh data structure (pointer to the structure)
   =============================================================================================== */

  void Willmore::fitSphere(Mesh& mesh)
  {
	int n_vertices = mesh.vertices.size();

	Eigen::MatrixXd A(n_vertices, 4);
	Eigen::VectorXd B(n_vertices);
	Eigen::VectorXd C(4);

	Vector pointA, centerA;

	double xA, yA, zA;
	int idx = 0;
	for(VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
		pointA = v_it->position2;
		xA = pointA[0]; yA = pointA[1]; zA = pointA[2];
		A(idx, 0) = 2*xA; A(idx, 1) = 2*yA; A(idx, 2) = 2*zA;
		A(idx, 3) = 1.0;
		B[idx]    = xA*xA + yA*yA + zA*zA;
		idx++;
	}

	C = A.bdcSvd(Eigen::ComputeThinU | Eigen::ComputeThinV).solve(B);

	centerA[0] = C[0]; centerA[1] = C[1]; centerA[2] = C[2];

	for(VertexIter v_it = mesh.vertices.begin(); v_it != mesh.vertices.end(); v_it++)
	{
		pointA = v_it->position2;
		pointA = pointA-centerA;
		pointA = pointA / pointA.norm();
		v_it->position2[0] = pointA[0];
		v_it->position2[1] = pointA[1];
		v_it->position2[2] = pointA[2];
	}
  }
