/* ===============================================================================================
  Geometry.h

   A set of Small functions that characterize the geometry of a mesh:
	- Area:	       		computes the surface area of a triangle
	- Volume:      		computes the volume of a tetrahedron
	- MeshMeasure: 		computes Surface Area of the mesh and the volume of
		       		the region that it encloses
	- ScaleMesh:   		scales the coordinates of the vertices of the mesh
		       		so that the scaled mesh is of area 1
	- VertexCurvature: 	computes the curvature at all vertices
	- MixedArea:   		computes the mixed area associated with each vertex

   Author:  Patrice Koehl
   Date:    10/11/2017
   Version: 1
   =============================================================================================== */

#pragma once

  /* ===============================================================================================
  Define MeshGeometry class
   =============================================================================================== */

  class MeshGeometry {

  public:
	// Computes surface area and volume of a mesh
	static void measureMesh(const Mesh& mesh, double *Area, double *Vol);

	// Scale mesh
	static void scaleMesh(Mesh& mesh, const double Scale);

	// center Mesh on (0, 0, 0)
	static void centerMesh(Mesh& mesh);

	// Compute curvature at each vertex
	static void curvatureMesh(const Mesh& mesh, double *Curvature);

	// Compute mixed area at each vertex
	static void mixedAreaMesh(const Mesh& mesh, double *MixedArea, double *Area);

	// Compute angles and cotans in a triangle
	static void trigAngles(const Vector& a, const Vector& b, const Vector& c, double *angs, double *cotans);

  private:
	// Computes surface area of a triangle
	static double  trigArea(const Vector& A, const Vector& B, const Vector& C);

	// Computes volume of tetrahedron associated with a triangle
	static double  trigVol(const Vector& A, const Vector& B, const Vector& C);

  };

  /* ===============================================================================================
   trigArea: Computes surface area of a triangle given by 3 points.

         if the three points are A, B, and C,
	 S = norm( A-B) x (A-C) )/2
	 where x is the cross product

   Input:
	  A, B, C: the three vertices of the triangle
   Output:
	  Area: surface area of the triangle
   =============================================================================================== */

  double MeshGeometry::trigArea(const Vector& A, const Vector& B, const Vector& C)
  {
	double Area;
	Area = 0.5 * cross(A-B , A-C).norm() ;
	return Area;
  }

  /* ===============================================================================================
   trigVol: Computes the volume associated with one triangle of a mesh

        Let the three vertices of the triangles be A, B, and C.
	We consider the tetrahedron formed by (A,B,C,O), where O is the origin
	of the coordinate system.
	We compute V, the signed volume of this tetrahedron. The sum of those
	volumes over the whole mesh will give us the volume enclosed in the mesh.
	This signed volume is:
	V = (A . (BxC))/6.0
	where A is the dot product and x the cross product

   Input:
	  A, B, C: the three vertices of the triangle
   Output:
	  Vol: the (signed) volume of the tetrahedron (A, B, C, O)
   =============================================================================================== */

  double MeshGeometry::trigVol(const Vector& A, const Vector& B, const Vector& C)
  {
	double V;
	V = dot(A , cross(B, C) )/6.0;
	return V;
  }

  /* ===============================================================================================
   trigAngles: Computes the three angles of a triangle and their cotans
   =============================================================================================== */

  void MeshGeometry::trigAngles(const Vector& a, const Vector& b, const Vector& c, double *ang, double *cotan)
  {
	double lab, lca, lbc;
	double val, alpha_a, alpha_b, alpha_c;
	double cot_a, cot_b, cot_c;

	lab = (a-b).norm();
	lca = (a-c).norm();
	lbc = (b-c).norm();

	val = (lab*lab + lca*lca - lbc*lbc)/(2.0*lab*lca);
	alpha_a = std::acos(val);
	cot_a   = 1.0/std::tan(alpha_a);
	val = (lab*lab + lbc*lbc - lca*lca)/(2.0*lab*lbc);
	alpha_b = std::acos(val);
	cot_b   = 1.0/std::tan(alpha_b);
	val = (lca*lca + lbc*lbc - lab*lab)/(2.0*lca*lbc);
	alpha_c = std::acos(val);
	cot_c   = 1.0/std::tan(alpha_c);

	ang[0] = alpha_a; ang[1] = alpha_b; ang[2] = alpha_c;
	cotan[0] = cot_a; cotan[1] = cot_b; cotan[2] = cot_c;
  }
  /* ===============================================================================================
   measureMesh: Computes surface area and volume of the mesh

   Uses repeatedly the functions "area" and "volume" that compute the contribution
   of one triangle of the mesh to the total area of the mesh, and total volume
   enclosed by the mesh

   Input:
	  mesh:	the mesh data structure
   Output:
	  Area: surface area of the mesh (reference)
	  Vol:  volume of the mesh (reference)
   =============================================================================================== */

  void MeshGeometry::measureMesh(const Mesh& mesh, double *Area, double *Vol)
  {

	double SA = 0;
	double V  = 0;

	HalfEdgeCIter he;
	VertexCIter va, vb, vc;
	Vector a, b, c;
	int idxa, idxb, idxc;

	for (FaceCIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		if (f->isReal()) {
			he = f->he;
			va = he->vertex;
			vb = he->next->vertex;
			vc = he->prev->vertex;
			idxa = va->index;
			idxb = vb->index;
			idxc = vc->index;
			a = va->position;
			b = vb->position;
			c = vc->position;
			SA += trigArea(a, b, c);
			V  += trigVol(a, b, c);
		}
	}

	*Area = SA;
	*Vol  = std::abs(V);
  }

  /* ===============================================================================================
   scaleMesh: Scale coordinates of all vertices in a mesh by a constant, Scale.

   Input:
	  mesh:	 the mesh data structure (pointer to the structure)
	  Scale: the scaling factor
   Output:
	  mesh:	 the scaled mesh
   =============================================================================================== */

  void MeshGeometry::scaleMesh(Mesh& mesh, const double Scale)
  {
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		v->position *= Scale;
	}
  }

  /* ===============================================================================================
   CenterMesh: Center mesh on (0,0,0)

   Input:
	  mesh:	 the mesh data structure (pointer to the structure)
   Output:
	  mesh:	 the scaled mesh
   =============================================================================================== */

  void MeshGeometry::centerMesh(Mesh& mesh)
  {
	Vector cm(0., 0., 0.);

	int nvertices=0;
	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		cm += v->position;
		nvertices++;
	}
	cm /= nvertices;

	for (VertexIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		v->position -= cm;
	}
   }

  /* ===============================================================================================
   curvatureMesh: computes the curvatures of all vertices

   The curvature is defined as the excess angle at each vertex i, i.e. to
   2pi - sum_t Theta(i,ijk)
   where sum_t indicates a sum over all triangles (ijk) that are incident to i
         Theta(i,ijk) is the angle at vertex i in the triangle (i,jk)

   Input:
	  mesh:	the mesh data structure (pointer to the structure)
   Output:
	  Vcurv: the curvature at each vertex
   =============================================================================================== */

  void MeshGeometry::curvatureMesh(const Mesh& mesh, double *Vcurv)
  {
	double TwoPI = 2*M_PI;

/* 	==========================================================================================
	Initialize Curvature to 2*PI
        ========================================================================================== */

	HalfEdgeCIter he;
	VertexCIter va, vb, vc;
	Vector a, b, c;

	int idxa, idxb, idxc;
	double angs[3], cotans[3];

	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		idxa = v->index;
		Vcurv[idxa] = TwoPI;
	}

/* 	==========================================================================================
	Iterate over all triangles and remove angle for each vertex
        ========================================================================================== */

	for (FaceCIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		if (f->isReal()) {
			he = f->he;
			va = he->vertex;
			vb = he->next->vertex;
			vc = he->prev->vertex;
			idxa = va->index;
			idxb = vb->index;
			idxc = vc->index;
			a = va->position;
			b = vb->position;
			c = vc->position;

			trigAngles(a, b, c, angs, cotans);

			Vcurv[idxa] -= angs[0];
			Vcurv[idxb] -= angs[1];
			Vcurv[idxc] -= angs[2];
		}
	}

  }

  /* ===============================================================================================
   mixedAreaMesh: computes the mixed area of all vertices (i.e. area of the Voronoi region
		associated to a vertex)

   We use the procedure described in:
   M. Meyer, M. Desbrun, P. Schroeder, A. Barr. "Discrete differential geometry operators
   for triangulated 2-manifolds", in: VisMath, Vol 2, pp 35-57 (2002)

   Input:
	  mesh:	the mesh data structure (pointer to the structure)
   Output:
	  MixedArea: mized area of each vertex
	  Area     : total area of the mesh
   =============================================================================================== */

  void MeshGeometry::mixedAreaMesh(const Mesh& mesh, double *MixedArea, double *Area)
  {
	double HalfPI = M_PI/2;

	HalfEdgeCIter he;
	VertexCIter va, vb, vc;
	Vector a, b, c;

	int idxa, idxb, idxc;

	double angs[3], cotans[3];

	double amix, bmix, cmix;
	double lab, lca, lbc;
	double Tarea;
	double tmix = 0;

/* 	==========================================================================================
	Initialize mixed area to 0
        ========================================================================================== */

	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++)
	{
		idxa = v->index;
		MixedArea[idxa] = 0;
	}

/* 	==========================================================================================
	Iterate over each triangle:
		For each triangle:
			Divide area into three parts, one for each vertex, using either:
			- Voronoi formula (see paper by Meyer et al) for non-obtuse triangles
			- simpler scheme for obtuse triangle (again, see paper by Meyer et al)
        ========================================================================================== */

	for (FaceCIter f = mesh.faces.begin(); f != mesh.faces.end(); f++)
	{
		if (f->isReal()) {
			he = f->he;
			va = he->vertex;
			vb = he->next->vertex;
			vc = he->prev->vertex;
			idxa = va->index;
			idxb = vb->index;
			idxc = vc->index;
			a = va->position;
			b = vb->position;
			c = vc->position;

			lab = (a-b).norm();
			lca = (c-a).norm();
			lbc = (b-c).norm();

			trigAngles(a, b, c, angs, cotans);
			Tarea = trigArea(a, b, c);

			if(angs[0] > HalfPI || angs[1] > HalfPI || angs[2] > HalfPI) {
				if(angs[0] > HalfPI) amix = Tarea/2;
				else amix = Tarea/4;
				if(angs[1] > HalfPI) bmix = Tarea/2;
				else bmix = Tarea/4;
				if(angs[2] > HalfPI) cmix = Tarea/2;
				else cmix = Tarea/4;
			}
			else {
				amix = (cotans[2]*lab*lab + cotans[1]*lca*lca)/8.0;
				bmix = (cotans[2]*lab*lab + cotans[0]*lbc*lbc)/8.0;
				cmix = (cotans[0]*lbc*lbc + cotans[1]*lca*lca)/8.0;
			}

			MixedArea[idxa] += amix;
			MixedArea[idxb] += bmix;
			MixedArea[idxc] += cmix;
		}

	}

/* 	==========================================================================================
	Compute total area: sum of the mixed areas of all vertices
        ========================================================================================== */

	for (VertexCIter v = mesh.vertices.begin(); v != mesh.vertices.end(); v++) 
	{
		idxa = v->index;
               	tmix += MixedArea[idxa];
	}

	*Area = tmix;

  }

