/* ===============================================================================================
   Map2Sphere: Conformal mapping of a genus zero surface represented by a triangular mesh onto
               the sphere.
               Different methods are implemented:
		1) Intrinsic methods:
			1.a: Tutte embedding
			1.b: Yamabe flow (Springborn et al, 2007)
			1.c: Ricci flow
                2) Extrinsic method:
			2.a: conformalized Mean Curvature Flow

   Author:  Patrice Koehl
   Date:    5/10/2019
   Version: 1
   =============================================================================================== */

/* ===============================================================================================
   Includes
   =============================================================================================== */

#include "Map2Sphere.h"
#include "staticvar.hpp"

/* ===============================================================================================
   Main program
   =============================================================================================== */

int main(int argc, char **argv)
{

/*	==========================================================================================
	Show usage if needed
	========================================================================================== */

	if( argc < 2 )
	{
		usage(argv);
		return -1;
	}

	std::string input = argv[1];
	if( input == "-h" || input == "-help" )
	{
		usage(argv);
		return -1;
	}

/*	==========================================================================================
	Read in all inputs (some values may have been preset)
	========================================================================================== */

	std::string INfile;
	std::string OUTfile;

        if (!parse_args(argc, argv, &INfile, &OUTfile)) return 1;

/*	==========================================================================================
	Read in mesh from input file
	========================================================================================== */

	Mesh model;

	std::string error;
	std::size_t found = INfile.find("obj");
	if(found !=std::string::npos) {
		MeshIO::readOBJ(INfile, model, error);
	} else {
		found = INfile.find("off");
		if(found !=std::string::npos) {
			MeshIO::readOFF(INfile, model, error);
		} else {
			std::cout << " " << std::endl;
			std::cout << "Input file format not recognized; program can only read OFF and OBJ files" << std::endl;
			std::cout << " " << std::endl;
			exit(1);
		}
	}

/*	==========================================================================================
	Characteristics of the mesh
	========================================================================================== */

	int nvertices = model.vertices.size();
	int nfaces    = model.faces.size();
	int nbound    = model.boundaries.size();
	int euler     = model.eulerCharacteristic();
	double genus  = (2-euler)/2;

	std::cout << " " << std::endl;
	std::cout << "Number of vertices in mesh         : " << nvertices << std::endl;
	std::cout << "Number of faces in mesh            : " << nfaces << std::endl;
	std::cout << "Number of boundaries in mesh       : " << nbound << std::endl;
	std::cout << "Euler characteristics              : " << euler << std::endl;
	std::cout << "Genus                              : " << genus << std::endl;

	double Area, Vol, Sphericity;
	MeshGeometry::measureMesh(model, &Area, &Vol);
	Sphericity = std::pow(M_PI,1.0/3.0) * std::pow(6.0*Vol,2.0/3.0) / Area;
			
	std::cout << " " << std::endl;
	std::cout << "Total surface area of scaled mesh  : " << Area << std::endl;
	std::cout << "Total volume of the mesh           : " << Vol << std::endl;
	std::cout << "Sphericity of the mesh             : " << Sphericity << std::endl;
	std::cout << " " << std::endl;

/*	==========================================================================================
	Call viewer
	========================================================================================== */

	for(VertexIter v = model.vertices.begin(); v != model.vertices.end(); v++)
	{
		v->position2 = v->position;
	}
	Viewer::init(model);
	
	return 0;

}

/* ===============================================================================================
   Usage
   =============================================================================================== */

static void usage(char** argv)
{
    std::cout << "\n\n" <<std::endl;
    std::cout << "     " << "================================================================================================"<<std::endl;
    std::cout << "     " << "================================================================================================"<<std::endl;
    std::cout << "     " << "=                                                                                              ="<<std::endl;
    std::cout << "     " << "=                                         Map2Sphere                                           ="<<std::endl;
    std::cout << "     " << "=                                                                                              ="<<std::endl;
    std::cout << "     " << "=     This program reads in a 3D surface represented by a triangular mesh, checks that         ="<<std::endl;
    std::cout << "     " << "=     it is a manifold and that its genus is 0, and map it conformally onto the sphere         ="<<std::endl;
    std::cout << "     " << "=                                                                                              ="<<std::endl;
    std::cout << "     " << "=     Usage is:                                                                                ="<<std::endl;
    std::cout << "     " << "=          Map2Sphere.exe -i INFILE -o OUTFILE                                                 ="<<std::endl;
    std::cout << "     " << "=     where:                                                                                   ="<<std::endl;
    std::cout << "     " << "=                 -i INFILE       --> Input Mesh file (OBJ or OFF format)                      ="<<std::endl;
    std::cout << "     " << "=                 -o OUTFILE      --> Ouput Mesh file in OFF format                            ="<<std::endl;
    std::cout << "     " << "================================================================================================"<<std::endl;
    std::cout << "     " << "================================================================================================"<<std::endl;
    std::cout << "\n\n" <<std::endl;
}

/* ===============================================================================================
   Parse Argument from command line:

   =============================================================================================== */

bool parse_args(int argc, char **argv, std::string *INfile, std::string *OUTfile) 
{
//
// Make sure we have at least two parameters....
//
	std::string param;
	if (argc == 1)
	{
		return false;
	}
	else
	{
		for (int i = 1; i < argc - 1; i = i + 2)
		{
			param = argv[i];

			if (param == "-i") {
				*INfile = argv[i + 1];
			}
			else if (param == "-o") {
				*OUTfile = argv[i + 1];
			}
		}
  	}
	return true;
}
